#include <windows.h>
#include <stdio.h>
#include "barbados.h"
#include "modules.h"
#include "graph2d.h"
#include "memory.h"
#include "array.h"
#include "tfc.h"
#include "name.h"



/*------------------- Static variables and other stuff -----------------------*/
interface
int PersistenceVersion           = 20218;

interface
b_err_type b_errno;

const
short int NamedObjHash::NH       = 20;

CNInfoList Conim::prereqinfo;
Conim *Conim::beingwritten       = NULL;
Conim **Conim::ContainersNeedingOfLocalSwizzle
                                 = NULL;

static const
void * TABLE_FINISH_MARK         = (void *) 0xFFFFFFFF;

static
char* Image;

static
char* Base;

static
uint compaction_offset;

Conim** Conim::Hash;
uint Conim::NumConims;
bool Conim::LinkGCRequired;


//==============================================================================
//------------------------------------- Interface implementation for this module
interface Conim* FindConim(container_id cid)
/* Do we have this already in memory? */
{
	return Conim::FindConim(cid);
}


interface void CloseContainer(container_id cid)
/* Close the specified container. */
{
	Conim* conim = FindConim(cid);
	b_errno      = E_NOERROR;
	if (conim)
        {
                conim->Close();
        }
	else b_errno = E_NOTOPEN;
}


interface void CloseConim(Conim *conim)
/* Close the specified container. */
{
	conim->Close();
}


interface Directory* CreateContainer(container_id parent, container_id *cidp)
/* Create a container of this type and this parent. */
{	Conim* conim;

	conim = Conim::CreateContainer(parent);
	if (conim == NULL)
	    return NULL;
	*cidp = conim->cid;
	return conim->directory();
}


interface Directory* OpenContainer(container_id cid, bool write_permission)
/* Open a pre-existing container into memory */
{
	Conim* conim = Conim::OpenContainer(cid, write_permission);
	if (conim == NULL)
	    return NULL;
	return conim->directory();
}


interface bool DeleteContainer(container_id cid)
/* Delete this container. The container must not be open. */
/* We delete the file from disk and the id from the tree-table. */
{	char path[PATHLENGTH];

	if (FindConim(cid))
	    return no;
	if (not DeleteContainerId(cid))
	    return no;
	CidToPath(cid, path, sizeof(path));
	DeleteFile(path);
	return yes;
}


interface Conim* Ptr_to_conim(void *ptr)
/* Mapping arbitrary pointers to conims */
{
	return Conim::Ptr_to_conim(ptr);
}



//==============================================================================
//------------------------------------------------------------ Auxiliary classes
//----------------------------------------------------------- NamedObjHash Class
/*
        While loading, we need to locate namedobj's by their u.location field,
        which is not the common one.
        This class is in charge of maintain a temporal mapping in order to make
        easy to access classes.
*/

Namedobj *NamedObjHash::getNamedObj(void *key)
/* Return the static NamedObj in this container whose location equals 'key'. */
{       Namedobj *obj, **list;

        list = h[((unsigned int) key) % NH];
        for (int each_aeli(obj, list))
            if (objLocation == key)
                return obj;
        return NULL;
}


NamedObjHash::~NamedObjHash()
// Freeing resources
{
        for (unsigned int i=0; i<NH; i++)
            if (h[i]!=NULL)
               Array_Free(h[i]);
        delete h;
}


NamedObjHash::NamedObjHash()
// Constructor -- prepare the hashing to operate
{
        h = new Namedobj**[NH];
        memset(h, 0, sizeof(void*) * NH);
}


void NamedObjHash::putNamedObj(Namedobj* obj)
// Inserts this Namedobj in the hashing
{       void* data;

        if (obj->storage == typedef_storage)
            data = obj->type;
        else
            data = ((StaticNamedobj*)obj)->location;
        Array_InsP(h[int(data) % NH], obj);
}

//---------------------------------------------------------- ConimIterator Class
//------------------------------------------------------- Iterating thru Conim's
void ConimIterator::reset()
{
	h = (uint)-1;
	conim = NULL;
}


Conim* ConimIterator::operator++(int c)
{
	if (conim)
	    conim = conim->next;
	while (conim == NULL) {
	    if (++h >= Array_Size(Conim::Hash))
		return NULL;
	    conim = Conim::Hash[h];
	}
	return conim;
}

//----------------------------------------------------------------- CNInfo class
inline
str  CNInfo::getName()
{
        return ((char*) databuffer + 1);
}

inline
char CNInfo::getTypeOfPointer() const
{
        return TypeOfPointer;
}

inline
bool CNInfo::isAbsolute()  const
{
        return (TypeOfPointer==Absolute_Ptr_Mark);
}

inline
bool CNInfo::isRelative() const
{
        return (TypeOfPointer==Relative_Ptr_Mark);
}

inline
bool CNInfo::isTypeClass() const
{
        return (TypeOfPointer==Relative_Ptr_Mark);
}

inline
bool CNInfo::isCorrupted() const
{
        return (TypeOfPointer!=Absolute_Ptr_Mark
                and TypeOfPointer!=Relative_Ptr_Mark)
                and TypeOfPointer!=TypeClass_Ptr_Mark;
}

inline
container_id CNInfo::getForeignContainer(void) const
{
        return container;
}

inline
Namedobj* CNInfo::getNamedobj(void) const
{
        return whereispointing;
}

inline
char *CNInfo::getReducedClassdef(void)
{
        return ReducedClassdef;
}

inline
void CNInfo::addPtr(void *ptr)
{
        Array_Add(whereis, ptr);
}


inline
void CNInfo::markasInvalidContainer()
/* We've tried to find this entry's container but we haven't been able to */
{
        container = 0;  // Invalid container !
}

void CNInfo::putPtrsNULL()
/* The location this entry's ptrs point to is no longer reachable, so
   we should put this ptrs to NULL */
{
     int i;
     void *ptr;

     for(each_aeli(ptr, whereis))
           *((void **)ptr) = NULL;
}

inline
int CNInfo::getPtrListSize() const
{
        return Array_Size(whereis);
}

void CNInfo::rebasePtrs(int base)
/* At the beginning, when loading, ptrs are relative. We have to add to them
   the base_address of the container */
{
     for(int i = 0;i<Array_Size(whereis); ++i)
        *(((int *) whereis) + i) += base;
}

void *CNInfo::getPtrIndx(int num) const
/* Returns a pointer to a ptr entry following a zero-based index */
{
        if (num>=getPtrListSize())
                return NULL;

        return Array_Idx(whereis, num);
}

CNInfo *CNInfo::loadCNInfo(char *&wr)
/* loads a CNInfo entry mapped on memory */
{
        int          namelength,
                     ptrslength,
                     rclength;
        CNInfo       *reg;
        container_id container;
        str          namedobj_name,
                     reducedclassdef;
        Namedobj    *obj;

        // Get the Namedobj while-saving-address
        obj              = *((Namedobj **) wr);
        wr              += sizeof(obj);

        // Get the container's id
        container        = *((container_id *) wr);
        wr              += sizeof(container);

        // Prepare the Name of the foreign object
        namelength       = *((int *) wr);
        wr              += sizeof(int); // name's length
        namedobj_name    = wr;
        wr              += namelength;

        // Prepare the ReducedClassdef
        rclength         = *((int *) wr);
        *((int *) wr)    = 0;           // Hack - zero for name (above)
        wr              += sizeof(int); // ReducedClassdef's length
        reducedclassdef  = wr;
        wr              += rclength;

        // Get the pointers list's length
        ptrslength       = *((int *) wr);
        *((int *) wr)    = 0;           // Hack - zero for reducedclassdef (above)
        wr              += sizeof(int); // PointerTable's length

        // Store this information
        reg = new CNInfo(*((void **) wr),
                         obj, container, namedobj_name, reducedclassdef);

        // Read all remaining ptrs
        ptrslength -= sizeof(void *);
        wr         += sizeof(void *);
        for (char* limit = wr + ptrslength; wr < limit; wr += sizeof(void *))
              reg->addPtr(*((void **) wr));

        return reg;
}

void CNInfo::saveCNInfo(int _FileHandle)
/* Save an existing CNInfo entry to disk */
{
        Namedobj*     obj;
        char              buffer[PATHLENGTH],
                         *buf;
        int               length;
        container_id      cont;
        void             *ptr;
        unsigned long int numwritten;
        HANDLE FileHandle = (HANDLE)_FileHandle;

        obj   = getNamedobj();
        assert(obj!=TABLE_FINISH_MARK);

        // Firstly, the location of the pointer
        WriteFile(FileHandle,
                    &obj,
                    sizeof(obj),
                    &numwritten,
                    NULL)
        ;

        // Next, the integer containing the container_id
        cont = getForeignContainer();
        WriteFile(FileHandle,
                     &cont,
                     sizeof(cont),
                     &numwritten, NULL)
        ;

        // Then, the string with the name of the namedobj_node
        // is written (firstly its length)
        // Its first position is the nature of the pointer:
        // absolute or relative
        NameUnameFromObj(obj, buffer + 1);
        *buffer = getTypeOfPointer();
        length  = strlen(buffer);
        WriteFile(FileHandle,
                     &length,
                     sizeof(length),
                     &numwritten, NULL)
        ;
        WriteFile(FileHandle,
                     (void *) buffer,
                     length,
                     &numwritten, NULL)
        ;

        // Then, the string with the reduceclassdef (provided this
        // is a TypeClass) is written (firstly its length)
        // Its first position is the nature of the pointer:
        // absolute, relative, typeclass ...
        if (getReducedClassdef() == NULL)
        {
            buf    = buffer;
            *buf   = 0;
            length = 1;
        }
        else
        {
            buf    = getReducedClassdef();
            length = strlen(buf);
        }
        WriteFile(FileHandle,
                     &length,
                     sizeof(length),
                     &numwritten, NULL)
        ;
        WriteFile(FileHandle,
                     buf,
                     length,
                     &numwritten, NULL)
        ;

        // And, finally, all pointers related to this entry
        length = getPtrListSize() * sizeof(void *);
        WriteFile(FileHandle,
                     &length,
                     sizeof(length),
                     &numwritten, NULL)
        ;
        for (int j=0;j < getPtrListSize(); ++j)
        {
            ptr = getPtrIndx(j);
            assert(((int) ptr) <= compaction_offset);
            assert(ptr != TABLE_FINISH_MARK);

            WriteFile(FileHandle,
                     &ptr,
                     sizeof(ptr),
                     &numwritten, NULL);
        }
}


CNInfo::CNInfo(void *ptr, Namedobj* obj, str RCinfo, char PtrType)
        : whereis(NULL), databuffer(NULL), container(0),
          whereispointing(obj), ReducedClassdef(NULL), TypeOfPointer(PtrType)
// Constructor suitable to be used while saving containers
{
        Conim *conim;

        // Deduce the foreign container
        if ((conim = Ptr_to_conim(obj)) != NULL)
        {
                container = conim->cid;

                // Copy the ReducedClassdef
                if (RCinfo != NULL)
                   ReducedClassdef = strdup(RCinfo);

                // Add the pointer to the list
                addPtr(ptr);
        }
        else assert(false);
}

CNInfo::CNInfo(void *ptr, Namedobj* obj, container_id cont, str info,
                                                            str RCinfo)
        : whereis(NULL), databuffer(NULL), whereispointing(NULL), container(0),
          TypeOfPointer(0), ReducedClassdef(NULL)
// Constructor suitable to be used while loading containers
{
        if (RCinfo != NULL)
           databuffer      = strdup(info);
        if (RCinfo != NULL)
           ReducedClassdef = strdup(RCinfo);

        if  (databuffer      != NULL
         and ReducedClassdef != NULL)
        {
                TypeOfPointer       = databuffer[0];
                whereispointing     = obj;
                container           = cont;
                addPtr(ptr);
        } else assert(false);
}

CNInfo::~CNInfo()
{
        // Delete list of pointers hold
        Array_Free(whereis);
        whereis = NULL;

        // Delete the databuffer information
        if (databuffer != NULL)
                free(databuffer);
}

//------------------------------------------------------------- CNInfoList class
CNInfoList::CNInfoList() : list(NULL)
{
}

CNInfoList::~CNInfoList()
{
        int i;
        CNInfo *ptr;

        for(each_aeli(ptr, list))
                delete ptr;

        Array_Free(list);
        list = NULL;
}

CNInfo *CNInfoList::lookforRegNamedObj(container_id cont, Namedobj* obj, char Type) const
/* Searches for the first CNInfo entry with a Namedobj address which equals
   to obj, if not found, it returns NULL */
{
        int i;
        CNInfo *ptr,
               *toret = NULL;

        for(each_aeli(ptr, list))
                if  (ptr->getNamedobj()==obj
                 and Type == ptr->getTypeOfPointer()
                 and cont == ptr->getForeignContainer())
                {
                        toret = ptr;
                        break;
                }

        return toret;
}

CNInfo *CNInfoList::addPtrToNamedobj(Namedobj* obj, char Type, void *ptr, container_id cont)
/* If an entry with obj Namedobj exists, then this ptr is added to it. It uses
    lookforRefNamedObj(). Otherwise, it returns NULL */
{
        CNInfo *toret = NULL;

        if (cont == 0)
                cont = Conim::Ptr_to_conim(ptr)->cid;

        if ((toret = lookforRegNamedObj(cont, obj, Type)) != NULL)
                toret->addPtr(ptr);

        return toret;
}

CNInfo *CNInfoList::addCNInfo(CNInfo *reg)
{
        Array_Add(list, reg);

        return reg;
}

inline
int CNInfoList::getNumberOfEntries() const
{
        return Array_Size(list);
}

inline
CNInfo *CNInfoList::getEntryIndx(int i) const
/* Returns the i-th element of the list of CNe's, being i a zero-based index */
{
        if (i>=getNumberOfEntries())
                return NULL;

        return Array_Idx(list, i);
}

inline
CNInfo *CNInfoList::operator[](int i) const
/* Uses getEntryIndx() member function in order to return an entry */
{
        return getEntryIndx(i);
}

void CNInfoList::clear()
/* Let the list prepared again for future uses. Free current resources. */
{
        int i;
        CNInfo *ptr;

        for(each_aeli(ptr, list))
                delete ptr;

        Array_Free(list);
        list = NULL;
}

void CNInfoList::deleteRedundantRClassdefs(void)
/* We look for all entries of a given obj. If we have more than one, we delete
   that one reducedclassdef. This class is a friend class of CNInfo */
{
        int       i,
                  j,
                  maxlist = Array_Size(list);
        CNInfo   *ptr,
                 *cn;
        Namedobj *obj;

        for (each_aeli(ptr, list))
        {
                if (ptr->getReducedClassdef() != NULL)
                {
                      obj  = ptr->getNamedobj();
                      j    = i + 1;
                      cn   = list[j];
                      while(j < maxlist)
                      {
                        if  (cn->whereispointing  == obj
                         and cn->ReducedClassdef  != NULL)
                                free(cn->ReducedClassdef);
                         cn   = list[++j];
                      }
                }
        }
}

//==============================================================================
//==============================================================================
//------------------------------------------------------------------ Conim Class

//------------------------------------ Member functions related to access rights
bool Conim::IsWriteable() const
/* Can I write in the pages of this Conim ? */
{
        return getProtection() == READWRITE;
}

inline
bool Conim::getProtection() const
// Returns current access mode to memory pages.
{
        return writeable;
}

// ----------------------------------------------------- delta-related functions
/*
                delta is zero when the container doesn't need
                to be swizzled locally. Many useful functions here ...
*/

inline
void Conim::putDelta(int x)
{
        delta = x;
}

inline
const int &Conim::getDelta(void) const
{
        return delta;
}

inline
void Conim::resetDelta()
{
        putDelta(0);
}

inline
bool Conim::isAlreadySwizzledLocally() const
{
        return getDelta()==0;
}

//------------------------------------------------- Misc. Conim member functions
const ContainerHeader &Conim::getContainerHeader() const
{
        return *header;
}

inline
Conim *Conim::getBeingwritten(void)
{
        return beingwritten;
}

inline
const void* &Conim::getPreferredAddress(void) const
/* Returns the preferred address of a container, in which it is expected to be
   loaded in memory by default */
{
        return header->base_address;
}

inline
Directory* Conim::directory()
{
	return &header->dir;
}

str Conim::Name()
/* Try to construct a useful name for this conim. */
{       static char buf[10];

	/*if (TypeEqual(root.type, direc_typstr)) {
	    direc = (Directory*)root.value;
	    return direc->h->name;
	}
	else {*/
	sprintf(buf, "B%d", cid);
	return buf;
}

inline
Conim* Conim::Ptr_to_conim(void *mem)
/* Map this pointer to a Conim. */
{
	Heap *heap = Ptr_to_heap(mem);
	if (heap == anon_heap)
	    return NULL;
	else if (heap)
            return (Conim*)heap;
        else if (mem >= StdlibLo and mem <= StdlibHi)
            return Stdlib;
        else return NULL;
}


void Conim::DumpConims(void)
{       Conim *conim, *child;
	ConimIterator iter;
	int i;

	for (each_conim) {
	    Pr("\n");
	    Pr("%8s:  B%ld (R%c,%d,%d) %s ",
		    conim->Name(),
		    conim->cid,
		    conim->writeable ? 'W' : ' ',
		    conim->link_count,
		    conim->reference_count,
		    //conim->interface_change,
		    //conim->remake,
		    conim->src ? "(src)" : "     ");
	    if (conim->prerequisites) {
		Pr("-> ");
		for (each_aeli(child, conim->prerequisites))
		    Pr("%s, ", child->cid);
	    }
	}
}

graph_type Conim::LinkGraph(void)
{       Conim *conim, *child;
	ConimIterator iter;
	graph_type G;
	int i;

	G = GraphNew();
	for (each_conim) {
	    GraphAddVertex(G, (void*)conim->cid, conim->Name(),
		    conim->writeable ? 4 : 1);
	    for (each_aeli(child, conim->prerequisites))
		GraphAddEdge(G, (void*)conim->cid, (void*)(child->cid), 15);
	}
	return G;
}

void Conim::UpdateHeaderClassdef()
/* If we created the container before the ContainerHeader classdef	*/
/* existed, but it exists now, then update the type of the header tile.	*/
{
	*(Classdef**)((char*)header - 4) = ContainerHeaderStruct;
}

//--------------------- Mapping container id's to heaps: -----------------------
Conim* Conim::FindConim(container_id cid)
/* Do we have this already in memory? */
{	Conim* conim;
	uint h;

	if (Hash == NULL)
	    return NULL;
	h = (uint)cid % Array_Size(Hash);
	for (conim=Hash[h]; conim; conim=conim->next) {
	    if (conim->cid == cid)
		return conim;
	}
	return NULL;
}


void Conim::MapAddConim()
/* Insert this conim into the hash-table. */
{	uint h;

	if (++NumConims > Array_Size(Hash)/2) {
	    /* Expand the hash-table: */
	    uint newh, NewHashSize;
	    Conim **NewHash=NULL;
	    Conim* conim;

	    NewHashSize = Array_Size(Hash)*2+1;
	    if (NewHashSize < 15)
		NewHashSize = 15;
	    Array_SetSize(NewHash, NewHashSize);
	    if (Hash) {
		for (h=Array_Size(Hash)-1; (int)h >= 0; h--) {
		    while (Hash[h]) {
			conim = Hash[h];
			Hash[h] = conim->next;
			newh = conim->cid % NewHashSize;
			conim->next = NewHash[newh];
			NewHash[newh] = conim;
		    }
		}
		Array_Free(Hash);
	    }
	    Hash = NewHash;
	}

	/* Ok - Hash is not too dense.  Just insert it. */
	h = cid;
	h %= Array_Size(Hash);
	this->next = Hash[h];
	Hash[h] = this;
}


void Conim::MapDelConim()
/* Take this out of the hash-table. */
{	Conim** conimp;
	uint h;

	/* Remove it from the linked list: */
	if (cid > 0) {
	    h = cid % Array_Size(Hash);
	    for (conimp=&Hash[h]; *conimp; conimp=&(*conimp)->next) {
		if (*conimp == this) {
		    *conimp = next;
		    break;
		}
	    }
	    /* Don't shrink Hash because this would confuse DestroyAll(). */
	    NumConims--;
	}
}



//--------------------- Finding pointers: --------------------

extern bool SwizzleWritePhase();
	/* Are we in the 'swizzle pointers of GC'd LGO Image' phase? */

extern bool SwizzleLoadPhase();
	/* Are we in the 'add delta to pointers of LGO image' phase? */

extern str BaseToImage(void* v);

interface PtrProcessor_fn PtrProcessor;



interface Type FindPtrsInTypstr(Type type)
{	int i;

	do {
	    switch (*type++) {
		case tp_reference:
		case tp_volatile:
		case tp_dynarray:
		case tp_pointer:
		case tp_const:	    continue;
		case tp_array:      type += sizeof(int);
				    continue;
	        case tp_ptrmemberfn:
		case tp_function:   for (i=(unsigned char)*type++ & 127; i; i--)
					type = FindPtrsInTypstr(type);
				    assert(*type == tp_terminated);
				    type = FindPtrsInTypstr(++type);
				    return type;
		case tp_enumerated: PtrProcessor((void**)type);
				    return type + 4;
		case tp_class:	    PtrProcessor((void**)type);
				    return type + 4;
		default:            return type;
	    }
	} forever;
}


interface void* FindPtrsInStaticNamedobj(StaticNamedobj *obj)
/* With this function and its brothers, we must find all    */
/* pointers in this _tile_.  That means that any sub-object */
/* that it points to within the tile must have its pointers */
/* found.  For static_storage named objects, this means	    */
/* the value embedded in it. */
{
	if (SwizzleWritePhase()) {
	    /* Process type-info _last_. */

            /* Static storage objects are the tricky ones, because  */
            /* we need to find all ptrs in a tile yet in order to   */
            /* do so requires following an unswizzled ptr. */
            PtrProcessor((void**)&obj->location);
            if (*obj->type == tp_class) {
                Classdef* classdef=*(Classdef**)(obj->type+1);
                if (classdef->VirtualFns)
                    PtrProcessor((void**)(BaseToImage((char*)obj->location - 4)));
                    /* Swizzle the virtual-function pointer. */
            }
            FindPtrs(BaseToImage(obj->location), obj->type);
            PtrProcessor((void**)&obj->type);
	    FindPtrsInTypstr((Type)BaseToImage(obj->type));
	    obj->make = NULL;
	}
	else {
	    /* Process type-info _first_. */
            PtrProcessor((void**)&obj->type);
	    FindPtrsInTypstr(obj->type);

            PtrProcessor((void**)&obj->location);
            if (*obj->type == tp_class) {
                Classdef* classdef=*(Classdef**)(obj->type+1);
                if (classdef->VirtualFns)
                    PtrProcessor((void**)((char*)obj->location - 4));
                    /* Swizzle the virtual-function pointer. */
            }
            FindPtrs(obj->location, obj->type);
	}

	/* Other pointers: */
	PtrProcessor((void**)&obj->name);
	PtrProcessor((void**)&obj->next);
	PtrProcessor((void**)&obj->owner);
        return obj;
}


interface void* FindPtrsInNamedobj(Namedobj *obj)
/* This is for simple namedobj's:  enums, data-members, typedefs. */
{
        /* common: */
        obj->make = NULL;
	if (SwizzleWritePhase()) {
            PtrProcessor((void**)&obj->type);
            FindPtrsInTypstr((Type)BaseToImage(obj->type));
        }
        else {
            PtrProcessor((void**)&obj->type);
            FindPtrsInTypstr(obj->type);
        }
	PtrProcessor((void**)&obj->name);
	PtrProcessor((void**)&obj->next);
	PtrProcessor((void**)&obj->owner);
        return obj;
}


interface void* FindPtrsInMacroNamedobj(MacroNamedobj *obj)
/* This is for simple namedobj's:  enums, data-members, typedefs. */
{
        PtrProcessor((void**)&obj->macro);

        /* common: */
        obj->make = NULL;
	if (SwizzleWritePhase()) {
            PtrProcessor((void**)&obj->type);
            FindPtrsInTypstr((Type)BaseToImage(obj->type));
        }
        else {
            PtrProcessor((void**)&obj->type);
            FindPtrsInTypstr(obj->type);
        }
	PtrProcessor((void**)&obj->name);
	PtrProcessor((void**)&obj->next);
	PtrProcessor((void**)&obj->owner);
        return obj;
}


interface void* FindPtrsInFunctionNamedobj(FunctionNamedobj *obj)
/* This is for simple namedobj's:  enums, data-members, typedefs. */
{
        PtrProcessor((void**)&obj->DefaultParameters);
        PtrProcessor((void**)&obj->Throws);
        if (obj->storage == straight_fn or obj->storage == member_fn)
            PtrProcessor((void**)&obj->u.fn);

        /* common: */
        obj->make = NULL;
	if (SwizzleWritePhase()) {
            PtrProcessor((void**)&obj->type);
            FindPtrsInTypstr((Type)BaseToImage(obj->type));
        }
        else {
            PtrProcessor((void**)&obj->type);
            FindPtrsInTypstr(obj->type);
        }
	PtrProcessor((void**)&obj->name);
	PtrProcessor((void**)&obj->next);
	PtrProcessor((void**)&obj->owner);
        return obj;
}


interface void* FindPtrsInNamedobjDerived(Namedobj *obj)
/* In some cases, we don't have a virtual function pointer. */
/* This happens inside classdef's.  So instead we use       */
/* obj->storage to choose between the different versions of */
/* the FindPtr functions. */
{
        switch (obj->storage) {
            case perprocess_storage:
            case perthread_storage:
            case local_static:
            case static_storage:
                        return FindPtrsInStaticNamedobj((StaticNamedobj*)obj);

            case member_storage:
            case inherit_storage:
            case const_storage:
            case typedef_storage:
                        return FindPtrsInNamedobj(obj);

            case straight_fn:
            case member_fn:
            case oneinstr_fn:
            case virtual_fn:
            case inline_fn:
                        return FindPtrsInFunctionNamedobj((FunctionNamedobj*)obj);

            case macro_storage:
                        return FindPtrsInMacroNamedobj((MacroNamedobj*)obj);

            default:    assert(false);
                        return NULL;
        }
}


interface void* FindPtrsInClassdef(Classdef *classdef)
{	Namedobj* obj;
	void **vf;
	int i;

	/* Maybe we've already processed this one! */
	if (SwizzleLoadPhase() and classdef->myself == classdef)
	    return classdef;

	/* With this function and its brothers, we must find all    */
	/* pointers in this _tile_.  That means that any sub-object */
	/* that it points to within the tile must have its pointers */
	/* found.  For static_storage Namedobj's, this means the    */
	/* value embedded in it. */

	/* Important pointers: */
	PtrProcessor((void**)&classdef->member);	// This _doesn't_ recurse because the
			// members are all in the same tile as the classdef.
	PtrProcessor((void**)&classdef->typedef_obj);
	PtrProcessor((void**)&classdef->myself);
	PtrProcessor((void**)&classdef->ptrfinder);
	/* Don't do this:  PtrProcessor((void**)&classdef->VirtualFns);  because VirtualFns is now an integer. */

	/* Virtual function pointers: */
	for (each_oeli(vf, (void**)(classdef+1)))	// The format of the virtualfn table
	    PtrProcessor(vf);					// matches the format of a dynamic array.

	/* The members: */
	if (SwizzleWritePhase()) {
	    for (obj=(Namedobj*)BaseToImage(classdef->member); obj;
			obj=(Namedobj*)BaseToImage(obj->next)) {
		FindPtrsInNamedobjDerived(obj);
	    }
	}
	else {
	    for (obj=classdef->member; obj; obj=obj->next) {
		FindPtrsInNamedobjDerived(obj);
	    }
	}
        return classdef;
}


interface void* FindPtrsInBitmap(void *p)
/* There are no real pointers here, just some Win32 handles.	*/
/* Since we have no hope of finding the same Win32 objects	*/
/* around when we reload, we simply zero out any such handles	*/
/* when writing out a container, and the applications must	*/
/* create them anew.  In future, we might try to extract the	*/
/* data out of Windows (it might mean getting it out of video-	*/
/* card memory!) and make it persistent. */
{	struct copyofbitmap_node {	// We copy it to avoid a dependency of types.cpp on bitmap.cpp.
	    int cx, cy;
	    void* mDC;
	    void* Bmp;
	} *bm=(struct copyofbitmap_node*)p;

	if (SwizzleWritePhase()) {
	    bm->mDC = NULL;
	    bm->Bmp = NULL;
	}
        return bm+1;
}


interface void FindPtrs(void* p, Type type)
/* This function finds all pointers in this object, and passes	*/
/* them through 'fn'.  It does not recurse into other memory	*/
/* blocks (unless 'fn' recurses into other memory blocks).	*/
/* 'fn' is recursive in the case of the garbage collection	*/
/* phase, but otherwise not. */
{	Classdef* classdef;
        int dimension, size;

	if (type == NULL or p == NULL)
	    return;
	switch (*type++) {
	    case tp_pointer:
            case tp_ptrmemberfn:
            case tp_reference:
		    PtrProcessor((void**)p);
		    break;

	    case tp_array:
                    size = 1;
                    do {
                        GetDimension(dimension, type);
                        size *= dimension;
                        while (*type == tp_const)
                            type++;
                        if (*type == tp_array)
                            type++;
                        else break;
                    } forever;
                    if (*type == tp_pointer) {
                        while (size-- > 0) {
                            PtrProcessor(((void**&)p)++);
                        }
                    }
                    else if (*type == tp_class) {
                        classdef = *(Classdef**)(++type);
                        if (classdef->ptrfinder) {
                            while (size-- > 0) {
                                classdef->ptrfinder(((void**&)p)++);
                            }
                        }
                    }
		    break;

	    case tp_function:
		    FindPtrsInFuncblock((funcblock_type)p, PtrProcessor);
		    break;

	    case tp_class:
		    GetPtr(classdef, type);
                    if (classdef->ptrfinder)
		        classdef->ptrfinder(p);
		    break;

	    default:// It must be a single datum.
		    break;
	}
}


interface void FindPtrsInTile(tile_type tile)
/* Find all pointers in this heap tile. */
/* That means in the header and in the  */
/* contents. */
{	Classdef* classdef;

        if (*(uint*)tile & 3)
	    return;	    // It's either free (bit0), or untyped (bit1).
	if (SwizzleWritePhase()) {
	    /*** In write phase, we need to swizzle first and span 3 address spaces. ***/
	    Classdef* orig_classdef = *(Classdef**)tile;
	    PtrProcessor((void**)tile);				// We need to swizzle first.
	    classdef = (Classdef*)BaseToImage(*(Classdef**)tile);
	    if ((char*)classdef < tile or (char*)classdef > tile + orig_classdef->RoundupSize) {
		/* It's a normal tile */
                assert(orig_classdef->signature == HELLO_BABE);
		orig_classdef->ptrfinder(tile + 4);	// Process the contents
	    }
	    else {
		/* It's a non-class-instance tile (or a tile which is larger than its class): */
		Type type = (uchar*)classdef + 4;	// equals the private Heap::TileTypestr(tile);
		FindPtrsInTypstr(type);		        // Process the typestr
		FindPtrs(tile + 4, type);		// Process the contents
	    }
            #define PTRFINDER_OFFSET            0x20
            assert((str)&BitmapStruct->ptrfinder - (str)BitmapStruct == PTRFINDER_OFFSET);
	}
	else {
	    /*** Load or Reachability phase: ***/
	    PtrProcessor((void**)tile);				// Process the header first of all

	    classdef = *(Classdef**)tile;
	    if ((char*)classdef < tile or (char*)classdef > tile + classdef->RoundupSize) {
		/* It's a normal tile */
                assert(classdef->signature == HELLO_BABE);
		classdef->ptrfinder(tile + 4);	        // Process the contents
	    }
	    else {
		/* It's a non-class-instance tile (or a tile which is larger than its class): */
		Type type = (uchar*)classdef + 4;	// equals the private Heap::TileTypestr(tile);
		FindPtrsInTypstr(type);		        // Process the typestr
		FindPtrs(tile + 4, type);		// Process the contents
	    }
	}
}





//------------------------------------------------------------- Swizzling Conims
static void SwizzleFn(void** p)
{
        Conim *thisone = Ptr_to_conim((char*)p);
	if (*p != NULL)
	    *((char**)p) += thisone->getDelta();
}


void Conim::localSwizzle()
/*
        Add 'delta' to all pointers in this heap.
        It does nothing if we try to do swizzling in a system
        container, such as 'stdlib'.
*/
{
        if (delta != 0 and cid > 0) {
            // Only if it is not a system container
            HeapIterator iter(this);
            tile_type tile;
            PtrProcessor = SwizzleFn;
            while ((tile=iter++) != NULL) {
                FindPtrsInTile(tile);
            }
        }

        resetDelta();    // Now the container is already swizzled
}



//------------------------- Loading Zone; Creating containers, and loading them

#define REACHABLE	128

Conim::Conim(char* mem, uint size, container_id _cid, bool _writeable,
	    void* filehandle, void* mappinghandle)
	: Heap(mem, size, _writeable), delta(0)
{
	writeable       = _writeable;
	reference_count = 0;
	link_count      = 0;
	prerequisites   = NULL;
	src             = NULL;
        NOHash          = NULL;
	cid             = _cid;
	MappingMem      = mappinghandle ? mem : NULL;
	FileHandle      = filehandle;
	MappingHandle   = mappinghandle;
	MapAddConim();
}


Conim::Conim(uint size, container_id _cid)
	: Heap(size, _cid < 0 ? (void*)0xb10000 : NULL), delta(0)
{
	writeable       = yes;
	reference_count = 0;
	link_count      = 0;
	prerequisites   = NULL;
	src             = NULL;
	cid             = _cid;
	MappingMem      = NULL;
	FileHandle      = NULL;
	MappingHandle   = NULL;
        NOHash          = NULL;
	MapAddConim();
}


//--------------------------- PhysicallyLoad()
Conim* Conim::PhysicallyLoad(container_id cid, bool writeable)
/* This will read the data and do basic swizzling, but it */
/* won't do C-N swizzling (because that would lead to     */
/* an infinite cycle).   Note that if 'writeable' false,  */
/* we might nevertheless return a container in writeable  */
/* mode: this is caused by e.g. relocation and is left    */
/* like this in case we need to do C-N swizzling.         */
{       int ContainerSize, size4K;
	unsigned long read_size;
        unsigned long bytesread;
        ContainerHeader header;
        char path[PATHLENGTH];
	char *mem, *cntable;
	Conim *conim;
	HANDLE file;
	int result;

	// Open the file:
	if (not CidToPath(cid, path, sizeof(path)))
	    return NULL;	// CidToPath has already set b_errno
	file = CreateFile(path,
		    GENERIC_READ,
		    FILE_SHARE_READ,
		    NULL,
		    OPEN_EXISTING,
		    FILE_ATTRIBUTE_NORMAL,
		    NULL);
	if (file == NULL or file == INVALID_HANDLE_VALUE) {
	    result = GetLastError();
	    if (result == ERROR_FILE_NOT_FOUND or result == ERROR_PATH_NOT_FOUND)
		b_errno = E_NO_SUCH_CONTAINER;
	    else if (result == ERROR_SHARING_VIOLATION or result == ERROR_LOCK_VIOLATION)
		b_errno = E_LOCK;
	    else b_errno = E_IOFAIL;
	    return NULL;
	}

        // Read the header of the container to get info about it
        ContainerSize = GetFileSize(file, &read_size);
        SetFilePointer(file, 4, NULL, FILE_BEGIN);       // pass the tile header
        if (!ReadFile(file,&header, sizeof(ContainerHeader), &bytesread, NULL)) {
            CloseHandle(file);
            b_errno = E_IOFAIL;
            return NULL;
        }
        SetFilePointer(file, 0, NULL, FILE_BEGIN);

	// Try to locate enough space in its preferred address.  If that
        // doesn't work, try any other address.
        if (writeable)
            size4K = ((header.lin_start-1)|(PAGE_SIZE-1))+1; // round up to 64K
        else size4K = ((header.lin_start-1)|4095)+1;   // round up to 4K
	mem = (char*)VirtualAlloc(header.base_address, size4K,
                                MEM_RESERVE|MEM_COMMIT, PAGE_READWRITE);

        // If the preferred address didn't work, then
        // we should try any other one
        if (mem == NULL) {
            mem = (char*)VirtualAlloc(NULL, size4K,
                                MEM_RESERVE|MEM_COMMIT, PAGE_READWRITE);
            if (mem == NULL) {
                b_errno = E_NOMEM;
                CloseHandle(file);
                return NULL;
            }
        }

        // Now we can copy the container in memory
	result = ReadFile(file, mem, header.lin_start, &read_size, NULL);
	if (not result) {
	    result = GetLastError();
	    b_errno = E_IOFAIL;
	    CloseHandle(file);
	    VirtualFree(mem, 0, MEM_RELEASE);
	    return NULL;
	}

        // Copy the CN Swizzling table in memory
        cntable = (char*)::malloc(anon_heap, ContainerSize - header.lin_start);
        if (cntable != NULL)
            result = ReadFile(file, cntable, ContainerSize - header.lin_start,
                                                              &read_size, NULL);
	if (not result or cntable == NULL) {
	    result = GetLastError();
	    b_errno = E_IOFAIL;
	    CloseHandle(file);
	    VirtualFree(mem, 0, MEM_RELEASE);
            ::free(anon_heap, cntable);
	    return NULL;
	}

        // Prepare the Conim object
	Base = Image = mem;
        compaction_offset = header.lin_start;
	((ContainerHeader*)(mem+4))->base_address = mem;

	/* Create the Conim object: */
	conim = new Conim(mem, header.lin_start, cid, writeable, file, NULL);

        /* Set delta if needed */
        conim->putDelta(mem - ((char*)header.base_address));

	/* Find the root object: */
	conim->header = (ContainerHeader*)((char*)mem + 4);

	/* Leave the FileHandle open in order to maintain a lock on the file. */
	conim->FileHandle = file;

      // CN Entries are loaded in memory too
      /*
        Summary of important tasks and ideas:
        * Container is already in memory, with its CN table in another address.
        * Read the list of foreign pointers for this containers and add them
          to the general list of foreign pointers to process.
        * Pointers must be prepared to be consistent with the base_address
          of this container.
        * Foreign containers must be linked.
        * All loaded containers must be stored in the list of containers needing
          of local_swizzle
      */

        CNInfo         *reg;
        Conim          *foreign;
        char           *wr;
        Namedobj       *obj;

        // Get ready
        Array_InsP(ContainersNeedingOfLocalSwizzle, conim);
        wr  = cntable;
        obj = *((Namedobj **) wr);

        // Reading entries
        while (obj != TABLE_FINISH_MARK and
                        wr < (cntable + (ContainerSize-header.lin_start))) {
            // Read entry
            reg = CNInfo::loadCNInfo(wr);
            reg->rebasePtrs(((int) conim->header->base_address));
            Conim::prereqinfo.addCNInfo(reg);

            // Link container
            foreign = conim->OpenPrerequisiteContainer(reg->getForeignContainer());
            if (foreign == NULL)
                reg->markasInvalidContainer();
            else conim->Link(foreign);

            // Next entry
            obj = *((Namedobj**)wr);
        }

        // We're done !
        if (not writeable) {
            DWORD OldProtect;
            VirtualProtect(mem, size4K, PAGE_READONLY, &OldProtect);
        }
        ::free(anon_heap, cntable);
	return conim;
}



// ------------------------------------------------------- CN Swizzling sub-zone
//-------------------------------------- fns part of processForeignPointerList()
static inline
void *applyDelta(void *x, int delta)
/* We need to get the swizzled pointer without modifying it */
{
        assert(Ptr_to_conim((((char*)x) + delta))!=NULL);
        return ((void*)(((char*)x) + delta));
}


typedef Namedobj *namedobj_type;


static
Namedobj* findNameMember(namedobj_type & clobj, str thename, int delta)
/*
        IMPORTANT ** This function is only used when the container
        in which we are looking
        for a named_object is locally unswizzled.
        *****************************************
        In this function, we parse the name of the named object
        we are looking for. We have the Namedobj of the classdef, obj, and
        we have to look for 'name' inside its members.
        This module therefore expects to use '::' in order to separate classes
        and namespaces in an unique name.
        Also, the overload version must be separated by a point.
*/
{
        Classdef* cldef = TypeToClassdef((unsigned char *) applyDelta(clobj->type, delta));
        Namedobj* obj   = (Namedobj*) applyDelta(cldef->member, delta);
        char *clname,
             *ovpos = NULL;
        int noverload       = 0;
        char *name = ::strdup(thename);

        if (name == NULL)
        {
                ErrorRun("Out of memory.-");
                return NULL;
        }

        // Is this the end of the a::b::c::d ... chain ?
        if ((clname = strchr((char *)name, ':')) != NULL)
                *clname = 0; // No, but let's do it as it were.
        else {
                // the final objective: the target.
                // extract the overloaded version if it is available
                if ((clname = strchr((char *)name, '.')) != NULL)
                {
                        *ovpos = 0;
                        noverload = atoi(ovpos + 1);
                }
        }



        // Now, let's find it.
        do {
                if (streq((char*)applyDelta(obj->name, delta), (char *)name)
                 && noverload == obj->overload_version)
                {
                        if (clname != NULL)
                                obj = findNameMember(obj, clname + 2, delta);
                        break;
                }

                if (obj->next != NULL)
                        obj = (Namedobj*) applyDelta(obj->next, delta);
                else obj = NULL;

        } while(obj != NULL);

        ::free(name);
        return obj;
}


Namedobj* Conim::findNamedObjByName(str thename)
/*
        This function has to deal with the possibility of the tiles of the
        foreign container not having been localSwizzle(). If they are, there is
        no problem, we just search. IF they aren't, then we have to run all over
        the hash of names. This is a problem, as we'll have to parse the name
        of the Namedobj looked for here, in the findNameMember() function.
*/
{
        Namedobj* toret   = NULL;
        char *name = ::strdup(thename);

        if (name == NULL)
        {
                ErrorRun("Out of memory.-");
                return NULL;
        }

        if (not isAlreadySwizzledLocally())
        {
                char *clname;
                Namedobj* obj;
                Namedobj **h = (Namedobj **)
                                   applyDelta(directory()->getHash(), getDelta());

                /* If it is a::b::c name, then we will have to find it within
                   a class
                */
                if ((clname = strchr((char*)name, ':')) != NULL)
                      *clname = 0;

                /* Start the search. If it is an a::b::c name, then we look
                   firstly for 'a'
                */
                for (unsigned int n=0; n<directory()->getHashLen(); ++n)
                {
                  if (h[n]!=NULL)
                  {
                     obj = (Namedobj*) applyDelta(h[n], getDelta());
                     do
                     {
                        // Search for the given name
                        if (streq((char *)applyDelta(obj->name, getDelta()),name))
                        {
                                toret = obj;

                                if (clname != NULL)
                                {
                                        toret = findNameMember(obj, clname + 2, getDelta());
                                }

                                if (clname != NULL)
                                        *clname = ':';
                                goto END;
                        }

                        // Get next tile
                        if (obj->next!=NULL)
                                obj = (Namedobj*) applyDelta(obj->next, getDelta());
                        else    obj = NULL;
                     }
                     while(obj != NULL);
                  }
                }


        }
        else toret = (directory()->ObjFromUname(name));
                        // Swizzled, so, operative hash


        // we're done !!
        END:
        ::free(name);
        if (toret==NULL)
                assert(false);
        return toret;
}

static
void ResolvePtr(const CNInfo &rec, Namedobj* where)
/*
        This function works closely -it's a private function of- with
        processForeignPointerList(). This is the place where the
        different types of pointers have to be resolved in their memory address
        to the correct Namedobj's value.
        This represents the container which will be modified (that will have
        its pointer modified).
        The purpose of this is to check if we really need to change the "thisone"
        container. If we can avoid every write, and the container is in READONLY
        mode, then the READONLY mode is preserved. This can lead to avoid the
        save process, later.
        We are subtracting delta to all pointers in *thisone*, because later,
        we are going to do localSwizzle(). Excepting with relative ones, which
        are sligthly different.
*/
{
        Conim  *foreign = Ptr_to_conim(where);
        Conim  *thisone;
        void ***ptr;
        char   *target;

        // Absolute pointers
        // many situations
        if (rec.isAbsolute())
        {
                for(int n=0; n<rec.getPtrListSize(); ++n)
                {
                  ptr     = (void ***) rec.getPtrIndx(n);
                  thisone = Ptr_to_conim(ptr);
                  assert(foreign != NULL && thisone != NULL);


                  target = (((char *)((StaticNamedobj*)where)->location) + foreign->getDelta());
                  if ((*ptr != ((void **) target))
                   or (thisone->getDelta() != 0))
                  {
                        thisone->protectionDowngrade();
                        *ptr = (void **)(target - thisone->getDelta());
                  }
                }
        }
        else
        if (rec.isRelative())
        // Relative pointers
        // only for function JMP's
        {
                for(int n=0; n<rec.getPtrListSize(); ++n)
                {
                  ptr     = (void ***) rec.getPtrIndx(n);
                  thisone = Ptr_to_conim(ptr);
                  assert(foreign != NULL && thisone != NULL);



                  target = ((char *)((int)((FunctionNamedobj*)where)->u.fn) + foreign->getDelta());

                  if (*((int *) ptr) != (((int)target)-((int) ptr) - 4))
                  {
                        if (thisone->getProtection()==READONLY)
                                thisone->protectionDowngrade();
                        *((int *) ptr) = ((int)target) - ((int) ptr) - 4;
                  }
                }
        }
        else {
            // TypeClass pointers for objects with their class definitions
            // in other containers
            for (int n=0; n<rec.getPtrListSize(); n++) {
                ptr = (void ***) rec.getPtrIndx(n);
                thisone = Ptr_to_conim(ptr);
                assert(foreign != NULL && thisone != NULL);
                target = *((char **)(where->type + 1)) + foreign->getDelta();
                if (*ptr != ((void **)target) or thisone->getDelta() != 0) {
                    if (thisone->getProtection() == READONLY)
                        thisone->protectionDowngrade();
                    *ptr = (void**)(target - thisone->getDelta());
                }
            }
        }
}


//-------------------------------------------------- processForeignPointerList()
bool Conim::needsOfCnSwizzling()
{       Namedobj* where;
        Conim *foreign;
        CNInfo *rec;

        for (int n=0; n < prereqinfo.getNumberOfEntries(); n++) {
            rec = prereqinfo[n];
            foreign = FindConim(rec->getForeignContainer());
            if (not foreign)
                return yes;
            where = foreign->findNamedObjByName(rec->getName());
            if (not where or where != rec->getNamedobj())
                return yes;
        }
        return no;
}



//-------------------------------------------------- processForeignPointerList()
void Conim::processForeignPointerList()
/*
     This function fixes all the pointers in the 'prereqinfo' pointer vector.
     This vector is filled in the physicallyload() function.
     The CN Swizzling table of each container is left ready here, although
     this function is executed only once all dependant containers are in memory.
     All of this happens in the load phase.
     Please note that this is a static function. It doesn't pertain to any
     concrete container, because the foreign pointer list (prereqinfo)
     contains fixings in this container and maybe in any other container of
     the prerequisite ones.
*/
{
        CNInfo        *rec;
        Conim         *foreign;
        Namedobj* where;


/*        if (getDelta()==0
         && needsOfCnSwizzling())
        {
*/
          for(int n=0;n<prereqinfo.getNumberOfEntries(); ++n)
          {
                // Process this entry
                rec = prereqinfo[n];

                // Get the foreign container
		if ((foreign=FindConim(rec->getForeignContainer())) != NULL)
                {
                        where   = foreign->findNamedObjByName(rec->getName());
			if (where != NULL)
                        {
                                if (!rec->isCorrupted())
                                        // Resolve the pointer
                                        ResolvePtr(*rec, where);
                                else ErrorWarning("Skipped corrupted C-N Swizzling reference");
                        }
                        else {
                                char buf[PATHLENGTH];
                                strcpy(buf, "Skipped reference to a non-existing public object: ");
                                strcat(buf, rec->getName());
                                ErrorWarning(buf);
                                rec->putPtrsNULL();
                                assert(false);
                        }
                }
                else
                {
                        rec->putPtrsNULL();
                        ErrorWarning("Skipped reference to an unexisting container");
                        assert(false);
                }

          }
/*
        }
*/
}

// ---------------------------------------------------- Container Links sub-zone
// ------------------------------------- Garbage Collection of Linked Containers
void Conim::FindReachable()
/* Recursively generate the link_counts. */
{       Conim *child;
	int i;

	/* Have we already visited this? */
	if (flags & REACHABLE)
	    return;
	else flags |= REACHABLE;

	/* Increment all its childrens' link counts and recurse: */
	for (each_aeli(child, prerequisites)) {
	    child->link_count++;
	    child->FindReachable();
	}
}


void Conim::LinkGC()		// A static function
/* Garbage collection of linked conim's. */
/* To do the garbage collection, we regenerate all the 'link_counts'	*/
/* based on what containers have OpenContainer()-style reference-counts,*/
/* and purge any that are now orphans. */
{     Conim *conim, **Orphans;
	ConimIterator iter;
	uint i;

	/* Reset the link_counts: */
	for (each_conim) {
	    conim->link_count = 0;
	    conim->flags &= ~REACHABLE;
	}

	/* Regenerate the link counts: */
	for (each_conim)
	    if (conim->reference_count)
		conim->FindReachable();


	/* Purge all the new orphans: */
	Orphans = NULL;
        for (each_conim) {
            if (not (conim->flags & REACHABLE))
                Array_Add(Orphans, conim);
        }
        for (each_aeli(conim, Orphans)) {
            Array_Free(conim->prerequisites);
            // These links have not been counted in the 'link_count's,
            // so we want to avoid the normal process of decrementing
            // a container's prerequisites' link_counts on destruction.
            delete conim;
        }
	LinkGCRequired = no;
}


//------------------------------------------------------- Create and Check Links
void Conim::Link(Conim *B)
/* Create a link from A to B, saying that A depends on B. */
{
	if (B and not Array_HasP(prerequisites, B)) {
            B->link_count++;
            Array_Add(prerequisites, B);
        }
}


bool Conim::LinksTo(Conim *B)
/* Does this Conim have B as a prerequisite? */
{
	return Array_HasP(prerequisites, B);
}


//------------------------------------------------------------ CreateContainer()
Conim* Conim::CreateContainer(container_id parent)
/* Create a new container and conim.  The new container is	*/
/* created with a reference count of 1, ie. you need a matching	*/
/* 'CloseContainter()' call to close it.  Return the conim,	*/
/* whose fields include the directory and the container_id.	*/
{	char path[PATHLENGTH];
	unsigned long written;
	container_id cid;
        Conim *conim;

	/*** Allocate a new container_id: ***/
	if (parent < 0) {
	    cid = parent;
            // This is to create the Stdlib Conim which exists only in memory.
        }
	else {
	    cid = NewContainerId(parent);
	    if (cid == 0)
		return NULL;
	}


	/*** Allocate the memory for it: ***/
        try {
            conim = new Conim(65536, cid);
        } catch (int) {
            TfcMessage("Stdlib", 'x', "Couldn't create Stdlib heap at "
                "required address 0xb10000");
            exit(0);
        }
	conim->cid = cid;
	if (ContainerHeaderStruct == NULL) {
	    static Classdef tmp_classdef;
	    ContainerHeaderStruct = &tmp_classdef;
	    ContainerHeaderStruct->RoundupSize = Heap::RoundUp(sizeof(ContainerHeader)) + 4;
	    ContainerHeaderStruct->signature = HELLO_BABE;
	}

        /* Prepare the header of the container */
        // Create ContainerHeader's tile
	conim->header = (ContainerHeader*)conim->New(ContainerHeaderStruct);
	clearS(*conim->header);
        // Assign init values
        conim->header->container_signature = BARBADOS_SIGNATURE;
	conim->header->dir.parent_cid = parent;
        conim->header->lin_start      = 0;

        // Set timestamp. If we are working with the Stdlib container,
        // we need a very old timestamp
        conim->header->timestamp      = (cid<0) ? 0 : time(NULL);

        // Set base_address. Very important !!
        conim->header->base_address   = conim->SuggestedBaseAddress();

        // Prepare the reference count
	conim->reference_count        = 1;

        /* Is this the StdLib container ? Then we're done ! */
	if (conim->cid <= 0)
	    return conim;


	/*** Create an empty file and keep it open: ***/
	if (not CidToPath(cid, path, sizeof(path))) {
	    delete conim;
	    return NULL;	// CidToPath has already set b_errno
	}
	conim->FileHandle = CreateFile(path,
		    GENERIC_READ|GENERIC_WRITE,
		    FILE_SHARE_READ,
		    NULL,
		    CREATE_ALWAYS,
		    FILE_ATTRIBUTE_NORMAL,
		    NULL);
	if (conim->FileHandle == INVALID_HANDLE_VALUE) {
	    conim->FileHandle = NULL;
	    int result = GetLastError();
	    if (result == ERROR_PATH_NOT_FOUND)
		b_errno = E_CORRUPT;
	    else if (result == ERROR_SHARING_VIOLATION or result == ERROR_LOCK_VIOLATION)
		b_errno = E_LOCK;
	    else b_errno = E_IOFAIL;
	    delete conim;
	    return NULL;
	}

        /* Write the header to the associated file */
        WriteFile(conim->FileHandle, conim->header, sizeof(ContainerHeader),
                                      &written, NULL);

        /* Go to the right position */
        SetFilePointer(conim->FileHandle,0, 0, FILE_BEGIN);

	return conim;
}


//-------------------------------------------------------------- OpenContainer()
Conim* Conim::OpenContainer(container_id cid, bool write_permission)
/* Open a pre-existing container into memory */
{	Conim *conim, *pqconim;

        // Perhaps it's already in memory
	conim = FindConim(cid);
        if (conim == NULL) {
            // Prepare the information entries
            prereqinfo.clear();
            ContainersNeedingOfLocalSwizzle = NULL;

            // load it in memory and do basic swizzling
	    conim = PhysicallyLoad(cid, write_permission);

            // is all right ?
	    if (conim == NULL)
               return NULL;

            // fix all pending CN - Swizzling entries
            processForeignPointerList();

            // Now please swizzle all containers we have loaded during this process
            for (int each_aeli(pqconim, ContainersNeedingOfLocalSwizzle)) {
                /* Swizzle it: */
                pqconim->localSwizzle();

                /* Is it healthy or sick ? */
                pqconim->Assert();

                /* Convert it to READONLY if it is a prerequisite */
                if (pqconim != conim)
                    pqconim->SetPagePermissions(READONLY);
            }

            // Cleaning structures for CN-Swizzling which will no longer be used
            prereqinfo.clear();
            Array_Free(ContainersNeedingOfLocalSwizzle);
            ContainersNeedingOfLocalSwizzle = NULL;
	}
        else {
            // Handle this request
            if (write_permission==READONLY)
                conim->protectionUpgrade();
            else conim->protectionDowngrade();
        }

        // Anyway, increment the reference count for this container
        conim->reference_count++;

	return conim;
}



//-------------------------------------------------- OpenPrerequisiteContainer()
inline
Conim *Conim::OpenPrerequisiteContainer(container_id cid)
{
        Conim *toret;

        if ((toret=FindConim(cid)) == NULL)
        {
                toret = PhysicallyLoad(cid, READONLY);
                Array_InsP(ContainersNeedingOfLocalSwizzle, toret);
        }

	  return toret;
}


//------------------------------------------------------- Memory access sub-zone
void Conim::SetPagePermissions(bool can_write)
/*
        This function is private.
        Make the whole conim unwriteable (or writeable again).
        This function is only called by PhysicallyLoad and other functions
        called by OpenContainer.
        The rest of access requests *must* be done through protectionUpgrade()
        and protectionDowngrade()
*/
{
	if (can_write != writeable)
            PageProtection(writeable = can_write);
}


void Conim::protectionUpgrade()
/*
        Upgrade protection means to change access of pages to READONLY mode.
        This function only applies when the container is already in memory,
        totally loaded.
        Changing access to READONLY mode is only possible in one situation:
        when the container is in READWRITE mode and reference_count is 0, which
        means this container is only linked. In this case, the call must come
        frome Conim::Close(), which should have saved the container before.
*/
{
        if (writeable == READWRITE and reference_count == 0)
            SetPagePermissions(READONLY);
}


void Conim::protectionDowngrade()
/*
        We are requesting this container to be READWRITE. This request is
        accomplished immediately, given the container is in READONLY mode.
*/
{
        SetPagePermissions(READWRITE);
}



//----------------- Saving Zone: Saving Containers and closing, destroying them.
Conim::~Conim()
{       Conim *B;
        uint i;

        // Save the container
	if (writeable == READWRITE and cid > 0) {
	    PhysicallySave();
	    if (b_errno) {
		char buf[PATHLENGTH];
		sprintf(buf, "Can't save B%d because: %s",
			    cid, BerrnoToString());
		TfcMessage("Barbados", '!', buf);
	    }
	}

        /* Remove all the links from this container to others.   */
        /* However, don't call the destructor for any of these   */
        /* other containers just yet, for fear of getting caught */
        /* in a cycle. Instead, flag the need for GC. Remember,  */
        /* if we're in here then it means we're in the middle of */
        /* the destructor for a Conim. */
        for (each_aeli(B, prerequisites)) {
            assert(B->link_count > 0);
            if (--B->link_count <= 0 and B->reference_count == 0)
            	LinkGCRequired = yes;
        }
        Array_Free(prerequisites);


        // Clean all the remaining trash ...
	if (MappingMem)
	    UnmapViewOfFile(MappingMem);
	if (MappingHandle)
	    CloseHandle(MappingHandle);
	if (FileHandle)
	    CloseHandle(FileHandle);

	MapDelConim();
}


void Conim::Close()
/* Decrease this Conim's reference count. */
{
	if (reference_count <= 0) {
	    b_errno = E_NOTOPEN;
	    return;
	}
	reference_count--;
	if (reference_count + link_count == 0)
	    delete this;
	else if (reference_count == 0) {
            if (writeable==READWRITE)
            {
                    ForceSave();            // Save container
                    protectionUpgrade();    // Because now it's only linked
            }
            LinkGCRequired = yes;
        }

	if (LinkGCRequired)
	    LinkGC();
}


void Conim::DestroyAll()
/*
        This is a static function. Destroy all remaining conims.
        Firstly, we run all over the list of containers in memory, saving them.
        Secondly, we delete them from memory.
        After the first step, we mark them as READONLY in order to avoid a
        second saving.
*/
{	ConimIterator iter;
	Conim* conim;

	for (each_conim) {
	    if (conim->writeable and conim->cid > 0) {
		conim->PhysicallySave();
                conim->writeable = READONLY;
		if (b_errno) {
		    char buf[PATHLENGTH];
		    sprintf(buf, "Can't save B%d because: %s",
			    conim->cid, BerrnoToString());
		    TfcMessage("Barbados", '!', buf);
		}
	    }
	}
	for (each_conim)
	    delete conim;
	FreeRes();
}


// ------------------------------------------------------------- Saving sub-zone
struct Saver_node {
	tile_type tile;
        union {
	    int offset;         // <- If we've processed it in the Reachability phase
            Saver_node *next;   // <- If we have yet to process it.
        };
        // There are 3 states a Saver can be in:  
        //      offset=-1   means unreached (yet)
        //      offset      means reached and processed
        //      next        means reached but unprocessed
};


static Saver_node* Savers, *current_saver, *NewReached_root;
static uint S_idx;


static Saver_node* PointerToSaver(void* ptr)
/* Map this ptr to its Saver_node. If it points inside */
/* a tile, rather than at the start of the tile, then  */
/* return the saver of the tile it points inside. */
{	tile_type tile;
        uint a, b, c;

	/* We use a binary search algorithm. */
	if (S_idx == 0)
	    return NULL;
	if (ptr < Savers[0].tile)
	    return NULL;
	a = 0;
	c = S_idx-1;
	while (a < c) {
	    b = (a + c + 1) / 2;
	    if (ptr < Savers[b].tile)
		c = b - 1;
	    else if (ptr > Savers[b].tile)
		a = b;
	    else
		return &Savers[b];
	}
	tile = Savers[a].tile;
	if (ptr == tile + 4)
	    return &Savers[a];	    // A shortcut.
	if ((char*)ptr > tile + Heap::AllocdTileToSize(tile))
	    return NULL;
	else return &Savers[a];
}


interface str BaseToImage(void* v)
/* A swizzling function.  It takes a base pointer, i.e. a pointer */
/* in the preferred-address address space, and converts it to an  */
/* image pointer, i.e. a pointer in the temporary contiguous saver*/
/* heap. */
{       
	if (v == NULL)
	    return NULL;
	if (v < Base or v >= Base + compaction_offset)
	    return (str)v;
	else return (str)v + Conim::getBeingwritten()->getDelta();
}


static void Reached(void** ptrp)
// Map this pointer to its tile and add that tile to NewReached_root if needed.
{	Saver_node* saver;

	saver = PointerToSaver(*ptrp);
        if (saver and saver->offset == -1) {
	    saver->next = NewReached_root;  // Thanks to the union, 
            // this also marks it as saver->offset != -1
            NewReached_root = saver;
        }
}





//------------------------------------------------------ findNamedObjByPointer()
inline static
bool isThisNamedObj(Namedobj* obj, void *ptr)
{
        if (objLocation == ptr)
            return yes;
        if (obj->storage == typedef_storage and NameIsOriginalClass(obj)
                 and NameClassdefFromObject(obj) == ptr)
            return yes;
        return no;
}


Namedobj* Conim::findNamedObjByPointer(void *ptr)
{       Namedobj* obj, *res = NULL;
        Classdef* classdef;
        Directory *direc;
        uint h;

        direc = this->directory();
        if (NOHash == NULL)
            NOHash = new NamedObjHash;
        else {
            res = NOHash->getNamedObj(ptr);
            if (res)
                return res;
        }
        for (each_dir_obj(direc)) {
            // Add this Namedobj to the hash
            NOHash->putNamedObj(obj);

            // Is this object the one we are looking for ?
            if (isThisNamedObj(obj, ptr))
                return obj;

            // But perhaps it is a class and then we have to inspect
            // its member functions
            classdef = NameClassdefFromObject(obj);
            if (classdef) {
                for (Namedobj* member=classdef->member; member; member=member->next) {
                    // Follow the members chain -- only member functions
                    // and member classes
                    if (*member->type == tp_function or
                            (member->storage == typedef_storage and
                            NameIsOriginalClass(member))) {
                        // tco to jose: we need to look for member classes too.
                        // For example, procbarb needs to CN-swizzle "list::cont".
                        // Add this Namedobj to the hash
                        NOHash->putNamedObj(member);

                        // Is this object the one we are looking for ?
                        if (isThisNamedObj(member, ptr))
                            return member;
                    }
                }
            }
        }

        // Not found:
        return NULL;
}


/*tco to jose: We need to improve on this algorithm.  The benefit of the
hash-table is lost because every time you don't find a pointer, you do
a full linear search.  You don't find a pointer if (a) the user has
given you a bad pointer, or more importantly (b) if the pointer was
never put into the hash-table.  Case (b) happens whenever you search
for a pointer, find it early on, so only half the objects have been
put into the hash-table.
        Also, member classes of member classes should be inserted
and static data members too.
        Also, you never put classdef pointers into the hash table.
I think you should have an extra parameter to 'putNamedObj()'.
        How about this: */

#if 0
void NamedObjHash::putNamedObjRecursively(Namedobj* obj)
{       Classdef* classdef;

        if (obj->storage == member_storage or obj->storage == const_storage)
            return;
        if (obj->storage != typedef_storage) {
            if (obj->u.location)
                putNamedObj(obj->u.location, obj);
        }
        else {
            /* If it's a class we also need to insert its members: */
            classdef = NameClassdefFromObject(obj);
            if (not classdef)
                return;
            if (not NameIsOriginalClass(obj))
                return;
            putNamedObj(classdef, obj);

            /* Insert its members: */
            for (Namedobj* obj=classdef->member; obj; obj=obj->next)
                putNamedObjRecursively(obj);
        }
}


static Namedobj* dot_obj;

Namedobj* Conim::findNamedObjByPointer(void *ptr)
{       Namedobj* obj,tmp;
        directory *dir;

        /* Do we have NOHash? */
        if (NOHash==NULL) {
            NOHash = new NamedObjHash;
            goto ADD_DIR;
        }

        /* Do we already know what the pointer maps to? */
        obj = NOHash->getNamedObj(ptr);
        if (obj)
            return obj;

        /* Has 'dir' been put into NOHash yet? */
        dir = directory();
        tmp = NOHash->getNamedObj(ptr);
        if (tmp)
            return NULL;

        /* Put dir into NOHash: */
        for (each_dir_obj(direc))
            NOHash->putNamedObjRecursively(obj);

        /* We need to remember for future reference that we've already */
        /* added this container to NOHash: */
        if (dot_obj == NULL)
            NameDeclare(stdlib_dir, ".", direc_typstr, static_storage, 0);
        NOHash->putNamedObj(dot_obj);   // Following the UNIX convention,
        // we'll use '.' to denote the directory, and therefore as a marker
        // to indicate whether we've inserted this directory yet or not.

        /* Can we find it now? */
        return NOHash->getNamedObj(ptr);
}
#endif



//----------------------------------------------------------------- WritePtr()
static void WritePtr(void** ptrp)
/*
        Save phase.
        Process all ABSOLUTE and TypeCLASS pointers in the container,
        before them are written to the image on disk.
        RELATIVE pointers are processed in ProcessRelativePtr()
        All pointers are stored in a vector of little records called CNSwizzList.
        CNSwizzList is processed at the end of PhysicallySave().
*/
{       Saver_node* saver;
        Conim *foreign;
        int whereis;

        // A shortcut for NULL pointers:
        if (*ptrp == NULL)
            return;

        // From where the ptr is pointing to ?
        whereis = (char*)ptrp - (char*)Image;
        assert(Conim::getBeingwritten() != NULL);

        // Is the pointer pointing to any location outside this container?
        foreign = Conim::Ptr_to_conim(*ptrp);
        if (foreign and Conim::getBeingwritten() != NULL
                and Conim::getBeingwritten() != foreign
                and whereis <= compaction_offset) {
            Namedobj* obj;
            CNInfo       *cns;
            char          Type = Absolute_Ptr_Mark;
            char         *rccl = NULL;
            CNInfoList   &list = Conim::getBeingwritten()->prereqinfo;

            // We must add an entry in the C-N Swizzling list with this ptrp
            // Also, test that we are dealing with a ptrp in the image:
            // static offset
            obj = foreign->findNamedObjByPointer(*ptrp);
            if (obj and whereis <= compaction_offset and whereis >= 0) {
                if (NameIsOriginalClass(obj)) {
                    Type = TypeClass_Ptr_Mark;
                    rccl = NameClassdefFromObject(obj)->getReducedClassdef();
                }

                // Check wether we simply add it or we have to create it.
                if (list.addPtrToNamedobj(obj, Type, (void*)whereis, foreign->cid) == NULL) {
                    // Store the CNSwizz node information
                    cns = new CNInfo((void*)whereis, obj, rccl, Type);
                    Conim::getBeingwritten()->prereqinfo.addCNInfo(cns);
                }
            }
            else {
                saver = PointerToSaver(*ptrp);
                if (saver)
                    // Swizzling the pointer in this Container
                    *ptrp = Base + saver->offset + ((char*)*ptrp - (char*)saver->tile);
                /* tco to jose: I believe this case is an error condition.
                You should assert false, not try to swizzle, you will
                swizzle to a bad location.  If there was an 'assert' here
                then I would have found the bug quicker. */
            }
        }
        else {
            // Points from a container to the same container
            saver = PointerToSaver(*ptrp);
            if (saver) {
                assert(saver->offset != -1);
                *ptrp = Base + saver->offset + ((char*)*ptrp - (char*)saver->tile);
            }
        }
}


//----------------------------------- WriteSwizzledToCorrect() ----------
interface void* WriteSwizzledToCorrect(void* ptr)
/* This function is only applicable in the write phase.  We have a  */
/* pointer which points into the original data-structure, not the   */
/* image being saved.  We need to map it to the corresponding	    */
/* pointer in the Image.					    */
/*	Note that there are effectively 3 address ranges: 1. The    */
/* original heap, 2. The Image and 3. The Preferred position.	    */
{
	assert(PtrProcessor == WritePtr);
	Saver_node* saver = PointerToSaver(ptr);
	if (saver == NULL)
	    return NULL;
	else return Image + saver->offset + ((char*)ptr - (char*)saver->tile);
}


interface bool SwizzleWritePhase()
{
	return PtrProcessor == WritePtr;
}


interface bool SwizzleLoadPhase()
/* Are we in the 'add delta to pointers of container image' phase? */
{
	return PtrProcessor == SwizzleFn;
}


interface uint MsizeSwizzled(void* ptr)
/* What is the size of this tile? Its header has already passed through 'fn'. */
{	uint *p, i;

	if (PtrProcessor != WritePtr)
	    return msize(ptr);
	p = *((uint**)ptr - 1);
	p = (uint*)BaseToImage(p);
	if (*p & 3)
	    return *p & ~3;
	i = *p;
	if (p < (uint*)ptr or (char*)p > (char*)ptr + i)
	    return *p - 4;			// A class instance
	else return (char*)p - (char*)ptr;	// A non-class instance typed tile
}


interface void ProcessRelativePtr(int* rpptr, PtrProcessor_fn fn)
/*
   Save Phase.
   Process this relpointer-pointer with function 'fn'.
   Relative pointers take place in compiled code, in
   the CALL instructions and JMP instructions.
   Absolute pointers are processed in WritePtr().
   All pointers to be fixed are stored in a vector of little records
   called CNSwizzList. This vector is processed in PhysicallySave().
*/
{	char tmp[256], tmp2[20];
        Conim *foreign, *conim;
        Namedobj* where;
        int *orig_rpptr;
	int tiledelta;
        void* whereis;
        str ptr;

	if (fn == WritePtr) {
	    tiledelta = (str)current_saver->tile - (Image+current_saver->offset);
            conim = Conim::getBeingwritten();
            assert(conim != NULL);

            // Find the relative address into the Image
	    orig_rpptr = (int*)((char*)rpptr + tiledelta);
	    assert(*orig_rpptr == *rpptr);

            // Where is it pointing to ?
	    ptr = (str)(orig_rpptr + 1) + *orig_rpptr;
	    WritePtr((void**)&ptr);

            // Is it pointing outside its container ?
            foreign = Ptr_to_conim(ptr);
	    if (foreign and conim != foreign) {
                where = foreign->findNamedObjByPointer(ptr);
                if (where != NULL) {
                    whereis = ((void*)((char*)rpptr - (int)Image));

                    // Store this information
                    if (conim->prereqinfo.addPtrToNamedobj(where, Relative_Ptr_Mark,
                            whereis, foreign->cid) == NULL)
                        conim->prereqinfo.addCNInfo(new CNInfo(whereis, where,
                                    (char*)NULL, (char)Relative_Ptr_Mark));
                }
                else {
                    strcpy(tmp, "Error in relative CN Swizzling: "
                                "target not found. C_id being written is: ");
                    strcat(tmp, itoa(Conim::getBeingwritten()->cid, tmp2, 10));
                    strcat(tmp, ", looking for an object in C_id: ");
                    strcat(tmp, itoa(foreign->cid, tmp2, 10));
                    ErrorRun(tmp);
                }
            }
            else {
                // Swizzle the pointer in the container
	        *rpptr = (int)((char*)ptr - ((str)rpptr + (Base - Image) + 4));
            }
	}
	else if (fn == SwizzleFn) {
        /* Load Phase
           We've found a local relative pointer. If it's local, then it's ok.
           If it is not, then it is ok, because, we have processForeignPointerList()'d
           just before.
           So at the present time (design time), we don't have to do anything here!!
        */

            Conim *where = Conim::Ptr_to_conim(rpptr);
            assert(where!=NULL);

/*	    ptr = (str)(rpptr + 1) + *rpptr;	// In Base space
	    if (ptr >= Base and ptr < Base + offset)
		return;
                // If it points inside its own container, then no delta required.
	    *rpptr -= where->getDelta();
*/
	}
        else {
            /* If a tile containing an executable function is only reachable */
            /* via a CALL instruction, and not via the corresponding         */
            /* Namedobj, then we don't attempt to deal with it. Such         */
            /* situations shouldn't occur.  Therefore, there's nothing to do */
            /* here in the reachability phase. */
            /*ptr = (char*)(rpptr+1) + *rpptr;*/
        }
}

//------------------------------------------------------------- PhysicallySave()
inline
void Conim::ForceSave(void)
// Forces this container to be saved.
{
        PhysicallySave();
}

inline
void Conim::cleanNOHash()
/* Cleans the hashing of NamedObj's keyed by u.pointer while loading the
   CN Swizzling entries.
   This method is called by a foreign container when it finishes saving.
*/
{
        if (NOHash!=NULL)
                delete NOHash;
        NOHash = NULL;
}

void Conim::PhysicallySave()
/*
     Save phase.
     This function, which should only be called when the container is opened
     in read and write mode, runs over the tiles and process all the pointers it
     finds through a call to WritePtr(). Information about the pointers to be
     stored in the CN table (i.e. the pointers that point outside) is stored
     in the CNSwizzList vector. This vector is processed at the end of this
     function, and then the CN-Swizzling table is appended.
     The container is written in an auxiliary file. Only if the operation is
     successful then it is written to the correct file.
*/
{       char path[PATHLENGTH], auxfilename[PATHLENGTH];
        ContainerHeader *headeraddr;
	unsigned long numwritten;
        uint size, HeaderSize;
	Saver_node* saver;
        HANDLE AuxHandle;
        int NumTiles;
	char* tile;

        /* Initialise: */
        assert(cid > 0);
        Assert();
	HeapIterator iter(this);
	b_errno = E_NOERROR;
        beingwritten = this;
        header->timestamp = time(NULL);

	/* delete all breakpoints leftover in this container. */
	DebugCloseContainer(this);

	/* Get the filename in advance: */
        if (not CidToPath(cid, path, sizeof(path)))
            goto RETURN;

        /* Create the auxiliary file: */
        strcpy(auxfilename, path);
        strcat(auxfilename, "$$SAV");
        AuxHandle = CreateFile(auxfilename, GENERIC_WRITE,
                                    0 /* No sharing */,
                                    NULL /* No process inheritance */,
                                    CREATE_ALWAYS,
                                    FILE_ATTRIBUTE_NORMAL,
                                    NULL);
        if (AuxHandle == INVALID_HANDLE_VALUE) {
            b_errno = E_IOFAIL;
            goto RETURN;
        }

        /* Give each tile a Saver_node */
        NumTiles = 0;
        for (each_tile)
            NumTiles++;
        size = NumTiles * sizeof(struct Saver_node) + 12;
        Savers = (Saver_node*) ::malloc(anon_heap, size);
        S_idx = 0;
        for (each_tile) {
            Savers[S_idx].offset = (uint)-1;
            Savers[S_idx++].tile = tile;
        }

        /* Recurse on the root tile. */
        HeaderSize = Heap::RoundUp(sizeof(struct ContainerHeader)) + 4;
        compaction_offset = 0;
        NewReached_root = NULL;
        PtrProcessor = Reached;
        headeraddr = header;
        Reached((void**)&headeraddr);
        while (NewReached_root) {
            // Take the next tile off the list:
            Saver_node* saver=NewReached_root;
            NewReached_root = saver->next;

            // Assign it an offset:
            saver->offset = compaction_offset;
	    compaction_offset += Heap::AllocdTileToSize((char*)saver->tile);

            // Find all pointers inside it
	    FindPtrsInTile(saver->tile);
        }

        /* Create the image: */
        Image = (char*) ::malloc(anon_heap, compaction_offset);
        *(uint*)Image = HeaderSize | 2;
        header->lin_start = 0;

        /*
           Here we are deciding wether we are going to put the saved
           container in its current location or not. If it has more than one
           segment, then surely it is going to clash with other container
           in the future, so better we save it in another position
           in memory.
        */
/*            if (getNumOfAllockedSegments() > 1)
             Base    = Image;
        else
*/
        Base    = (char *) header->base_address;

        // Create a copy of the current container
        putDelta(Image - Base);
        PtrProcessor = WritePtr;
        for (saver=Savers; saver < &Savers[S_idx]; saver++) if (saver->offset != (uint)-1) {
            current_saver = saver;
            memcpy(Image + saver->offset, (char*)saver->tile, AllocdTileToSize((char*)saver->tile));
            // FindPtrs and swizzling phase
            FindPtrsInTile(Image + saver->offset);
        }

        /* Save the image to disk. */

        // Prepare the size of the Image
        ((ContainerHeader*) (Image+4))->lin_start    = compaction_offset;
                    // Size without the CNSwizz table
        SetFilePointer(AuxHandle, 0, 0, FILE_BEGIN);

        /* Write the Image */
        WriteFile(AuxHandle, Image, compaction_offset, &numwritten, NULL);
        if (numwritten < compaction_offset)
            b_errno = E_IOFAIL;
        else {
            /*
               Now I put here the dictionary of the C-N Swizzling entries.
               Structure is as explained in the CNInfo class declaration.
               Finally, a TABLE_FINISH_MARK value is appended.
            */
            prereqinfo.deleteRedundantRClassdefs(); // only one RC per class
            for (int i = 0; i < prereqinfo.getNumberOfEntries(); i++) {
                prereqinfo[i]->saveCNInfo((int)AuxHandle);
                Ptr_to_conim(prereqinfo[i]->getNamedobj())->cleanNOHash();
            }

            // Write End of the CN-Swizzling table
            WriteFile(AuxHandle, (const void*) &TABLE_FINISH_MARK,
                      sizeof(void*), &numwritten, NULL);
        }

        // We're done !  Close the original container's file and move
        // the auxiliary file over.
        CloseHandle(FileHandle);
        FileHandle = NULL;
        CloseHandle(AuxHandle);
        DeleteFile(path);
        MoveFile(auxfilename, path);
        ::free(anon_heap, Savers);
        ::free(anon_heap, Image);

        // Free allocated resources and finish
        RETURN:
        prereqinfo.clear();
        beingwritten = NULL;
}

