/** MEMORY.H: The interface to the MEM super-module. **/
#ifndef MEMORY_H
#define MEMORY_H


#include "array.h"
#include "sch_evol.h"

/*----- Heaps: -----*/

#define HEAP_NUMPOWERS	56
#define MAX_HEAP_SIZE	(4<<(HEAP_NUMPOWERS/2))
#define PAGE_SIZE	65536
#define PATHLENGTH      8192
        // PAGE_SIZE is 64k because this is the granularity of VirtualAlloc().

#define READONLY        0
#define READWRITE       1


/*
        I use the word 'tile' to mean a malloc block.
        A 'tile_type' is a ptr to the start of the
        tile including the header, i.e. this is NOT
        the pointer returned to the user.
        For the pointers returned to the user I always use 'void*'.
*/
typedef char* tile_type;


/* WholesaleRegion's: interfacing with WIN32 virtual memory calls. */
struct WholesaleRegion {
        uint numbytes;
        WholesaleRegion* next;

        char* FirstTile() { return (char*)this - numbytes; }
        // The linked list of WholesaleRegions is always sorted in
        // ascending order.
};


class HeapIterator {
        WholesaleRegion *region;
	tile_type tile;
	class Heap* heap;

public:
	HeapIterator(class Heap* heap);
	tile_type operator++(int);
	void reset();
};

#define each_tile	iter.reset(); (tile = iter++) != NULL;



class Heap {
	tile_type freechain[HEAP_NUMPOWERS];
	WholesaleRegion *wholesaleregions;
        int NumFreedSinceMerge, NumUsed;
        volatile int Lock;

	static uint CalcSize2Power(uint size);
	static uint Size2Power(uint size);
        void FreeTile(char* p, uint size, uint good_h=0);
	void MergeTiles();
	void* RawMalloc(uint size, uint h, uint header);
	bool AddNewRegion(uint numbytes, void* insistAddr=NULL);
                // Create a new WholesaleRegion for this heap.
                // The new memory goes straight into freechain[].
	void PageMapAddRegion(void* mem, uint size);	// Add this region to the PageHash.
	void PageMapDelRegion(void* mem, uint size);	// Remove this region from the PageHash.
        void WaitForLock();
        void Unlock();

        /* Class members: */
	static uint size2power[512];
	static uint power2size[HEAP_NUMPOWERS];
	static Heap **PageToHeap;		// An array mapping pages to heaps.
	static int LowestMem, HighestMem;	// The range of PageToHeap

protected:
	static bool TileIsTyped(tile_type tile)
                { return ((*(uint*)tile & 3) == 0); }
	static bool TileTypeIsClass(tile_type tile);
	static Type TileTypestr(tile_type tile);
	static uint FreeTileToSize(tile_type tile);
	static bool IsFree(tile_type tile) { return (*(int*)tile & 1); }
	static uint TileRoundUp(uint size); /* Includes the header size */

public:
	void* malloc(uint size);
	void* realloc(void* p, uint size);
	void* calloc(uint size, uint n);
	str strdup(str s);
	void free(void* p);
	    /* The normal malloc functions. */

	void* New(Classdef* structure);
	void* New(Type typestr);
	void* NewPlus(Type typestr, int extra);
	    /* Create a new object of this class or typestr. */

	void* realloc(void* p, uint size, Type type);
	    /* A typed realloc, related to NewPlus().  */

	static uint msize(void* p);
	    /* What is the rounded-up size of this block, excluding the header? */

	int MemTotalBytes(void);
	    /* How many bytes has this heap used? */

	static Heap* Ptr_to_heap(void* ptr);
	    /* Map this pointer to a heap. */

	static bool AssertTile(tile_type tile);
	bool Assert(void);
	    /* For checking for corruption in the heap. */

	static uint AllocdTileToSize(tile_type tile);
	    /* (Used by Heap and the saver Recurse fn). */

	static Type Typestr(void* ptr, unsigned char dest[5]);
	    /* Querying the type information in tiles. */

	static uint RoundUp(uint size) { return TileRoundUp(size+4) - 4; }
	    /* Round up the size you pass to malloc to what it will */
	    /* really be created with. */

	void PageProtection(bool can_write);
	    /* Make the whole heap unwriteable (or writeable again). */

        void *SuggestedBaseAddress(void);
            /* For getting a suggested base address. */

	Heap() { }                                              // For anon_heap
	Heap(uint init_size, void* insistAddr=NULL);            // For new heaps
	Heap(char* mem, uint init_size, bool writeable);        // for Conims
	~Heap();

        static void InitialiseAnon();
        static void CheckInitialised() { if (power2size[0] == 0) InitialiseAnon(); }

	friend class HeapIterator;
};




/*----- Hash of Namedobj's ---------------------------------------------------*/
/* This has is used while loading a container, and its prerequisites are
   being elicited. We will have to search in foreign containers looking by its
   u.location, which is not a key used within containers, so a linear search is
   used and we at least try to do that linear search only one time
*/

class NamedObjHash {
        private:
                const static short int NH;
                Namedobj ***h;
        public:
                NamedObjHash();
                // Prepare the hash to operate
                ~NamedObjHash();
                // Freeing resources

                Namedobj *getNamedObj(void * key);
                // returns a reference to a NamedObj in this container
                // which its u.location equals the argument passed

                void putNamedObj(Namedobj* obj);
                // Inserts this reference to a Namedobj in the hashing
};



/*----- Conim's: -------------------------------------------------------------*/

/* Iterate through the conim's */
class ConimIterator {
	uint h;
	class Conim *conim;

public:
	void reset();
	Conim* operator++(int);
};

/* While doing CNSwizzling,
   we can find type-class, absolute and relative pointers */
const char Absolute_Ptr_Mark  = 'A';
const char Relative_Ptr_Mark  = 'R';
const char TypeClass_Ptr_Mark = 'C';

/* The entries from the CN Swizzling table while saving/loading a container   */
/*
        Structure of the CN-Swizzling table at the very end of the container
        *on disk* is:
                - Namedobj*    // Which was the address of the foreign Namedobj ?
                - container_id     // Container of the foreign Namedobj.
                - num              // Number of chars for name (counting Type of pointer)
                - char             // Containing one of the values above
                - char[]           // Name of the foreign Namedobj.
                - num              // Number of characters for the classdef
                - char[]           // Reduced classdef.
                - num              // Number of bytes for the next list
                - void **[]        // Location of ptrs affected in the container

        This is repeated n times, being n the number of different types of
        Namedobj's referenced. A CN_TABLE_MARK instead of a legal Namedobj*
        finishes the table.
*/
class CNInfo {
        char *databuffer;
        char *ReducedClassdef;
        void **whereis;
        Namedobj* whereispointing;
        container_id container;
        char TypeOfPointer;
   public:
        bool isAbsolute()  const;
        bool isRelative()  const;
        bool isTypeClass() const;
        bool isCorrupted() const;

        char getTypeOfPointer(void) const;
        int  getPtrListSize(void) const;
        void *getPtrIndx(int) const;
        container_id getForeignContainer(void) const;
        Namedobj* getNamedobj(void) const;
        str getReducedClassdef(void);
        str  getName(void);

        static CNInfo *loadCNInfo(char *&);  // loads a CNInfo entry mapped in memory
        void           saveCNInfo(int);  // saves a CNInfo entry to a disk

        void addPtr(void *);
        void markasInvalidContainer(void);
        void putPtrsNULL(void);
        void rebasePtrs(int);

        // Constructors, destructors
        CNInfo(void *ptr, Namedobj* obj, str RCinfo, char PtrType); // save
        CNInfo(void *ptr, Namedobj* obj, int container, str info, str RCindx);  // load
        ~CNInfo();

        friend class CNInfoList;
};

/* A list of CNEntries, using the dynarray facilities                         */
class CNInfoList {
        CNInfo **list;
  public:
        CNInfoList();
        ~CNInfoList();

        CNInfo *lookforRegNamedObj(container_id, Namedobj*, char) const;
        CNInfo *addPtrToNamedobj(Namedobj* obj, char Type, void *ptr, container_id cont = 0);
        CNInfo *addCNInfo(CNInfo *reg);

        int getNumberOfEntries(void) const;
        CNInfo *getEntryIndx(int) const;
        CNInfo *operator[](int) const;

        void deleteRedundantRClassdefs(void);
        void clear(void);
};

// Containers in memory class
class Conim : public Heap {		// This represents a container in memory.

private:
        static Conim * beingwritten;    // The Conim beingwritten to disk when some Container is being saved.
	static Conim ** Hash;		// A hash-table mapping container id's to Conims.
	static uint NumConims;		// How many elements in Hash?
	static bool LinkGCRequired;	// Do we need a garbage collection of linked containers?
        static bool ReallyCloseAll;     // Are finally closing all containers ?
	Conim *next;			// (For the hash-table)
	char reference_count;		// How many unmatched OpenContainer()'s?
	char link_count;		// How many other containers link to it?
	bool writeable;	        	// Does it have write permission?
	char flags;			// For temporary use
        int  delta;                     // How far is its base memory address ?
        static Conim **ContainersNeedingOfLocalSwizzle;
                                        // Load phase. We have to keep track of
                                        // all containers loaded when one
                                        // container is asked to be loaded
	Conim **prerequisites;	        // Prerequisite containers (for C-N swizzling)
        NamedObjHash *NOHash;           // Used the first time that a Namedobj is searched for
                                        // during CNSwizzling
	char *MappingMem;		// What memory is it mapped into?
	void *FileHandle;		// The handle of the file
	void *MappingHandle;		// The handle of the file-mapping object
	struct ContainerHeader *header;	// The first tile of the container, which is a header describing lin-ptrs etc.

	void MapAddConim();		// Insert this conim into the hash-table.
	void MapDelConim();		// Take this out of the hash-table.
	void localSwizzle();
                                        // Add 'delta' to all pointers in this heap
	static Conim* PhysicallyLoad(container_id cid, bool writeable);
					// Map it into memory
        Conim *OpenPrerequisiteContainer(container_id cid);
                                        // Map a prerequisite in memory if doesn't exist
	void PhysicallySave();		// The garbage-collector saver
	void FindReachable();		// For conim-level garbage-collecting,
	static void LinkGC();		//  i.e. containers that are no longer prerequisites.
	void Link(Conim *B);		// Create a prerequisite link from this to B
	void SetPagePermissions(bool can_write);
		                        // Make the whole heap writeable or unwriteable.
        void putDelta(int);             // delta for local swizzling.
        bool needsOfCnSwizzling(void);  // determines whether CN Swizzling is needed.
public:
	container_id cid;		// The conim's cid.  0=no conim.
        static
        CNInfoList prereqinfo;          // A list of structures for pointers
                                        // that point to outside this container
                                        // and therefore, will be needed to be C-N Swizzled
                                        // used only while saving & loading.

	time_t interface_change;	// Dates of last interface change
	time_t remake;			//          and last re-make.
	time_t interaddr_change;	//          last change to an interface object's address
	Directory *src;		        // The .SRC directory, if it is linked in.
                                        // (void*)0x1 means there's nothing to link.

	Conim(uint size, container_id _pid);
	~Conim();
		// Constructor and destructor
	Conim(char* mem, uint size, container_id _pid, bool _writeable,
		    void* filehandle, void* mappinghandle);

        Namedobj* findNamedObjByPointer(void * ptr);
                // Finds a Namedobj by its u.location pointer
                // Uses the NOHash object hashing
                // If the Namedobj is a class member, then the object owner is
                // filled

        Namedobj* findNamedObjByName(char *);
                // Finds a Namedobj by its name. Even if the container is not
                // localSwizzle()'d, this function will find it anyway.

        void ForceSave(void);
                // Forces this container to be saved.

        void cleanNOHash();
                // Cleans the hashing of NamedObj's keyed by u.pointer
                // while loading the CN Swizzling entries.
                // This method is called by a foreign container when it finishes

	void Close();
		// Decrement this conim's reference count and perhaps close it.

	bool LinksTo(Conim *B);
		// Does this Conim have B as a prerequisite?

	str Name();
		// Construct a useful name for this container.

	void PublicLink(Conim *B) { Link(B); }
		// A public version of the private function.

	Directory* directory();
                // Returns the directory of a container

        const void* &getPreferredAddress(void) const;
                // Returns the preferred address of a container, in which it is
                // expected to be loaded in memory by default

	/* Static functions: */
	static Conim* CreateContainer(container_id parent);
		// Create a new container and conim.

	static Conim* OpenContainer(container_id cid, bool write_permission);
		// Open a pre-existing container into memory

        static void processForeignPointerList();
                // Finishes the CN Swizzling process in this container and its
                // prerequisites ones.

	static Conim* FindConim(container_id cid);
		// Do we have this already in memory?

	static void DestroyAll();
		// Call the destructor on all remaining conims.

	static Conim* Ptr_to_conim(void *mem);
		// Map this pointer to a Conim.

        static Conim *getBeingwritten(void);
                // The container which is being saved (when one of them is saved)

	static struct graph_node* LinkGraph(void);
		// For diagnostic purposes, output the links as a graph.

	static void DumpConims(void);
		// For diagnostic purposes, output the links to stdout.

	void UpdateHeaderClassdef();
		// If we created the container before the ContainerHeader classdef
		// existed, but it exists now, then update the type of the header tile.

	friend class ConimIterator;

        const ContainerHeader &getContainerHeader() const;

        /*
           Protection of memory pages functions. Take a look to the
           implementation for further information.
           The functions below use SetPagePermissions()
        */
        bool IsWriteable(void)   const;
	bool getProtection(void) const;
                // Returns current access mode to memory pages.
        void protectionDowngrade();
                // Change pages to READWRITE mode, if needed.
        void protectionUpgrade();
                // Change pages to READONLY mode, if *possible*.

        /*
                delta is zero when the container doesn't need
                to be swizzled locally. Many useful functions here ...
        */
        const int &getDelta(void) const;
        void resetDelta();
        bool isAlreadySwizzledLocally() const;
};


extern Heap* anon_heap;
extern Heap* default_heap;
extern Conim* Stdlib;
extern void *StdlibLo, *StdlibHi;

#define each_conim	iter.reset(); (conim = iter++) != NULL;









/*------------- The container_id interface to Conim's: ------------*/

#define ROOT_CID	1
// This is the only container with a hardwired id.
extern int COMMON_CID;


Conim* FindConim(container_id cid);
	/* Do we have this already in memory? */

void CloseContainer(container_id cid);
void CloseConim(Conim *conim);
	/* Close the specified container. */

Directory* CreateContainer(container_id parent, container_id *cidp);
	/* Create a container of this parent.  The root directory is	  */
	/* returned formally and the container_id is returned via direcp. */
	/* This function is more for the user-level : system-level	  */
	/* functions should preferably use Conim::CreateContainer() which */
	/* returns a Conim.	  */

Directory* OpenContainer(container_id cid, bool write_permission);
	/* Open a pre-existing container into memory. Return its directory. */
	/* This is more a user-level function: system-level functions	    */
	/* should preferably use Conim::OpenContainer() which returns a	    */
	/* Conim. */

bool DeleteContainer(container_id cid);
	/* Delete this container. The container must not be open. */
	/* We delete the file from disk and the id from the tree-table. */

Conim* Ptr_to_conim(void *ptr);
	/* Mapping arbitrary pointers to conims */





/*------ Heap operations with checking: ------*/

void* malloc_user(uint size);
void free_user(void *mem);
void* realloc_user(void* mem, uint size);
void* calloc_user(uint num, uint size);
str strdup_user(str s);
void* operator_new_user(Type type);
void operator_delete_user(void *vp);
uint msize(void* ptr);
void AccessAddInterval(void* m, uint len);
void AccessDeleteInterval(void* m, uint len);

void HeapCheck(container_id cid);
bool MemorySetDefaultHeap(container_id cid);
container_id MemoryGetDefaultHeap(void);





/*------- FindPtr functions: --------*/

void* FindPtrsInStaticNamedobj(StaticNamedobj *obj);
void* FindPtrsInNamedobj(Namedobj *obj);
void* FindPtrsInMacroNamedobj(MacroNamedobj *obj);
void* FindPtrsInFunctionNamedobj(FunctionNamedobj *obj);
void* FindPtrsInClassdef(Classdef* classdef);




/*------- Reporting failure of OS operations: --------*/

typedef enum {  E_NOERROR, E_NOMEM, E_NO_SUCH_CONTAINER, E_ACCESS,
		E_LOCK, E_NOTOPEN, E_CORRUPT, E_IOFAIL, E_HASCHILD, 
		E_CYCLE, E_NO_OWNERSHIP_FILE, E_CREATE_WINDIR,
                E_MUTEX_BUSY, E_COULDNT_INIT_MUTEX,
                E_BAD_CONTAINER_ID, E_INVALID_PARENT,
                E_FILE_MOVE_ERROR,
		E_NO_DEBUG_OBJ, E_BREAKPOINT_NO_CODE  } b_err_type;

extern b_err_type b_errno;

str BerrnoToString(void);

#endif
