/****************************************************************\
** RUNLIB.CPP:							**
**          Here we provide run-time support for Barbados.	**
** This includes setting up standard types, standard classes,	**
** standard functions.						**
\****************************************************************/


#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>
#include <conio.h>
#include <time.h>
#include "barbados.h"
#include "modules.h"
#include "graph2d.h"
#include "bitmap.h"
#include "memory.h"
#include "source.h"
#include "browse.h"
#include "runlib.h"
#include "trace.h"
#include "debug.h"
#include "name.h"
#include "join.h"
//#include "mem.h"




interface Conim* Stdlib;		/* The Standard Library. */
	/* This is a pseudo-Conim:  it has an Conim object and everything */
	/* that an open container would have, except that it does not have a    */
	/* file on disk, and its container_id = -1 which has no entry in the    */
	/* ownership file.  The reason for doing it this way is so that    */
	/* C-N swizzling works where the referenced object is in the      */
	/* Standard Library. */

interface Directory* stdlib_dir;
interface void *StdlibLo, *StdlibHi;





/*================= Initialising Types: ===================*/

interface Classdef *AnyStruct, *OstreamStruct, *DirecStruct,
		*ClassdefStruct, *TypeStruct, *TyexStruct,
                *BitmapStruct, *FileStruct, *ContainerHeaderStruct,
                *NamedobjStruct, *StaticNamedobjStruct, *MacroNamedobjStruct,
                *FunctionNamedobjStruct, *PlaceStruct, *BGraphStruct;


static Namedobj* LibraryDeclare(str name, Type type, storage_enum storage)
/* Declare this entity.  If it's a static object, its value will */
/* presumably be stored in the system heap.  */
{
	return NameDeclare(curdir, name, type, storage, NAME_OVERLOAD);
}


static void SetUpTypeName(uchar tp, str name)
/* Set up this built-in type as a typedef, */
/* for the benefit of user-defined types.  */
{
	NameDeclare(curdir, name, &tp, typedef_storage, 0);
}


static Classdef* MakeMetaClass(int def_len, Type type, int size)
{       Classdef *classdef;

        classdef = (Classdef*)Stdlib->calloc(1,def_len);
        *type++ = tp_class;
        *(Classdef**)type = classdef;
	classdef->signature = HELLO_BABE;
	classdef->size = size;
	classdef->RoundupSize = Heap::RoundUp(size) + 4;
	classdef->member = NULL;
	classdef->def_len = def_len;
        return classdef;
}


static void AssertMetaClass(str classname, int address)
{       Classdef* classdef;
        Namedobj* obj;

        obj = stdlib_dir->Find(classname);
        classdef = *(Classdef**)(obj->type+1);
        assert(classdef == (Classdef*)address);
}


static void CreatePredefinedType(str name,
		Classdef* *StructPtr, Type dest)
/* 'name' specifies a typedef object which we assert that */
/* we've compiled.  Import it into the world of Barbados  */
/* by pulling out the pointer in the typedef to the	  */
/* specified destinations. */
{	Type type;
	Namedobj *obj;

	obj = curdir->Find(name);
	assert(obj);
	type = obj->type;
	assert(*type == tp_class or *type == tp_enumerated);
	*dest++ = *type++;
	GetPtr(*StructPtr, type);
	PutPtr(*StructPtr, dest);
}


static void Compile(str s)
{
        Compile(s,curdir,'P');
}


static void LibraryInitMetaClasses(void)
/* Set up the fundamental types in the name space. */
/* Must be called from the compiler initialisation */
/* routines. */
{       static char type_buf[] = { tp_pointer, tp_char, tp_pointer, tp_class, 0,0,0,0 };
	extern char print_fn_type[];
	Namedobj* obj;
	Type type;

	/* This function consists of an extremely delicate bootstrapping process. */
        /* Get some basic values into ClassdefStruct manually: */
	ClassdefStruct->signature = HELLO_BABE;
	ClassdefStruct->size = sizeof(Classdef);
	ClassdefStruct->RoundupSize = Heap::RoundUp(sizeof(Classdef)) + 4;
	ClassdefStruct->member = NULL;

        /* Compile the meta-classes into these addresses: */
	LibraryDeclare("str", (Type)type_buf, typedef_storage);
	LibraryDeclare("NULL", voidptr_typstr,   const_storage);
        Compile("class any { type t; void *value; "
		    "public:    type Type();"
		    "void* Address();"
		    "any ArrayLookup(int);"
		    "int ArrayLength();"
		    "any FieldLookup(str s);"
		    "any FieldLookup(Namedobj *obj);"
		    "any FunctionCall(any *Params);"
		    "any Dereference();"
		    "};");
        Compile("struct type_expansion { public:"
		    "int primary; "
		    "class type remainder;"
		    "int n; "
		    "class type params;"
		    "class Namedobj *obj, *typedef_obj;"
		    "};");
	Compile("class type { public: char* ts; public:"
		    "char Expand(class type_expansion*);"
		    "type NextParam();"
		    "type operator=(str s);"
		    "void* New();"
		    "};");
	Compile("class classdef { public:"
		    "int RoundupSize; "
		    "int signature; "
		    "int size; "
		    "int def_len; "
		    "class classdef *myself; "
		    "class Namedobj *typedef_obj; "
                    "void *ptrfinder; "
                    "void **Friends; "
		    "class Namedobj *member; "
		    "void** VirtualFns; "
		    "};");
	Compile("class Namedobj { "
		    "int make; "
		    "class Namedobj* next; "
		    "void* location; "
		    "str name; "
		    "unsigned char storage; "
		    "unsigned char visibility; "
		    "char overload_version; "
		    "unsigned char type[1]; "
		    "public: "
		    "Namedobj* Next(); "
		    "str Name(); "
		    "any Any();"
		    "bool Exists();"
		    "ostream& Print(ostream&);"
		    "Namedobj* Dependency(int n);"
		    "};");
	Compile("class directory { "
		    "Namedobj **Hash; "
		    "unsigned int Hash_len, NumEntries; "
		    "directory *parent; "
		    "container parent_cid; "
		    "public: any FindAny(str);"
		    "Namedobj* Find(str);"
		    "Namedobj* Create(str, type);"
		    "void Delete(str);"
		    "ostream& Print(ostream&);"
		    "};");
	Compile("class ostream { "
		    "str buf,buf_end,s; "
		    "int line_buffering; "
		    "public: int (*flush)(); "
		    "ostream& operator<<(str s); "
		    "ostream& operator<<(bool b); "
		    "ostream& operator<<(unsigned char ch); "
		    "ostream& operator<<(char ch); "
		    "ostream& operator<<(int n); "
		    "ostream& operator<<(long n); "
		    "ostream& operator<<(unsigned int n); "
		    "ostream& operator<<(double g); "
		    "ostream& operator<<(void*); "
		    "ostream& operator<<(type t); "
		    "}; ");
	Compile("class ContainerHeader { public: "
        		"int signature; "
                        "int base_address; "
                        "int lin_start; "
                        "int interface_change, remake, timestamp; "
                        "directory dir; };");
	Compile("class StaticNamedobj : public Namedobj {"
                "public: void* location; };");
	Compile("class MacroNamedobj : public Namedobj {"
                "public: str macro; };");
	Compile("class FunctionNamedobj : public Namedobj {"
                "public: void *fn;"
                "char CallingConvention; "
                "int MinimumArity;"
                "char* DefaultParameters;"
                "classdef** Throws;};");
	Compile("struct FILE { int a[6]; };");
	Compile("class Place {"
		"public: int x,y;"
		"Place(int _x, int _y);"
		"};");
	Compile("class Bitmap { "
		"int cx, cy; "
		"void* mDC; "
		"void* Bmp; "
		"public:  "
		"void Init(int X, int Y);"
		"void Clear(int col);"
		"void Clear();"
		"void Line(int x0,int y0,int x1,int y1, int col);"
		"void Line(Place A, Place B, int col);"
		"void Arrow(Place A, Place B, int col);"
		"void Rect(int x0, int y0, int x1, int y1, int col);"
		"void SetPixel(int x, int y, int col);"
		"void Display();"
		"ostream& Print(ostream &out);"
		"};");
	Compile("class BGraph { void *bg;"
		"public: "
		"void BGraph();"
		"void Free();"
		"void AddVertex(void*, str, int);"
		"void AddVertex(int, str, int);"
		"void AddEdge(void*, void*, int);"
		"void AddEdge(int, int, int);"
		"Bitmap ToBitmap();"
		"ostream& Print(ostream&);"
		"};");

	assert(sizeof(Classdef) == TypeSize(classdef_typstr));
	assert(sizeof(Directory) == TypeSize(direc_typstr));
	assert(sizeof(ContainerHeader) == TypeSize(containerheader_typstr));

	/* Update the first tile of the Stdlib conim to be of type 'ContainerHeaderStruct' */
	/* as opposed to the proto-classdef we used to create the container before */
	/* this classdef existed. */
	Stdlib->UpdateHeaderClassdef();

        /* Create some other stuff: */
	obj = LibraryDeclare("cout", ostream_typstr, static_storage);
	cout = (Ostream*)((StaticNamedobj*)obj)->location;
	InitStdio();
	*(Classdef**)(print_fn_type + 4) = OstreamStruct;
	*(Classdef**)(print_fn_type + 11) = OstreamStruct;
	type = (Type)(dirref_typstr + 1);
	PutPtr(DirecStruct, type);
}









/*=================== Installing standard functions: ==================*/

typedef enum { Voi, Chr, Int, Lng, Flt, Dou, Str, Fil, Ell, Arg, Ty1, Ty2,
		    Boo, ANy, Dir, Cid, Cip, Spt, Vfp, Vpt, Anp, Nob, Ost,
                    Lop, Bmp } Typ;
#define _   Voi






static Type AddTyp(Type s, Typ arg)
/* Adds this type to the str & returns the new str. */
{
	switch (arg) {
	    case Voi:   *s++ = tp_void;
			break;
	    case Boo:   *s++ = tp_bool;
			break;
	    case Chr:   *s++ = tp_char;
			break;
	    case Int:   *s++ = tp_int;
			break;
	    case Lng:   *s++ = tp_long;
			break;
	    case Flt:   *s++ = tp_float;
			break;
	    case Dou:   *s++ = tp_double;
			break;
	    case Fil:   *s++ = tp_pointer;
			*s++ = tp_class;
			PutPtr(FileStruct, s);
			break;
	    case Spt:   *s++ = tp_pointer;
			/* And carry on. */
	    case Ty1:
	    case Str:   *s++ = tp_pointer;
			*s++ = tp_char;
			break;
	    case Arg:   *s++ = tp_pointer;
			*s++ = tp_void;
			break;
	    case Lop:   *s++ = tp_pointer;
                        *s++ = tp_long;
			break;
	    case Vfp:   *s++ = tp_pointer;
			*s++ = tp_function;
			*s++ = 0;
			*s++ = tp_terminated;
			*s++ = tp_void;
			break;
	    case Vpt:   *s++ = tp_pointer;
			*s++ = tp_void;
			break;
	    case Anp:   *s++ = tp_pointer;
			/* And carry on. */
	    case ANy:   memcpy(s, any_typstr, 1 + sizeof(str));
			s += 1 + sizeof(str);
			break;
            case Nob:	*s++ = tp_pointer;
	    		memcpy(s, namedobj_typstr, 1 + sizeof(str));
			s += 1 + sizeof(str);
			break;
	    case Ty2:   memcpy(s, typetype_typstr, 1 + sizeof(str));
			s += 1 + sizeof(str);
			break;
	    case Ost:   *s++ = tp_reference;
			memcpy(s, ostream_typstr, 1 + sizeof(str));
			s += 1 + sizeof(str);
			break;
	    case Dir:   *s++ = tp_reference;
			memcpy(s, direc_typstr, 1 + sizeof(str));
			s += 1 + sizeof(str);
			break;
	    case Cid:   *s++ = tp_container;
			break;
	    case Cip:   *s++ = tp_pointer;
			*s++ = tp_container;
			break;
	    case Bmp:   memcpy(s, bitmap_typstr, 1 + sizeof(str));
			s += 1 + sizeof(str);
			break;
	    default:    assert(false);
	}

	return s;
}


static void Add(str name, void* fn, ...)
/* Add this function to the library.  */
/* Return the CALLSYS number it gets. */
{	int NoParams, FrameSize;
        FunctionNamedobj* fobj;
	uchar *NoParamsPtr;
	uchar buf[80];
	Typ arg, Ret;
	Type type,t;
	va_list ap;


	/* Construct a type-str from the types given. */
	type = buf;
	*type++ = tp_function;
	NoParams = 0;


	/* Grab the return value */
	va_start(ap, fn);
	Ret = va_arg(ap,Typ);
	NoParamsPtr = type++;


	/* Do the parameters */
	FrameSize = 0;
	do {
	    arg = va_arg(ap,Typ);
	    if (arg == Ell) {
		NoParams |= 128;
		break;
	    }
	    else if (arg == Voi)
		break;
	    else {
		t = type;
		type = AddTyp(type, arg);
		FrameSize += TypeSizeWord((Type)t);
		NoParams++;
	    }
	} forever;

	*NoParamsPtr = NoParams;
	*type++ = tp_terminated;
	va_end(ap);
	AddTyp(type, Ret);


	/*** Declare it in the name-space: ***/
	fobj = (FunctionNamedobj*)LibraryDeclare(name, (Type)buf, straight_fn);


        /*** Initialise it: */
	fobj->u.fn = (funcblock_type)fn;
        fobj->Throws = NULL;
        fobj->MinimumArity = NoParams&127;
        fobj->DefaultParameters = NULL;
        fobj->CallingConvention = '+';

        
        /*** Update StdlibLo, StdlibHi: ***/
        if (StdlibHi == NULL)
            StdlibLo = StdlibHi = fn;
        else {
            if (fn < StdlibLo)
                StdlibLo = fn;
            if (fn > StdlibHi)
                StdlibHi = fn;
        }
}










/*---------------- Ostream Functions: -------------------*/

interface bool OstreamFlush(Ostream* out)
/* Call the user-defined 'flusher' function and reset 's'. */
/* If there's no flusher, then it means we're writing into */
/* a simple character buffer: in this case just leave the  */
/* 's' at the end, in order to ignore all subsequent input.*/
{	bool success;

	if (out->flush)
	    success = out->flush() != 0;
	else return no;
	out->s = out->buf;
	return success;
}


interface int OstreamPutch(Ostream* out, int ch)
/* Output a character. */
{
	if (out->s >= out->buf_end) {
	    if (not OstreamFlush(out))
		return -1;
	}
	if (ch)
	    *(out->s)++ = ch;
	return ch;
}


interface int OstreamPutsR(Ostream* out, str s, int len)
/* Output a str with no line-buffering and with length supplied. */
{	int delta;

	while (out->s + len >= out->buf_end) {
	    if (out->flush == NULL) {
		/* We're writing into a simple character buffer. */
		/* Leave 's' at the end so we ignore all subsequent input. */
		if (out->s + 4 > out->buf_end)
		    out->s = out->buf_end - 4;
		strcpy(out->s, "...");
		out->s = out->buf_end;
		return 0;
	    }
	    delta = out->buf_end - out->s - 1;
	    if (delta < 0)
		delta = 0;
	    memcpy(out->s, s, delta);
	    out->buf_end[-1] = '\0';
	    s += delta;
	    len -= delta;
	    if (not OstreamFlush(out))
		return -1;
	}
	memcpy(out->s, s, len);
	out->s += len;
	return 0;
}


interface int OstreamPuts(Ostream* out, str s)
/* Output a str, perhaps with line-buffering. */
{	str t;
	int len;

	if (not out->line_buffering)
	    return OstreamPutsR(out, s, strlen(s));
	else {
	    while ((t=strchr(s, '\n')) != NULL or (t=strchr(s, '\r')) != NULL) {
		if (t == s and *t == '\r' and cout->s == cout->buf) {
		    s++;
		    continue;
		}
		len = t - s + 1;
		memcpy(out->s, s, len);
		out->s += len;
		s += len;
		out->s[0] = '\0';
		if (not OstreamFlush(out))
		    return -1;
	    }
	    return OstreamPutsR(out, s, strlen(s));
	}
}


interface int OstreamPrintf(Ostream* out, const str fmt, ...)
/* Output a printf str. */
{       va_list argp;
	int n;

	if (out == NULL)
	    return 0;
	va_start(argp, fmt);
	if (out->buf_end - out->s >= 512) {
	    n = vsprintf(out->s, fmt, argp);
	    out->s += n;
	}
	else {
	    char buf[4096];
	    n = vsprintf(buf, fmt, argp);
	    OstreamPuts(out, buf);
	}
	return n;
}


interface int Pr(const str fmt, ...)
/* Output a printf str to 'cout'. */
{       va_list argp;
	int n;

	if (cout == NULL)
	    return 0;
	va_start(argp, fmt);
	if (cout->buf_end - cout->s >= 512) {
	    n = vsprintf(cout->s, fmt, argp);
	    cout->s += n;
	}
	else {
	    char buf[4096];
	    n = vsprintf(buf, fmt, argp);
	    OstreamPuts(cout, buf);
	}
	return n;
}


interface void CoutPuts(str s)
{
	OstreamPuts(cout, s);
}


#define Pro	OstreamPrintf




/*-------------------- Outputting of values -------------------*/

static str PrintableChar(char ch)
/* Converts this char to a printable str. */
{       static char r[6];
	static char s1[] = "\n\r\t\b\"\'\\\0";
	static char s2[] = "nrtb\"\'\\0";
	str s;

	s = strchr(s1, ch);
	if (s) {
	    r[0] = '\\';
	    r[1] = s2[s-s1];
	    r[2] = '\0';
	}
	else if (ch >= ' ' or ch < 0) {
	    r[0] = ch;
	    r[1] = '\0';
	}
	else {
	    r[0] = '\\';
	    r[1] = 'x';
	    itoa(ch, r+2, 16);
	}
	return r;
}


static int NumDecimals(double f)
/* How many decimal places should be used to represent 'f'? */
{	int n=0;

	while (n < 6 and fabs((int)f - f) > 0.0001)
	    n++, f *= 10;
	return n;
}


static void OutputOstream(Ostream* out, Ostream* vout)
{
	if (vout->buf == cout->buf)
	    Pro(out, "cout");
	else Pro(out, "(ostream)");
}


interface bool OutputMembersRaw(Ostream* out, Classdef* classdef,
		str value, char need_separator)
{	bool something_done=no;
        FieldNamedobj *mobj;

        for (Namedobj *obj=classdef->member; obj; obj=obj->next) {
            /* Print member 'obj'. */
            if (obj->storage == member_storage) {
                mobj = (FieldNamedobj*)obj;
            	if (need_separator)
                    Pro(out, "%c ", need_separator);
                if (mobj->offset < 0) {
                    int x, offset, bit_offset, bit_width;

                    DecodeBitField(mobj->offset, &offset,
                                        &bit_offset, &bit_width);
                    x = *(unsigned int*)((char*)value + offset);
                    x <<= 32 - bit_offset - bit_width;
                    if (TpIsUnsigned((tp_enum)*obj->type))
                        x = (unsigned int)x >> (32 - bit_width);
                    else x >>= 32 - bit_width;
                    OutputValueRaw(out, &x, obj->type);
                }
                else OutputValueRaw(out, (int*)((char*)value +
                                mobj->offset), obj->type);
                need_separator = ',';
                something_done = yes;
            }
            else if (obj->storage == inherit_storage) {
                mobj = (FieldNamedobj*)obj;
		if (OutputMembersRaw(out, TypeToClassdef(obj->type),
                	value + mobj->offset, need_separator))
                    need_separator = ';';
            }
        }
        return something_done;
}


interface void OutputValueRaw(Ostream* out, void* value, Type type)
/* Take a (value,type) pair and output it to 'out'. */
/* Don't use any user functions.		    */
/* If it's an array type, then 'value' points to    */
/* the start of the array. */
{       int i, size, enum_value, dimension;
	Classdef* classdef;
	Directory* dir;
	Namedobj* obj;
	char *loc;
	uint h;

        LOOP:
	switch (*type++) {
            case tp_const:
            case tp_volatile:
            	    goto LOOP;

	    case tp_bool:
		    Pro(out, *(short*)value ? "true" : "false");
		    break;

	    case tp_short:
		    Pro(out, "%d", *(short*)value);
		    break;

	    case tp_int:
		    Pro(out, "%d", *(int4*)value);
		    break;

	    case tp_long:
		    Pro(out, "%dL", *(int4*)value);
		    break;

	    case tp_ulong:
		    Pro(out, "%uL", *(int4*)value);
		    break;

	    case tp_uchar:
	    case tp_char:
		    Pro(out, "'%s'", PrintableChar(*(char*)value));
		    break;

	    case tp_int64:
		    Pro(out, "%I64dL", *(__int64*)value);
		    break;

	    case tp_float:
		    Pro(out, "%1.*f", NumDecimals(*(float*)value), *(float*)value);
		    break;

	    case tp_double:
		    Pro(out, "%1.*f", NumDecimals(*(double*)value), *(double*)value);
		    break;

	    case tp_longdouble:
		    Pro(out, "%1.*fL", NumDecimals(*(long double*)value), *(long double*)value);
		    break;

	    case tp_ushort:
		    Pro(out, "%u", *(unsigned short*)value);
		    break;

	    case tp_uint:
		    Pro(out, "%u", *(unsigned int*)value);
		    break;

	    case tp_ptrmemberfn:
	    case tp_pointer:
		    if (*type == tp_char) {
			str s = *(str*)value;
			if (s == NULL) {
			    OstreamPuts(out, "NULL");
			    break;
			}
			OstreamPuts(out, "\"");
			while (*s)
			    OstreamPuts(out, PrintableChar(*s++));
			OstreamPuts(out, "\"");
		    }
		    else {
			Pro(out, "0x%lx", *(void**)value);
		    }
		    break;

	    case tp_reference:
		    value = *(int**)value;
		    goto LOOP;

	    case tp_array:
		    GetDimension(dimension, type);
		    loc = (char*)value;
		    if (*type == tp_char) {
			OstreamPuts(out, "\"");
                        while (*loc)
			    OstreamPuts(out, PrintableChar(*loc++));
			OstreamPuts(out, loc);
			OstreamPuts(out, "\"");
			break;
		    }
		    Pro(out, "{ ");
		    if (*type == tp_char)
			size = 1;
		    else size = TypeSize(type);
		    for (i=0; i < dimension; i++) {
			OutputValueRaw(out, (int*)(loc + i*size), type);
			if (i+1 < dimension)
			    Pro(out, ", ");
		    }
		    Pro(out, " }");
		    break;

	    case tp_dynarray:
		    dimension = DynLength(value);
		    loc = (char*)DynVector(value);
		    if (*type == tp_char) {
			OstreamPuts(out, "\"");
			OstreamPuts(out, loc);
			OstreamPuts(out, "\"");
			break;
		    }
		    Pro(out, "{ ");
		    if (*type == tp_char)
			size = 1;
		    else size = TypeSizeWord(type);
		    for (i=0; i < dimension; i++) {
			OutputValueRaw(out, (int*)(loc + i*size), type);
			if (i+1 < dimension)
			    Pro(out, ", ");
		    }
		    Pro(out, " }");
		    break;

	    case tp_class:
		    GetPtr(classdef, type);
		    if (classdef == AnyStruct) {
			OutputAny(out, value);
		    }
		    else if (classdef == OstreamStruct) {
			OutputOstream(out, (Ostream*)value);
		    }
                    else if (classdef == NamedobjStruct) {
			NamedObjPrint(out, (Namedobj*)value);
                    }
                    else {
                        Pro(out, "{ ");
                        OutputMembersRaw(out, classdef, (str)value, '\0');
                        Pro(out, " }");
                    }
		    break;

	    case tp_enumerated:
		    enum_value = *(int4*)value;
		    GetPtrE(obj, type);
		    type -= 5;
                    NameOwner(obj, &dir, &classdef);
                    if (dir) {
                        /* The owner is a directory */
                        for (each_dir_obj(dir)) {
                            if (obj->storage == const_storage and
                                        ((IntNamedobj*)obj)->constval == enum_value
                                        and TypeEqual(obj->type, type))
                                goto FOUND;
                        }
                    }
                    else {
                        /* The owner is a class */
                        for (obj=classdef->member; obj; obj=obj->next) {
                            if (obj->storage == const_storage and
                                        ((IntNamedobj*)obj)->constval == enum_value
                                        and TypeEqual(obj->type, type))
                                goto FOUND;
                        }
                    }
		    Pro(out, "(enum)%d", enum_value);
		    break;
		    FOUND:
		    Pro(out, "%s", obj->name);
		    break;

	    case tp_function:
		    Pro(out, "Function");
		    break;

	    case tp_container:
		    Pro(out, "B%d", *(int4*)value);
		    break;

	    default:
		    assert(false);
	}
}


interface void OutputValueUser(Ostream* out, void* value, Type type)
/* Output this value with either a user-defined output function */
/* or the native ones. */
{
	OutputValueRaw(out, value, type);
}


interface str ValueToString(char buf[], int maxsize, void* value, Type type, bool user_fns)
/* Output this value into this caller-supplied str.	    */
/* If 'user_fn's, then use any applicable user output fn's. */
{	Ostream out;

	out.s = out.buf = buf;
	out.buf_end = buf + maxsize;
	out.line_buffering = no;
	out.flush = NULL;
	__try {
	    if (user_fns)
		OutputValueUser(&out, value, type);
	    else OutputValueRaw(&out, value, type);
	} __except(1) {
	    if (*type == tp_pointer and type[1] == tp_char) {
		__try {
		    Pro(&out, "0x%08X <can't access>", *(void**)value);
		}
		__except (1) {
		    Pro(&out, "[0x%08X] <can't access it>", value);
		}
	    }
	    else Pro(&out, "<can't calculate>");
	}
	*out.s = '\0';
	return buf;
}


interface void OutputAny(Ostream* out, void *value)
/* 'value' points to a type,value object. */
{       class Any any_struct;/*struct Any {
	    Type type;
	    str value;
	} any_struct;    */
	static str NIL=NULL;

	Pro(out, "{ ");
	any_struct = *(class Any*)value;
	if (not TypeValidated(any_struct.type)) {
	    Pro(out, "Invalid }");
	    return;
	}
	OutputType(out, any_struct.type);
	Pro(out, ", ");
	if (any_struct.value == NULL)
	    any_struct.value = (str)&NIL;
	if (any_struct.type[0] == tp_class and
		    TypeEqual(any_struct.type, any_typstr))
	    Pro(out, "??");       // Otherwise we could get unbounded recursion.
	else OutputValueRaw(out, any_struct.value, any_struct.type);
	Pro(out, " }");
}


interface str TypeToString(Type type, char dest[], int sizeofdest)
/* Convert this type to a str and put it in 'dest'. Also return 'dest'. */
{	Ostream out;

	out.s = out.buf = dest;
	out.buf_end = dest + sizeofdest;
	out.flush = NULL;
	OutputType(&out, type);
	*out.s = '\0';
	return dest;
}


interface void OutputType(Ostream* out, Type type)
/* Output this type-str to 'out'. */
{       Classdef* classdef;
	Directory* dir;
	Namedobj* obj;
	int i, dimension;
	bool first;
	uint h;

	forever {
	    switch (*type++) {
		case tp_void:   Pro(out, "void"); return;
		case tp_bool:	Pro(out, "bool"); return;
		case tp_char:   Pro(out, "char"); return;
		case tp_short:  Pro(out, "short"); return;
		case tp_int:    Pro(out, "int"); return;
		case tp_long:   Pro(out, "long"); return;
		case tp_int64:  Pro(out, "int64"); return;
		case tp_float:  Pro(out, "float"); return;
		case tp_double: Pro(out, "double"); return;
		case tp_longdouble:
                                Pro(out, "long double"); return;
		case tp_container:
				Pro(out, "container"); return;
		case tp_volatile:
				Pro(out, "volatile "); break;
		case tp_uint:	Pro(out, "unsigned int"); return;
		case tp_uchar:	Pro(out, "unsigned char"); return;
		case tp_ushort:	Pro(out, "unsigned short"); return;

		case tp_const:	Pro(out, "const "); 
				break;
		case tp_reference:Pro(out, "reference to ");
				break;
		case tp_dynarray:Pro(out, "dynarray of ");
				break;
		case tp_pointer:Pro(out, "ptr to ");
				break;

		case tp_enumerated:
                		/* First search for a typedef name: */
				GetPtrE(obj, type);
				type -= 5;
                                if (((Classdef*)obj->owner)->signature == HELLO_BABE) {
                                    classdef = (Classdef*)obj->owner;
                                    for (obj=classdef->member; obj; obj=obj->next) {
                                        if (obj->storage == typedef_storage and TypeEqual(obj->type, type))
                                    	    goto FOUND_ENUM_TYPEDEF;
                                    }
                                }
                                else {
                                    dir = (Directory*)obj->owner;
                                    for (each_dir_obj(dir)) {
                                        if (obj->storage == typedef_storage and TypeEqual(obj->type, type)) {
                                            FOUND_ENUM_TYPEDEF:
                                            Pro(out, "enum %s", obj->name);
                                            return;
                                        }
                                    }
                                }

                                /* No? Okay: display all the members: */
				Pro(out, "enum { ");
                                type++;
				GetPtrE(obj, type);
				type -= 5;
				first = yes;
                                if (((Classdef*)obj->owner)->signature == HELLO_BABE) {
                                    classdef = (Classdef*)obj->owner;
                                    for (obj=classdef->member; obj; obj=obj->next) {
                                        if (obj->storage == const_storage and TypeEqual(obj->type, type)) {
                                            Pro(out, "%s%s=%d", first ? "" : ", ",
                                                obj->name, ((IntNamedobj*)obj)->constval);
                                            first = no;
                                        }
                                    }
                                }
                                else {
                                    dir = (Directory*)obj->owner;
                                    for (each_dir_obj(dir)) {
                                        if (obj->storage == const_storage and TypeEqual(obj->type, type)) {
                                            Pro(out, "%s%s=%d", first ? "" : ", ",
                                                obj->name, ((IntNamedobj*)obj)->constval);
                                            first = no;
                                        }
                                    }
                                }
				Pro(out, " }");
				return;

		case tp_array:  GetDimension(dimension, type);
				if (dimension == 0)
				    Pro(out, "array of ? ");
				else Pro(out, "array of %d ", dimension);
				break;

	        case tp_ptrmemberfn:
                                Pro(out, "ptr-to-member");
                                /* and carry on */
		case tp_function:
				Pro(out, "function (");
				if (*type == 0)
				    Pro(out, " ");
				for (i=(unsigned char)*type++ & ~128; i; i--) {
				    OutputType(out, type);
				    type += LengthOfTypeString(type);
				    if (i > 1)
					Pro(out, ",");
				}
				assert(*type == tp_terminated);
				type++;
				Pro(out, ") returning ");
				OutputType(out, type);
				return;

		case tp_class:	GetPtr(classdef, type);
				if (classdef->typedef_obj) {
				    Pro(out, "%s", classdef->typedef_obj->name);
				    return;
				}
				return;

		default:        Pro(out, "???");
				return;
	    }
	}
}


interface void OutputClassdef(Ostream* out, Classdef* classdef)
{	Namedobj *obj;

	if (classdef->typedef_obj and classdef->typedef_obj->storage == typedef_storage
		and classdef->typedef_obj->type[0] == tp_class)
	    Pro(out, "class %s {\n", classdef->typedef_obj->name);
	else Pro(out, "class {\n");
	for (obj=classdef->member; obj; obj=obj->next) {
            if (obj->overload_version)
            	Pro(out, "\t%s.%d : ", obj->name, obj->overload_version);
	    else Pro(out, "\t%s : ", obj->name);
	    OutputType(out, obj->type);
	    Pro(out, ";\n");
	}
	Pro(out, "};\n",  classdef->size);
}


static Ostream* TypePrint(Ostream* out, Type type)
{
	OutputType(out, type);
	return out;
}





/*---------------- Misc Functions: -------------------*/

interface bool CheckTypes(Type A, Type B)
/* Are these types of equal size and assignable to each other? */
/* Used to check 'Any'->'Type' projections. */
{
	if ((*A == tp_short or *A == tp_enumerated) and
	    (*B == tp_short or *B == tp_enumerated))
		return yes;
	if (*A == tp_pointer and *B == tp_pointer and
	    (A[1] == tp_void or B[1] == tp_void))
		return yes;
	return TypeEqual(A, B);
}


static void HelpType(Type type, bool print_source)
/* Given this type-str, print a list of the member functions for it. */
{	Classdef* classdef;
	Namedobj* obj;
	str s;

	if (print_source)
	    SourceLinkSource(curconim,no);

	classdef = TypeToClassdef(type);
	if (classdef == NULL)
	    TypePrint(cout, type);
	else {
	    assert(classdef->typedef_obj != NULL);
	    Pr("class %s {\n", classdef->typedef_obj->name);
	    for (obj=classdef->member; obj; obj=obj->next) {
		if (obj->visibility) {
		    Pr("\t%s : ", obj->name);
		    OutputType(cout, obj->type);
		    Pr(";\n");
		}
		if (print_source) {
		    s = SourceHeader(obj);
		    if (s)
			Pr("\t\t%s\n", s);
		}
	    }
	    Pr("};\n");
	}
}


static void GenericArrayConstructor(char* A, int num, int sizeof1,
			void (*Constructor)(void *el))
/* Use this one for static arrays and arrays with fixed dimensions. */
{
	while (--num >= 0) {
            Constructor(A);
            A += sizeof1;
        }
}


static void* GenericArrayConstructor2(Classdef* classdef, int dim,
			void (*Constructor)(void *el))
/* Use this one for dynamic arrays with variable dimensions. */
/* 'classdef' is either a classdef, or (if it's < 256) a     */
/* tp_enum.  'dim' is the number of elements. */
{	uchar typebuf[512];
	char *A, *A0;
	int sizeof1;

	typebuf[0] = tp_array;
        *(int*)(typebuf+1) = dim;
        if ((int)classdef < 256) {
            typebuf[5] = (int)classdef;
            if (typebuf[5] == tp_pointer)
                typebuf[6] = tp_void;
            sizeof1 = TpSize((tp_enum)(int)classdef);
        }
        else {
            typebuf[5] = tp_class;
            *(Classdef**)(typebuf+6) = classdef;
            sizeof1 = classdef->size;
        }
        A = A0 = (char*)operator_new_user(typebuf);
        if (Constructor) {
            while (--dim >= 0) {
                Constructor(A);
                A += sizeof1;
            }
        }
        return A0;
}


static void GenericArrayDestructor(char* A, void (*Destructor)(void *el))
/* Use this one for static arrays and arrays with fixed dimensions. */
{	int dim, sizeof1;
	uchar typebuf[5];
	Type type;

	type = Heap::Typestr(A,typebuf);
        if (type == NULL or *type++ != tp_array) {
            ErrorRun("delete[] on non-array");
            return;
        }
        dim = *((int*&)type)++;
        sizeof1 = TypeSize(type);
	while (--dim >= 0) {
            Destructor(A);
            A += sizeof1;
        }
        free(A);
}


interface str BerrnoToString(void)
{
	switch (b_errno) {
	    case E_NOERROR:	return "No error";
	    case E_NOMEM:	return "Out of memory";
	    case E_NO_SUCH_CONTAINER:	
				return "No such container";
	    case E_ACCESS:	return "Can't write to a read-only container";
	    case E_LOCK:	return "Container is locked by some other process";
	    case E_NOTOPEN:	return "Can't close a container that's not open";
	    case E_CORRUPT:	return "This container's file is corrupt";
	    case E_IOFAIL:	return "Hardware failure";
	    case E_HASCHILD:	return "Can't delete a container with children";
	    case E_CYCLE:	return "That would create an ownership cycle";
	    case E_NO_OWNERSHIP_FILE:	
				return "Couldn't open the PS\\Ownership.dat file";
	    case E_NO_DEBUG_OBJ:return "No debug object";
	    case E_BREAKPOINT_NO_CODE:
				return "No machine-code for this line";
	    default:		return ">Bad errno!<";
	}
}


interface void CheckErrno(container_id cid)
/* If there's been an operating-system level error, */
/* report it to the screen. */
{
	if (b_errno) {
	    ErrorRun("%s : B%d", BerrnoToString(), cid);
	    b_errno = E_NOERROR;
	}
}


interface void PureVirtual()
{
	ErrorRun("A pure virtual function (=0) was called.\n");
}


static void* L_calloc(int size, int num)
{
	return calloc_user(size, num);
}


static void* L_realloc(void* v, int size)
{
	if (not Heap::AssertTile((char*)v-4)) {
	    ErrorRun("Bad realloc()");
	    return NULL;
	}
	else return realloc_user(v, size);
}


static void L_free(void *v)
{
	if (not Heap::AssertTile((char*)v-4))
	    ErrorRun("Bad free()");
	else free_user(v);
}


static bool LibraryCheckAccess(str s)
{
	if (not AccessCheckAccess(s)) {
	    ErrorRun("Memory overwrite!");
	    return no;
	}
	else return yes;
}


static str L_strncpy(str d, str s, int len)
{
	if (not LibraryCheckAccess(d))
	    return NULL;
	else return strncpy(d,s,len);
}


static str L_strcpy(str d, str s)
{
	if (not LibraryCheckAccess(d))
	    return NULL;
	else return strcpy(d,s);
}


static str L_strcat(str d, str s)
{
	if (not LibraryCheckAccess(d))
	    return NULL;
	else return strcat(d,s);
}


/*
static Ostream* prt_char(Ostream* out, char ch)
{
	Pro(out, "%c", ch);
	return out;
}


static Ostream* prt_short(Ostream* out, short i)
{
	Pro(out, "%d", i);
	return out;
}


static Ostream* prt_int(Ostream* out, int i)
{
	Pro(out, "%d", i);
	return out;
}


static Ostream* prt_int64(Ostream* out, __int64 i)
{
	Pro(out, "%I64d", i);
	return out;
}


static Ostream* prt_float(Ostream* out, float f)
{
	Pro(out, "%f", f);
	return out;
}


static Ostream* prt_double(Ostream* out, double f)
{
	Pro(out, "%f", f);
	return out;
}


static Ostream* prt_longdouble(Ostream* out, long double f)
{
	Pro(out, "%fL", f);
	return out;
}


static Ostream* prt_string(Ostream* out, str s)
{
	OstreamPuts(out, s);
	return out;
}


static Ostream* prt_container_id(Ostream* out, container_id cid)
{
	Pro(out, "B%d", cid);
	return out;
}
*/

static int L_puts(str s)
{
	OstreamPuts(cout, s);
	return 0;
}


static str L_gets(str d)
{
	flush_display();
	if (not LibraryCheckAccess(d))
	    return NULL;
	return my_gets(d);
}


static int L_getch(void)
{
	flush_display();
	return EditorGetch();
}


static str L_fgets(str d, int len, FILE *input)
{
	if (not LibraryCheckAccess(d))
	    return NULL;
	else return fgets(d, len, input);
}


static int L_putch(int ch)
{
	Pro(cout, "%c", ch);
	return 0;
}



#undef isalpha
#undef isalnum
#undef isdigit
#undef isupper
#undef islower
#undef isspace
#undef toupper
#undef tolower


static str L_itoa(int n, str d, int radix)
{
	if (not LibraryCheckAccess(d))
	    return NULL;
	else return itoa(n,d,radix);
}


static double L_floor(double f)
{
        return floor(f);
}


static double L_fabs(double f)
{
        return fabs(f);
}


static str L_memcpy(str d, str s, int len)
{
	if (not LibraryCheckAccess(d))
	    return NULL;
	else return (str)memcpy(d,s,len);
}


static str L_memmove(str d, str s, int len)
{
	if (not LibraryCheckAccess(d))
	    return NULL;
	else return (str)memmove(d,s,len);
}


static str L_memset(str d, int val, int len)
{
	if (not LibraryCheckAccess(d))
	    return NULL;
	else return (str)memset(d,val,len);
}


static void L_dir_c(void)
{
	DirectoryPrint(curdir);
}


static int L_size_of(Type type)
{
	if (not TypeValidated(type)) {
	    ErrorRun("Bad type in function call");
	    return 0;
	}
	else return TypeSize(type);
}


static void L_cd_str(str s)
{	container_id cid;
        void *location;
        Namedobj* obj;

	obj = NamePathToObj(s);
	if (obj == NULL or obj->storage != static_storage) {
	    ErrorRun("Can't find %s.", s);
            return;
        }
        location = ((StaticNamedobj*)obj)->location;
	if (TypeEqual(obj->type, direc_typstr))
	    Chdir((Directory*)location);
	else if (TypeEqual(obj->type, container_typstr)) {
	    cid = *(container_id*)location;
	    if (not ChdirContainer(cid))
		ErrorRun("Unable to cd(B%d)", cid);
	}
	else ErrorRun("Can't cd to %s.", s);
}


static void L_cd_container(container_id cid)
{
	if (not ChdirContainer(cid))
	    ErrorRun("Unable to cd(B%d)", cid);
}


static str DirToPath(Directory* dir, str buf)
/* Construct the absolute path-name of this directory */
/* and put it into 'buf[]'. */
{	Directory* parentdir;
	container_id parentcid;
	Namedobj* obj;
	uint h;

	/* Bottoming out: */
	if (dir->parent == NULL) {
            Conim *conim = Ptr_to_conim(dir);
            if (conim == NULL) {
            	strcpy(buf, "?/");
                return buf+2;
            }
            container_id cid = conim->cid;
            if (cid == ROOT_CID) {
                strcpy(buf, "/");
                return buf + 1;
            }
            parentcid = RetParent(cid);
            if (parentcid == 0 or (parentdir=OpenContainer(parentcid, READONLY)) == NULL) {
            	strcpy(buf, "<error>/");
		return buf+strlen(buf);
	    }
	    curconim->PublicLink(Conim::FindConim(parentcid));
	    CloseContainer(parentcid);

	    /* Get the parent's path: */
	    buf = DirToPath(parentdir, buf);
	    if (buf[-1] != '/')
		*buf++ = '/';

	    /* Find the name of the current directory: */
            for (each_dir_obj(parentdir)) {
                if (obj->storage == static_storage and obj->storage == static_storage
                	and cid == *(container_id*)((StaticNamedobj*)obj)->location) {
                    strcpy(buf, obj->name);
                    return buf + strlen(buf);
                }
            }
            strcpy(buf, "Not implemented - recursively search for cid");
            // The cid must be in some subdirectory. We should do a recursive
            // search for it.
            return buf+1;
	}

	/* Get the parent's path: */
	buf = DirToPath(dir->parent, buf);
	if (buf[-1] != '/')
	    *buf++ = '/';

	/* Find the name of the current directory: */
	for (each_dir_obj(dir->parent)) {
	    if (obj->storage == static_storage and
                        dir == (Directory*)((StaticNamedobj*)obj)->location) {
		strcpy(buf, obj->name);
		return buf + strlen(buf);
	    }
	}

	/* If something goes wrong: */
	strcpy(buf, "?");
	return buf+1;
}


static str L_pwd(void)
{	char buf[1024];
	static str s;

	if (s)
	    free(NULL, s);
	s = DirToPath(curdir, buf);
	*s = '\0';
	s = strdup(anon_heap, buf);
	return s;
}


static container_id L_pwc(void)
{	Conim *conim;

	conim = Ptr_to_conim(curdir);
	if (conim == NULL)
	    return 0;
	return conim->cid;
}


static void L_exit(int n)
{
	ErrorRun("Program exited: %d", n);
}


static bool L_feof(FILE *input)
{
	return feof(input) != 0;
}


static int L_errno()
{
	return errno;
}


static void L_YieldToWindows()
{       extern bool TfcYield(bool DoCommands);

        TfcYield(no);
}


static int L_time()
{
	return time(NULL);
}


static void L_PrintClass(Type type)
{	Classdef* classdef;

	classdef = TypeToClassdef(type);
        if (classdef == NULL)
            Pro(cout, "Not a class\n");
	else OutputClassdef(cout, classdef);
}


static void* L_any_project(Any A, Type type)
{
	if (not CheckTypes(A.type, type))
	    ErrorRun("Bad 'Any' projection");
	return A.value;
}


static void* L_dynarray(void* dyn, int n, int size)
{       int oldsize, newsize;

	if ((unsigned int)n < (unsigned int)DynLength(dyn))
	    return (char*)DynVector(dyn) + n*size;
	if (n < 0) {
	    ErrorRun("Negative dynarray index");
	    return NULL;
	}
	oldsize = size * DynLength(dyn);
	newsize = size * (n + 1);
	newsize = ((newsize-1) | 63) + 1;
	if (newsize > 128000) {
	    ErrorRun("Dynarray index too big.");
	    return NULL;
	}
	DynLength(dyn) = n + 1;
	if (DynVector(dyn))
	    DynVector(dyn) = Ptr_to_conim(dyn)->realloc(DynVector(dyn), newsize);
	else DynVector(dyn) = Ptr_to_conim(dyn)->malloc(newsize);
	memset((char*)DynVector(dyn) + oldsize, 0, newsize - oldsize);
	return (char*)DynVector(dyn) + n*size;
}


static int L_dynarray_len(void *dyn)
{
	return DynLength(dyn);
}


static void L_Help1(str s)
{
	HelpType((Type)s, no);
}


static void L_Help2(str s)
{
	HelpType((Type)s, yes);
}


/* For some reason, the FPU supports sin/cos/tan but not pow,log or exp. */
/* (Although it does have base 2 log & exp). */

static double L_pow(double a, double b)
{       return (double)pow(a,b); }

static double L_log(double x)
{	return (double)log(x); }

static double L_log10(double x)
{	return (double)log10(x); }

static double L_exp(double x)
{	return (double)exp(x); }


static void L_GraphOfCalls(str s)
{       BrowseWord(s, yes); }

static void L_GraphOfDepends(str s)
{       BrowseWord(s, no); }

static void L_GraphOfContainers(void)
{       BrowseConims(); }

static int L_LineCount(void)
{       int r;
	extern int StatLineCount;

	r = StatLineCount;
	StatLineCount = 0;
	return r;
}

static void * L_getContainerHeader(void)
{
        return ((void*)const_cast<ContainerHeader *>(&curconim->getContainerHeader()));
}


static void BreakIn(str s)
{   	Namedobj* obj;
	char buf[512];
	bool success;

	obj = NamePathToObj(s);
	if (obj == NULL) {
	    ErrorRun("No such object \"%s\" found. Maybe you mistyped it, "
		    "or you're in the wrong directory.", s);
	    return;
	}
	if (obj->storage != straight_fn and obj->storage != member_fn) {
	    ErrorRun("\"%s\" is not a function.  It's a %s.",
		    obj->name,
		    (obj->storage == macro_storage) ? "macro" :
		    (obj->storage == typedef_storage) ? "typedef" :
		    TypeToString(obj->type, buf, 80));
	    return;
	}
        if (obj->make == NULL)
            SourceLinkSource(Ptr_to_conim(obj),no);
	success = DebugInsertBreakpoint(obj, 0, NULL);
	if (not success)
	    ErrorRun("Unable to set breakpoint in \"%s\" because: %s", s, BerrnoToString());
}


struct scanf_args {
	void* A[32];
};			/* Unfortunately, there's no 'vsscanf()' so I have to use this hack! */
			/* I allow up to 32 arguments. */
static int my_scanf(str fmt, scanf_args args)
{	char buf[16384];

	buf[0] = '\0';
	my_gets(buf);
	return sscanf(buf, fmt, args);
}


interface Ostream* PrintAny(Ostream* ostr, int size, Type type, int value)
{
	if (*type == tp_array)
	    OutputValueRaw(ostr, (char*)value, type);
	else OutputValueRaw(ostr, &value, type);
	return ostr;
}


static void any_function_call(Any a, Any b)
{
	assert(false);
}


static Ostream* PrtPtr(Ostream* out, void* p)
{
	Pro(out, "0x%x", (int)p);
	return out;
}





/*------------- Standard I/O buffering -----------------*/

static char stdout_buf[400];
static char stdin_buf[80], *stdinp;
static int width;
interface Ostream* cout;


interface void InitStdio(void)
{
	stdinp = stdin_buf;
	width = sizeof(stdout_buf)-1;
	assert(cout != NULL);
	cout->s = cout->buf = stdout_buf;
	cout->buf_end = stdout_buf + width - 1;
	cout->flush = flush_stdout;
	cout->line_buffering = yes;
	AccessAddInterval(stdout_buf, width);
}


interface int flush_stdout(void)
/* Flush my stdout buffer */
{
	if (cout->s == stdout_buf)
	    return 1;
	if (cout->s[-1] == '\n' or cout->s[-1] == '\r')
	    cout->s--;
	*cout->s = '\0';
	OutputLine(stdout_buf);
	cout->s = stdout_buf;
	return 1;
}


interface void flush_display(void)
/* Flush the stdout buffer and update the screen immediately. */
{
	flush_stdout();
	EditorReprint();
	width = EditorWidth();
	if (width > 120)
	    width = 120;
}


interface str my_gets(char dest[])
/* Gets a str from standard input. */
{
	if (cout->s > cout->buf) {
	    *cout->s = '\0';
	    OutputLine(stdout_buf);
	    cout->s = stdout_buf;
	}
	return EditorGets(dest, 160);
}





/*---------------- Object & Directory Manipulation -------------------*/

interface Any DirFindObj(Directory* dir, str uname)
/* Look for an object of this name in 'dir'. */
{	Namedobj* obj;
	Any r;

	obj = dir->ObjFromUname(uname);
	if (obj == NULL) {
	    r.type = (Type)"";
	    r.value = NULL;
	}
	else if (obj->storage == typedef_storage) {
	    r.type = obj->type;
	    r.value = NULL;
	}
	else {
	    r.type = obj->type;
	    r.value = (str)((StaticNamedobj*)obj)->location;
	}
	return r;
}


interface bool CanModifyObj(Namedobj* obj)
/* Is this object modifiable?  If not, it's because either: */
/* (a) it's a built-in,  or:                 */
/* (b) it's in use in the current execution. */
{	extern bool MakeEntityInUse(Make *A);

	if (obj == NULL) {
	    ErrorRun("No such object.");
	    return no;
	}
	else if (obj->storage == oneinstr_fn) {
	    ErrorRun("Object %s is a built-in", obj->name);
	    return no;
	}
	else if (MakeEntityInUse(obj->make)) {
	    ErrorRun("Object %s is in use", obj->name);
	    return no;
	}
	else return yes;
}



interface Namedobj* DirCreateObj(Directory* dir, str name, Type type)
/* The function users use to create an object in a directory. */
/* If it already exists then we return the existing object.   */
{	Namedobj* obj;

	if (not TypeValidated(type)) { // Out of mem or bad type.
	    ErrorRun("Bad type in 'create()'.");
	    return NULL;
	}

	if (strchr(name, '/')) {
	    ErrorRun("Path names not allowed in DirCreateObj.");
	    return NULL;
	}

	obj = NameDeclare(dir, name, type, static_storage, NAME_OVERLOAD);
	return obj;
}


interface void DirDeleteObj(Directory* dir, str name)
/* The users function for deleting objects from the persistent name space. */
{	Namedobj* obj;

	obj = dir->ObjFromUname(name);
	if (obj == NULL) {
	    ErrorRun("Object not found: %s", name);
	    return;
	}
	if (CanModifyObj(obj))
	    NameDelete(dir, obj);
}




static Directory* s_dir, *d_dir;
static Namedobj* s_obj, *d_obj;
static str d2,d3, new_name;

static bool GetSourceAndDest(str s, str d)
/* Get source object and directory and destination */
/* name and directory given their path names.      */
/* Returns 1 for success. */
{
	if (s == NULL or d == NULL)
	    return no;

	/* Get the source object: */
	s_obj = NamePathToObj(s);
	if (s_obj == NULL)
	    return no;
	s_dir = (Directory*)s_obj->owner;

	/* Get the destination directory and name: */
	d3 = NULL;
	for (d2=d; *d2; d2++)
	    if (*d2 == '/')
		d3 = d2;
	if (d3 == NULL)
	    d_dir = curdir, new_name = d;
	else if (d3 == d)
	    d_dir = curconim->directory(),
	    new_name = d+1;
	else {
	    *d3++ = '\0';
	    new_name = d3;
	    d_obj = NamePathToObj(d);
	    if (d_obj == NULL)
		return no;
	    d_dir = (Directory*)d_obj->owner;
	    if (not TypeEqual(d_obj->type, direc_typstr)) {
		ErrorRun("Bad destination for move()");
		return no;
	    }
	    d_dir = (Directory*)((StaticNamedobj*)d_obj)->location;
	}
	if (*new_name == '\0')
	    return no;
	return yes;
}


interface void CopyObj(str s, str d)
/* Move object 's' to destination 'd'. */
{       void *slocation, **dlocation;
	bool ValidParams;
        int *temp;
	int size;

	/* Get the parameters: */
	ValidParams = GetSourceAndDest(s,d);
	if (not ValidParams)
	    return;

	/* Copy the source: */
	d_obj = NameDeclare(d_dir, new_name, s_obj->type, static_storage, NAME_OVERLOAD);
	if (Error.err)
	    return;
        slocation = ((StaticNamedobj*)s_obj)->location;
        dlocation = &((StaticNamedobj*)d_obj)->location;
	if (TypeEqual(s_obj->type, direc_typstr))
	    *dlocation = slocation;
	else if (*s_obj->type == tp_function) {
	    size = SizeOfFunction((FunctionNamedobj*)s_obj);
	    *dlocation = temp = (int*)malloc(anon_heap, size);
	    memcpy(temp, (str)slocation, size);
	}
	else {
	    size = TypeSizeWord(s_obj->type);
	    if (size)
		memcpy(*dlocation, slocation, size);
	}
}


interface void MoveObj(str s, str d)
/* Move object 's' to destination 'd'. */
{       void **slocation, **dlocation;
	bool ValidParams;
	int size;

	/* Get the parameters: */
	ValidParams = GetSourceAndDest(s,d);
	if (not ValidParams)
	    return;
	if (TypeEqual(s_obj->type, direc_typstr)) {
	    ErrorRun("Can't move directories like this.");
	    return;
	}

	/* Copy the source: */
	d_obj = NameDeclare(d_dir, new_name, s_obj->type, static_storage, NAME_OVERLOAD);
	if (Error.err)
	    return;
        slocation = &((StaticNamedobj*)s_obj)->location;
        dlocation = &((StaticNamedobj*)d_obj)->location;
	if (*s_obj->type == tp_function) {
	    *dlocation = *slocation;
	    *slocation = NULL;
	}
	else {
	    size = TypeSizeWord(s_obj->type);
	    if (size)
		memcpy(*dlocation, *slocation, size);
	}

	/* Delete the source object : */
	Chdir(s_dir);
	if (CanModifyObj(s_obj))
	    NameDelete(s_dir, s_obj);
}



/*--------------- Types, any's and named_any's: --------------*/

interface int TypeExpand(Type type, struct type_expansion *tyex)
/* Splits up 'type' into a 'tyex' classdef for easy querying. */
{       Classdef* classdef;
	int arity;

	if (tyex == NULL)
	    return *type;

	clearS(*tyex);
	tyex->primary = *type;

	switch (*type++) {

	    case tp_container:
	    case tp_ushort:
	    case tp_short:
	    case tp_float:
	    case tp_uchar:
	    case tp_int64:
	    case tp_long:
	    case tp_char:
	    case tp_uint:
	    case tp_int:
		    break;

	    case tp_pointer:
	    case tp_reference:
		    tyex->remainder = type;
		    break;

	    case tp_array:
		    GetDimension(tyex->n, type);
		    tyex->remainder = type;
		    break;

	    case tp_dynarray:
		    tyex->n = -1;
		    tyex->remainder = type;
		    break;

	    case tp_class:
		    GetPtr(classdef, type);
		    tyex->obj = classdef->member;
		    tyex->typedef_obj = classdef->typedef_obj;
		    break;

	    case tp_enumerated:
		    tyex->obj = *(Namedobj**)type;
		    break;

	    case tp_ptrmemberfn:
	    case tp_function:
		    tyex->n = *type++;
		    tyex->params = type;
		    arity = tyex->n & 127;
		    while (arity-- > 0)
			type += LengthOfTypeString(type);
		    assert(*type == tp_terminated);
		    type++;
		    tyex->remainder = type;
		    break;

	    default:tyex->primary = tp_error;
		    break;
	}
	return tyex->primary;
}


interface Type TypeNextParam(Type type)
{
	type += LengthOfTypeString(type);
	return type;
}


static void* TypeNew(Type type)
/* Create a new object of this type. I would have returned */
/* an 'Any' as a value, except that this causes a     */
/* conflict between my calling conventions with > 4 byte   */
/* values, versus Borland's. */
{       static Any A;

	if (not TypeValidated(type)) {
	    ErrorRun("Bad type");
	    type = bool_typstr;
	}
        return default_heap->New(type);
}


interface Type AnyType(Any A)
{
	return A.type;
}


interface Any* AnyArrayLookup(Any A, int i)
{       static Any E;
	Type type;
	int dimension;
	void *dyn;

	E.type = NULL;
	E.value = NULL;
	type = A.type;
	if (*type == tp_array) {
	    type++;
	    GetDimension(dimension, type);
	    E.type = type;
	    if (i >= dimension or i < 0)
		goto ERROR;         // It's out of bounds!
	    E.value = (char*)A.value + i * TypeSize(type);
	    return &E;
	}
	else if (*type == tp_dynarray) {
	    type++;
	    E.type = type;
	    dyn = A.value;
	    dimension = DynLength(dyn);
	    if (i >= dimension or i < 0)
		goto ERROR;         // It's out of bounds!
	    E.value = (char*)DynVector(dyn) + i * TypeSize(type);
	    return &E;
	}
	else {
	    ERROR:
	    ErrorRun("Error in AnyArrayLookup()");
	    return &E;
	}
}


interface int AnyArrayLength(Any A)
/* Returns the number of elements in this array. */
{       Type type;
	int dimension;
	void* dyn;

	type = A.type;
	if (*type == tp_array) {
	    type++;
	    GetDimension(dimension, type);
	    return dimension;
	}
	else if (*type == tp_dynarray) {
	    dyn = A.value;
	    return DynLength(dyn);
	}
	else return -1;
}


interface Any* AnyFieldLookup(Any A, Namedobj* this_obj)
{       Classdef* classdef;
	Namedobj* obj;
	static Any E;
	Type type;

	E.type = NULL;
	E.value = NULL;

	type = A.type;
	if (*type++ != tp_class)
	    goto ERROR;         // It's not a struct!
	GetPtr(classdef, type);
	for (obj=classdef->member; obj; obj=obj->next)
	    if (obj == this_obj)
		goto FOUND;
	ERROR:
	ErrorRun("Error in AnyFieldLookup()");
	return &E;

	FOUND:
	E.type = obj->type;
	E.value = (char*)A.value + ((FieldNamedobj*)obj)->offset;
	return &E;
}


interface Any* AnyFieldLookupS(Any A, str s)
{       Classdef* classdef;
	Namedobj* obj;
	static Any E;
	Type type;

	E.type = NULL;
	E.value = NULL;

	type = A.type;
	if (*type++ != tp_class)
	    goto ERROR;         // It's not a struct!
	GetPtr(classdef, type);
	for (obj=classdef->member; obj; obj=obj->next)
	    if (strcmp(obj->name, s) == 0)
		goto FOUND;
	ERROR:
	ErrorRun("Error in AnyFieldLookup()");
	return &E;

	FOUND:
	E.type = obj->type;
	E.value = (char*)A.value + ((FieldNamedobj*)obj)->offset;
	return &E;
}


interface Any* AnyDereference(Any A)
{       static Any E;
	Type type;

	E.type = NULL;
	E.value = NULL;
	type = A.type;
	if (*type != tp_pointer and *type != tp_reference) {
	    ErrorRun("Error in AnyDereference!");
	    return &E;
	}
	E.type = ++type;
	if (A.value == NULL) {
	    ErrorRun("Null pointer in AnyDereference!");
	    return &E;
	}
	E.value = *(void**)A.value;
	return &E;
}


interface int* AnyFunctionCall(int **SPptr, Any *Fn, Any *Params)
/* Call this user-function with these parameters.       */
/* If it's a member function, Params[0] contains 'this'.*/
/* Returns the result as an 'any' on the stack.  Returns*/
/* the stack pointer formally. */
{
	assert(false);
#if 0
	int i, arity, ret_size, frame_size, size;
	Type type, ret;
	instruction code[8];
	int *SP, *SP_old;
	static Any E;


	/*** Is it a function? ***/
	type = Fn->type;
	if (*type++ != tp_function) {
	    ErrorRun("Your 'any' is not a function!");
	    return *SPptr;
	}


	/*** Get the return value: ***/
	ret = E.type = type;
	type += LengthOfTypeString(type);


	/*** Check each of the parameters: ***/
	arity = *type++;
	for (i=0; i < arity; i++) {
	    if (not CheckTypes(Params[i].type, type)) {
		ErrorRun("Parameter %d doesn't match!", i);
		return *SPptr;
	    }
	    type += LengthOfTypeString(type);
	}
	assert(*type == tp_terminated);


	/*** Load the values onto the stack: ***/
	SP = SP_old = *SPptr;
	frame_size = 0;
	for (i=0; i < arity; i++) {
	    size = TypeSizeWord(Params[i].type);
	    (char*)SP -= size;
	    memcpy(SP, Params[i].value, size);
	    frame_size += size;
	}


	/*** Call the function: ***/
	*--(op_type*)SP = Fn->value;
	code[0] = CALL_SP;
	code[1] = POPDX;
	code[2] = frame_size;
	code[3] = DEREF_RV;
	code[4] = ret_size = TypeSizeWord(ret);
	code[5] = RETI;
	*SPptr = SP;
	SP = ExecutorSubExecute(code);
	assert(ret_size == (char*)SP_old - (char*)SP);
	E.value = malloc_ault(ret_size);
	memcpy(E.value, SP, ret_size);
	(char*)SP += ret_size;
	*--(Any**)SP = &E;
	*SPptr = SP;
	return SP;
#endif
	return NULL;
}


interface Namedobj* NamedObjNext(Namedobj* obj)
{
	if (obj == NULL)
	    return NULL;
	else return obj->next;
}


interface str NamedObjName(Namedobj* obj)
{
	if (obj == NULL)
	    return NULL;
	else return obj->name;
}


interface Any* NamedObjAny(Namedobj* obj)
{       static Any E;

	if (obj == NULL)
	    return NULL;
	if (obj->storage != static_storage and obj->storage != straight_fn) {
	    E.type = NULL;
	    E.value = NULL;
	    return &E;
	}
	E.type = obj->type;
	E.value = ((StaticNamedobj*)obj)->location;
	return &E;
}


interface Namedobj* NamedObjDependency(Namedobj* obj, int n)
{       int i;

	if (obj == NULL)
	    return NULL;
	if (obj->make == NULL)
	    return NULL;
	if (obj->make->depends_on == NULL)
	    return NULL;
	if (n < 0)
	    return NULL;
	for (i=0; i <= n; i++)
	    if (obj->make->depends_on[i] == NULL)
		return NULL;
	return obj->make->depends_on[n]->obj;
}


interface Namedobj* NamedObjFromString(Directory* dir, str s)
/* Find a named-object by looking up this string in the current name-space. */
{
	if (dir == NULL)
            dir = curdir;
	return dir->Find(s);
}


interface Ostream* NamedObjPrint(Ostream* out, Namedobj* obj)
{	
	if (obj->overload_version)
            Pro(out, "{ \"%s\"'%d ",
        	obj->name,
                obj->overload_version);
        else Pro(out, "{ \"%s\" ", obj->name);
	OutputType(out, obj->type);
	Pro(out, " <%s> at 0x%X }",
                StorageToString(obj->storage),
                ((StaticNamedobj*)obj)->location);
        return out;
}









/*--------------- Fixing a corrupt persistent store: ---------------*/

static container_id* WhatContainersDoWeHave(Directory* dir)
/* Obtain a list of all the child containers sprouting from 'dir'. */
{       Namedobj* obj;
	container_id *A;
	int A_idx;
	uint h;

	A = (container_id*)malloc(anon_heap, sizeof(container_id));
	A_idx = 0;
	for (each_dir_obj(dir)) {
	    if (obj->type[0] == tp_container) {
		A = (container_id*)realloc(anon_heap, A, Heap::RoundUp((A_idx+2)*sizeof(container_id)));
		A[A_idx++] = *(container_id*)((StaticNamedobj*)obj)->location;
	    }
	}
	A[A_idx++] = 0;
	return A;
}


static int compar1(container_id *Ap, container_id *Bp)
{
	return *Ap - *Bp;
}


static void SortList(container_id *A)
/* Sort this list of container id's into ascending order. */
{       int A_idx;

	A_idx = 0;
	while (A[A_idx])
	    A_idx++;
	qsort(A, A_idx, sizeof(A[0]), (cmp_fn)compar1);
}


static str PidToName(container_id cid)
{       static char buf[20];

	strcpy(buf, "B");
	itoa(cid, buf+1, 10);
	return buf;
}


static Namedobj* DirectoryFindContainerPtr(Directory* dir, container_id cid)
/* Find the object which is a reference to 'cid'. */
{       Namedobj* obj;
	uint h;

	for (each_dir_obj(dir)) {
	    if (obj->type[0] == tp_container) {
		if (*(container_id*)((StaticNamedobj*)obj)->location == cid)
		    return obj;
	    }
	}
	return NULL;
}


static container_id L_curcid()
/* Return the container id of the current container. */
{
	return curconim->cid;
}


static container_id CreateContainerCid(container_id cid)
{
	CreateContainer(cid,&cid);
	return cid;
}


static container_id MkdirContainer(container_id cid, container_id parent_cid)
/* When the user calls 'mkdir(X)', we declare X to be a container, but */
/* then if X is non-null we create a container id, but if X already */
/* exists then we don't modify it. */
{
	if (cid == 0)
	    CreateContainer(parent_cid,&cid);
	return cid;
}


static bool RmdirContainer(container_id cid)
{
        return DeleteContainer(cid);
}


interface void FixChildContainers(void)
/* Get the current directory consistent with the ownership */
/* tree (the underlying directory tree) by adding containers     */
/* we don't have and removing ones we do.                  */
/*          The Barbados directory hierarchy allows you to */
/* create an arbitrary graph structure with container id's,	   */
/* but the underlying operating systems maintains a strict */
/* tree-like hierarchy: each container has one owner.		   */
{       Directory* dir;
	container_id *Have, *Need;
	Namedobj* obj;
	container_id cid;
	int h,n;

	/* What containers are in the Barbados directory at the C:\\PS level? */
	dir = curdir;
	if (dir->parent) {
	    Pr("You must call this from the root of a container.\n");
	    return;
	}
	Have = WhatContainersDoWeHave(dir);

	/* What containers are in the underlying operating system's directory? */
	Need = DiskListOfChildren(curconim->cid);

	/* Sort and compare them: */
	SortList(Have);
	SortList(Need);
	h = n = 0;
	while (Have[h] or Need[n]) {
	    if (Have[h] == Need[n]) {
		/* They equal each other. */
		h++, n++;
		continue;
	    }
	    else if (Have[h] == 0 or Have[h] > Need[n]) {
		/* We're missing this one. */
		cid = Need[n++];
		obj = NameDeclare(dir, PidToName(cid), container_typstr, static_storage, NAME_OVERLOAD);
		*(container_id*)((StaticNamedobj*)obj)->location = cid;
	    }
	    else {
		/* We have an extraneous one. */
		cid = Have[h++];
		obj = DirectoryFindContainerPtr(dir, cid);
		assert(obj != NULL);
		*(container_id*)((StaticNamedobj*)obj)->location = 0;
		NameDelete(dir, obj);
	    }
	}

	/* Clean up: */
	free(anon_heap, Have);
	free(anon_heap, Need);
}


static Directory* StdlibDir()
{
	return stdlib_dir;
}



/*----------------- Bitmap stuff: -------------------*/


static void CompileBitmapStuff(void)
{
	Compile("Place::Place(int _x, int _y)"
		"{ x = _x; y = _y; } ");

	Compile("void Bitmap::Init(int X, int Y)"
		"{ cx = X, cy = Y; mDC = NULL; Bmp = NULL; }");
	Compile("void Bitmap::Clear(int col)"
		"{ BM_Clear(this,col); }");
	Compile("void Bitmap::Clear()"
		"{ BM_Clear(this,0); }");
	Compile("void Bitmap::Line(int x0,int y0,int x1,int y1, int col)"
		"{ BM_Line(this, x0,y0,x1,y1, col); }");
	Compile("void Bitmap::Line(Place A, Place B, int col)"
		"{ BM_Line(this, A.x,A.y, B.x,B.y, col); }");
	Compile("void Bitmap::Rect(int x0,int y0,int x1,int y1, int col)"
		"{ BM_Rect(this, x0,y0,x1,y1, col); }");
	Compile("void Bitmap::Arrow(Place A, Place B, int col)"
		"{ BM_Arrow(this, A.x,A.y, B.x,B.y, col); }");
	Compile("void Bitmap::SetPixel(int x, int y, int col)"
		"{ BM_SetPixel(this, x,y, col); }");
	Compile("void Bitmap::Display()"
		"{ BM_Display(this); }");
	Compile("ostream& Bitmap::Print(ostream &out)"
		"{ out << BM_Output(this); return out; }");
}




/*----------------- BGraph stuff: -------------------*/


static void CompileBGraphStuff(void)
{
	Compile("BGraph::BGraph()"
		"{ bg = _bgraph_new(); }");
	Compile("void BGraph::Free()"
		"{ _bgraph_free(bg); }");
	Compile("void BGraph::AddVertex(void* v, str name, int color)"
		"{ _bgraph_addvertex(bg, v, name, color&15); }");
	Compile("void BGraph::AddVertex(int v, str name, int color)"
		"{ _bgraph_addvertex(bg, *(void**)&v, name, color&15); }");
	Compile("void BGraph::AddEdge(void* from, void* to, int color)"
		"{ _bgraph_addedge(bg,from,to, color&15); }");
	Compile("void BGraph::AddEdge(int from, int to, int color)"
		"{ _bgraph_addedge(bg, *(void**)&from, *(void**)&to, color&15); }");
	Compile("Bitmap BGraph::ToBitmap()"
		"{ Bitmap B;   *(void**)&B = _bgraph_to_bitmap(bg);   return B; }");
	Compile("ostream& BGraph::Print(ostream& out)"
		"{ void* bm;  bm = _bgraph_to_bitmap(bg);  out << BM_Output(bm); "
		" BM_Free(bm);  return out; }");
}




/*----------------- Type/Any/NamedObj/Directory classes: -----------------*/

static void CompileRTTIStuff(void)
/* Compile all the code to do with Run-Time Type Information. */
{
	/* The 'any' stuff: */
	Compile("type any::Type() { return t; }");
	Compile("void* any::Address() { return value; }");
	Compile("any any::ArrayLookup(int i) "
		      "{ return *AnyArrayLookup(*this, i); }");
	Compile("int any::ArrayLength() "
		      "{ return AnyArrayLength(*this); }");
	Compile("any any::FieldLookup(Namedobj *obj) "
		      "{ return *AnyFieldLookup(*this, obj); }");
	Compile("any any::FieldLookup(str s) "
		      "{ return *AnyFieldLookupS(*this, s); }");
	Compile("any any::Dereference() { return *AnyDereference(*this); }");
	Compile("any any::FunctionCall(any *Params) "
		      "{ return *AnyFunctionCall(this, Params); }");


	/* The 'type' stuff: */
	Compile("char type::Expand(type_expansion* tyex)"
		      "{ return TypeExpand(ts, tyex); }");
	Compile("type type::NextParam()"
		      "{ type E; E.ts = TypeNextParam(ts); return E; }");
	Compile("type type::operator=(str s)"
		      "{ ts = s; return *this; }");
	Compile("void* type::New()"
		      "{ return TypeNew(ts); }");


	/* The 'Namedobj' stuff: */
	Compile("Namedobj* Namedobj::Next() { return NamedObjNext(this); }");
	Compile("str Namedobj::Name() { return NamedObjName(this); }");
	Compile("any Namedobj::Any() { return *NamedObjAny(this); }");
	Compile("bool Namedobj::Exists() { return this != NULL; }");
	Compile("Namedobj* Namedobj::Dependency(int n) { return NamedObjDependency(this, n); }");
	Compile("ostream& Namedobj::Print(ostream& out) { return NamedObjPrint(out, this); }");


	/* The 'directory' stuff: */
	/*Compile("any directory::FindAny(str name) "
		    "{ return DirFindObj(*this, name); }");*/
	Compile("Namedobj* directory::Find(str name) "
		    "{ return NamedObjFromString(this, name); }");
	Compile("Namedobj* directory::Create(str name, type t) "
		    "{ return DirCreateObj(*this, name, *(char**)&t); }");
	Compile("void directory::Delete(str name) "
		    "{ DirDeleteObj(*this, name); }");
	Compile("ostream& directory::Print(ostream& out) "
		    "{ dir(*this); return out; }");
	Compile("ostream& ostream::operator<<(bool b) { puts(b?\"yes\":\"no\"); return *this; }");
	Compile("ostream& ostream::operator<<(str s) { puts(s); return *this; }");
	Compile("ostream& ostream::operator<<(char ch) { putch(ch); return *this; }");
	Compile("ostream& ostream::operator<<(unsigned char ch) { putch(ch); return *this; }");
	Compile("ostream& ostream::operator<<(int n) { printf(\"%d\", n); return *this; }");
	Compile("ostream& ostream::operator<<(long n) { printf(\"%ld\", n); return *this; }");
	Compile("ostream& ostream::operator<<(unsigned int n) { printf(\"%uU\", n); return *this; }");
	Compile("ostream& ostream::operator<<(double g) { printf(\"%f\", g); return *this; }");
	Compile("ostream& ostream::operator<<(void *v) { printf(\"0x%08x\", (int)v); return *this; }");
	Compile("ostream& ostream::operator<<(type t) { return TypePrint(*this, t); }");
}


static void AddMacro(str name, int value)
/* Create a macro of the ascii form of value. */
{	char buf[512];

	itoa(value, buf, 10);
	MacroEnter(name, -1, buf);
}


static void LibraryInitFunctions(void)
/* Set up the intrinsic functions. */
{

	/******************************************\
	**                                        **
	** Setting up the intrinsic functions:    **
	**                                        **
	\******************************************/

	Add("CreateContainer",	CreateContainer,Dir, Cid,Cip,_);
	Add("OpenContainer",	OpenContainer,  Dir, Cid,Boo,_);
	Add("CloseContainer",	CloseContainer, Voi, Cid,_);
	Add("DeleteContainer",	DeleteContainer,Boo, Cid,_);
	Add("any_project",	L_any_project,	Vpt, ANy,Ty1,_);
	Add("dynarray_access",	L_dynarray,	Vpt, Vpt,Int,Int,_);
	Add("dynarray_len",	L_dynarray_len, Int, Vpt,_);

        Add("assert",   L_assert,       Voi, Boo, _);
	Add("malloc",   malloc_user,    Str, Int,_);
	Add("calloc",   L_calloc,       Str, Int,Int,_);
	Add("realloc",  L_realloc,      Str, Str,Int,_);
	Add("free",     L_free,         Voi, Vpt,_);
	Add("operator new", operator_new_user, Vpt, Str,_);
	Add("operator delete", free_user, Voi, Vpt,_);
	Add("strcpy",   L_strcpy,       Str, Str,Str,_);
	Add("strcat",   L_strcat,       Str, Str,Str,_);
	Add("strdup",   strdup_user,    Str, Str,_);
	Add("strlen",   strlen,         Int, Str,_);
	Add("strcmp",   strcmp,         Int, Str,Str,_);
	Add("stricmp",  stricmp,	Int, Str,Str,_);
	Add("strchr",   (str (*)(str,int))strchr,         Str, Str,Int,_);
	Add("strncpy",  L_strncpy,      Str, Str,Str,Int,_);
	Add("getch",    L_getch,        Int, _);
	Add("gets",     L_gets,         Str, Str,_);
	Add("fgets",    L_fgets,        Str, Str,Int,Fil,_);
	Add("fputs",    fputs,          Int, Str,Fil,_);
	Add("fgetc",    fgetc,          Int, Fil,_);
	Add("fputc",    fputc,          Int, Int,Fil,_);
	Add("feof",     L_feof,         Boo, Fil,_);
	Add("fopen",    fopen,          Fil, Str,Str,_);
	Add("fseek",	fseek,		Int, Fil,Lng,Int,_);
	Add("fscanf",	fscanf,		Int, Fil,Str,Ell,_);
        Add("fflush",   fflush,         Int, Fil, _);
	Add("fclose",   fclose,         Int, Fil,_);
	Add("printf",	printf,		Int, Str,Ell,_);
	Add("fprintf",	fprintf,	Int, Fil,Str,Ell,_);
	Add("sprintf",	sprintf,	Int, Str,Str,Ell,_);
	Add("vsprintf",	vsprintf,	Int, Str,Str,Arg,_);
	Add("sscanf",	sscanf,		Int, Str,Str,Ell,_);
	Add("fscanf",	fscanf,		Int, Fil,Str,Ell,_);
	Add("fread",	fread,		Int, Vpt,Int,Int,Fil,_);
	Add("fwrite",	fwrite,		Int, Vpt,Int,Int,Fil,_);
        Add("ftell",    ftell,          Lng, Fil,_);
	Add("strerror",	strerror, 	Str, Int,_);
	Add("L_errno",	L_errno, 	Int, _);
	Add("YieldToWindows", L_YieldToWindows, Voi, _);
	AddMacro("SEEK_SET", SEEK_SET);
	AddMacro("SEEK_CUR", SEEK_CUR);
	AddMacro("SEEK_END", SEEK_END);
	Add("isalpha",  isalpha,        Int, Int,_);
	Add("isalnum",  isalnum,        Int, Int,_);
	Add("isdigit",  isdigit,        Int, Int,_);
	Add("isspace",  isspace,        Int, Int,_);
	Add("isupper",  isupper,        Int, Int,_);
	Add("islower",  islower,        Int, Int,_);
	Add("toupper",  toupper,        Chr, Int,_);
	Add("tolower",  tolower,        Chr, Int,_);
	Add("itoa",     L_itoa,         Str, Int,Str,Int,_);
	Add("atoi",     atoi,           Int, Str,_);
	Add("rand",     rand,           Int, _);
	Add("srand",    srand,          Voi, Int,_);
	Add("memcpy",   L_memcpy,       Vpt, Vpt,Vpt,Int,_);
	Add("memmove",  L_memmove,      Vpt, Vpt,Vpt,Int,_);
	Add("memset",   L_memset,       Vpt, Vpt,Int,Int,_);
	Add("memcmp",   memcmp,         Int, Vpt,Vpt,Int,_);
	Add("getchar",	L_getch,	Chr, _);
	Add("putchar",	L_putch,	Int, Int,_);
	Add("putch",	L_putch,	Int, Int,_);
	Add("puts",	L_puts,		Int, Str,_);
	Add("fabs",	L_fabs,		Dou, Dou,_);
	Add("floor",	L_floor,	Dou, Dou,_);
	Add("pow",	L_pow,		Dou, Dou,Dou,_);
	Add("log",	L_log,		Dou, Dou,_);
	Add("log10",	L_log10,	Dou, Dou,_);
	Add("exp",	L_exp,		Dou, Dou,_);
	Add("dir",    	L_dir_c,        Voi, Voi,_);
	Add("dir",    	DirectoryPrint, Voi, Dir,_);
	Add("dir",    	DirectoryPrintC,Voi, Cid,_);
	Add("DirFindObj",	DirFindObj,  ANy, Dir,Str,_);
	Add("DirCreateObj",	DirCreateObj,Nob, Dir,Str,Ty1,_);
	Add("DirDeleteObj",	DirDeleteObj,Voi, Dir,Str,_);
        Add("PathToNamedobj",	NamePathToObj,Nob,Str,_);
	Add("Stdlib",	StdlibDir,	Dir, _);
	Add("move",     MoveObj,        Voi, Str,Str,_);
	Add("copy",     CopyObj,        Voi, Str,Str,_);
	Add("size_of",  L_size_of,      Int, Ty2,_);
        Add("GenericArrayConstructor", GenericArrayConstructor,
        				Voi, Vpt,Int,Int,Vpt,_);
        Add("GenericArrayConstructor2", GenericArrayConstructor2,
        				Vpt, Vpt,Int,Vpt,_);
        Add("GenericArrayDestructor", GenericArrayDestructor,
        				Voi, Vpt,Vpt,_);
	Add("cd",   	Chdir,          Voi, Dir,_);
	Add("cd",   	L_cd_str,       Voi, Str,_);
	Add("cd",   	L_cd_container, Voi, Cid,_);
	Add("PathToNob",NamePathToObj,	Nob, Str,_);
	Add("pwd",	L_pwd,		Str, _);
	Add("pwc",	L_pwc,		Cid, _);
	Add("dump",     Conim::DumpConims,Voi, _);
	Add("exit",     L_exit,         Voi, Int,_);
	Add("time",     time,           Lng, Lop,_);
	Add("time",     L_time,         Lng, _);
	Add("ctime",    ctime,          Str, Lop,_);
	Add("edit_",    JoinEditSource, Voi, Str,_);
	Add("compile",  UserCompileString, Vfp, Str,_);
	Add("src_del_",	Source_src_del, Voi, Str,_);
	Add("src_mv_",	Source_src_mv,  Voi, Str,Dir,_);
	Add("operator<<",  PrtPtr,	Ost, Ost, Vpt,_);
	Add("BM_New",   BM_New,         Vpt, Int,Int,_);
	Add("BM_Free",  BM_Free,        Voi, Vpt,_);
	Add("BM_Clear", BM_Clear,       Voi, Vpt,Int,_);
	Add("BM_SetPixel",BM_SetPixel,  Voi, Vpt,Int,Int,Int,_);
	Add("BM_Line",  BM_Line,        Voi, Vpt,Int,Int,Int,Int,Int,_);
	Add("BM_Rect",  BM_Rect,        Voi, Vpt,Int,Int,Int,Int,Int,_);
	Add("BM_Arrow", BM_Arrow,       Voi, Vpt,Int,Int,Int,Int,Int,_);
	Add("BM_Bubble",BM_Bubble,      Voi, Vpt,Str,Int,Int,Int,Int,Int,_);
	Add("BM_BubbleIntersect",   BM_BubbleIntersect,
					Voi, Vpt,Str,Int,Int,Int,Int,Arg,_);
	Add("BM_Display",BM_Display,    Voi, Vpt,_);
	Add("BM_Output",BM_Output,      Str, Vpt,_);
	Add("Help_",    L_Help1,        Voi, Ty2,_);
	Add("Help2_",   L_Help2,        Voi, Ty2,_);
	Add("TypeExpand",       TypeExpand,             Int, Ty1,Vpt,_);
	Add("TypeNextParam",    TypeNextParam,          Ty1, Ty1,_);
	Add("TypePrint",        TypePrint,              Ost, Ost,Ty2,_);
        Add("PrintClass",	L_PrintClass,	        Voi, Ty2,_);
	Add("TypeNew",          TypeNew,                Anp, Vpt,_);
	Add("AnyType",          AnyType,                Ty1, ANy,_);
	Add("AnyArrayLookup",   AnyArrayLookup,         Anp, ANy,Int,_);
	Add("AnyFieldLookup",   AnyFieldLookup,         Anp, ANy,Nob,_);
	Add("AnyFieldLookupS",  AnyFieldLookupS,        Anp, ANy,Str,_);
	Add("AnyDereference",   AnyDereference,         Anp, ANy,_);
	Add("NamedObjNext",     NamedObjNext,           Nob, Nob,_);
	Add("NamedObjName",     NamedObjName,           Str, Nob,_);
	Add("NamedObjAny",      NamedObjAny,            Anp, Nob,_);
	Add("_BreakIn",		BreakIn,	        Voi, Str,_);
	Add("stop",		stop,	                Voi, Voi,_);

	Add("AnyFunctionCall",	    any_function_call,	Anp, Anp,Anp,_);
	Add("AnyArrayLength",	    AnyArrayLength,     Int, ANy,_);
	Add("NamedObjDependency",   NamedObjDependency,	Nob, Nob,Int,_);
	Add("NamedObjFromString",   NamedObjFromString,	Nob, Vpt,Str,_);
	Add("NamedObjPrint",	    NamedObjPrint,      Ost, Ost,Nob,_);

	SetParamCheckerFn(Stdlib->directory()->Find("printf"), 'f');
	SetParamCheckerFn(Stdlib->directory()->Find("sprintf"), 'f');
	SetParamCheckerFn(Stdlib->directory()->Find("fprintf"), 'f');
	SetParamCheckerFn(Stdlib->directory()->Find("scanf"), 's');
	SetParamCheckerFn(Stdlib->directory()->Find("sscanf"), 's');
	SetParamCheckerFn(Stdlib->directory()->Find("fscanf"), 's');

        NamedobjStruct->ptrfinder = (PtrFinder_fn)FindPtrsInNamedobj;
        StaticNamedobjStruct->ptrfinder = (PtrFinder_fn)FindPtrsInStaticNamedobj;
        MacroNamedobjStruct->ptrfinder = (PtrFinder_fn)FindPtrsInMacroNamedobj;
        FunctionNamedobjStruct->ptrfinder = (PtrFinder_fn)FindPtrsInFunctionNamedobj;
        ClassdefStruct->ptrfinder = (PtrFinder_fn)FindPtrsInClassdef;




	/******************************************\
	**                                        **
	** Stuff requiring compilation:           **
	**                                        **
	\******************************************/


#if 1
	CompileRTTIStuff();
#else
	Compile("ostream& ostream::operator<<(str s) { puts(s); return *this; }");
	Compile("ostream& ostream::operator<<(char ch) { putch(ch); return *this; }");
	Compile("ostream& ostream::operator<<(int n) { printf(\"%d\", n); return *this; }");
	Compile("ostream& ostream::operator<<(unsigned int n) { printf(\"%u\", n); return *this; }");
	Compile("ostream& ostream::operator<<(double g) { printf(\"%f\", g); return *this; }");
	Compile("ostream& ostream::operator<<(void *v) { printf(\"0x%08x\", v); return *this; }");
#endif

	Compile("#define and &&");
	Compile("#define or ||");
	Compile("#define not !");
	Compile("#define until(c) while (not (c))");
	Compile("#define forever while (1)");
	Compile("#define yes true");
	Compile("#define no false");
	Compile("#define Help(o) Help_(typeof(o))");
	Compile("#define Help2(o) Help2_(typeof(o))");
	Compile("#define edit(o) edit_(#o)");
	Compile("#define del(o) src_del_(#o)");
	Compile("#define mv(o) src_mv_(#o)");
	Compile("#define mkdir(d) container d = MkdirContainer(d,curcid())");
	Compile("#define BreakIn(d) _BreakIn(#d)");
	Compile("#define each_dir_entry(d)   Namedobj o=DirectoryFirst(d);"
		      " o.Exists(); o=NamedObjNext(o)");
	Compile("#define each_member(tyex)   Namedobj o=tyex.obj;"
		      " o.Exists(); o=NamedObjNext(o)");
	Compile("#define extern");
	Compile("#define errno	L_errno()");
	Compile("#define rmdir(d) RmdirContainer(d); src_del_(#d)");
	Compile("typedef long time_t;");
	CompileAndRun("const int EOF=-1;");
	CompileAndRun("const str endl=\"\n\";");

	/* Bitmaps: */
	CompileBitmapStuff();
	CreatePredefinedType("Bitmap", &BitmapStruct, bitmap_typstr);
	Namedobj* obj = LibraryDeclare("stdout", file_typstr, static_storage);
	*(void**)(((StaticNamedobj*)obj)->location) = &cout;
	obj = LibraryDeclare("stderr", file_typstr, static_storage);
	*(void**)(((StaticNamedobj*)obj)->location) = &cout;
	Add("GraphOfCalls",     L_GraphOfCalls,     Voi, Str,_);
	Add("GraphOfDepends",   L_GraphOfDepends,   Voi, Str,_);
	Add("GraphOfContainers",L_GraphOfContainers,Voi, Voi,_);
	Add("LineCount",        L_LineCount,        Int, _);
	Add("greps",            GrepS,              Voi, Str,_);
	Add("grepo",            GrepO,              Voi, Str,_);
	Add("grepd",            GrepD,              Voi, Nob,_);
	Add("_bgraph_new",      GraphNew,           Vpt, _);
	Add("_bgraph_free",     GraphFree,          Voi, Vpt,_);
	Add("_bgraph_addvertex",GraphAddVertex,     Voi, Vpt,Vpt,Str,Int,_);
	Add("_bgraph_addedge",  GraphAddEdge,       Voi, Vpt,Vpt,Vpt,Int,_);
	Add("_bgraph_to_bitmap",GraphToBitmap,      Vpt, Vpt,_);
	CompileBGraphStuff();

	//not needed:  CreatePredefinedType("BGraph", &BGraphStruct, BGraph_typstr);
	Add("clear",		EditorClearLog,		Voi, _);
	Add("curcid",		L_curcid,		Cid, _);
	Add("CreateContainerCid",CreateContainerCid,	Cid, Cid,_);
	Add("MkdirContainer",	MkdirContainer,		Cid, Cid,Cid,_);
	Add("RmdirContainer",	RmdirContainer,		Boo, Cid,_);
	Add("FixChildContainers",FixChildContainers,	Voi, _);
	Add("MoveContainer",	MoveContainerId,	Boo, Cid,Cid,_);
        Add("getContainerHeader", L_getContainerHeader, Vpt, Voi, _);
}







/*------------------ Initialising the Standard Library: -------------------*/

interface void StdlibInit(void)
/* Create a special Conim and heap for all standard types */
/* and library functions.  Then compile or otherwise      */
/* create all those library functions. */
{       unsigned char place_typstr[5], bitmap_typstr[5], bgraph_typstr[5];

	/* Initialise the Stdlib conim: */
	InitProvisionalTypstrs();
	Stdlib = Conim::CreateContainer(-1);
	curdir = stdlib_dir = Stdlib->directory();
	curconim = Stdlib;
	default_heap = Stdlib;
	PredefinitionPhase = yes;

        /* Assign fixed addresses to all the meta-class's Classdef objects. */
        /* This is necessary to ensure that as we develop Barbados, the     */
        /* Stdlib pointers within legacy persistent stores are not          */
        /* invalidated.  (It also avoids the need for double compilation    */
        /* of meta-classes for getting the right links).   IMPORTANT:       */
        /* don't change the sizes assigned or order or starting address.    */
        /* Do anything rather than invalidate all existing ptrs. */
        AnyStruct = MakeMetaClass(1532, any_typstr, 8);
        TyexStruct = MakeMetaClass(764, tyex_typstr, 24);
        TypeStruct = MakeMetaClass(764, typetype_typstr, 4);
        ClassdefStruct = MakeMetaClass(1020, classdef_typstr, 40);
        NamedobjStruct = MakeMetaClass(1532, namedobj_typstr, 25);
        StaticNamedobjStruct = MakeMetaClass(764, staticnamedobj_typstr, 32);
        MacroNamedobjStruct = MakeMetaClass(764, macronamedobj_typstr, 32);
        FunctionNamedobjStruct = MakeMetaClass(764, functionnamedobj_typstr, 48);
        DirecStruct = MakeMetaClass(1532, direc_typstr, 20);
        OstreamStruct = MakeMetaClass(2044, ostream_typstr, 20);
        ContainerHeaderStruct = MakeMetaClass(764, containerheader_typstr, 44);
        FileStruct = MakeMetaClass(380, file_typstr, 24);
        PlaceStruct = MakeMetaClass(380, place_typstr, 8);
        BitmapStruct = MakeMetaClass(1532, bitmap_typstr, 16);
        BGraphStruct = MakeMetaClass(1532, bgraph_typstr, 4);
        #define MakeTypedef(name, Struct, type) 	Struct->typedef_obj = \
                        NameDeclare(stdlib_dir, name, type, typedef_storage, 0);
        MakeTypedef("any", AnyStruct, any_typstr);
        MakeTypedef("type_expansion", TyexStruct, tyex_typstr);
        MakeTypedef("type", TypeStruct, typetype_typstr);
        MakeTypedef("classdef", ClassdefStruct, classdef_typstr);
        MakeTypedef("Namedobj", NamedobjStruct, namedobj_typstr);
        MakeTypedef("StaticNamedobj", StaticNamedobjStruct, staticnamedobj_typstr);
        MakeTypedef("MacroNamedobj", MacroNamedobjStruct, macronamedobj_typstr);
        MakeTypedef("FunctionNamedobj", FunctionNamedobjStruct, functionnamedobj_typstr);
        MakeTypedef("directory", DirecStruct, direc_typstr);
        MakeTypedef("ostream", OstreamStruct, ostream_typstr);
        MakeTypedef("ContainerHeader", ContainerHeaderStruct, containerheader_typstr);
        MakeTypedef("FILE", FileStruct, file_typstr);
        MakeTypedef("Place", PlaceStruct, place_typstr);
        MakeTypedef("Bitmap", BitmapStruct, bitmap_typstr);
        MakeTypedef("BGraph", BGraphStruct, bgraph_typstr);

	/* Create all the keywords as objects in stdlib: */
        ConstructKeywordsInStdlib(Stdlib);

	/* Compile all standard types: */
	LibraryInitMetaClasses();

	/* Compile all standard functions: */
	LibraryInitFunctions();

        /* Check that the meta-classes get the right addresses: */
        AssertMetaClass("any", 0xb1f004);
        AssertMetaClass("classdef", 0xb1c004);
        AssertMetaClass("directory", 0xb1d004);
        AssertMetaClass("Namedobj", 0xb1c404);
        AssertMetaClass("StaticNamedobj", 0xb1f904);
        AssertMetaClass("FunctionNamedobj", 0xb1cd04);

        /* Finished: */
	PredefinitionPhase = no;
	Stdlib->Assert();
}


