/******************* SOURCE *****************************\
** Summary: This sub-module manages the source code for	**
**          the objects in each directory.		**
**							**
\********************************************************/


#include <string.h>
#include <stdio.h>
#include "barbados.h"
#include "modules.h"
#include "memory.h"
#include "source.h"
#include "array.h"
#include "name.h"
#include "make.h"


static const str SRC = "SRC";







/*-------------------- The .SRC subdirectory: -------------------*/

static Directory* rootdir;
static bool want_to_create;
static Conim* this_conim;


static Directory* SourceDirectoryRecurse(Directory* dir)
{       Directory* src;
	Namedobj* obj;
	Conim *conim;

	/* Can we bottom out now? */
	assert(dir != NULL);
	if (dir == rootdir) {

	    /* We're at the directory at the root of this container. */
	    obj = dir->Find(SRC);
	    if (obj) {
		if (not TypeEqual(obj->type, direc_typstr)) {
		    ErrorRun("%s is not a directory", SRC);
		    return NULL;
		}
		return (Directory*)objLocation;
	    }
	    else if (not want_to_create)
		return NULL;
	    else {
		obj = NameDeclare(dir, SRC, direc_typstr, static_storage, 0);
		return (Directory*)objLocation;
	    }
	}


	/* Check we're still in the same container: */
	conim = Ptr_to_conim(dir->parent);
	if (conim != NULL and conim != this_conim) {
	    ErrorRun("We leave this container");
	    if (Ptr_to_conim(dir->parent) != this_conim)
		ErrorRun("We leave this container");
	    return NULL;
	}


	/* Look for 'dir's corresponding src directory: */
	src = SourceDirectoryRecurse(dir->parent);
	if (src == NULL)
	    return NULL;
	obj = src->Find("$$$");
	if (obj != NULL) {
	    if (not TypeEqual(obj->type, direc_typstr))
		goto DECLARE;
	    return (Directory*)objLocation;
	}
	else if (not want_to_create)
	    return NULL;
	else {
	    DECLARE:
	    obj = NameDeclare(src, "$$$", direc_typstr, static_storage, 0);
	    return (Directory*)objLocation;
	}
}


interface Directory* SourceDirectory(Directory* dir, bool create)
/* Find the subdirectory of the SRC container which */
/* corresponds to this directory. */
{
	/* Get the directory at the root of this container: */
	this_conim = Ptr_to_conim(dir);
	if (this_conim == NULL) {
	    ErrorRun("Loading SRC:  invalid directory pointer.");
	    return NULL;
	}
	rootdir = this_conim->directory();
	want_to_create = create;
	return SourceDirectoryRecurse(dir);
}






/*-------------------- Storing Source away: -------------------*/

/* This sub-module is all about stashing away source-code strings to    */
/* one side to allow us to exit the compiler normally and only then     */
/* link in source-code.  It's all about saving the compiler from being  */
/* re-entrant.  We assume that one compileable which declares multiple  */
/* objects will have one source-code string but multiple debug-objects  */
/* (if functions). */


#define MAX_STASH   200
struct {
	Namedobj* obj;
	Classdef* owner;
} Later[MAX_STASH];
static int L_idx;



static bool ObjInStash(Namedobj* obj)
/* Is this Namedobj in our short-term stash? */
{
	for (int i=0; i < L_idx; i++) {
	    if (Later[i].obj == obj)
		return yes;
	}
	return no;
}


void SourceStashRename(Namedobj* oldobj, Namedobj* newobj)
/* 'oldobj' has been moved to address 'newobj'. */
{
	for (int i=0; i < L_idx; i++) {
	    if (Later[i].obj == oldobj) {
		Later[i].obj = newobj;
                return;
            }
	}
}


static str SourceStore(str name, str source, bool make_copy)
/* Store this source-text. We will either create a new copy of it, */
/* (like strdup), or we will create a string pointing to this src. */
/* It returns the source that is given to the object.              */
{	char type_buf[8], *type;
	Directory* src;
	Namedobj* obj;
	int dimension;
	str r;


	/* Create the .SRC directory: */
	src = SourceDirectory(curdir, yes);
	if (src == NULL)
	    return NULL;


	/* Create the source object: */
	if (make_copy) {

	    /* Set up the source as an array of chars */
	    type = type_buf;
	    *type++ = tp_array;
	    dimension = strlen(source) + 1;
	    PutDimension(dimension, type);
	    *type++ = tp_char;
            HeapCheck(0);
	    obj = NameDeclare(src, name, (Type)type_buf, static_storage, 0);
            HeapCheck(0);
	    r = strcpy((str)objLocation, source);
            HeapCheck(0);
	}
	else {

	    /* Set up the source as a char* to the given source. */
	    type = type_buf;
	    *type++ = tp_pointer;
	    *type++ = tp_char;
	    obj = NameDeclare(src, name, (Type)type_buf, static_storage, 0);
	    *(str*)objLocation = r = source;
	}


	/* Link in all the source in SRC with the compiled objects: */
	SourceLinkSource(curconim, yes);

	return r;
}



interface str SourceFromObject(Namedobj* obj)
/* Gets the source corresponding to this object. */
/* But be warned:  it's possible for 'obj' to be */
/* moved during this operation!!                 */
/*      Also be warned that if you try to get the    */
/* source of a non-function class member, you'll */
/* get an 'assert' error. */
{       Make *make;

	/* Hopefully, the source is already linked in: */
	if (obj->make)
	    return obj->make->source.s;

	/* We'll link in the source.  However, we need to get a 'make-node' */
	/* as a persistent pointer to this object as it gets recompiled.    */
	obj->make = make = MakeEntityNew(obj);
	SourceLinkSource(curconim, no);
	assert(obj->make == make);
	if (make->source.s == NULL) {
	    MakeDelete(make);
	    obj->make = NULL;
	    return NULL;
	}
	else return make->source.s;
}


interface void SourceClearStashed(void)
/* Clear the list of recently-created objects. */
{
	L_idx = 0;
}


interface void SourceStash(Namedobj* obj, Classdef* owner)
/* Store up this tuple so that later they can be added to */
/* the source directory.  This is done because the source */
/* directory needs to call DeclareVariable() to store     */
/* source, and we want to avoid that function being       */
/* re-entrant. */
{
	if (PredefinitionPhase)
	    return;
	if (CurrentSourceIsPersistent) {
	    /* We must be compiling from an installed source-str already. */
	    if (obj->make) {
		obj->make->source.s = CurrentSource;
		obj->make->source.dir = curdir;
	    }
	}
	else {
	    /* Store off this object to be updated later when the */
	    /* full source is there. */
	    if (L_idx >= MAX_STASH)
		ErrorMake("Too many objects in one declaration");
	    else {
		Later[L_idx].obj = obj;
		Later[L_idx].owner = owner;
		L_idx++;
	    }
	}
}



interface void SourceStoreStashed(void)
/* In 'Later[]' we have a list of objects declared during the */
/* last compilation.  Here we take the source-text that       */
/* created them, and store it.   (The source text is given by */
/* 'CurrentSource').  */
{       Classdef* owner;
	Namedobj* obj;
	char buf[1024];
	str uname;

	while (L_idx > 0) {
	    --L_idx;
	    obj = Later[L_idx].obj;
	    owner = Later[L_idx].owner;

	    uname = NameUnameFromObj(obj, buf);
	    CurrentSource = SourceStore(uname, CurrentSource, yes);
	    CurrentSourceIsPersistent = yes;

	    if (obj->make) {
                HeapCheck(0);
                assert(obj->make->obj == obj);
		obj->make->source.s = CurrentSource;
		obj->make->source.dir = curdir;
                HeapCheck(0);
	    }
	}
}





/*------------------ Making Whole containers: -----------------*/

static str GetStringFromObj(Namedobj* obj)
/* If this is an array of chars, or a pointer to char, */
/* return the str it points to.  Otherwise return   */
/* NULL. */
{       Type type=obj->type;
	int dimension;

	if (*type == tp_array) {
	    type++;
	    GetDimension(dimension, type);
	    if (*type != tp_char)
		return NULL;
	    else return (str)objLocation;
	}
	else if (*type == tp_pointer) {
	    type++;
	    if (*type++ != tp_char)
		return NULL;
	    else return *(str*)objLocation;
	}
	else return NULL;
}


static void RecursivelyCompile(Directory* srcdir, Directory* dir, char errmsg[512])
/* We are given a source directory to compile.  */
/* Since it may contain subdirectories, compile */
/* it recursively.   NB :- Upon return, the     */
/* caller must restore the current directory.   */
{       Directory* subsrc, *subdir;
	Namedobj* obj;
	uint h;
        str s;

	Chdir(dir);

	for (each_dir_obj(srcdir)) {
	    s = GetStringFromObj(obj);
	    if (s != NULL and not ObjInStash(obj)) {
		CurrentSourceIsPersistent = yes;
		MakeTimeAdvance();
		Compile(s, curdir, 'M');
                if (Error.err and not *errmsg)
                    strcpy(errmsg, Error.message);
	    }
	}

	for (each_dir_obj(srcdir)) {
	    if (IsDirectory(obj, no, no)) {
		subsrc = *(Directory**)objLocation;
		subdir = dir->FindDirectory(obj->name);
		if (subsrc != NULL and subdir != NULL)
		    RecursivelyCompile(subsrc, subdir, errmsg);
	    }
	}
}


static void AddMakeNode(str uname, Directory* dir)
/* Add a make-node to the object whose 'uname' is 's' and */
/* which is in the current directory. */
{       Namedobj* obj;

	obj = dir->ObjFromUname(uname);
	if (obj != NULL and obj->make == NULL)
	    obj->make = MakeEntityNew(obj);
}


static void RecursivelyAddMakeNodes(Directory* srcdir, Directory* dir)
/* Add a make-node to every object here that has a source str. */
{       Directory* subsrc, *subdir;
	Namedobj* obj;
	uint h;
	str s;

	for (each_dir_obj(srcdir)) {
	    if (strchr(obj->name, ':'))
		continue;
	    s = GetStringFromObj(obj);
	    if (s != NULL and not ObjInStash(obj))
		AddMakeNode(obj->name, dir);
	}

	for (each_dir_obj(srcdir)) {
	    if (IsDirectory(obj, no, no)) {
		subsrc = *(Directory**)objLocation;
		subdir = dir->FindDirectory(obj->name);
		if (subsrc != NULL and subdir != NULL)
		    RecursivelyAddMakeNodes(subsrc, subdir);
	    }
	}
}


interface void SourceLinkSource(Conim *conim, bool CreateIfNeeded)
/* Link in the "SRC" for this container, and compile everything in it	*/
/* so that all the objects in this container have their make nodes.	*/
{       Directory* temp_dir, *root_dir, *src_dir;
        char errmsg[512];

        if (conim->src == (void*)0x1) {
            if (not CreateIfNeeded)
                return; // We tried previously to link, but nothing there.
        }
	else if (conim->src)
	    return;     // It's already linked in.

	temp_dir = curdir;
	root_dir = conim->directory();
	src_dir = root_dir->FindDirectory(SRC);
	if (src_dir == NULL) {
            if (not CreateIfNeeded) {
                conim->src = (Directory*)0x1;
                return;
            }
	    /* The SRC directory needs to be created. */
	    Namedobj* obj = NameDeclare(curdir, SRC, direc_typstr, static_storage, 0);
	    conim->src = (Directory*)objLocation;
	    return;
	}
	conim->src = src_dir;

	/* Give make-nodes to everything that has source: */
	MakeEntityReset();
	RecursivelyAddMakeNodes(src_dir, root_dir);

	/* Compile all source: */
	OstreamPrintf(cout, "<< Linking SRC (ie. compiling everything)... >>\n");
        *errmsg = '\0';
	RecursivelyCompile(src_dir, root_dir, errmsg);
	Chdir(temp_dir);
        if (*errmsg)
            OstreamPrintf(cout, "%s\n", errmsg);
}


interface bool SourceNeedsLinkSource()
/* Do we need to call LinkSource() on curdir's Conim?	*/
/* The answer is yes if curconim doesn't already	*/
/* have src linked. */
{
	return curconim->src == NULL;
}








/*-------------------- User functions: --------------------*/

static bool MemberFunctionName(str name, str classname)
/* Is 'name' of the form: 'classname::???'? */
{       str s,t;

	s = name;
	t = classname;
	while (*t)
	    if (*s++ != *t++)
		return no;
	return (s[0] == ':' and s[1] == ':');
}


static bool SourceDeleteClass(Namedobj* typedef_obj)
/* Remove this class:  */
/* - typedef_obj       */
/* - member fns        */
/* - class source      */
/* - member fn source  */
{	Namedobj* obj, **list=NULL;
	Directory* srcdir;
	str name=NULL;
	uint h;
	int i;

	/* Delete the class source: */
	srcdir = SourceDirectory(curdir, no);
	if (srcdir) {
	    for (each_dir_obj(srcdir)) {
		if (MemberFunctionName(obj->name, name))
		    Array_Add(list, obj);
	    }
	    for (each_aeli(obj, list))
		NameDelete(srcdir, obj);
	    Array_Free(list);
	    obj = srcdir->Find(name);
	    if (obj)
		NameDelete(srcdir, obj);
	}

	/* Now delete the class itself: */
	NameDeleteClass(typedef_obj);

	return yes;
}


static bool SourceDeleteEnum(Namedobj* enum_obj)
/* Remove this enum:   */
/* - typedef_obj       */
/* - value objects     */
/* - source            */
{	Namedobj* obj, **list=NULL;
	Directory* srcdir;
	uint h;
	int i;

	/* First deal with the source: */
	srcdir = SourceDirectory(curdir, no);
	if (srcdir) {
	    obj = srcdir->Find(enum_obj->name);
	    if (obj)
		NameDelete(srcdir, obj);
	}

	/* First get a list of all the objects we want to delete from curdir: */
	for (each_dir_obj(curdir)) {
	    if ((obj->storage == const_storage or obj->storage == typedef_storage)
		    and TypeEqual(obj->type, enum_obj->type))
		Array_Add(list, obj);
	}

	/* Now delete them: */
	for (each_aeli(obj, list))	// This 2-step process is necessary because
	    NameDelete(curdir, obj);	// deletion interferes with the iterator.
	Array_Free(list);

	return yes;
}


interface bool Source_src_del(str name)
/* Delete this object and its associated source. */
{       bool done_something=no;
	Directory* src;
	Namedobj* obj;

	/* Try to delete the object itself: */
	obj = NamePathToObj(name);
	if (obj) {
	    if (obj->storage == static_storage or obj->storage == straight_fn) {
		NameDelete(curdir, obj);
		done_something = yes;
	    }
	    else if (obj->storage == typedef_storage) {
		if (NameIsOriginalClass(obj))
		    return SourceDeleteClass(obj);
		else if (obj->storage == typedef_storage and obj->type[0] == tp_enumerated)
		    return SourceDeleteEnum(obj);
		NameDelete(curdir, obj);
		done_something = yes;
	    }
	}

	/* Now delete its source code: */
	src = SourceDirectory(curdir, no);
	if (src) {
	    obj = src->Find(name);
	    if (obj) {
		NameDelete(src, obj);
		done_something = yes;
	    }
	}

	/* Finish up: */
	if (not done_something)
	    ErrorRun("Couldn't find any \"%s\"", name);
	return done_something;
}


interface bool Source_src_mv(str name, Directory* dir)
/* Move this object and its associated source to the new directory. */
{
	ErrorRun("Move %s to dir not implemented", name);
	return no;
}


interface str SourceHeader(Namedobj* obj)
/* Return the first comment from the source of this object. */
/* Returns NULL if it can't find anything. */
{       static char buf[256];
	str s,d;

	/* Get the source: */
	s = SourceFromObject(obj);
	if (s == NULL)
	return NULL;

	/* Find the start of the first comment: */
	for (; *s; s++) {
	    if (s[0] == '/' and s[1] == '*')
		break;
	    if (s[0] == '/' and s[1] == '/')
		break;
	    if (*s == '{')
		return NULL;
	}
	if (*s == '\0')
	    return NULL;

	/* Find the end of this comment: */
	for (d=s+1; *d and d-s < sizeof(buf)-2; d++) {
	    if (s[1] == '/') {
		/* C++ comments: */
		if (d[0] == '\n') {
		    if (memcmp(d, "\n//", 5) == 0)
			continue;       /* The comment continues */
		    else break;
		}
	    }
	    else {
		/* C comments: */
		if (d[0] == '*' and d[1] == '/') {
		    if (memcmp(d, "*/\n/*", 5) == 0)
			continue;       /* The comment continues */
		    else break;
		}
	    }
	}

	/* Copy it into the return buffer: */
	if (*d == '\0')
	    return NULL;
	else if (*d == '*')
	    d += 2;
	memcpy(buf, s, d-s);
	buf[d-s] = '\0';
	return buf;
}




/*---------------- Mapping obj's to directories: ----------------*/

interface Directory* ObjectToDirectory(Namedobj* obj)
/* Find the directory that this object belongs in. */
{
	assert(obj->storage != auto_storage and obj->storage != member_storage);
	return (Directory*)obj->owner;
}


