/* $Id: rowcolumn.h 618 2005-10-26 06:32:35Z jla $ */


// RowColumn.h:   This class is used for displaying tabular reports.
//              As such, it bears some similarities to GridWin:  you need
//    to know what the differences are in order to know which one
//    to use:
//  Use a RowColumnWin if you have a very large number of rows of the
//    same format.
//  Use a GridWin if you need functionality more like a spreadsheet.
//
//
//  User Instructions:
//  ------------------
//  RowColumn was modeled on the Windows Explorer 'Detail' view and the
//  MS Outlook screen.  You have a set of default columns, plus a large
//  set of possible columns.  You can add columns, hide columns and re-order
//  columns:  the user adds a column by right-clicking on any column
//  heading (actually, the place where he wants it to be put) and selecting
//  from the popup menu which field to add.  The user hides a column by
//  right-clicking on the heading and selecting 'hide'.  The user
//  rearranges columns by dragging the heading from one place to another.
//    You can also sort the rows according to each column:  click
//  on a column heading or right-click and select 'Sort'. This will sort
//  all rows according to that column, (ascending or descending as the
//  application-writer deems appropriate).  Click again to get the reverse
//  order.  There are actually 3 sort columns: primary, secondary and
//  tertiary, but by clicking in this way you get to move them down and
//  put a new column as the primary sort column.
//
//  Application-writer Instructions:
//  --------------------------------
//      See the sample program below.
//

#ifndef TFC_ROWCOLUMNWIN
#define TFC_ROWCOLUMNWIN

#include <stdio.h>
#include <map>
#include "excel.h"
#include "list.h"

#define NEARPIXEL 4

typedef int column_id;    // Identifies one possible column as defined
        // by the application-writer.

class RowColExpDlg {
public:
    RowColExpDlg(class RowColumnWin* pWin);
    ~RowColExpDlg();

    void    ConstructUI();
    int     DisplayDialog();
    int*    GetExportOrderColumnIds() {return mExportOrderColIds;}
private:

    str*    mUnselectedHeadings;
    str*    mSelectedHeadings;
    int*    mExportOrderColIds;

    char*   mUnselected;
    char*   mSelected;        

    RowColumnWin* pColWin;

    control mUnselectedList;
    control mSelectedList;
    control mMainControl;
    control mExportButton;

    void    SelectButtonClicked();
    void    UnselectButtonClicked();
    void    UpButtonClicked();
    void    DownButtonClicked();
    void    ExportButtonClicked();

    static void SelectClicked(RowColExpDlg* dlg);
    static void UnselectClicked(RowColExpDlg* dlg);
    static void UpClicked(RowColExpDlg* dlg);
    static void DownClicked(RowColExpDlg* dlg);
    static void ExportClicked(RowColExpDlg* dlg);
};



class RowColumnWin : public ScrollWin {

    friend class Report;
    friend class Excel;

public:
    struct SortKey {
        double f;
        const char* s;
        char transition;
        bool strdup_s;
        void *row;
        // If you override the sortkey functions, you can treat {f,s}
        // as 12 bytes of data, or use the default interpretation.
    };

private:
    bool bHighlightOnMouse2;
    bool mRepaintOnMousestroke;
    bool mCalibrateUsingFullWin;
    bool mCalibrateAllCols;
    bool mAllowSorting;
    int summaryFg,summaryBg;
    RowColExpDlg* mpExportDialog;

    void PaintCell(void* row, column_id cid, int x1, int x2, int y, char status);
    void PaintRowViaVirtual(int j);
    void AddCol_1int(int cid8i);
    str MakePopupName(int cid, char tmp[30], str prefix);
    void PaintRow(void *row, int y, int x0, int x1);
            // This is private because we can't trust application programmers
            // to set 'paint_rownum' before calling this.  Use PaintRow(int j) instead.
    void InSituSort(SortKey **AA, int n, int sortcolm);
    void DoSort(SortKey **AA, int a, int b, int sortary);

protected:
    /* Options */
    int LeftmostAlwaysVisible;  // Stops leftmost column scrolling off.
    int RowJumpers;              // AutoScroll when within 'RowJumpers' from bottom(top)

    /* Extra pass-in parameters for GetField: */
    int paint_x;            // What column number is being painted?
    int paint_rownum;       // What row is being painted?

    /* Extra pass-out parameters for GetField: */
    double number;          // What number?
    char exceltype;         // n)umber, $)currency, d)ate or t)ime.
                            // Dates are represented as days since 1/1/1900,
                            // which equals a Smarts date + 24481.

    /* Local Fields */
    column_id BeingDragged; // If we're re-ordering a column.
    bool      BeingResized; // If we're re-sizing a column.
    bool      HeadingClicked;
    int       FirstShifted;
    int *X;                 // A dynamic array of column x-start values (n+1 entries for n columns, X[0]==0)
    int *U;                 // A dynamic array of booleans: has the user resized this row?
    void* CurrentRow;       // What row has focus?
    int   CurrentRowId;     // What's the index of the focus row?
    int one;                // The width of a line in pixels
    char searchstring[40];  // For searching
    str summary;
    int tooltip_id, tooltip_cid, tooltip_tim;
    int _x, _y;
    void **Rows;            // The list of rows, a TFC dynamic array.

    TfcGenericMenuItem *RightButtonMenu(int x, int y);
    virtual TfcGenericMenuItem *ExtraColList(int i);

public:
    bool FromMousestroke;
    TfcFont font;           // The font used for all text fields except where overridden.
    TfcFont cellfont;       // To override 'font', set 'cellfont' within GetField().
    int ColumnGap;          // How many pixels between columns?
    int RowHeight;          // The height in pixels of each row
    int SummaryHeight;      // The height of the summary section
    int HeadingHeight;      // The height of the heading row
    int TopHeight;          // = SummaryHeight + HeadingHeight
    bool HighlightedOnly;   // While printing - Print only highlighted rows
    bool AllowNoSelection;  //
    bool AllowNoMultiSelection;
    column_id *ColumnIds;   // The column_id of each column
    column_id sort[3];      // sort[0]=primary, sort[1]=secondary, sort[2]=tertiary
    column_id currSort;     //++ (jla) The column currently being sorted --
    bool ForExcel;          // Strip out formatting, we're exporting to Excel.
    void **HighlightedRows; // ==All selected rows


    RowColumnWin(str caption, int width, int height, int flag = 0, ScrollWin *Owner=NULL);
    RowColumnWin(RowColumnWin& orig);

	virtual RowColumnWin* Clone() { return NULL; }

    /* Adding rows: */
    void SetRows(void** _Rows); // Install a new or updated dynamic-array of rows.
                                    // The RowColumnWin will hereafter own this list.
    void AddRow(void* Row);         // Add one row
    // Important: The RowColumnWin owns the 'Rows' array, but does not
    // own the rows.  You need to destruct the rows in your application code.
    void DeleteRow(void* Row);

    bool isHighlighted(int row) ;
    bool isHighlighted(void* row) ;
    char rowStatus(void* row);
    int  NumRows(void) { return ListSize(Rows); }
    void* GetRow(int i) { if (i < 0 or i >= ListSize(Rows))
                    return NULL; else return Rows[i]; }
    void Clear();
    void HighlightAll();

    void SetFocusRow(void *row);
    bool MoveToRow(int i, bool Highlight=no);       // i==-1 means the 'no focus' state
    bool HighlightRow(int i, bool clearprevioushighlight); /* (aha) MoveToRow programatically - like CTRL+Click */
    bool SetRowWithoutRedraw(int i);

    void LockCol(column_id cid);
    virtual void HideCol(column_id cid);
    virtual void AddCol(column_id cid, int i);
    void SortOn(column_id cid);
    void HelpOn(column_id cid);
    void MoveColumn(column_id cid, int newpos, bool moveMouse = false);
    void ResizeColumn(column_id cid, int dx);
    
    void Calibrate(bool AllowShrinkage);    // Re-calculate all column widths
    
    void Export(str filename = NULL)
    { Export(filename, NULL, (ListSize(HighlightedRows) > 1 ? HighlightedRows : Rows), NULL); }
               // Exports table to Excel.
               // if filename == NULL opens Excel application with data
               // if filename == "__clipboard__" copies data to clipboard
               // otherwise saves data as .xls file

    void Export(str filename, void ** headerRows, void ** footerRows)
    { Export(filename, headerRows, (ListSize(HighlightedRows) > 1 ? HighlightedRows : Rows), footerRows); }

    void Export(str filename, void ** headerRows, void ** bodyRows, void ** footerRows);
                /* Added this function so you could specify an optional header/footer rows */

    void SetSummary(str summary, int Colour, int Background);
            // This displays a paragraph-sized string at the top of the screen,
            // above even the heading line.  To clear it, pass text=NULL.
    void SaveAsCsv(str givenfilename=NULL); // Separator is default (semi-colon)
    bool SaveAsCsvWithSeparator(char separator, str filename=NULL)
    { return SaveAsCsvWithSeparator(separator, filename, false, false); }
    bool SaveAsCsvWithSeparator(char separator, str filename, bool useUnicode, bool printRaw);
    void PaintHeadingRow(int x2);
    void** GetHighlightedData();
    void*  GetFocusRow();
    int  GetFocusRowInt() { return CurrentRowId; }
    int  YtoI(int y) { return (y-TopHeight)/RowHeight; }
    void* YtoRow(int y) { int i = YtoI(y); if (i < 0 or i >= ListSize(Rows))
                    return NULL; else return Rows[i]; }
    int  XtoCid(int x);
    int  RowToI(void *row) { return ListFindP(Rows, row); }
    void FocusRowIntoView();
    virtual void ReadConfigDefCols() {}
    virtual void SetColumnsToDefaults();
    void SetColumnsToDefaultsAndRedraw();
    virtual void ShowAllColumns();
    void StringToColumns(str buf, bool call_default = true);
    virtual str ColumnsToString(str buf);
    void StringToSorts(str buf);
    str  SortsToString(str buf);
    void StringToColumnWidths(str buf);
    str ColumnWidthsToString(str buf);

    char* SortAndReturnTransitions(bool bFocusRowIntoView=false);
    virtual void Sort(bool bFocusRowIntoView=false);
    virtual void MakeSortKey(void* row, column_id cid, SortKey *sokey);
    virtual int  CompareSortKey(column_id cid, SortKey *A, SortKey *B);
    virtual void FreeSortKey(column_id cid, SortKey *A);
            // You can override either 'MakeSortKey' or 'CompareSortKey' or both.
    
    void ExpandRightmostColumn();   // Rightmost column expands to clientWidth.
	void SetUnscrollableAreas();
    void SetLeftmostColumnAlwaysVisible(int set=1);// Stops it scrolling off.
    void SetRowJumpers(int set);
    void PaintRow(int j);
	void PaintObj(void *obj);		// Repaint the row 'obj'.
    void Print();        
    void ClearSelection();			// Clear highlighted rows plus focus row
    void ClearHighlight();			// Clear highlighted rows
    void Search(str s);             // Moves focus row to any matching row.
    void Search(bool Continuation); // Ask for the string and call Search(s)
    void ShowTooltip();
    virtual bool Mousemove(int x, int y);
    void SetHighlightOnMouse2(bool bHighlight); // (awa) Will this rowcolumnwin
	                                    // Highlight on second mouse press
    void SetRepaintOnMousestroke(bool setting) {mRepaintOnMousestroke = setting;}
    void SetCalibrateUsingFullWin(bool setting) {mCalibrateUsingFullWin = setting;}
    void SetCalibrateAllCols(bool setting) {mCalibrateAllCols = setting;}        
    void SetAllowSorting(bool setting) {mAllowSorting = setting;}

    virtual ~RowColumnWin();

    //----- Subclasses MUST define these functions: -----
    virtual kstr GetField(void* row, column_id cid, char dest[],
            char status /* F)ocus, H)ighlight, X)=both, ' '=normal. */,
            int *fgcolour, int *bgcolour,
            char *alignment/*'L','R' or 'C'*/) = 0;
            // 'row', 'cid', 'status' and 'dest[]' are the pass-in parameters.
            // 'paint_rowno' is an implicit pass-in parameter, useful for
            // painting a 'Row number' column.
            // 'fgcolour', 'bgcolour', 'alignment', 'number' and 'exceltype'
            // are the pass-out parameters.
            // 'number' and 'exceltype' are implicit pass-out parameters:
            // set them if you want to have numerical sorts, or you want
            // to have exact values exported to Excel.
    virtual kstr GetHeading(column_id cid, kstr *helpp, char *statusp) = 0;
            // status:  'D'=on by default, 'N'=off by default, '\0'=not applicable.
            // Returns the heading name as the formal return value.
            // When it returns NULL, then we know we have all the possible columns.

    //----- Subclasses can optionally define these functions: -----
    virtual void PaintField(void* row, column_id cid, char status,
            int x0,int y0,int x1,int y1) { }
            // If it's a text field, they'll return the string via
            // GetField() - either a pre-existing string, or they
            // construct a string in dest[].   If it's a graphics
            // field, then GetField() will return NULL and PaintField
            // will be called instead.
    virtual void PerRowPaint(void* row, int x0, int y0, int x1, int y1) {}
            // Override this function to add functionality to draw extra
            // graphics for each row specifically
            // Added to allow DeltaWindow to use this to draw a line
    virtual int FieldWidth(void *row, column_id cid) { return 10; } // for graphics fields
    virtual int  GetFieldWidth(void* row, column_id cid);
    virtual bool Keystroke(int key);        // Litewin functions...
    virtual bool Mousestroke(int op, int x, int y);
    virtual void Paint(int x0, int y0, int x1, int y1);
    virtual void CursorPaint();
    virtual void Measure(int *xp, int *yp);
    virtual void Resized();
    virtual void CalcHeight();
    virtual bool DescendingByDefault(column_id cid) { return no; }
	virtual void DecideOnPageBreakPosX(int x, int *endX, int *startOfNextX);
	virtual void DecideOnPageBreakPosY(int y, int *endY, int *startOfNextY);
};

/* Sample program:

class ArithWin : public RowColumnWin {
public:
    ArithWin();
    virtual str GetField(void* row, column_id cid, char dest[], char status,
                    int *fgcolor, int *bgcolor, char *alignment);
    virtual kstr GetHeading(column_id cid, kstr *helpp, char *statusp);
};


ArithWin::ArithWin()
    : RowColumnWin("Arithmetic", 400,400)
{
    SetColumnsToDefaults();         // Must be called here,
    // after we have a validly constructed ArithWin.
}


struct Sum {
    int a,b;
    Sum(int a, int b) { this->a = a; this->b = b; }
};


kstr ArithWin::GetHeading(column_id cid, kstr *helpp, char *statusp)
{
    *helpp = "Not available";
    *statusp = 'D';
    switch (cid) {
        case 0:     return "X";
        case 1:     return "Y";
        case 2:     return "X+Y";
        default:    return NULL;    // NULL=no more columns
    }
}


str ArithWin::GetField(void* row, column_id cid, char dest[], char status,
      int *fgcolor, int *bgcolor, char *alignment)
{
    Sum* sum=(Sum*)row;
    *fgcolor = WHITE;
    *bgcolor = (status != ' ') ? DARK(BLUE) : BLACK;
    *alignment = 'R';
    switch (cid) {
        case 0:     sprintf(dest, "%d", sum->a);
                    break;
        case 1:     sprintf(dest, "%d", sum->b);
                    break;
        case 2:     sprintf(dest, "%d", sum->a+sum->b);
                    break;
    }
    return dest;
}


int main(int argc, str argv[])
{
    ArithWin A;
    A.AddRow(new Sum(4,5));
    A.AddRow(new Sum(14,50));
    A.AddRow(new Sum(60,11));
    A.AddRow(new Sum(6,110));
    A.EventLoop();
    return 0;
}

*/

#endif
