/* $Id: canvas.cpp 437 2005-10-14 05:00:17Z jla $ */


#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <memory.h>
#include <math.h>
#include <ctype.h>
#include <set>
#include "tfc.h"
#include "list.h"
#include "canvas.h"






class Quatree {
public:
        TfcRect Rect;
        CanvasObj **ObjList;
        class Quatree *Q[2][2];

        Quatree(int x1, int y1, int x2, int y2);
        Quatree();
        /* Constructors. */

        void AddObj(CanvasObj *obj);
        /* Insert a canvas obj. */

        void DelObj(CanvasObj *obj);
        /* Delete a canvas obj. */

        bool Intersects(TfcRect *Rect);
        bool Intersects(int x, int y);
        /* Does this intersect Rect? */

        CanvasObj** AddToDrawList(TfcRect *Rect, CanvasObj** List);
        /* Add to 'List' every object in this that intersects Rect. */
        /* Don't worry if we get duplicates in the list - we expect */
        /* this. */

        CanvasObj** AddToPointList(int x, int y, CanvasObj** List);
        /* Add to 'List' every object in this that intersects (x,y). */
        /* Don't worry if we get duplicates in the list - we expect */
        /* this. */

        ~Quatree();
};


Quatree::Quatree(int x1, int y1, int x2, int y2)
{
        clearS(*this);
        Rect.top = y1;
        Rect.left = x1;
        Rect.right = x2;
        Rect.bottom = y2;
        ObjList = NULL;
        for(int i = 0; i < 2; i++)
        {
            for(int j = 0; j < 2; j++)
                Q[i][j] = NULL;
        }
}


Quatree::Quatree()
{
        clearS(*this);
        Rect.top = 0;
        Rect.left = 0;
        Rect.right = 0;
        Rect.bottom = 0;        
        ObjList = NULL;
        for(int i = 0; i < 2; i++)
        {
            for(int j = 0; j < 2; j++)
                Q[i][j] = NULL;
        }
}


bool Quatree::Intersects(TfcRect *R)
{
        if (R->left > Rect.right)
            return no;
        if (R->right < Rect.left)
            return no;
        if (R->top > Rect.bottom)
            return no;
        if (R->bottom < Rect.top)
            return no;
        return yes;
}


bool Quatree::Intersects(int x, int y)
{
        return x >= Rect.left and x <= Rect.right
                and y >= Rect.top and y <= Rect.bottom;
}


void Quatree::AddObj(CanvasObj *obj)
{       int centre_x, centre_y;
        int obj_area, q_area;

        assert(obj->width >= 0 and obj->width < 1e7);
        assert(obj->height >= 0 and obj->height < 1e7);
        obj_area = obj->width * obj->height;
        if (obj_area == 0)
            return;
        q_area = (Rect.bottom - Rect.top) * (Rect.right - Rect.left);
        if (obj_area * 4 > q_area) {
            /* If it's a significant proportion of this quatree's area, */
            /* then place it into this quatree. */
            ListAdd(ObjList, obj);
        }
        else {
            centre_x = (Rect.right + Rect.left) / 2;
            centre_y = (Rect.top + Rect.bottom) / 2;

            short x1,y1,x2,y2;

            x1 = obj->x;
            y1 = obj->y;
            x2 = x1 + obj->width;
            y2 = y1 + obj->height;
            if (y1 <= centre_y and x1 <= centre_x) {
                if (Q[0][0] == NULL)
                    Q[0][0] = new Quatree(Rect.left, Rect.top,
                                centre_x, centre_y);
                Q[0][0]->AddObj(obj);
            }
            if (y1 <= centre_y and x2 >= centre_x) {
                if (Q[0][1] == NULL)
                    Q[0][1] = new Quatree(centre_x, Rect.top,
                                Rect.right, centre_y);
                Q[0][1]->AddObj(obj);
            }
            if (y2 >= centre_y and x1 <= centre_x) {
                if (Q[1][0] == NULL)
                    Q[1][0] = new Quatree(Rect.left, centre_y,
                                centre_x, Rect.bottom);
                Q[1][0]->AddObj(obj);
            }
            if (y2 >= centre_y and x2 >= centre_x) {
                if (Q[1][1] == NULL)
                    Q[1][1] = new Quatree(centre_x, centre_y,
                                Rect.right, Rect.bottom);
                Q[1][1]->AddObj(obj);
            }
        }
}


void Quatree::DelObj(CanvasObj *obj)
{       extern int L;

        if (this == NULL)
            return;
        else if (ListHasP(ObjList, obj)) {
            ListDelP(ObjList, obj);
        }
        else {

			// (awa) It seems when spread is in almas, the zoomer object sometimes
			// isn't removed from the quatree properly. This causes problems when we
			// delete it later in spread then change days which then causes canvas to
			// try and do a delete on a deleted object.
			//
			// I'll get Tim and Natalija to investigate this later (or if your Tim or natalija
			// reading this then please talk to me).

            /*int centre_x = (Rect.right + Rect.left) / 2;
            int centre_y = (Rect.top + Rect.bottom) / 2;

            short x1 = obj->x;
            short y1 = obj->y;
            int x2 = x1 + obj->width;
            int y2 = y1 + obj->height;
            if (x1 > Rect.right or y1 > Rect.bottom
                    or x2 < Rect.left or y2 < Rect.top)
                return;*/
            //if (y1 <= centre_y and x1 <= centre_x)
                Q[0][0]->DelObj(obj);
            //if (y1 <= centre_y and x2 >= centre_x)
                Q[0][1]->DelObj(obj);
            //if (y2 >= centre_y and x1 <= centre_x)
                Q[1][0]->DelObj(obj);
            //if (y2 >= centre_y and x2 >= centre_x)
                Q[1][1]->DelObj(obj);
		}
}


CanvasObj** Quatree::AddToDrawList(TfcRect *Rect, CanvasObj** List)
/* Add to 'List' every object in this that intersects Rect. */
/* Don't worry if we get duplicates in the list - we expect */
/* this. */
{       int i;

        if (not this)
            return List;
        if (not Intersects(Rect))
            return List;
        ListConcat(List, ObjList);
        for (i=0; i < 4; i++)
            List = Q[0][i]->AddToDrawList(Rect, List);
        return List;
}


CanvasObj** Quatree::AddToPointList(int x, int y, CanvasObj** List)
/* Add to 'List' every object in this that intersects Rect. */
/* Don't worry if we get duplicates in the list - we expect */
/* this. */
{       int i;

        if (not this)
            return List;
        if (not Intersects(x,y))
            return List;
        ListConcat(List, ObjList);
        for (i=0; i < 4; i++)
            List = Q[0][i]->AddToPointList(x,y, List);
        return List;
}


CanvasObj::CanvasObj(int _x, int _y, int _width, int _height, depth_id _depth,
            int _colour, int _border, TfcCallback _DefaultAction)
{
        depth = _depth;
        colour = _colour;
        border = _border;
        x = _x;
        y = _y;
        width = _width;
        height = _height;
        if (border != NOCOLOUR) {
            width++;
            height++;
        }
        DefaultAction = _DefaultAction;
        parent = NULL;
}


void CanvasObj::Measure(int *widthp, int *heightp)
/* Get the required width and height */
{
        *widthp = width;
        *heightp = height;
}


void CanvasObj::Paint(int x1, int y1, int x2, int y2)
/* To paint a canvas object with proper clipping, */
/* you need to ask the parent to paint this area. */
{       Canvas *Can=(Canvas*)parent;

        Can->Paint(x1 + x, y1 + y, x2 + x, y2 + y);
}


void CanvasObj::GetWinXY(int *xp, int *yp)
/* Interpret 'xy' as (x,y) in Windows coordinates */
{       int px,py;

        parent->GetWinXY(&px,&py);
        *xp = x + px;
        *yp = y + py;
}


bool CanvasObj::GetWinRect(TfcRect *Rect)
/* Get the display rectangle in Windows coordinates */
{       int px,py;

        parent->GetWinXY(&px,&py);
        Rect->left = x + px;
        Rect->top = y + py;
        Rect->right = Rect->left + width;
        Rect->bottom = Rect->top + height;
        return no;
}



void CanvasObj::SetDepth(depth_id _depth)
/* Change the depth to this value. */
{
        if (depth == _depth)
            return;
        depth = _depth;
        if (parent == NULL)
            return;
        PaintWhole();

        Canvas *Can=(Canvas*)parent;
        if (depth < Can->MinDepth)
            Can->MinDepth = depth;
        if (depth > Can->MaxDepth)
            Can->MaxDepth = depth;
}


void CanvasObj::MoveTo(int _x, int _y)
/* Move the object to this new location. */
{       int x1,y1,x2,y2,area1,area2;
        TfcRect Big;

        /* Is it already there? */
        if (x == _x and y == _y)
            return;

        /* What is the original rectangle? */
        x1 = x;
        y1 = y;
        x2 = x1 + width;
        y2 = y1 + height;

        /* Take it out, change the position, put it back: */
        Canvas *Can=(Canvas*)parent;
        Can->DelObjectSimple(this);
        x = _x;
        y = _y;
        Can->AddObjectSimple(this);

        /* Are we suppressing paints? */
        if (Can->ShowState != tfc_shown)
            return;

        /* Calculate the areas with the two methods: */
        Big.top = (y1 < y) ? y1 : y;
        Big.left = (x1 < x) ? x1 : x;
        Big.right = (x2 > x+width) ? x2 : x+width;
        Big.bottom = (y2 > y+height) ? y2 : y+height;
        area1 = (Big.bottom - Big.top) * (Big.right - Big.left);
        area2 = 2 * width * height;

        /* Repaint the relevant areas: */
        if (area1 < area2)
            Can->Paint(Big.left,Big.top, Big.right,Big.bottom);
        else {
            Can->Paint(x1,y1,x2,y2);
            Can->Paint(x,y,x+width,y+height);
        }
}


void Canvas::Initialise()
{
        Q = new Quatree(0,0,256,256);
        MinDepth = 9999;
        MaxDepth = -9999;
        focus = NULL;
        ScrollbarMode = 'N';
        extraObjects = NULL;
}


Canvas::Canvas(str caption, int width, int height)
        : ScrollWin(caption, width, height)
{
        SetCursorBlinkRate(0,0);
        Initialise();
        RepaintOnRefocus = no;
        Background2 = NOCOLOUR;
}

Canvas::Canvas(str caption, int width, int height, int flag)
        : ScrollWin(caption, width, height, flag)
{
        SetCursorBlinkRate(0,0);
        Initialise();
        RepaintOnRefocus = no;
        Background2 = NOCOLOUR;
}


Canvas::Canvas(Canvas& orig)
/* Copy constructor */
        : ScrollWin(orig)
{
        SetCursorBlinkRate(0,0);
        Initialise();
        RepaintOnRefocus = no;
        Background2 = orig.Background2;
}


std::set<CanvasObj *> DestructorList;


Quatree::~Quatree()
{       CanvasObj *obj = NULL;
        int i, j;

        for (each_aeli(obj, ObjList))
            DestructorList.insert(obj);
        ListFree(ObjList);
        for(i = 0; i < 2; i++)
        {
            for(j = 0; j < 2; j++)
                delete Q[i][j];
        }
}


void Canvas::FreeAll()
/* Delete all objects in this canvas. */
{       CanvasObj *obj = NULL;
        int i = 0;

        DestructorList.clear();
        delete Q;
        Q = NULL;
        for (std::set<CanvasObj *>::iterator it = DestructorList.begin();
                    it != DestructorList.end(); ++it)
            delete *it;
        DestructorList.clear();

        for (i = 0; i < ListSize(extraObjects); ++i)
            delete extraObjects[i];
        ListFree(extraObjects);
}


void Canvas::Clear()
{
        FreeAll();
        Initialise();
        realWidth = realHeight = 0;
}


void Canvas::ClearSimple()
{
        delete Q;
        Initialise();
        realWidth = realHeight = 0;        
}


Canvas::~Canvas()
{
        Clear();
        delete Q;
}


void Canvas::AddObjectSimple(CanvasObj *obj)
{
        /* Is it covered by the existing root Quatree? */
        short x2 = obj->x + obj->width;
        short y2 = obj->y + obj->height;
        while (x2 > Q->Rect.right or y2 > Q->Rect.bottom) {
            /* Make the old root the top left-hand Quatree of the new root. */
            Quatree *Parent = new Quatree();
            Parent->Rect.top = 0;
            Parent->Rect.left = 0;
            Parent->Rect.right = Q->Rect.right * 2;
            Parent->Rect.bottom = Q->Rect.bottom * 2;
            Parent->Q[0][0] = Q;
            Q = Parent;
        }

        /* Now that obj is fully inside the root, add it: */
        Q->AddObj(obj);
        obj->parent = this;

        /* Update MinDepth/MaxDepth: */
        if (obj->depth < MinDepth)
            MinDepth = obj->depth;
        if (obj->depth > MaxDepth)
            MaxDepth = obj->depth;

        /* Update realWidth/realHeight: */
        short x = obj->x;
        short y = obj->y;
        if (x + obj->width > realWidth)
            realWidth = x + obj->width;
        if (y + obj->height > realHeight)
            realHeight = y + obj->height;

        // this prevents memory leak
        if (obj->height == 0 || obj->width == 0)
            ListAdd(extraObjects, obj);
}


CanvasObj* Canvas::AddObject(CanvasObj *obj)
{
        assert(Q->Rect.right - Q->Rect.left != 0);
        AddObjectSimple(obj);
        if (ShowState == tfc_shown)
            obj->PaintWhole();
        return obj;
}


void Canvas::DelObjectSimple(CanvasObj *obj)
{
        Q->DelObj(obj);
        if (obj->height == 0 || obj->width == 0)
            ListDelP(extraObjects, obj);
}


void Canvas::DelObject(CanvasObj *obj)
{
        DelObjectSimple(obj);
        if (focus == obj)
            focus = NULL;
        if (ShowState == tfc_shown)
            obj->PaintWhole();  // It won't actually be there,
                                // but it will paint its old area.
        delete obj;
}


static int compar1(CanvasObj **obj1p, CanvasObj **obj2p)
/* Returns the deepest ones first. */
{       CanvasObj *obj1=*obj1p;
        CanvasObj *obj2=*obj2p;

        if (obj1->depth != obj2->depth)
            return obj2->depth - obj1->depth;
        if (obj1->x != obj2->x)
            return obj1->x - obj2->x;
        if (obj1->y != obj2->y)
            return obj1->y - obj2->y;
        return (int)obj1 - (int)obj2;
}


static int compar2(CanvasObj **obj1p, CanvasObj **obj2p)
/* Returns the closest ones first. */
{
        return -compar1(obj1p, obj2p);
}


void Canvas::Paint(int x1, int y1, int x2, int y2)
{       CanvasObj **List = NULL, *obj = NULL, *prev = NULL;
        TfcRect Rect;
        int i;

        // (awa) Cannot draw what isn't here
        if (not Q)
            return;

        /* Get a list of all objects whose extents */
        /* intersect this part of the canvas. */
        Rect.top = y1;
        Rect.left = x1;
        Rect.right = x2;
        Rect.bottom = y2;

        List = Q->AddToDrawList(&Rect, NULL);
        ListSort(List, compar1);

        /* Clip the background: */
        SetClipper();
        if (not inPrint)
            SetClipper(x1,y1,x2,y2);

        /* Blank out the background: */
        DrawBlendRectangle(x1,0,x2,y2>realHeight?y2:realHeight,
                        Background,Background2,y1,y2);
        // Usually this just maps to a simple FillRect() call.

        /* Paint them. */
        prev = NULL;
        for (each_aeli(obj, List)) {
            if (obj == prev)
                continue;
             obj->PaintJustMe(
                        (x1 <= obj->x) ? 0 : x1 - obj->x,
                        (y1 <= obj->y) ? 0 : y1 - obj->y,
                        (x2 >= obj->x+obj->width) ? obj->width : x2-obj->x,
                        (y2 >= obj->y+obj->height) ? obj->height : y2-obj->y);
            prev = obj;
        }
        ListFree(List);
        SetClipper();
}


bool Canvas::Mousestroke(int op, int x, int y)
{       CanvasObj **List, *obj = NULL, *prev;
        int i,lx,ly;

        if (op == 0)
            return no;

        if (op == MOUSE_DRAG and focus) {
            lx = x - focus->x;
            ly = y - focus->y;
            return focus->Mousestroke(op,lx,ly);
        }

        /* Get the list of objects that lie under this point. */
        List = Q->AddToPointList(x,y,NULL);
        ListSort(List, compar2);

        /* Go through the list from top down. */
        prev = NULL;
        for (each_aeli(obj, List)) {
            if (obj == prev)
                continue;
            lx = x - obj->x;            // Local coordinates (local to object)
            ly = y - obj->y;
            if (lx < 0 or lx > obj->width)        // Check that it is at least in the extent.
                continue;
            if (ly < 0 or ly > obj->height)
                continue;
            if (not obj->Intersects(lx,ly))
                continue;
            if ((obj->colour & TFC_TRANSLUCENT) and i+1 < ListSize(List)) {
                CanvasObj *obj2=List[i+1];
                if (obj->width * obj->height < obj2->width * obj2->height
                    and not (obj2->colour & TFC_TRANSLUCENT)) {
                    // If we have a transparent obj in front of a larger non-transparent
                    // obj, swap them.
                    List[i] = obj2, List[i+1] = obj;
                    i--;
                    continue;
                }
            }
            if (not obj->Mousestroke(op,lx,ly)) {
                if ((op == MOUSE_PRESS or op == MOUSE_DOUBLECLICK)
                                and not obj->DefaultAction.IsNull()) {
                    /* We want a double-click to result in two invocations
                    of DefaultAction, e.g. so you can press a 'button' twice
                    in quick succession.  As things stand, the first press
                    is a MOUSE_PRESS and the second one is a MOUSE_DOUBLECLICK. */
                    obj->DefaultAction(this, obj);
                    ListFree(List);
                    return yes;
                }
                ListFree(List);
                return no;
            }
            ListFree(List);
            return yes;
        }
        ListFree(List);
        return no;
}


bool Canvas::Keystroke(int key)
{
        // (awa) If ShowState is not shown, we could be processing a message *before*
        // everything is initialised causing bad crashes.
        if (this->ShowState == tfc_shown and focus) {
            if (focus->Keystroke(key))
                return yes;
            if (key == ENTER) {
                focus->DefaultAction(this,nullcontrol);
                return yes;
            }
            else return no;
        }
        else return no;
}


void Canvas::SetFocus(CanvasObj *obj)
/* Change which object has the focus. */
{       CanvasObj *tmp;

        if (obj == focus)
            return;
        tmp = focus;
        focus = obj;
        if (RepaintOnRefocus) {
            if (tmp)
                tmp->PaintWhole();
            if (obj)
                obj->PaintWhole();
        }
}


interface void Raise(CanvasObj *obj)
{       Canvas *Can=(Canvas*)obj->parent;

        obj->SetDepth(DEPTH(Can->MinDepth-1));
}


interface void Lower(CanvasObj *obj)
{       Canvas *Can=(Canvas*)obj->parent;

        obj->SetDepth(DEPTH(Can->MaxDepth+1));
}





/*---------------------- The objects: ------------------------*/

CanvasRect::CanvasRect(int x, int y, int width, int height, depth_id depth,
                int colour, int border, TfcCallback DefaultAction)
        : CanvasObj(x,y,width,height,depth,colour,border,DefaultAction)
{
}


void CanvasRect::PaintJustMe(int x1, int y1, int x2, int y2)
{
        DrawRectangle(0,0,width,height,colour,border);
}


CanvasCircle::CanvasCircle(int x, int y, int width, int height, depth_id depth,
                        int colour, int border, TfcCallback DefaultAction)
        : CanvasObj(x,y,width,height,depth,colour,border,DefaultAction)
{
}


void CanvasCircle::PaintJustMe(int x1, int y1, int x2, int y2)
{       Canvas *Can=(Canvas*)parent;

        DrawEllipse(0,0,width,height,colour, (Can->focus == this ? WHITE : border));
}


bool CanvasCircle::Intersects(int x, int y)
/* Does this point lie inside this circle? */
{       double dx,dy,w,h;       // 4-byte int's are just not big enough!

        dx = x - width / 2;
        dy = y - height / 2;
        dx *= dx;
        dy *= dy;
        w = width*width;
        h = height*height;
        return dx*h + dy*w < h*w/4;
}


CanvasLine::CanvasLine(int x1, int y1, int x2, int y2,
                        depth_id depth, int colour, int _width,
                        TfcCallback DefaultAction)
        : CanvasObj(x1,y1,x2,y2,depth,colour,NOCOLOUR,DefaultAction)
{
        if (x1 > x2)
            x = x2, width = x1 - x2, from00 = no;
        else x = x1, width = x2 - x1, from00 = yes;
        if (y1 > y2)
            y = y2, height = y1 - y2, from00 = not from00;
        else y = y1, height = y2 - y1;
        linewidth = _width;
        x -= linewidth/2;
        width += linewidth;
        y -= linewidth / 2;
        height += linewidth;
}


void CanvasLine::PaintJustMe(int x1, int y1, int x2, int y2)
{
        if (from00)
            DrawLine(0+linewidth/2,0+linewidth/2,
                width-(linewidth+1)/2,height-(linewidth+1)/2,
                colour,linewidth);
        else
            DrawLine(0+linewidth/2,height-linewidth/2,
                width-(linewidth+1)/2,0+(linewidth+1)/2,
                colour,linewidth);
}


bool CanvasLine::Intersects(int x, int y)
{       int x1,y1,x2,y2,a,b,c,d2;

        if (from00) {
            x1 = 0+linewidth/2;
            y1 = 0+linewidth/2;
            x2 = width-linewidth/2;
            y2 = height-linewidth/2;
        }
        else {
            x1 = 0+linewidth/2;
            y1 = height-linewidth/2;
            x2 = width-linewidth/2;
            y2 = 0+linewidth/2;
        }
        a = y2 - y1;
        b = x1 - x2;
        c = y1*(x2-x1) - x1*(y2-y1);
        d2 = (a*x + b*y + c);
        d2 *= d2;
        d2 /= a*a+b*b;
        return d2 <= linewidth*linewidth;
}


CanvasString::CanvasString(int x, int y, int w, int h, depth_id depth,
                    str text, int foreground, int background,
                    TfcFont _font, int _flags,
                    TfcCallback DefaultAction)
        : CanvasObj(x,y,w,h,depth,foreground,background,DefaultAction)
{
        buf = strdup(text);
        font = _font;
        flags = _flags;
        if (width <= 1 or height <= 1) {
            TextDimensions(text, -1, _font, &w, &h, flags);
            if (width <= 1)
                    width = w;
            if (height <= 1)
                    height = h;
        }
}


void CanvasString::SetText(str txt)
/* Change the string and recalc dimension */
{
        if (buf) {
            free(buf);
            buf = NULL;
        }
        buf = strdup(txt);
        TextDimensions(buf, -1, font, &width, &height, flags);
}


CanvasString::~CanvasString()
{
        free(buf);
        buf = NULL;
}


void CanvasString::PaintJustMe(int x1, int y1, int x2, int y2)
{
        DrawStringU(buf, strlen(buf), font, 0,0,width,height,colour,border,flags);
// was -1
}


CanvasBitmap::CanvasBitmap(int x, int y, int width, int height, depth_id depth,
                        int _bitmap_id, TfcCallback DefaultAction,
                        int fg_colour, int bg_colour)
        : CanvasObj(x,y,width,height,depth,fg_colour,bg_colour,DefaultAction)
{
        bitmap_id = _bitmap_id;
        TfcBitmapDimensions(bitmap_id,0, &sourcewidth, &sourceheight);
}


void CanvasBitmap::MoveAndChange(int _x, int _y, int bitmap_id)
/* Move to this new location and change the shape to 'bitmap'. */
{
        if (_x == x and _y == y and this->bitmap_id == bitmap_id)
            return;             // Nothing to do.
        if (this->bitmap_id != bitmap_id) {
			this->bitmap_id = bitmap_id;
			TfcBitmapDimensions(bitmap_id,0, &sourcewidth, &sourceheight);            
        }
        if (_x == x and _y == y) {
            Canvas *Can = (Canvas*)parent;
            Can->Paint(x,y,x+width,y+height);
        }
        else MoveTo(_x, _y);
}


void CanvasBitmap::PaintJustMe(int x1, int y1, int x2, int y2)
{
        DrawMonoBitmap(0,0,width,height,bitmap_id,colour,border,
                        sourcewidth,sourceheight,0);
}


void CanvasPolygon::ComputeExtent()
{       TfcPoint *P;
        TfcRect R;
        int i;

        R.top = List[0].y;
        R.left = List[0].x;
        R.right = List[0].x;
        R.bottom = List[0].y;
        for (i=0; i < ListLen; i++) {
            P = &List[i];
            if (P->x < R.left)
                R.left = P->x;
            if (P->x > R.right)
                R.right = P->x;
            if (P->y < R.top)
                R.top = P->y;
            if (P->y > R.bottom)
                R.bottom = P->y;
        }
        x = R.left;
        y = R.top;
        width = R.right - R.left;
        height = R.bottom - R.top;
        for (i=0; i < ListLen; i++) {
            P = &List[i];
            P->x -= x;
            P->y -= y;
        }
        if (border != NOCOLOUR) {
            width++;
            height++;
        }
}


bool CanvasPolygon::Intersects(int x, int y)
{       int i, i1, i2, i_1, p1;

        if (!CanvasObj::Intersects(x,y))
            return false;

        p1 = 0;
        for (i=0; i<ListLen; i++) {
            i1 = i + 1;
            if (i1 == ListLen)
                i1 = 0;

            /* check either the edge can be below the point*/
            if (x < List[i].x and x < List[i1].x)
                continue;
            if (x > List[i].x and x > List[i1].x)
                continue;
            if (y > List[i].y and y > List[i1].y)
                continue;

	    /* check special cases*/
	    if (x == List[i].x) {
                i_1 = i == 0 ? ListLen - 1 : i - 1;
	    	i2 = i1;
	    	while (List[i2].x == x)
                    if (++i2 == ListLen)
		        i2 = 0;

                i = i2 == 0 ? ListLen-1 : i2 - 1;
                if (List[i_1].x < x and x < List[i2].x  or
		    List[i_1].x > x and x > List[i2].x)
		    continue;
		i1 = i2;
            }

	    /* check either line down through the point would intersect the edge*/
	    if ((double)List[i].y*(x - List[i1].x)/(List[i].x - List[i1].x) +
                (double)List[i1].y*(x - List[i].x)/(List[i1].x - List[i].x) > y)
                p1++;
        }

        return p1&1;
}


CanvasPolygon::CanvasPolygon(depth_id depth, int colour, int border,
                TfcCallback DefaultAction,
                TfcPoint *_List, int _ListLen)
        : CanvasObj(0,0,0,0,depth,colour,border,DefaultAction)
{
        List = _List;
        ListLen = _ListLen;
        ComputeExtent();
}


CanvasPolygon::CanvasPolygon(depth_id depth, int colour, int border,
                TfcCallback DefaultAction, int x, int y, ...)
        : CanvasObj(0,0,0,0,depth,colour,border,DefaultAction)
{       TfcPoint P[100];
        va_list args;
        int P_idx;

        P[0].x = x;
        P[0].y = y;
        P_idx = 1;
        va_start(args, y);
        do {
            P[P_idx].x = va_arg(args, int);
            if (P[P_idx].x == TFC_ENDOFPOLY)
                break;
            P[P_idx++].y = va_arg(args, int);
        } forever;
        List = (TfcPoint*)malloc(P_idx * sizeof(TfcPoint));
        memcpy(List, P, P_idx*sizeof(TfcPoint));
        ListLen = P_idx;
        ComputeExtent();
}


CanvasPolygon::~CanvasPolygon()
{
        free(List);
}


void CanvasPolygon::PaintJustMe(int x1, int y1, int x2, int y2)
{
        DrawPolygon(colour, border, List, ListLen);
}



CanvasPlot::CanvasPlot(depth_id depth, int colour, int border, int x1, int x2,
                        int *Y, TfcCallback DefaultAction)
        : CanvasObj(0,0,0,0,depth,colour,border,DefaultAction)
{       int i, maxY, minY;

        /* Get the dimensions: */
        maxY = minY = NoNum;
        for (i=x2-x1-1; i >= 0; i--) {
            if (Y[i] == NoNum)
                continue;
            if (Y[i] > maxY or maxY == NoNum)
                maxY = Y[i];
            if (Y[i] < minY or minY == NoNum)
                minY = Y[i];
        }
        if (minY == NoNum)
            minY = maxY = 0;
        x = x1;
        y = minY;
        width = x2 - x1;
        height = maxY - minY;
        if (border != NOCOLOUR)
            width++, height+=2;
        if (height == 0)
            height = 1;

        /* Convert Y to plot coordinates: */
        this->Y = (int*)malloc(width * sizeof(int));
        for (i=0; i < width; i++) {
            if (Y[i] != NoNum)
                this->Y[i] = Y[i] - minY;
            else this->Y[i] = NoNum;
        }
}


void CanvasPlot::PaintJustMe(int x1, int y1, int x2, int y2)
{       int i,prev_y;

        /* Draw the outline first: */
        if (border != NOCOLOUR) {
            prev_y = Y[0];
            for (i=1; i < width; i++) {
                if (prev_y != NoNum and Y[i] != NoNum)
                    DrawLine(i, prev_y==Y[i]?prev_y-1:prev_y, i, Y[i], border, 2);
                prev_y = Y[i];
            }
            if (colour == border)
                return;
        }

        /* Draw the plot: */
        prev_y = Y[0];
        for (i=1; i < width; i++) {
            if (prev_y != NoNum and Y[i] != NoNum)
                DrawLine(i, prev_y, i, Y[i], colour, 1);
            prev_y = Y[i];
        }
}


bool CanvasPlot::Intersects(int x, int y)
/* Does this point lie inside this plot? */
{
        if (x < 0 or x >= width)
            return no;
        if (y >= Y[x] - 3 and y <= Y[x] + 3)
            return yes;
        if (x > 0 and y >= Y[x-1] - 3 and y <= Y[x-1] + 3)
            return yes;
        if (x + 1 < width and y >= Y[x+1] - 3 and y <= Y[x+1] + 3)
            return yes;
        for (int t = 0; t <= 3; t++)
        {
            if (x >= t and x + 1 < width and y >= Y[x+t] and y <= Y[x-t])
                return yes;
            if (x >= t and x + 1 < width and y >= Y[x-t] and y <= Y[x+t])
                return yes;
        }
        return no;
}



CanvasBlot::CanvasBlot(depth_id depth, int colour, int _border, int x1, int x2, ab *Y,
                        TfcCallback DefaultAction)
        : CanvasObj(0,0,0,0,depth,colour,0,DefaultAction)
{       int i, maxY, minY;

        /* Get the dimensions: */
        minY = 9999;
        maxY = -9999;        
        border = _border; 
        for (i=x2-x1-1; i >= 0; i--) {
            if (Y[i].a >= Y[i].b)
                continue;
            if (Y[i].b > maxY and Y[i].b != 9999)
                maxY = Y[i].b;
            if (Y[i].a < minY and Y[i].a != -9999)
                minY = Y[i].a;
        }
        if (maxY == -9999)
            maxY = minY = -1;
        x = x1;
        y = minY;
        width = x2 - x1;
        height = maxY - minY;

        /* Convert Y to plot coordinates: */
        this->Y = (ab*)malloc(width * sizeof(ab));
        for (i=0; i < width; i++) {
            if (Y[i].a == Y[i].b) {
                this->Y[i].a = this->Y[i].b = 0;
                continue;
            }
            this->Y[i].a = Y[i].a - minY;
            this->Y[i].b = Y[i].b - minY;
        }
}


void CanvasBlot::PaintJustMe(int x1, int y1, int x2, int y2)
/* Tim(?): I used to call DrawLine for 1-pixel rectangles but this */
/* stuffs up in print mode because Tfc prints pen-lines darker */
/* than brush-regions. */
/* Tom: the border stuff is for Oslo.  I don't know why   */
/* we aren't using a polygon object, which would make this a bit */
/* cleaner */
{
        /* Draw the plot: */
        if (x1 < 0)
            x1 = 0;
        if (x2 > width)
            x2 = width;
        int *end=(int*)&Y[x2];
        for (int i=x1; i < x2; ) {
            /* Optimise out runs of the same (a,b): */
            int a_comma_b=*(int*)&Y[i];
            int runlength = 0;
            for (int *nxt=(int*)&Y[i+1]; nxt < end and *nxt == a_comma_b; nxt++)
                runlength++;
         
            if (colour!=border)
                DrawRectangle(i, Y[i].a, i+runlength+1, Y[i].b, colour, NOCOLOUR);            

            if (border!=NOCOLOUR) {
                if (i > 0) {
                    if (Y[i-1].a > 0 and Y[i].a > 0)
                        DrawLine(i, Y[i-1].a, i, Y[i].a, border, 1.5);
                    if (Y[i-1].b > 0 and Y[i].b > 0)
                        DrawLine(i, Y[i-1].b, i, Y[i].b, border, 1.5);
                } else DrawLine(i, Y[i].a, i, Y[i].b);
                DrawLine(i, Y[i].a, i+runlength+1, Y[i].a, border, 1.5);
                DrawLine(i, Y[i].b, i+runlength+1, Y[i].b, border, 1.5);           
                if (i + runlength + 1 >= x2)
                    DrawLine(i+runlength+1, Y[i].a, i+runlength+1, Y[i].b, border, 1.5);           
            }


            i += runlength + 1;            
        }        
}


bool CanvasBlot::Intersects(int x, int y)
/* Does this point lie inside this plot? */
{
        if (x < 0 or x >= width)
            return no;
        if (y >= Y[x].a - 1 and y <= Y[x].b + 1)
            return yes;
        return no;
}



/*----------------CanvasSprite ---------------*/

void* TfcGetBitmap(int bitmap_id, int &w, int &h, void* *bmpMsk, int mask_colour);


CanvasSprite::CanvasSprite(int x, int y, depth_id depth, int bitmap_id,
                        int _width/*=0*/, int _height/*=0*/,
                        int transparentCol/*NOCOLOUR means top-left-hand pixel*/,
                        TfcCallback DefaultAction,
                        int type)
        : CanvasObj(x,y,_width,_height,depth,NULL,border,DefaultAction)
{
        bmpImg = TfcGetBitmap(bitmap_id, width, height, &bmpMsk,transparentCol);
        /* This function will modify 'width' and 'height' to the bitmap's */
        /* default size if either is <= 1. */
        this->transp_colour = transparentCol;
        this->type = type;
        this->bitmap_id = bitmap_id;
}


void CanvasSprite::PaintJustMe(int x1, int y1, int x2, int y2)
{
        DrawTransparentBitmap(0,0,width,height,bmpImg,bmpMsk);
}


void CanvasSprite::MoveAndChange(int _x, int _y, int bitmap_id)
/* Move to this new location and change the shape to 'bitmap'. */
{
        if (_x == x and _y == y and this->bitmap_id == bitmap_id)
            return;             // Nothing to do.
        if (this->bitmap_id != bitmap_id) {
            bmpImg = TfcGetBitmap(bitmap_id, width, height, &bmpMsk, transp_colour);
            this->bitmap_id = bitmap_id;
        }
        if (_x == x and _y == y) {
            Canvas *Can = (Canvas*)parent;
            Can->Paint(x,y,x+width,y+height);
        }
        else MoveTo(_x, _y);
}


CanvasSprite::~CanvasSprite()
{
}


CanvasSprite** Canvas::SpriteTouches(TfcRect *rect, CanvasObj *current, int type)
/* Find all other sprites which touch this sprite.  If 'type' is nonzero, */
/* only take sprites of type 'type'. */
{       CanvasObj **List, *obj;
        int s,d;

        /* Get a list of all objects whose extents */
        /* intersect this part of the canvas. */
        List = Q->AddToDrawList(rect, NULL);
        s = d = 0;
        for ( ; s < ListSize(List); s++) {
            obj = List[s];
            if (obj->x > rect->right or obj->x + obj->width < rect->left)
                continue;
            if (obj->y > rect->bottom or obj->y + obj->height < rect->top)
                continue;
            if (obj == current)
                continue;
            if (obj->Type() == 0 or (type and type != obj->Type()))
                continue;
            List[d++] = obj;
        }
        ListSetSize(List, d);
        return (CanvasSprite**)List;
}


CanvasSprite** CanvasSprite::Touches(int type)
/* Find all other sprites which touch this sprite.  If 'type' is nonzero, */
/* only take sprites of type 'type'. */
{       Canvas *Can=(Canvas*)parent;
        TfcRect Rect;

        Rect.top = y;
        Rect.left = x;
        Rect.right = x + width;
        Rect.bottom = y + height;
        return Can->SpriteTouches(&Rect, this, type);
}


CanvasArrow::CanvasArrow(int Ax, int Ay, int Bx, int By,
                int deviation, depth_id depth,
                int colour, char* arrow_style, TfcCallback DefaultAction)
                : CanvasObj(0,0,0,0, depth,colour, NOCOLOUR, DefaultAction)
/* The constructor for CURVED arrows */
{       TfcPoint* pt = NULL;
        double AN2;
        float d;
        int i;

        point = NULL;
        headpoint1 = headpoint2 = NULL;
        linewidth = arrow_style ? *arrow_style - '0' : 1;
        if (linewidth < 1)
            linewidth = 1;
        head1 = arrow_style[1];
        head2 = arrow_style[2];
        d = abs(deviation);
        if (d == 0)
            d = 1;
        ListSetSize(point,2);
        if (deviation > 0) {
            point[0].x = Ax;
            point[0].y = Ay;
            point[1].x = Bx;
            point[1].y = By;
        }
        else {
            point[1].x = Ax;
            point[1].y = Ay;
            point[0].x = Bx;
            point[0].y = By;
        }

        // calculate the radius of the circle:
        AN2 = ((Ax - Bx)*(Ax - Bx) + (Ay - By)*(Ay - By)) / 4;
        r = (d*d + AN2) / (2*d);

        // Calculate the center of the circle (x0, y0)
        if (Ax == Bx) {
            Cx = Ax + ((Ay < By)==(deviation>0) ? r-d : d-r);
            Cy = (Ay + By) / 2;
        }
        else {
            TfcPoint M,dM;
            M.x = (Ax + Bx) / 2;
            M.y = (Ay + By) / 2;
            dM.x = Ax - M.x;
            dM.y = Ay - M.y;
            double g = dM.x * (r-d) / sqrt((double)(dM.x*dM.x + dM.y*dM.y));
            double f = -dM.y * (r-d) / sqrt((double)(dM.x*dM.x + dM.y*dM.y));
            if (deviation < 0)
                g = -g, f = -f;
            Cx = M.x + f;
            Cy = M.y + g;
        }

        // Calculate the extent of the arc:
        int QA, QB;     // Which quadrants are A and B in?
        if (Ax > Cx)
            QA = (Ay > Cy) ? 0 : 1;
        else QA = (Ay > Cy) ? 3 : 2;
        if (Bx > Cx)
            QB = (By > Cy) ? 0 : 1;
        else QB = (By > Cy) ? 3 : 2;
        // There are 4 points: T,L,B,R, corresponding to the top of
        // the circle's bounding rectangle, Left, Bottom and Right.
        // Which of these 4 points does our arc touch?
        enum { T=1, L=2, B=4, R=8 };
        static int TSarray[4][4] = {
                { 0, R, R|T, R|T|L },
                { T|L|B, 0, T, T|L },
                { L|B, L|B|R, 0, L },
                { B, B|R, B|R|T, 0 } };
        int TS;
        if (QA == QB)
            TS = ((Bx > Ax) == (QA==0 or QA==3) == (deviation > 0)) ? 0 : T|L|B|R;
        else TS = TSarray[QA][QB];
        if (deviation < 0)
            TS ^= 0xf;           // If we're going round the reverse
            // direction, then we touch every side we would not touch
            // in the normal direction.
        TfcRect Bound;
        if (Ax > Bx)
            Bound.left = Bx, Bound.right = Ax;
        else Bound.left = Ax, Bound.right = Bx;
        if (Ay > By)
            Bound.top = By, Bound.bottom = Ay;
        else Bound.top = Ay, Bound.bottom = By;
        if (TS & T) {
            if (Cy - r < Bound.top)
                Bound.top = Cy - r;
        }
        if (TS & L) {
            if (Cx - r < Bound.left)
                Bound.left = Cx - r;
        }
        if (TS & B) {
            if (Cy + r > Bound.bottom)
                Bound.bottom = Cy + r;
        }
        if (TS & R) {
            if (Cx + r > Bound.right)
                Bound.right = Cx + r;
        }

        // Shift the points to this new frame:
        x = Bound.left;
        y = Bound.top;
        for (each_oeli(pt,point)) {
            pt->x -= x;
            pt->y -= y;
        }
        Cx -= x;
        Cy -= y;
        Ax -= x;
        Ay -= y;
        Bx -= x;
        By -= y;
        width = Bound.right - Bound.left;
        height = Bound.bottom - Bound.top;

        // Compute arrow heads
        int sgn = (deviation > 0) ? 1 : -1;
        if (IsUTF8Alpha(head1)) {
            TfcPoint p1,p2;
            p1.x = Bx + sgn*(Cy-By);
            p1.y = By - sgn*(Cx-Bx);
            p2.x = Bx;
            p2.y = By;
            headpoint2 = ComputeArrowHead(p1, p2, head2);
        }
        if (IsUTF8Alpha(head2)) {
            TfcPoint p1,p2;
            p1.x = Bx + sgn*(Cy-By);
            p1.y = By - sgn*(Cx-Bx);
            p2.x = Bx;
            p2.y = By;
            headpoint2 = ComputeArrowHead(p1, p2, head2);
        }
}


CanvasArrow::CanvasArrow(TfcPoint* pointArray,
                depth_id depth, int colour, char* arrow_style,   TfcCallback DefaultAction ) :
                CanvasObj(0,0,0,0, depth,colour, NOCOLOUR, DefaultAction)
/*  Constructs a polyline arrow  */
{
        if (ListSize(pointArray) < 2) {
            assert(false);
            return;
        }
        r = 0;
        point = pointArray;
        headpoint1 = headpoint2 = NULL;
        linewidth = *arrow_style - '0';
        if (linewidth < 1)
            linewidth = 1;
        else if (linewidth > 9)
            linewidth = 9;
        Cx = Cy = 0;
        head1 = arrow_style[1];
        head2 = arrow_style[2];

        ComputeExtent();

        headpoint1 = ComputeArrowHead(point[1], point[0], head1);
        headpoint2 = ComputeArrowHead(point[ListSize(point) - 2], point[ListSize(point) - 1], head2);
}


CanvasArrow::CanvasArrow(int xStart, int yStart, int xEnd, int yEnd, depth_id depth,
                         int colour, char* arrow_style,   TfcCallback DefaultAction ) :
			   CanvasObj(xStart, yStart, xEnd,  yEnd, depth,colour, NOCOLOUR, DefaultAction)
/*  Constructs simple straight arrow  */
{
        r = 0;
        width = abs(xEnd  - xStart);
        height = abs(yEnd  - yStart);
        if (width == 0 and height == 0) {
            assert(false);
            return;
        }
        Cx = Cy = 0;
        linewidth = arrow_style ? *arrow_style - '0' : 1;
        if (linewidth < 1)
            linewidth = 1;
        if (arrow_style) {
            head1 = arrow_style[1];
            head2 = arrow_style[2];
        }
        else
            head1 = head2 = 0;

        point = NULL;
        headpoint1 = headpoint2 = NULL;

        ListIdx(point,1);
        point[0].x = xStart;
        point[0].y = yStart;
        point[1].x = xEnd ;
        point[1].y = yEnd ;
        ComputeExtent();
        headpoint1 = ComputeArrowHead(point[1], point[0], head1);
        headpoint2 = ComputeArrowHead(point[ListSize(point) - 2],
                    point[ListSize(point) - 1], head2);
}


CanvasArrow::~CanvasArrow()
{
        ListFree(point);
        ListFree(headpoint1);
        ListFree(headpoint2);
}


TfcPoint* CanvasArrow::ComputeArrowHead(TfcPoint p1, TfcPoint p2, char style)
{       double PI = 3.14159265358979;
        TfcPoint* endpoint = NULL, *p = NULL;
        double alfa, narrowness;
        int i, X = 0, Y = 0;
        double xp1, yp1;
        double xp2, yp2;
        double size;

        if (style >= 'A' and style <= 'F')
            size = style - 'A' + 1, narrowness = PI/6;
        else if (style >= 'a' and style <= 'f')
            size = style - 'a' + 1, narrowness = PI/9;
        else return NULL;
        size *= linewidth;
        size += 2;

        alfa = atan2((double)(p2.y - p1.y), (double)(p2.x - p1.x));
        xp1 = p2.x + 2*size*cos(PI-alfa-narrowness);
        yp1 = p2.y - 2*size*sin(PI-alfa-narrowness);
        xp2 = p2.x - 2*size*cos(alfa-narrowness);
        yp2 = p2.y - 2*size*sin(alfa-narrowness);

        ListSetSize(endpoint,4);
        endpoint[0].x = xp1;
        endpoint[0].y = yp1;
        endpoint[1].x = p2.x - size*cos(alfa);
        endpoint[1].y = p2.y - size*sin(alfa);
        endpoint[2].x = xp2;
        endpoint[2].y = yp2;
        endpoint[3].x = p2.x + (linewidth/2)*cos(-PI/2+alfa);
        endpoint[3].y = p2.y - (linewidth/2)*sin(-PI/2+alfa);

        for (each_oeli(p,endpoint)) {
            X = p->x < X ? p->x : X;
            width = width < p->x ? p->x : width;
            Y = p->y < Y ? p->y : Y;
            height = height < p->y ? p->y : height;
        }

        if (X < 0) {
            for (each_oeli(p,point))
                p->x -= X;
            for (each_oeli(p,endpoint))
                p->x -= X;
            for (each_oeli(p,headpoint1))
                p->x -= X;
            x += X;
            width -= X;
            Cx -= X;
        }

        if (Y < 0) {
            for (each_oeli(p,point))
                p->y -= Y;
            for (each_oeli(p,endpoint))
                p->y -= Y;
            for (each_oeli(p,headpoint1))
                p->y -= Y;
            y += Y;
            height -= Y;
            Cy -= Y;
        }
        return endpoint;
}


bool CanvasArrow::Intersects(int x, int y)
{       int i, aa,bb,cc,x1,x2, y1, y2;
        float d2;

        if (not DefaultAction)
            return no;      // It's not going to do anything anyway,
            // so don't hog the screen space.

        if (r) {        // It's a curved arrow
            d2 = (x - Cx)*(x - Cx) + (y - Cy)*(y - Cy);
            d2 = sqrt(d2);
            d2 -= r;
            if (d2 < 0)
                d2 = -d2;
            return (d2 < linewidth+2);
        }

        for (i=1; i< ListSize(point);i++) {
            x1 = point[i-1].x; y1 = point[i-1].y;
            x2 = point[i].x; y2 = point[i].y;
            if (x1 < x2 and (x < x1 or x > x2))
                continue;
            if (x1 > x2 and (x < x2 or x > x1))
                continue;
            if (y1 < y2 and (y < y1 or y > y2))
                continue;
            if (y1 > y2 and (y < y2 or y > y1))
                continue;
            aa = y2 - y1;
            bb = x1 - x2;
            cc = y1*(x2-x1) - x1*(y2-y1);
            d2 = (aa*x + bb*y + cc);
            d2 *= d2;
            d2 /= aa*aa+bb*bb;
            if (d2 <= linewidth*linewidth)
                return yes;
        }
        return no;
}


void CanvasArrow::ComputeExtent()
{       TfcPoint *P;
        TfcRect R;
        int i;

        R.top = point[0].y;
        R.left = point[0].x;
        R.right = point[0].x;
        R.bottom = point[0].y;
        for (i=0; i < ListSize(point); i++) {
            P = &point[i];
            if (P->x < R.left)
                R.left = P->x;
            if (P->x > R.right)
                R.right = P->x;
            if (P->y < R.top)
                R.top = P->y;
            if (P->y > R.bottom)
                R.bottom = P->y;
        }
        x = R.left;
        y = R.top;
        width = R.right - R.left;
        height = R.bottom - R.top;
        for (i=0; i < ListSize(point); i++) {
            P = &point[i];
            P->x -= x;
            P->y -= y;
        }
        if (border != NOCOLOUR) {
            width++;
            height++;
        }
}


void CanvasArrow::PaintJustMe(int x1, int y1, int x2, int y2)
{
        if (!point)
            return;

        if (r) {        // A curved arrow
            TfcRect crect;
            crect.top = Cy - r;
            crect.bottom = Cy + r;
            crect.left = Cx - r;
            crect.right = Cx + r;
            DrawCurve(point[0].x,point[0].y,point[1].x,point[1].y,
                        crect,
                        colour, linewidth);
        }
        else {
            for (int i=0; i < ListSize(point) - 1; i++)
                DrawLine(point[i].x,point[i].y,point[i+1].x,point[i+1].y,
                                colour,linewidth);
        }

        DrawPolygon(colour,colour,headpoint1, ListSize(headpoint1));
        DrawPolygon(colour,colour,headpoint2, ListSize(headpoint2));
}

