/* $Id: canvas.h 437 2005-10-14 05:00:17Z jla $ */


/* Canvases:


        A canvas is a ScrollWin which has lots of 2D graphics objects on it,
        displayed at different depths.
                The objects can be circles, rectangles, polygons, text strings,
        plots, bitmaps, sprites or objects you construct yourself.  Each
        object has a certain depth and they are displayed with the correct
        objects overlapping other objects.  The objects can also respond to
        user events.  The objects can be moved up/down/left/right
        or raised/lowered/set to any depth.
                Some people call this type of display, with 2D objects
        displayed some on top of others, a "2-and-a-half D display".

        WARNING:  If you've come here because you need to write a visualisation
        and have a lot of power to customise what's displayed and how it's
        displayed, you might be better off using a ScrollWin directly and
        writing your own 'Paint()' function.   The 'Canvas' is used really
        if you need this "2-and-a-half D display" functionality.

*/

#ifndef CANVAS_H
#define CANVAS_H


#ifndef TFC_H
#include "tfc.h"
#endif

typedef enum { depth_uppermost=-65536, depth_bottommost=65536 } depth_id;
#define DEPTH(d)        ((depth_id)((int)d))
/* A 'depth_id' is really just an integer with type-checking.
We need to use type-checking because of the large number of
parameters we have in typical Canvas?() constructors. Just
put e.g.  DEPTH(8)  where it's expecting the depth. */


class CanvasObj : public Litewin {        // A generic object on a canvas.
protected:
        depth_id depth;
        friend int compar1(CanvasObj **obj1p, CanvasObj **obj2p);
        friend class Canvas;

public:
        short x,y;
        int colour, border;
        int width, height;
        TfcCallback DefaultAction;      // This simplifies the creation of objects that
                                        // only respond to a single event: mouse-clicks.
                                        // For more complex behaviour, you need to
                                        // inherit from a CanvasObj and define the
                                        // Keystroke/Mousestroke functions.

        CanvasObj(int x, int y, int width, int height, depth_id depth,
                    int colour, int border,
                    TfcCallback DefaultAction=0);
        /* Constructor. */

        virtual void PaintJustMe(int x1, int y1, int x2, int y2) = 0;
        /* How to paint it. It paints it on top, i.e. regardless */
        /* of what else is in the way. */

        virtual bool Mousestroke(int op, int x, int y) { return no; }
        /* Process a mousestroke.  */

        virtual bool Intersects(int x, int y) { return yes; }
        /* Does this point intersect this object? */

        virtual int Type() { return 0; }
        /* A form of run-time type information for CanvasSprites. */

        void Measure(int *widthp, int *heightp);
        /* Get the required width and height */

        void GetWinXY(int *xp, int *yp);
        /* Interpret 'xy' as (x,y) in Windows coordinates */

        bool GetWinRect(TfcRect *Rect);
        /* Get the display rectangle in Windows coordinates */

        void Paint(int x1, int y1, int x2, int y2);
        /* Use this to invoke the painter, not the above, */
        /* because otherwise it won't clip overlapping objects. */

        void SetDepth(depth_id depth);
        /* Change the depth to this value. */

        void MoveTo(int x, int y);
        /* Move the object to this new location. */

        virtual ~CanvasObj() { }
        /* The destructor. */
};


class Canvas : public ScrollWin {        // The canvas itself.
        class Quatree *Q;
        CanvasObj **extraObjects;
        void Initialise();
        void FreeAll();

public:
        CanvasObj *focus;
        bool RepaintOnRefocus;                // Do we repaint objects if they gain/lose focus?
                            // Set this to yes if your canvas-object paint functions paint
                            // an object differently depending on whether it has focus or not.
        int MinDepth, MaxDepth;
        int Background2;                // Normally=NOCOLOUR, but otherwise gives
                                        // a blended rectangle background.

        void AddObjectSimple(CanvasObj *obj);        // No paints
        void DelObjectSimple(CanvasObj *obj);        // No paints, no destructor.
                // Low-level adding and deleting objects

        CanvasObj* AddObject(CanvasObj *obj);
        void DelObject(CanvasObj *obj);
                // Adding and deleting objects.  The canvas 'owns' all the
                // objects you give it.

        Canvas(str caption, int width, int height);
        Canvas(str caption, int width, int height, int flag);
        Canvas(Canvas& orig);
                // Constructor and copy-constructor

        virtual void Paint(int x1, int y1, int x2, int y2);
                // How to paint it

        virtual void Measure(int *widthp, int *heightp)
                { *widthp = realWidth; *heightp = realHeight; }
                // Get the required width and height

        virtual bool Mousestroke(int op, int x, int y);
                // Process a mousestroke.

        virtual bool Keystroke(int key);
                // Process a keystroke.

        virtual void SetFocus(CanvasObj *obj);
                // Change which object has the focus.

        void Clear();
                // Delete all the objects in the canvas.

        void ClearSimple();
                /* Clear the canvas without freeing all the 
                   canvas objects (because it doesn't belong to canvas).
                   Use in conjunction with AddObjectSimple and DelObjectSimple */

        class CanvasSprite** SpriteTouches(TfcRect *rect, CanvasObj *current, int type);
                // Find all other sprites which touch this sprite.  If 'type'
                // is nonzero, only take sprites of type 'type'.

        virtual ~Canvas();
};


interface void Raise(CanvasObj *obj);
interface void Lower(CanvasObj *obj);




/*--------------- A selection of canvas objects: ------------------*/

class CanvasRect : public CanvasObj {                        // A simple rectangle with optional border
public:
        CanvasRect(int x, int y, int width, int height, depth_id depth,
                        int colour, int border, TfcCallback DefaultAction=0);
        void PaintJustMe(int x1, int y1, int x2, int y2);

        virtual ~CanvasRect() { }
};



class CanvasCircle : public CanvasObj {                        // A circle or ellipse with optional border
public:
        CanvasCircle(int x, int y, int width, int height, depth_id depth,
                        int colour, int border, TfcCallback DefaultAction=0);
        void PaintJustMe(int x1, int y1, int x2, int y2);
        bool Intersects(int x, int y);
        virtual ~CanvasCircle() { }
};



class CanvasLine : public CanvasObj {                        // A line
        char linewidth;
        bool from00;
public:
        CanvasLine(int x1, int y1, int x2, int y2, depth_id depth,
                        int colour, int linewidth=1,
                        TfcCallback DefaultAction=0);
        void PaintJustMe(int x1, int y1, int x2, int y2);
        bool Intersects(int x, int y);

        virtual ~CanvasLine() { }
};


class CanvasString : public CanvasObj { // A text string.  Specify NOCOLOUR for the
public:                                 // background for a transparent background.
        str buf;
        TfcFont font;
        int flags;

        CanvasString(int x1, int y1, int width, int height, depth_id depth,
                    str text, int foreground,
                    int background=NOCOLOUR,
                    TfcFont font=NULL, int flags=0,
                    TfcCallback DefaultAction=0);
                    // If width==-1 then it means calculate the required width.
                    // Ditto if height==-1.
            
        void SetText(str txt);

        void PaintJustMe(int x1, int y1, int x2, int y2);
        virtual ~CanvasString();
};



class CanvasBitmap : public CanvasObj {                        // A simple rectangle with optional border
protected:
        int bitmap_id;
        int fg_colour, bg_colour;
        int sourcewidth, sourceheight;
public:
        CanvasBitmap(int x, int y, int width, int height, depth_id depth,
                        int bitmap_id,
                        TfcCallback DefaultAction=0,
                        int fg_colour=WHITE, int bg_colour=BLACK);
	
		void MoveAndChange(int _x, int _y, int bitmap_id);
        void PaintJustMe(int x1, int y1, int x2, int y2);

        virtual ~CanvasBitmap() { }
};


class CanvasSprite : public CanvasObj {  // A transparent bitmap
protected:
        void* bmpImg;
        void* bmpMsk;
        int transp_colour;
public:
        int type;
        int bitmap_id;

        CanvasSprite(int x, int y, depth_id depth, int bitmap_id,
                        int width=0, int height=0,
                        int transp_colour=NOCOLOUR,
                        TfcCallback DefaultAction=0,
                        int type=0
                        );
                        // if colour is not specified,
                        // the colour if top-left point is considered
                        // as a transparence colour
        void PaintJustMe(int x1, int y1, int x2, int y2);
        void MoveAndChange(int _x, int _y, int bitmap_id);
        CanvasSprite** Touches(int type=0);      // Find all other sprites
        // which touch this sprite.  If 'type' is nonzero, only take
        // sprites of type 'type'.
        virtual int Type() { return type; }

        virtual ~CanvasSprite();
};


class CanvasPolygon : public CanvasObj {                // A polygon. Use it e.g. for triangles or arrows.
protected:
        TfcPoint *List;
        int ListLen;
public:
        CanvasPolygon(depth_id depth, int colour, int border, TfcCallback DefaultAction, TfcPoint *List, int ListLen);
        CanvasPolygon(depth_id depth, int colour, int border, TfcCallback DefaultAction, int x, int y, ...);
                        // Here you put any number of points in the function call.
                        // Terminate the variable argument list with a single TFC_TERMINATOR.
        void PaintJustMe(int x1, int y1, int x2, int y2);
        void ComputeExtent();
        bool Intersects(int x, int y);
        virtual ~CanvasPolygon();
};


class CanvasPlot : public CanvasObj {
        // A plot.  You specify the values at each x-pixel value.
        // You must create the axes yourself however.
        // It comes with a border outline if you want one
        // (specify NOCOLOUR if not).
protected:
        int *Y, Oy;
public:
        CanvasPlot(depth_id depth, int colour, int border, int x1, int x2, int *Y,
                        TfcCallback DefaultAction=0);
                /* Set:  Y[i] = -1e9  for undefined values. */
        void PaintJustMe(int x1, int y1, int x2, int y2);
        bool Intersects(int x, int y);
        virtual ~CanvasPlot() { free(Y); }
};


class CanvasBlot : public CanvasObj {                
        // A bar-graph-ish version of CanvasPlot.
        // You specify the y0..y1 values at each x-pixel value.        
        // border=NOCOLOUR for no border
        // border=colour for no body.
public:
        struct ab { short a, b; };
private:ab *Y;
        int Oy;
        bool blackborder;
public:
        CanvasBlot(depth_id depth, int colour, int border, int x1, int x2, ab *Y,
                        TfcCallback DefaultAction=0);
        void PaintJustMe(int x1, int y1, int x2, int y2);
        bool Intersects(int x, int y);
        virtual ~CanvasBlot() { free(Y); }
};


class CanvasArrow : public CanvasObj {
protected:
        int linewidth;
        TfcPoint* point;
        TfcPoint* headpoint1;
        TfcPoint* headpoint2;
        char head1;     // Style of arrowhead at A
        char head2;     // Style of arrowhead at B
        float r;        // radius of circle, for a curve;  0=straight line.
        float Cx, Cy;   // Centre of circle, for a curve.

public:
        /* Explanation of the 3-symbol 'arrow_style':
           The first char defines line width ('1'..'9'),
           the 2nd char defines the arrowhead at the start (usually='-')
           the 3rd char defines the arrowhead at the end.
           The arrowhead chars can be:
                '-' for nonexistent
                'A'..'F' for a wide arrowhead ('A'=smallest, 'F'=largest)
                'a'..'f' for a narrow arrowhead ('a'=smallest, 'f'=largest)

           A good default is "1-c"
        */

        CanvasArrow(int xStart, int yStart, int xEnd, int yEnd,
                depth_id depth, int colour,
                char* arrow_style = "1-c",
                TfcCallback DefaultAction = 0);
                    /* straight arrow constructor*/

        CanvasArrow(TfcPoint* pointArray,
                    depth_id depth, int colour,
                    char* arrow_style="1-c",
                    TfcCallback DefaultAction = 0);
                    /*polyline arrow constructor*/

        CanvasArrow(int xStart, int yStart,
                    int xEnd, int yEnd,
                    int deviation,
                    depth_id depth, int colour,
                    char* arrow_style="1-c",
                    TfcCallback DefaultAction = 0);
        /* The CURVED arrow constructor: */
        /* deviation defines a curvature */
        /* if deviation > 0, the curve is drawn counter-clockwise from start to end */
        /* if deviation > 0, the curve is drawn clockwise from start to end */

        virtual ~CanvasArrow();

        void ComputeExtent();
        TfcPoint* ComputeArrowHead(TfcPoint p1, TfcPoint p2, char style);
        virtual bool Intersects(int x, int y);
        virtual void PaintJustMe(int x1, int y1, int x2, int y2);
};


/* An example of use:

        MyCan->AddObject(new CanvasRect(150,100,30,30,RED,NOCOLOUR));
        MyCan->AddObject(new CanvasRect(150,100,30,30,RED,NOCOLOUR,Raise));        <-- Raise it on click
        MyCan->AddObject(new CanvasRect(150,100,30,30,RED,NOCOLOUR,
                            TfcCallback(MyFunc,MyData));        <-- Call your functions on click



        Or for more flexibility with the way the objects respond to user events:

        class MyCircle : public CanvasCircle {
        public:
            MyCircle(...)
                : CanvasCircle(...) { }                // Pass the constructor arguments straight thru
            bool Mousestroke(int op, int x, int y) { ..respond to mouse event.. }
            bool Keystroke(int op, int x, int y) { ..respond to keyboard event.. }
        }


        For more flexibility with how the object is displayed, create
        your own class that inherits from CanvasObj and provides a 'PaintJustMe()'
        function.

        Note that the Canvas owns all objects inside it, i.e. they are freed
        (including calling their destructors) when the Canvas is deleted. This is
        important if you put your own data into objects which inherit from CanvasObj.
*/

#endif


