/* $Id: editor.cpp 881 2005-11-08 00:20:00Z jla $ */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include "header.h"
#include "editor.h"
#include "list.h"





/* Data-structure routines:

    This powerful sub-module provides a simple API to the editor.  The API consists of function
    calls to:  insert line, delete line, replace line, undo and redo.  (Also: break line and
    join line which could alternatively be implemented on top of the other fns).  Note that
    there are only line-level functions, not character-level functions.  Note also that the
    API is in terms of strings, not lines, i.e. lines are constructed from string in this
    module, not in the editor.
            This module ensures that the screen is up-to-date with the editor.


Undoing works as follows:
    Each undo transaction is reduced to the form:  (delete_this, insert_this).
    We aim to have the exact same pointers in use after the undo as at the original time.
    This is important for avoiding problems with identifying lines in the presence of
    change.  It also falls out easily from the RowWin API.
        The undo object owns the 'insert_this' line (for undoing deletes), but the
    'delete_this' refers to something currently inside the Editor->RW.

    Redo transactions work similarly.  Each time we do an undo, we invert the
    operation and move it across to the Redo list.  To invert an operation
    involves swapping the 'insert_this' and 'delete_this' members over, and
    reversing the order of the operations (if it's a compound operation such
    as Cut/Paste).

    When a single action of the user causes multiple updates, we package these undo's into
    a single undo object:  first we create a chain of undo's, but then we take this chain
    out of the Undo chain and make it hang off a special node tagged with 'insert_this ==
    multiundo'.  (The chain hangs off 'delete_this').


States:
    For use as a programming editor, we have a concept of 'state'.  This can be used to
    represent lexical state (or grammatical state?).  This module provides the invariant
    that:
        *The first line's state is zero
        *After that, each line's state equals Editor->DefineState(ln->prev).

*/

/////static int utf8IndexToChar(str buf, int charIndex);


Line* EditorWin::NewLine(int n)
{   Line *ln;

    if (n > this->nMaxLineLength)
        return NULL;

    ln = NewLine();                // A virtual fn provided by the application-writer.
    ln->parent = this;
    ln->type = 0;
    ln->state = 0;
    ln->buf = (str)malloc(n+1);
    return ln;
}


Line* EditorWin::NewLine(str s)
{   Line *ln;    

    ln = NewLine();                // A virtual fn provided by the application-writer.
    ln->parent = this;
    ln->type = 0;
    ln->state = 0;

            // (awa) Instead of returning NULL which causes internal
            // crashes, let's chop a bit off the end
    if (strlen(s) > nMaxLineLength)
            {
        ln->buf = reinterpret_cast<char *>(malloc(nMaxLineLength + 1));
                    memcpy(ln->buf, s, nMaxLineLength);
                    ln->buf[nMaxLineLength] = '\0';
            }
            else
            {
                    ln->buf = strdup(s);
            }
    return ln;
}


void EditorWin::CompressUndo(undo_type undo)
/* Because each keystroke causes an undo object to be created, we need to */
/* compress this at some stage.  The solution is to compress multiple     */
/* updates to the same line not immediately but after a pause of, say, 4  */
/* updates. */
{   undo_type next;

    if (undo == NULL or undo->insert_this == NULL or undo->delete_this == NULL)
        return;
    next = undo->next;
    while (next and next->insert_this and next->delete_this
            and next->delete_this == undo->insert_this) {
        undo->insert_this = next->insert_this;
        undo->next = next->next;
        undo->c = next->c;
        delete next->delete_this;  /* This was actually undo->insert_this, i.e. formerly owned */
        /* by the 'undo' object, so it's Ok to free it. */
        delete next;
        next = undo->next;
        NumUndos--;
    }
}


void EditorWin::FreeUndoChain(undo_type undo)
{   undo_type tmp;

    while (undo) {
        tmp = undo->next;
        if (undo->insert_this == multiundo)
            FreeUndoChain((undo_type)undo->delete_this);
        else if (undo->insert_this)
            delete undo->insert_this;
        delete undo;
        undo = tmp;
        NumUndos--;
    }
}


void EditorWin::CullUndos(void)
/* We have too many undo's.  Delete any of them beyond number MAX_UNDOS. */
{   undo_type undo, tmp;
    int n;

    n = MAX_UNDOS;
    undo = UndoRoot;
    while (n-- > 0 and undo)
        undo = undo->next;
    if (undo == NULL)
        return;
    tmp = undo->next;
    undo->next = NULL;
    FreeUndoChain(tmp);
}


void EditorWin::ClearAllUndos(void)
/* Clear out all our undo's and redo's. */
{
    FreeUndoChain(UndoRoot), UndoRoot = NULL;
    FreeUndoChain(RedoRoot), RedoRoot = NULL;
}



/*
static bool InEditor(Line *this)
{   Line *ln;

    for (ln=(Line*)Editor->RW.root; ln; ln=(Line*)ln->next) {
        if (ln == this)
            return yes;
    }
    return no;
}


static void CheckUndoChain(undo_type undo)
{   undo_type D[100];
    int i=0;

    for (; undo; undo=undo->next, i++) {
        D[i] = undo;
        if (undo->insert_this == multiundo)
            CheckUndoChain((undo_type)undo->delete_this);
        else {
            if (undo->delete_this) {
                assert(undo->delete_this->buf[0] != -35);
                assert(((int)undo->delete_this & 4) == 0);
                assert(((int)undo->delete_this->next & 4) == 0);
                assert(((int)undo->delete_this->prev & 4) == 0);
            }
            if (undo->insert_this) {
                assert(undo->insert_this->buf[0] != -35);
                assert(((int)undo->insert_this & 4) == 0);
                assert(((int)undo->insert_this->next & 4) == 0);
                assert(((int)undo->insert_this->prev & 4) == 0);
            }
        }
    }
}


static bool InUndoChain(undo_type undo, Line *ln)
{   undo_type D[100];
    int i=0;

    for (; undo; undo=undo->next, i++) {
        D[i] = undo;
        if (undo->insert_this == multiundo) {
            if (InUndoChain((undo_type)undo->delete_this, ln))
                return yes;
        }
        else {
            if (undo->delete_this == ln)
                return yes;
            if (undo->insert_this == ln)
                return yes;
        }
    }
    return no;
}


static void debug(void)
{
    CheckUndoChain(UndoRoot);
    CheckUndoChain(OldRoot);
}
*/


void EditorWin::AddUndo(Line *insert_this, Line *delete_this, LineN_struct c)
/* Add an 'undo' transaction.  The 'undo' must insert 'insert_this' and */
/* delete 'delete_this', i.e. the names represent what the 'undo' must  */
/* do, i.e. the opposite of what was done.  Either Line can be NULL. */
{   undo_type undo;
    int n;

    /* Each time we make a fresh modification, we need to clear the Redo list. */
    if (RedoRoot)
        FreeUndoChain(RedoRoot), RedoRoot = NULL;

    /* Check the list for possible compression: */
    undo = UndoRoot;
    for (n=4; n > 0 and undo; n--)
        /* We always check the fourth operation ago, so you can continue to do little undos. */
        undo = undo->next;
    if (undo)
        CompressUndo(undo);

    /* Create the fresh undo object. */
    undo = new undo_node;
    undo->insert_this = insert_this;
    undo->delete_this = delete_this;
    undo->c = c;
    undo->next = UndoRoot;
    UndoRoot = undo;

    /* Do we have too many undo objects? */
    NumUndos++;
    if (OldRoot == NULL and NumUndos > MAX_UNDOS and NumUndos % 10 == 0)
        CullUndos();

    NeedsSave = yes;
}


void EditorWin::InsertLineAfter(Line *prev, str buf, LineN_struct c)
/* Insert a line with this string after 'prev'. If 'prev==NULL', */
/* insert it at the top.  Don't create any undo info. */
{   int state;
    Line *ln;

    ln = NewLine(buf);
    ln->parent = this;
    ln->state = DefineState(prev);
    Insert(prev, ln);
    AddUndo(NULL, ln, c);
    while (ln->next) {
        state = DefineState(ln);
        if (state == ((Line*)ln->next)->state)
            break;
        ln = (Line*)ln->next;
        ln->state = state;
        ln->PaintWhole();
    }
}


Line *EditorWin::InsertLineNoUndo(Line *prev, str buf, char type/*=0*/)
/* Like above except that we ignore undo information. */
{   Line *ln;

    ln = NewLine(buf);
    ln->parent = this;
    ln->state = DefineState(prev);
    ln->type = type;
    ln->FindDC();
    Insert(prev, ln);
    return ln;
}


void EditorWin::DeleteLine(Line *ln, LineN_struct c)
/* Insert a line with this string after 'ln'. */
{   int state;

    Delete(ln);
    AddUndo(ln, NULL, c);

    state = ln->state;
    while (ln->next) {
        ln = (Line*)ln->next;
        if (state == ln->state)
            break;
        ln->state = state;
        ln->PaintWhole();
        state = DefineState(ln);
    }
}


void EditorWin::DeleteLineNoUndo(Line *ln)
/* Delete this line. */
{   int state;

    Delete(ln);

    while (ln->next) {
        state = DefineState(ln);
        if (state == ((Line*)ln->next)->state)
            break;
        ln = (Line*)ln->next;
        ln->state = state;
        ln->PaintWhole();
    }
}


void EditorWin::JoinLines(Line *ln, LineN_struct c)
/* Join this line with its following line. */
{   Line *New, *ln_next;
    int state;

    if (ln->next == NULL)
        return;
    StartTransaction();
    ln_next = (Line*)ln->next;
    New = NewLine(strlen(ln->buf) + strlen(ln_next->buf));
    strcpy(New->buf, ln->buf);
    strcat(New->buf, ln_next->buf);
    New->state = ln->state;
    New->Measure();
    Replace(ln, New);
    AddUndo(ln, New, c);
    Delete(ln_next);
    AddUndo(ln_next, NULL, c);
    FinishTransaction();

    /* The states: */
    ln = New;
    while (ln->next) {
        state = DefineState(ln);
        ln = (Line*)ln->next;
        if (state == ln->state)
            break;
        ln->state = state;
        ln->PaintWhole();
    }
}


void EditorWin::BreakLine(Line *ln, int n, bool indent, LineN_struct c)
/* Break this line at this point. */
{   Line *first, *second;
    int i, state;
    int indentlen = strlen(IndentString);

    StartTransaction();
    //Auto Identing
    i = 0;
    while (strbegins(c.ln->buf+i, IndentString))        
        i += indentlen;
    assert((uint)n <= strlen(ln->buf));
    second = NewLine(strlen(ln->buf+n) + i);
    strcpy(second->buf + i, ln->buf+n);
    while (i) {
        memcpy(second->buf+(i-indentlen), IndentString, indentlen);            
        i-= indentlen;
    }
    second->Measure();
    first = NewLine(n);
    memcpy(first->buf, ln->buf, n);
    first->buf[n] = '\0';
    first->state = ln->state;
    first->Measure();
    Replace(ln, first);
    AddUndo(ln, first, c);
    second->state = DefineState(first);
    Insert(first, second);
    AddUndo(NULL, second, c);
    FinishTransaction();

    /* States: */
    ln = second;
    while (ln->next) {
        state = DefineState(ln);
        if (state == ((Line*)ln->next)->state)
            break;
        ln = (Line*)ln->next;
        ln->state = state;
        ln->PaintWhole();
    }
}


Line* EditorWin::ReplaceLine(Line *ln, str buf, LineN_struct c)
/* Substitute this string with this line. */
{   Line *New;
    int state;

    New = NewLine(buf);
    if (New == NULL)
        return New;
    New->state = ln->state;
    New->type = ln->type;
    New->Measure();
    Replace(ln, New);
    AddUndo(ln, New, c);

    /* States: */
    ln = New;
    while (ln->next) {
        state = DefineState(ln);
        ln = (Line*)ln->next;
        if (state == ln->state)
            break;
        ln->state = state;
        ln->PaintWhole();
    }

    return New;
}


Line *EditorWin::ReplaceLineNoUndo(Line *ln, str buf)
{   Line *New;
    int state;

    New = NewLine(buf);
    if (New == NULL)
        return New;
    New->state = ln->state;
    New->Measure();
    Replace(ln, New);

    /* States: */
    ln = New;
    while (ln->next) {
        state = DefineState(ln);
        if (state == ((Line*)ln->next)->state)
            break;
        ln = (Line*)ln->next;
        ln->state = state;
        ln->PaintWhole();
    }

    return New;
}


void EditorWin::DeleteAllLines()
/* Clear out the whole file. */
{   Row *ln;

    SuppressPaints();
    while (root) {
        ln = root->next;
        delete root;
        root = (Line*)ln;
    }
    focus = NULL;
    Marker.ln = NULL;
    Marker1.ln = NULL;
    Marker2.ln = NULL;
    c.ln = NULL;
    realHeight = 0;
}


void EditorWin::PerformOperation(undo_type undo)
{
    if (undo->insert_this == multiundo) {
        /* Compound undo's: */
        for (undo=(undo_type)undo->delete_this; undo; undo=undo->next)
            PerformOperation(undo);
    }
    else {
        /* Atomic undo's: */
        if (undo->insert_this and undo->delete_this) {
            Replace(undo->delete_this, undo->insert_this);
        }
        else if (undo->insert_this) {
            Insert(undo->insert_this->prev, undo->insert_this);
        }
        else if (undo->delete_this) {
            Delete(undo->delete_this);
        }
    }
}


static void InvertOperation(undo_type undo)
/* Invert this operation.  It involves reversing the order of */
/* compound operations and swapping inserts & deletes. */
{
    if (undo->insert_this == multiundo) {
        undo_type tmp, tmpnext;
        tmp = (undo_type)undo->delete_this;        // the original root
        undo->delete_this = NULL;
        while (tmp) {
                tmpnext = tmp->next;
                InvertOperation(tmp);
            tmp->next = (undo_type)undo->delete_this;
            undo->delete_this = (Line*)tmp;
            tmp = tmpnext;
        }
    }
    else {
        Line* tmp;
        tmp = undo->insert_this;
        undo->insert_this = undo->delete_this;
        undo->delete_this = tmp;
    }
}


void EditorWin::Undo()
/* Undo one operation.  Returns the position of the modification. */
{   LineN_struct c0=c;
    undo_type undo;

    ClearSelection();
    if (UndoRoot == NULL)
        return;
    undo = UndoRoot, UndoRoot = UndoRoot->next; // Take it off the list
    PerformOperation(undo);                            // Do it
    InvertOperation(undo);                            // Invert it
    undo->next = RedoRoot, RedoRoot = undo;            // Put it onto the redo list
    c = undo->c;                                // Move to the appropriate place.
    RedoRoot->c = c0;                           // Where would we return to?
}


void EditorWin::Redo()
/* Redo one operation.  Returns the position of the modification. */
{   LineN_struct c0=c;
    undo_type redo;

    ClearSelection();
    if (RedoRoot == NULL)
        return;
    redo = RedoRoot, RedoRoot = RedoRoot->next; // Take it off the list
    PerformOperation(redo);                            // Do it
    InvertOperation(redo);                            // Invert it
    redo->next = UndoRoot, UndoRoot = redo;            // Put it onto the undo list
    c = redo->c;                                // Move to the appropriate place.
    UndoRoot->c = c0;                           // Where would we return to?
}


void EditorWin::StartTransaction(void)
/* Set up a multi-operation transaction. */
{
    if (TransLevel++)
        return;
    OldRoot = UndoRoot;
    UndoRoot = NULL;
}


void EditorWin::FinishTransaction(void)
/* Close the transaction.  (Can be nested). */
{   undo_type undo, tmp;

    if (--TransLevel)
        return;
    if (UndoRoot == NULL)
        return;
    undo = new undo_node;
    undo->insert_this = multiundo;
    undo->delete_this = (Line*)UndoRoot;
    undo->next = OldRoot;
    for (tmp=UndoRoot; tmp->next; tmp=tmp->next)
        ;
    undo->c = tmp->c;
    UndoRoot = undo;
    OldRoot = NULL;
}


Line* EditorWin::InsertBigString(Line *AfterThis, str bigbuf)
/* Insert this string, possibly containing newlines, into the editor. */
{   Line *current=AfterThis;
    char buf[EA_MAX_LINELEN];
    str s,d;

    for (s=bigbuf, d=buf; *s; s++) {
        if (*s == '\n') {
            *d = '\0';
            current = InsertLineNoUndo(current, buf);
            d = buf;
        }
        else if (*s == '\r') {
            *d++ = '\r';
            *d = '\0';
            current = InsertLineNoUndo(current, buf);
            d = buf;
        }
        else {
            if (d+1 >= buf+sizeof(buf)) {
                *d = '\0';
                current = InsertLineNoUndo(current, buf);
                d = buf;
            }
            *d++ = *s;
        }
    }
    *d = '\0';
    if (*buf)
        current = InsertLineNoUndo(current, buf);
        // ensure that there's always a newline at the end of the last line.
    if (c.ln == NULL)
        c.ln = current;
    return current;
}


bool EditorWin::LoadFile(char* _filename, int maxSize)
/* Read this file into the editor.  */
/* Return 'yes' for success.  If    */
/* the file doesn't exist, it            */
/* creates a blank file. */
/* A maximum file size can be specified (number of chars/bytes). */
{   char buf[EA_MAX_LINELEN];
    char *s, *oldtitle;
    int NumLines = 0;
    int numChars = 0;
    FILE *input;
    Line *ln;

    DeleteAllLines();
    ClearAllUndos();
    input = fopen(_filename, "rt");
    oldtitle = strdup(windowTitle);
    strcpy(filename, _filename);
    PaginatedWidth = clientWidth;
    if (input == NULL) {
        ln = InsertLineNoUndo(NULL, "");
    }
    else {
        ln = NULL;
        NumLines = 0;
        while (fgets(buf, this->nMaxLineLength, input) && (!maxSize || numChars < maxSize)) {
            s = strchr(buf, '\n');
            if (s)
                *s = '\0';
            s = strchr(buf, '\r');
            if (s)
                *s = '\0';
            numChars += strlen(buf);
            if (maxSize && numChars >= maxSize) {
                char msg[256];
                sprintf(msg, "***File longer than %d characters - truncated...", maxSize);
                ln = InsertLineNoUndo(ln, msg);
            } else
                ln = InsertLineNoUndo(ln, buf);
            NumLines++;
            if (NumLines > 1999 and NumLines % 1000 == 0) {
                sprintf(buf, "%d lines", NumLines);
                SetTitle(buf);
            }
        }
        fclose(input);
        if (NumLines == 0) // t'was an empty file
            ln = InsertLineNoUndo(NULL, "");
    }

    c.ln = parse_line = (Line*)root;
    c.n = 0;
    c.nbuf = 0;
    focus = root;
    NeedsSave = no;
    Resized();
    if (NumLines > 1999) {
        SetTitle(oldtitle);
        free(oldtitle);
    }
    return yes;
}


bool EditorWin::SaveFile()
/* Save the contents of the editor to disk. */
/* Return 'yes' for success. */
{   FILE *output;
    Line *ln;

    output = fopen(filename, "wt");
    if (output == NULL)
        return no;
    for (ln=(Line*)root; ln; ln=(Line*)ln->next) {
        if (ln->type)
            continue;
        fputs(ln->buf, output);
        fputc('\n', output);
    }
    fclose(output);
    NeedsSave = no;
    return yes;
}


void EditorWin::LoadString(char* text)
{
    SuppressPaints();
    InsertBigString(NULL, text);
    c.ln = (Line*)root;
    c.n = 0;
    c.nbuf = 0;
}

void EditorWin::AppendString(char* text)
{
    if (root == NULL) // If empty of text, just load the string
        LoadString(text);
    else {    // else append
        Line* ln;
        SuppressPaints();
        for (ln=(Line*)root; ln->next; ln=(Line*)ln->next)
            ;
        InsertBigString(ln, text);
    }
}


str EditorWin::SaveString()
/* Package the contents of the editor into a single null-terminated */
/* string allocated with strdup, and give it to the caller. */
{   Line* ln;
    str s,s0;
    int n;

    n = 1;
    for (ln=(Line*)root; ln; ln=(Line*)(ln->next))
        n += strlen(ln->buf) + 1;
    s = s0 = (str)malloc(n);
    for (ln=(Line*)root; ln; ln=(Line*)(ln->next)) {
        strcpy(s, ln->buf);
        s += strlen(s);
        *s++ = '\n';
    }
    if (s > s0)
        s--;
    *s = '\0';
    return s0;
}


void EditorWin::Clear()
/* Clear all the rows and start afresh. */
{
    RowWin::Clear();        // Suppress paints etc.

    // The editor owns the lines.
    while (root) {
        focus = root->next;
        delete root;
        root = focus;
    }
    c.ln = NULL;
    Marker.ln = NULL;
    Marker1.ln = NULL;
}







/*----------------- Screen display and Input routines: --------------------*/

void Line::Paint(int x1, int y1, int x2, int y2)
/* Repaint this text-row at these row coordinates (i.e. (0,0) means the *
/* top left-hand corner of this row). */
{   EditorWin* RW=(EditorWin*)parent;
    struct Paragrapher_node P;

    /* This rectangle is the row coords of the full */
    /* display dimensions of the row. */
    P.x1 = 0;
    P.y1 = 0;
    P.x2 = x2;
    P.y2 = height;
    if (P.y2 + y < RW->scrollY or P.y1 + y > RW->clientHeight + RW->scrollY)
        return;
    if (P.y2 + y > RW->clientHeight + RW->scrollY)
        P.y2 = RW->clientHeight + RW->scrollY - y;
    P.pa = pa_paint;
    if (RW->Marker1.ln and y >= RW->Marker1.ln->y and y <= RW->Marker2.ln->y) {
        /* This line is at least partially selected. */
        if (y > RW->Marker1.ln->y)
            P.ha = 0, P.habuf = 0;
        else P.ha = RW->Marker1.n, P.habuf = RW->Marker1.nbuf;
        if (y < RW->Marker2.ln->y)
            P.hb = 99999, P.hbbuf = 99999;
        else P.hb = RW->Marker2.n, P.hbbuf = RW->Marker2.nbuf;
    }
    else P.ha = P.hb = -1;
    P.IsCurrent = (this == RW->c.ln);
    Paragrapher(&P);
}


int Line::MapFromXY(int x, int y, int* nbuf)
/* Convert this (ln,x,y) to a character position. */
/* (x and y in ScrollWin coordinates). */
{   struct Paragrapher_node P;

    /* This rectangle is the row coords of the full */
    /* display dimensions of the row. */
    P.x1 = 0;
    P.y1 = 0;
    P.x2 = width;
    P.y2 = height;
    P.mouse_x = x;
    P.mouse_y = y;
    P.cn = -1;
    P.cnbuf = -1;
    P.pa = pa_xy2cn;
    P.ha = P.hb = -1;
    Paragrapher(&P);
    assert(P.cn >= 0);
    if (nbuf)
        *nbuf = utf8IndexToChar(buf, P.cn);
    return P.cn;
}


void Line::MapToXY(int n, int *xp, int *yp)
/* Map from this (ln,n) to a (ln,x,y).  (ScrollWin coordinates). */
{   struct Paragrapher_node P;

    P.x1 = P.y1 = 0;
    P.x2 = width;
    P.y2 = height;
    P.cn = n;
    P.cnbuf = utf8IndexToChar(buf, n);
    P.pa = pa_cn2rect;
    P.ha = P.hb = -1;
    Paragrapher(&P);
    *xp = P.x1;
    *yp = P.y1;
}


void Line::Measure()
/* Set up the 'width' and 'height' parameters for this line. */
{   EditorWin* Editor=(EditorWin*)parent;
    struct Paragrapher_node P;

    if (Editor->realHeight > 500000) {
        /* For very large files: */
        width = Editor->realWidth;
        height = Editor->root->height;
        return;
    }
    P.x1 = P.y1 = 0;
    P.x2 = Editor->PaginatedWidth;
    P.y2 = 99999;
    P.pa = pa_measure;
    P.ha = P.hb = -1;
    Paragrapher(&P);
    width = P.x2;
    height = P.y2;
}


Line::~Line()
{
    free(buf);
}


void EditorWin::PointerHasChanged(Row *_Old, Row *_New)
{   Line *Old=(Line*)_Old, *New=(Line*)_New;

    if (c.ln == Old)
        c.ln = New;
    if (temp.ln == Old)
        temp.ln = New;
    if (temp2.ln == Old)
        temp2.ln = New;
    if (parse_line == Old)
        parse_line = New;
    if (ReplaceStart.ln == Old)
        ReplaceStart.ln = New;
    if (Marker.ln == Old)
        Marker.ln = New;
    if (Marker1.ln == Old) {
        Marker1.ln = New;
        if (Marker1.ln == NULL)
            Marker2.ln = NULL;
    }
    if (Marker2.ln == Old) {
        Marker2.ln = New;
        if (Marker2.ln == NULL)
            Marker1.ln = NULL;
    }
}


int EditorWin::AverageCharWidth(void)
{   unsigned int width, height;

    TfcFont font = TfcFindFont(fontheight);
    TextDimensions("H", 1, font, &width, &height);
    return width;
}


EditorWin::EditorWin(char *title, int width, int height, int _Background, int _Foreground,
								ScrollWin *OwnerSw/*=NULL*/)
		: RowWin(title, width, height, OwnerSw)
{
    Background = _Background;
    Foreground = _Foreground;
    CursorColour = RED;
    HighlightColour = DARK(CYAN);
    insert_mode = yes;

    fontheight = 16;
    *fontname = '\0';
    LeftIndent = 0;
    TabSize = 4 * AverageCharWidth();
    PaginatedWidth = clientWidth - 16;
            // Subtract the width of the vertical scrollbar.
    FixedWidth = no;

            nMaxLineLength = EA_MAX_LINELEN - 1;

    OverrideFont = NULL;
    EditingOneLine = no;
    Marker.ln = NULL;
    Marker1.ln = NULL;
    Marker2.ln = NULL;
    c.ln = NULL;
    temp.ln = NULL;
    temp2.ln = NULL;
    Marker.n = 0;
    Marker1.n = 0;
    Marker2.n = 0;
    Marker.nbuf = 0;
    Marker1.nbuf = 0;
    Marker2.nbuf = 0;
    c.n = 0;
    c.nbuf = 0;

    UndoRoot = RedoRoot = 0;
    NumUndos = 0;
    TransLevel = 0;
    OldRoot = NULL;
    desired_x = 0;
    Cursor.x1 = 0;
    Cursor.y1 = 0;
    Cursor.x2 = 0;
    Cursor.y2 = 0;
    parse_line = NULL;
    strcpy(filename, "unnamed.txt");

    /* Assign an appropriate foreground colour if they have NOCOLOUR. */
    if (Foreground == NOCOLOUR) {
        int brightness = (Background & 0xff) + ((Background>>8)&0xff) + ((Background>>16)&0xff);
        if (brightness < 200)
            Foreground = YELLOW;
        else Foreground = BLACK;
    }

    /* Avoid having root == NULL. */
    c.ln = InsertLineNoUndo(NULL, "");

    /* Have a valid cursor: */
    if (c.ln)
        SetCursor();

    IndentString = NULL;
    UseTabIndenting();            
}


void EditorWin::WindowClose()
{
    SaveFile();
    exit(0);
}


void Line::CursorPaint()
{   EditorWin *Editor=(EditorWin*)parent;

    DrawRectangle(Editor->Cursor.x1, Editor->Cursor.y1,
                Editor->Cursor.x2, Editor->Cursor.y2, Editor->CursorColour, NOCOLOUR);
}


void EditorWin::CursorPaint()
{
    if (c.ln)
        c.ln->CursorPaint();
}


void Line::SetCursor(int n)
{   EditorWin *Editor=(EditorWin*)parent;
    struct Paragrapher_node P;

    P.cn = n;
    P.cnbuf = utf8IndexToChar(buf, n);
    P.x1 = 0;
    P.x2 = 99999;
    P.y1 = 0;
    P.y2 = (next?next->y:Editor->realHeight) - y;
    P.pa = pa_cn2rect;
    P.ha = P.hb = -1;
    Paragrapher(&P);
    Editor->Cursor.x1 = P.x1;
    Editor->Cursor.y1 = P.y1;
    Editor->Cursor.x2 = P.x2;
    Editor->Cursor.y2 = P.y2;
    if (Editor->insert_mode)
        Editor->Cursor.x2 = Editor->Cursor.x1 + 3;
    else {
        if (Editor->Cursor.x2 <= Editor->Cursor.x1 + 3)
            Editor->Cursor.x2 = Editor->Cursor.x1 + 3;
    }
    Row::SetCursor(Editor->Cursor.x1, Editor->Cursor.y1, Editor->Cursor.x2, Editor->Cursor.y2);
}






/*--------------------- Selections: -------------------*/

void EditorWin::UpdateSelection(LineN_struct c)
{   Line *ln, *Min, *Max;

    /* Store the previous selection range: */
    Min = Marker1.ln;
    Max = Marker2.ln;

    /* Is there anything selected? */
    if (Marker.ln == NULL) {
        Marker1.ln = Marker2.ln = NULL;
        goto REPAINT;
    }

    /* Sort Marker1 and Marker2 into the appropriate order: */
    if (Marker.ln->y < c.ln->y)
        Marker1 = Marker, Marker2 = c;
    else if (Marker.ln->y > c.ln->y)
        Marker1 = c, Marker2 = Marker;
    else if (Marker.nbuf < c.nbuf)
        Marker1 = Marker, Marker2 = c;
    else Marker1 = c, Marker2 = Marker;

    /* Take the union with the old selection. */
    if (Min == NULL or Marker1.ln->y < Min->y)
        Min = Marker1.ln;
    if (Max == NULL or Marker2.ln->y > Max->y)
        Max = Marker2.ln;

    /* Invalidate an area containing the union of the old */
    /* and new selections. */
    REPAINT:
    for (ln=Min; ln and ln->y <= Max->y; ln=(Line*)ln->next)
        ln->PaintWhole();
}


void EditorWin::StartSelection(void)
{
    Marker = c;
}


void EditorWin::ClearSelection(void)
{
    Marker.ln = NULL;
    if (Marker1.ln == NULL)
        return;
    UpdateSelection(Marker);
    Marker1.ln = NULL;
    Marker2.ln = NULL;
}


bool EditorWin::HasSelection(void)
{
    return (Marker1.ln != NULL && Marker2.ln != NULL);
}


bool EditorWin::IntersectSelection(int coln, int liney)
{
    if (!HasSelection())
        return false;

    if (Marker1.ln == Marker2.ln)
        return (coln > Marker1.n  && coln < Marker2.n);
    else
        return (liney > Marker1.ln->y && liney < Marker2.ln->y);
}


void EditorWin::Copy(void)
/* Copy the selected text. */
{   str s, Clipboard;
    Line *ln;
    int len;

    if (Marker1.ln == NULL)
        return;

    /* How much space do we need? */
    len = 1;
    for (ln=Marker1.ln; ln and ln->y <= Marker2.ln->y; ln=(Line*)ln->next)
        len += strlen(ln->buf) + 2;

    /* Create the clipboard: */
    Clipboard = s = (char*)malloc(len);
    if (Marker1.ln == Marker2.ln) {
        memcpy(s, Marker1.ln->buf + Marker1.nbuf, Marker2.nbuf - Marker1.nbuf);
        s[Marker2.nbuf - Marker1.nbuf] = '\0';
    }
    else {
        strcpy(s, Marker1.ln->buf + Marker1.nbuf);
        s += strlen(s);
        *s++ = '\r';
        *s++ = '\n';
        for (ln=(Line*)Marker1.ln->next; ln and ln != Marker2.ln; ln=(Line*)ln->next) {
            strcpy(s, ln->buf);
            s += strlen(s);
            *s++ = '\r';
            *s++ = '\n';
        }
        memcpy(s, Marker2.ln->buf, Marker2.nbuf);
        s[Marker2.n] = '\0';
    }

    /* Copy it into the Windows clipboard: */
    ClipboardSetText(Clipboard);
    free(Clipboard);
    c.ln->SetCursor(c.n);
}


void EditorWin::CopyAndClear()
/* Copy the selection into the clipboard, and then */
/* clear the selection so the user knows that */
/* something has happened: */
{
    Copy();
    ClearSelection();
    c.ln->SetCursor(c.n);
}


void EditorWin::Gobble(void)
/* Gobble the selected text. */
{   char buf[EA_MAX_LINELEN];
    Line *ln;

    if (Marker1.ln == NULL)
        return;
    if (EditingOneLine and (Marker1.ln != c.ln or Marker2.ln != c.ln))
        return;
    if (Marker1.ln == Marker2.ln) {
        ln = Marker1.ln;
        memcpy(buf, ln->buf, Marker1.n);
        strcpy(buf+Marker1.n, ln->buf + Marker2.n);
        Marker2.n = Marker1.n;
        ReplaceLine(ln, buf, c);
        c = temp = Marker1;
    }
    else {
        SuppressPaints();
        StartTransaction();
        memcpy(buf, Marker1.ln->buf, Marker1.n);
        strcpy(buf+Marker1.n, Marker2.ln->buf + Marker2.n);
        ReplaceLine(Marker1.ln, buf, c);
        until ((Line*)Marker1.ln->next == Marker2.ln)
            DeleteLine((Line*)Marker1.ln->next, c);
        ln = Marker2.ln;
        c = temp = Marker1;
        DeleteLine(ln, c);
        FinishTransaction();
        SuppressPaints();
    }
    Marker.ln = NULL;
    Marker1.ln = NULL;
    c.ln->SetCursor(c.n);
}


void EditorWin::Cut(void)
/* Cut the selected text. */
{
    if (Marker1.ln == NULL)
        return;
    Copy();
    Gobble();
    c.ln->SetCursor(c.n);
}


void EditorWin::Paste(void)
/* Paste the clipboard into the editor. */
{   char buf[EA_MAX_LINELEN + 1], buf2[EA_MAX_LINELEN + 1];
    str s,d,s0,se,Clipboard;

    Clipboard = ClipboardGetText();
    if (Clipboard == NULL)
        return;

    s = strchr(Clipboard, '\n');
    if (s == NULL) {
        /* A partial line of text. */
        d = buf;
        memcpy(d, c.ln->buf, c.nbuf);
        d += c.nbuf;
        strcpy(d, Clipboard);
        strcat(d, c.ln->buf + c.nbuf);
        if (ReplaceLine(c.ln, buf, c) == NULL)
            return;
        c.nbuf += strlen(Clipboard);
        c.n += utf8len(Clipboard);

                    if (c.n > nMaxLineLength)
                            c.n = nMaxLineLength;
    }
    else {
        SuppressPaints();
        StartTransaction();
        strcpy(buf2, c.ln->buf + c.nbuf);
        d = buf;
        memcpy(d, c.ln->buf, c.nbuf);
        d += c.nbuf;
        if (s[-1] == '\r')
            se = s-1;
        else se = s;
        memcpy(d, Clipboard, se - Clipboard);
        d[se - Clipboard] = '\0';
        ReplaceLine(c.ln, buf, c);
        do {
            s0 = s+1;
            s = strchr(s0, '\n');
            if (s == NULL)
                break;
            se = (s[-1] == '\r') ? s-1 : s;
            memcpy(buf, s0, se - s0);
            buf[se - s0] = '\0';
            InsertLineAfter(c.ln, buf, c);
            c.ln = (Line*)c.ln->next;
        } forever;
        strcpy(buf, s0);
        c.n = utf8len(buf);
        c.nbuf = strlen(buf);
        strcat(buf, buf2);
        InsertLineAfter(c.ln, buf, c);
        c.ln = (Line*)c.ln->next;
        FinishTransaction();
    }
    free(Clipboard);
    c.ln->SetCursor(c.n);
}


void EditorWin::SetMaximumInputLength(int nMaximumLineLength)
{
    nMaxLineLength = nMaximumLineLength;
}


bool EditorWin::UsingTabIndenting()
{
    return (IndentString and *IndentString == '\t');
}


int EditorWin::UsingSpaceIndenting()
{
    if (IndentString == NULL or *IndentString == '\t')
        return 0;
    return strlen(IndentString);                   
}


void EditorWin::UseSpaceIndenting(int numspaces)
{
    if (numspaces < 2)
        return;
        
    if (IndentString) {
        free(IndentString);
        IndentString = NULL;
    }
    
    IndentString = (char *) calloc (numspaces + 1, 1);
    memset(IndentString, ' ', numspaces);
}
    

void EditorWin::UseTabIndenting()
{
    if (IndentString) {
        free(IndentString);
        IndentString = NULL;
    }
    IndentString = (char *) calloc (2, 1);
    IndentString[0] = '\t';         
}


void EditorWin::Indent(bool right)
/* Indent (or unindent) the selected text. */
{   char buf[EA_MAX_LINELEN];
    Line *ln, *EndLn;
    int indentlen = strlen(IndentString);

    if (Marker2.ln == NULL)
        return;     // Only happens if they hit TAB while dragging.
    if (Marker2.n == 0 and Marker2.ln->prev)
        EndLn = Marker2.ln;
    else EndLn = (Line*)Marker2.ln->next;
    StartTransaction();
    for (ln=Marker1.ln; ln and ln != EndLn; ln=(Line*)ln->next) {
        if (right) {
            strcpy(buf, IndentString);
            strcpy(buf+indentlen, ln->buf);
            if (ln == c.ln)
            {
                c.n +=indentlen;
                c.nbuf += indentlen;
            }
            if (ln == Marker.ln)
            {
                Marker.n +=indentlen;
                Marker.nbuf +=indentlen;
            }
            if (ln == Marker1.ln)
            {
                Marker1.n +=indentlen;
                Marker1.nbuf +=indentlen;
            }
            if (ln == Marker2.ln)
            {
                Marker2.n +=indentlen;
                Marker2.nbuf +=indentlen;
            }
        }
        else {
            if (!strbegins(ln->buf, IndentString))
                continue;
            strcpy(buf, ln->buf+indentlen);
            if (ln == c.ln and c.n > 0)
            {
                c.n -=indentlen;
                c.nbuf -=indentlen;
            }
            if (ln == Marker.ln and Marker.n)
            {
                Marker.n-=indentlen;
                Marker.nbuf-=indentlen;
            }
            if (ln == Marker1.ln and Marker1.n)
            {
                Marker1.n-=indentlen;
                Marker1.nbuf-=indentlen;
            }
            if (ln == Marker2.ln and Marker2.n)
            {
                Marker2.n-=indentlen;
                Marker2.nbuf-=indentlen;
            }
        }
        ReplaceLine(ln, buf, c);
    }
    FinishTransaction();
    c.ln->SetCursor(c.n);
}



void EditorWin::Resized(void)
{   int MaxHeight, MinHeight;
    Line *ln;

    if (clientWidth < 50)
        return;
    if (clientWidth == PaginatedWidth + LeftIndent)
        return;
    /*if (clientWidth >= PaginatedWidth - 16 and
                    clientWidth <= PaginatedWidth - 16)
        return;
        /* Do we care about the small resizing scrollbars */
        /* cause?  Yes. */
    PaginatedWidth = clientWidth - LeftIndent;
    GetFontRange(&MinHeight, &MaxHeight);
    MinHeight = MinHeight * 2 / 3;  // MinHeight is the minimum
    // comfortable size, but we allow uncomfortable fonts too.
    if (fontheight < MinHeight or fontheight > MaxHeight)
        fontheight = MinHeight * 4 / 3;
    for (ln=(Line*)root; ln; ln=(Line*)ln->next)
        ln->Measure();
    RowWin::Resized();
    if (c.ln)
        c.ln->SetCursor(c.n);
}





/*--------------------- The editor itself: ---------------------*/

bool EditorWin::MoveRight(void)
/* Move right one character. Move to the next line if you're on the end. */
/* Returns 'yes' on success. */
{
    if (c.ln->buf[c.nbuf]) {
        c.n++;
        c.nbuf += charlen(c.ln->buf + c.nbuf);
        return yes;
    }
    if (c.ln->next) {
        c.ln = (Line*)c.ln->next;
        c.n = 0;
        c.nbuf = 0;
        return yes;
    }
    return no;
}


bool EditorWin::MoveLeft(void)
/* Move left one character. Move to the previous line if you're at the beginning. */
/* Returns 'yes' on success. */
{
    if (c.n > 0) {
        c.n--;
        c.nbuf -= charlen_bwd(c.ln->buf, c.nbuf -1);
        return yes;
    }
    if (c.ln->prev and c.ln != parse_line) {
        c.ln = (Line*)c.ln->prev;
        c.n = utf8len(c.ln->buf);
        c.nbuf = strlen(c.ln->buf);
        return yes;
    }
    return no;
}


bool EditorWin::IsWordStart(int n, bool for_ctrl_t)
/* Are we at the start of a word at this position? */
{   int a, b;

    if (n == 0)
        return yes;
    int  nbuf = utf8IndexToChar(c.ln->buf, n);
    int  nbuf1 = utf8IndexToChar(c.ln->buf, n-1);
    b = c.ln->buf[nbuf];
    a = c.ln->buf[nbuf1];
    if (isalnum(b) or b == '_')
        return not (isalnum(a) or a == '_');
    else if (isspace(b))
        return no; // for some reason this the case for most editors
    else if (not isalnum(b))
        return yes;
    else if (IsBracket(b) or b == ',')
        return yes;
    else if (for_ctrl_t and b == '/')
        return yes;
    else
        return isspace(a) or a == ',';
}


int EditorWin::IsBracket(int ch)
/* 1=open bracket, -1=close bracket, 0=not a bracket. */
{
    if (ch == '{' or ch == '[' or ch == '(')
        return 1;
    if (ch == '}' or ch == ']' or ch == ')')
        return -1;
    return 0;
}


void EditorWin::FindMatchingBracket(void)
/* Move to the bracket matching this one. */
{   int n, b, och, cch;
    Line *ln;

    n = c.n;
    if (not IsBracket(och=c.ln->buf[n])) {
        if (n == 0)
            return;
        n--;
        if (not IsBracket(och=c.ln->buf[n]))
            return;
    }
    if (IsBracket(och) == 1) {
        if (och == '{')
            cch = '}';
        else if (och == '[')
            cch = ']';
        else if (och == '(')
            cch = ')';
        else
            cch = '\0';   // Should never happen - just fixes a compiler warning for uninitialised var
        n++;
        b = 0;
        for (ln=c.ln; ln; ln=(Line*)ln->next, n=0) {
            for ( ; ln->buf[n]; n++) {
                if (ln->buf[n] == och)
                    b++;
                else if (ln->buf[n] == cch) {
                    if (--b < 0)
                        goto FOUND;
                }
            }
        }
    }
    else {
        cch = och;
        if (cch == '}')
            och = '{';
        else if (cch == ']')
            och = '[';
        else if (cch == ')')
            och = '(';
        b = 0;
        n--;
        for (ln=c.ln; ln; ln=(Line*)ln->prev, (n = ln ? strlen(ln->buf) : 0)) {
            for ( ; n >= 0; n--) {
                if (ln->buf[n] == cch)
                    b++;
                else if (ln->buf[n] == och) {
                    if (--b < 0)
                        goto FOUND;
                }
            }
        }
    }
    /* Not found. */
    return;

    FOUND:
    c.ln = ln;
    c.nbuf = n;
    c.n = utf8IndexToChar(ln->buf, n);
}


bool EditorWin::MoveTo(int x, int y)
/* Move to this location (x and y in ScrollWin coordinates). */
{   Line *old;

    if (x < 0) x = 0;
    if (y < 0) y = 0;
    temp2.ln = (Line*)FindRow(y);
    if (temp2.ln == NULL)
        return no;

    // (aha) supress paints so cursors don't paint out of bounds
    SuppressPaints();

    if (c.ln == NULL)
        c.ln = temp2.ln;
    else if (c.ln->y < temp2.ln->y) {
        while (c.ln->y < temp2.ln->y) {
            old = c.ln;
            c.ln = (Line*)c.ln->next;
            if (not MoveDown(old)) {
                RETURN:
                if (c.n > utf8len(c.ln->buf))
                    c.n = utf8len(c.ln->buf);
                return no;
            }
            if (temp2.ln == NULL)
                goto RETURN;
        }
    }
    else {
        while (c.ln->y > temp2.ln->y) {
            old = c.ln;
            c.ln = (Line*)c.ln->prev;
            if (not MoveUp(old) or temp2.ln == NULL)
                goto RETURN;
        }
    }
    c.n = c.ln->MapFromXY(x, y - temp2.ln->y, &c.nbuf);
    return yes;
}


bool EditorWin::MoveTo(Line *ln, int n)
/* Move to this location (ln,n). */
{   Line *old;

    if (ln == NULL)
        return no;
    while (c.ln->y < ln->y) {
        old = c.ln;
        c.ln = (Line*)c.ln->next;
        if (not MoveDown(old)) {
            RETURN:
            if (c.n > utf8len(c.ln->buf))
            {
                c.n = utf8len(c.ln->buf);
                c.nbuf = strlen(c.ln->buf);
            }
            return no;
        }
    }
    while (c.ln->y > ln->y) {
        old = c.ln;
        c.ln = (Line*)c.ln->prev;
        if (not MoveUp(old))
            goto RETURN;
    }
    c.ln = ln;        // (in case for any reason we didn't get there)
    int ulen = utf8len(c.ln->buf);
    c.n = (n <= ulen) ? n : ulen;
    c.nbuf = utf8IndexToChar(c.ln->buf, n);
    /* The second case can happen if we move down into an output/error  */
    /* line which is then deleted from inside the MoveDown/Up call.            */
    return yes;
}



/*--------------------- The application interacting with the text: ---------------------*/

void EditorWin::InsertChar(Line* ln, int n, char ch)
{   char* buf;

    if (n > utf8len(ln->buf))
        return;
    int nbuf = utf8IndexToChar(ln->buf, n);        
    buf = (str)malloc(strlen(ln->buf) + 2);        
    memcpy(buf, ln->buf, nbuf );
    buf[n] = ch;
    strcpy(buf+n+1, ln->buf+n);
    ln = ReplaceLine(ln, buf, c);
    if (c.ln == ln and c.n >= n)
        c.n++;
    if (Marker1.ln == ln and Marker1.n >= n)
        Marker1.n++;
    if (Marker2.ln == ln and Marker2.n >= n)
        Marker2.n++;
    if (Marker.ln == ln and Marker.n >= n)
        Marker.n++;
}


void EditorWin::InsertWord(Line* ln, int n, str word)
{   char* buf;
    int len;

    if (n > strlen(ln->buf))
        return;
    if (word == NULL)
        return;
    len = strlen(word);
    buf = (str)malloc(strlen(ln->buf) + len + 1);
    memcpy(buf, ln->buf, n);
    memcpy(buf+n, word, len);
    strcpy(buf+n+len, ln->buf+n);
    ln = ReplaceLine(ln, buf, c);
    if (c.ln == ln and c.n >= n)
    {
        c.n+=utf8len(word);
        c.nbuf += len;
    }
    if (Marker1.ln == ln and Marker1.n >= n)
    {
        Marker1.n+=utf8len(word);
        Marker1.nbuf+=len;
    }
    if (Marker2.ln == ln and Marker2.n >= n)
    {
        Marker2.n+=utf8len(word);;
        Marker2.nbuf+=len;
    }
    if (Marker.ln == ln and Marker.n >= n)
    {
        Marker2.n+=utf8len(word);;
        Marker2.nbuf+=len;
    }
}



/*--------------------- High-level operations: ---------------------*/

static char* stristr(char *big, char* target)
/* Find 'target' in 'big', case-insensitive. */
{   int len=strlen(target);

    while (*big) {
        if (strnicmp(big, target, len) == 0)
            return big;
        else big++;
    }
    return NULL;
}


void EditorWin::FindOrReplace(char *findstring, char *replacewith,
        bool case_sensitive, bool whole_word, bool prompt, bool backwards)
{   bool first_time, last_time, found, last_was_replace;
    int response, num_replacements=0;
    char buf[1024];        
    char *s, *d;
    Line *ln;
    int n;

    first_time = yes;
    last_time = no;

    found = no;
    ReplaceStart = c;
    if (replacewith) {
        StartTransaction();
        last_was_replace = yes;
    }
    else
        last_was_replace = no;
    for(ln = c.ln; ln != ReplaceStart.ln or first_time or last_time; )
    {
        s = ln->buf;
        if(first_time)
        {
            s += ReplaceStart.n;
            if(backwards)
                s -= strlen(findstring) + 1;
        }
        else if (backwards)
            s += strlen(ln->buf) - 1;
        
        do {
            /* Find the next occurrence of the word on this line: */
            if (backwards) {
                char *t = ln->buf, *t_1=NULL;
                do {
                    if (case_sensitive)
                        t = strstr(t, findstring);
                    else t = stristr(t, findstring);
                    if (t > s or t == NULL)
                        break;
                    t_1 = t++;
                } forever;
                s = t_1;
            }
            else {
                if (case_sensitive)
                    s = strstr(s, findstring);
                else s = stristr(s, findstring);
            }
            if (s == NULL)
                break;
            if (whole_word) {
                if (s > ln->buf and isalnum(s[-1]) or s[-1] == '_') {
                    if (backwards)
                        s--;
                    else
                        s++;
                    continue;
                }
                n = strlen(findstring);
                if (isalnum(s[n]) or s[n] == '_') {
                    if (backwards)
                        s--;
                    else
                        s++;
                    continue;
                }
            }

            /* Now that we've found it, what are we going to do with it? */
            found = yes;
            if (prompt or replacewith == NULL) {
                /* Highlight the string we found: */
                ClearSelection();
                MoveTo(ln, s - ln->buf);
                StartSelection();
                MoveTo(ln, s + strlen(findstring) - ln->buf);
                UpdateSelection(c);
                SetCursor();
            }
            if (replacewith == NULL)
                return;
            if (prompt) {
                response = TfcChoose("Replace",
                            last_was_replace ?
                                  "~&Replace\0Replace, do&n't ask\0&Skip\0Cancel\0"
                                : "&Replace\0Replace, do&n't ask\0~&Skip\0Cancel\0",
                                "Replace this one?");
                if (response == 0 or response == 4) {
                    FinishTransaction();
                    return;
                }
                if (response == 2)
                    prompt = no;
                last_was_replace = (response == 1 or response == 2);
            }
            else last_was_replace = yes;
            if (last_was_replace) {
                n = s - ln->buf;
                memcpy(buf, ln->buf, n);
                d = buf + n;
                strcpy(d, replacewith);
                d += strlen(replacewith);
                strcpy(d, s + strlen(findstring));
                ln = ReplaceLine(ln, buf, c);
                s = ln->buf + n + strlen(replacewith);
                num_replacements++;
            }
            else s++;
        } forever;
        

        ln = backwards ? (Line*)ln->prev : (Line*)ln->next;
        if (ln == NULL) {
            if (backwards) {
                for (ln=(Line*)root; ln->next; ln=(Line*)ln->next)
                    ;
            }
            else ln = (Line*)root;
        }
        first_time = no;
        
        if (last_time)
            break;
        else if (ln == ReplaceStart.ln)
            last_time = yes;

    }
    if (replacewith)
        FinishTransaction();
    if (not found) {
        if (prompt){
            sprintf(buf, "Couldn't find:  %s", findstring);
            TfcMessage("Find", '!', buf);
        }
        F3meansAgain = no;
    }
    else if (not prompt and num_replacements) {
        sprintf(buf, "%d replacements made.", num_replacements);
        TfcMessage("Replace", '!', buf);
        F3meansAgain = no;
    }
    ClearSelection();
    if (ReplaceStart.ln)
        c = ReplaceStart;
    ReplaceStart.ln = NULL;
}


void EditorWin::GuiFindOrReplace(bool find_again, bool replace, bool backwards)
/* If 'find_again', then we dispense with the dialog box and */
/* just repeat the last find. */
{   static char replacewith[512];
    static char findstring[512];
    static bool case_sensitive;
    static bool whole_word;
    control d,r,l,def;
    int result;

    if (*findstring == '\0' or not find_again) {
        /* Do the dialog box: */
        if (Marker1.ln and Marker1.ln == Marker2.ln) {
            /* If there's selected text, copy it into 'findstring'. */
            memcpy(findstring, Marker1.ln->buf + Marker1.n,
                        Marker2.n - Marker1.n);
            findstring[Marker2.n - Marker1.n] = '\0';
        }
        if (replace) {
            d = (def=Control(findstring, sizeof(findstring), "&Find:", 15))
                    | Control(replacewith, sizeof(replacewith), "Replace with:", 15);
            r = Control(&case_sensitive, "Match &case")
                    -
                Control(&whole_word, "Match &whole word only")
                    -
                Control(&backwards, "&Backwards");
            l = Button("Replace, do&n't ask", 3)
                    -
                (OkButton() | CancelButton());
            result = DoDialog("Search & replace", d - (r | l), def);
        }
        else {
            d = (def=Control(findstring, sizeof(findstring), "&Find:", 15))
                    -
                Control(&case_sensitive, "Match &case")
                    -
                Control(&whole_word, "Match &whole word only")
                    -
                Control(&backwards, "&Backwards")
                    -
                (OkButton() | CancelButton());
            result = DoDialog("Find", d, def);
        }
    }
    else result = 1;
    if (result <= 0 or *findstring == '\0')
        return;

    /* Do the action: */
    F3meansAgain = yes;
    FindOrReplace(findstring, replace ? replacewith : NULL,
            case_sensitive, whole_word, result == 1, backwards);
    c.ln->SetCursor(c.n);
}

// Globals for the 'Find' modeless dialog box
bool find_SearchUp = no, find_MatchCase = no, find_MatchWholeWord = no;
char find_fstring[512];
void* find_dlg = NULL;

void EditorWin::FindDlgDied()
{
	find_dlg = NULL;
}


void EditorWin::FindNextFunction()
{
	FindOrReplace(find_fstring, NULL, find_MatchCase, find_MatchWholeWord, no, find_SearchUp);
    c.ln->SetCursor(c.n);
}


void EditorWin::GuiFind() // Modeless dialog box
{    
	control root, stringEntryBox, findNextButton, def;
	if (find_dlg != NULL)
		return;
	stringEntryBox = Control(find_fstring, sizeof(find_fstring), "&Find:", 15, &EditorWin::FindNextFunction);
	findNextButton = Button("&Find Next", &EditorWin::FindNextFunction, TFC_DEFPUSHBUTTON); // Set 'Return' focus to this button
	// See if there is currently a selection, if so, store in find_fstring
	if (Marker1.ln and Marker1.ln == Marker2.ln)	{
		memcpy(find_fstring, Marker1.ln->buf + Marker1.n, Marker2.n - Marker1.n);
		find_fstring[Marker2.n - Marker1.n] = '\0';
	}
	root = (def = stringEntryBox)
						-
					Control(&find_MatchCase, "Match &case")
						-
					Control(&find_MatchWholeWord, "Match &whole word only")
						-
					Control(&find_SearchUp, "&Search Up")
						-
					(findNextButton | CancelButton());
	find_dlg = CreateModelessDialog("Find", root, def, &EditorWin::FindDlgDied);
}


void EditorWin::GuiReplace()
{
    GuiFindOrReplace(no, yes, no);
}


void EditorWin::Edit()
{
    /* Begin editing at the top: */
    if (root == NULL) {
        InsertLineNoUndo(NULL, "");
        c.ln = parse_line = (Line*)root;
        focus = root;
    }

    /* Initialise this line: */
    c.n = 0;
    desired_x = 0;

    /* Start processing events: */
    c.ln->SetCursor(c.n);
    EventLoop();
}


bool Line::Mousestroke(int op, int x, int y)
{   EditorWin* Editor=(EditorWin*)parent;
    int n, nbuf;

    /* If we're doing a 1-line 'gets' edit, then don't allow */
    /* the mouse to be used to exit the line. */
    /*if (Editor->EditingOneLine and this != Editor->c.ln)
        return yes;*/

    switch (op) {
        case MOUSE_PRESS:
            if (Editor->Marker.ln)
                Editor->ClearSelection();
            SelectKeyDown = yes;
            n = MapFromXY(x, y, &nbuf);
            if (Editor->EditingOneLine) {
                Editor->Marker.ln = this, Editor->Marker.n = n, Editor->Marker.nbuf = nbuf;
                Editor->temp2 = Editor->Marker;
            }
            else {
                Editor->c.ln = this;
                Editor->c.nbuf = n;
                Editor->c.n = n;
                Editor->StartSelection();
                SetCursor(n);
            }
            break;

        case MOUSE_DRAG:
            SelectKeyDown = yes;
            n = MapFromXY(x, y, &nbuf);
            if (Editor->EditingOneLine) {
                Editor->Marker2.ln = this, Editor->Marker2.n = n, Editor->Marker2.nbuf = nbuf;
                Editor->temp2 = Editor->Marker2;
            }
            else {
                Editor->c.ln = this;
                Editor->c.n = n;
                Editor->c.nbuf = utf8IndexToChar(buf, n);
                Editor->UpdateSelection(Editor->c);
                SetCursor(n);
            }
            break;

        case MOUSE_RELEASE:
            if (Editor->Marker1.ln == Editor->Marker2.ln and
                Editor->Marker1.n == Editor->Marker2.n) {
                Editor->ClearSelection();
                SelectKeyDown = no;
            }
            else
                SelectKeyDown = yes;
            break;
    }
    return yes;
}


bool EditorWin::Keystroke(int key)
{   int x,y,w,response;
    char buf[EA_MAX_LINELEN +1];
    Line *ln;
    int indentlen = strlen(IndentString);

    temp = c;
    switch (key) {

        case WINDOW_QUIT:
            if (NeedsSave) {
                response = TfcChoose("Open", "~&Save\0&Don't save\0Cancel\0",
                        "Do you want to save:\n\n\t%s\n", filename);
                if (response == 1) {
                    Save();
                    QuitEventLoop = yes;
                }
                else if (response == 2)
                    QuitEventLoop = yes;
            }
            else QuitEventLoop = yes;
            return yes;

        case CTRL('Y'):
            if (Marker.ln)
                ClearSelection();
            if (c.ln->next)
                DeleteLine(c.ln, c);
            else if (c.ln->prev and c.ln != parse_line) {
                c.ln = (Line*)c.ln->prev;
                DeleteLine((Line*)c.ln->next, c);
            }
            else {
                ReplaceLine(c.ln, "", c);
            }
            c.n = 0;
            break;

        case CTRL('U'):
        case CTRL('Z'):
        case UNDO:
            Undo();
            break;

        case CTRL('R'):
            Redo();
            break;

        case CTRL('S'):
            Save();
            //TfcMessage("Save", 'i', "File saved.");
            // TJ: At best, this message is just annoying.  
            // But also, it is inconsistent:
            // with this Message here, if the user saves through 
            // the File/Save option, they
            // don't get the annoying pop-up, but if they use ctrl-s, then
            // they do.
            // An experienced programmer will save all the time, and
            // does not need a popup to see it has succeeded.  e.g. put a
            // "*" in the document title bar for unsaved files.
            break;

        case CTRL('P'):
			Print(pageTitle ? pageTitle : windowTitle);
            break;

        case CTRL('F'):
            GuiFind();
            return yes;

        case F3:
            GuiFindOrReplace(F3meansAgain, no, no);
            return yes;

        case SHIFT(F3):
            GuiFindOrReplace(F3meansAgain, no, yes);
            return yes;

        case UP:
		case SHIFT(UP):
            if (desired_x)
                x = desired_x;
            else x = desired_x = Cursor.x1;
            y = Cursor.y1 + c.ln->y - 1;
            desired_x = x;
            do {
                if (y < 0)
                    return no;
                if (not MoveTo(x, y))
                    break;
                y -= 8;
            } while (c == temp);
            break;

        case DOWN:
		case SHIFT(DOWN):
            if (desired_x)
                x = desired_x;
            else x = desired_x = Cursor.x1;
            y = Cursor.y2 + c.ln->y + 1;
            do {
                if (y >= realHeight)
                    return no;
                if (not MoveTo(x, y))
                    break;
                y += 8;
            } while (c == temp);
            break;

        case RIGHT:
		case SHIFT(RIGHT):
            MoveRight();
            break;

        case LEFT:
		case SHIFT(LEFT):
            MoveLeft();
            break;

        case INS:
            insert_mode = not insert_mode;
            break;

        case CTRL_(RIGHT):
        case CTRL_(SHIFT(RIGHT)):
            do {
                if (not MoveRight())
                    break;
            } until (IsWordStart(c.n, no));
            break;

        case CTRL_(LEFT):
        case CTRL_(SHIFT(LEFT)):
            do {
                if (not MoveLeft())
                    break;
            } until (IsWordStart(c.n, no));
            break;

        case HOME:
        case SHIFT(HOME):
            x = 0;
            y = Cursor.y1 + c.ln->y;
            MoveTo(x, y);
            break;

        case END:
        case SHIFT(END):
            x = 99999;
            y = Cursor.y1 + c.ln->y;
            MoveTo(x, y);
            break;

        case DEL:
            if (Marker.ln)
                Gobble();
            else if (c.ln->buf[c.nbuf]) {
                int cn = charlen(c.ln->buf + c.nbuf);
                memcpy(buf, c.ln->buf, c.nbuf);
                strcpy(buf+c.nbuf, c.ln->buf+c.nbuf+cn);
                ReplaceLine(c.ln, buf, c);
            }
            else {
                JoinLines(c.ln, c);
            }
            break;

        case BACKSPACE:
            if (Marker.ln)
                Gobble();
            else if (not insert_mode) {
                MoveLeft();
                break;
            }
            else if (c.n > 0) {
                int cn = charlen_bwd(c.ln->buf, c.nbuf); // that's the current character. we need the previsous one
                cn = charlen_bwd(c.ln->buf, c.nbuf-cn);                    
                memcpy(buf, c.ln->buf, c.nbuf-cn);
                strcpy(buf+c.nbuf-cn, c.ln->buf+c.nbuf);
                ReplaceLine(c.ln, buf, c);
                c.n--;
                c.nbuf -= cn;
            }
            else if (c.ln->prev) {
                c.ln = (Line*)c.ln->prev;
                c.nbuf = strlen(c.ln->buf);
                c.n = utf8len(c.ln->buf);
                JoinLines(c.ln, c);
            }
            break;

        case CTRL('T'):
            if (c.ln->buf[c.nbuf] == '\0') {
                JoinLines(c.ln, c);
            }
            else {
                for (w=c.nbuf+1; c.ln->buf[w]; w++) {
                    if (IsWordStart(w, yes))
                        break;
                }
                memcpy(buf, c.ln->buf, c.nbuf);
                strcpy(buf + c.nbuf, c.ln->buf + w);
                ReplaceLine(c.ln, buf, c);
            }
            break;

        case ENTER:
            if (not insert_mode and c.ln->next) {
                c.ln = (Line*)c.ln->next;
                c.n = 0;
                c.nbuf = 0;
                break;
            }
            BreakLine(c.ln, c.nbuf, yes, c);
            c.ln = (Line*)c.ln->next;
            c.n = 0;
            c.nbuf = 0;
            if (not MoveDown((Line*)c.ln->prev))
                break;
            while (strbegins(c.ln->buf+c.nbuf, IndentString))
            {
                c.n+=indentlen;
                c.nbuf+=indentlen;
            }
            break;

        case CTRL('N'):
            BreakLine(c.ln, c.nbuf, yes, c);
            break;

        case SHIFT(INS):
        case CTRL('V'):
            if (Marker.ln)
                Gobble();
            Paste();
            break;

        case CTRL_(INS):
        case CTRL('C'):
            CopyAndClear();
            break;

        case SHIFT(DEL):
        case CTRL('X'):
            Cut();
            break;

        case TAB:
            if (SelectKeyDown) {
                if (Marker.ln) {
                    SelectKeyDown = yes;
                    Indent(yes);
                }
                else {
                    SelectKeyDown = no;
                    if (c.n)
                        c.n--;
                }
            }
            else if (Marker.ln) {
                SelectKeyDown = yes;
                Indent(yes);
            }
            else goto INSERT;
            break;

        case CTRL(']'):
            FindMatchingBracket();
            break;

        case PG_DOWN:
        case SHIFT(PG_DOWN):
            ln = c.ln;
            MoveTo(Cursor.x1, c.ln->y + clientHeight - 30);
            UpdateScrollbars(scrollX, scrollY + (c.ln->y - ln->y));
            break;

        case PG_UP:
        case SHIFT(PG_UP):
            ln = c.ln;
            MoveTo(Cursor.x1, c.ln->y - clientHeight + 30);
            UpdateScrollbars(scrollX, scrollY + (c.ln->y - ln->y));
            break;

        case CTRL_(HOME):
        case CTRL_(PG_UP):
            x = Cursor.x1;
            y = 0;
            MoveTo(x,y);
            break;

        case CTRL_(END):
        case CTRL_(PG_DOWN):
            x = Cursor.x1;
            y = realHeight - 3;
            MoveTo(x,y);
            break;

        case CTRL_(UP):
            if (c.ln->prev)
                y = c.ln->prev->y - c.ln->y;
            else if (c.ln->next)
                y = c.ln->y - c.ln->next->y;
            else break;
            scrollY -= y;
            if (scrollY < 0)
                scrollY = 0;
            PaintWhole();
            break;

        case CTRL_(DOWN):
            if (c.ln->prev)
                y = c.ln->prev->y - c.ln->y;
            else if (c.ln->next)
                y = c.ln->y - c.ln->next->y;
            else break;
            scrollY += y;
            if (scrollY > realHeight - clientHeight)
                scrollY = realHeight - clientHeight;
            PaintWhole();
            break;

        default:
            if (key >= 32) {
                INSERT:
                if (Marker.ln) {
                    StartTransaction();
                    Gobble();
                }
                wchar_t wkey[2];
                wkey[0] = key; wkey[1] = 0;
                WideToUtf8(wkey, buf, sizeof(buf));
                str utf8key = strdup(buf);
                int cn = strlen(utf8key);
                if (insert_mode) {
                    memcpy(buf, c.ln->buf, c.nbuf);                    
                    if (key == TAB) {
                        strcpy(buf+c.nbuf, IndentString);
                        strcpy(buf+c.nbuf+indentlen, c.ln->buf+c.nbuf);
                    }
                    else {
                        strcpy(buf + c.nbuf, utf8key);//, cn);
                        //buf[c.nbuf] = key;
                        strcpy(buf+c.nbuf+cn, c.ln->buf+c.nbuf);  
                    }
                }
                else {     
                    strcpy(buf, c.ln->buf);                        
                    if (key == TAB)
                        strcpy(buf+c.nbuf, IndentString);
                    else {
                        if (buf[c.nbuf] == '\0')                                                                                
                            buf[c.nbuf+1] = '\0';                                
                        buf[c.nbuf] = key; 
                    }
                }
                                    if (strlen(buf) > this->nMaxLineLength)
                {
                    free(utf8key);
                    return no;
                }
                ReplaceLine(c.ln, buf, c);
                if (Marker.ln)
                    FinishTransaction();                    
                if (key == TAB)
                {
                    c.n+=indentlen;                    
                    c.nbuf+=indentlen;                    
                }
                else
                {
                    c.n++;
                    c.nbuf+=cn;
                }
                free(utf8key);
            }
            else return no;
            break;
    }

    /* What about selected text? */
    if (SelectKeyDown and Marker.ln == NULL and temp.ln) {
        /* They have begun a selection */
        Marker = temp;
        UpdateSelection(c);
    }
    else if (Marker.ln and not SelectKeyDown) {
        /* They have aborted a selection. */
        ClearSelection();
    }
    else if (Marker.ln and SelectKeyDown) {
        /* They are continuing to mark a selection. */
        UpdateSelection(c);
    }

    /* What about 'desired_x'? */
    if (key != UP and key != DOWN)
        desired_x = 0;

    /* Reset the cursor regardless of what the keystroke was. */
    /* (Almost any keystroke will cause it to change). */
    SetCursor();
    return yes;
}


static int *KeystrokeBuffer;

str EditorWin::EditOneLine()
/* Like above, except that we only edit a single line. */
{   int response, HistoryIdx;
    static str History[20];
    char buf[16384];
    int orig_n;
    str s,t;

    EditingOneLine = yes;
    if (c.nbuf > strlen(c.ln->buf))
    {
        c.nbuf = strlen(c.ln->buf);
        c.n = utf8len(c.ln->buf);
    }
    orig_n = c.nbuf;
    HistoryIdx = 0;

    do {
        temp = c;
        SetCursor();
        if (ListSize(KeystrokeBuffer)) {
            response = KeystrokeBuffer[0];
            ListDelN(KeystrokeBuffer, 0);
        }
        else response = ScrollWin::GetKey();
        switch (response) {

            case WINDOW_QUIT:
                // (aha) Leave it to the app to quit the main event loop
                //QuitEventLoop = yes;
                EditingOneLine = no;
                return NULL;

            case CTRL('U'):
            case CTRL('Z'):
            case UNDO:
                Undo();
                break;

            case CTRL('R'):
                Redo();
                break;

			case CTRL('A'):
				Marker.ln = (Line*)root;
				Marker.n = 0;
				Marker.nbuf = 0;
                c.nbuf = strlen(c.ln->buf);
				c.n = utf8len(c.ln->buf);
				UpdateSelection(c);
				continue;

            case RIGHT:
                if (c.ln->buf[c.nbuf])
                    MoveRight();
                break;

            case LEFT:
                if (c.nbuf > orig_n)
                    MoveLeft();
                break;

            case INS:
                insert_mode = not insert_mode;
                break;

            case CTRL_(RIGHT):
                do {
                    if (c.ln->buf[c.nbuf] == '\0')
                        break;
                    if (not MoveRight())
                        break;
                } until (IsWordStart(c.nbuf, no));
                break;

            case CTRL_(LEFT):
                do {
                    if (c.nbuf <= orig_n)
                        break;
                    if (not MoveLeft())
                        break;
                } until (IsWordStart(c.nbuf, no));
                break;

            case HOME:
                while (c.nbuf > orig_n)
                {
                    MoveLeft();
                }
                break;

            case END:                    
                while (c.ln->buf[c.nbuf])
                    MoveRight();
                break;

            case DEL:
                if (Marker.ln)
                    Gobble();
                else if (c.ln->buf[c.nbuf]) {
                    int cn = charlen(c.ln->buf + c.nbuf);
                    memcpy(buf, c.ln->buf, c.nbuf);
                    strcpy(buf+c.nbuf, c.ln->buf+c.nbuf+cn);
                    ReplaceLine(c.ln, buf, c);
                }
                break;

            case BACKSPACE:
                {
					if (c.nbuf <= orig_n)
						break;
					if (Marker.ln)
						Gobble();
					if (not insert_mode) {
						MoveLeft();
						break;
					}
					int cn = charlen_bwd(c.ln->buf, c.nbuf);
					memcpy(buf, c.ln->buf, c.nbuf-cn);
					strcpy(buf+c.n-cn, c.ln->buf+c.n);
					ReplaceLine(c.ln, buf, c);
					c.n--;
					c.nbuf -= cn;
					break;
                }

            case '\n':
            case ENTER:
            case ESC:
                goto RETURN;

            case SHIFT(INS):
            case CTRL('V'):
                if (Marker.ln)
                    Gobble();
                s = ClipboardGetText();
                if (strchr(s, '\n')) {
                    t = s;
                    while (*t) {
                        if (*t != '\r')
                            ListAdd(KeystrokeBuffer, *t);
                        t++;
                    }
                    free(s);
                    break;
                }
                free(s);
                Paste();
                break;

            case CTRL_(INS):
            case CTRL('C'):
                CopyAndClear();
                break;

            case SHIFT(DEL):
            case CTRL('X'):
                Cut();
                break;

            case TAB:
                goto INSERT;

            case UP:
                if (HistoryIdx < 19 and History[HistoryIdx+1]) {
                    if (HistoryIdx == 0)
                        free(History[0]),
                        History[0] = strdup(c.ln->buf);
                    HistoryIdx++;
                    ReplaceLine(c.ln, History[HistoryIdx], c);
                    c.n = utf8len(c.ln->buf);
                    c.nbuf = strlen(c.ln->buf);
                }
                break;

            case DOWN:
                if (HistoryIdx > 0 and History[HistoryIdx-1]) {
                    HistoryIdx--;
                    ReplaceLine(c.ln, History[HistoryIdx], c);
                    c.n = utf8len(c.ln->buf);
                    c.nbuf = strlen(c.ln->buf);
                }
                break;

            default:
                if (response >= 32 and response <= 255) {
                    INSERT:
                    if (Marker.ln) {
                        StartTransaction();
                        Gobble();
                    }
                    if (insert_mode) {
                        memcpy(buf, c.ln->buf, c.nbuf);
                        buf[c.nbuf] = response;
                        strcpy(buf+c.nbuf+1, c.ln->buf+c.nbuf);
                    }
                    else {
                        strcpy(buf, c.ln->buf);
                        if (buf[c.nbuf] == '\0')
                            buf[c.nbuf+1] = '\0';
                        buf[c.nbuf] = response;
                    }
                    ReplaceLine(c.ln, buf, c);
                    if (Marker.ln)
                        FinishTransaction();
                    c.n++;
                    c.nbuf++;
                }
                else if (response >= 1000) {
                    QuitEventLoop = yes;
                    goto RETURN;   /* Don't clear the selection. */
                }
                break;
        }

        /* What about selected text? */
        if (SelectKeyDown and Marker.ln == NULL and temp.ln) {
            /* They have begun a selection */
            Marker = temp;
            UpdateSelection(c);
        }
        else if (Marker.ln and not SelectKeyDown) {
            /* They have aborted a selection. */
            ClearSelection();
        }
        else if (Marker.ln and SelectKeyDown) {
            /* They are continuing to mark a selection. */
            if (temp2.ln)
                UpdateSelection(temp2);
        }

    } forever;

    RETURN:
    if (c.ln->buf[0]) {
        /* Add this line to our history: */
        free(History[19]);
        memmove(History+1,History,sizeof(History)-sizeof(str));
        History[1] = strdup(c.ln->buf);
    }
    BreakLine(c.ln, strlen(c.ln->buf), yes, c);
    s = c.ln->buf + orig_n;
    c.ln = (Line*)c.ln->next;
    c.n = 0;
    EditingOneLine = no;
    return s;
}

Line* EditorWin::GetCurrentLine()
{
return c.ln;
}

int EditorWin::GetCurrentColumn()
{
return c.n;
}

/*======================= Some real applications of the class Editor: ====================*/


typedef enum { le_normal=0, le_bold=1, le_italic=2, le_link=4, le_highlight=8, le_heading=16 } le_enum;


static int LexicalAnalyse(str buf, str toks, int les)
{   str s, t;

    s = buf, t = toks;
    les &= 7;
    while (*s) {
        switch (*s++) {
            case 1:     les ^= le_bold;
                        break;
            case 2:     les ^= le_italic;
                        break;
            case 3:     les ^= le_link;
                        break;
            case 4:     les ^= le_heading;
                        break;
        }
        if (t)
            *t++ = les;
    }
    return les;
}


static TfcFont LeToFontAndColour(int le, int *fg_colourp, int *bg_colourp, EditorWin *Editor)
{
    *fg_colourp = (le&le_link) ? Editor->CursorColour : Editor->Foreground;
    *bg_colourp = (le&le_highlight) ? Editor->HighlightColour : Editor->Background;
    if (Editor->OverrideFont) {
        int fontflag = 0;
        int fontsize;
        char buf[128];

        if (le&le_bold)
            fontflag |= TFC_BOLD;
        if (le&le_italic)
            fontflag |= TFC_ITALIC;

        fontsize = TfcFontHeight(Editor->OverrideFont);
        if (le&le_heading)
            fontsize *= 1.2;

        return TfcFindFont(fontsize, fontflag, TfcFontName(Editor->OverrideFont, buf, 128));
    }
    return TfcFindFont(Editor->fontheight + ((le&le_heading) ? 5 : 0),
                (le&le_heading) or ((le&le_bold) != 0), (le&le_italic) != 0,
                Editor->FixedWidth, Editor->fontname);
}


interface void Line::Paragrapher(Paragrapher_type P)
/* Do everything related to the display of this paragraph. */
{   int i, j, W_idx, minY, maxY, maxX, limitX;
    EditorWin *Editor=(EditorWin*)parent;
    struct {
        int x;
        int height;
        str s;
        int sn;
        int snbuf;
        int le;
    } W[200];
    int fg_colour, bg_colour, les;
    unsigned int height, width;
    int le, linebreak, marginX;
    static char *LE;
    str s, sl, sw;
    TfcFont font;
    int x, y;

    LE = (str)realloc(LE, strlen(buf)+1);
    les = LexicalAnalyse(buf, LE, state);
    sl = buf;
    y = P->y1;
    limitX = Editor->PaginatedWidth;
    if (limitX < 10) {
        assert(false);
        return;
    }
    if (*sl == '\5')
        marginX = 20, sl++;
    else marginX = 0;
    maxX = 0;
    do {
        /* Take the first (next) line of the paragraph: */
        W_idx = 0;
        minY = 99999;
        maxY = 0;
        x = P->x1 + marginX;
        sw = sl;

        /* Moving to Home: */
        if (P->pa == pa_xy2cn and P->mouse_x <= marginX and P->mouse_y <= y) {
            P->cn = utf8len(buf,sl);// - buf;
            P->cnbuf = sl  - buf;
            return;
        }

        do {
            /* Take the first (next) word of the line: */
            while (*sw > 0 and *sw < 3)
                sw++;
            s = sw;
            if (*s == '\0')
                break;
            le = (le_enum)LE[s - buf];
            if (*s == '\t') {
                height = Editor->fontheight;
                width = Editor->TabSize - (x % Editor->TabSize);
                if (P->ha >= 0) {
                    i = s - buf;
                    if (i >= P->habuf and i < P->hbbuf)
                        le |= le_highlight;
                }
                s++;
                goto HAVE_WIDTH;
            }
            if (*s > 0 and *s <= ' ') {
                if (*s == '\r') {
                    sw = s + 1;
                    break;
                }
                while (*s <= ' ' and *s and *s != '\t') {
                    s++;
                    if (LE[s - buf] != le)
                        break;
                }
            }
            else {
                while (*s < 0 or *s > ' ') {
                    s++;
                    if (LE[s - buf] != le)
                        break;
                }
                if (*s == ' ')
                    s++;
            }

            /* What about highlighting? */
            if (P->ha >= 0) {
                i = sw - buf;
                j = s - buf;
                if (i < P->hbbuf and j >= P->habuf) {
                    if (i >= P->habuf and j < P->hbbuf)
                        le |= le_highlight;        // The whole word is highlighted
                    else if (i >= P->habuf) {
                        //s = buf;
                        //for (int u=0; u<P->hbbuf;u++)
                        //       s += charlen(s);
                        s = buf + P->hbbuf;        // Highlighting stops halfway through the word
                        le |= le_highlight;
                    }
                    else {
                       //s = buf;
                       //for (int u=0; u<P->ha;u++)
                       //        s += charlen(s); // Highlighting starts halfway through the word
                       s = buf + P->habuf;        
                    }
                }
            }

            /* Measure this word: */
            font = LeToFontAndColour(le, &fg_colour, &bg_colour, Editor);
            if ((unsigned int) *sw > 4)
                TextDimensions(sw,  utf8len(sw, s), font, &width, &height);
            else width = height = 0;
            HAVE_WIDTH:
            if (x + width > limitX) {
                char *old_s=s, *old_sw=sw;
                int old_W_idx=W_idx, old_x=x;

                if (width >= limitX / 2) {
                    /* We have one very long token. Extend the limit to the right. */
                    limitX = 999999;
                    goto HAVE_WIDTH;
                }

                /* Rewind all the tokens that are joined to this token: */
                while (W_idx > 0 and sw[-1] != ' ' and sw[-1] != '\t') {
                    sw = s = W[--W_idx].s;
                    x = W[W_idx].x;
                }

                if (W_idx == 0 or W[W_idx-1].x < limitX / 3) {
                    /* We have a very long connected token. */
                    limitX = 999999;
                    s = old_s;
                    sw = old_sw;
                    W_idx = old_W_idx;
                    x = old_x;
                    goto HAVE_WIDTH;
                }
                break;
            }
            if (minY > height and height)
                minY = height;
            if (maxY < height)
                maxY = height;
            W[W_idx].x = x + Editor->LeftIndent;
            W[W_idx].le = le;
            x += width;
            W[W_idx].height = height;
            W[W_idx].s = sw;
            W[W_idx].sn = utf8len(sw, s);
            W[W_idx].snbuf = s - sw;
            if (++W_idx >= sizeof(W)/sizeof(W[0]))
                assert(false);
            sw = s;
        } while (*sw);

        W[W_idx].x = x + Editor->LeftIndent;
        y += maxY;

        /* Take care of left (or centre) alignment: */
        /*if (sl > buf) {
            if (limitX > 99999)
                alignX = 50;
            else alignX = limitX - 5 - x;
            for (i=0; i <= W_idx; i++)
                W[i].x += alignX;
            x += alignX;
        }*/

        if (maxX < x)
            maxX = x;
        if (maxY == 0)
            minY = 0;
        linebreak = 0;

        if (P->pa == pa_paint) {
            /* We are painting. */
            if (maxY != minY)
                DrawRectangle(P->x1, y - maxY, P->x2, y - minY,
                        Editor->Background, NOCOLOUR);
            if (W[0].x > 0)
                DrawRectangle(0, y - maxY, W[0].x, y,
                            Editor->Background, NOCOLOUR);
            for (i=0; i < W_idx; i++) {
                font = LeToFontAndColour(W[i].le, &fg_colour, &bg_colour, Editor);
                if (W[i].le&le_italic)                      /* do this since letter's rectangle in italic */
                    DrawRectangle(W[i].x, y - W[i].height,  /* not always coincides with bounding rectangle */
                                W[i+1].x, y, bg_colour, NOCOLOUR);
                if (W[i].s[0] == '\t')
                    DrawRectangle(W[i].x, y - W[i].height,
                                W[i+1].x, y, bg_colour, NOCOLOUR);
                else DrawStringU(W[i].s, W[i].sn, font,
                        W[i].x, y - W[i].height,
                        W[i+1].x, y,
                        fg_colour, bg_colour);
            }
            if (W[i].x < P->x2)
                DrawRectangle(W[i].x,
                            y - minY,
                            P->x2,
                            y,
                            (P->hb > 999) ? DARK(CYAN) : Editor->Background, NOCOLOUR);
            if (linebreak)    // Leave a bigger gap after a '\n' as opposed to a '\r'.
                DrawRectangle(P->x1, y, P->x2, y + linebreak, Editor->Background, NOCOLOUR);
        }
        else if (P->pa == pa_cn2rect and (sw - buf > P->cnbuf or *sw == '\0')) {
            /* Have we found the cursor position already? */
            if (W_idx == 0) {
                P->x1 = 0;
                P->x2 = 3;
                return;
            }
            for (i=0; i < W_idx; i++) {
                if (W[i].s[0] > 0 and W[i].s[0] <= 3)
                    continue;
                if (W[i].s + W[i].snbuf >= buf + P->cnbuf)
                    break;
            }
            if (i == W_idx)
                i--;
            font = LeToFontAndColour(W[i].le, &fg_colour, &bg_colour, Editor);
            int n = utf8len(W[i].s, buf + P->cnbuf);
            if (n <= 0)
                width = 0;
            else TextDimensions(W[i].s, n, font, &width, &height);
            P->x1 = W[i].x + width;
            P->y1 = y - W[i].height;
            P->y2 = y;
            if (n + 1 <= 0)
                width = 0;
            else TextDimensions(W[i].s, n+1, font, &width, &height);
            P->x2 = W[i].x + width;
            return;
        }
        else if (P->pa == pa_xy2cn and y + linebreak > P->mouse_y) {
            if (W_idx == 0) {
                P->cn = 0;
                P->cnbuf = 0;
                return;
            }
            else if (P->mouse_x < W[0].x) {
                P->cn = utf8len(buf, W[0].s);
                P->cnbuf = W[0].s - buf;
                return;
            }
            for (i=0; i < W_idx; i++) {
                if (W[i+1].x > P->mouse_x)
                    break;
            }
            if (i >= W_idx) {
                P->cn = utf8len(buf, sw);
                P->cnbuf = sw - buf;
                if (buf[P->cnbuf])
                {
                    P->cn--;
                    P->cnbuf -= charlen_bwd(buf, P->cnbuf);
                }
                return;
            }
            s = W[i].s;
            int len=0, len0=0;
            assert(*s);
            for (j=0; j <= W[i].sn; j++) {
                font = LeToFontAndColour(W[i].le, &fg_colour, &bg_colour, Editor);                    
                TextDimensions(W[i].s, j, font, &width, &height);
                if (W[i].x + width > P->mouse_x) {
                    P->cnbuf = len0;//W[i].s + j - 1 - buf;
                    P->cn = utf8len(buf, W[i].s + len0);
                    assert(P->cn >= 0);
                    return;
                }
                len0 = len;
                len += charlen(W[i].s + len);
            }
            P->cnbuf = W[i].s + j - 1 - buf;
            P->cn = utf8len(buf, W[i].s + j - 1);
            return;
        }

        y += linebreak;
        sl = sw;
    } while (*sl);

    if (P->pa == pa_measure) {
        if (y < Editor->fontheight)
            y = Editor->fontheight;
        P->x2 = maxX;
        P->y2 = y;
    }
    else if (P->pa == pa_cn2rect) {
        if (maxY == 0)
            maxY = Editor->fontheight, y += maxY;
        P->x1 = x;
        P->x2 = x;
        P->y1 = y - maxY;
        P->y2 = y;
    }
    else if (P->pa == pa_xy2cn) {
        P->cnbuf = sl - buf;
        P->cn = utf8len(buf, sl);
    }
    else if (P->pa == pa_paint) {
        if (y < P->y2)
            DrawRectangle(P->x1, y, P->x2, P->y2, Editor->Background);
    }
}





/*----------------- The help viewer: -----------------*/

class HelpViewer : public EditorWin {
public:
    HelpViewer(str title, int width, int height);
    Line* NewLine();
    int DefineState(Line *ln);
    bool Keystroke(int key);
    void WindowClose() { delete this; }
};


static HelpViewer *Viewer=NULL;


HelpViewer::HelpViewer(str title, int width, int height)
: EditorWin(title, width, height, 0xdcffff)
{   str GetSystemFontName(str , int);

    fontheight = 18;
    LeftIndent = 5;
    GetSystemFontName(fontname,50);
    SetCursorBlinkRate(0,0);
}


bool HelpViewer::Keystroke(int key)
{
    switch (key) {
        case WINDOW_QUIT:
        case ESC:        delete Viewer;
                    Viewer = NULL;
                    return yes;

        case UP:
        case DOWN:
        case LEFT:
        case RIGHT:
        case PG_UP:
        case PG_DOWN:
        case CTRL_(LEFT):
        case CTRL_(RIGHT):
        case CTRL_(PG_UP):
        case CTRL_(PG_DOWN):
        case CTRL('F'):
        case CTRL_(INS):
        case CTRL('C'):
        case F3:
        case HOME:
        case END:        return EditorWin::Keystroke(key);

        default:        return yes;
    }
}


interface int HelpDefineState(Line *prev)
{
    if (prev == NULL)
        return le_normal;
    return LexicalAnalyse(prev->buf, NULL, prev->state);
}


int HelpViewer::DefineState(Line *prev)
/* Define this line's state by processing the previous line. */
{
    return HelpDefineState(prev);
}


Line* HelpViewer::NewLine()
/* Construct a line. */
{
    return new Line;
}


interface void ViewHelp(kstr text)
{
	if (text == NULL)
		return;		// No help available.
	ViewHelpWithPosition(400, 600, TfcDisplayStringConverter(text));
}


interface void ViewHelpWithPosition(int w, int h, kstr text)
{
    if (Viewer == NULL)
        Viewer = new HelpViewer("Help", w, h);
    else Viewer->Show();
    Viewer->Clear();
    Viewer->LoadString((str)text);
    int height = Viewer->realHeight + 30 ;
    if (height > 600)
        height = 600;
    Viewer->SetClientHeight(height);
    Viewer->Focus(yes);
    Viewer->CursorOff();
}


interface void ViewHelpFile(str filename)
{
    if (Viewer == NULL)
        Viewer = new HelpViewer("Help", 400, 600);
    else Viewer->Show();
    Viewer->Clear();
    if (not Viewer->LoadFile(filename)) {
        Viewer->LoadString(filename);
        Viewer->LoadString(strerror(errno));
    }
    Viewer->Focus();
}


TfcCallback HelpCallback(kstr helpText)
{
	return TfcCallback(ViewHelp, helpText);
}


TfcGenericMenuItem TfcHelpMenuItem(kstr menuText, kstr helpText)
{
	return TfcMenuItem(menuText, TfcCallback(ViewHelp, helpText));
}



/*----------------- A TTY window: -----------------*/

static Tty *tty;  // This is what 'tfc_printf()' et al use.


class TtyLine : public Line {
public:
    bool Mousestroke(int op, int x, int y);

    friend class Tty;
};


Line* Tty::NewLine()
{
    return new TtyLine;
}


Tty::Tty(str title, int width, int height, int Background, unsigned int _MaxLines,
					ScrollWin *OwnerSw)
		: EditorWin(title, width, height, Background, NOCOLOUR, OwnerSw)
{
    c.ln = InsertLineNoUndo(NULL, "");
    LogFile = NULL;
    gets_buf = NULL;
    MaxLines = _MaxLines > 10 ? _MaxLines : 250;
    prompt_n = 0;
    FixedWidth = yes;
    *buffer = '\0';
    OverrideFont = TfcFindFont(fontheight=14,TFC_FIXED,"Lucida Console");
}


interface bool CtrlCPressed;

bool Tty::Keystroke(int key)
/* This gets called when keystrokes are sent to a Tty when the */
/* Tty is not expecting input, i.e. outside the 'gets' call. */
{
    switch (key) {
        case WINDOW_QUIT:
                    EditorWin::Keystroke(ENTER);
                    if (gets_buf) {
                        *gets_buf = -1;
                        gets_buf[1] = '\0';
                    }
                    else OnEnter("\377");
                    //exit(0);  We don't want to quit here, because this is
                    // a decision the application should make.
                    QuitEventLoop = yes;
                    return yes;

        case CTRL_(INS):
        case CTRL('C'):
                    if (Marker.ln)
                        return EditorWin::Keystroke(key);
                    else CtrlCPressed = yes;
                    return yes;

        default:    if ((key >= ' ' and key < 128) or key == ENTER)
                        ListAdd(KeystrokeBuffer, key);
                    // otherwise 'response' will hold the keystroke.
                    return yes;
    }
}


bool TtyLine::Mousestroke(int op, int x, int y)
{
    Line::Mousestroke(op,x,y);
    return yes;
}


void Tty::RedirectToFile(kstr filename, bool FlushEachLine,
                            bool AppendMode, bool AlsoDisplay)
{
    LogFile = fopen(filename, AppendMode ? "at" : "wt");
    if (LogFile == NULL) {
        TfcMessage(filename, '!', "Can't open log file:\n%s\n\nReason: %s",
                    filename, strerror(errno));
        return;
    }
    this->FlushEachLine = FlushEachLine;
    this->AlsoDisplay = AlsoDisplay;
}


void Tty::Finish(bool WaitForKeystroke)
/* Close the file if we've redirected.           */
/* Wait for a keystroke if we have a log window. */
{   char buf[512];

    if (LogFile)
        fclose(LogFile), LogFile = NULL;
    if (windowTitle and WaitForKeystroke)
        gets(buf);
}


str Tty::puts(const char* text)
/* Buffer the outputs. */
{   str s,d;

    if (this == NULL)
        return (str)text;

    s = (str)text;
    d = buffer+strlen(buffer);
    for (s=(str)text; *s; s++) {
        *d++ = *s;
        if (*s < ' ' or d >= buffer+sizeof(buffer)-1) {
            if (*s == '\r')
                d--;
            *d = '\0';
            Flush();
            d = buffer;
        }
    }
    *d = '\0';
    return (str)text;
}


void Tty::Flush()
{   char obuf[EA_MAX_LINELEN];
    str s,d,t;

    if (LogFile) {
        fputs(buffer, LogFile);
        if (FlushEachLine)
            fflush(LogFile);
        if (not AlsoDisplay)
            return;
    }

    s = buffer;        
    d = obuf;
    if (c.ln == NULL)
        c.ln = InsertLineNoUndo(NULL, "");
        /* if Clear() has been called then need to avoid root == NULL */
    while (c.ln->next)    // We should always be at the bottom!
        c.ln = (Line*)c.ln->next;
            strcpy(obuf, c.ln->buf);                
    d += strlen(d);
    do {
        t = strchr(s, '\n');
        if (t == NULL)
            t = s + strlen(s);
        memcpy(d, s, t - s);
        d[t-s] = '\0';
        s = t;
        t = strchr(obuf, '\b');
        if (t and t > obuf)
            strcpy(t-1,t+1);
        ReplaceLineNoUndo(c.ln, obuf);
        c.n = utf8len(c.ln->buf);
        c.nbuf = strlen(c.ln->buf);
        if (c.n > 1000) {
            c.ln = InsertLineNoUndo(c.ln, "");
            c.n = 0;
        }
        if (*s == '\0')
            break;
        s++;
        c.ln = InsertLineNoUndo(c.ln, "");
        c.n = 0;
        d = obuf;
    } while (*s);
    prompt_n = c.n;

    /* Do we need to cull the lines at the top? */
    if (realHeight > MaxLines * c.ln->height) {
        SuppressPaints();
        if (realHeight > MaxLines * c.ln->height)
            DeleteLine((Line*)root, c);
    }

    /* Paint the changes: */
    *buffer = '\0';
	if (ShowState != tfc_shown or IsMinimised())
		return;
    SetCursor();
    TfcYield(no);
    return;
}


int Tty::printf(const char *fmt, ...)
{   char buf[16384];
    va_list args;
    int n;

    /* Construct the text: */
    va_start(args, fmt);
    buf[sizeof(buf)-1] = '\0';
    n = vsnprintf(buf, sizeof(buf), fmt, args);
    puts(buf);
    assert(buf[sizeof(buf)-1] == '\0');     // Or did we overrun the buffer?
    return n;
}


str Tty::gets(str buf, int sizeof_buf)
{   str s;

    Flush();
    c.ln->SetCursor(c.n);
    prompt_n = c.n;
    gets_buf = buf;
    sizeof_gets_buf = sizeof_buf;
    TfcYield(no);
    response = 0;
    s = EditOneLine();
    if (s == NULL)
        strcpy(buf, "\4");
    else if (strlen(s) < sizeof_buf)
        strcpy(buf, s);
    else {
        strncpy(buf, s, sizeof_buf-1);
        buf[sizeof_buf-1] = '\0';
    }
    gets_buf = NULL;
    if (LogFile)
        fprintf(LogFile, "%s\n", buf);
    return buf;
}


static void EnsureWeHaveTty()
{
    if (tty == NULL) {
        tty = new Tty("Tty", 700, 230, BLACK, 1024);
        tty->Focus();
    }
}


interface void tfc_initTty(str title, int width, int height, int Background)
{
    tty = new Tty(title, width, height, Background, 1024);
    tty->Focus();
}


interface void tfc_printHardcopy(str jobname)
{
	if (tty) {
		tty->Foreground = BLACK;
		tty->Background = WHITE;
		tty->Print(jobname);
	}
}


interface str tfc_puts(const char* buf)
{   int key, save;
    str s;

    save = response;
    EnsureWeHaveTty();
    s = tty->puts(buf);
    if (response == CTRL('S')) {
        do {
            key = tty->GetKey();
        } until (key > 0 and key < 128);
    }
    else if (response == CTRL('C'))
        CtrlCPressed = yes;
    response = save;
    return s;
}


interface char* tfc_gets(char buf[])
{
    EnsureWeHaveTty();
    tty->Focus();
    return tty->gets(buf);
}


interface int tfc_printf(const char *fmt, ...)
{   char buf[16384];
    va_list args;
    int r;

    /* Construct the message: */
    va_start(args, fmt);
    r = vsnprintf(buf, sizeof(buf), fmt, args);
    tfc_puts(buf);
    TfcYield(yes);
    return r;
}


interface int tfc_fprintf(FILE *output, const char *fmt, ...)
{   char buf[16384];
    va_list args;
    int r;

    /* Construct the message: */
    va_start(args, fmt);
    r = vsnprintf(buf, sizeof(buf), fmt, args);
    if (output == stdout or output == stderr)
        tfc_puts(buf);
    else fputs(buf, output);
    return r;
}


interface void tfc_LogToFile(str filename, bool FlushEachLine,
            bool Append, bool AlsoDisplay)
{
    EnsureWeHaveTty();
    tty->RedirectToFile(filename, FlushEachLine, Append, AlsoDisplay);
}


interface void settitle(str title, bool maximise)
{
    EnsureWeHaveTty();
    tty->SetTitle(title);
    if (maximise)
        tty->Maximise();
}


interface void tfc_clearTty()
{
   if (tty)
	   tty->Clear();
}



//------------- utf8 functions ------------------------
int charlen(str charBuf)
{   int len = 0;
	str ptr = charBuf;

	if (!*ptr)                 // empty char
	   return 0;

	if (!(*ptr  & 0x80)) // simple ASCII byte
		 return 1;

	ptr++; len = 1;
	do {   

		 if (!*ptr)
				 return len;

		 if ((*ptr  & 0x80) && !(*ptr & 0x40)) // 0x10xx xxxx byte -> not  leading
				 ptr++,len++;
		 else
				 break;

	} forever;

	return len;
}

int charlen_bwd(char* charBuf, int idx)
{
	int len = 0;
	str ptr = charBuf + idx;

	if (!(*ptr  & 0x80)) // simple ASCII byte
		 return 1;

	do {    
	     
		 if ((*ptr  & 0x80) && !(*ptr & 0x40)) // 0x10xx xxxx byte -> not leading
		 {
				 len++;
				 ptr--;
		 } else
				 return len+1;                                  
	     
		 if (ptr == charBuf)             
				 return len + 1;
	     
	} forever;
	return len;
}

int utf8IndexToChar(char* buf, int charIndex)
{
    str ptr = buf;
    int i=0; 
    do {
            if (i++ == charIndex)
                    break;
            if (!*ptr)
                    break;
            ptr += charlen(ptr);
    } forever;
    return ptr - buf;
}

