/* $Id: textgrid.cpp 437 2005-10-14 05:00:17Z jla $ */


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include "tfc.h"
#include "list.h"
#include "textgrid.h"


#define XY_FLOATER        0xffff


interface str ContextHelp = "The application-writer has not provided any help string.";
static bool Resizing=no;





/*---------------------- Misc fn's: ----------------------*/

interface int ContrastWith(int colour, int background)
/* If 'colour' is sufficiently different from 'background',  */
/* return it.  Otherwise try to modify it to be sufficiently */
/* different from 'background'. */
{       int difference, r0, g0, b0, r1, g1, b1;

        r0 = colour & 0xff;
        g0 = (colour>>8) & 0xff;
        b0 = (colour>>16) & 0xff;
        r1 = background & 0xff;
        g1 = (background>>8) & 0xff;
        b1 = (background>>16) & 0xff;
        difference = abs(r0-r1) + abs(g0-g1) + abs(b0-b1);
        if (difference >= 0xff)
            return colour;
        if (r0 >= r1 and r1 <= 0x80)
            r0 = 0xff;
        else r0 = 0;
        if (g0 >= g1 and g1 <= 0x80)
            g0 = 0xff;
        else g0 = 0;
        if (b0 >= b1 and b1 <= 0x80)
            b0 = 0xff;
        else b0 = 0;
        colour = r0 + (g0<<8) + (b0<<16);
        return colour;
}



/*---------------------- TextCell's: ----------------------*/

class TextCell : public GridCell {
        char line_right, line_below;
        char flags;            // 0=centre, 1=italics
        int fg_colour, bg_colour;
        str buf;
        struct {
            char ch;
            int colour;
        } symbol;

public:
        TextCell(str text, char line_below=1, char line_right=1,
                    int fg_colour=BLACK, int bg_colour=WHITE, void* data=NULL);
        TextCell(TextCell &orig);
        ~TextCell() { free(buf); }

        str GetText() { return buf; }
        void SetText(str s);

        int GetFgColour() { return fg_colour; }
        int GetBgColour() { return bg_colour; }
        void SetColour(int _fg_colour, int _bg_colour);

        char GetFormatBelow() { return line_below; }
        char GetFormatRight() { return line_right; }
        void SetFormatBelow(char line_below);
        void SetFormatRight(char line_below);

        char GetSymbol(int *colourp) { *colourp = symbol.colour; return symbol.ch; }
        void SetSymbol(char ch, int colour);

        void *data;
        void* GetData() { return data; }
        void SetData(void* _data) { data = _data; }

        void Paint(int x1, int y1, int x2, int y2);
                                        // How to paint it
        void CursorPaint();
                                        // How to paint a cursor.
        void Measure(int *widthp, int *heightp);
                                        // Get the width and height
        bool Keystroke(int key);
        bool Mousestroke(int op, int x, int y);

        void GetWinXY(int *xp, int *yp);
        bool GetWinRect(TfcRect *Rect);

        virtual Litewin* DuplicateMe();

        friend class TextGrid;
};



static int LineWidth(char line)
{
        switch (line) {
            case 0: return 0;
            case 1: return 1;
            case 2: return 2;
            case 3: return 3;
            case 4: return 1;            // grey
            default:        return 1;
        }
}


TextCell::TextCell(str text, char _line_below, char _line_right,
        int _fg_colour, int _bg_colour, void* _data)
{
        fg_colour = _fg_colour;
        bg_colour = _bg_colour;
        line_right = _line_right;
        line_below = _line_below;
        buf = strdup(text);
        req_width = req_height = 0;
        /* The dimensions will be measured after we have the TextGrid. */
        symbol.ch = 0;
        symbol.colour = 0;
        data = _data;
}


TextCell::TextCell(TextCell &orig)
{
        i = orig.i;
        j = orig.j;
        fg_colour = orig.fg_colour;
        bg_colour = orig.bg_colour;
        line_right = orig.line_right;
        line_below = orig.line_below;
        buf = strdup(orig.buf);
        req_width = orig.req_width;
        req_height = orig.req_height;
        symbol = orig.symbol;
        flags = orig.flags;
}


Litewin* TextCell::DuplicateMe()
/* Do a deep-copy of this. */
{
        return new TextCell(*this);
}


void TextCell::SetText(str s)
{
        free(buf);
        buf = strdup(s);
        PaintWhole();
}


void TextCell::SetColour(int _fg_colour, int _bg_colour)
{
        if (fg_colour == _fg_colour and bg_colour == _bg_colour)
            return;
        fg_colour = _fg_colour;
        bg_colour = _bg_colour;
        PaintWhole();
}


void TextCell::SetFormatBelow(char _line_below)
{
        if (line_below == _line_below)
            return;
        line_below = _line_below;
        PaintWhole();
}


void TextCell::SetFormatRight(char _line_right)
{
        if (line_right == _line_right)
            return;
        line_right = _line_right;
        PaintWhole();
}


void TextCell::SetSymbol(char ch, int colour)
{
        if (symbol.ch == ch and symbol.colour == colour)
            return;
        symbol.ch = ch;
        symbol.colour = colour;
        PaintWhole();
}


static str SplitLine(char dest[], str buf)
/* Insert a newline character into some place in this str */
/* for less elongated display.  If it's a single word of  */
/* less than 12 chars, don't break it at all. */
{        struct {
            str s;
            int score;
        } best;
        int score, slen;
        str s, d;

        /* Do we even need to break it? */
        slen = strlen(buf);
        if (strlen(buf) <= 8 or (slen <= 14 and not strchr(buf, ' ')))
            return buf;

        /* What's the optimal place to break it? */
        best.s = NULL;
        best.score = -999;
        for (s=buf+1; *s; s++) {
            str mid = buf + slen / 2;
            score = (s[-1] == ' ') ? 20 : isalnum(s[-1]) ? 0 : 10;
            score -= abs(s - mid);
            if (score > best.score) {
                best.s = s;
                best.score = score;
            }
        }
        for (s=buf, d=dest; *s; ) {
            if (s == best.s) {
                *d++ = '\n';
                if (*s == ' ')
                    s++;
                else *d++ = *s++;
            }
            else *d++ = *s++;
        }
        *d = '\0';
        return dest;
}


void TextCell::Measure(int *widthp, int *heightp)
{        TextGrid* grid=(TextGrid*)parent;
        char splitbuf[1024], *s;

        /* Split it into 2 lines if necessary: */
        if (grid->split_lines)
            s = SplitLine(splitbuf, buf);
        else s = buf;

        /* Now get the dimensions. */
        int width, height;
        TextDimensions(s, -1, grid ? grid->font : NULL, &width, &height);
        if (width < 3)
            width = grid->fontheight;
        if (height < grid->fontheight)
            height = grid->fontheight;
        width += 8 + (LineWidth(line_right)+1)*grid->line_width;
        height += LineWidth(line_below)*grid->line_width;
        if (symbol.ch)
            width += grid->fontheight;

        *widthp = width > req_width ? width : req_width;
        *heightp = height > req_height ? height : req_height;
        *widthp = width;
        *heightp = height;
}


void TextCell::Paint(int x1, int y1, int x2, int y2)
{       TextGrid* grid=(TextGrid*)parent;
        char splitbuf[1024], *s;

        /* Are we suppressing paints? */
        if (grid->GetShowState() != tfc_shown)
            return;

        /* What are the dimensions? */
        int w, h;
        if (i == XY_FLOATER)
            w = req_width, h = req_height;
        else w=Width(), h=Height();

        /* Split it into 2 lines if necessary: */
        if (grid->split_lines and not (flags & CENTRED>>4))
            s = SplitLine(splitbuf, buf);
        else s = buf;

        /* Draw the text: */
        TfcFont font = (flags&(ITALICS>>4)) ? TfcFindFont(grid->fontheight, no, yes) : grid->font;
        int Align = (flags&(CENTRED>>4)) ? TFC_CENTRE : (flags&(RIGHTALIGN>>4)) ? TFC_RIGHTALIGN : 0;
        DrawLine(0,0,0,h, bg_colour, 1);
        DrawString(s, -1, font, 1,0,w,h, fg_colour, bg_colour, Align);

        /* Draw the ruler lines: */
        int l = grid->line_width;
        switch (line_right) {
            case 1:     DrawLine(w-1*l,0,w-1*l,h,BLACK);
                        break;
            case 2:     DrawLine(w-1*l,0,w-1*l,h,BLACK);
                        DrawLine(w-2*l,0,w-2*l,h,BLACK);
                        break;
            case 3:     DrawLine(w-1*l,0,w-1*l,h,BLACK);
                        DrawLine(w-2*l,0,w-2*l,h,bg_colour);
                        DrawLine(w-3*l,0,w-3*l,h,BLACK);
                        break;
            case 4:     DrawLine(w-1*l,0,w-1*l,h,GREY);
                        break;
        }
        switch (line_below) {
            case 1:     DrawLine(0,h-1*l,w,h-1*l,BLACK);
                        break;
            case 2:     DrawLine(0,h-1*l,w,h-1*l,BLACK);
                        DrawLine(0,h-2*l,w,h-2*l,BLACK);
                        break;
            case 3:     DrawLine(0,h-1*l,w,h-1*l,BLACK);
                        DrawLine(0,h-2*l,w,h-2*l,bg_colour);
                        DrawLine(0,h-3*l,w,h-3*l,BLACK);
                        break;
            case 4:     DrawLine(0,h-1*l,w,h-1*l,GREY);
                        break;
        }
        if (i == XY_FLOATER) {
            /* It's the floater.cell. */
            DrawRectangle(0,0,w,h,NOCOLOUR,BLACK);
        }

        /* Draw the symbol: */
        if (symbol.ch) {
            int x1,y1,x2,y2;
            x2 = w - LineWidth(line_right)*l - 1;
            y2 = h - LineWidth(line_below)*l - 1;
            x1 = x2 - grid->fontheight + 1;
            y1 = y2 - grid->fontheight + 1;
            DrawEllipse(x1,y1,x2,y2,symbol.colour,NOCOLOUR);
            char two[2] = { symbol.ch, '\0' };
            DrawString(two,-1,grid->font,
                    x1,y1,x2,y2,
                    TfcContrastingColour(symbol.colour), NOCOLOUR,
                    TFC_CENTRE);
        }

        /* If we're drawing a character cursor, don't let the cell cursor blink. */
        if (i == grid->focus.i and j == grid->focus.j)
            CursorPaint();
}


void TextCell::CursorPaint()
{
        int w=Width(),h=Height();
        int colour = ContrastWith(RED, bg_colour);
        w -= LineWidth(line_right);
        h -= LineWidth(line_below);
        DrawRectangle(0, 0, w, h, NOCOLOUR, colour);
        DrawRectangle(1, 1, w-1, h-1, NOCOLOUR, colour);

        /* Draw the character cursor: */
        TextGrid* grid=(TextGrid*)parent;
        if (grid->cursor_n >= 0) {
            unsigned int cx0,cx1,cy0,cy1,ch;
            char splitbuf[2048];
            str s, t;
            int n;

            if (grid->split_lines)
                s = SplitLine(splitbuf, buf);
            else s = buf;
            if ((t=strchr(s, '\n')) != NULL and t < s + grid->cursor_n) {
                n = grid->cursor_n - (t - s);
                s = t + 1;
                cy0 = grid->fontheight;
            }
            else cy0 = 0, n = grid->cursor_n;
            TextDimensions(s, n, grid->font, &cx0, &ch);
            if (flags & (CENTRED>>4)) {
                TextDimensions(s, -1, grid->font, &cx1, &ch);
                cx0 += (w - cx1) / 2;
            }
            if (grid->InsertMode)
                cx1 = cx0 + 3;
            else {
                TextDimensions(s, n+1, grid->font, &cx1, &ch);
                if (cx1 <= cx0 + 3)
                    cx1 = cx0 + 3;
            }
            cy1 = cy0 + ch;
            DrawRectangle(cx0,cy0, cx1,cy1, RED);
        }
}


bool TextCell::Keystroke(int key)
{       extern str ContextHelp;

        if (key == F1) {
            ViewHelp(ContextHelp);
            return yes;
        }
        return no;
}


bool TextCell::Mousestroke(int op, int x, int y)
{       TextGrid* grid=(TextGrid*)parent;

        grid->mouse_x = i;
        grid->mouse_y = j;

        if (grid->floater.cell) {

            /*** If we have a floater.cell: ***/
            return yes;            // Leave it to TextGrid::Mousestroke() to process.
        }
        else {
            grid->MoveTo(i,j);
            if (op == MOUSE_PRESS) {
                grid->floater.offset_x = x;
                grid->floater.offset_y = y;
            }
            return yes;
        }
}







/*---------------------- TextGrid's: ----------------------*/

TextGrid::TextGrid(str title, int width, int height, int bg_colour, tfcgridgrowmode_enum GrowMode)
        : GridWin(title, width, height, bg_colour, GrowMode)
{
        font = NULL;
        fontheight = 0;
        split_lines = no;
        floater.cell = NULL;
        cursor_n = -1;
        Dirty = 2;
        line_width = 1;
        InsertMode = yes;
        
        /* in case the first call to resized comes from ScrollWin */
        origClientWidth = 0;
        origClientHeight = 0;
}


TextGrid::TextGrid(TextGrid &orig)
        : GridWin(orig)
{
        font = orig.font;
        fontheight = orig.fontheight;
        split_lines = orig.split_lines;
        floater.cell = NULL;
        cursor_n = orig.cursor_n;
        line_width = orig.line_width;
        Dirty = orig.Dirty;
}


Litewin *TextGrid::DuplicateMe()
/* Deep copy. */
{
        return new TextGrid(*this);
}


void TextGrid::Set(int x, int y, str text, char line_below, char line_right,
                    int fg_colour, int bg_colour, void* data, int cwidth, int cheight)
/* Create a cell with these properties. */
{       TextCell *cell;
        char flags;

        if (bg_colour == NOCOLOUR)
            bg_colour = Background;
        if (fg_colour == NOCOLOUR)
            fg_colour = ContrastWith(BLACK, bg_colour);
        flags = line_right >> 4;
        line_right &= 15;

        /* Optimise out updates that leave it unchanged: */
        if (cwidth == 1 and cheight == 1) {
            cell = (TextCell*)GridWin::Get(x,y);
            if (cell and
                        (cell->buf == NULL ? text == NULL : streq(cell->buf, text))
                        and
                        cell->line_below == line_below
                        and
                        cell->line_right == line_right
                        and
                        cell->flags == flags
                        and
                        cell->fg_colour == fg_colour
                        and
                        cell->bg_colour == bg_colour
                ) {
                cell->data = data;      // This stuff has nothing to do with layout
                                        // the display, so it doesn't hurt to update it.
                return;
            }
        }

        /* Create a new cell and use it to replace what's there: */
        cell = new TextCell(text,
                line_below, line_right,
                fg_colour, bg_colour,
                data);
        cell->flags = flags;
        if (!font)
          SetParameters(0, split_lines);
        cell->parent = this;
        cell->Measure(&cell->req_width, &cell->req_height);

        if (cwidth == 1 and cheight == 1)
            GridWin::Set(x,y, cell);
        else GridWin::SetExtended(x,y, cell, cwidth, cheight);
}


void TextGrid::Set(int x, int y, int num, char line_below, char line_right,
                    int fg_colour, int bg_colour, void* data, int cwidth, int cheight)
{       char buf[512];

        sprintf(buf, "%d", num);
        Set(x,y,buf,line_below,line_right,fg_colour,bg_colour,data,cwidth,cheight);
}


void TextGrid::Set(int x, int y, double d, char line_below, char line_right,
                    int fg_colour, int bg_colour, void* data, int cwidth, int cheight)
{        char buf[512];

        sprintf(buf, "%f", d);
        Set(x,y,buf,line_below,line_right,fg_colour,bg_colour,data,cwidth,cheight);
}


void TextGrid::SetSymbol(int x, int y, char symbol, int colour)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            return;
        GridCell->SetSymbol(symbol, colour);
}


void TextGrid::SetFormatBelow(int x, int y, char line_below)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            return;
        GridCell->SetFormatBelow(line_below);
}


void TextGrid::SetFormatRight(int x, int y, char line_right)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            return;
        GridCell->SetFormatRight(line_right);
}


void TextGrid::SetColour(int x, int y, int fg_colour, int bg_colour)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            Set(x,y,"",0,0,fg_colour,bg_colour,NULL);
        else GridCell->SetColour(fg_colour, bg_colour);
}


void TextGrid::SetData(int x, int y, void* obj)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            return;
        GridCell->SetData(obj);
}


void TextGrid::SetText(int x, int y, str text)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            return;
        GridCell->SetText(text);
        int new_width, new_height;
        GridCell->Measure(&new_width, &new_height);
        GridCell->Resize(new_width, new_height);
}


char TextGrid::GetSymbol(int x, int y, int *colourp)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            return '\0';
        *colourp = GridCell->symbol.colour;
        return GridCell->symbol.ch;
}


char TextGrid::GetFormatBelow(int x, int y)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            return '\0';
        return GridCell->line_below;
}


char TextGrid::GetFormatRight(int x, int y)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            return '\0';
        return GridCell->line_right;
}


int TextGrid::GetFlags(int x, int y)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            return 0;
        return GridCell->flags << 4;
}


int TextGrid::GetFgColour(int x, int y)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            return NOCOLOUR;
        return GridCell->fg_colour;
}


int TextGrid::GetBgColour(int x, int y)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            return Background;
        return GridCell->bg_colour;
}


void* TextGrid::GetData(int x, int y)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            return NULL;
        return GridCell->data;
}


str TextGrid::GetText(int x, int y)
{
        TextCell *GridCell = Get(x,y);
        if (GridCell == NULL)
            return NULL;
        return GridCell->buf;
}


void TextGrid::SetParameters(int _fontheight, bool _split_lines, int bold, bool italic, bool FixedWidth, char* Face)
/* Set this font and split_lines and recalculates all */
/* row and column sizes based on them. */
{
        SuppressPaints();
        font = TfcFindFont(_fontheight, bold != 0, italic, FixedWidth, Face);
        fontheight = TfcFontHeight(font);
        split_lines = _split_lines;
        RemeasureCells();
        MakeMeasurements();
}

/* Half finished half baked
void TextGrid::ResizeFont(int NewHeight)
{
        char Buf[80];
        char *FontStr;
        int  TmpHeight;

        if (!font)
          font = TfcFindFont(0);

        TfcFontToString(font, Buf + 5);
        fontheight = NewHeight;
        fontheight = (fontheight > 0) ? fontheight : 1;
        // Find where the height substring ends
        FontStr  = strchr(Buf + 5, '-');
        // Wipe the avecharwidth
        while (*++FontStr != '-') *FontStr = '0';
        while (*--FontStr != '-') ;
        // And just rebuild that bit
        while (fontheight)
        {
          FontStr--;
          *FontStr = '0' + (fontheight % 10);
          fontheight /= 10;
        }

        font = TfcStringToFont(FontStr);
        fontheight = TfcFontHeight(font);

        SuppressPaints();
        RemeasureCells();
        MakeMeasurements();
}
Half finished half baked */

void TextGrid::Resized()
{       tfccolourmode_enum colourmode;
        struct {
            int minheight,maxheight;
        } FontRange;
        struct {
            int gwidth, gheight;
            int fontheight;
        } A, B, C;

        /* Skip iconised windows. */
        if (clientWidth <= 32 or clientHeight <= 32)
            return;
        if (ListSize(this->A) == 0)
            return;

        /* Is this a scrollbar call? */
        int dx,dy;
        dx = clientWidth - origClientWidth;
        dy = clientHeight - origClientHeight;
        if (dx < 0)
            dx = -dx;
        if (dy < 0)
            dy = -dy;
        if (dx <= 24 and dy <= 24)
            return;

        // Only adjust font sizes if you are in tfc_colsgrowA mode
        if (GrowMode != tfc_colsgrowA) {
            GridWin::Resized();
            return;
        }

        origClientWidth = clientWidth;
        origClientHeight = clientHeight;

        /* Is this a re-entrant call? */
        /* This happens e.g. when the resizing affects the scrollbars */
        /* which affect the client area which causes a WM_SIZE message. */
        if (Resizing)
            return;
        Resizing = yes;

        /* Recalculate column widths because font will change */
        ListFree(Column);
        Column = NULL;

        /* Set up the line_width: */
        FindDC(&colourmode, &line_width);
        SuppressPaints();

        /* What are the biggest and smallest reasonable fonts? */
        GetFontRange(&FontRange.minheight, &FontRange.maxheight);

        /* Try it with the smallest font: */
        SetParameters(FontRange.minheight, no);

        /* If the average cell is very elongated, try splitting lines: */
        int avWidth = realWidth / numcols;
        int avHeight = realHeight / numrows;
        if (4 * avHeight * clientWidth < avWidth * clientHeight
                and avWidth > avHeight * 8
                and realWidth * FontRange.maxheight / FontRange.minheight > clientWidth)
            SetParameters(FontRange.minheight, yes);
            /* In this case we try to reduce the width at the expense */
            /* of increasing the height by allowing cells to use 2 lines. */

        if (realWidth < clientWidth and realHeight < clientHeight) {
            /* Increase the font-size until we're over 105%. */
            A.fontheight = FontRange.minheight;
            A.gwidth = realWidth;
            A.gheight = realHeight;
            C.fontheight = FontRange.maxheight + 1;
            C.gwidth = 99999;
            C.gheight = 99999;
            do {
                /* At this stage, we think A is too small and C is too big. */
                if (A.fontheight >= C.fontheight - 1)
                    break;
                B.fontheight = (A.fontheight + C.fontheight) / 2;
                SetParameters(B.fontheight, split_lines);
                B.gwidth = realWidth;
                B.gheight = realHeight;
                if (B.gwidth > clientWidth or B.gheight > clientHeight) {
                    /* It's too big. */
                    A = B;
                }
                else {
                    C = B;
                }
            } forever;
            SetParameters(A.fontheight, split_lines);
        }

        /* Set the new cursor rectangle: */
        CalcCursorRect();

        /* Skip all generated WM_SIZE messages: */
        TfcYield(no);   // Actually, this call is
        // suspect - we might have to ditch it.

        Resizing = no;
}


bool TextGrid::Mousestroke(int op, int x, int y)
/* Process a mouse event. */
{
        if (floater.cell) {
            TfcRect R;
            x -= floater.offset_x;
            y -= floater.offset_y;
            R.top = y < floater.y ? y : floater.y;
            R.left = x < floater.x ? x : floater.x;
            R.right = (x > floater.x ? x : floater.x) + floater.cell->req_width;
            R.bottom = (y > floater.y ? y : floater.y) + floater.cell->req_height;
            floater.x = x;
            floater.y = y;
            InvalidateRect(R.left, R.top, R.right, R.bottom);
            if (op == MOUSE_RELEASE) {
                SuppressPaints();
                GetFromXY(  x + floater.offset_x - (floater.middle_x?floater.cell->req_width/2:0),
                            y + floater.offset_y - (floater.middle_y?floater.cell->req_height/2:0),
                            &floater.i,&floater.j);
                delete floater.cell;
                floater.cell = NULL;
                QuitEventLoop = yes;
            }
            return yes;
        }

        return GridWin::Mousestroke(op,x,y);
}


void TextCell::GetWinXY(int *xp, int *yp)
{
        if (i == XY_FLOATER) {
            TextGrid *grid = (TextGrid*)parent;
            parent->GetWinXY(xp,yp);
            *xp += grid->floater.x;
            *yp += grid->floater.y;
        }
        else GridCell::GetWinXY(xp,yp);
}


bool TextCell::GetWinRect(TfcRect *rect)
{
        if (i == XY_FLOATER) {
            TextGrid *grid = (TextGrid*)parent;
            rect->top = grid->floater.y;
            rect->left = grid->floater.x;
            rect->right = grid->floater.y + req_width;
            rect->bottom = grid->floater.x + req_height;
            return no;
        }
        else return GridCell::GetWinRect(rect);
}


void TextGrid::AddPrintHeader()
{        char buf[1024];

        Set(0, numrows, PrintHeader(title, buf), 0, 3<<4,
                    NOCOLOUR, Background, NULL, numcols, 1);

}


void TextGrid::DoDrag(int i, int j, int bg_colour, bool middle_x, bool middle_y, int *ip, int *jp)
/* Set up this cell as a floater.cell and then allow the user */
/* to drag it somewhere. Return the destination in         */
/* *ip and *jp. */
{        int ms_on, ms_off;

        if (floater.cell)
            return;        // We already have a floater!!
        floater.cell = Get(i,j);
        if (floater.cell == NULL)
            return;
        A[j][i] = NULL;
        if (bg_colour == NOCOLOUR)
            Set(i,j,floater.cell->buf,floater.cell->line_below,floater.cell->line_right,
                        floater.cell->fg_colour,floater.cell->bg_colour);
        else Set(i,j,"",floater.cell->line_below,floater.cell->line_right,BLACK,bg_colour);
        floater.cell->i = XY_FLOATER;
        floater.cell->j = XY_FLOATER;
        floater.cell->parent = this;
        floater.cell->line_below = 0;
        floater.cell->line_right = 0;
        floater.x = Column[i];
        floater.y = Row[j];
        floater.middle_x = middle_x;
        floater.middle_y = middle_y;
        if (floater.offset_x > floater.cell->req_width)
            floater.cell->req_width = floater.offset_x + 5;
        if (floater.offset_y > floater.cell->req_height)
            floater.cell->req_height = floater.offset_y + 5;
        GetCursorBlinkRate(&ms_on, &ms_off);
        SetCursorBlinkRate(0,0);
        floater.cell->PaintWhole();
        EventLoop();
        *ip = floater.i;
        *jp = floater.j;
        SetCursorBlinkRate(ms_on, ms_off);
}


void TextGrid::Paint(int x1, int y1, int x2, int y2)
{
        GridWin::Paint(x1,y1,x2,y2);
        if (floater.cell) {
            if (floater.x > x2 or floater.y > y2)
                ;
            else if (floater.x + floater.cell->req_width < x1)
                ;
            else if (floater.y + floater.cell->req_height < y1)
                ;
            else {
                floater.cell->PaintWhole();
            }
        }
}


int TextGrid::GetKey(int i, int j)
/* Get a keystroke for this cell. Some keystrokes are processed */
/* directly by the grid.  */
{
        if (Dirty == 2) {
            origClientWidth = origClientHeight = 0;
            Resized();
            Dirty = 0;
        }
        else if (Dirty == 1)
            Dirty = 0;        //
        MakeMeasurements();
        UpdateScrollbars(scrollX, scrollY);
        cursor_n = -1;
        MoveTo(i,j);
        do {
            /* Get the next keystroke: */
            response = ScrollWin::GetKey();

            /* Is it one we're swallowing? */
            switch (response) {
            case CTRL('P'): Print();
                            break;

            case CTRL('X'): ExportToCsv(title);
                            break;

            /*case AUTO_SAVE: extern void AutoSave(void);
                            AutoSave();
                                continue;*/

            case ZOOM_IN:   Resizing = yes;
                            SetParameters(++fontheight, split_lines);
                            TfcYield(no);
                            Resizing = no;
                            break;

            case ZOOM_OUT:  Resizing = yes;
                            SetParameters(--fontheight, split_lines);
                            TfcYield(no);
                            Resizing = no;
                            break;

            default:        return response;

            }
        } forever;
}


int TextGrid::GetKey(int i, int j, int n, bool insert_cursor)
/* Get a keystroke for character 'n' of the string in this cell. */
{
        cursor_n = n;
        MoveTo(i,j);
        ScrollWin::GetKey();
        cursor_n = -1;
        A[j][i]->PaintWhole();
        return response;
}


str TextGrid::Edit(int x, int y, bool WantIdentifier, int maxwidth, int& response)
/* Edit a text cell. */
{       static str Punctuation = " ,;={}#()[]\t\n\377";
        int n, wantright=0;
        str buf, orig;
        str t;
        
        if (maxwidth > TG_MAX_LINELEN or maxwidth < 2)
            maxwidth = TG_MAX_LINELEN;
        buf = (char*) malloc (maxwidth+1);
        orig = (char*) malloc (maxwidth+1);
        strcpy(buf, GetText(x,y));
        buf[maxwidth] = '\0';
        strcpy(orig, buf);
        if (response == DEL)
            n = 0, response = '\0';
        /*else if (response == BACKSPACE)
            n = strlen(buf), response = '\0';*/
        else if (response == SHIFT(INS) or response == SHIFT(DEL) or
                response == CTRL_(INS) or response == END)
            n = strlen(buf);
        else //buf[n=0] = '\0';
            n = strlen(buf);
        do {
            switch (response) {
                case UNDO:
                            strcpy(buf, orig);
                            SetText(x,y,buf);
                            response = '\0';
                            goto RETURN;
                case BACKSPACE:
                            if (n > 0)
                                strcpy(buf+n-1, buf+n), n--;
                            break;
                case LEFT:  if (n > 0)
                                n--;
                            else goto RETURN;
                            break;
                case RIGHT: if (buf[n] != '\0')
                                n++;
                            else {
                                if (++wantright == 2)
                                    goto RETURN;
                            }
                            break;
                case DEL:   if (buf[n])
                                strcpy(buf+n, buf+n+1);
                            break;
                case HOME:  n = 0;
                            break;
                case END:   n = strlen(buf);
                            break;
                case CTRL_(UP):
                case CTRL_(DOWN):
                            break;
                case BACKTAB:
                case TAB:
                case UP:
                case DOWN:  
                case PG_DOWN:
                case PG_UP: goto RETURN;
                case ESC:
                case ENTER:
                case WINDOW_QUIT:
                case MOUSE_PRESS:
                case MOUSE_DRAG:
                            goto RETURN;
                case INS:   InsertMode = not InsertMode;
                            break;
                case SHIFT(INS):
                            t = ClipboardGetText();
                            if (t) {
                                str d=buf,t0=t;
                                while (*t and t - t0 < 100) {
                                    if (WantIdentifier and strchr(Punctuation, *t))
                                        t++;
                                    else *d++ = *t++;
                                }
                                *d = '\0';
                                free(t0);
                                n = strlen(buf);
                            }
                            break;
                case CTRL_(INS):
                            ClipboardSetText(GetText(x,y));
                            break;
                case SHIFT(DEL):
                            ClipboardSetText(GetText(x,y));
                            buf[0] = '\0';
                            SetText(x,y,buf);
                            response = 0;
                            goto RETURN;
                case CTRL('G'):
                            buf[0] = '\0';
                            response = 0;
                            goto RETURN;
                case 0:     break;
                default:    
                            if (response != SHIFT(DOWN) and response != SHIFT(UP) and 
                                (response >= 170 or response < ' ')) 
                                    goto RETURN;
                            if (response < ' ' or response >= 127)
                                break;
                            if (strlen(buf) >= maxwidth)
                                break;
                            if (response == '"')
                                break;
                            if (WantIdentifier and strchr(Punctuation, response))
                                response = '_';
                            if (InsertMode) {
                                memmove(buf+1+n, buf+n, strlen(buf+n)+1);
                                buf[n++] = response;
                            }
                            else {
                                if (buf[n] == '\0')
                                    buf[n+1] = '\0';
                                buf[n++] = response;
                            }
                            break;
            }
            if (response != RIGHT)
                wantright = 0;
            SetText(x,y,buf);
            if (response == SHIFT(INS) or response == CTRL_(INS)) {
                response = DOWN;
                goto RETURN;
            }
            response = GetKey(x, y, n, InsertMode);

        } forever;

        RETURN:
        free(buf);
        free(orig);
        return GetText(x,y);
}


void TextGrid::ExportToCsv(str suggestedname, int fromrow, int torow)
{       char filename[512];
        FILE *output;
        int x,y, maxrows;
        str s,d;

        /* Clean the suggestedname: */
        d = filename;
        for (s=suggestedname; *s; s++) {
            if (*s < ' ' or *s >= 127 or *s == '/')
                *d++ = '-';
            else *d++ = *s;
        }
        *d = '\0';

        /* Do the file selection box and open the file: */
        if (not TfcSelectFilename(yes, filename, sizeof(filename),
                "Comma-separated values (eg. excel)\0*.csv\0"
                "Tab-separated values (eg. OASIS)\0*.txt\0",
                "csv"))
            return;

        /* Prompt if we're overwriting a file: */
        output = fopen(filename, "rt");
        if (output) {
            fclose(output);
            int result = TfcChoose("Overwrite file", "~Cancel\0Overwrite\0",
                        "The file:\n\n%s\n\nalready exists - do you want to overwrite?", filename);
            if (result != 2)
                return;
        }

        /* Open the file: */
        output = fopen(filename, "wt");
        if (output == NULL) {
            TfcMessage("Export error", '!',
                    "Can't write to:\n\n%s\n\nReason: %s", filename, strerror(errno));
            return;
        }

        /* Do the work: */
        y = fromrow < 0 or fromrow >= numrows ? 0 : fromrow;
        maxrows = torow < 0 or torow < y or torow > numrows ? numrows : torow;
        for (; y < maxrows; y++) {
            for (x=0; x < numcols; x++) {
                s = strdup(GetText(x, y));
                if (y == 0) {                    
                    for (d = s; *d; d++) {
                        if (*d < ' ' or *d >= 127 or *d == '/')
                            *d = ' ';
                    }
                }
                fprintf(output, "%s,", s ? s : "");
                free(s);
            }
            fprintf(output, "\n");
        }
        fclose(output);

        /* The success message: */
        TfcMessage("Export to csv", 'i', "Okay - successful output to:\n\n%s", filename);
}


void TextGrid::SetCharCursor(int n)
/* For a character cursor ('caret') */
{       TextCell *cell;

        if (n == cursor_n)
            return;
        cursor_n = n;
        cell = Get(focus.i,focus.j);
        if (cell)
            cell->PaintWhole();
}



