(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)

INTERFACE Data;
IMPORT Word;

(* Pointee: The smallest addressable unit; its size is given in number of
   bits. Pointer: The address of a pointee; its size is a multiple of the
   pointee size. Immediate: A non-pointer value; its size is the same as a
   pointer. Polymorph: A pointer or immediate value.

   Int: An integer as big as an Immediate (aligned).
   SmallInt: an integer of intermediate size between Pointee and Int, whose 
   size is a multiple of the pointee size (aligned). *)

TYPE (* Storage types *)
  Byte = [0..255]; (* BITSIZE(Byte) = 8 *)
  Short = [0..65535]; (* BITSIZE(Short) = 16 *)
  Long = Word.T;  (* BITSIZE(Long) = 32 *)
  Float = REAL;  (* BITSIZE(Real) = 32 *)

  BytePtr = UNTRACED REF Byte;
  ShortPtr = UNTRACED REF Short;
  LongPtr = UNTRACED REF Long;
  FloatPtr = UNTRACED REF Float;

TYPE (* Program types *)
  Bool = BOOLEAN;
  Char = CHAR;
  SmallInteger = [-32768..32767];  (* BITSIZE(Byte) = 8 *)
  SmallCardinal = [0..65535]; 
  Integer = INTEGER; 
  Cardinal = Word.T; 
  Real = REAL;
  
CONST
  Kilo = 1024; 
  Mega = Kilo * Kilo;

  FirstByteInt = -128; (* = -2**(BitsPerByteInt-1) *) 
  LastByteInt = 127; (* = 2**(BitsPerByteInt-1)-1 *) 
  ByteIntSize = 256; (* = 2**BitsPerByteInt *) 
  FirstSmallInt = -32768; (* = -2**(BitsPerSmallInt-1) *) 
  LastSmallInt = 32767; (* = 2**(BitsPerSmallInt-1)-1 *) 
  SmallIntSize = 65536; (* = 2**BitsPerSmallInt *) 
  FirstSmallCard = 0;
  LastSmallCard = 65535; (* = 2**(BitsPerSmallCard)-1 *) 
  SmallCardSize = 65536; (* = 2**BitsPerSmallInt *) 
  FirstInt = -1073741824; (* = -2**(BitsPerInt-2) *) 
  LastInt = 1073741823; (* = 2**(BitsPerInt-2)-1 *) 
  FirstCard : Cardinal = 0;
  LastCard : Cardinal = Word.Not(0); (* 2**(BitsPerCard)-1 *) 
  FirstFloat = -1073741824.0; (* -- REAL *)
  LastFloat = -1073741823.0; (* -- REAL *)

  OpSize = 2; (* Bytes; size of bytecode opcode. *)
  SmallJumpSize = 2; (* Bytes; size of bytecode relJump *) 

(* Dimensioning. *)
CONST
  LevelStackSize = 64;
  StackSize = 65534;
  TopLevelFP = 4096;
  SharedStackSize = 4096;
  WalkStackSize = 262144;

(* System and version. *)
CONST
  ExternId1 = 'V'; 
  ExternId2 = 'A'; 
  ExternId3 = 'X'; 
  ExternVersion = 1;

  ExternPortableId1 = 'P';
  ExternPortableId2 = 'R';
  ExternPortableId3 = 'T';
  ExternPortableVersion = 1;

CONST
  (* Sizes of Quest Machine data types. *)
  PointeesPerPointee = 1;
  PointeesPerPolymorph = 4;
  PointeesPerImmediate = PointeesPerPolymorph;
  PointeesPerPointer = PointeesPerPolymorph;
  PointeesPerSmallInt = 2;
  PointeesPerSmallCard = 2;
  PointeesPerInt = PointeesPerImmediate;
  PointeesPerCard = PointeesPerImmediate;
  PointeesPerFloat = PointeesPerImmediate;
  BitsPerPointee = 8;
  BitsPerPolymorph = BitsPerPointee * PointeesPerPolymorph;
  BitsPerImmediate = BitsPerPolymorph;
  BitsPerPointer = BitsPerPolymorph;
  BitsPerSmallInt = BitsPerPointee * PointeesPerSmallInt;
  BitsPerSmallCard = BitsPerPointee * PointeesPerSmallCard;
  BitsPerInt = BitsPerPointee * PointeesPerInt;
  BitsPerCard = BitsPerPointee * PointeesPerCard;

CONST
  (* Ranges of unencoded Quest Machine basic types. *)
  Ok = 0;
  FalseBool = 0;
  TrueBool = 1;

  (* Ranges of encoded Quest Machine basic types. *)
  OkImmediate = 2*Ok+1;
  FalseImmediate = 2*FalseBool+1;
  TrueImmediate = 2*TrueBool+1;
  ZeroImmediate = 2*0+1;
  MinPointee (* : Pointee *) = '\000';
  MaxPointee (* : Pointee *) = '\377'; (* = 2**BitsPerPointee-1 *) (* ARCH *)
  PointeeSize = 256; (* = 2**BitsPerPointee *) (* ARCH *)
  MinImmediate = 1; (* odd *)
  MaxImmediate = LastCard; (* odd *)
  MinPointer = 0; (* even *)
  MaxPointer = LastCard  (*-1*); (* -- even *)
  MinPolymorph = MinPointer;

TYPE (* -- ? *)
  (* Unencoded Quest Machine basic types. *)
  SmallCard = SmallCardinal (* [FirstSmallCard .. LastSmallCard] *);
  SmallInt = SmallInteger (* [FirstSmallInt .. LastSmallInt] *);
  Card = Cardinal (* [0 .. LastInt] *);
  Int = Integer (* [FirstInt .. LastInt] *);

  (* Encoded Quest Machine basic types. *)
  Pointee = Char; (* COMP: ARRAY OF Pointee must be a packed type! *)
  Polymorph = Cardinal (* [0 .. MaxCardinal] *);
  Pointer = Cardinal (* even Polymorph *);
  Immediate = Cardinal (* odd Polymorph *);

  (* Dereference. *)
  PointeePtr = UNTRACED REF Pointee;
  PolymorphPtr = UNTRACED REF Polymorph;
  PointerPtr = UNTRACED REF Pointer;
  ImmediatePtr = UNTRACED REF Immediate;
  SmallCardPtr = UNTRACED REF SmallCard;
  SmallIntPtr = UNTRACED REF SmallInt;
  CardPtr = UNTRACED REF Card;
  IntPtr = UNTRACED REF Int;
  RealPtr = UNTRACED REF Real;

PROCEDURE Setup();

(* Errors. *)
PROCEDURE Fault(msg: TEXT);

(* Unencoded types tests. *)
<*INLINE*> PROCEDURE IsRepresentableInt(l: Long): Bool;
<*INLINE*> PROCEDURE IsRepresentableFloat(l: Long): Bool;
<*INLINE*> PROCEDURE IsSmallInt(i: Int): Bool;

(* Encoded types tests. *)
<*INLINE*> PROCEDURE IsImmediate(val: Polymorph): Bool;
<*INLINE*> PROCEDURE IsPointer(val: Polymorph): Bool;

(* Encoding. *)
<*INLINE*> PROCEDURE ImmediateOfBool(bool: Bool): Immediate;
<*INLINE*> PROCEDURE ImmediateOfChar(char: Char): Immediate;
<*INLINE*> PROCEDURE ImmediateOfInt(int: Int): Immediate;
<*INLINE*> PROCEDURE ImmediateOfFloat(float: Float): Immediate;

(* Decoding. *)
<*INLINE*> PROCEDURE BoolOfImmediate(imm: Immediate): Bool;
<*INLINE*> PROCEDURE CharOfImmediate(imm: Immediate): Char;
<*INLINE*> PROCEDURE IntOfImmediate(imm: Immediate): Int;
<*INLINE*> PROCEDURE FloatOfImmediate(imm: Immediate): Float;

(* Alignment. *)
<*INLINE*> PROCEDURE AlignUp(pointer: Pointer; alignment: Card): Pointer;
<*INLINE*> PROCEDURE AlignDown(pointer: Pointer; alignment: Card): Pointer;

END Data.
