(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)

UNSAFE MODULE Exec;
IMPORT Text, QOS, Data, Store, Code, Format, Value;

(*
|				+-----------------------+	\
|		  FP[4]	|        SavePC			|	|
|  FP --------> +-----------------------+   |
|		  FP[0]	|        SaveCP			|	|
|				+-----------------------+	|
|		  FP[-4]|						| 0	|
|				|						| 1	|
|				|		 Temps			| :	|
|				|						|	|
|				|						|	|
|				+-----------------------+	|  Frame
|				|						|	|  size
|				|						|	|
|				|		  Args			| :	|
|				|						| 1	|
|				|						| 0	|
|				+-----------------------+	|
|				|						|	|
|				|						|	|
|				|		  Res			| :	|
|				|						| 1	|
|				|						| 0	|
|				+-----------------------+	/
|
|
|
|				|						|	\
|  TP --------> +-----------------------+   |
|				|   oldTP-stackBase		|	|
|				+-----------------------+	|
|				|		   CP			|	|
|				+-----------------------+	| Temps
|				|     PC-codeBase		|	|
|				+-----------------------+	|
|				|     FP-stackBase		|	|
|				+-----------------------+	|
|				|						|	/
|
|
|				|						|
|  TL --------> +-----------------------+
|				|						|
|
|
|				|						|
|  stackBase -> +-----------------------+
|
|
|  initial state: FP = TL; TP = stackBase; EX = noException; fault = false
|
|
|  normal final state: FP = TL; TP = stackBase; fault = false
|  abnormal final state: exceptionPacket; fault = true
*)

CONST
  FrameSavePCDispl = Data.PointeesPerPolymorph;
  FrameSaveCPDispl = 0;

VAR
  fatalError, breakError, traceError, intError, stringError, realError,
  arrayError,
  readerError, writerError, (* RPC rpcError, *) dynamicError, storeError, 
  osError, caseError, caseCheckError: Value.T;
  fatalErrorCrash, fatalErrorStackOvf, fatalErrorUnimpl: Value.T;

VAR
  (* Machine meta-state *)
  levelStack: Value.T; (* The stack of machine states *)
  LP: Data.Int; (* The level pointer *)
VAR
  (* Machine state *)
  stack: Value.T; (* The current stack *)
  TL: Data.Pointer; (* The top-level (i.e. minimum FP) pointer *)
  TP: Data.Pointer; (* The trap frame pointer *)
  FP: Data.Pointer; (* The frame pointer *)
  CP: Value.T; (* The closure pointer *)
  PC: Data.Pointer; (* The program counter *)
  EX: Value.T; (* Exception packet *)
  home: Value.T; (* A special location *)
  fault: Data.Bool; (* Fault indicator *)

VAR
  selNil, selList, selLitStringCase, selLitBadgeCase, selLitProgCase,
  selLitFormatCase, selLitTypeCase, selLitValueProgCase, selLitValueCase:
    Data.Int;

  noException, saveMachineState: Value.T;

PROCEDURE Setup() =
  VAR string: Value.T;
  BEGIN
    traceOnRaise := FALSE;
    breakOnRaise := FALSE;
    IF Value.NewValString(0, Data.MinPointee, (*out*) string) THEN END;
    noException := Value.NewValRaisePacket(string, Value.valOk);
    saveMachineState := Data.MinPointer;
    home := Data.MinPolymorph;
    levelStack := Value.NewValLevelStack(Data.LevelStackSize);
    LP := Data.FirstCard;
    EX := noException;
    fatalError := Value.ValStringOfChars("fatalError");
    fatalErrorCrash := Value.ValStringOfChars("crash");
    fatalErrorStackOvf := Value.ValStringOfChars("stackOverflow");
    fatalErrorUnimpl := Value.ValStringOfChars("unimplemented");
    breakError := Value.ValStringOfChars("<break>");
    traceError := Value.ValStringOfChars("<trace>");
    intError := Value.ValStringOfChars("intError");
    realError := Value.ValStringOfChars("realError");
    stringError := Value.ValStringOfChars("stringError");
    arrayError := Value.ValStringOfChars("arrayError");
    readerError := Value.ValStringOfChars("readerError");
    writerError := Value.ValStringOfChars("writerError");
    dynamicError := Value.ValStringOfChars("dynamicError");
    storeError := Value.ValStringOfChars("storeError");
    osError := Value.ValStringOfChars("osError");
    caseError := Value.ValStringOfChars("caseError");
    caseCheckError := Value.ValStringOfChars("\'!\'Error");
    (* -- Keep compatible with Literal_TList *)
    selNil := 0;
    selList := 1;
    (* -- Keep compatible with Literal_T, ByteCode_Lit *)
    selLitStringCase := 0;
    selLitBadgeCase := 1;
    selLitProgCase := 2;
    selLitTypeCase := 3;
    selLitValueProgCase := 4;
    selLitValueCase := 5;
  END Setup;

PROCEDURE Msg(msg: TEXT) =
  BEGIN
    IF Text.Length(msg) > 0 THEN
      QOS.OutString(QOS.stdout, "Exec: ");
      QOS.OutString(QOS.stdout, msg);
      QOS.OutChar(QOS.stdout, QOS.newLine);
    END;
  END Msg;

PROCEDURE GetHome(): Value.T = BEGIN RETURN home; END GetHome;

PROCEDURE SetHome(val: Value.T) = BEGIN home := val; END SetHome;

PROCEDURE GetAddr(): Value.T =
  VAR
    int: Data.Int;
    real: Data.Real;
    opAddrClass: Data.SmallCard;
    tuple: Value.T;
    index: INTEGER;
  BEGIN
    (* INLINE opAddrClass := Store.GetSmallCard(PC); *)
    opAddrClass := LOOPHOLE(Store.heapAdr + PC, Data.SmallCardPtr)^;
    (* END INLINE *)
    INC(PC, Data.PointeesPerSmallCard);
    CASE VAL(opAddrClass DIV Data.PointeeSize, Code.OpAddrClass) OF
    | Code.OpAddrClass.OpAddrImmedOkCase => RETURN Value.valOk;
    | Code.OpAddrClass.OpAddrImmedBoolCase =>
    	int := opAddrClass MOD Data.PointeeSize;
        IF int = Data.TrueBool THEN
          RETURN Value.valTrue
        ELSIF int = Data.FalseBool THEN
          RETURN Value.valFalse
	ELSE Data.Fault("");
        END;
    | Code.OpAddrClass.OpAddrImmedCharCase =>
	RETURN Value.NewValChar(VAL(opAddrClass MOD Data.PointeeSize, CHAR));
    | Code.OpAddrClass.OpAddrImmedIntSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
        RETURN Value.NewValInt(int);
    | Code.OpAddrClass.OpAddrImmedIntFullCase =>
        int := Store.GetInt(PC);
        INC(PC, Data.PointeesPerInt);
        RETURN Value.NewValInt(int);
    | Code.OpAddrClass.OpAddrImmedRealCase =>
        real := Store.GetFloat(PC);
        INC(PC, Data.PointeesPerFloat);
        RETURN Value.NewValFloat(real);
    | Code.OpAddrClass.OpAddrLiteralSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
        RETURN
          Value.ValLiteralsGet(
            Value.ValProgGetLiterals(Value.ValClosureProg(CP)), int);
    | Code.OpAddrClass.OpAddrFrameSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
        RETURN Store.GetPolymorph(LOOPHOLE(FP, Data.Int) + int);
    | Code.OpAddrClass.OpAddrFrameLocativeSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
        tuple := Store.GetPolymorph(LOOPHOLE(FP, Data.Int) + int);
        index := Data.IntOfImmediate( Store.GetPolymorph(
           LOOPHOLE(FP, Data.Int) + int + Data.PointeesPerPolymorph));
        RETURN Value.ValTupleGet(tuple, index);
    | Code.OpAddrClass.OpAddrGlobalSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
        RETURN Value.ValClosureGetGlobal(CP, int);
    | Code.OpAddrClass.OpAddrIndexedSmallCase =>
        tuple := GetAddr();
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
        RETURN Value.ValTupleGet(tuple, int);
    | Code.OpAddrClass.OpAddrStackDisplCase =>
	RETURN Value.NewValInt( (LOOPHOLE(FP, Data.Int) - LOOPHOLE(stack, Data.Int)) +
	    Data.IntOfImmediate(GetAddr()));
    | Code.OpAddrClass.OpAddrSpecialCase =>
        CASE VAL(opAddrClass MOD Data.PointeeSize, Code.OpAddrSpecialClass) OF
        | Code.OpAddrSpecialClass.OpAddrSpecialHomeCase => RETURN GetHome();
        | Code.OpAddrSpecialClass.OpAddrSpecialStackCase => RETURN stack;
        | Code.OpAddrSpecialClass.OpAddrSpecialPacketExcCase => 
		RETURN Value.ValRaisePacketExc(EX);
        | Code.OpAddrSpecialClass.OpAddrSpecialPacketValCase =>
		RETURN Value.ValRaisePacketVal(EX);
        ELSE Data.Fault("GetAddr Special");
        END;
    ELSE Data.Fault("GetAddr");
    END;
  END GetAddr;

PROCEDURE GetAddrListSize(): Data.Int =
  VAR int: Data.Int;
  BEGIN
    int := Store.GetSmallInt(PC);
    INC(PC, Data.PointeesPerSmallInt);
    RETURN int;
  END GetAddrListSize;

PROCEDURE SetAddr(val: Value.T) =
  VAR
    int: Data.Int;
    opAddrClass: Data.SmallCard;
    tuple: Value.T;
    index: INTEGER;
  BEGIN
    (* INLINE opAddrClass := Store.GetSmallCard(PC); *)
    opAddrClass := LOOPHOLE(Store.heapAdr + PC, Data.SmallCardPtr)^;
    (* END INLINE *)
    INC(PC, Data.PointeesPerSmallCard);
    CASE VAL(opAddrClass DIV Data.PointeeSize, Code.OpAddrClass) OF
    | Code.OpAddrClass.OpAddrLiteralSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
        Value.ValLiteralsSet(
          Value.ValProgGetLiterals(Value.ValClosureProg(CP)), int, val);
    | Code.OpAddrClass.OpAddrFrameSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
        Store.SetPolymorph(LOOPHOLE(FP, Data.Int) + int, val);
    | Code.OpAddrClass.OpAddrFrameLocativeSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
        tuple := Store.GetPolymorph(LOOPHOLE(FP, Data.Int) + int);
        index := Data.IntOfImmediate( Store.GetPolymorph(
           LOOPHOLE(FP, Data.Int) + int + Data.PointeesPerPolymorph));
        Value.ValTupleSet(tuple, index, val);
    | Code.OpAddrClass.OpAddrGlobalSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
        Value.ValClosureSetGlobal(CP, int, val);
    | Code.OpAddrClass.OpAddrIndexedSmallCase =>
        tuple := GetAddr();
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
        Value.ValTupleSet(tuple, int, val);
    | Code.OpAddrClass.OpAddrSpecialCase =>
        CASE VAL(opAddrClass MOD Data.PointeeSize, Code.OpAddrSpecialClass) OF
        | Code.OpAddrSpecialClass.OpAddrSpecialHomeCase => SetHome(val);
        | Code.OpAddrSpecialClass.OpAddrSpecialStackCase => stack := val;
        ELSE Data.Fault("SetAddr Special");
        END;
    ELSE Data.Fault("SetAddr");
    END;
  END SetAddr;

PROCEDURE RelativeState(
    stack: Value.T;
    VAR (*in-out*) TL, TP, FP: Data.Pointer;
    CP: Value.T;
    VAR (*in-out*) PC: Data.Pointer) =
  BEGIN
    DEC(TL, stack);
    DEC(TP, stack);
    DEC(FP, stack);
    DEC(PC, Value.ValProgCodeStart(Value.ValClosureProg(CP)));
  END RelativeState;

PROCEDURE AbsoluteState(
    stack: Value.T;
    VAR (*in-out*) TL, TP, FP: Data.Pointer;
    CP: Value.T;
    VAR (*in-out*) PC: Data.Pointer) =
  BEGIN
    INC(TL, stack);
    INC(TP, stack);
    INC(FP, stack);
    INC(PC, Value.ValProgCodeStart(Value.ValClosureProg(CP)));
  END AbsoluteState;

PROCEDURE PushTrapFrame(
    stack: Value.T;
    displ: Data.Int;
    TL, TP, FP: Data.Pointer;
    CP: Value.T;
    PC: Data.Pointer)
    : Data.Pointer =
  VAR trapFrame, i: Data.Pointer;
  BEGIN
    trapFrame := LOOPHOLE(LOOPHOLE(FP, Data.Int) + displ, Data.Pointer);
    RelativeState(stack, (*in-out*) TL, (*in-out*) TP, (*in-out*) FP, CP,
      (*in-out*) PC);
    i := trapFrame;
    Store.SetImmediate(i, Data.ImmediateOfInt(TP));
    i := i - Data.PointeesPerPolymorph;
    Store.SetPointer(i, CP);
    i := i - Data.PointeesPerPolymorph;
    Store.SetImmediate(i, Data.ImmediateOfInt(PC));
    i := i - Data.PointeesPerPolymorph;
    Store.SetImmediate(i, Data.ImmediateOfInt(FP));
    RETURN trapFrame;
  END PushTrapFrame;

PROCEDURE FlushTrapFrame(
    stack: Value.T;
    TP: Data.Pointer)
    : Data.Pointer =
  BEGIN
    IF TP = stack THEN Data.Fault("Trap frames undeflow"); END;
    TP := Data.IntOfImmediate(Store.GetImmediate(TP));
    TP := TP + stack;
    RETURN TP;
  END FlushTrapFrame;

PROCEDURE PopTrapFrame(
    stack: Value.T;
    TL, TP: Data.Pointer;
    VAR (* out *) FP: Data.Pointer;
    VAR (* out *) CP: Value.T;
    VAR (* out *) PC: Data.Pointer)
    : Data.Pointer =
  VAR i: Data.Pointer;
  BEGIN
    IF TP = stack THEN Data.Fault("Trap frames undeflow"); END;
    i := TP;
    TP := Data.IntOfImmediate(Store.GetImmediate(i));
    i := i - Data.PointeesPerPolymorph;
    CP := Store.GetPointer(i);
    i := i - Data.PointeesPerPolymorph;
    PC := Data.IntOfImmediate(Store.GetImmediate(i));
    i := i - Data.PointeesPerPolymorph;
    FP := Data.IntOfImmediate(Store.GetImmediate(i));
    AbsoluteState(stack, (*in-out*) TL, (*in-out*) TP, (*in-out*) FP, CP,
      (*in-out*) PC);
    RETURN TP;
  END PopTrapFrame;

PROCEDURE Exec(machineState: Value.T): Value.T =
  BEGIN
    Value.ValMachineStateGet(machineState, (* out *) stack,
      (* out *) TL, (* out *) TP, (* out *) FP, (* out *) CP, (* out *) PC,
      (* out *) EX, (* out *) home, (*out*) fault);
    AbsoluteState(stack, (* in-out *) TL,
      (* in-out *) TP, (* in-out *) FP, CP, (* in-out *) PC);
    Execute();
    RelativeState(stack, (* in-out *) TL, (* in-out *) TP,
      (* in-out *) FP, CP, (* in-out *) PC);
    RETURN
      Value.NewValMachineState(stack, TL, TP, FP, CP, PC, EX, home, fault);
  END Exec;

PROCEDURE ReentrantExecProg(
    prog: Value.T;
    newStack: Value.T;
    newRelTL: Data.Pointer)
    : Value.T =
  VAR machineState: Value.T;
  BEGIN
    IF saveMachineState # Data.MinPointer THEN
      Data.Fault("ReentrantExecProg");
    END;
    saveMachineState :=
      Value.NewValMachineState(stack, TL, TP, FP, CP, PC, EX, home, fault);
    machineState := ExecProg(prog, newStack, newRelTL);
    Value.ValMachineStateGet(saveMachineState, (* out *) stack,
      (* out *) TL, (* out *) TP, (* out *) FP, (* out *) CP,
      (* out *) PC, (* out *) EX, (*out*) home, (*out*) fault);
    saveMachineState := Data.MinPointer;
    RETURN machineState;
  END ReentrantExecProg;

PROCEDURE ExecProg(
    prog: Value.T;
    newStack: Value.T;
    newRelTL: Data.Pointer)
    : Value.T =
  BEGIN
    stack := newStack;
    TL := newRelTL + stack;
    TP := stack;
    FP := TL;
    CP := Value.NewValClosure(prog, 0);
    PC := Value.ValProgCodeStart(prog);
    EX := noException;
    fault := FALSE;
    Execute();
    RelativeState(stack, (* in-out *) TL, (* in-out *) TP,
      (* in-out *) FP, CP, (* in-out *) PC);
    RETURN
      Value.NewValMachineState(stack, TL, TP, FP, CP, PC, EX, home, fault);
  END ExecProg;

PROCEDURE MachineExec() =
  VAR
    val, val1, val2, val3: Value.T;
    bootProg: Value.T;
    newStack: Value.T;
    newRelTL, newRelTP, newRelFP: Data.Pointer;
    newCP: Value.T;
    newRelPC: Data.Pointer;
    newEX: Value.T;
    newHome: Value.T;
    newFault: Data.Bool;
  BEGIN
    val1 := GetAddr();
    val2 := GetAddr();
    val3 := GetAddr();
    bootProg := BootConvertProg(val1);
    newStack := val2;
    newRelTP := 0;
    newRelTL := Data.IntOfImmediate(val3);
    newRelFP := newRelTL;
    newCP := Value.NewValClosure(bootProg, 0);
    newRelPC := 0;
    newEX := noException;
    newHome := Value.valOk;
    newFault := FALSE;
    val :=
      Value.NewValMachineState(newStack, newRelTL, newRelTP, newRelFP,
        newCP, newRelPC, newEX, newHome, newFault);
    Start(val);
  END MachineExec;

PROCEDURE MachineResume() =
  VAR val: Value.T;
  BEGIN val := GetAddr(); Start(val); END MachineResume;

PROCEDURE MachineGoto() =
  VAR val1, val2: Value.T; gotoProg: Value.T;
  BEGIN
    val1 := GetAddr();
    val2 := GetAddr();
    gotoProg := BootConvertProg(val2);
    StartGoto(val1, gotoProg);
  END MachineGoto;

PROCEDURE ValProgFrameSize(prog: Data.Pointer): Data.Card =
  (* Assumes the first instruction is an OpFrame *)
  VAR start: Data.Pointer;
  BEGIN
    start := Value.ValProgCodeStart(prog);
    INC(start, Data.OpSize); (* skip OpFrame opcode *)
    RETURN Store.GetSmallInt(start);
  END ValProgFrameSize;

PROCEDURE PopFrame(machineState: Value.T): Value.T =
  VAR
    outStack: Value.T;
    outTL, outTP, outFP: Data.Pointer;
    outCP: Value.T;
    outPC: Data.Pointer;
    outEX: Value.T;
    outHome: Value.T;
    outFault: Data.Bool;
  BEGIN
    Value.ValMachineStateGet(machineState, (* out *) outStack,
      (* out *) outTL, (* out *) outTP, (* out *) outFP, (* out *) outCP,
      (* out *) outPC, (* out *) outEX, (* out *) outHome,
      (*out*) outFault);
    AbsoluteState(outStack, (* in-out *) outTL,
      (* in-out *) outTP, (* in-out *) outFP, outCP, (* in-out *) outPC);
    DEC(outFP,
      ValProgFrameSize(Value.ValClosureProg(outCP)) -
        Data.PointeesPerPolymorph);
    (* FP is now 1 above normal *)
    outCP := Store.GetPolymorph(
        LOOPHOLE(outFP, Data.Int) - Data.PointeesPerPolymorph + FrameSaveCPDispl);
    outPC := Store.GetPolymorph(
        LOOPHOLE(outFP, Data.Int) - Data.PointeesPerPolymorph + FrameSavePCDispl);
    DEC(outFP, Data.PointeesPerPolymorph);
    (* -- Pop TP if pointing inside current frame? *)
    RelativeState(outStack, (* in-out *) outTL, (* in-out *) outTP,
      (* in-out *) outFP, outCP, (* in-out *) outPC);
    RETURN
      Value.NewValMachineState(outStack, outTL, outTP, outFP, outCP, outPC,
        outEX, outHome, outFault);
  END PopFrame;

VAR
  debugTP, debugFP: Data.Pointer;
  debugCP: Value.T;
  debugPC: Data.Pointer; 

PROCEDURE Execute() =
  CONST PathBufferSize = 256;
  VAR
    opClass: Data.SmallCard;
    pointee: Data.Pointee;
    val, val1, val2, val3, val4, val5, val6: Value.T;
    size, i, index, addrListSize, resultsNo, displ: Data.Int;
    r: Data.Float;
    jump: BOOLEAN;
    trace, break, exit: BOOLEAN;
    frameSize, frameTempNo: Data.Int;
    test: Data.Bool;
    char: Data.Char;
    int: Data.Int;
    long: Data.Long;
    dirNameStart, dirNameLength, length: Data.Int;
    nextCP: Value.T;
  BEGIN
    trace := FALSE;
    break := FALSE;
    exit := FALSE;
    LOOP
      (* INLINE opClass := Store.GetSmallCard(PC); *)
      opClass := LOOPHOLE(Store.heapAdr + PC, Data.SmallCardPtr)^;
      (* END INLINE *)
      INC(PC, Data.PointeesPerSmallCard);
      (* Do trace and break at the end, so it can execute an OpFrame first *)
      IF opClass >= 16_4000 THEN
        IF opClass >= 16_8000 THEN trace := TRUE; DEC(opClass, 16_8000); END;
        IF opClass >= 16_4000 THEN break := TRUE; DEC(opClass, 16_4000); END;
      END;
      CASE VAL(opClass DIV Data.PointeeSize, Code.OpClass) OF
      | Code.OpClass.OpMoveCase => SetAddr(GetAddr());
      | Code.OpClass.OpApplyCase =>
          Store.SetPolymorph(LOOPHOLE(FP, Data.Int) + FrameSaveCPDispl,CP);
          CP := GetAddr();
          (* JSB *)
          INC(FP, Data.PointeesPerPolymorph);
          Store.SetPolymorph( LOOPHOLE(FP, Data.Int) -
             Data.PointeesPerPolymorph + FrameSavePCDispl, PC);
          PC := Value.ValProgCodeStart(Value.ValClosureProg(CP));
      | Code.OpClass.OpFrameCase =>
          (* FP is 1 above normal *)
          frameSize := Store.GetSmallInt(PC);
          INC(PC, Data.PointeesPerSmallInt);
          frameTempNo := Store.GetSmallInt(PC);
          INC(PC, Data.PointeesPerSmallInt);
          (* -- Clear result. -- assumes single result. Actually, OpArguments
		should do this because it knows the no of results *)
          Store.SetPolymorph(LOOPHOLE(LOOPHOLE(FP, Data.Int) +
             Data.PointeesPerPolymorph, Data.Pointer), Data.MinPolymorph);
          (* Reposition FP for new frame. *)
          FP :=
            LOOPHOLE(
              LOOPHOLE(FP, Data.Int) + (frameSize - Data.PointeesPerPolymorph), Data.Pointer);
	  size := Value.ValArraySize(stack);
          IF FP > stack + LOOPHOLE(size, Data.Pointer) THEN
	    Data.Fault("Stack Overflow");
            (* ---- Msg("Stack Overflow"); *)
            exit := RaiseAndUnwind(fatalError, fatalErrorStackOvf);
          ELSE
            (* -- Clear temps *)
            FOR i := 0 TO frameTempNo - 1 DO
              Store.SetPolymorph( LOOPHOLE(LOOPHOLE(FP, Data.Int) -
                 Data.PointeesPerPolymorph * (i + 1 (* saveCP *) ), Data.Pointer),
                 Data.MinPolymorph);
            END;
          END;
      | Code.OpClass.OpReturnCase => (* -- assumes 1 result *)
          frameSize := Store.GetSmallInt(PC);
          INC(PC, Data.PointeesPerSmallInt);
          FP :=
            LOOPHOLE(
              LOOPHOLE(FP, Data.Int) -
                (frameSize - Data.PointeesPerPolymorph), 
	      Data.Pointer);
          (* FP is now 1 above normal *)
          CP := Store.GetPolymorph( LOOPHOLE(FP, Data.Int) -
             Data.PointeesPerPolymorph + FrameSaveCPDispl);
          (* RET *)
          PC := Store.GetPolymorph( LOOPHOLE(FP, Data.Int) -
             Data.PointeesPerPolymorph + FrameSavePCDispl);
          DEC(FP, Data.PointeesPerPolymorph);
      | Code.OpClass.OpReturnObsoleteCase => (* -- assumes 1 result *)
          frameSize := Store.GetSmallInt(PC);
          INC(PC, Data.PointeesPerSmallInt);
          FP :=
            LOOPHOLE(
              LOOPHOLE(FP, Data.Int) -
                (frameSize - Data.PointeesPerPolymorph), 
	      Data.Pointer);
          (* FP is now 1 above normal *)
          CP := Store.GetPolymorph( LOOPHOLE(FP, Data.Int) -
             Data.PointeesPerPolymorph + FrameSaveCPDispl);
          (* RET *)
          PC := Store.GetPolymorph( LOOPHOLE(FP, Data.Int) -
             Data.PointeesPerPolymorph + FrameSavePCDispl);
          DEC(FP, Data.PointeesPerPolymorph);
      | Code.OpClass.OpArgumentsCase =>
          resultsNo := Store.GetSmallInt(PC);
          INC(PC, Data.PointeesPerSmallInt);
          addrListSize := GetAddrListSize();
          i := (1 (* SavePC *) + resultsNo) * Data.PointeesPerPolymorph;
          WHILE addrListSize > 0 DO
            INC(i, Data.PointeesPerPolymorph);
            Store.SetPolymorph(LOOPHOLE(LOOPHOLE(FP, Data.Int) + i, Data.Pointer),
               GetAddr());
            DEC(addrListSize);
          END;
      | Code.OpClass.OpApplyArguments0Results1Case =>
          Store.SetPolymorph(LOOPHOLE(FP, Data.Int) + FrameSaveCPDispl,CP);
          CP := GetAddr();
          (* JSB *)
          INC(FP, Data.PointeesPerPolymorph);
          Store.SetPolymorph( LOOPHOLE(FP, Data.Int) -
             Data.PointeesPerPolymorph + FrameSavePCDispl, PC);
          PC := Value.ValProgCodeStart(Value.ValClosureProg(CP));
      | Code.OpClass.OpApplyArguments1Results1Case =>
	  nextCP := GetAddr();
          i := Data.PointeesPerPolymorph (* SavePC *) + 
		Data.PointeesPerPolymorph (* resultsNo *) +
		Data.PointeesPerPolymorph;
          Store.SetPolymorph(LOOPHOLE(LOOPHOLE(FP, Data.Int) + i, 
	    Data.Pointer), GetAddr());
          Store.SetPolymorph(LOOPHOLE(FP, Data.Int) + FrameSaveCPDispl,CP);
          CP := nextCP;
          (* JSB *)
          INC(FP, Data.PointeesPerPolymorph);
          Store.SetPolymorph( LOOPHOLE(FP, Data.Int) -
             Data.PointeesPerPolymorph + FrameSavePCDispl, PC);
          PC := Value.ValProgCodeStart(Value.ValClosureProg(CP));
      | Code.OpClass.OpApplyArguments2Results1Case =>
	  nextCP := GetAddr();
          i := Data.PointeesPerPolymorph (* SavePC *) + 
		Data.PointeesPerPolymorph (* resultsNo *); 
          INC(i, Data.PointeesPerPolymorph);
          Store.SetPolymorph(LOOPHOLE(LOOPHOLE(FP, Data.Int) + i, 
	    Data.Pointer), GetAddr());
          INC(i, Data.PointeesPerPolymorph);
          Store.SetPolymorph(LOOPHOLE(LOOPHOLE(FP, Data.Int) + i, 
	    Data.Pointer), GetAddr());
          Store.SetPolymorph(LOOPHOLE(FP, Data.Int) + FrameSaveCPDispl,CP);
          CP := nextCP;
          (* JSB *)
          INC(FP, Data.PointeesPerPolymorph);
          Store.SetPolymorph( LOOPHOLE(FP, Data.Int) -
             Data.PointeesPerPolymorph + FrameSavePCDispl, PC);
          PC := Value.ValProgCodeStart(Value.ValClosureProg(CP));
      | Code.OpClass.OpApplyArgumentsResults1Case =>
	  nextCP := GetAddr();
          addrListSize := GetAddrListSize();
          i := Data.PointeesPerPolymorph (* SavePC *) + 
		Data.PointeesPerPolymorph (* resultsNo *); 
          WHILE addrListSize > 0 DO
            INC(i, Data.PointeesPerPolymorph);
            Store.SetPolymorph(LOOPHOLE(LOOPHOLE(FP, Data.Int) + i, 
	      Data.Pointer), GetAddr());
            DEC(addrListSize);
          END;
          Store.SetPolymorph(LOOPHOLE(FP, Data.Int) + FrameSaveCPDispl,CP);
          CP := nextCP;
          (* JSB *)
          INC(FP, Data.PointeesPerPolymorph);
          Store.SetPolymorph( LOOPHOLE(FP, Data.Int) -
             Data.PointeesPerPolymorph + FrameSavePCDispl, PC);
          PC := Value.ValProgCodeStart(Value.ValClosureProg(CP));
      | Code.OpClass.OpResultsCase =>
          addrListSize := GetAddrListSize();
          i := Data.PointeesPerPolymorph; (* SavePC *)
          WHILE addrListSize > 0 DO
            i := i + Data.PointeesPerPolymorph;
            SetAddr(Store.GetPolymorph(LOOPHOLE(FP, Data.Int) + i));
            DEC(addrListSize);
          END;
      | Code.OpClass.OpResults1Case =>
          SetAddr(Store.GetPolymorph(LOOPHOLE(FP, Data.Int) + 
	    Data.PointeesPerPolymorph(*SavePC*) + Data.PointeesPerPolymorph));
      | Code.OpClass.OpEndCase => (* -- being replaced by OpStop *)
          IF LP # 0 THEN
            Msg("OpEnd: levelStack not empty");
            exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          ELSE
            exit := TRUE;
          END;
      | Code.OpClass.OpStartCase =>
          Start(GetAddr());
          (* Destination address not used until later *)
      | Code.OpClass.OpStopCase =>
          (* IF FP # TL THEN Msg("FP # TL on OpStop") END; *)
          (* IF TP # stack THEN Msg("TP # stack on OpStop") END; *)
          fault := FALSE;
          IF LP = 0 THEN exit := TRUE;
          ELSE
            val := Stop();
            (* This result address was left around by OpStart or OpMachineExec!
            *)
            SetAddr(val);
          END;
      | Code.OpClass.OpClosureCase =>
          val1 := GetAddr();
          addrListSize := GetAddrListSize();
          val2 := Value.NewValClosure(val1, addrListSize);
          i := 0;
          WHILE i < addrListSize DO
            Value.ValClosureSetGlobal(val2, 
		i * Data.PointeesPerPolymorph,
              GetAddr());
            INC(i);
          END;
          SetAddr(val2);
      | Code.OpClass.OpDumClosureCase =>
          val1 := GetAddr();
          val2 :=
            Value.NewValClosure(val1, Data.IntOfImmediate(GetAddr()));
          SetAddr(val2);
      | Code.OpClass.OpRecClosureCase =>
          val := GetAddr();
          addrListSize := GetAddrListSize();
          i := 0;
          WHILE i < addrListSize DO
		Value.ValClosureSetGlobal(val, i * Data.PointeesPerPolymorph,
              GetAddr());
            INC(i);
          END;
      | Code.OpClass.OpNoOp =>
      | Code.OpClass.OpJumpCase =>
          i := Store.GetRelJump(PC);
          PC := LOOPHOLE(LOOPHOLE(PC, Data.Int) + i, Data.Pointer);
      | Code.OpClass.OpJumpWhenCase =>
          val1 := GetAddr();
          val2 := GetAddr();
          CASE VAL(opClass MOD Data.PointeeSize, Code.OpJumpClass) OF
          | Code.OpJumpClass.OpJumpAlwaysCase => jump := TRUE;
          | Code.OpJumpClass.OpJumpEqCase => 
	    jump := Value.Same(val1, val2);
          | Code.OpJumpClass.OpJumpNotEqCase => 
	    jump := NOT Value.Same(val1, val2);
          | Code.OpJumpClass.OpJumpLessCase =>
              jump := Data.IntOfImmediate(val1) <
		Data.IntOfImmediate(val2);
          | Code.OpJumpClass.OpJumpLessEqCase =>
              jump := Data.IntOfImmediate(val1) <=
		Data.IntOfImmediate(val2);
          | Code.OpJumpClass.OpJumpMoreCase =>
              jump := Data.IntOfImmediate(val1) >
		Data.IntOfImmediate(val2);
          | Code.OpJumpClass.OpJumpMoreEqCase =>
              jump := Data.IntOfImmediate(val1) >=
		Data.IntOfImmediate(val2);
          END;
          IF jump THEN
            i := Store.GetRelJump(PC);
            PC := LOOPHOLE(LOOPHOLE(PC, Data.Int) + i, Data.Pointer);
          ELSE
            INC(PC, Data.SmallJumpSize);
          END;
      | Code.OpClass.OpDataCompareCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataCompareClass) OF
          | Code.OpDataCompareClass.OpSameCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(Value.NewValBool(Value.Same(val1,val2)));
          | Code.OpDataCompareClass.OpOtherCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(Value.NewValBool(
		NOT (Value.Same(val1, val2))));
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataAsciiCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataAsciiClass) OF
          | Code.OpDataAsciiClass.OpAsciiCharCase =>
              val1 := GetAddr();
              SetAddr(Value.NewValChar(
		VAL(Data.IntOfImmediate(val1), CHAR)));
          | Code.OpDataAsciiClass.OpAsciiValCase =>
              val1 := GetAddr();
              (* Compiler bug? expression broken into pieces:
                 SetAddr(Value.NewValInt(ORD(Data.CharOfImmediate(val1))));
              *)
              char := Data.CharOfImmediate(val1);
              int := ORD(char);
              val := Value.NewValInt(int);
              SetAddr(val);
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataBoolCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataBoolClass) OF
          | Code.OpDataBoolClass.OpBoolNotCase =>
              val1 := GetAddr();
              SetAddr(Value.NewValBool(NOT (Data.BoolOfImmediate(val1))));
          | Code.OpDataBoolClass.OpBoolAndCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValBool(
                  Data.BoolOfImmediate(val1)
                  AND Data.BoolOfImmediate(val2)));
          | Code.OpDataBoolClass.OpBoolOrCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValBool(
                  Data.BoolOfImmediate(val1)
                  OR Data.BoolOfImmediate(val2)));
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataIntCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataIntClass) OF
          | Code.OpDataIntClass.OpIntPlusCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val :=
                Value.NewValInt(
                  Data.IntOfImmediate(val1) + Data.IntOfImmediate(val2)); 
              IF Data.IsImmediate(val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(intError, Value.valOk);
              END;
          | Code.OpDataIntClass.OpIntDiffCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val :=
                Value.NewValInt(
                  Data.IntOfImmediate(val1) - Data.IntOfImmediate(val2));
              IF Data.IsImmediate(val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(intError, Value.valOk);
              END;
          | Code.OpDataIntClass.OpIntMulCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val :=
                Value.NewValInt(
                  Data.IntOfImmediate(val1) * Data.IntOfImmediate(val2));
              IF Data.IsImmediate(val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(intError, Value.valOk);
              END;
          | Code.OpDataIntClass.OpIntDivCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              i := Data.IntOfImmediate(val2);
              IF i = 0 THEN
                exit := RaiseAndUnwind(intError, Value.valOk);
              ELSE
                val := Value.NewValInt(Data.IntOfImmediate(val1) DIV i);
                IF Data.IsImmediate(val) THEN
                  SetAddr(val);
                ELSE
                  exit := RaiseAndUnwind(intError, Value.valOk);
                END;
              END;
          | Code.OpDataIntClass.OpIntModCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              i := Data.IntOfImmediate(val2);
              IF i = 0 THEN
                exit := RaiseAndUnwind(intError, Value.valOk);
              ELSE
                val := Value.NewValInt(Data.IntOfImmediate(val1) MOD i);
                IF Data.IsImmediate(val) THEN
                  SetAddr(val);
                ELSE
                  exit := RaiseAndUnwind(intError, Value.valOk);
                END;
              END;
          | Code.OpDataIntClass.OpIntLessCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValBool(
                  Data.IntOfImmediate(val1) < Data.IntOfImmediate(val2)));
          | Code.OpDataIntClass.OpIntMoreCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValBool(
                  Data.IntOfImmediate(val1) > Data.IntOfImmediate(val2)));
          | Code.OpDataIntClass.OpIntLessEqCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValBool(
                  Data.IntOfImmediate(val1) <=
                    Data.IntOfImmediate(val2)));
          | Code.OpDataIntClass.OpIntMoreEqCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValBool(
                  Data.IntOfImmediate(val1) >=
                    Data.IntOfImmediate(val2)));
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataRealCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataRealClass) OF
          | Code.OpDataRealClass.OpRealIntCase =>
              (* -- check overflow *)
              val1 := GetAddr();
              SetAddr(Value.NewValFloat(FLOAT(Data.IntOfImmediate(val1))));
              (* | OpRealAbsCase => ---- (* -- check overflow *) *)
          | Code.OpDataRealClass.OpRealFloorCase =>
              (* -- check overflow *)
              val1 := GetAddr();
              SetAddr(Value.NewValInt(TRUNC(Data.FloatOfImmediate(val1))));
          | Code.OpDataRealClass.OpRealRoundCase =>
              (* -- check overflow *)
              Msg("OpRealRound unimplemnted"); (*--*)
              exit := RaiseAndUnwind(fatalError, fatalErrorUnimpl);
          | Code.OpDataRealClass.OpRealPlusCase =>
              (* -- check overflow *)
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValFloat(
                  Data.FloatOfImmediate(val1) +
                    Data.FloatOfImmediate(val2)));
          | Code.OpDataRealClass.OpRealDiffCase =>
              (* -- check overflow *)
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValFloat(
                  Data.FloatOfImmediate(val1) -
                    Data.FloatOfImmediate(val2)));
          | Code.OpDataRealClass.OpRealMulCase =>
              (* -- check overflow *)
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValFloat(
                  Data.FloatOfImmediate(val1) *
                    Data.FloatOfImmediate(val2)));
          | Code.OpDataRealClass.OpRealDivCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              r := Data.FloatOfImmediate(val2);
              IF r = 0.0 THEN
                exit := RaiseAndUnwind(realError, Value.valOk);
              ELSE
                val := Value.NewValFloat(Data.FloatOfImmediate(val1) / r);
                IF Data.IsImmediate(val) THEN
                  SetAddr(val);
                ELSE
                  exit := RaiseAndUnwind(realError, Value.valOk);
                END;
              END;
          | Code.OpDataRealClass.OpRealExpCase =>
              (* -- check overflow *)
              Msg("OpRealExp unimplemnted"); (*--*)
              exit := RaiseAndUnwind(fatalError, fatalErrorUnimpl);
          | Code.OpDataRealClass.OpRealLogCase =>
              (* -- check overflow *)
              Msg("OpRealLog unimplemnted"); (*--*)
              exit := RaiseAndUnwind(fatalError, fatalErrorUnimpl);
          | Code.OpDataRealClass.OpRealLessCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValBool(
                  Data.FloatOfImmediate(val1) <
                    Data.FloatOfImmediate(val2)));
          | Code.OpDataRealClass.OpRealMoreCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValBool(
                  Data.FloatOfImmediate(val1) >
                    Data.FloatOfImmediate(val2)));
          | Code.OpDataRealClass.OpRealLessEqCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValBool(
                  Data.FloatOfImmediate(val1) <=
                    Data.FloatOfImmediate(val2)));
          | Code.OpDataRealClass.OpRealMoreEqCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValBool(
                  Data.FloatOfImmediate(val1) >=
                    Data.FloatOfImmediate(val2)));
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataStringCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataStringClass) OF
          | Code.OpDataStringClass.OpStringErrorCase => SetAddr(stringError);
          | Code.OpDataStringClass.OpStringNewCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              (* Compiler bug? expression broken into pieces: *)
              char := Data.CharOfImmediate(val2);
              IF Value.NewValString(
	        Data.IntOfImmediate(val1),
                   LOOPHOLE(char, Data.Pointee), (*out*) val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(stringError, Value.valOk);
              END;
          | Code.OpDataStringClass.OpStringIsEmptyCase =>
              val1 := GetAddr();
              SetAddr(Value.NewValBool(Value.ValStringIsEmpty(val1)));
          | Code.OpDataStringClass.OpStringLengthCase =>
              val1 := GetAddr();
              SetAddr(Value.NewValInt(Value.ValStringLength(val1)));
          | Code.OpDataStringClass.OpStringEqualCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(Value.NewValBool(Value.ValStringEqual(val1, val2)));
          | Code.OpDataStringClass.OpStringEqualSubCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              val4 := GetAddr();
              val5 := GetAddr();
              val6 := GetAddr();
              IF Value.ValStringEqualSub(val1, 
	        Data.IntOfImmediate(val2),
	        Data.IntOfImmediate(val3),
		val4,
	        Data.IntOfImmediate(val5),
	        Data.IntOfImmediate(val6),
                   (*out*) test) THEN
                SetAddr(Value.NewValBool(test));
              ELSE
                exit := RaiseAndUnwind(stringError, Value.valOk);
              END;
          | Code.OpDataStringClass.OpStringPrecedesCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(
                Value.NewValBool(Value.ValStringPrecedes(val1, val2)));
          | Code.OpDataStringClass.OpStringPrecedesSubCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              val4 := GetAddr();
              val5 := GetAddr();
              val6 := GetAddr();
              IF Value.ValStringPrecedesSub(val1,
	        Data.IntOfImmediate(val2),
	        Data.IntOfImmediate(val3),
                   val4, 
	        Data.IntOfImmediate(val5),
	        Data.IntOfImmediate(val6),
		(*out*) test) THEN
                SetAddr(Value.NewValBool(test));
              ELSE
                exit := RaiseAndUnwind(stringError, Value.valOk);
              END;
          | Code.OpDataStringClass.OpStringGetCharCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              IF Value.ValStringGetPointee(val1,
	        Data.IntOfImmediate(val2),
                   (*out*) pointee) THEN
                SetAddr(Value.NewValChar(LOOPHOLE(pointee, CHAR)));
              ELSE
                exit := RaiseAndUnwind(stringError, Value.valOk);
              END;
          | Code.OpDataStringClass.OpStringSetCharCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              (* Compiler bug? Expression broken into pieces: *)
              char := Data.CharOfImmediate(val3);
              IF Value.ValStringSetPointee(val1,
	        Data.IntOfImmediate(val2),
		LOOPHOLE(char, Data.Pointee)) THEN
                SetAddr(Data.Ok);
              ELSE
                exit := RaiseAndUnwind(stringError, Value.valOk);
              END;
          | Code.OpDataStringClass.OpStringGetSubCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              IF Value.ValStringGetSub(val1, 
	        Data.IntOfImmediate(val2),
	        Data.IntOfImmediate(val3),
		(*out*) val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(stringError, Value.valOk);
              END;
          | Code.OpDataStringClass.OpStringSetSubCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              val4 := GetAddr();
              val5 := GetAddr();
              IF Value.ValStringSetSub(val1, 
	        Data.IntOfImmediate(val2),
                val3, 
	        Data.IntOfImmediate(val4),
	        Data.IntOfImmediate(val5)) THEN
                SetAddr(Data.Ok);
              ELSE
                exit := RaiseAndUnwind(stringError, Value.valOk);
              END;
          | Code.OpDataStringClass.OpStringCatCase, 
		Code.OpDataStringClass.OpStringConcatCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(Value.ValStringCat(val1, val2));
          | Code.OpDataStringClass.OpStringCatSubCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              val4 := GetAddr();
              val5 := GetAddr();
              val6 := GetAddr();
              IF Value.ValStringCatSub(val1, 
	        Data.IntOfImmediate(val2),
	        Data.IntOfImmediate(val3),
		val4,
	        Data.IntOfImmediate(val5),
	        Data.IntOfImmediate(val6),
                   (*out*) val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(stringError, Value.valOk);
              END;
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataArrayCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataArrayClass) OF
          | Code.OpDataArrayClass.OpArrayErrorCase => SetAddr(arrayError);
          | Code.OpDataArrayClass.OpArrayNewCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              IF Value.NewValArray(
	        Data.IntOfImmediate(val2),
		val3,
                   (*out*) val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(arrayError, Value.valOk);
              END;
          | Code.OpDataArrayClass.OpArraySizeCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(Data.ImmediateOfInt(Value.ValArraySize(val2)));
          | Code.OpDataArrayClass.OpArrayGetCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              IF Value.ValArrayGetItem(val2,
	        Data.IntOfImmediate(val3)
			* Data.PointeesPerPolymorph,
                   (*out*) val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(arrayError, Value.valOk);
              END;
          | Code.OpDataArrayClass.OpArraySetCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              val4 := GetAddr();
              IF Value.ValArraySetItem(val2,
	        Data.IntOfImmediate(val3)
			* Data.PointeesPerPolymorph,
                   val4) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(arrayError, Value.valOk);
              END;
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataReaderCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataReaderClass) OF
          | Code.OpDataReaderClass.OpReaderErrorCase => SetAddr(readerError);
          | Code.OpDataReaderClass.OpReaderInputCase => SetAddr(Value.ValReaderInput());
          | Code.OpDataReaderClass.OpReaderFileCase =>
              val1 := GetAddr();
              IF Value.NewValReaderFile(val1, (*out*) val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(readerError, Value.valOk);
              END;
          | Code.OpDataReaderClass.OpReaderMoreCase =>
              val1 := GetAddr();
              IF Value.ValReaderMore(val1, (*out*) test) THEN
                SetAddr(Value.NewValBool(test));
              ELSE
                exit := RaiseAndUnwind(readerError, Value.valOk);
              END;
          | Code.OpDataReaderClass.OpReaderReadyCase =>
              val1 := GetAddr();
              IF Value.ValReaderReady(val1, (*out*) int) THEN
                SetAddr(Value.NewValInt(int));
              ELSE
                exit := RaiseAndUnwind(readerError, Value.valOk);
              END;
          | Code.OpDataReaderClass.OpReaderGetCharCase =>
              val1 := GetAddr();
              IF Value.ValReaderGetPointee(val1, (*out*) pointee) THEN
                SetAddr(Value.NewValChar(LOOPHOLE(pointee, CHAR)));
              ELSE
                exit := RaiseAndUnwind(readerError, Value.valOk);
              END;
          | Code.OpDataReaderClass.OpReaderGetStringCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              IF Value.ValReaderGetString(val1, Data.IntOfImmediate(val2),
                   (*out*) val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(readerError, Value.valOk);
              END;
          | Code.OpDataReaderClass.OpReaderGetSubStringCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              val4 := GetAddr();
              IF Value.ValReaderGetSubString(val1, val2,
                   Data.IntOfImmediate(val3), Data.IntOfImmediate(val4))
              THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(readerError, Value.valOk);
              END;
          | Code.OpDataReaderClass.OpReaderCloseCase =>
              val1 := GetAddr();
              IF Value.ValReaderClose(val1) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(readerError, Value.valOk);
              END;
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataWriterCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataWriterClass) OF
          | Code.OpDataWriterClass.OpWriterErrorCase => SetAddr(writerError);
          | Code.OpDataWriterClass.OpWriterOutputCase => SetAddr(Value.ValWriterOutput());
          | Code.OpDataWriterClass.OpWriterFileCase =>
              val1 := GetAddr();
              IF Value.NewValWriterFile(val1, (*out*) val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(readerError, Value.valOk);
              END;
          | Code.OpDataWriterClass.OpWriterFlushCase =>
              val1 := GetAddr();
              IF Value.ValWriterFlush(val1) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(writerError, Value.valOk);
              END;
          | Code.OpDataWriterClass.OpWriterPutCharCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              char := Data.CharOfImmediate(val2);
              IF Value.ValWriterPutPointee(val1, LOOPHOLE(char, Data.Pointee)) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(writerError, Value.valOk);
              END;
          | Code.OpDataWriterClass.OpWriterPutStringCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              IF Value.ValWriterPutString(val1, val2) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(writerError, Value.valOk);
              END;
          | Code.OpDataWriterClass.OpWriterPutSubStringCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              val4 := GetAddr();
              IF Value.ValWriterPutSubString(val1, val2,
	        Data.IntOfImmediate(val3),
	        Data.IntOfImmediate(val4))
              THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(writerError, Value.valOk);
              END;
          | Code.OpDataWriterClass.OpWriterCloseCase =>
              val1 := GetAddr();
              IF Value.ValWriterClose(val1) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(writerError, Value.valOk);
              END;
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataDynamicCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataDynamicClass) OF
          | Code.OpDataDynamicClass.OpDynamicErrorCase => SetAddr(dynamicError);
          | Code.OpDataDynamicClass.OpDynamicNewCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(Value.NewValDynamic(val1, val2));
          | Code.OpDataDynamicClass.OpDynamicBeCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              IF TRUE (* -- Do dynamic type check *) THEN
                SetAddr(Value.ValDynamicValue(val2))
              ELSE
                exit := RaiseAndUnwind(dynamicError, Value.valOk);
              END;
          | Code.OpDataDynamicClass.OpDynamicCopyCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              IF Data.IsImmediate(val2) THEN
                SetAddr(val2);
              ELSE
                SetAddr(Format.Copy(val2));
              END;
          | Code.OpDataDynamicClass.OpDynamicInternCase =>
              val1 := GetAddr();
              IF Value.ValDynamicIntern(val1, (*out*) val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(dynamicError, Value.valOk);
              END;
          | Code.OpDataDynamicClass.OpDynamicExternCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              IF Value.ValDynamicExtern(val1, val2) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(dynamicError, Value.valOk);
              END;
          | Code.OpDataDynamicClass.OpDynamicInternPortableCase =>
              val1 := GetAddr();
              IF Value.ValDynamicInternPortable(val1, (*out*) val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(dynamicError, Value.valOk);
              END;
          | Code.OpDataDynamicClass.OpDynamicExternPortableCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              IF Value.ValDynamicExternPortable(val1, val2) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(dynamicError, Value.valOk);
              END;
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataTimeCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataTimeClass) OF
          | Code.OpDataTimeClass.OpTimeZeroCase => SetAddr(Value.valTimeZero);
          | Code.OpDataTimeClass.OpTimeNowCase => SetAddr(Value.ValTimeNow());
          | Code.OpDataTimeClass.OpTimeSecondsCase =>
              val1 := GetAddr();
              SetAddr(Value.NewValInt(Value.ValTimeSeconds(val1)));
          | Code.OpDataTimeClass.OpTimeMicrosecondsCase =>
              val1 := GetAddr();
              SetAddr(Value.NewValInt(Value.ValTimeMicroSeconds(val1)));
          | Code.OpDataTimeClass.OpTimeEqualCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(Value.NewValBool(Value.ValTimeEqual(val1, val2)));
          | Code.OpDataTimeClass.OpTimeBeforeCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(Value.NewValBool(Value.ValTimeBefore(val1, val2)));
          | Code.OpDataTimeClass.OpTimeAfterCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(Value.NewValBool(Value.ValTimeAfter(val1, val2)));
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataValueCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataValueClass) OF
          | Code.OpDataValueClass.OpValueNewCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(val2);
          | Code.OpDataValueClass.OpValueBeCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              SetAddr(val2);
          | Code.OpDataValueClass.OpValueFetchCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              int := Data.IntOfImmediate(val2);
              SetAddr(Store.GetPolymorph(val1 + int));
         | Code.OpDataValueClass.OpValueStoreCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              int := Data.IntOfImmediate(val2);
              Store.SetPolymorph(val1 + int, val3);
              SetAddr(Value.valOk);
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataMachineCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataMachineClass) OF
          | Code.OpDataMachineClass.OpMachineExecCase => MachineExec();
          | Code.OpDataMachineClass.OpMachineResumeCase => MachineResume();
          | Code.OpDataMachineClass.OpMachineGotoCase => MachineGoto();
          | Code.OpDataMachineClass.OpMachinePopFrameCase =>
              val1 := GetAddr();
              SetAddr(PopFrame(val1));
          | Code.OpDataMachineClass.OpMachineTraceCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              Value.ValClosureTrace(val1, Data.BoolOfImmediate(val2));
              SetAddr(Value.valOk);
          | Code.OpDataMachineClass.OpMachineTraceExcCase =>
              val1 := GetAddr();
              traceOnRaise := Data.BoolOfImmediate(val1);
              SetAddr(Value.valOk);
          | Code.OpDataMachineClass.OpMachineBreakCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              Value.ValClosureBreak(val1, Data.BoolOfImmediate(val2));
              SetAddr(Value.valOk);
          | Code.OpDataMachineClass.OpMachineBreakExcCase =>
              val1 := GetAddr();
              breakOnRaise := Data.BoolOfImmediate(val1);
              SetAddr(Value.valOk);
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataStoreCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataStoreClass) OF
          | Code.OpDataStoreClass.OpStoreErrorCase =>  SetAddr(storeError);
          | Code.OpDataStoreClass.OpStoreLoopholeCase => SetAddr(GetAddr());
          | Code.OpDataStoreClass.OpStoreAddressCase =>
	      val1 := GetAddr();
	      IF Data.IsPointer(val1) THEN
	        SetAddr(Data.ImmediateOfInt(val1));
              ELSE
                exit := RaiseAndUnwind(storeError, Value.valOk);
              END;
          | Code.OpDataStoreClass.OpStoreAtCase =>
	      val1 := GetAddr();
	      i := Data.IntOfImmediate(val1);
	      IF Value.PlausiblePointer(i) THEN
		SetAddr(i);
              ELSE
                exit := RaiseAndUnwind(storeError, Value.valOk);
              END;
          | Code.OpDataStoreClass.OpStoreClearCase => 
	      val1 := GetAddr();
	      val2 := GetAddr();
	      Store.Clear(Data.IntOfImmediate(val1), 
		Data.IntOfImmediate(val2));
              SetAddr(Value.valOk);
          | Code.OpDataStoreClass.OpStoreCopyCase => 
	      val1 := GetAddr();
	      val2 := GetAddr();
	      val3 := GetAddr();
	      Store.Copy(Data.IntOfImmediate(val1), 
		Data.IntOfImmediate(val2), Data.IntOfImmediate(val3));
              SetAddr(Value.valOk);
          | Code.OpDataStoreClass.OpStoreGetByteCase => 
	      val1 := GetAddr();
	      int := Store.GetByte(Data.IntOfImmediate(val1));
	      IF int>Data.LastByteInt THEN DEC(int,Data.ByteIntSize) END;
	      SetAddr(Data.ImmediateOfInt(int));
          | Code.OpDataStoreClass.OpStoreSetByteCase => 
	      val1 := GetAddr();
	      val2 := GetAddr();
	      int := Data.IntOfImmediate(val2);
	      IF int<0 THEN INC(int,Data.ByteIntSize) END;
	      Store.SetByte(Data.IntOfImmediate(val1), int);
              SetAddr(Value.valOk);
          | Code.OpDataStoreClass.OpStoreGetShortCase => 
	      val1 := GetAddr();
	      int := Store.GetShort(Data.IntOfImmediate(val1));
	      IF int>Data.LastSmallInt THEN DEC(int,Data.SmallIntSize) END;
	      SetAddr(Data.ImmediateOfInt(int));
          | Code.OpDataStoreClass.OpStoreSetShortCase => 
	      val1 := GetAddr();
	      val2 := GetAddr();
	      int := Data.IntOfImmediate(val2);
	      IF int<0 THEN INC(int,Data.SmallIntSize) END;
	      Store.SetShort(Data.IntOfImmediate(val1), int);
              SetAddr(Value.valOk);
          | Code.OpDataStoreClass.OpStoreGetLongCase => 
	      val1 := GetAddr();
	      long := Store.GetLong(Data.IntOfImmediate(val1));
	      IF Data.IsRepresentableInt(long) THEN
	        SetAddr(Data.ImmediateOfInt(LOOPHOLE(long, Data.Int)));
	      ELSE
                exit := RaiseAndUnwind(storeError, Value.valOk);
	      END;
          | Code.OpDataStoreClass.OpStoreSetLongCase => 
	      val1 := GetAddr();
	      val2 := GetAddr();
	      long := LOOPHOLE(Data.IntOfImmediate(val2), Data.Long);
	      Store.SetLong(Data.IntOfImmediate(val1), long);
              SetAddr(Value.valOk);
          | Code.OpDataStoreClass.OpStoreGetRealCase => 
	      val1 := GetAddr();
	      long := Store.GetLong(Data.IntOfImmediate(val1));
	      IF Data.IsRepresentableFloat(long) THEN
	        SetAddr(Data.ImmediateOfFloat(LOOPHOLE(long, Data.Float)));
	      ELSE
                exit := RaiseAndUnwind(storeError, Value.valOk);
	      END;
          | Code.OpDataStoreClass.OpStoreSetRealCase => 
	      val1 := GetAddr();
	      val2 := GetAddr();
	      Store.SetFloat(Data.IntOfImmediate(val1), 
		Data.FloatOfImmediate(val2));
              SetAddr(Value.valOk);
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      | Code.OpClass.OpDataOsCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataOsClass) OF
          | Code.OpDataOsClass.OpOsErrorCase => SetAddr(osError);
          | Code.OpDataOsClass.OpOsChdirCase =>
              val1 := GetAddr();
              IF QOS.ChDir(LOOPHOLE(Store.heapAdr, ADDRESS), 
		   Value.ValStringStart(val1),
                   Value.ValStringLength(val1)) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(osError, Value.valOk);
              END;
          | Code.OpDataOsClass.OpOsWhereCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              IF Value.NewValString(PathBufferSize, Data.MinPointee,
                   (*out*) val3)
                 AND QOS.Lookup(LOOPHOLE(Store.heapAdr, ADDRESS), 
		       Value.ValStringStart(val1),
                       Value.ValStringLength(val1), 
		       LOOPHOLE(Store.heapAdr, ADDRESS),
                       Value.ValStringStart(val3), PathBufferSize,
                       (*out*) length)
                 AND QOS.FindAlongPath(LOOPHOLE(Store.heapAdr, ADDRESS),
                       Value.ValStringStart(val3), length, 
                       QOS.dirSeparator, QOS.pathSeparator,
                       LOOPHOLE(Store.heapAdr, ADDRESS), 
		       Value.ValStringStart(val2),
                       Value.ValStringLength(val2),
                       (*out*) dirNameStart, (*out*) dirNameLength)
                 AND Value.ValStringOfBuff(LOOPHOLE(Store.heapAdr, ADDRESS), 
		       dirNameStart,
                       dirNameLength, (*out*) val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(osError, Value.valOk);
              END;
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      (* RPC:
      | Code.OpClass.OpDataRPCCase =>
         CASE VAL(opClass MOD Data.PointeeSize, Code.OpDataRPCClass) OF
          | Code.OpDataRPCClass.OpRPCErrorCase => SetAddr(rpcError);
          | Code.OpDataRPCClass.OpRPCImportCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              IF Value.ValRPCImport(val1, val2, val3, (*out*) val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCCloseCase =>
              val1 := GetAddr();
              Value.ValRPCClose(val1);
              SetAddr(Value.valOk);
          | Code.OpDataRPCClass.OpRPCInitCallCase =>
              val1 := GetAddr();
              IF Value.ValRPCInitCall(val1) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCSendCallCase =>
              val1 := GetAddr();
              IF Value.ValRPCSendCall(val1) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCEndCallCase =>
              val1 := GetAddr();
              IF Value.ValRPCEndCall(val1) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCExportCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              val3 := GetAddr();
              IF Value.ValRPCExport(val1, val2, val3) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCGetRequestCase =>
              val1 := Value.ValRPCGetRequest();
              SetAddr(val1);
          | Code.OpDataRPCClass.OpRPCStartResponseCase =>
              val1 := GetAddr();
              Value.ValRPCStartResponse(val1);
              SetAddr(Value.valOk);
          | Code.OpDataRPCClass.OpRPCEndResponseCase =>
              val1 := GetAddr();
              Value.ValRPCEndResponse(val1);
              SetAddr(Value.valOk);
          | Code.OpDataRPCClass.OpRPCPutIntCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              int := Data.IntOfImmediate(val2);
              IF Value.ValRPCPut(SYSTEM.ADR(int), SYSTEM.BYTESIZE(int),
                   val1) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCPutBoolCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              test := Data.BoolOfImmediate(val2);
              IF Value.ValRPCPut(SYSTEM.ADR(test), SYSTEM.BYTESIZE(test),
                   val1) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCPutCharCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              char := Data.CharOfImmediate(val2);
              IF Value.ValRPCPut(SYSTEM.ADR(char), SYSTEM.BYTESIZE(char),
                   val1) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCPutStringCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              IF Value.ValRPCPut(
                   SYSTEM.ADR(Store.heap[Value.ValStringStart(val2)]),
                   Value.ValStringLength(val2), val1) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCPutDynamicCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              IF Value.ValRPCPutDynamic(val1, val2) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCPutLocalSpaceIdCase =>
              val1 := GetAddr();
              IF Value.ValRPCPutLocalSpaceId(val1) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCGetIntCase =>
              val1 := GetAddr();
              IF Value.ValRPCGet(SYSTEM.ADR(int), SYSTEM.BYTESIZE(int),
                   val1) THEN
                SetAddr(Data.ImmediateOfInt(int));
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCGetBoolCase =>
              val1 := GetAddr();
              IF Value.ValRPCGet(SYSTEM.ADR(test), SYSTEM.BYTESIZE(test),
                   val1) THEN
                SetAddr(Data.ImmediateOfBool(test));
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCGetCharCase =>
              val1 := GetAddr();
              IF Value.ValRPCGet(SYSTEM.ADR(char), SYSTEM.BYTESIZE(char),
                   val1) THEN
                SetAddr(Data.ImmediateOfChar(char));
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCGetStringCase =>
              val1 := GetAddr();
              val2 := GetAddr();
              IF Value.ValRPCGet(
                   SYSTEM.ADR(Store.heap[Value.ValStringStart(val2)]),
                   Value.ValStringLength(val2), val1) THEN
                SetAddr(Value.valOk);
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          | Code.OpDataRPCClass.OpRPCGetDynamicCase =>
              val1 := GetAddr();
              IF Value.ValRPCGetDynamic(val1, val) THEN
                SetAddr(val);
              ELSE
                exit := RaiseAndUnwind(rpcError, Value.valOk);
              END;
          ELSE
              Msg("Unknown Data Op");
              exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
          END;
      *)
      | Code.OpClass.OpTupleCase =>
          size := GetAddrListSize();
          val := Value.NewValTuple(size);
          i := 0;
          WHILE i < size DO
            Value.ValTupleSet(val, i * Data.PointeesPerPolymorph,
              GetAddr());
            INC(i);
          END;
          SetAddr(val);
      | Code.OpClass.OpCaseCheckCase =>
          val := GetAddr();
          index := Data.IntOfImmediate(GetAddr());
	  IF index # Data.IntOfImmediate(Value.ValTupleGet(val, 0)) THEN
            exit := RaiseAndUnwind(caseCheckError, Value.valOk);
          END;
      | Code.OpClass.OpCaseCase =>
          val1 := GetAddr();
          INC(PC, Data.PointeesPerSmallInt);
          index:=Data.IntOfImmediate(Value.ValTupleGet(val1,0));
          PC := LOOPHOLE(LOOPHOLE(PC, Data.Int) + (index * Data.SmallJumpSize), 
		  Data.Pointer);
          i := Store.GetRelJump(PC);
          PC := LOOPHOLE(LOOPHOLE(PC, Data.Int) + i, Data.Pointer);
      | Code.OpClass.OpCaseFaultCase =>
          exit := RaiseAndUnwind(caseError, Value.valOk);
      | Code.OpClass.OpArrayCase =>
          size := GetAddrListSize();
          IF Value.NewValArray(size, Value.valOk, (*out*) val) THEN END;
          i := 0;
          WHILE i < size DO
            IF Value.ValArraySetItem(val, i * Data.PointeesPerPolymorph,
                 GetAddr()) THEN
            END;
            INC(i);
          END;
          SetAddr(val);
      | Code.OpClass.OpTrapCase =>
          displ := Data.IntOfImmediate(GetAddr());
          i := Store.GetRelJump(PC);
          TP :=
            PushTrapFrame(stack, displ, TL, TP, FP, CP,
              LOOPHOLE(LOOPHOLE(PC, Data.Int) + i, Data.Pointer));
          INC(PC, Data.SmallJumpSize);
      | Code.OpClass.OpUntrapCase => TP := FlushTrapFrame(stack, TP);
      | Code.OpClass.OpRaiseCase =>
          debugPC := PC;
          debugCP := CP;
          debugFP := FP;
          debugTP := TP;
          val1 := GetAddr();
          val2 := GetAddr();
          EX := Value.NewValRaisePacket(val1, val2);
          trace := trace OR traceOnRaise;
          break := break OR breakOnRaise;
      | Code.OpClass.OpUnwindCase =>
          IF TP = stack THEN
            fault := TRUE;
            exit := DebugStop(stack, debugPC, debugCP, debugFP, debugTP);
          ELSE
            TP :=
              PopTrapFrame(stack, TL, TP, (*out*) FP, (*out*) CP, (*out*) PC);
          END;
      | Code.OpClass.OpCrashCase =>
          Msg("OpCrash");
          exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
      ELSE
          Msg("Unknown Op");
          exit := RaiseAndUnwind(fatalError, fatalErrorCrash);
      END;
      IF exit THEN EXIT END;
      IF trace THEN
        trace := FALSE;
        IF RaiseAndUnwind(traceError, Value.valOk) THEN EXIT END;
      END;
      IF break THEN
        break := FALSE;
        IF RaiseAndUnwind(breakError, Value.valOk) THEN EXIT END;
      END;
    END;
  END Execute;

PROCEDURE RaiseAndUnwind(val1, val2: Value.T): BOOLEAN =
  (* Do what OpRaise and OpUnwind do. *)
  BEGIN
    debugPC := PC;
    debugCP := CP;
    debugFP := FP;
    debugTP := TP;
    EX := Value.NewValRaisePacket(val1, val2);
    IF TP = stack THEN
      fault := TRUE;
      RETURN DebugStop(stack, debugPC, debugCP, debugFP, debugTP);
    ELSE
      TP := PopTrapFrame(stack, TL, TP, (*out*) FP, (*out*) CP, (*out*) PC);
      RETURN FALSE;
    END;
  END RaiseAndUnwind;

PROCEDURE DebugStop(
    stack: Value.T;
    debugPC: Data.Pointer;
    debugCP: Value.T;
    debugFP, debugTP: Data.Pointer)
    : BOOLEAN =
  VAR machineState: Value.T;
  BEGIN
    PC := debugPC;
    CP := debugCP;
    FP := debugFP;
    TP := debugTP;
    IF EX = Data.MinPointer THEN Msg("Null EX on DebugStop") END;
    IF LP = 0 THEN
      RETURN TRUE;
    ELSE
      machineState := Stop();
      (* This result address was left around by OpStart or OpMachineExec! *)
      SetAddr(machineState);
      RETURN FALSE;
    END;
  END DebugStop;

PROCEDURE BootConvertProg(
    prog: Data.Pointer (* ByteCode_Prog *))
    : Value.T =
  VAR
    code, literals, rtProg, rtLiterals, scan: Data.Pointer;
    codeSize, i: Data.Int;
    pointee: Data.Pointee;
  BEGIN
    code := Value.ValTupleGet(prog, 0 * Data.PointeesPerPolymorph);
    literals := Value.ValTupleGet(prog, 1 * Data.PointeesPerPolymorph);
    codeSize := Value.ValStringLength(code);
    rtProg := Value.NewValProg(codeSize);
    rtLiterals := BootConvertLiterals(literals);
    Value.ValProgSetLiterals(rtProg, rtLiterals);
    scan := Value.ValProgCodeStart(rtProg);
    i := 0;
    WHILE i < codeSize DO
      IF NOT (Value.ValStringGetPointee(code, i, (*out*) pointee)) THEN
        Data.Fault("")
      END;
      Store.SetPointee(scan, pointee);
      INC(scan);
      INC(i);
    END;
    RETURN rtProg;
  END BootConvertProg;

PROCEDURE BootConvertLiterals(
    literals: Data.Pointer (* ByteCode_LitList *))
    : Value.T =
  VAR scan, case, item, rtLiterals: Data.Polymorph; i, index: Data.Int;
  BEGIN
    scan := literals;
    i := 0;
    LOOP
      index := Data.IntOfImmediate(Value.ValTupleGet(scan, 0));
      case := scan;
      IF index = selNil THEN EXIT END;
      IF index # selList THEN Data.Fault("BootConvertLiterals"); END;
      scan := Value.ValTupleGet(case, 2 * Data.PointeesPerPolymorph);
      INC(i);
    END;
    rtLiterals := Value.NewValLiterals(i);
    scan := literals;
    i := 0;
    LOOP
      index := Data.IntOfImmediate(Value.ValTupleGet(scan, 0));
      case := scan;
      IF index = selNil THEN EXIT END;
      item := Value.ValTupleGet(case, 1 * Data.PointeesPerPolymorph);
      Value.ValLiteralsSet(rtLiterals, i * Data.PointeesPerPolymorph,
        BootConvertLiteral(item));
      scan := Value.ValTupleGet(case, 2 * Data.PointeesPerPolymorph);
      INC(i);
    END;
    RETURN rtLiterals;
  END BootConvertLiterals;

PROCEDURE BootConvertString(string: Value.T): Value.T =
  BEGIN RETURN string END BootConvertString;

PROCEDURE BootConvertBadge(badge: Value.T): Value.T =
  BEGIN RETURN badge END BootConvertBadge;

PROCEDURE BootConvertFormat(format: Value.T): Value.T =
  BEGIN
    (* --format Convert the string "format" to a real format *)
    RETURN format
  END BootConvertFormat;

PROCEDURE BootConvertType(type: Value.T): Value.T =
  BEGIN RETURN Data.MinPointer; (* -- *) END BootConvertType;

PROCEDURE BootConvertLiteral(
    literal: Data.Pointer (* ByteCode_Lit *))
    : Value.T =
  VAR index: Data.Int; case: Value.T;
  BEGIN
    index := Data.IntOfImmediate(Value.ValTupleGet(literal, 0));
    case := Value.ValTupleGet(literal, 1 * Data.PointeesPerPolymorph);
    IF index = selLitStringCase THEN
      RETURN BootConvertString(case);
    ELSIF index = selLitBadgeCase THEN
      RETURN BootConvertBadge(case);
    ELSIF index = selLitProgCase THEN
      RETURN BootConvertProg(case);
    ELSIF index = selLitFormatCase THEN
      RETURN BootConvertFormat(case);
    ELSIF index = selLitTypeCase THEN
      RETURN BootConvertType(case);
    ELSIF index = selLitValueProgCase THEN
      RETURN BootConvertProg(case);
    ELSIF index = selLitValueCase THEN
      RETURN case;
    ELSE
      Data.Fault("BootConvertLiteral");
    END;
  END BootConvertLiteral;

PROCEDURE Start(newMachineState: Value.T) =
  VAR machineState: Value.T;
  BEGIN
    RelativeState(stack, (* in-out *) TL, (* in-out *) TP, (* in-out *) FP,
      CP, (* in-out *) PC);
    machineState :=
      Value.NewValMachineState(stack, TL, TP, FP, CP, PC, EX, home, fault);
    INC(LP);
    IF NOT (Value.ValArraySetItem(levelStack,
              LP * Data.PointeesPerPolymorph, machineState)) THEN
      Data.Fault("");
    END;
    Value.ValMachineStateGet(newMachineState, (* out *) stack,
      (* out *) TL, (* out *) TP, (* out *) FP, (* out *) CP, (* out *) PC,
      (* out *) EX, (* out *) home, (*out*) fault);
    AbsoluteState(stack, (* in-out *) TL,
      (* in-out *) TP, (* in-out *) FP, CP, (* in-out *) PC);
  END Start;

PROCEDURE StartGoto(newMachineState: Value.T; gotoProg: Value.T) =
  VAR machineState: Value.T;
  BEGIN
    RelativeState(stack, (* in-out *) TL, (* in-out *) TP, (* in-out *) FP,
      CP, (* in-out *) PC);
    machineState :=
      Value.NewValMachineState(stack, TL, TP, FP, CP, PC, EX, home, fault);
    INC(LP);
    IF NOT (Value.ValArraySetItem(levelStack,
              LP * Data.PointeesPerPolymorph, machineState)) THEN
      Data.Fault("")
    END;
    Value.ValMachineStateGet(newMachineState, (* out *) stack,
      (* out *) TL, (* out *) TP, (* out *) FP, (* out *) CP, (* out *) PC,
      (* out *) EX, (* out *) home, (*out*) fault);
    AbsoluteState(stack, (* in-out *) TL,
      (* in-out *) TP, (* in-out *) FP, CP, (* in-out *) PC);
    PC := 0;
    INC(PC, Value.ValProgCodeStart(gotoProg));
  END StartGoto;

PROCEDURE Stop(): Value.T =
  VAR machineState, oldMachineState: Value.T;
  BEGIN
    IF LP <= 0 THEN Data.Fault("levelStack underflow"); END;
    RelativeState(stack, (* in-out *) TL, (* in-out *) TP,
      (* in-out *) FP, CP, (* in-out *) PC);
    machineState :=
      Value.NewValMachineState(stack, TL, TP, FP, CP, PC, EX, home, fault);
    IF NOT (Value.ValArrayGetItem(levelStack,
              LP * Data.PointeesPerPolymorph, (*out*) oldMachineState)) THEN
      Data.Fault("")
    END;
    IF NOT (Value.ValArraySetItem(levelStack,
              LP * Data.PointeesPerPolymorph, Data.MinPointer)) THEN
      Data.Fault("")
    END;
    DEC(LP);
    Value.ValMachineStateGet(oldMachineState, (* out *) stack, (* out *) TL,
      (* out *) TP, (* out *) FP, (* out *) CP,
      (* out *) PC, (* out *) EX, (*out*) home, (*out*) fault);
    AbsoluteState(stack, (* in-out *) TL,
      (* in-out *) TP, (* in-out *) FP, CP, (* in-out *) PC);
    RETURN machineState;
  END Stop;

(* Debug aid *)
<*NOWARN*> PROCEDURE ClosureInfo(CP: Value.T) =
  VAR rTBadge: Data.Pointer; moduleName, progName: Value.T;
  BEGIN
    rTBadge :=
      Value.ValLiteralsGet(
        Value.ValProgGetLiterals(Value.ValClosureProg(CP)), 0);
    Value.ValBadgeName(rTBadge, (* out *) moduleName, (* out *) progName);
    QOS.OutString(QOS.stdout, "Frame: ");
    Value.PrintValString(QOS.stdout, moduleName);
    QOS.OutChar(QOS.stdout, '.');
    Value.PrintValString(QOS.stdout, progName);
    QOS.OutChar(QOS.stdout, QOS.newLine);
  END ClosureInfo;

<*NOWARN*> PROCEDURE PopIt() =
  BEGIN
    DEC(FP,
      ValProgFrameSize(Value.ValClosureProg(CP)) -
        Data.PointeesPerPolymorph);
    (* FP is now 1 above normal *)
    CP :=
      Store.GetPolymorph(
        LOOPHOLE(FP, Data.Int) - Data.PointeesPerPolymorph + FrameSaveCPDispl);
    PC :=
      Store.GetPolymorph(
        LOOPHOLE(FP, Data.Int) - Data.PointeesPerPolymorph + FrameSavePCDispl);
    DEC(FP, Data.PointeesPerPolymorph);
  END PopIt;

BEGIN
END Exec.

