(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)

INTERFACE Format;
IMPORT QOS, (*-- QRPC, *) Data;

(*
|	           <---size-->
|	+----+----+-- - - - --+
|	| o  |size|           |                  format
|	+----+----+-- - - - --+
|	          ^
|	          |        <---size-->
|	        +-|--+----+-- - - - --+
|	        | o  |size|           |          data
|	        +----+----+-- - - - --+
|	                  ^
|	                  |
*)

(*
| Format =
|   Alt(	
|     codeSize: Data.PointeesPerSmallInt,		-- "code" field
|     empty: Empty,					-- code = "Empty"
|     pointee: Empty,					-- code = "Pointee"
|     smallInt: Empty,					-- code = "SmallInt"
|     int: Empty,					-- code = "Int"
|     immediate: Empty,					-- code = "Immediate"
|     pointer: Empty,					-- code = "Pointer"
|     polymorph: Empty,					-- code = "Polymorph"
|     seq: 						-- code = "Seq"
|       Iter(
|         seqsNoSize: Data.PointeesPerSmallInt,
|         seqsFormat: Format),
|     alt: 						-- code = "Alt"
|       Seq(
|         altTagPointees: Pointee[Data.PointeesPerSmallInt],
|         (* -- jump table *)
|         altItems:
|           Iter(
|             altItemsNoNoSize: Data.PointeesPerSmallInt,
|             altItemsFormat: Format)),
|     iter: 						-- code = "Iter"
|       Seq(
|         iterCountPointees: Pointee[Data.PointeesPerSmallInt],
|         itemFormat: Format),
|     area:						-- code = "Area"
|       Seq(
|         areaSizePointees: Pointee[Data.PointeesPerSmallInt],
|         areaFillPointees: Pointee[Data.PointeesPerSmallInt],
|         areaFormat: Format),
|     repeat:						-- code = "[]"
|       Seq(
|         repeatCount: Pointee[Data.PointeesPerInt],
|         repeatFormat: Format)
|     run: Empty,					-- code = "Run"
|      ).
*)

TYPE
  Class =
    {NoneCase, EmptyCase, PointeeCase, SmallIntCase, IntCase,
     ImmediateCase, PointerCase, PolymorphCase, SeqCase, AltCase, IterCase,
     AreaCase, RepeatCase, RunCase};

  T = UNTRACED ROOT OBJECT END;
  TNone = T BRANDED OBJECT END;
  TEmpty = T BRANDED OBJECT END;
  TPointee = T BRANDED OBJECT END;
  TSmallInt = T BRANDED OBJECT END;
  TInt = T BRANDED OBJECT END;
  TImmediate = T BRANDED OBJECT END;
  TPointer = T BRANDED OBJECT END;
  TPolymorph = T BRANDED OBJECT END;
  TSeq = T BRANDED OBJECT seqItems: TList END;
  TAlt = T BRANDED OBJECT altTagPointees: Data.Int; altItems: TList END;
  TIter = T BRANDED OBJECT iterCountPointees: Data.Int; itemFormat: T END;
  TArea = T BRANDED OBJECT 
      areaSizePointees: Data.Int;
      areaFillPointees: Data.Int;
      areaFormat: T; 
    END;
  TRepeat = T BRANDED OBJECT 
      repeatCount: Data.Int; repeatFormat: T;
    END;
  TRun = T BRANDED OBJECT END;

  TList = UNTRACED ROOT OBJECT first: T; rest: TList END;

VAR
  none, empty, pointee, smallInt, int, pointer, immediate, polymorph,
  type: T;

PROCEDURE Setup();

PROCEDURE NewNone(): T;
PROCEDURE NewEmpty(): T;
PROCEDURE NewPointee(): T;
PROCEDURE NewSmallInt(): T;
PROCEDURE NewInt(): T;
PROCEDURE NewImmediate(): T;
PROCEDURE NewPointer(): T;
PROCEDURE NewPolymorph(): T;
PROCEDURE NewSeq(seqItems: TList): T;
PROCEDURE NewAlt(altTagPointees: Data.Int; altItems: TList): T;
PROCEDURE NewIter(iterCountPointees: Data.Int; itemFormat: T): T;
PROCEDURE NewArea(
    areaSizePointees, areaFillPointees: Data.Int;
    areaFormat: T)
    : T;
PROCEDURE NewRepeat(repeatCount: Data.Int; repeatFormat: T): T;
PROCEDURE NewRun(): T;

VAR emptyList: TList;
PROCEDURE NewList(first: T; rest: TList): TList;
PROCEDURE ListLength(list: TList): Data.Int;

VAR
  formatFormat, errorFormat: Data.Pointer;

PROCEDURE New(format: T): Data.Pointer;
PROCEDURE DataGetSize(hp: Data.Pointer): Data.Int;
PROCEDURE DataSetSize(hp: Data.Pointer; size: Data.Int);
PROCEDURE DataGetFormat(hp: Data.Pointer): Data.Pointer;
PROCEDURE DataSetFormat(hp: Data.Pointer; fmt: Data.Pointer);
PROCEDURE FormatStart(fmt: Data.Pointer): Data.Pointer;

PROCEDURE Copy(hp: Data.Pointer): Data.Pointer;
PROCEDURE Extern(wr: QOS.Writer; hp: Data.Pointer): BOOLEAN;
PROCEDURE Intern(rd: QOS.Reader; VAR (*out*) hp: Data.Pointer): BOOLEAN;
PROCEDURE ExternPortable(wr: QOS.Writer; hp: Data.Pointer): BOOLEAN;
PROCEDURE InternPortable(rd: QOS.Reader; VAR (*out*) hp: Data.Pointer): BOOLEAN;

(* RPC:
PROCEDURE ExternMarshal(VAR rpc: QRPC.T; hp: Data.Pointer): BOOLEAN;
PROCEDURE InternMarshal(
    VAR rpc: QRPC.T;
    VAR (*out*) hp: Data.Pointer)
    : BOOLEAN;
*)

(* PROCEDURE Print(wr: QOS.Writer; format: T); *)
(* PROCEDURE PrintRun(hp: Data.Pointer; printDepth: INTEGER); *)

END Format.
