(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)

UNSAFE MODULE Format;
IMPORT Word, QOS, (* RPC: QRPC,*) Data, Store, Code;

(*
|			  31                       0   bit
|			 +--+-- -- -- -- -- -- --+--+
| Polymorph:             |  |                    |  |
|			 +--+-- -- -- -- -- -- --+--+
|			                           ^
|			    Immediate/Pointer mark |
|
|
|			  31                    1  0   bit
|			 +--+-- -- -- -- -- --+--+--+
| Immediate:             |  |                 |  | 1|
|			 +--+-- -- -- -- -- --+--+--+
|			  \________Data_________/
|
|
|			  31 30 29              1  0   bit
|			 +--+--+--+-- -- -- --+--+--+
| Pointer:               |  |  |  |           |  | 0|
|			 +--+--+--+-- -- -- --+--+--+
|			  ^  ^  \_____Data______/
|			  |  |	
|			  |  "shared" mark
|			  "visited" mark
*)

TYPE
  SharedStack =
    ARRAY [0 .. Data.SharedStackSize - 1] OF RECORD
      shared, copy: Data.Pointer
    END;
VAR
  topSharedStack: INTEGER;
  sharedStack: SharedStack;

TYPE WalkStackItem = RECORD hp, at: Data.Pointer END;
TYPE WalkStack = ARRAY [0 .. Data.WalkStackSize - 1] OF WalkStackItem;
VAR
  walkStack: WalkStack;
  walkStackTop: INTEGER;

PROCEDURE Setup() =
  BEGIN
    emptyList := NIL;
    none := NewNone();
    empty := NewEmpty();
    pointee := NewPointee();
    pointer := NewPointer();
    immediate := NewImmediate();
    polymorph := NewPolymorph();
    smallInt := NewSmallInt();
    int := NewInt();
    type := pointer; (* --format *)

    formatFormat := Data.MinPointer;
    errorFormat := Data.MaxPointer;

    topSharedStack := -1;
    FOR i := 0 TO Data.SharedStackSize - 1 DO
      sharedStack[i].shared := Data.MinPointer;
      sharedStack[i].copy := Data.MinPointer;
    END;

    (* -- init walk stack *)

  END Setup;

(* === New === *)

PROCEDURE NewNone(): T =
  BEGIN
    RETURN NEW(TNone);
  END NewNone;

PROCEDURE NewEmpty(): T =
  BEGIN
    RETURN NEW(TEmpty);
  END NewEmpty;

PROCEDURE NewPointee(): T =
  BEGIN
    RETURN NEW(TPointee);
  END NewPointee;

PROCEDURE NewSmallInt(): T =
  BEGIN
    RETURN NEW(TSmallInt);
  END NewSmallInt;

PROCEDURE NewInt(): T =
  BEGIN
    RETURN NEW(TInt);
  END NewInt;

PROCEDURE NewImmediate(): T =
  BEGIN
    RETURN NEW(TImmediate);
  END NewImmediate;

PROCEDURE NewPointer(): T =
  BEGIN
    RETURN NEW(TPointer);
  END NewPointer;

PROCEDURE NewPolymorph(): T =
  BEGIN
    RETURN NEW(TPolymorph);
  END NewPolymorph;

PROCEDURE NewSeq(seqItems: TList): T =
  BEGIN
    RETURN NEW(TSeq, seqItems:=seqItems);
  END NewSeq;

PROCEDURE NewAlt(altTagPointees: Data.Int; altItems: TList): T =
  BEGIN
    RETURN NEW(TAlt,altTagPointees:=altTagPointees,altItems:=altItems);
  END NewAlt;

PROCEDURE NewIter(iterCountPointees: Data.Int; itemFormat: T): T =
  BEGIN
    RETURN NEW(TIter, iterCountPointees:=iterCountPointees, itemFormat:=itemFormat);
  END NewIter;

PROCEDURE NewArea(
    areaSizePointees, areaFillPointees: Data.Int;
    areaFormat: T)
    : T =
  BEGIN
    RETURN NEW(TArea, areaSizePointees:=areaSizePointees, areaFillPointees:=areaFillPointees, areaFormat:=areaFormat);
  END NewArea;

PROCEDURE NewRepeat(repeatCount: Data.Int; repeatFormat: T): T =
  BEGIN
    RETURN NEW(TRepeat, repeatCount:=repeatCount, repeatFormat:=repeatFormat);
  END NewRepeat;

PROCEDURE NewRun(): T =
  BEGIN
    RETURN NEW(TRun);
  END NewRun;

PROCEDURE NewList(first: T; rest: TList): TList =
  BEGIN
    RETURN NEW(TList, first:=first, rest:=rest);
  END NewList;

PROCEDURE ListLength(list: TList): Data.Int =
  VAR i: Data.Int;
  BEGIN
    i := 0;
    WHILE list # emptyList DO INC(i); list := list.rest; END;
    RETURN i;
  END ListLength;

(* === Print === *)

(*

   PROCEDURE Print(wr: Wr.T; format: T); BEGIN PrintFmt(wr, format);
   Wr.PutChar(wr, '\n'); END Print;

   PROCEDURE PrintFmt(wr: Wr.T; fmt: T); BEGIN
   (* -- Wr.PutStr(wr, "<format>"); *) CASE fmt^.class OF | NoneCase:
   Wr.PutStr(wr, "None"); | EmptyCase: Wr.PutStr(wr, "Empty"); | PointeeCase:
   Wr.PutStr(wr, "Pointee"); | SmallIntCase: Wr.PutStr(wr, "SmallInt"); |
   IntCase: Wr.PutStr(wr, "Int"); | ImmediateCase: Wr.PutStr(wr, "Immediate");
   | PointerCase: Wr.PutStr(wr, "Pointer"); | PolymorphCase: Wr.PutStr(wr,
   "Polymorph"); | SeqCase: Wr.PutStr(wr, "Seq("); PrintFmtList(wr,
   fmt^.seqItems); Wr.PutChar(wr, ')'); | AltCase: Wr.PutStr(wr, "Alt(");
   QOS.OutInt(wr, fmt^.altTagPointees); Wr.PutStr(wr, ", "); PrintFmtList(wr,
   fmt^.altItems); Wr.PutChar(wr, ')'); | IterCase: Wr.PutStr(wr, "Iter(");
   QOS.OutInt(wr, fmt^.iterCountPointees); Wr.PutStr(wr, ", "); PrintFmt(wr,
   fmt^.itemFormat); Wr.PutChar(wr, ')'); | AreaCase: Wr.PutStr(wr, "Area(");
   QOS.OutInt(wr, fmt^.areaSizePointees); Wr.PutStr(wr, ", "); QOS.OutInt(wr,
   fmt^.areaFillPointees); Wr.PutStr(wr, ", "); PrintFmt(wr, fmt^.areaFormat);
   Wr.PutChar(wr, ')'); | RepeatCase: Wr.PutStr(wr, "Repeat("); QOS.OutInt(wr,
   fmt^.repeatCount); Wr.PutStr(wr, ", "); PrintFmt(wr, fmt^.repeatFormat);
   Wr.PutChar(wr, ')'); | RunCase: Wr.PutStr(wr, "Run"); END; END PrintFmt;

   PROCEDURE PrintFmtList(wr: Wr.T; list: TList); BEGIN WHILE list # emptyList
   DO PrintFmt(wr, list^.first); IF list^.rest # emptyList THEN Wr.PutStr(wr,
   ", ") END; list := list^.rest; END; END PrintFmtList;

*)

(* === Pre-fields === *)

<*INLINE*> PROCEDURE DataGetSize(hp: Data.Pointer): Data.Int =
  BEGIN RETURN Store.GetInt(hp - Data.PointeesPerInt); END DataGetSize;

<*INLINE*> PROCEDURE DataSetSize(hp: Data.Pointer; size: Data.Int) =
  BEGIN Store.SetInt(hp - Data.PointeesPerInt, size); END DataSetSize;

<*INLINE*> PROCEDURE DataGetFormat(hp: Data.Pointer): Data.Pointer =
  BEGIN
    RETURN
      Store.GetPointer(hp - Data.PointeesPerInt - Data.PointeesPerPointer);
  END DataGetFormat;

<*INLINE*> PROCEDURE DataSetFormat(hp: Data.Pointer; fmt: Data.Pointer) =
  BEGIN
    Store.SetPointer(hp - Data.PointeesPerInt - Data.PointeesPerPointer, fmt);
  END DataSetFormat;

<*INLINE*> PROCEDURE FormatStart(fmt: Data.Pointer): Data.Pointer =
  BEGIN RETURN fmt; END FormatStart;

(* === New === *)

PROCEDURE LayFmt(format: T) =
  VAR scan: TList;
  BEGIN
    TYPECASE format OF
    | TEmpty => Store.LaySmallInt(ORD(Class.EmptyCase));
    | TPointee => Store.LaySmallInt(ORD(Class.PointeeCase));
    | TSmallInt => Store.LaySmallInt(ORD(Class.SmallIntCase));
    | TInt => Store.LaySmallInt(ORD(Class.IntCase));
    | TImmediate => Store.LaySmallInt(ORD(Class.ImmediateCase));
    | TPointer => Store.LaySmallInt(ORD(Class.PointerCase));
    | TPolymorph => Store.LaySmallInt(ORD(Class.PolymorphCase));
    | TSeq(formatSeq) =>
        Store.LaySmallInt(ORD(Class.SeqCase));
        Store.LaySmallInt(ListLength(formatSeq.seqItems));
        scan := formatSeq.seqItems;
        WHILE scan # emptyList DO
          LayFmt(scan.first);
          scan := scan.rest;
        END;
    | TAlt(formatAlt) =>
        Store.LaySmallInt(ORD(Class.AltCase));
        Store.LaySmallInt(formatAlt.altTagPointees);
        (* -- lay jump table for fast access *)
        Store.LaySmallInt(ListLength(formatAlt.altItems));
        scan := formatAlt.altItems;
        WHILE scan # emptyList DO
          LayFmt(scan.first);
          scan := scan.rest;
        END;
    | TIter(formatIter) =>
        Store.LaySmallInt(ORD(Class.IterCase));
        Store.LaySmallInt(formatIter.iterCountPointees);
        LayFmt(formatIter.itemFormat);
    | TArea(formatArea) =>
        Store.LaySmallInt(ORD(Class.AreaCase));
        Store.LaySmallInt(formatArea.areaSizePointees);
        Store.LaySmallInt(formatArea.areaFillPointees);
        LayFmt(formatArea.areaFormat);
    | TRepeat(formatRepeat) =>
        Store.LaySmallInt(ORD(Class.RepeatCase));
        Store.LayInt(formatRepeat.repeatCount);
        LayFmt(formatRepeat.repeatFormat);
    | TRun => Store.LaySmallInt(ORD(Class.RunCase));
    END;
  END LayFmt;

PROCEDURE New(format: T): Data.Pointer =
  VAR pointer, sizePointer: Data.Pointer;
  BEGIN
    Store.Align();
    Store.LayPointer(formatFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    LayFmt(format);
    Store.SetInt(sizePointer, Store.hp - pointer);
    RETURN pointer;
  END New;

(* === WALK === *)

(* "WalkMark(hp, TRUE)" walks a data structure pointed by hp.
   Marks "visited" the headers of every data and format run it finds
   (uses a visitedStack to remember intermediate structures).
   If it finds a run already marked "visited", it also marks it
   "shared" and stores a (unique) pointer to it in the shared stack.
   At the end, the entire data structure is left marked, the visiteStack
   is empty, and the sharedStack contains the structures marked shared.

   "WalkMark(hp, FALSE)" walks a data structure pointed by hp removing
   all the marks.
*)

(* === Visited and Shared === *)

CONST
  w80000000 = Word.Shift(1, 31); (* ARCH *)
  w40000000 = Word.Shift(1, 30); (* ARCH *)
  w3FFFFFFF = Word.Not(Word.Or(w80000000,w40000000)); (* ARCH *)


<*INLINE*> PROCEDURE UnMark(hp: Data.Pointer): Data.Pointer =
  BEGIN
    (* ARCH: hp & 16_3FFFFFFF *)
    (* RETURN hp MOD 16_40000000; *)
    RETURN Word.And(hp,w3FFFFFFF);
  END UnMark;

<*INLINE*> PROCEDURE TestVisited(hp: Data.Pointer): BOOLEAN =
  BEGIN
    (* ARCH: (hp & 16_80000000) # 0 *)
    (* RETURN hp >= 16_80000000; *)
    RETURN Word.And(hp, w80000000) # 0;
  END TestVisited;

<*INLINE*> PROCEDURE TestShared(hp: Data.Pointer): BOOLEAN =
  BEGIN
    (* ARCH: (hp & 16_40000000) # 0 *)
    (* RETURN (hp MOD 16_80000000) >= 16_40000000; *)
    RETURN Word.And(hp,w40000000) # 0;
  END TestShared;

<*INLINE*> PROCEDURE MarkVisited(hp: Data.Pointer): Data.Pointer =
  BEGIN
    (* ARCH: hp | 16_80000000 *)
    (* IF hp >= 16_80000000 THEN RETURN hp ELSE RETURN hp + 16_80000000 END;*)
    RETURN Word.Or(hp,w80000000);
  END MarkVisited;

<*INLINE*> PROCEDURE MarkShared(hp: Data.Pointer): Data.Pointer =
  BEGIN
    (* ARCH: hp | 16_40000000 *)
    (* IF (hp MOD 16_80000000) >= 16_40000000 THEN
      RETURN hp
    ELSE
      RETURN hp + 16_40000000
    END; *)
    RETURN Word.Or(hp,w40000000);
  END MarkShared;

PROCEDURE RememberShared(hp: Data.Pointer) =
  VAR scan: INTEGER;
  BEGIN
    IF hp = Data.MinPointer THEN Data.Fault("") END;
    scan := topSharedStack;
    WHILE scan >= 0 DO
      IF sharedStack[scan].shared = hp THEN RETURN END;
      DEC(scan);
    END;
    IF topSharedStack >= Data.SharedStackSize - 1 THEN
      Data.Fault("SharedStack overflow");
    END;
    INC(topSharedStack);
    sharedStack[topSharedStack].shared := hp;
    sharedStack[topSharedStack].copy := Data.MinPointer;
  END RememberShared;

PROCEDURE RememberSharedCopy(hp, copy: Data.Pointer) =
  VAR scan: INTEGER;
  BEGIN
    IF hp = Data.MinPointer THEN Data.Fault("") END;
    IF copy = Data.MinPointer THEN Data.Fault("") END;
    scan := topSharedStack;
    WHILE scan >= 0 DO
      IF sharedStack[scan].shared = hp THEN
        sharedStack[scan].copy := copy;
        RETURN;
      END;
      DEC(scan);
    END;
    Data.Fault("");
  END RememberSharedCopy;

PROCEDURE CopiedShared(VAR (*in-out*) hp: Data.Pointer): BOOLEAN =
  VAR scan: INTEGER;
  BEGIN
    IF hp = Data.MinPointer THEN Data.Fault("") END;
    scan := topSharedStack;
    WHILE scan >= 0 DO
      IF sharedStack[scan].shared = hp THEN
        IF sharedStack[scan].copy = Data.MinPointer THEN
          RETURN FALSE;
        ELSE
          hp := sharedStack[scan].copy;
          RETURN TRUE;
        END;
      END;
      DEC(scan);
    END;
    Data.Fault("");
  END CopiedShared;

PROCEDURE WalkHeaderUnMark(VAR (*in-out*) hp: Data.Pointer): BOOLEAN =
  VAR header: Data.Pointer;
  BEGIN
    header := DataGetFormat(hp);
    IF NOT TestVisited(header) THEN RETURN FALSE END;
    DataSetFormat(hp, UnMark(header));
    RETURN TRUE;
  END WalkHeaderUnMark;

(* 
PROCEDURE WalkHeaderMarkVisited(VAR (*in-out*) hp: Data.Pointer): BOOLEAN =
  VAR header: Data.Pointer; pointerAddr: Data.PointerPtr;
  BEGIN
    header := DataGetFormat(hp);
    IF TestVisited(header) THEN RETURN FALSE END;
    DataSetFormat(hp, MarkVisited(header));
    RETURN TRUE;
  END WalkHeaderMarkVisited;
*)

PROCEDURE WalkHeaderMarkShared(VAR (*in-out*) hp: Data.Pointer): BOOLEAN =
  VAR header: Data.Pointer;
  BEGIN
    header := DataGetFormat(hp);
    IF TestShared(header) THEN RETURN FALSE END;
    IF TestVisited(header) THEN
      DataSetFormat(hp, MarkShared(header));
      RememberShared(hp);
      RETURN FALSE;
    END;
    DataSetFormat(hp, MarkVisited(header));
    RETURN TRUE;
  END WalkHeaderMarkShared;

(* === Relative Pointers === *)

VAR
  externStart: Data.Pointer;
  internStart: Data.Pointer;

PROCEDURE RelativePointer(pointer: Data.Pointer): Data.Pointer =
  BEGIN
    IF (pointer = Data.MinPointer) OR (pointer = formatFormat) THEN
      RETURN pointer;
    ELSE
      RETURN pointer - externStart;
    END;
  END RelativePointer;

PROCEDURE AbsolutePointer(pointer: Data.Pointer): Data.Pointer =
  BEGIN
    IF (pointer = Data.MinPointer) OR (pointer = formatFormat) THEN
      RETURN pointer;
    ELSE
      RETURN pointer + internStart;
    END;
  END AbsolutePointer;

(* === Mark === *)

PROCEDURE WalkMark(hp: Data.Pointer; markOnOff: BOOLEAN) =
  BEGIN
    walkStackTop := 0;
    walkStack[walkStackTop].hp := hp;
    WHILE walkStackTop >= 0 DO
      hp := walkStack[walkStackTop].hp;
      DEC(walkStackTop);
      WalkMarkHp(hp, markOnOff);
    END;
  END WalkMark;

PROCEDURE WalkMarkHp(hp: Data.Pointer; markOnOff: BOOLEAN) =
  VAR fmt, fmtScan, hpScan: Data.Pointer; continue: BOOLEAN;
  BEGIN
    <*ASSERT NOT( (hp < Store.MinIndex) OR (hp > Store.MaxIndex) ) *>
    IF hp = Data.MinPointer THEN RETURN END;
    IF markOnOff THEN
      continue := WalkHeaderMarkShared( (*in-out*) hp);
    ELSE
      continue := WalkHeaderUnMark( (*in-out*) hp);
    END;
    IF continue THEN
      fmt := UnMark(DataGetFormat(hp));
      IF fmt = formatFormat THEN
        WalkMarkFmt(hp, markOnOff);
      ELSE
        WalkMarkFmt(fmt, markOnOff);
        fmtScan := FormatStart(fmt);
        hpScan := hp;
        WalkMarkData( (*in-out*) fmtScan, (*in-out*) hpScan, markOnOff);
      END;
    END;
  END WalkMarkHp;

PROCEDURE WalkMarkFmt(fmt: Data.Pointer; markOnOff: BOOLEAN) =
  VAR continue: BOOLEAN;
  BEGIN
    <*ASSERT NOT( (fmt < Store.MinIndex) OR (fmt > Store.MaxIndex)) *>
    IF fmt = formatFormat THEN Data.Fault("WalkFmt") END;
    IF markOnOff THEN
      continue := WalkHeaderMarkShared( (*in-out*) fmt);
    ELSE
      continue := WalkHeaderUnMark( (*in-out*) fmt);
    END;
  END WalkMarkFmt;

PROCEDURE WalkMarkData(
    VAR (*in-out*) fmt, hp: Data.Pointer;
    markOnOff: BOOLEAN) =
  VAR
    class: Data.SmallInt;
    count: Data.Int;
    countPointees: Data.Int;
    fmtSave, pointer: Data.Pointer;
    polymorph: Data.Polymorph;
  BEGIN
    <*ASSERT NOT (((fmt < Store.MinIndex) OR (fmt > Store.MaxIndex)) 
	OR ((hp < Store.MinIndex) OR (hp > Store.MaxIndex)) ) *>
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    CASE VAL(class, Class) OF
    | Class.NoneCase => Data.Fault("WalkData None");
    | Class.EmptyCase =>
    | Class.PointeeCase => INC(hp, Data.PointeesPerPointee);
    | Class.SmallIntCase => INC(hp, Data.PointeesPerSmallInt);
    | Class.IntCase => INC(hp, Data.PointeesPerInt);
    | Class.ImmediateCase => INC(hp, Data.PointeesPerImmediate);
    | Class.PointerCase =>
        pointer := Store.GetPointer(hp);
        IF walkStackTop >= Data.WalkStackSize - 1 THEN
          Data.Fault("Walk stack")
        END;
        INC(walkStackTop);
        walkStack[walkStackTop].hp := pointer;
        INC(hp, Data.PointeesPerPointer);
    | Class.PolymorphCase =>
        polymorph := Store.GetPolymorph(hp);
        IF Data.IsPointer(polymorph) THEN
          IF walkStackTop >= Data.WalkStackSize - 1 THEN
            Data.Fault("Walk stack")
          END;
          INC(walkStackTop);
          walkStack[walkStackTop].hp := polymorph;
        END;
        INC(hp, Data.PointeesPerPolymorph);
    | Class.SeqCase =>
        count := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
        FOR index := 0 TO count - 1 DO
          WalkMarkData( (*in-out*) fmt, (*in-out*) hp, markOnOff);
        END;
    | Class.AltCase => Data.Fault("Walk: Unimplemented AltCase");
    | Class.IterCase =>
        countPointees := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
        IF countPointees = Data.PointeesPerSmallInt THEN
          count := Store.GetSmallInt(hp);
          INC(hp, Data.PointeesPerSmallInt);
        ELSIF countPointees = Data.PointeesPerInt THEN
          count := Store.GetInt(hp);
          INC(hp, Data.PointeesPerInt);
        ELSE
          Data.Fault("");
        END;
        fmtSave := fmt;
        FOR index := 0 TO count - 1 DO
          fmt := fmtSave;
          WalkMarkData( (*in-out*) fmt, (*in-out*) hp, markOnOff);
        END;
    | Class.AreaCase => Data.Fault("Walk Area");
    | Class.RepeatCase =>
        count := Store.GetInt(fmt);
        INC(fmt, Data.PointeesPerInt);
        fmtSave := fmt;
        FOR index := 0 TO count - 1 DO
          fmt := fmtSave;
          WalkMarkData( (*in-out*) fmt, (*in-out*) hp, markOnOff);
        END;
    | Class.RunCase => Data.Fault("Walk Run");
    END;
  END WalkMarkData;

(* === Copy === *)

PROCEDURE WalkCopy(hp: Data.Pointer; relative: BOOLEAN): Data.Pointer =
  VAR resHp, newHp, at: Data.Pointer; root: BOOLEAN;
  BEGIN
    walkStackTop := 0;
    walkStack[walkStackTop].hp := hp;
    walkStack[walkStackTop].at := Data.MinPointer;
    root := TRUE;
    WHILE walkStackTop >= 0 DO
      hp := walkStack[walkStackTop].hp;
      at := walkStack[walkStackTop].at;
      DEC(walkStackTop);
      newHp := WalkCopyHp(hp, relative);
      IF root THEN
        resHp := newHp;
        root := FALSE;
      ELSE
        IF relative THEN newHp := RelativePointer(newHp); END;
        WalkPointerCopy(at, newHp);
      END;
    END;
    RETURN resHp;
  END WalkCopy;

PROCEDURE WalkCopyHp(hp: Data.Pointer; relative: BOOLEAN): Data.Pointer =
  VAR fmt, fmtScan, hpScan, newHp, newFmt, newHpScan: Data.Pointer;
  BEGIN
    <*ASSERT NOT( (hp < Store.MinIndex) OR (hp > Store.MaxIndex)) *>
    IF hp = Data.MinPointer THEN RETURN Data.MinPointer END;
    fmt := UnMark(DataGetFormat(hp));
    IF fmt = formatFormat THEN
      newHp := WalkCopyFmt(hp);
      RETURN newHp;
    ELSE
      IF WalkHeaderCopy( (*in-out*) hp) THEN
        newFmt := WalkCopyFmt(fmt);
        IF relative THEN newFmt := RelativePointer(newFmt); END;
        newHp := WalkRunCopy(hp, newFmt);
        fmtScan := FormatStart(fmt);
        hpScan := hp;
        newHpScan := newHp;
        WalkCopyData( (*in-out*) fmtScan,
          (*in-out*) hpScan, (*in-out*) newHpScan);
        RETURN newHp;
      ELSE
	RETURN hp;
      END;
    END;
  END WalkCopyHp;

PROCEDURE WalkCopyFmt(fmt: Data.Pointer): Data.Pointer =
  BEGIN
    <*ASSERT NOT ((fmt < Store.MinIndex) OR (fmt > Store.MaxIndex)) *>
    IF fmt = formatFormat THEN Data.Fault("WalkFmt") END;
    IF WalkHeaderCopy( (*in-out*) fmt) THEN
      RETURN WalkFormatCopy(fmt);
    ELSE
      RETURN fmt;
    END;
  END WalkCopyFmt;

PROCEDURE WalkCopyData(VAR (*in-out*) fmt, hp, newHp: Data.Pointer) =
  VAR
    class: Data.SmallInt;
    count: Data.Int;
    countPointees: Data.Int;
    fmtSave, pointer: Data.Pointer;
    polymorph: Data.Polymorph;
  BEGIN
    <*ASSERT NOT( ((fmt < Store.MinIndex) OR (fmt > Store.MaxIndex)) 
	OR ((hp < Store.MinIndex) OR (hp > Store.MaxIndex)) OR
       ((newHp < Store.MinIndex) OR (newHp > Store.MaxIndex)) ) *>
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    CASE VAL(class, Class) OF
    | Class.NoneCase => Data.Fault("WalkData None");
    | Class.EmptyCase =>
    | Class.PointeeCase =>
        WalkPointeeCopy(newHp, Store.GetPointee(hp));
        INC(newHp, Data.PointeesPerPointee);
        INC(hp, Data.PointeesPerPointee);
    | Class.SmallIntCase =>
        WalkSmallIntCopy(newHp, Store.GetSmallInt(hp));
        INC(newHp, Data.PointeesPerSmallInt);
        INC(hp, Data.PointeesPerSmallInt);
    | Class.IntCase =>
        WalkIntCopy(newHp, Store.GetInt(hp));
        INC(newHp, Data.PointeesPerInt);
        INC(hp, Data.PointeesPerInt);
    | Class.ImmediateCase =>
        WalkImmediateCopy(newHp, Store.GetImmediate(hp));
        INC(newHp, Data.PointeesPerImmediate);
        INC(hp, Data.PointeesPerImmediate);
    | Class.PointerCase =>
        pointer := Store.GetPointer(hp);
        IF walkStackTop >= Data.WalkStackSize - 1 THEN
          Data.Fault("Walk stack")
        END;
        INC(walkStackTop);
        walkStack[walkStackTop].hp := pointer;
        walkStack[walkStackTop].at := newHp;
        INC(newHp, Data.PointeesPerPointer);
        INC(hp, Data.PointeesPerPointer);
    | Class.PolymorphCase =>
        polymorph := Store.GetPolymorph(hp);
        IF Data.IsImmediate(polymorph) THEN
          WalkImmediateCopy(newHp, polymorph);
          INC(newHp, Data.PointeesPerImmediate);
        ELSE
          IF walkStackTop >= Data.WalkStackSize - 1 THEN
            Data.Fault("Walk stack")
          END;
          INC(walkStackTop);
          walkStack[walkStackTop].hp := polymorph;
          walkStack[walkStackTop].at := newHp;
          INC(newHp, Data.PointeesPerPointer);
        END;
        INC(hp, Data.PointeesPerPolymorph);
    | Class.SeqCase =>
        count := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
        FOR index := 0 TO count - 1 DO
          WalkCopyData( (*in-out*) fmt, (*in-out*) hp, (*in-out*) newHp);
        END;
    | Class.AltCase => Data.Fault("Walk: Unimplemented AltCase");
    | Class.IterCase =>
        countPointees := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
        IF countPointees = Data.PointeesPerSmallInt THEN
          count := Store.GetSmallInt(hp);
          WalkSmallIntCopy(newHp, count);
          INC(newHp, Data.PointeesPerSmallInt);
          INC(hp, Data.PointeesPerSmallInt);
        ELSIF countPointees = Data.PointeesPerInt THEN
          count := Store.GetInt(hp);
          WalkIntCopy(newHp, count);
          INC(newHp, Data.PointeesPerInt);
          INC(hp, Data.PointeesPerInt);
        ELSE
          Data.Fault("");
        END;
        fmtSave := fmt;
        FOR index := 0 TO count - 1 DO
          fmt := fmtSave;
          WalkCopyData( (*in-out*) fmt, (*in-out*) hp, (*in-out*) newHp);
        END;
    | Class.AreaCase => Data.Fault("Walk Area");
    | Class.RepeatCase =>
        count := Store.GetInt(fmt);
        INC(fmt, Data.PointeesPerInt);
        fmtSave := fmt;
        FOR index := 0 TO count - 1 DO
          fmt := fmtSave;
          WalkCopyData( (*in-out*) fmt, (*in-out*) hp, (*in-out*) newHp);
        END;
    | Class.RunCase => Data.Fault("Walk Run");
    END;
  END WalkCopyData;

PROCEDURE WalkHeaderCopy(VAR (*in-out*) hp: Data.Pointer): BOOLEAN =
  VAR header: Data.Pointer;
  BEGIN
    header := DataGetFormat(hp);
    IF TestShared(header) THEN
      RETURN NOT CopiedShared( (*in-out*) hp); 
    END;
    RETURN TRUE;
  END WalkHeaderCopy;

PROCEDURE WalkFormatCopy(fmt: Data.Pointer): Data.Pointer =
  VAR fmtHeader, newFmt, scanFmt: Data.Pointer; size: Data.Int; 
  BEGIN
    size := DataGetSize(fmt);
    Store.Align();
    Store.LayPointer(formatFormat);
    Store.LayInt(size);
    newFmt := Store.hp;
    scanFmt := fmt;
    FOR i := 0 TO size - 1 DO
      Store.LayPointee(Store.GetPointee(scanFmt));
      INC(scanFmt, Data.PointeesPerPointee);
    END;
    fmtHeader := DataGetFormat(fmt);
    IF TestShared(fmtHeader) THEN
      RememberSharedCopy(fmt, newFmt);
    END;
    RETURN newFmt;
  END WalkFormatCopy;

PROCEDURE WalkRunCopy(hp, newFmt: Data.Pointer): Data.Pointer =
  VAR
    header, newHp: Data.Pointer;
    size: Data.Int;
    fmt: Data.Pointer;
  BEGIN
    fmt := UnMark(DataGetFormat(hp));
    size := DataGetSize(hp);
    Store.Align();
    Store.LayPointer(newFmt);
    Store.LayInt(size);
    newHp := Store.hp;
    INC(Store.hp, size); (* dirty memory *)
    header := DataGetFormat(hp);
    IF TestShared(header) THEN
      RememberSharedCopy(hp, newHp); 
    END;
    RETURN newHp;
  END WalkRunCopy;

PROCEDURE WalkPointeeCopy(hp: Data.Pointer; pointee: Data.Pointee) =
  BEGIN Store.SetPointee(hp, pointee); END WalkPointeeCopy;

PROCEDURE WalkPointerCopy(hp: Data.Pointer; pointer: Data.Pointer) =
  BEGIN Store.SetPointer(hp, pointer); END WalkPointerCopy;

PROCEDURE WalkImmediateCopy(hp: Data.Pointer; immediate: Data.Immediate) =
  BEGIN Store.SetImmediate(hp, immediate); END WalkImmediateCopy;

PROCEDURE WalkSmallIntCopy(hp: Data.Pointer; smallInt: Data.SmallInt) =
  BEGIN Store.SetSmallInt(hp, smallInt); END WalkSmallIntCopy;

PROCEDURE WalkIntCopy(hp: Data.Pointer; int: Data.Int) =
  BEGIN Store.SetInt(hp, int); END WalkIntCopy;

PROCEDURE MarkAndCopy(hp: Data.Pointer): Data.Pointer =
  VAR newHp: Data.Pointer;
  BEGIN
    WalkMark(hp, TRUE); (* -- compute size *)
    (* -- IF doesn't fit THEN oldHp := WalkMark(hp, FALSE); Collect(); oldHp
       := WalkMark(hp, TRUE); (* -- compute size *) IF still doesnt' fit THEN
       abort END; END; *)
    newHp := WalkCopy(hp, FALSE);
    WalkMark(hp, FALSE);
    RETURN newHp;
  END MarkAndCopy;

PROCEDURE Copy(hp: Data.Pointer): Data.Pointer =
  VAR saveTopSharedStack: Data.Int; newHp: Data.Pointer;
  BEGIN
    saveTopSharedStack := topSharedStack;
    newHp := MarkAndCopy(hp);
    topSharedStack := saveTopSharedStack;
    RETURN newHp;
  END Copy;

(* === EXTERN === *)

(* Extern:
	1) Copy to contiguous segment, preserving sharing.
	2) Walk copy, subtracting relative origin from all pointers.
	3) Write single-shot to file.
	4) Free copy.
   -- Note: no collection allowed between 1 and 4. *)

PROCEDURE MarkAndExtern(hp: Data.Pointer): Data.Pointer =
  VAR newHp: Data.Pointer;
  BEGIN
    WalkMark(hp, TRUE); (* -- compute size *)
    (* -- IF doesn't fit THEN oldHp := WalkMark(hp, FALSE); Collect(); oldHp
       := WalkMark(hp, TRUE); (* -- compute size *) IF still doesnt' fit THEN
       abort END; END; *)
    newHp := WalkCopy(hp, TRUE);
    WalkMark(hp, FALSE);
    RETURN newHp;
  END MarkAndExtern;

PROCEDURE ExternCommon(hp: Data.Pointer): Data.Int =
  VAR saveTopSharedStack: Data.Int; newHp, externStop: Data.Pointer;
  BEGIN
    saveTopSharedStack := topSharedStack;
    Store.Align();
    externStart := Store.hp;
    newHp := MarkAndExtern(hp);
    Store.Align();
    Store.LayPointer(RelativePointer(newHp));
    externStop := Store.hp;
    topSharedStack := saveTopSharedStack;
    RETURN externStop - externStart;
  END ExternCommon;

PROCEDURE Extern(wr: QOS.Writer; hp: Data.Pointer): BOOLEAN =
  TYPE IntPtr = UNTRACED REF Data.Int;
  VAR header: ARRAY [0..7] OF CHAR;
    size: Data.Int; sizeAdr: IntPtr;
  BEGIN
    size := ExternCommon(hp);
    header[0] := Data.ExternId1;
    header[1] := Data.ExternId2;
    header[2] := Data.ExternId3;
    header[3] := VAL(Data.ExternVersion, CHAR);
    sizeAdr := LOOPHOLE(ADR(header[4]), IntPtr);
    sizeAdr^ := size;
    IF NOT QOS.PutSub(wr, ADR(header[0]), 0, 8) THEN
      RETURN FALSE
    END;
    IF NOT QOS.PutSub(wr, LOOPHOLE(Store.heapAdr, ADDRESS), 
	externStart, size) THEN
      RETURN FALSE
    END;
    Store.hp := externStart;
    RETURN TRUE;
  END Extern;

(* -- old
PROCEDURE Extern(wr: QOS.Writer; hp: Data.Pointer): BOOLEAN =
  VAR size: Data.Int; word: IntWord;
  BEGIN
    size := ExternCommon(hp);
    word := IntWord(size);
    IF NOT QOS.PutChar(wr, Data.ExternId1) THEN RETURN FALSE END;
    IF NOT QOS.PutChar(wr, Data.ExternId2) THEN RETURN FALSE END;
    IF NOT QOS.PutChar(wr, Data.ExternId3) THEN RETURN FALSE END;
    IF NOT QOS.PutChar(wr, CHR(Data.ExternVersion)) THEN RETURN FALSE END;
    IF NOT QOS.PutChar(wr, CHAR(word[0])) THEN RETURN FALSE END;
    IF NOT QOS.PutChar(wr, CHAR(word[1])) THEN RETURN FALSE END;
    IF NOT QOS.PutChar(wr, CHAR(word[2])) THEN RETURN FALSE END;
    IF NOT QOS.PutChar(wr, CHAR(word[3])) THEN RETURN FALSE END;
    IF NOT QOS.PutSub(wr, Store.heapAdr, externStart, size) THEN
      RETURN FALSE
    END;
    Store.hp := externStart;
    RETURN TRUE;
  END Extern;
  *)

(* RPC: PROCEDURE ExternMarshal(VAR rpc: QRPC.T; hp: Data.Pointer):
BOOLEAN =
   VAR size: Data.Int; BEGIN size := ExternCommon(hp); IF NOT
   QRPC.Put(SYSTEM.ADR(size), SYSTEM.BYTESIZE(size), rpc) THEN RETURN FALSE
   END; IF NOT QRPC.Put(SYSTEM.ADR(Store.heap[externStart]), size, rpc) THEN
   RETURN FALSE END; Store.hp := externStart; RETURN TRUE; END ExternMarshal;
*)

(* === INTERN === *)

(* Intern:
	1) Read single-shot to contiguous region.
	2) Walk region, adding relative origin to all pointers.
   -- Note: no collection allowed between 1 and 2. *)

PROCEDURE WalkInternData(VAR (*in-out*) fmt, hp: Data.Pointer) =
  VAR
    class: Data.SmallInt;
    count: Data.Int;
    countPointees: Data.Int;
    fmtSave: Data.Pointer;
    polymorph: Data.Polymorph;
  BEGIN
    <* ASSERT NOT( ((fmt < Store.MinIndex) OR (fmt > Store.MaxIndex)) 
	OR ((hp < Store.MinIndex) OR (hp > Store.MaxIndex)) ) *>
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    CASE VAL(class, Class) OF
    | Class.NoneCase => Data.Fault("WalkData None");
    | Class.EmptyCase =>
    | Class.PointeeCase => INC(hp, Data.PointeesPerPointee);
    | Class.SmallIntCase => INC(hp, Data.PointeesPerSmallInt);
    | Class.IntCase => INC(hp, Data.PointeesPerInt);
    | Class.ImmediateCase => INC(hp, Data.PointeesPerImmediate);
    | Class.PointerCase =>
        Store.SetPointer(hp, AbsolutePointer(Store.GetPointer(hp)));
        INC(hp, Data.PointeesPerPointer);
    | Class.PolymorphCase =>
        polymorph := Store.GetPolymorph(hp);
        IF Data.IsPointer(polymorph) THEN
          Store.SetPointer(hp, AbsolutePointer(polymorph));
        END;
        INC(hp, Data.PointeesPerPolymorph);
    | Class.SeqCase =>
        count := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
        FOR index := 0 TO count - 1 DO
          WalkInternData( (*in-out*) fmt, (*in-out*) hp);
        END;
    | Class.AltCase => Data.Fault("Walk: Unimplemented AltCase");
    | Class.IterCase =>
        countPointees := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
        IF countPointees = Data.PointeesPerSmallInt THEN
          count := Store.GetSmallInt(hp);
          INC(hp, Data.PointeesPerSmallInt);
        ELSIF countPointees = Data.PointeesPerInt THEN
          count := Store.GetInt(hp);
          INC(hp, Data.PointeesPerInt);
        ELSE
          Data.Fault("");
        END;
        fmtSave := fmt;
        FOR index := 0 TO count - 1 DO
          fmt := fmtSave;
          WalkInternData( (*in-out*) fmt, (*in-out*) hp);
        END;
    | Class.AreaCase => Data.Fault("Walk Area");
    | Class.RepeatCase =>
        count := Store.GetInt(fmt);
        INC(fmt, Data.PointeesPerInt);
        fmtSave := fmt;
        FOR index := 0 TO count - 1 DO
          fmt := fmtSave;
          WalkInternData( (*in-out*) fmt, (*in-out*) hp);
        END;
    | Class.RunCase => Data.Fault("Walk Run");
    END;
  END WalkInternData;

PROCEDURE InternCommon(size: Data.Int): Data.Pointer =
  VAR hp, internStop, scan, fmt, hpScan, newHpScan, fmtScan: Data.Pointer;
  BEGIN
    INC(Store.hp, size - Data.PointeesPerPointer);
    hp := AbsolutePointer(Store.GetPointer(Store.hp));
    internStop := Store.hp;
    scan := internStart;
    LOOP
      scan := Data.AlignUp(scan, Store.DataAlignment);
      IF scan >= internStop THEN EXIT END;
      fmt := AbsolutePointer(Store.GetPointer(scan));
      INC(scan, Data.PointeesPerPointer);
      IF fmt = formatFormat THEN
        size := Store.GetInt(scan);
        INC(scan, Data.PointeesPerInt);
        INC(scan, size);
      ELSE
        size := Store.GetInt(scan);
        INC(scan, Data.PointeesPerInt);
        DataSetFormat(scan, fmt);
        fmtScan := FormatStart(fmt);
        hpScan := scan;
        newHpScan := scan;
        WalkInternData( (*in-out*) fmtScan, (*in-out*) hpScan);
        INC(scan, size);
      END;
    END;
    RETURN hp;
  END InternCommon;

PROCEDURE Intern(rd: QOS.Reader; VAR (*out*) hp: Data.Pointer): BOOLEAN =
  TYPE IntPtr = UNTRACED REF Data.Int;
  VAR header: ARRAY [0..7] OF CHAR;
    size, readSize: Data.Int; sizeAdr: IntPtr;
  BEGIN
    IF NOT QOS.GetSub(rd, ADR(header[0]), 0, 8,
             (*out*) readSize) THEN
      RETURN FALSE
    END;
    IF readSize # 8 THEN RETURN FALSE END;
    IF (header[0] # Data.ExternId1) OR (header[1] # Data.ExternId2) OR
        (header[2] # Data.ExternId3) OR (ORD(header[3]) # Data.ExternVersion)
    THEN
      RETURN FALSE
    END;
    sizeAdr := LOOPHOLE(ADR(header[4]), IntPtr);
    size := sizeAdr^;
    (* -- make sure it fits in memory, or call the collector *)
    Store.Align();
    internStart := Store.hp;
    IF NOT QOS.GetSub(rd, LOOPHOLE(Store.heapAdr, ADDRESS), internStart, size,
             (*out*) readSize) THEN
      RETURN FALSE
    END;
    IF readSize # size THEN RETURN FALSE END;
    hp := InternCommon(size);
    RETURN TRUE;
  END Intern;

(* -- old  
PROCEDURE Intern(rd: QOS.Reader; VAR (*out*) hp: Data.Pointer): BOOLEAN =
  VAR ch: CHAR; word: IntWord; size, readSize: Data.Int;
  BEGIN
    IF NOT QOS.GetChar(rd, (*out*) ch) THEN RETURN FALSE END;
    IF ch # Data.ExternId1 THEN RETURN FALSE END;
    IF NOT QOS.GetChar(rd, (*out*) ch) THEN RETURN FALSE END;
    IF ch # Data.ExternId2 THEN RETURN FALSE END;
    IF NOT QOS.GetChar(rd, (*out*) ch) THEN RETURN FALSE END;
    IF ch # Data.ExternId3 THEN RETURN FALSE END;
    IF NOT QOS.GetChar(rd, (*out*) ch) THEN RETURN FALSE END;
    IF ORD(ch) # Data.ExternVersion THEN RETURN FALSE END;
    IF NOT QOS.GetChar(rd, (*out*) ch) THEN RETURN FALSE END;
    word[0] := Data.Pointee(ch);
    IF NOT QOS.GetChar(rd, (*out*) ch) THEN RETURN FALSE END;
    word[1] := Data.Pointee(ch);
    IF NOT QOS.GetChar(rd, (*out*) ch) THEN RETURN FALSE END;
    word[2] := Data.Pointee(ch);
    IF NOT QOS.GetChar(rd, (*out*) ch) THEN RETURN FALSE END;
    word[3] := Data.Pointee(ch);
    size := Data.Int(word);
    (* -- make sure it fits in memory, or call the collector *)
    Store.Align();
    internStart := Store.hp;
    IF NOT QOS.GetSub(rd, Store.heapAdr, internStart, size,
             (*out*) readSize) THEN
      RETURN FALSE
    END;
    IF readSize # size THEN RETURN FALSE END;
    hp := InternCommon(size);
    RETURN TRUE;
  END Intern;
*)

(* RPC: PROCEDURE InternMarshal( VAR rpc: QRPC.T; VAR (* out *) hp:
   Data.Pointer) : BOOLEAN = VAR size: Data.Int; BEGIN IF NOT
   QRPC.Get(SYSTEM.ADR(size), SYSTEM.BYTESIZE(size), rpc) THEN RETURN FALSE
   END; (* -- make sure it fits in memory, or call the collector *)
   Store.Align(); internStart := Store.hp; IF NOT
   QRPC.Get(SYSTEM.ADR(Store.heap[internStart]), size, rpc) THEN RETURN FALSE
   END; hp := InternCommon(size); RETURN TRUE; END InternMarshal; *)

(* === EXTERN PORTABLE === *)

(* Extern in architecture-independent representation:
	1) Mark
	2) Walk, outputting ascii
	3) Unmark
   Each pickle starts with '[' and ends with ']'. Between the
   brackets there is a sequence of N varhex's (the sizes of the following
   segments) then '|', then N segments (first all the format segments
   starting with 'F', then all the data segments starting with 'D'),
   and finally the "root" pointer run (see below). 
   Within each segment, another segment is referred to by "^n", 
   where n is the number of that segment in the sequence; the first segment 
   is number 0.
   Each segment is a possibly nested, flattened, run; the points of nesting 
   in a data run are determined by the position of pointers in its 
   corresponding format run. 
   A format run starts with 'F' and its size, then contains the format codes, 
   which are encode as 'n'Null, 'e'Empty, 'b'Pointee, 's'SmallInt, 'i'Int, 
   'k'Immediate, 'p'Pointer, 'm'Polymorph, 'S'Seq+smallInt+runs, 
   'A'Alt(unimp.), 'I'Iter+smallInt+run, 'E'Area(unimp.), 'R'Repeat+int+run, 
   'U'Run(unimp.). 
   A data run starts with 'D', then comes its format run and the data size, 
   and then the rest is determined by the format.
   Pointees, ints, smallints and (decoded-)immediates are written as varhex'es.
   Pointers are written as either '-' (null pointer), or '+' followed
   by a run (starting with 'F' or 'D'), or '^' followed by a varhex referring 
   to a different segment. Polymorphs are written as either 'k' followed by 
   an immediate, or 'p' followed by a pointer.

   -- Note: no collection allowed between 1 and 3. *)

PROCEDURE PutPositiveVarHex(wr: QOS.Writer; n: INTEGER) =
  VAR i: INTEGER; buffer: ARRAY [0..19] OF CHAR;
  BEGIN
    IF n=0 THEN EVAL QOS.PutChar(wr, '@')
    ELSE
      i := 0;
      WHILE n#0 DO
	buffer[i] := VAL(ORD('@') + (n MOD 16), CHAR);
	n := n DIV 16;
        INC(i);
      END;
      FOR j := i-1 TO 0 BY -1 DO EVAL QOS.PutChar(wr, buffer[j]) END;
    END;
    EVAL QOS.PutChar(wr, ' ')
  END PutPositiveVarHex;

PROCEDURE PutVarHex(wr: QOS.Writer; n: INTEGER) =
  BEGIN
    IF n<0 THEN
      EVAL QOS.PutChar(wr, '-');
      n := -n;
    END;
    PutPositiveVarHex(wr, n);
  END PutVarHex;

PROCEDURE ReorderSharedStack(bot, top: INTEGER) =
  (* Move all the format runs before all the data runs (before==topmost) *)
  VAR fstDat, lstFmt: INTEGER; dat, fmt: Data.Pointer;
  BEGIN
    fstDat := top;
    lstFmt := bot+1;
    IF lstFmt >= fstDat THEN RETURN END;
    LOOP
      WHILE 
	  (UnMark(DataGetFormat(sharedStack[fstDat].shared)) = formatFormat) 
	  AND (lstFmt < fstDat) DO
        DEC(fstDat);
      END;
      WHILE (UnMark(DataGetFormat(sharedStack[lstFmt].shared)) # formatFormat)
	  AND (lstFmt < fstDat) DO
        INC(lstFmt);
      END;
      IF lstFmt >= fstDat THEN EXIT END;
      fmt := sharedStack[lstFmt].shared;
      dat := sharedStack[fstDat].shared;
      sharedStack[lstFmt].shared := dat;
      sharedStack[fstDat].shared := fmt;
    END;
  END ReorderSharedStack;

PROCEDURE ExternPortable(wr: QOS.Writer; hp: Data.Pointer): BOOLEAN =
  VAR saveTopSharedStack: Data.Int; scan: Data.Int;
  BEGIN
    IF NOT QOS.PutChar(wr, Data.ExternPortableId1) THEN RETURN FALSE END;
    IF NOT QOS.PutChar(wr, Data.ExternPortableId2) THEN RETURN FALSE END;
    IF NOT QOS.PutChar(wr, Data.ExternPortableId3) THEN RETURN FALSE END;
    IF NOT QOS.PutChar(wr, VAL(Data.ExternPortableVersion, CHAR)) 
      THEN RETURN FALSE END;
    saveTopSharedStack := topSharedStack;
    WalkMark(hp, TRUE);
    ReorderSharedStack(saveTopSharedStack, topSharedStack);
    IF NOT QOS.PutChar(wr, '[') THEN RETURN FALSE END;
    scan := topSharedStack;
    WHILE scan > saveTopSharedStack DO
      PutVarHex(wr, DataGetSize(sharedStack[scan].shared));
      DEC(scan);
    END;
    IF NOT QOS.PutChar(wr, '|') THEN RETURN FALSE END;
    scan := topSharedStack;
    WHILE scan > saveTopSharedStack DO
      PExternSegment(wr, sharedStack[scan].shared);
      DEC(scan);
    END;
    IF NOT QOS.PutChar(wr, ']') THEN RETURN FALSE END;
    PExternPointer(wr, hp);
    IF NOT QOS.PutChar(wr, '.') THEN RETURN FALSE END;
    WalkMark(hp, FALSE);
    topSharedStack := saveTopSharedStack;
    RETURN TRUE;
  END ExternPortable;

PROCEDURE PExternSegment(wr: QOS.Writer; hp: Data.Pointer) =
  (* Ignores the "shared" mark of hp^ *)
  VAR fmt: Data.Pointer;
  BEGIN
    <* ASSERT NOT ((hp < Store.MinIndex) OR (hp > Store.MaxIndex)) *>
    IF hp = Data.MinPointer THEN
      Data.Fault("PExtern: Null pointer");
    END;
    fmt := UnMark(DataGetFormat(hp));
    IF fmt = formatFormat THEN
      PExternFmt(wr, hp);
    ELSE
      PExternData(wr, fmt, hp);
    END;
  END PExternSegment;

PROCEDURE PExternPointer(wr: QOS.Writer; hp: Data.Pointer) =
  VAR fmt: Data.Pointer;
  BEGIN
    <* ASSERT NOT( (hp < Store.MinIndex) OR (hp > Store.MaxIndex) ) *>
    IF hp = Data.MinPointer THEN
      EVAL QOS.PutChar(wr, '-');
    ELSE
      fmt := DataGetFormat(hp);
      IF TestShared(fmt) THEN
        PExternShared(wr, hp);
      ELSE
        fmt := UnMark(fmt);
        EVAL QOS.PutChar(wr, '+');
        IF fmt = formatFormat THEN
          PExternFmt(wr, hp);
        ELSE
          PExternData(wr, fmt, hp);
        END;
      END;
    END;
  END PExternPointer;

PROCEDURE PExternShared(wr: QOS.Writer; hp: Data.Pointer) =
  VAR i, scan: INTEGER;
  BEGIN
    scan := topSharedStack;
    i := 0;
    WHILE scan >= 0 DO
      IF hp = sharedStack[scan].shared THEN
        EVAL QOS.PutChar(wr, '^');
	PutVarHex(wr, i);
	RETURN;
      END;
      INC(i);
      DEC(scan);
    END;
    Data.Fault("PExternShared: not found");
  END PExternShared;

PROCEDURE PExternFmt(wr: QOS.Writer; fmt: Data.Pointer) =
  VAR fmtFmt, scanFmt: Data.Pointer;
  BEGIN
    <*ASSERT NOT( (fmt < Store.MinIndex) OR (fmt > Store.MaxIndex) ) *>
    IF fmt = formatFormat THEN Data.Fault("PExternFmt") END;
    fmtFmt := UnMark(DataGetFormat(fmt));
    IF fmtFmt # formatFormat THEN Data.Fault("PExternFmt:  not Format") END;
    EVAL QOS.PutChar(wr, 'F');
    PutVarHex(wr, DataGetSize(fmt));
    scanFmt := FormatStart(fmt);
    PExternFmtRun(wr, (*in-out*)scanFmt);
  END PExternFmt;

PROCEDURE PExternFmtRun(wr: QOS.Writer; VAR (*in-out*) fmt: Data.Pointer) =
  VAR class, count, countPointees: INTEGER;
  BEGIN
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    CASE VAL(class, Class) OF
    | Class.NoneCase => 
	EVAL QOS.PutChar(wr, 'n');
    | Class.EmptyCase =>
	EVAL QOS.PutChar(wr, 'e');
    | Class.PointeeCase =>
	EVAL QOS.PutChar(wr, 'b');
    | Class.SmallIntCase =>
	EVAL QOS.PutChar(wr, 's');
    | Class.IntCase => 
	EVAL QOS.PutChar(wr, 'i');
    | Class.ImmediateCase =>
	EVAL QOS.PutChar(wr, 'k');
    | Class.PointerCase =>
	EVAL QOS.PutChar(wr, 'p');
    | Class.PolymorphCase =>
	EVAL QOS.PutChar(wr, 'm');
    | Class.SeqCase =>
	EVAL QOS.PutChar(wr, 'S');
        count := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
        PutVarHex(wr, count);
	FOR i := 1 TO count DO
	  PExternFmtRun(wr, (*in-out*)fmt);
        END;
    | Class.AltCase =>
	EVAL QOS.PutChar(wr, 'A');
	Data.Fault("Walk: Unimplemented AltCase");
    | Class.IterCase =>
	EVAL QOS.PutChar(wr, 'I');
        countPointees := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
        PutVarHex(wr, countPointees);
	PExternFmtRun(wr, (*in-out*)fmt);
    | Class.AreaCase => 
	EVAL QOS.PutChar(wr, 'E');
	Data.Fault("Walk Area");
    | Class.RepeatCase =>
	EVAL QOS.PutChar(wr, 'R');
        count := Store.GetInt(fmt);
        INC(fmt, Data.PointeesPerInt);
        PutVarHex(wr, count);
	PExternFmtRun(wr, (*in-out*)fmt);
    | Class.RunCase => 
	EVAL QOS.PutChar(wr, 'U');
	Data.Fault("Walk Run");
    END;
  END PExternFmtRun;

PROCEDURE PExternData(wr: QOS.Writer; fmt, hp: Data.Pointer) =
  VAR scanFmt, scanHp: Data.Pointer;
  BEGIN
    <*ASSERT NOT( ((fmt < Store.MinIndex) OR (fmt > Store.MaxIndex))
       OR ((hp < Store.MinIndex) OR (hp > Store.MaxIndex)) ) *>
    EVAL QOS.PutChar(wr, 'D');
    PExternPointer(wr, fmt);
    PutVarHex(wr, DataGetSize(hp));
    scanFmt := FormatStart(fmt);
    scanHp := hp;
    PExternDataRun(wr, (*in-out*) scanFmt, (*in-out*) scanHp);
  END PExternData;

PROCEDURE PExternDataRun(wr: QOS.Writer; VAR (*in-out*) fmt, hp: Data.Pointer) =
  VAR
    class: Data.SmallInt;
    count: Data.Int;
    countPointees: Data.Int;
    fmtSave, ptr: Data.Pointer;
    polymorph: Data.Polymorph;
  BEGIN
    IF IsProgFormat(fmt) THEN 
      PExternProg(wr, (*in-out*)fmt, (*in-out*)hp);
      RETURN;
    END;
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    CASE VAL(class, Class) OF
    | Class.NoneCase => Data.Fault("PExternDataRun None");
    | Class.EmptyCase =>
    | Class.PointeeCase =>
        PutVarHex(wr, ORD(Store.GetPointee(hp)));
        INC(hp, Data.PointeesPerPointee);
    | Class.SmallIntCase =>
        PutVarHex(wr, Store.GetSmallInt(hp));
        INC(hp, Data.PointeesPerSmallInt);
    | Class.IntCase => 
	PutVarHex(wr, Store.GetInt(hp)); 
	INC(hp, Data.PointeesPerInt);
    | Class.ImmediateCase =>
        PutVarHex(wr, Data.IntOfImmediate(Store.GetImmediate(hp)));
        INC(hp, Data.PointeesPerImmediate);
    | Class.PointerCase =>
        ptr := UnMark(Store.GetPointer(hp));
	PExternPointer(wr, ptr);
        INC(hp, Data.PointeesPerPointer);
    | Class.PolymorphCase =>
        polymorph := Store.GetPolymorph(hp);
        IF Data.IsImmediate(polymorph) THEN
	  EVAL QOS.PutChar(wr, 'k');
          PutVarHex(wr, Data.IntOfImmediate(polymorph));
        ELSE
	  EVAL QOS.PutChar(wr, 'p');
	  PExternPointer(wr, polymorph);
        END;
        INC(hp, Data.PointeesPerPolymorph);
    | Class.SeqCase =>
        count := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
        FOR i := 0 TO count - 1 DO
          PExternDataRun(wr, (* in-out *) fmt, (* in-out *) hp);
	END;
    | Class.AltCase => Data.Fault("Walk: Unimplemented AltCase");
    | Class.IterCase =>
        countPointees := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
        IF countPointees = Data.PointeesPerSmallInt THEN
          count := Store.GetSmallInt(hp);
          INC(hp, Data.PointeesPerSmallInt);
        ELSIF countPointees = Data.PointeesPerInt THEN
          count := Store.GetInt(hp);
          INC(hp, Data.PointeesPerInt);
        ELSE
          Data.Fault("");
        END;
        PutVarHex(wr, count);
        fmtSave := fmt;
        FOR i := 0 TO count - 1 DO
          fmt := fmtSave;
          PExternDataRun(wr, (* in-out *) fmt, (* in-out *) hp);
        END;
    | Class.AreaCase => Data.Fault("Walk Area");
    | Class.RepeatCase =>
        count := Store.GetInt(fmt);
        INC(fmt, Data.PointeesPerInt);
        fmtSave := fmt;
        FOR i := 0 TO count - 1 DO
          fmt := fmtSave;
          PExternDataRun(wr, (* in-out *) fmt, (* in-out *) hp);
        END;
    | Class.RunCase => Data.Fault("Walk Run");
    END;
  END PExternDataRun;

PROCEDURE IsProgFormat(fmt: Data.Pointer): BOOLEAN =
  VAR
    class: Data.SmallInt;
    count: Data.Int;
  BEGIN
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    IF VAL(class, Class) # Class.SeqCase THEN RETURN FALSE END;
    count := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    IF count # 2 THEN RETURN FALSE END;
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    IF VAL(class, Class) # Class.PointerCase THEN RETURN FALSE END;
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    IF VAL(class, Class) # Class.IterCase THEN RETURN FALSE END;
    count := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    IF count # Data.PointeesPerInt THEN RETURN FALSE END;
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    IF VAL(class, Class) # Class.PointeeCase THEN RETURN FALSE END;
    RETURN TRUE;
  END IsProgFormat;

PROCEDURE PExternProg(wr: QOS.Writer; VAR (*in-out*) fmt, hp: Data.Pointer) =
  VAR
    class: Data.SmallInt;
    count, countPointees: Data.Int;
    ptr: Data.Pointer;
  BEGIN
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    <*ASSERT (VAL(class, Class) = Class.SeqCase) *>
    count := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    <*ASSERT (count = 2) *>
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    <*ASSERT (VAL(class, Class) = Class.PointerCase) *>
    ptr := UnMark(Store.GetPointer(hp));
    PExternPointer(wr, ptr);
    INC(hp, Data.PointeesPerPointer);
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    <*ASSERT (VAL(class, Class) = Class.IterCase) *>
    countPointees := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    <*ASSERT (countPointees = Data.PointeesPerInt) *>
    count := Store.GetInt(hp);
    INC(hp, Data.PointeesPerInt);    
    PutVarHex(wr, count);
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    <*ASSERT (VAL(class, Class) = Class.PointeeCase) *>
    PExternCode(wr, count, hp);
    INC(hp, count);
  END PExternProg;

PROCEDURE PExternSmallInt(wr: QOS.Writer; VAR (*in-out*) PC: Data.Pointer) =
  VAR i: Data.Int;
  BEGIN
    i := Store.GetSmallInt(PC);
    INC(PC, Data.PointeesPerSmallInt);
    PutVarHex(wr, i);
  END PExternSmallInt;

PROCEDURE PExternAddrN(wr: QOS.Writer; i: Data.Int; 
    VAR (*in-out*) PC: Data.Pointer) =
  BEGIN
    WHILE i > 0 DO
      PExternAddr(wr, (*in-out*) PC);
      DEC(i);
    END;
  END PExternAddrN;

PROCEDURE PExternTarget(wr: QOS.Writer; VAR (*in-out*) PC: Data.Pointer) =
  VAR rel: Data.Int;
  BEGIN
    rel := Store.GetSmallInt(PC);
    INC(PC, Data.PointeesPerSmallInt);
    PutVarHex(wr, rel);
  END PExternTarget;

PROCEDURE PExternTargetList(wr: QOS.Writer; VAR (*in-out*) PC: Data.Pointer) =
  VAR size: Data.Int;
  BEGIN
    size := Store.GetSmallInt(PC);
    INC(PC, Data.PointeesPerSmallInt);
    PutVarHex(wr, size);
    WHILE size > 0 DO
      PExternTarget(wr, (*in-out*) PC);
      DEC(size);
    END;
  END PExternTargetList;

PROCEDURE PExternAddrList(wr: QOS.Writer; VAR (*in-out*) PC: Data.Pointer) =
  VAR size: Data.Int;
  BEGIN
    size := Store.GetSmallInt(PC);
    INC(PC, Data.PointeesPerSmallInt);
    PutVarHex(wr, size);
    WHILE size > 0 DO
      PExternAddr(wr, (*in-out*) PC);
      DEC(size);
    END;
  END PExternAddrList;

PROCEDURE PExternAddr(wr: QOS.Writer; VAR (*in-out*) PC: Data.Pointer) =
  VAR opAddrClass, int: Data.Int; float: Data.Float;
  BEGIN
    opAddrClass := Store.GetSmallCard(PC);
    INC(PC, Data.PointeesPerSmallCard);
    PutVarHex(wr, opAddrClass);
    CASE VAL(opAddrClass DIV Data.PointeeSize, Code.OpAddrClass) OF
    | Code.OpAddrClass.OpAddrImmedOkCase =>
    | Code.OpAddrClass.OpAddrImmedBoolCase =>
    | Code.OpAddrClass.OpAddrImmedCharCase =>
    | Code.OpAddrClass.OpAddrImmedIntSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
	PutVarHex(wr, int);
    | Code.OpAddrClass.OpAddrImmedIntFullCase =>
        int := Store.GetInt(PC);
        INC(PC, Data.PointeesPerInt);
	PutVarHex(wr, int);
    | Code.OpAddrClass.OpAddrImmedRealCase =>
        float := Store.GetFloat(PC);
        INC(PC, Data.PointeesPerFloat);
	PutVarHex(wr, Data.ImmediateOfFloat(float));
    | Code.OpAddrClass.OpAddrLiteralSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
	PutVarHex(wr, int);
    | Code.OpAddrClass.OpAddrFrameSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
	PutVarHex(wr, int);
    | Code.OpAddrClass.OpAddrFrameLocativeSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
	PutVarHex(wr, int);
    | Code.OpAddrClass.OpAddrGlobalSmallCase =>
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
	PutVarHex(wr, int);
    | Code.OpAddrClass.OpAddrIndexedSmallCase =>
	PExternAddr(wr, (*in-out*)PC);
        int := Store.GetSmallInt(PC);
        INC(PC, Data.PointeesPerSmallInt);
	PutVarHex(wr, int);
    | Code.OpAddrClass.OpAddrStackDisplCase =>
	PExternAddr(wr, (*in-out*)PC);
    | Code.OpAddrClass.OpAddrSpecialCase =>
    END;
  END PExternAddr;

PROCEDURE PExternCode(wr: QOS.Writer; count: Data.Int; PC: Data.Pointer) =
  VAR limit: Data.Pointer; opClass: Data.Int;
  BEGIN
    limit := PC + count;
    WHILE PC < limit DO
      opClass := Store.GetSmallCard(PC);
      INC(PC, Data.PointeesPerSmallCard);
      PutVarHex(wr, opClass);
      CASE VAL(opClass DIV Data.PointeeSize, Code.OpClass) OF
      | Code.OpClass.OpMoveCase =>
        PExternAddrN(wr, 2, (*in-out*)PC);
      | Code.OpClass.OpFrameCase =>
	PExternSmallInt(wr, (*in-out*)PC);
	PExternSmallInt(wr, (*in-out*)PC);
      | Code.OpClass.OpApplyCase =>
        PExternAddr(wr, (*in-out*)PC);
      | Code.OpClass.OpReturnCase =>
        PExternSmallInt(wr, (*in-out*)PC);
      | Code.OpClass.OpReturnObsoleteCase =>
	PExternSmallInt(wr, (*in-out*)PC);
      | Code.OpClass.OpArgumentsCase =>
	PExternSmallInt(wr, (*in-out*)PC);
        PExternAddrList(wr, (*in-out*)PC);
      | Code.OpClass.OpApplyArguments0Results1Case =>
	PExternAddr(wr, (*in-out*)PC);
      | Code.OpClass.OpApplyArguments1Results1Case =>
	PExternAddr(wr, (*in-out*)PC);
        PExternAddr(wr, (*in-out*)PC);
      | Code.OpClass.OpApplyArguments2Results1Case =>
	PExternAddr(wr, (*in-out*)PC);
        PExternAddr(wr, (*in-out*)PC);
        PExternAddr(wr, (*in-out*)PC);
      | Code.OpClass.OpApplyArgumentsResults1Case =>
	PExternAddr(wr, (*in-out*)PC);
        PExternAddrList(wr, (*in-out*)PC);
      | Code.OpClass.OpResultsCase =>
        PExternAddrList(wr, (*in-out*)PC);
      | Code.OpClass.OpResults1Case =>
        PExternAddr(wr, (*in-out*)PC);
      | Code.OpClass.OpClosureCase =>
        PExternAddr(wr, (*in-out*)PC);
        PExternAddrList(wr, (*in-out*)PC);
        PExternAddr(wr, (*in-out*)PC);
      | Code.OpClass.OpDumClosureCase =>
        PExternAddrN(wr, 3, (*in-out*)PC);
      | Code.OpClass.OpRecClosureCase =>
        PExternAddr(wr, (*in-out*)PC);
        PExternAddrList(wr, (*in-out*)PC);
      | Code.OpClass.OpTupleCase =>
        PExternAddrList(wr, (*in-out*)PC);
        PExternAddr(wr, (*in-out*)PC);
      | Code.OpClass.OpJumpCase =>
        PExternTarget(wr, (*in-out*)PC);
      | Code.OpClass.OpJumpWhenCase =>
        PExternAddrN(wr, 2, (*in-out*)PC);
        PExternTarget(wr, (*in-out*)PC);
      | Code.OpClass.OpCaseCheckCase =>
        PExternAddrN(wr, 2, (*in-out*)PC);
      | Code.OpClass.OpCaseCase =>
        PExternAddr(wr, (*in-out*)PC);
        PExternTargetList(wr, (*in-out*)PC);
      | Code.OpClass.OpCaseFaultCase =>
      | Code.OpClass.OpArrayCase =>
        PExternAddrList(wr, (*in-out*)PC);
        PExternAddr(wr, (*in-out*)PC);
      | Code.OpClass.OpTrapCase =>
        PExternAddr(wr, (*in-out*)PC);
        PExternTarget(wr, (*in-out*)PC);
      | Code.OpClass.OpUntrapCase =>
      | Code.OpClass.OpRaiseCase =>
        PExternAddrN(wr, 2, (*in-out*)PC);
      | Code.OpClass.OpUnwindCase =>
      | Code.OpClass.OpCrashCase =>
      | Code.OpClass.OpEndCase =>
      | Code.OpClass.OpStartCase =>
        PExternAddrN(wr, 2, (*in-out*)PC);
      | Code.OpClass.OpStopCase =>
      | Code.OpClass.OpNoOp =>
      | Code.OpClass.OpDataCompareCase =>
        PExternAddrN(wr, 
	  Code.dataCompareArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataCompareClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataBoolCase =>
        PExternAddrN(wr, 
	  Code.dataBoolArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataBoolClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataAsciiCase =>
        PExternAddrN(wr, 
	  Code.dataAsciiArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataAsciiClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataIntCase =>
        PExternAddrN(wr, 
	  Code.dataIntArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataIntClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataRealCase =>
        PExternAddrN(wr, 
	  Code.dataRealArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataRealClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataStringCase =>
        PExternAddrN(wr, 
	  Code.dataStringArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataStringClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataArrayCase =>
        PExternAddrN(wr, 
	  Code.dataArrayArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataArrayClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataReaderCase =>
        PExternAddrN(wr, 
	  Code.dataReaderArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataReaderClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataWriterCase =>
        PExternAddrN(wr, 
	  Code.dataWriterArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataWriterClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataDynamicCase =>
        PExternAddrN(wr, 
	  Code.dataDynamicArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataDynamicClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataTimeCase =>
        PExternAddrN(wr, 
	  Code.dataTimeArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataTimeClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataValueCase =>
        PExternAddrN(wr, 
	  Code.dataValueArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataValueClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataMachineCase =>
        PExternAddrN(wr, 
	  Code.dataMachineArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataMachineClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataStoreCase =>
        PExternAddrN(wr, 
	  Code.dataStoreArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataStoreClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataOsCase =>
        PExternAddrN(wr, 
	  Code.dataOsArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataOsClass)], 
	  (*in-out*)PC);
      END;
    END;
    IF NOT PC=limit THEN Data.Fault("") END; (* -- *)
    <*ASSERT (PC=limit) *>
  END PExternCode;

(* === INTERN PORTABLE === *)

VAR 
  charReady: BOOLEAN; 
  lookAheadChar: CHAR;

PROCEDURE GetChar(rd: QOS.Reader): CHAR =
  BEGIN
    IF charReady THEN charReady := FALSE;
    ELSE EVAL QOS.GetChar(rd, (*out*) lookAheadChar);
    END;
    RETURN lookAheadChar;
  END GetChar;

PROCEDURE LookChar(rd: QOS.Reader): CHAR =
  BEGIN
    IF NOT charReady THEN
      EVAL QOS.GetChar(rd, (*out*) lookAheadChar);
      charReady := TRUE;
    END;
    RETURN lookAheadChar;
  END LookChar;

PROCEDURE HasChar(rd: QOS.Reader; ch: CHAR): BOOLEAN =
  BEGIN
    IF LookChar(rd) = ch THEN
      charReady := FALSE;
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END HasChar;

PROCEDURE GetPositiveVarHex(rd: QOS.Reader): INTEGER =
  VAR i: INTEGER; ch: CHAR;
  BEGIN
    i := 0;
    LOOP
      ch := GetChar(rd);
      IF ch = ' ' THEN EXIT END;
      i := i*16 + (ORD(ch)-ORD('@'));
    END;
    RETURN i;
  END GetPositiveVarHex;

PROCEDURE GetVarHex(rd: QOS.Reader): INTEGER =
  BEGIN
    IF HasChar(rd, '-') THEN 
      RETURN -GetPositiveVarHex(rd);
    ELSE
      RETURN GetPositiveVarHex(rd);
    END;
  END GetVarHex;

VAR segment0Index: INTEGER;

PROCEDURE InternPortable(rd: QOS.Reader; VAR (*out*) hp: Data.Pointer)
    : BOOLEAN =
  VAR saveTopSharedStack, scanSharedStack, size: Data.Int; seg: Data.Pointer;
  BEGIN
    charReady := FALSE;
    IF NOT HasChar(rd, Data.ExternPortableId1) THEN (*RETURN FALSE*)Data.Fault("") END;
    IF NOT HasChar(rd, Data.ExternPortableId2) THEN (*RETURN FALSE*)Data.Fault("") END;
    IF NOT HasChar(rd, Data.ExternPortableId3) THEN (*RETURN FALSE*)Data.Fault("") END;
    IF NOT HasChar(rd, VAL(Data.ExternPortableVersion, CHAR)) 
      THEN (*RETURN FALSE*)Data.Fault("") END;
    IF NOT HasChar(rd, '[') THEN (*RETURN FALSE*)Data.Fault("") END;
    saveTopSharedStack := topSharedStack;
    segment0Index := topSharedStack+1;
    WHILE NOT HasChar(rd, '|') DO
      size := GetVarHex(rd);
      IF size < 0 THEN (*RETURN FALSE*)Data.Fault("") END;
      Store.Align();
      Store.LayPointer(errorFormat); (* temporary *)
      Store.LayInt(size);
      seg := Store.hp;
      INC(Store.hp,size);
      INC(topSharedStack);
      sharedStack[topSharedStack].shared := seg;
    END;
    scanSharedStack := saveTopSharedStack;
    WHILE scanSharedStack < topSharedStack DO
      INC(scanSharedStack);
      seg := sharedStack[scanSharedStack].shared;
      IF NOT PInternSegment(rd, seg) THEN (*RETURN FALSE*)Data.Fault("") END;
    END;
    IF NOT HasChar(rd, ']') THEN (*RETURN FALSE*)Data.Fault("") END;
    IF NOT PInternPointer(rd, (*out*) hp) THEN (*RETURN FALSE*)Data.Fault("") END;
    IF NOT HasChar(rd, '.') THEN (*RETURN FALSE*)Data.Fault("") END;
    topSharedStack := saveTopSharedStack;
    RETURN TRUE;
  END InternPortable;

PROCEDURE PInternSegment(rd: QOS.Reader; seg: Data.Pointer): BOOLEAN =
  VAR size: INTEGER; scanFmt, scanHp, fmt: Data.Pointer;
  BEGIN
    IF HasChar(rd, 'F') THEN
      DataSetFormat(seg, formatFormat);
      scanFmt := FormatStart(seg);
      size := GetVarHex(rd); (* redundant *)
      RETURN PInternFmtRun(rd, (*in-out*) scanFmt);
    ELSIF HasChar(rd, 'D') THEN
      IF NOT PInternPointer(rd, (*out*) fmt) THEN (*RETURN FALSE*)Data.Fault("") END;
      IF DataGetFormat(fmt) = errorFormat THEN (*RETURN FALSE*)Data.Fault("") END;
      DataSetFormat(seg, fmt);
      size := GetVarHex(rd); (* redundant *)
      scanFmt := FormatStart(fmt);
      scanHp := seg;
      RETURN PInternDataRun(rd, (*in-out*) scanFmt, (*in-out*)scanHp);
    ELSE (*RETURN FALSE*)Data.Fault("");
    END;
  END PInternSegment;

PROCEDURE PInternPointer(rd: QOS.Reader; VAR (*out*) hp: Data.Pointer)
    : BOOLEAN =
  VAR i: INTEGER;
  BEGIN
    IF HasChar(rd, '-') THEN
      hp := Data.MinPointer;
      RETURN TRUE;
    ELSIF HasChar(rd, '^') THEN
      i := GetVarHex(rd);
      IF segment0Index + i > topSharedStack THEN (*RETURN FALSE*)Data.Fault("") END;
      hp := sharedStack[segment0Index + i].shared;
      RETURN TRUE;
    ELSIF HasChar(rd, '+') THEN
      IF LookChar(rd) = 'F' THEN
        RETURN PInternFmt(rd, (*out*) hp);
      ELSIF LookChar(rd) = 'D' THEN
	RETURN PInternData(rd, (*out*) hp);
      ELSE (*RETURN FALSE*)Data.Fault("");
      END;
    ELSE (*RETURN FALSE*)Data.Fault("");
    END;
  END PInternPointer;

PROCEDURE PInternFmt(rd: QOS.Reader; VAR (*out*) fmt: Data.Pointer)
    : BOOLEAN =
  VAR size: INTEGER; scanFmt: Data.Pointer;
  BEGIN
    IF NOT HasChar(rd, 'F') THEN (*RETURN FALSE*)Data.Fault("") END;
    size := GetVarHex(rd);
    IF size < 0 THEN (*RETURN FALSE*)Data.Fault("") END;
    Store.Align();
    Store.LayPointer(formatFormat);
    Store.LayInt(size);
    fmt := Store.hp;
    INC(Store.hp,size);
    scanFmt := FormatStart(fmt);
    RETURN PInternFmtRun(rd, (*in-out*)scanFmt);
  END PInternFmt;

PROCEDURE PInternFmtRun(rd: QOS.Reader; VAR (*in-out*) fmt: Data.Pointer)
    : BOOLEAN =
  VAR count, countPointees: INTEGER; ch: CHAR;
  BEGIN
    ch := GetChar(rd);
    IF ch = 'n' THEN
      Store.SetSmallInt(fmt, ORD(Class.NoneCase));
      INC(fmt, Data.PointeesPerSmallInt);
    ELSIF ch = 'e' THEN
      Store.SetSmallInt(fmt, ORD(Class.EmptyCase));
      INC(fmt, Data.PointeesPerSmallInt);
    ELSIF ch = 'b' THEN
      Store.SetSmallInt(fmt, ORD(Class.PointeeCase));
      INC(fmt, Data.PointeesPerSmallInt);
    ELSIF ch = 's' THEN
      Store.SetSmallInt(fmt, ORD(Class.SmallIntCase));
      INC(fmt, Data.PointeesPerSmallInt);
    ELSIF ch = 'i' THEN
      Store.SetSmallInt(fmt, ORD(Class.IntCase));
      INC(fmt, Data.PointeesPerSmallInt);
    ELSIF ch = 'k' THEN
      Store.SetSmallInt(fmt, ORD(Class.ImmediateCase));
      INC(fmt, Data.PointeesPerSmallInt);
    ELSIF ch = 'p' THEN
      Store.SetSmallInt(fmt, ORD(Class.PointerCase));
      INC(fmt, Data.PointeesPerSmallInt);
    ELSIF ch = 'm' THEN
      Store.SetSmallInt(fmt, ORD(Class.PolymorphCase));
      INC(fmt, Data.PointeesPerSmallInt);
    ELSIF ch = 'S' THEN
      Store.SetSmallInt(fmt, ORD(Class.SeqCase));
      INC(fmt, Data.PointeesPerSmallInt);
      count := GetVarHex(rd);
      Store.SetSmallInt(fmt, count);
      INC(fmt, Data.PointeesPerSmallInt);
      FOR i := 1 TO count DO
	IF NOT PInternFmtRun(rd, (*in-out*)fmt) THEN (*RETURN FALSE*)Data.Fault("") END;
      END;
    ELSIF ch = 'A' THEN
      Store.SetSmallInt(fmt, ORD(Class.AltCase));
      INC(fmt, Data.PointeesPerSmallInt);
      Data.Fault("PInternFmtRun AltCase");
    ELSIF ch = 'I' THEN
      Store.SetSmallInt(fmt, ORD(Class.IterCase));
      INC(fmt, Data.PointeesPerSmallInt);
      countPointees := GetVarHex(rd);
      Store.SetSmallInt(fmt, countPointees);
      INC(fmt, Data.PointeesPerSmallInt);
      IF NOT PInternFmtRun(rd, (*in-out*)fmt) THEN (*RETURN FALSE*)Data.Fault("") END;
    ELSIF ch = 'E' THEN
      Store.SetSmallInt(fmt, ORD(Class.AreaCase));
      INC(fmt, Data.PointeesPerSmallInt);
      Data.Fault("PInternFmtRun AreaCase");
    ELSIF ch = 'R' THEN
      Store.SetSmallInt(fmt, ORD(Class.RepeatCase));
      INC(fmt, Data.PointeesPerSmallInt);
      count := GetVarHex(rd);
      Store.SetInt(fmt, count);
      INC(fmt, Data.PointeesPerInt);
      IF NOT PInternFmtRun(rd, (*in-out*)fmt) THEN (*RETURN FALSE*)Data.Fault("") END;
    ELSIF ch = 'U' THEN
      Store.SetSmallInt(fmt, ORD(Class.RunCase));
      INC(fmt, Data.PointeesPerSmallInt);
      Data.Fault("PInternFmtRun RunCase");
    ELSE (*RETURN FALSE*)Data.Fault("");
    END;
    RETURN TRUE;
  END PInternFmtRun;

PROCEDURE PInternData(rd: QOS.Reader; VAR (*out*) hp: Data.Pointer)
      : BOOLEAN =
  VAR fmt, scanFmt, scanHp: Data.Pointer; size:  INTEGER;
  BEGIN
    IF NOT HasChar(rd, 'D') THEN (*RETURN FALSE*)Data.Fault("") END;
    IF NOT PInternPointer(rd, (*out*) fmt) THEN (*RETURN FALSE*)Data.Fault("") END;
    IF fmt = Data.MinPointer THEN fmt := formatFormat END;
    size := GetVarHex(rd);
    IF size < 0 THEN (*RETURN FALSE*)Data.Fault("") END;
    Store.Align();
    Store.LayPointer(fmt);
    Store.LayInt(size);
    hp := Store.hp;
    INC(Store.hp,size);
    scanFmt := FormatStart(fmt);
    scanHp := hp;
    RETURN PInternDataRun(rd, (*in-out*) scanFmt, (*in-out*) scanHp);
  END PInternData;

PROCEDURE PInternDataRun(rd: QOS.Reader; VAR (*in-out*) fmt, hp: Data.Pointer)
      : BOOLEAN =
  VAR class, count, countPointees: INTEGER; ptr, fmtSave: Data.Pointer;
  BEGIN
    IF IsProgFormat(fmt) THEN 
      RETURN PInternProg(rd, (*in-out*)fmt, (*in-out*)hp);
    END;
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    CASE VAL(class, Class) OF
    | Class.NoneCase => Data.Fault("PInternData None");
    | Class.EmptyCase =>
    | Class.PointeeCase =>
        Store.SetPointee(hp, VAL(GetVarHex(rd), Data.Pointee));
        INC(hp, Data.PointeesPerPointee);
    | Class.SmallIntCase =>
        Store.SetSmallInt(hp, GetVarHex(rd));
        INC(hp, Data.PointeesPerSmallInt);
    | Class.IntCase => 
        Store.SetInt(hp, GetVarHex(rd));
	INC(hp, Data.PointeesPerInt);
    | Class.ImmediateCase =>
        Store.SetImmediate(hp, Data.ImmediateOfInt(GetVarHex(rd)));
        INC(hp, Data.PointeesPerImmediate);
    | Class.PointerCase =>
	EVAL PInternPointer(rd, (*out*) ptr);
        Store.SetPointer(hp, ptr);
        INC(hp, Data.PointeesPerPointer);
    | Class.PolymorphCase =>
	IF HasChar(rd, 'k') THEN
          Store.SetImmediate(hp, Data.ImmediateOfInt(GetVarHex(rd)));
          INC(hp, Data.PointeesPerImmediate);
	ELSIF HasChar(rd, 'p') THEN
	  EVAL PInternPointer(rd, (*out*) ptr);
          Store.SetPointer(hp, ptr);
          INC(hp, Data.PointeesPerPointer);
	ELSE (*RETURN FALSE*)Data.Fault("");
	END;
    | Class.SeqCase =>
        count := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
        FOR i := 0 TO count - 1 DO
          IF NOT PInternDataRun(rd, (* in-out *) fmt, (* in-out *) hp) THEN
            (*RETURN FALSE*)Data.Fault("")
          END;
        END;
    | Class.AltCase => Data.Fault("PInternDataRun AltCase");
    | Class.IterCase =>
        countPointees := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
	count := GetVarHex(rd);
        IF countPointees = Data.PointeesPerSmallInt THEN
	  Store.SetSmallInt(hp, count);
          INC(hp, Data.PointeesPerSmallInt);
        ELSIF countPointees = Data.PointeesPerInt THEN
	  Store.SetInt(hp, count);
          INC(hp, Data.PointeesPerInt);
        ELSE
          Data.Fault("");
        END;
        fmtSave := fmt;
        FOR i := 0 TO count - 1 DO
          fmt := fmtSave;
          IF NOT PInternDataRun(rd, (* in-out *) fmt, (* in-out *) hp) THEN
            (*RETURN FALSE*)Data.Fault("")
          END;
        END;
    | Class.AreaCase => Data.Fault("PInternDataRun Area");
    | Class.RepeatCase =>
        count := Store.GetInt(fmt);
        INC(fmt, Data.PointeesPerInt);
        fmtSave := fmt;
        FOR i := 0 TO count - 1 DO
          fmt := fmtSave;
          IF NOT PInternDataRun(rd, (* in-out *) fmt, (* in-out *) hp) THEN
            (*RETURN FALSE*)Data.Fault("")
          END;
        END;
    | Class.RunCase => Data.Fault("PInternDataRun Run");
    END;
    RETURN TRUE;
  END PInternDataRun;

PROCEDURE PInternProg(rd: QOS.Reader; VAR (*in-out*) fmt, hp: Data.Pointer)
      : BOOLEAN =
  VAR
    class: Data.SmallInt;
    count, countPointees: Data.Int;
    literals: Data.Pointer;
  BEGIN
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    <*ASSERT (VAL(class, Class) = Class.SeqCase) *>
    count := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    <*ASSERT (count = 2) *>
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    <*ASSERT (VAL(class, Class) = Class.PointerCase) *>
    IF NOT PInternPointer(rd, (* out *) literals) THEN (*RETURN FALSE*)Data.Fault("") END;
    Store.SetPointer(hp, literals);
    INC(hp, Data.PointeesPerPointer);
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    <*ASSERT (VAL(class, Class) = Class.IterCase) *>
    countPointees := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    <*ASSERT (countPointees = Data.PointeesPerInt) *>
    count := GetVarHex(rd);
    Store.SetInt(hp, count);
    INC(hp, Data.PointeesPerInt);    
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    <*ASSERT (VAL(class, Class) = Class.PointeeCase) *>
    IF NOT PInternCode(rd, count, hp) THEN (*RETURN FALSE*)Data.Fault("") END;
    INC(hp, count);
    RETURN TRUE;
  END PInternProg;

PROCEDURE PInternSmallInt(rd: QOS.Reader; VAR (*in-out*) PC: Data.Pointer) =
  VAR i: Data.Int;
  BEGIN
    i := GetVarHex(rd);
    Store.SetSmallInt(PC, i);
    INC(PC, Data.PointeesPerSmallInt);
  END PInternSmallInt;

PROCEDURE PInternAddrN(rd: QOS.Reader; i: Data.Int; 
    VAR (*in-out*) PC: Data.Pointer) =
  BEGIN
    WHILE i > 0 DO
      PInternAddr(rd, (*in-out*) PC);
      DEC(i);
    END;
  END PInternAddrN;

PROCEDURE PInternTarget(rd: QOS.Reader; VAR (*in-out*) PC: Data.Pointer) =
  VAR rel: Data.Int;
  BEGIN
    rel := GetVarHex(rd);
    Store.SetSmallInt(PC, rel);
    INC(PC, Data.PointeesPerSmallInt);
  END PInternTarget;

PROCEDURE PInternTargetList(rd: QOS.Reader; VAR (*in-out*) PC: Data.Pointer) =
  VAR size: Data.Int;
  BEGIN
    size := GetVarHex(rd);
    Store.SetSmallInt(PC, size);
    INC(PC, Data.PointeesPerSmallInt);
    WHILE size > 0 DO
      PInternTarget(rd, (*in-out*) PC);
      DEC(size);
    END;
  END PInternTargetList;

PROCEDURE PInternAddrList(rd: QOS.Reader; VAR (*in-out*) PC: Data.Pointer) =
  VAR size: Data.Int;
  BEGIN
    size := GetVarHex(rd);
    Store.SetSmallInt(PC, size);
    INC(PC, Data.PointeesPerSmallInt);
    WHILE size > 0 DO
      PInternAddr(rd, (*in-out*) PC);
      DEC(size);
    END;
  END PInternAddrList;

PROCEDURE PInternAddr(rd: QOS.Reader; VAR (*in-out*) PC: Data.Pointer) =
  VAR opAddrClass, int: Data.Int; float: Data.Float;
  BEGIN
    opAddrClass := GetVarHex(rd);
    Store.SetSmallCard(PC, opAddrClass);
    INC(PC, Data.PointeesPerSmallCard);
    CASE VAL(opAddrClass DIV Data.PointeeSize, Code.OpAddrClass) OF
    | Code.OpAddrClass.OpAddrImmedOkCase =>
    | Code.OpAddrClass.OpAddrImmedBoolCase =>
    | Code.OpAddrClass.OpAddrImmedCharCase =>
    | Code.OpAddrClass.OpAddrImmedIntSmallCase =>
	int := GetVarHex(rd);
        Store.SetSmallInt(PC, int);
        INC(PC, Data.PointeesPerSmallInt);
    | Code.OpAddrClass.OpAddrImmedIntFullCase =>
	int := GetVarHex(rd);
        Store.SetInt(PC, int);
        INC(PC, Data.PointeesPerInt);
    | Code.OpAddrClass.OpAddrImmedRealCase =>
	float := Data.FloatOfImmediate(GetVarHex(rd));
        Store.SetFloat(PC, float);
        INC(PC, Data.PointeesPerFloat);
    | Code.OpAddrClass.OpAddrLiteralSmallCase =>
	int := GetVarHex(rd);
        Store.SetSmallInt(PC, int);
        INC(PC, Data.PointeesPerSmallInt);
    | Code.OpAddrClass.OpAddrFrameSmallCase =>
	int := GetVarHex(rd);
        Store.SetSmallInt(PC, int);
        INC(PC, Data.PointeesPerSmallInt);
    | Code.OpAddrClass.OpAddrFrameLocativeSmallCase =>
	int := GetVarHex(rd);
        Store.SetSmallInt(PC, int);
        INC(PC, Data.PointeesPerSmallInt);
    | Code.OpAddrClass.OpAddrGlobalSmallCase =>
	int := GetVarHex(rd);
        Store.SetSmallInt(PC, int);
        INC(PC, Data.PointeesPerSmallInt);
    | Code.OpAddrClass.OpAddrIndexedSmallCase =>
	PInternAddr(rd, (*in-out*)PC);
	int := GetVarHex(rd);
        Store.SetSmallInt(PC, int);
        INC(PC, Data.PointeesPerSmallInt);
    | Code.OpAddrClass.OpAddrStackDisplCase =>
	PInternAddr(rd, (*in-out*)PC);
    | Code.OpAddrClass.OpAddrSpecialCase =>
    END;
  END PInternAddr;

PROCEDURE PInternCode(rd: QOS.Reader; count: Data.Int; PC: Data.Pointer)
      : BOOLEAN =
  VAR limit: Data.Pointer; opClass: Data.Int;
  BEGIN
    limit := PC + count;
    WHILE PC < limit DO
      opClass := GetVarHex(rd);
      Store.SetSmallCard(PC, opClass);
      INC(PC, Data.PointeesPerSmallCard);
      CASE VAL(opClass DIV Data.PointeeSize, Code.OpClass) OF
      | Code.OpClass.OpMoveCase =>
        PInternAddrN(rd, 2, (*in-out*)PC);
      | Code.OpClass.OpFrameCase =>
	PInternSmallInt(rd, (*in-out*)PC);
	PInternSmallInt(rd, (*in-out*)PC);
      | Code.OpClass.OpApplyCase =>
        PInternAddr(rd, (*in-out*)PC);
      | Code.OpClass.OpReturnCase =>
        PInternSmallInt(rd, (*in-out*)PC);
      | Code.OpClass.OpReturnObsoleteCase =>
        PInternSmallInt(rd, (*in-out*)PC);
      | Code.OpClass.OpArgumentsCase =>
	PInternSmallInt(rd, (*in-out*)PC);
        PInternAddrList(rd, (*in-out*)PC);
      | Code.OpClass.OpApplyArguments0Results1Case =>
	PInternAddr(rd, (*in-out*)PC);
      | Code.OpClass.OpApplyArguments1Results1Case =>
	PInternAddr(rd, (*in-out*)PC);
        PInternAddr(rd, (*in-out*)PC);
      | Code.OpClass.OpApplyArguments2Results1Case =>
	PInternAddr(rd, (*in-out*)PC);
        PInternAddr(rd, (*in-out*)PC);
        PInternAddr(rd, (*in-out*)PC);
      | Code.OpClass.OpApplyArgumentsResults1Case =>
	PInternAddr(rd, (*in-out*)PC);
        PInternAddrList(rd, (*in-out*)PC);
      | Code.OpClass.OpResultsCase =>
        PInternAddrList(rd, (*in-out*)PC);
      | Code.OpClass.OpResults1Case =>
        PInternAddr(rd, (*in-out*)PC);
      | Code.OpClass.OpClosureCase =>
        PInternAddr(rd, (*in-out*)PC);
        PInternAddrList(rd, (*in-out*)PC);
        PInternAddr(rd, (*in-out*)PC);
      | Code.OpClass.OpDumClosureCase =>
        PInternAddrN(rd, 3, (*in-out*)PC);
      | Code.OpClass.OpRecClosureCase =>
        PInternAddr(rd, (*in-out*)PC);
        PInternAddrList(rd, (*in-out*)PC);
      | Code.OpClass.OpTupleCase =>
        PInternAddrList(rd, (*in-out*)PC);
        PInternAddr(rd, (*in-out*)PC);
      | Code.OpClass.OpJumpCase =>
        PInternTarget(rd, (*in-out*)PC);
      | Code.OpClass.OpJumpWhenCase =>
        PInternAddrN(rd, 2, (*in-out*)PC);
        PInternTarget(rd, (*in-out*)PC);
      | Code.OpClass.OpCaseCheckCase =>
        PInternAddrN(rd, 2, (*in-out*)PC);
      | Code.OpClass.OpCaseCase =>
        PInternAddr(rd, (*in-out*)PC);
        PInternTargetList(rd, (*in-out*)PC);
      | Code.OpClass.OpCaseFaultCase =>
      | Code.OpClass.OpArrayCase =>
        PInternAddrList(rd, (*in-out*)PC);
        PInternAddr(rd, (*in-out*)PC);
      | Code.OpClass.OpTrapCase =>
        PInternAddr(rd, (*in-out*)PC);
        PInternTarget(rd, (*in-out*)PC);
      | Code.OpClass.OpUntrapCase =>
      | Code.OpClass.OpRaiseCase =>
        PInternAddrN(rd, 2, (*in-out*)PC);
      | Code.OpClass.OpUnwindCase =>
      | Code.OpClass.OpCrashCase =>
      | Code.OpClass.OpEndCase =>
      | Code.OpClass.OpStartCase =>
        PInternAddrN(rd, 2, (*in-out*)PC);
      | Code.OpClass.OpStopCase =>
      | Code.OpClass.OpNoOp =>
      | Code.OpClass.OpDataCompareCase =>
        PInternAddrN(rd, 
	  Code.dataCompareArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataCompareClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataBoolCase =>
        PInternAddrN(rd, 
	  Code.dataBoolArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataBoolClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataAsciiCase =>
        PInternAddrN(rd, 
	  Code.dataAsciiArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataAsciiClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataIntCase =>
        PInternAddrN(rd, 
	  Code.dataIntArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataIntClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataRealCase =>
        PInternAddrN(rd, 
	  Code.dataRealArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataRealClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataStringCase =>
        PInternAddrN(rd, 
	  Code.dataStringArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataStringClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataArrayCase =>
        PInternAddrN(rd, 
	  Code.dataArrayArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataArrayClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataReaderCase =>
        PInternAddrN(rd, 
	  Code.dataReaderArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataReaderClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataWriterCase =>
        PInternAddrN(rd, 
	  Code.dataWriterArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataWriterClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataDynamicCase =>
        PInternAddrN(rd, 
	  Code.dataDynamicArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataDynamicClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataTimeCase =>
        PInternAddrN(rd, 
	  Code.dataTimeArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataTimeClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataValueCase =>
        PInternAddrN(rd, 
	  Code.dataValueArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataValueClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataMachineCase =>
        PInternAddrN(rd, 
	  Code.dataMachineArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataMachineClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataStoreCase =>
        PInternAddrN(rd, 
	  Code.dataStoreArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataStoreClass)], 
	  (*in-out*)PC);
      | Code.OpClass.OpDataOsCase =>
        PInternAddrN(rd, 
	  Code.dataOsArity[VAL(opClass MOD Data.PointeeSize, 
	    Code.OpDataOsClass)], 
	  (*in-out*)PC);
      END;
    END;
    IF NOT PC=limit THEN Data.Fault("") END; (* -- *)
    <*ASSERT (PC=limit) *>
    RETURN TRUE;
  END PInternCode;

(* === PRINT === *)

(* 

VAR
  currBulk, currDepth, currWidth, maxWidth: INTEGER;

PROCEDURE WalkHeaderPrint(hp: Data.Pointer): BOOLEAN =
  BEGIN
    IF (hp < Store.MinIndex) OR (hp > Store.MaxIndex) THEN
      QOS.OutString(QOS.stdout, "<Bogus pointer ");
      QOS.OutInt(QOS.stdout, Data.IntOfImmediate(hp));
      QOS.OutString(QOS.stdout, ">");
      RETURN FALSE;
    ELSIF currBulk < 0 THEN
      RETURN FALSE;
    ELSIF currDepth <= 0 THEN
      QOS.OutString(QOS.stdout, "... ");
      RETURN FALSE;
    ELSE
      RETURN TRUE;
    END;
  END WalkHeaderPrint;

PROCEDURE WalkRunPrint() =
  BEGIN
    QOS.OutChar(QOS.stdout, '(');
    currDepth := currDepth - 1;
    currWidth := maxWidth;
  END WalkRunPrint;

PROCEDURE WalkPointeePrint(pointee: Data.Pointee) =
  BEGIN
    currBulk := currBulk - 1;
    IF currBulk = 0 THEN QOS.OutString(QOS.stdout, "etc. "); RETURN END;
    IF currBulk < 0 THEN RETURN END;
    currWidth := currWidth - 1;
    IF currWidth = 0 THEN QOS.OutString(QOS.stdout, "... "); RETURN END;
    IF currWidth < 0 THEN RETURN END;
    IF (pointee >= Data.Pointee(' ')) AND (pointee <= Data.Pointee('~')) THEN
      QOS.OutChar(QOS.stdout, '\'');
      QOS.OutChar(QOS.stdout, CHAR(pointee));
    ELSE
      QOS.OutChar(QOS.stdout, '\\');
      QOS.OutInt(QOS.stdout, ORD(pointee));
    END;
    QOS.OutChar(QOS.stdout, ' ');
  END WalkPointeePrint;

PROCEDURE WalkPointerPrint() =
  BEGIN
    currDepth := currDepth + 1;
    currWidth := maxWidth;
    IF currBulk < 0 THEN RETURN END;
    QOS.OutString(QOS.stdout, ") ");
  END WalkPointerPrint;

PROCEDURE WalkImmediatePrint(immediate: Data.Immediate) =
  BEGIN
    currBulk := currBulk - 1;
    IF currBulk = 0 THEN QOS.OutString(QOS.stdout, "etc. "); RETURN END;
    IF currBulk < 0 THEN RETURN END;
    currWidth := currWidth - 1;
    IF currWidth = 0 THEN QOS.OutString(QOS.stdout, "... "); RETURN END;
    IF currWidth < 0 THEN RETURN END;
    QOS.OutInt(QOS.stdout, Data.IntOfImmediate(immediate));
    QOS.OutChar(QOS.stdout, ' ');
  END WalkImmediatePrint;

PROCEDURE WalkSmallIntPrint(smallInt: Data.SmallInt) =
  BEGIN
    currBulk := currBulk - 1;
    IF currBulk = 0 THEN QOS.OutString(QOS.stdout, "etc. "); RETURN END;
    IF currBulk < 0 THEN RETURN END;
    currWidth := currWidth - 1;
    IF currWidth = 0 THEN QOS.OutString(QOS.stdout, "... "); RETURN END;
    IF currWidth < 0 THEN RETURN END;
    QOS.OutInt(QOS.stdout, smallInt);
    QOS.OutChar(QOS.stdout, ' ');
  END WalkSmallIntPrint;

PROCEDURE WalkIntPrint(int: Data.Int) =
  BEGIN
    currBulk := currBulk - 1;
    IF currBulk = 0 THEN QOS.OutString(QOS.stdout, "etc. "); RETURN END;
    IF currBulk < 0 THEN RETURN END;
    currWidth := currWidth - 1;
    IF currWidth = 0 THEN QOS.OutString(QOS.stdout, "... "); RETURN END;
    IF currWidth < 0 THEN RETURN END;
    QOS.OutInt(QOS.stdout, int);
    QOS.OutChar(QOS.stdout, ' ');
  END WalkIntPrint;

PROCEDURE PrintRun(hp: Data.Pointer; printDepth: INTEGER) =
  CONST printWidth = 16; printBulk = 64; (* -- make parameter *)
  BEGIN
    currBulk := printBulk;
    currDepth := printDepth;
    maxWidth := printWidth;
    currWidth := maxWidth;
    WalkPrint(hp);
    IF currBulk < 0 THEN RETURN END;
    QOS.OutChar(QOS.stdout, ')');
  END PrintRun;

PROCEDURE WalkPrint(hp: Data.Pointer) =
  VAR fmt, fmtScan, hpScan: Data.Pointer;
  BEGIN
    <* ASSERT NOT( (hp < Store.MinIndex) OR (hp > Store.MaxIndex) ) *>
    IF hp = Data.MinPointer THEN RETURN END;
    IF WalkHeaderPrint(hp) THEN
      fmt := DataGetFormat(hp);
      IF fmt # formatFormat THEN
        WalkRunPrint();
        fmtScan := FormatStart(fmt);
        hpScan := hp;
        WalkPrintData( (* in-out *) fmtScan, (* in-out *) hpScan);
      END;
    END;
  END WalkPrint;

PROCEDURE WalkPrintData(VAR (* in-out *) fmt, hp: Data.Pointer) =
  VAR
    class: Data.SmallInt;
    count, i: Data.Int;
    countPointees: Data.Int;
    fmtSave, pointer: Data.Pointer;
    polymorph: Data.Polymorph;
  BEGIN
    <*ASSERT NOT( ((fmt < Store.MinIndex) OR (fmt > Store.MaxIndex))
       OR ((hp < Store.MinIndex) OR (hp > Store.MaxIndex)) ) *>
    class := Store.GetSmallInt(fmt);
    INC(fmt, Data.PointeesPerSmallInt);
    CASE VAL(Class, class) OF
    | Class.NoneCase => Data.Fault("WalkData None");
    | Class.EmptyCase =>
    | Class.PointeeCase =>
        WalkPointeePrint(Store.GetPointee(hp));
        INC(hp, Data.PointeesPerPointee);
    | Class.SmallIntCase =>
        WalkSmallIntPrint(Store.GetSmallInt(hp));
        INC(hp, Data.PointeesPerSmallInt);
    | Class.IntCase => WalkIntPrint(Store.GetInt(hp)); 
	INC(hp, Data.PointeesPerInt);
    | Class.ImmediateCase =>
        WalkImmediatePrint(Store.GetImmediate(hp));
        INC(hp, Data.PointeesPerImmediate);
    | Class.PointerCase =>
        pointer := Store.GetPointer(hp);
        WalkPrint(pointer);
        WalkPointerPrint();
        INC(hp, Data.PointeesPerPointer);
    | Class.PolymorphCase =>
        polymorph := Store.GetPolymorph(hp);
        IF Data.IsImmediate(polymorph) THEN
          WalkImmediatePrint(polymorph);
        ELSE
          WalkPrint(polymorph);
          WalkPointerPrint();
        END;
        INC(hp, Data.PointeesPerPolymorph);
    | Class.SeqCase =>
        count := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
        FOR i := 0 TO count - 1 DO
          WalkPrintData( (* in-out *) fmt, (* in-out *) hp);
        END;
    | Class.AltCase => Data.Fault("Walk: Unimplemented AltCase");
    | Class.IterCase =>
        countPointees := Store.GetSmallInt(fmt);
        INC(fmt, Data.PointeesPerSmallInt);
        IF countPointees = Data.PointeesPerSmallInt THEN
          count := Store.GetSmallInt(hp);
          WalkSmallIntPrint(count);
          INC(hp, Data.PointeesPerSmallInt);
        ELSIF countPointees = Data.PointeesPerInt THEN
          count := Store.GetInt(hp);
          WalkIntPrint(count);
          INC(hp, Data.PointeesPerInt);
        ELSE
          Data.Fault("");
        END;
        fmtSave := fmt;
        FOR i := 0 TO count - 1 DO
          fmt := fmtSave;
          WalkPrintData( (* in-out *) fmt,
            (* in-out *) hp);
        END;
    | Class.AreaCase => Data.Fault("Walk Area");
    | Class.RepeatCase =>
        count := Store.GetInt(fmt);
        INC(fmt, Data.PointeesPerInt);
        fmtSave := fmt;
        FOR i := 0 TO count - 1 DO
          fmt := fmtSave;
          WalkPrintData(
            (* in-out *) fmt, (* in-out *) hp);
        END;
    | Class.RunCase => Data.Fault("Walk Run");
    END;
  END WalkPrintData;

*)

BEGIN
END Format.
