(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)

INTERFACE QOS;

(* Operating system requirements of the Quest implementation. Many routines
   return FALSE for failure. *)

(* Assumes CHAR = [CHR(0)..CHR(255)] *)

TYPE
  Reader <: ADDRESS;
  (* A source of characters. *)
  Writer <: ADDRESS;
  (* A sink of characters. *)

VAR
  stdin: Reader;
  (* A standard input reader (e.g. terminal). *)
  stdout: Writer;
  (* A standard output reader (e.g. terminal). *)
  newLine: CHAR;
  (* The newline character. *)
  dirSeparator: CHAR;
  (* The character separating directories in a file name. *)
  pathSeparator: CHAR;
  (* The character separating file names in a path. *)

PROCEDURE Setup();
(* To be called before any other use of this module. *)

PROCEDURE Alloc(size: INTEGER): ADDRESS;
(* Allocate a chunk of memory of given size. *)

PROCEDURE TimeNow(VAR (*out*) seconds, microseconds: INTEGER);
(* The current absolute time. *)

PROCEDURE Id(VAR (*out*) cpu, process: INTEGER);
(* CPU and process id's, for building unique identifiers. *)

PROCEDURE ChDir(
    name: ADDRESS;
    start, length: INTEGER)
    : BOOLEAN;
(* Change current work directory. *)

PROCEDURE OpenRead(
    fileName: ADDRESS;
    start, length: INTEGER;
    VAR (*out*) rd: Reader)
    : BOOLEAN;
(* Obtain a reader out of a file name. *)

PROCEDURE OpenWrite(
    fileName: ADDRESS;
    start, length: INTEGER;
    VAR (*out*) wr: Writer)
    : BOOLEAN;
(* Obtain a writer out of a file name. *)

PROCEDURE FileExists(
    fileName: ADDRESS;
    start, length: INTEGER)
    : BOOLEAN;
(* Test existence of a file. *)

PROCEDURE FindAlongPath(
    path: ADDRESS;
    pathStart, pathLength: INTEGER;
    dirSeparator, pathSeparator: CHAR;
    fileName: ADDRESS;
    fileNameStart, fileNameLength: INTEGER;
    VAR (*out*) dirNameStart, dirNameLength: INTEGER)
    : BOOLEAN;
(* Try to find the first directory in the path containing a file called
   fileName, and return the directory name as it appears in the path. Return
   false if not found. If path is empty, fileName alone is considered ] (it
   can be an absolute name, or a name relative to the current directory). If
   path is a sequence of directory names (separated by the pathSeparator
   character) each directory in turn is concatenated with the dirSeparator and
   the fileName, and the resulting file name is tested for existence. The
   output parameters are indeces into path. *)

PROCEDURE Lookup(
    inBuff: ADDRESS;
    inStart, inLength: INTEGER;
    (*out*) outBuff: ADDRESS;
    outStart, outMaxLength: INTEGER;
    VAR (*out*) outLength: INTEGER)
    : BOOLEAN;
(* If in = "$d" where d is a digit, the effective out string is taken from the
   program arguments. If in = "$name", the value of the environment variable
   name is used. Otherwise out is a copy of in. Return false if the lookup
   fails for any reason, or if outMaxLength is too small to contain the
   result. *)

PROCEDURE OutChar(wr: Writer; ch: CHAR);
(* Light-duty I/O: output a character. *)

PROCEDURE OutString(wr: Writer; string: TEXT);
(* Light-duty I/O: output a string. *)

PROCEDURE OutInt(wr: Writer; int: INTEGER);
(* Light-duty I/O: output a string representation of an integer. *)

PROCEDURE GetChar(rd: Reader; VAR (*out*) ch: CHAR): BOOLEAN;
(* Get a character from a reader (blocks). *)

PROCEDURE GetSub(
    rd: Reader;
    (*out*) string: ADDRESS;
    start, length: INTEGER;
    VAR (*out*) read: INTEGER)
    : BOOLEAN;
(* Get a string of given lenght from a reader (blocks). *)

PROCEDURE ReaderClose(rd: Reader): BOOLEAN;
(* Close a reader. Operations on it will then fail. *)

PROCEDURE ReaderMore(rd: Reader; VAR (*out*) more: BOOLEAN): BOOLEAN;
(* Test whether there are more character on a reader (blocks). *)

PROCEDURE ReaderReady(rd: Reader; VAR (*out*) ready: INTEGER): BOOLEAN;
(* Check how many characters can be read right now (does not block). *)

PROCEDURE PutChar(wr: Writer; ch: CHAR): BOOLEAN;
(* Put a character to a writer. *)

PROCEDURE PutSub(
    wr: Writer;
    string: ADDRESS;
    start, length: INTEGER)
    : BOOLEAN;
(* Put a string to a writer. *)

PROCEDURE WriterClose(wr: Writer): BOOLEAN;
(* Close a writer. Operations on it will then fail. *)

PROCEDURE WriterFlush(wr: Writer): BOOLEAN;
(* Flush buffered characters to their final destination. *)

END QOS.
