(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)

UNSAFE MODULE QOS;
IMPORT Unix, Uuio, Utime, Ctypes, Text, Time, Params, Env;

TYPE
  UnixFileDesc = Ctypes.int;
  UnixTimeVal = Utime.struct_timeval;

REVEAL 
  Reader = UNTRACED BRANDED REF ReaderBase;
  Writer = UNTRACED BRANDED REF WriterBase;

TYPE
  ReaderBase =
    RECORD
      file: UnixFileDesc;
      lookAheadReady: BOOLEAN;
      lookAheadByte: CHAR;
    END;
    
  WriterBase =
    RECORD
      file: UnixFileDesc;
    END;
    
  CharPtr = UNTRACED REF CHAR;

  CBuff = ARRAY [0..255] OF CHAR;
    
PROCEDURE Alloc(size: INTEGER): ADDRESS =
  TYPE StorePtr = UNTRACED REF ARRAY OF CHAR;
  VAR storePtr: StorePtr;
  BEGIN
    storePtr := NEW(StorePtr, size);
    RETURN ADR(storePtr[0]);
  END Alloc;

PROCEDURE TimeNow(VAR (*out*) seconds, microseconds: INTEGER) =
  VAR time: Time.T;
  BEGIN
    time := Time.Now();
    seconds := TRUNC(time);
    microseconds := TRUNC((time-FLOAT(seconds, LONGREAL))*1000000.0d0);
  END TimeNow;

PROCEDURE Id(VAR (*out*) cpu, process: INTEGER) =
  BEGIN
    (* -- *)
    cpu := 0;
    process := 0;
  END Id;

PROCEDURE CString(
    string: ADDRESS; start, length: INTEGER;
    VAR (*out*) cString: CBuff) =
  VAR i: INTEGER; charPtr: CharPtr; scanString: ADDRESS;
  BEGIN
    scanString := string;
    INC(scanString, start);
    i := 0;
    LOOP
      IF i >= length THEN EXIT END;
      charPtr := LOOPHOLE(scanString, CharPtr);
      cString[i] := charPtr^;
      INC(scanString);
      INC(i);
    END;
    cString[i] := '\000';
  END CString;

PROCEDURE ChDir(
    name: ADDRESS;
    start, length: INTEGER)
    : BOOLEAN =
  VAR cString: CBuff;
  BEGIN
    CString(name, start, length, (*out*)cString);
    RETURN Unix.chdir(ADR(cString[0])) = 0;
  END ChDir;

PROCEDURE OpenRead(
    fileName: ADDRESS;
    start, length: INTEGER;
    VAR (*out*) rd: Reader)
    : BOOLEAN =
  VAR cString: CBuff; file :UnixFileDesc;
  BEGIN
    CString(fileName, start, length, (*out*)cString);
    file :=
      Unix.open(ADR(cString), Unix.O_RDONLY, 0);
    IF file < 0 THEN RETURN FALSE
    ELSE
      rd := NEW(Reader);
      rd^.file := file;
      rd^.lookAheadReady := FALSE;
      RETURN TRUE;
    END;
  END OpenRead;

PROCEDURE OpenWrite(
    fileName: ADDRESS;
    start, length: INTEGER;
    VAR (*out*) wr: Writer)
    : BOOLEAN =
  VAR cString: CBuff; file: UnixFileDesc;
  BEGIN
    CString(fileName, start, length, (*out*)cString);
    file :=
      Unix.open(ADR(cString), Unix.O_CREAT+Unix.O_TRUNC+Unix.O_WRONLY,
	Unix.MROTHER+Unix.MRGROUP+Unix.MROWNER+Unix.MWOWNER);
    IF file < 0 THEN RETURN FALSE;
    ELSE
      wr := NEW(Writer);
      wr^.file := file;
      RETURN TRUE;
    END;
  END OpenWrite;

PROCEDURE FileExists(
    fileName: ADDRESS;
    start, length: INTEGER)
    : BOOLEAN =
  VAR cString: CBuff;
      file :UnixFileDesc; ok: INTEGER;
  BEGIN
    CString(fileName, start, length, (*out*)cString);
    file :=
      Unix.open(ADR(cString), Unix.O_RDONLY, 0);
    IF file < 0 THEN RETURN FALSE
    ELSE
      ok := Unix.close(file);
      RETURN TRUE;
    END;
  END FileExists;

PROCEDURE FindAlongPath(
    path: ADDRESS;
    pathStart, pathLength: INTEGER;
    dirSeparator, pathSeparator: CHAR;
    fileName: ADDRESS;
    fileNameStart, fileNameLength: INTEGER;
    VAR (*out*) dirNameStart, dirNameLength: INTEGER)
    : BOOLEAN =
  VAR buff: ARRAY [0..255] OF CHAR;
    scanPath, scanFileName: ADDRESS; charPtr: CharPtr;
  BEGIN

    IF pathLength = 0 THEN
      IF FileExists(fileName, fileNameStart, fileNameLength) THEN
        dirNameStart := 0;
        dirNameLength := 0;
        RETURN TRUE;
      ELSE
        RETURN FALSE;
      END;
    ELSE
      dirNameStart := pathStart;
      LOOP
        IF dirNameStart >= pathStart + pathLength THEN RETURN FALSE END;
        dirNameLength := 0;
        LOOP
          IF dirNameStart + dirNameLength >= pathStart + pathLength THEN
            EXIT
          END;
          scanPath := path;
          INC(scanPath, dirNameStart+dirNameLength);
          charPtr := LOOPHOLE(scanPath, CharPtr);
          IF charPtr^ = pathSeparator THEN EXIT END;
          INC(dirNameLength);
        END;
	    FOR i := 0 TO dirNameLength-1 DO
	      scanPath := path;
	      INC(scanPath, dirNameStart+i);
	      charPtr := LOOPHOLE(scanPath, CharPtr);
	      buff[i] := charPtr^;
	    END;
	    buff[dirNameLength] := dirSeparator;
	    FOR i := 0 TO fileNameLength-1 DO
	      scanFileName := fileName;
	      INC(scanFileName, fileNameStart+i);
	      charPtr := LOOPHOLE(scanFileName, CharPtr);
	      buff[i+dirNameLength+1] := charPtr^;
	    END;
	    IF FileExists(ADR(buff), 0, 
	      dirNameLength+1+fileNameLength) THEN
	        RETURN TRUE;
	    END;
        INC(dirNameStart, dirNameLength + 1);
      END;
    END;
  END FindAlongPath;

PROCEDURE Lookup(
    inBuff: ADDRESS;
    inStart, inLength: INTEGER;
    (*out*) outBuff: ADDRESS;
    outStart, outMaxLength: INTEGER;
    VAR (*out*) outLength: INTEGER)
    : BOOLEAN =
  VAR text: TEXT; ch: CHAR; scanInBuff: ADDRESS; charPtr: CharPtr;
  BEGIN
    scanInBuff := inBuff;
    INC(scanInBuff, inStart);
    charPtr := LOOPHOLE(scanInBuff, CharPtr);
    IF (inLength > 0) AND (charPtr^ = '$') THEN
      INC(scanInBuff);
      charPtr := LOOPHOLE(scanInBuff, CharPtr);
      ch := charPtr^;
      IF (inLength > 1) AND (ch >= '0') AND (ch <= '9') THEN
        text := Params.Get(ORD(ch) - ORD('0'));
      ELSE
        text :=
          Env.Get(
            ToText(inBuff, inStart + 1, inLength - 1));
      END;
    ELSE
      text := ToText(inBuff, inStart, inLength);
    END;
    outLength := Text.Length(text);
    IF outLength > outMaxLength THEN RETURN FALSE END;
    FromText(text, outBuff, outStart, outLength);
    RETURN TRUE;
  END Lookup;

PROCEDURE ToText(
    string: ADDRESS; start, length: INTEGER)
    : TEXT =
  VAR charPtr: CharPtr;
    buff: ARRAY [0..255] OF CHAR;
    scanString: ADDRESS;
  BEGIN
    scanString := string;
    INC(scanString, start);
    FOR i := 0 TO length-1 DO
      charPtr := LOOPHOLE(scanString, CharPtr);
      buff[i] := charPtr^;
      INC(scanString);
    END;
    RETURN Text.FromChars(SUBARRAY(buff, 0, length));
  END ToText;

PROCEDURE FromText(
    text: TEXT;
    string: ADDRESS; start, length: INTEGER) =
  VAR charPtr: CharPtr; scanString: ADDRESS;
  BEGIN
    (* ASSERT(Text.Length(text)=length); *)
    scanString := string;
    INC(scanString, start);
    FOR i := 0 TO length-1 DO
      charPtr := LOOPHOLE(scanString, CharPtr);
      charPtr^ := Text.GetChar(text, i);
      INC(scanString);
    END;
  END FromText;

PROCEDURE OutChar(wr: Writer; ch: CHAR) =
  VAR ok: BOOLEAN;
  BEGIN ok := PutChar(wr, ch); END OutChar;

PROCEDURE OutString(wr: Writer; string: TEXT) =
  CONST buffSize = 256;
  VAR buff: ARRAY [0..buffSize-1] OF CHAR;
  BEGIN
    Text.SetChars((*out*)buff, string);
    EVAL PutSub(wr, ADR(buff[0]), 0, Text.Length(string));
    IF Text.Length(string) > buffSize THEN OutString(wr, "... ") END; 
  END OutString;

PROCEDURE OutInt(wr: Writer; int: INTEGER) =
  BEGIN
    OutString(wr, "<int>");
(* --
    IF int >= 0 THEN
      Wr.PrintF(wr, "%d", int);
    ELSE
      Wr.PutChar(wr, '~');
      Wr.PrintF(wr, "%d", -int);
    END;
    *)
  END OutInt;

PROCEDURE GetChar(rd: Reader; VAR (*out*) ch: CHAR): BOOLEAN =
  VAR buff: ARRAY [0..0] OF CHAR;
  BEGIN
    IF rd=NIL THEN RETURN FALSE
    ELSIF rd^.lookAheadReady THEN
      ch := LOOPHOLE(rd^.lookAheadByte, CHAR);
      rd^.lookAheadReady := FALSE;
      RETURN TRUE;
    ELSE
      IF Uuio.read(rd^.file, ADR(buff[0]), 1) # 1 THEN
	RETURN FALSE
      ELSE
	ch := buff[0];
	RETURN TRUE;
      END;
    END;
  END GetChar;

PROCEDURE GetSub(
    rd: Reader;
    (*out*) string: ADDRESS;
    start, length: INTEGER;
    VAR (*out*) read: INTEGER)
    : BOOLEAN =
  VAR buff: ADDRESS; charPtr: CharPtr; extra: INTEGER;
  BEGIN
    IF rd=NIL THEN RETURN FALSE END;
    extra := 0;
    buff := string;
    INC(buff,start);
    IF rd^.lookAheadReady AND (length>0) THEN
      charPtr := LOOPHOLE(buff, CharPtr);
      charPtr^ := LOOPHOLE(rd^.lookAheadByte, CHAR);
      INC(buff);
      rd^.lookAheadReady := FALSE;
      extra := 1;
    END;
    read := Uuio.read(rd^.file, buff, length-extra) + extra;
    RETURN read = length;
  END GetSub;

PROCEDURE ReaderClose(rd: Reader): BOOLEAN =
  BEGIN
    IF rd=NIL THEN RETURN FALSE
    ELSIF Unix.close(rd^.file) = 0 THEN
      DISPOSE(rd);
      RETURN TRUE; 
    ELSE RETURN FALSE;
    END;
  END ReaderClose;

PROCEDURE ReaderMore(rd: Reader; VAR (*out*) more: BOOLEAN): BOOLEAN =
  VAR buff: ARRAY [0..0] OF CHAR;
  BEGIN
    IF rd=NIL THEN RETURN FALSE
    ELSIF rd^.lookAheadReady THEN 
      more := TRUE;
      RETURN TRUE;
    ELSIF Uuio.read(rd^.file, ADR(buff[0]), 1) = 1 THEN
      rd^.lookAheadByte := buff[0];
      rd^.lookAheadReady := TRUE;
      more := TRUE;
      RETURN TRUE;
    ELSE
      more := FALSE;
      RETURN TRUE;
    END;
  END ReaderMore;

PROCEDURE ReaderReady(rd: Reader; VAR (*out*) ready: INTEGER): BOOLEAN =
  VAR 
    timeout: UnixTimeVal; rs, ws, es: Unix.FDSet; nfound: INTEGER;
  BEGIN
    IF rd=NIL THEN RETURN FALSE
    ELSIF rd^.lookAheadReady THEN 
      ready := 1;
      RETURN TRUE;
    ELSE
      rs := Unix.FDSet{rd^.file};
      ws := Unix.FDSet{};
      es := Unix.FDSet{};
      timeout.tv_sec := 0;
      timeout.tv_usec := 0;
      nfound := 
	Unix.select(rd^.file+1, ADR(rs), ADR(ws), ADR(es), ADR(timeout));
      IF nfound<0 THEN RETURN FALSE
      ELSIF nfound = 0 THEN ready := 0; RETURN TRUE;
      ELSE ready := 1; RETURN TRUE;
      END;
    END;
  END ReaderReady;

(*
      readfds := Word.Shift(1, rd^.file);
      writefds := 0;
      exceptfds := 0;

PROCEDURE ReaderReady(rd: Reader; VAR (*out*) ready: INTEGER): BOOLEAN;
  VAR buff: ARRAY [0..0] OF CHAR; i:INTEGER;
    flags, newFlags: Unix.FileDescFlags;
  BEGIN
    IF rd=NIL THEN RETURN FALSE
    ELSIF rd^.lookAheadReady THEN 
      ready := 1;
    ELSE
      flags := Unix.FileDescFlags(Unix.fcntl(rd^.file, Unix.fGetFL, 0));
      newFlags := flags + Unix.FileDescFlags{Unix.fNDelay};
      i := Unix.fcntl(rd^.file, Unix.fSetFL, INTEGER(newFlags));
      IF Uuio.read(rd^.file, ADR(buff[0]), 1) = 1 THEN
        rd^.lookAheadByte := buff[0];
        rd^.lookAheadReady := TRUE;
	ready := 1;
      ELSE
        ready := 0;
      END;
      i := Unix.fcntl(rd^.file, Unix.fSetFL, INTEGER(flags));
    END;
    RETURN TRUE;
  END ReaderReady;
*)

PROCEDURE PutChar(wr: Writer; ch: CHAR): BOOLEAN =
  VAR buff: ARRAY [0..0] OF CHAR;
  BEGIN
    IF wr=NIL THEN RETURN FALSE
    ELSE
      buff[0] := ch;
      RETURN Uuio.write(wr^.file, ADR(buff[0]), 1) = 1;
    END;
  END PutChar;

PROCEDURE PutSub(
    wr: Writer;
    string: ADDRESS;
    start, length: INTEGER)
    : BOOLEAN =
  VAR buff: ADDRESS; written: INTEGER; from: ADDRESS;
  BEGIN
    IF wr=NIL THEN RETURN FALSE END;
    buff := string;
    INC(buff, start);
    from := buff;
    written := Uuio.write(wr^.file, from, length);
    RETURN written = length;
  END PutSub;

PROCEDURE WriterClose(wr: Writer): BOOLEAN =
  BEGIN
    IF wr=NIL THEN RETURN FALSE
    ELSIF Unix.close(wr^.file) = 0 THEN
      DISPOSE(wr);
      RETURN TRUE; 
    ELSE RETURN FALSE;
    END;
  END WriterClose;

PROCEDURE WriterFlush(wr: Writer): BOOLEAN =
  BEGIN
    IF wr=NIL THEN RETURN FALSE END;
    RETURN TRUE;
    (* -- *)
  END WriterFlush;

PROCEDURE Setup() =
  BEGIN
    stdin := NEW(Reader);
    stdin^.file := 0;
    stdin^.lookAheadReady := FALSE;
    stdout := NEW(Writer);
    stdout^.file := 1;
    newLine := '\n';
    dirSeparator := '/';
    pathSeparator := ':';
  END Setup;

BEGIN
END QOS.

(*
TYPE
  CStringPointer = UNTRACED REF ARRAY [0..2147483647] OF CHAR;
  CParamPointer  = UNTRACED REF ARRAY [0..2147483647] OF CStringPointer;

PROCEDURE GetParameter(
    num: CARDINAL; 
    VAR (*out*) outBuff: ADDRESS;
    VAR (*out*) outLength: INTEGER) =
  VAR argv: CParamPointer;
  BEGIN
    IF num >= Cstdarg.argc THEN 
      outBuff := NIL;
      outLength := 0;
    ELSE
      argv := LOOPHOLE(Cstdarg.argv, CParamPointer);
      outBuff := ADR(argv^[num]^[0]);
      outLength := 0;
      WHILE argv^[num]^[outLength] # '\000' DO INC(outLength) END;
    END;
  END GetParameter;

PROCEDURE GetEnvironment(
    inBuff: ADDRESS;
    inStart, inLength: INTEGER;
    VAR (*out*) outBuff: ADDRESS;
    VAR (*out*) outLength: INTEGER) =
  VAR cString: CBuff; scanBuff: CStringPointer;
  BEGIN
    CString(inBuff, inStart, inLength, (*out*) cString);
    outBuff := Cstdlib.getenv(ADR(cString[0]));
    scanBuff := LOOPHOLE(outBuff, CStringPointer);
    outLength := 0;
    WHILE scanBuff^[outLength] # '\000' DO INC(outLength) END;
  END GetEnvironment;
*)

(*
PROCEDURE GetEnvironment(
    inBuff: ADDRESS;
    inStart, inLength: INTEGER;
    VAR (*out*) outBuff: ADDRESS;
    VAR (*out*) outLength: INTEGER) =
  VAR envp: CParamPointer; scanInBuff: ADDRESS; scanPtr: CharPtr;
    v, i: INTEGER;
  BEGIN
    v := 0;
    envp := LOOPHOLE(Cstdarg.envp, CParamPointer);
    LOOP
      scanInBuff := inBuff;
      INC(scanInBuff, inStart);
      IF envp^[v] = NIL THEN 
	outBuff := NIL;
	outLength := 0;
	RETURN; 
      END;
      i := 0;
      LOOP
	IF i >= inLength THEN EXIT END;
	scanPtr := LOOPHOLE(scanInBuff, CharPtr);
	IF NOT (scanPtr^ = envp^[v]^[i]) THEN EXIT END;
	INC(scanInBuff);
	INC(i);
      END;
      IF (envp^[v]^[i] = '=') AND (i = inLength) THEN EXIT END;
      INC(v);
    END;
    INC(i);
    outBuff := ADR(envp^[v]^[i]);
    outLength := 0;
    WHILE envp^[v]^[i+outLength] # '\000' DO INC(outLength) END;
  END GetEnvironment;
*)

(* 
PROCEDURE Lookup(
    inBuff: ADDRESS;
    inStart, inLength: INTEGER;
    (*out*) outBuff: ADDRESS;
    outStart, outMaxLength: INTEGER;
    VAR (*out*) outLength: INTEGER)
    : BOOLEAN =
  VAR ch: CHAR; scanInBuff, scanTempBuff, scanOutBuff: ADDRESS; 
    tempBuff: ADDRESS; tempStart, tempLength: INTEGER;
    tempPtr, outPtr, charPtr: CharPtr;
  BEGIN
    scanInBuff := inBuff;
    INC(scanInBuff, inStart);
    charPtr := LOOPHOLE(scanInBuff, CharPtr);
    IF (inLength > 0) AND (charPtr^ = '$') THEN
      INC(scanInBuff);
      charPtr := LOOPHOLE(scanInBuff, CharPtr);
      ch := charPtr^;
      IF (inLength > 1) AND (ch >= '0') AND (ch <= '9') THEN
        GetParameter(ORD(ch) - ORD('0'),
	  (*out*) tempBuff, (*out*) tempLength);
	tempStart := 0;
      ELSE
        GetEnvironment(inBuff, inStart+1, inLength-1,
	  (*out*) tempBuff, (*out*) tempLength);
	tempStart := 0;
      END;
    ELSE
      tempBuff := inBuff;
      tempStart := inStart;
      tempLength := inLength;
    END;
    IF tempLength > outMaxLength THEN RETURN FALSE END;
    scanTempBuff := tempBuff;
    INC(scanTempBuff, tempStart);
    scanOutBuff := outBuff;
    INC(scanOutBuff, outStart);
    FOR i := 0 TO tempLength-1 DO
      tempPtr := LOOPHOLE(scanTempBuff, CharPtr);
      outPtr := LOOPHOLE(scanOutBuff, CharPtr);
      outPtr^ := tempPtr^;
      INC(scanTempBuff);
      INC(scanOutBuff);
    END;
    outLength := tempLength;
    RETURN TRUE;
  END Lookup;
*)

