(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)

UNSAFE MODULE Value;
IMPORT Text, QOS, (*--QRPC, Text, *) Data, Format, Store;

CONST
  ReaderTableSize = 20;
  WriterTableSize = 20;
(* RPC
  RPCTableSize = 5;
  RPCServer = 0;
*)

TYPE
  ReaderTable =
    ARRAY [0 .. ReaderTableSize - 1] OF RECORD
      inUse: BOOLEAN;
      rd: QOS.Reader;
    END;

TYPE
  WriterTable =
    ARRAY [0 .. WriterTableSize - 1] OF RECORD
      inUse: BOOLEAN;
      wr: QOS.Writer;
    END;

(* RPC:
TYPE
  RPCTable =
    ARRAY [0 .. RPCTableSize - 1] OF RECORD inUse: BOOLEAN; rpc: QRPC.T; END;
*)

VAR
  readerTable: ReaderTable;
  writerTable: WriterTable;
  inputReader, outputWriter: T;
  (* RPC: rpcTable: RPCTable; *)

CONST
  TupleFormatCache = 16;
  ClosureFormatCache = 64;

VAR
  stringFormat, badgeFormat, literalsFormat, progFormat,
  optionFormat, arrayFormat, readerFormat, writerFormat, timeFormat,
  dynamicFormat, typeFormat, excPacketFormat: Data.Pointer;
  tupleFormat: ARRAY [0 .. TupleFormatCache - 1] OF Data.Pointer;
  closureFormat: ARRAY [0 .. ClosureFormatCache - 1] OF Data.Pointer;

PROCEDURE ClosureFormat(size: INTEGER): Data.Pointer =
  BEGIN
    RETURN
      Format.New(
        Format.NewSeq(
          Format.NewList(Format.pointer,
            Format.NewList(Format.NewRepeat(size, Format.polymorph),
              Format.emptyList))));
  END ClosureFormat;

PROCEDURE TupleFormat(size: INTEGER): Data.Pointer =
  BEGIN
    RETURN Format.New(Format.NewRepeat(size, Format.polymorph));
  END TupleFormat;

PROCEDURE PlausiblePointer(ptr: Data.Pointer): BOOLEAN =
  VAR fmt, fmtFmt: Data.Pointer;
  BEGIN
    IF NOT Data.IsPointer(ptr) THEN RETURN FALSE END;
    IF (ptr<Store.MinIndex) OR (ptr>Store.MaxIndex) THEN RETURN FALSE END;
    fmt := Format.DataGetFormat(ptr);
    IF NOT Data.IsPointer(fmt) THEN RETURN FALSE END;
    IF (fmt<Store.MinIndex) OR (fmt>Store.MaxIndex) THEN RETURN FALSE END;
    fmtFmt := Format.DataGetFormat(fmt);
    IF fmtFmt # Format.formatFormat THEN RETURN FALSE END;
    RETURN TRUE;
  END PlausiblePointer;

PROCEDURE NewValBool(bool: Data.Bool): T =
  BEGIN RETURN Data.ImmediateOfBool(bool); END NewValBool;

PROCEDURE NewValChar(char: CHAR): T =
  BEGIN RETURN Data.ImmediateOfChar(char); END NewValChar;

PROCEDURE NewValInt(int: Data.Int): T =
  BEGIN RETURN Data.ImmediateOfInt(int); END NewValInt;

PROCEDURE NewValFloat(float: Data.Float): T =
  BEGIN RETURN Data.ImmediateOfFloat(float); END NewValFloat;

PROCEDURE NewValString(
    size: Data.Int;
    init: Data.Pointee;
    VAR (*out*) string: T)
    : BOOLEAN =
  VAR pointer, sizePointer: Data.Pointer;
  BEGIN
    IF size < 0 THEN RETURN FALSE END;
    Store.Align();
    Store.LayPointer(stringFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayInt(size);
    FOR i := 0 TO size - 1 DO Store.LayPointee(init); END;
    Store.SetInt(sizePointer, Store.hp - pointer);
    string := pointer;
    RETURN TRUE;
  END NewValString;

PROCEDURE ValStringOfBuff(
    buff: ADDRESS;
    start, length: INTEGER;
    VAR (*out*) string: T)
    : BOOLEAN =
  TYPE CharPtr = UNTRACED REF CHAR;
  VAR scanBuff: ADDRESS; charPtr: CharPtr;
  BEGIN
    IF NOT (NewValString(length, Data.MinPointee, (*out*) string)) THEN
      RETURN FALSE
    END;
    scanBuff := buff;
    INC(scanBuff, start);
    FOR i := 0 TO length - 1 DO
      charPtr := LOOPHOLE(scanBuff, CharPtr);
      IF NOT ValStringSetPointee(string, i, LOOPHOLE(charPtr^, Data.Pointee))
      THEN
        RETURN FALSE
      END;
      INC(scanBuff);
    END;
    RETURN TRUE;
  END ValStringOfBuff;

PROCEDURE ValStringLength(string: T): Data.Int =
  BEGIN RETURN Store.GetInt(string); END ValStringLength;

PROCEDURE ValStringStart(string: T): Data.Pointer =
  BEGIN RETURN string + Data.PointeesPerInt (*length*) END ValStringStart;

PROCEDURE ValStringIsEmpty(string: T): Data.Bool =
  BEGIN RETURN Store.GetInt(string) = 0; END ValStringIsEmpty;

PROCEDURE ValStringGetPointee(
    string: T;
    i: Data.Int;
    VAR (*out*) pointee: Data.Pointee)
    : BOOLEAN =
  VAR length: Data.Int;
  BEGIN
    length := Store.GetInt(string);
    IF (i < 0) OR (i >= length) THEN RETURN FALSE END;
    pointee :=
      Store.GetPointee(
        string + Data.PointeesPerInt (* length *) + LOOPHOLE(i, Data.Card));
    RETURN TRUE;
  END ValStringGetPointee;

PROCEDURE ValStringSetPointee(
    string: T;
    i: Data.Int;
    pointee: Data.Pointee)
    : BOOLEAN =
  VAR length: Data.Int;
  BEGIN
    length := Store.GetInt(string);
    IF (i < 0) OR (i >= length) THEN RETURN FALSE END;
    Store.SetPointee(
      string + Data.PointeesPerInt (* length *) + LOOPHOLE(i, Data.Card), pointee);
    RETURN TRUE;
  END ValStringSetPointee;

PROCEDURE ValStringGetSub(
    string: T;
    start, size: Data.Int;
    VAR (*out*) sub: T)
    : BOOLEAN =
  VAR
    pointer, sizePointer: Data.Pointer;
    scan: Data.Pointer;
    length: Data.Int;
  BEGIN
    length := Store.GetInt(string);
    IF (start < 0) OR (size < 0) THEN RETURN FALSE END;
    IF start + size > length THEN RETURN FALSE END;
    scan := string + Data.PointeesPerInt (*length*) + LOOPHOLE(start, Data.Card);
    Store.Align();
    Store.LayPointer(stringFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayInt(size);
    FOR i := 0 TO size - 1 DO
      Store.LayPointee(Store.GetPointee(scan));
      INC(scan);
    END;
    Store.SetInt(sizePointer, Store.hp - pointer);
    sub := pointer;
    RETURN TRUE;
  END ValStringGetSub;

PROCEDURE ValStringSetSub(
    string1: T;
    start1: Data.Int;
    string2: T;
    start2, size: Data.Int)
    : BOOLEAN =
  VAR scan1, scan2: Data.Pointer; length1, length2: Data.Int;
  BEGIN
    IF size = 0 THEN RETURN TRUE END;
    length1 := Store.GetInt(string1);
    length2 := Store.GetInt(string2);
    IF (start1 < 0) OR (start2 < 0) OR (size < 0) THEN RETURN FALSE END;
    IF (start1 + size > length1) OR (start2 + size > length2) THEN
      RETURN FALSE;
    END;
    scan1 := string1 + Data.PointeesPerInt (*length*) + LOOPHOLE(start1, Data.Card);
    scan2 := string2 + Data.PointeesPerInt (*length*) + LOOPHOLE(start2, Data.Card);
    IF scan1 <= scan2 THEN
      FOR i := 0 TO size - 1 DO
        Store.SetPointee(scan1, Store.GetPointee(scan2));
        INC(scan1);
        INC(scan2);
      END;
    ELSE
      INC(scan1, size - 1);
      INC(scan2, size - 1);
      FOR i := 0 TO size - 1 DO
        Store.SetPointee(scan1, Store.GetPointee(scan2));
        DEC(scan1);
        DEC(scan2);
      END;
    END;
    RETURN TRUE;
  END ValStringSetSub;

PROCEDURE ValStringCat(string1, string2: T): T =
  VAR
    pointer, sizePointer: Data.Pointer;
    length1, length2: Data.Int;
    scan1, scan2: Data.Pointer;
  BEGIN
    length1 := Store.GetInt(string1);
    length2 := Store.GetInt(string2);
    scan1 := string1 + Data.PointeesPerInt (*length*);
    scan2 := string2 + Data.PointeesPerInt (*length*);
    Store.Align();
    Store.LayPointer(stringFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayInt(length1 + length2);
    FOR i := 0 TO length1 - 1 DO
      Store.LayPointee(Store.GetPointee(scan1));
      INC(scan1);
    END;
    FOR i := 0 TO length2 - 1 DO
      Store.LayPointee(Store.GetPointee(scan2));
      INC(scan2);
    END;
    Store.SetInt(sizePointer, Store.hp - pointer);
    RETURN pointer;
  END ValStringCat;

PROCEDURE ValStringCatSub(
    string1: T;
    start1, size1: Data.Int;
    string2: T;
    start2, size2: Data.Int;
    VAR (*out*) string3: T)
    : BOOLEAN =
  VAR
    pointer, sizePointer: Data.Pointer;
    length1, length2: Data.Int;
    scan1, scan2: Data.Pointer;
  BEGIN
    length1 := Store.GetInt(string1);
    length2 := Store.GetInt(string2);
    scan1 := string1 + Data.PointeesPerInt (*length*) + LOOPHOLE(start1, Data.Card);
    scan2 := string2 + Data.PointeesPerInt (*length*) + LOOPHOLE(start2, Data.Card);
    IF (start1 < 0) OR (start2 < 0) OR (size1 < 0) OR (size2 < 0) THEN
      RETURN FALSE
    END;
    IF (start1 + size1 > length1) OR (start2 + size2 > length2) THEN
      RETURN FALSE
    END;
    Store.Align();
    Store.LayPointer(stringFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayInt(size1 + size2);
    FOR i := 0 TO size1 - 1 DO
      Store.LayPointee(Store.GetPointee(scan1));
      INC(scan1);
    END;
    FOR i := 0 TO size2 - 1 DO
      Store.LayPointee(Store.GetPointee(scan2));
      INC(scan2);
    END;
    Store.SetInt(sizePointer, Store.hp - pointer);
    string3 := pointer;
    RETURN TRUE;
  END ValStringCatSub;

PROCEDURE ValStringEqual(string1, string2: T): Data.Bool =
  VAR length: Data.Int; scan1, scan2: Data.Pointer;
  BEGIN
    length := Store.GetInt(string1);
    IF Store.GetInt(string2) # length THEN RETURN FALSE; END;
    scan1 := string1 + Data.PointeesPerInt (*length*);
    scan2 := string2 + Data.PointeesPerInt (*length*);
    LOOP
      IF length <= 0 THEN RETURN TRUE END;
      IF Store.GetPointee(scan1) # Store.GetPointee(scan2) THEN
        RETURN FALSE;
      END;
      INC(scan1);
      INC(scan2);
      DEC(length);
    END;
  END ValStringEqual;

PROCEDURE ValStringEqualSub(
    string1: T;
    start1, size1: Data.Int;
    string2: T;
    start2, size2: Data.Int;
    VAR (*out*) eq: Data.Bool)
    : BOOLEAN =
  VAR length, length1, length2: Data.Int; scan1, scan2: Data.Pointer;
  BEGIN
    length1 := Store.GetInt(string1);
    length2 := Store.GetInt(string2);
    IF (start1 < 0) OR (size1 < 0) OR (start1 + size1 > length1) THEN
      RETURN FALSE
    END;
    IF (start2 < 0) OR (size2 < 0) OR (start2 + size2 > length2) THEN
      RETURN FALSE
    END;
    IF size1 # size2 THEN eq := FALSE; RETURN TRUE END;
    length := size1;
    scan1 := string1 + Data.PointeesPerInt (*length*) + LOOPHOLE(start1, Data.Card);
    scan2 := string2 + Data.PointeesPerInt (*length*) + LOOPHOLE(start2, Data.Card);
    LOOP
      IF length <= 0 THEN eq := TRUE; RETURN TRUE END;
      IF Store.GetPointee(scan1) # Store.GetPointee(scan2) THEN
        eq := FALSE;
        RETURN TRUE
      END;
      INC(scan1);
      INC(scan2);
      DEC(length);
    END;
  END ValStringEqualSub;

PROCEDURE ValStringPrecedes(string1, string2: T): Data.Bool =
  VAR
    i, length1, length2: Data.Int;
    scan1, scan2: Data.Pointer;
    pointee1, pointee2: Data.Pointee;
  BEGIN
    length1 := Store.GetInt(string1);
    length2 := Store.GetInt(string2);
    IF length1 > length2 THEN RETURN FALSE END;
    scan1 := string1 + Data.PointeesPerInt (*length*);
    scan2 := string2 + Data.PointeesPerInt (*length*);
    i := length1;
    LOOP
      IF i <= 0 THEN RETURN TRUE END;
      pointee1 := Store.GetPointee(scan1);
      pointee2 := Store.GetPointee(scan2);
      IF pointee1 # pointee2 THEN RETURN pointee1 < pointee2 END;
      INC(scan1);
      INC(scan2);
      DEC(i);
    END;
  END ValStringPrecedes;

PROCEDURE ValStringPrecedesSub(
    string1: T;
    start1, size1: Data.Int;
    string2: T;
    start2, size2: Data.Int;
    VAR (*out*) prec: Data.Bool)
    : BOOLEAN =
  VAR
    i, length1, length2: Data.Int;
    scan1, scan2: Data.Pointer;
    pointee1, pointee2: Data.Pointee;
  BEGIN
    length1 := Store.GetInt(string1);
    length2 := Store.GetInt(string2);
    IF (start1 < 0) OR (size1 < 0) OR (start1 + size1 > length1) THEN
      RETURN FALSE
    END;
    IF (start2 < 0) OR (size2 < 0) OR (start2 + size2 > length2) THEN
      RETURN FALSE
    END;
    IF size1 > size2 THEN prec := FALSE; RETURN TRUE END;
    scan1 := string1 + Data.PointeesPerInt (*length*) + LOOPHOLE(start1, Data.Card);
    scan2 := string2 + Data.PointeesPerInt (*length*) + LOOPHOLE(start2, Data.Card);
    i := size1;
    LOOP
      IF i <= 0 THEN prec := TRUE; RETURN TRUE END;
      pointee1 := Store.GetPointee(scan1);
      pointee2 := Store.GetPointee(scan2);
      IF pointee1 # pointee2 THEN
        prec := pointee1 < pointee2;
        RETURN TRUE
      END;
      INC(scan1);
      INC(scan2);
      DEC(i);
    END;
  END ValStringPrecedesSub;

PROCEDURE NewValTuple(size: Data.Int): T =
  VAR
    pointer, sizePointer: Data.Pointer;
    cacheLimit: Data.Int;
    format: Data.Pointer;
  BEGIN
    IF size < 0 THEN Data.Fault("") END;
    cacheLimit := TupleFormatCache;
    IF size < cacheLimit THEN
      format := tupleFormat[size];
    ELSE
      format := TupleFormat(size);
    END;
    Store.Align();
    Store.LayPointer(format);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    FOR i := 0 TO (Data.PointeesPerPolymorph * size) - 1 DO
      Store.LayPointee(Data.MinPointee);
    END;
    Store.SetInt(sizePointer, Store.hp - pointer);
    RETURN pointer;
  END NewValTuple;

PROCEDURE ValTupleGet(tuple: T; i: Data.Card): T =
  BEGIN RETURN Store.GetPolymorph(tuple + i); END ValTupleGet;

PROCEDURE ValTupleSet(tuple: T; i: Data.Card; item: T) =
  BEGIN Store.SetPolymorph(tuple + i, item); END ValTupleSet;

PROCEDURE NewValRaisePacket(exc, val: T): T =
  VAR pointer, sizePointer: Data.Pointer;
  BEGIN
    Store.Align();
    Store.LayPointer(excPacketFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayPointer(exc);
    Store.LayPolymorph(val);
    Store.SetInt(sizePointer, Store.hp - pointer);
    RETURN pointer;
  END NewValRaisePacket;

PROCEDURE ValRaisePacketExc(val: T): T =
  BEGIN RETURN Store.GetPointer(val); END ValRaisePacketExc;

PROCEDURE ValRaisePacketVal(val: T): T =
  BEGIN
    RETURN Store.GetPolymorph(val + Data.PointeesPerPointer (* exc *));
  END ValRaisePacketVal;

PROCEDURE NewValMachineState(
    stack: T;
    TL, TP, FP: Data.Pointer;
    CP: T;
    PC: Data.Pointer;
    EX: T;
    home: T;
    fault: Data.Bool)
    : T =
  VAR machineState: T;
  BEGIN
    machineState := NewValTuple(9);
    ValTupleSet(machineState, 0, stack);
    ValTupleSet(machineState, Data.PointeesPerPolymorph,
      Data.ImmediateOfInt(TL));
    ValTupleSet(machineState, 2 * Data.PointeesPerPolymorph,
      Data.ImmediateOfInt(TP));
    ValTupleSet(machineState, 3 * Data.PointeesPerPolymorph,
      Data.ImmediateOfInt(FP));
    ValTupleSet(machineState, 4 * Data.PointeesPerPolymorph, CP);
    ValTupleSet(machineState, 5 * Data.PointeesPerPolymorph,
      Data.ImmediateOfInt(PC));
    ValTupleSet(machineState, 6 * Data.PointeesPerPolymorph, EX);
    ValTupleSet(machineState, 7 * Data.PointeesPerPolymorph, home);
    ValTupleSet(machineState, 8 * Data.PointeesPerPolymorph,
      Data.ImmediateOfBool(fault));
    RETURN machineState;
  END NewValMachineState;

PROCEDURE ValMachineStateSet(
    machineState: T;
    stack: T;
    TL, TP, FP: Data.Pointer;
    CP: T;
    PC: Data.Pointer;
    EX: T;
    home: T;
    fault: Data.Bool) =
  BEGIN
    ValTupleSet(machineState, 0, stack);
    ValTupleSet(machineState, Data.PointeesPerPolymorph,
      Data.ImmediateOfInt(TL));
    ValTupleSet(machineState, 2 * Data.PointeesPerPolymorph,
      Data.ImmediateOfInt(TP));
    ValTupleSet(machineState, 3 * Data.PointeesPerPolymorph,
      Data.ImmediateOfInt(FP));
    ValTupleSet(machineState, 4 * Data.PointeesPerPolymorph, CP);
    ValTupleSet(machineState, 5 * Data.PointeesPerPolymorph,
      Data.ImmediateOfInt(PC));
    ValTupleSet(machineState, 6 * Data.PointeesPerPolymorph, EX);
    ValTupleSet(machineState, 7 * Data.PointeesPerPolymorph, home);
    ValTupleSet(machineState, 8 * Data.PointeesPerPolymorph,
      Data.ImmediateOfBool(fault));
  END ValMachineStateSet;

PROCEDURE ValMachineStateGet(
    machineState: T;
    VAR (*out*) stack: T;
    VAR (*out*) TL, TP, FP: Data.Pointer;
    VAR (*out*) CP: T;
    VAR (*out*) PC: Data.Pointer;
    VAR (*out*) EX: T;
    VAR (*out*) home: T;
    VAR (*out*) fault: Data.Bool) =
  BEGIN
    stack := ValTupleGet(machineState, 0);
    TL :=
      Data.IntOfImmediate(
        ValTupleGet(machineState, Data.PointeesPerPolymorph));
    TP :=
      Data.IntOfImmediate(
        ValTupleGet(machineState, 2 * Data.PointeesPerPolymorph));
    FP :=
      Data.IntOfImmediate(
        ValTupleGet(machineState, 3 * Data.PointeesPerPolymorph));
    CP := ValTupleGet(machineState, 4 * Data.PointeesPerPolymorph);
    PC :=
      Data.IntOfImmediate(
        ValTupleGet(machineState, 5 * Data.PointeesPerPolymorph));
    EX := ValTupleGet(machineState, 6 * Data.PointeesPerPolymorph);
    home := ValTupleGet(machineState, 7 * Data.PointeesPerPolymorph);
    fault :=
      Data.BoolOfImmediate(
        ValTupleGet(machineState, 8 * Data.PointeesPerPolymorph));
  END ValMachineStateGet;

PROCEDURE NewValArray(
    items: Data.Int;
    init: T;
    VAR (*out*) array: T)
    : BOOLEAN =
  VAR pointer, sizePointer: Data.Pointer;
  BEGIN
    IF items < 0 THEN RETURN FALSE END;
    Store.Align();
    Store.LayPointer(arrayFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayInt(items);
    FOR i := 0 TO items - 1 DO Store.LayPolymorph(init); END;
    Store.SetInt(sizePointer, Store.hp - pointer);
    Store.Align();
    array := pointer;
    RETURN TRUE;
  END NewValArray;

PROCEDURE ValArraySize(array: T): Data.Int =
  BEGIN RETURN Store.GetInt(array); END ValArraySize;

PROCEDURE ValArrayGetItem(
    array: T;
    i: Data.Int;
    VAR (*out*) item: T)
    : BOOLEAN =
  VAR size: Data.Int;
  BEGIN
    size := ValArraySize(array);
    IF (i < 0) OR (i >= size * Data.PointeesPerPolymorph) THEN
      RETURN FALSE;
    END;
    item :=
      Store.GetPolymorph(
        array + Data.PointeesPerInt (* size *) + LOOPHOLE(i, Data.Card));
    RETURN TRUE;
  END ValArrayGetItem;

PROCEDURE ValArraySetItem(array: T; i: Data.Int; item: T): BOOLEAN =
  VAR size: Data.Int;
  BEGIN
    size := ValArraySize(array);
    IF (i < 0) OR (i >= size * Data.PointeesPerPolymorph) THEN
      RETURN FALSE;
    END;
    Store.SetPolymorph(
      array + Data.PointeesPerInt (* size *) + LOOPHOLE(i, Data.Card), item);
    RETURN TRUE;
  END ValArraySetItem;

PROCEDURE NewValBadge(
    moduleName, progName: Data.Pointer;
    machineId, processId, timeSecs, timeMicroSecs, timeSerialNo: Data.Card)
    : T =
  VAR pointer, sizePointer: Data.Pointer;
  BEGIN
    Store.Align();
    Store.LayPointer(badgeFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayPointer(moduleName);
    Store.LayPointer(progName);
    Store.LayImmediate(Data.ImmediateOfInt(machineId));
    Store.LayImmediate(Data.ImmediateOfInt(processId));
    Store.LayImmediate(Data.ImmediateOfInt(timeSecs));
    Store.LayImmediate(Data.ImmediateOfInt(timeMicroSecs));
    Store.LayImmediate(Data.ImmediateOfInt(timeSerialNo));
    Store.SetInt(sizePointer, Store.hp - pointer);
    RETURN pointer;
  END NewValBadge;

PROCEDURE ValBadgeName(badge: T; VAR (*out*) moduleName, progName: T) =
  VAR scan: T;
  BEGIN
    scan := badge;
    moduleName := Store.GetPointer(scan);
    INC(scan, Data.PointeesPerPointer);
    progName := Store.GetPointer(scan);
  END ValBadgeName;

PROCEDURE ValBadgeId(
    badge: T;
    VAR (*out*)
      machineId, processId, timeSecs, timeMicroSecs, timeSerialNo:
      Data.Card) =
  VAR scan: T;
  BEGIN
    scan := badge;
    INC(scan, Data.PointeesPerPointer (*moduleName*));
    INC(scan, Data.PointeesPerPointer (*progName*));
    machineId := Data.IntOfImmediate(Store.GetImmediate(scan));
    INC(scan, Data.PointeesPerImmediate);
    processId := Data.IntOfImmediate(Store.GetImmediate(scan));
    INC(scan, Data.PointeesPerImmediate);
    timeSecs := Data.IntOfImmediate(Store.GetImmediate(scan));
    INC(scan, Data.PointeesPerImmediate);
    timeMicroSecs := Data.IntOfImmediate(Store.GetImmediate(scan));
    INC(scan, Data.PointeesPerImmediate);
    timeSerialNo := Data.IntOfImmediate(Store.GetImmediate(scan));
  END ValBadgeId;

PROCEDURE NewValLiterals(size: Data.Int): T =
  VAR pointer, sizePointer: Data.Pointer;
  BEGIN
    IF size < 0 THEN Data.Fault("") END;
    Store.Align();
    Store.LayPointer(literalsFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayInt(size);
    FOR i := 0 TO (Data.PointeesPerPolymorph * size) - 1 DO
      Store.LayPointee(Data.MinPointee);
    END;
    Store.SetInt(sizePointer, Store.hp - pointer);
    RETURN pointer;
  END NewValLiterals;

PROCEDURE ValLiteralsSize(literals: T): Data.Int =
  BEGIN RETURN Store.GetInt(literals); END ValLiteralsSize;

PROCEDURE ValLiteralsGet(literals: T; i: Data.Card): T =
  BEGIN
    RETURN Store.GetPolymorph(literals + Data.PointeesPerInt (*size*) + i);
  END ValLiteralsGet;

PROCEDURE ValLiteralsSet(literals: T; i: Data.Card; literal: T) =
  BEGIN
    Store.SetPolymorph(literals + Data.PointeesPerInt (*size*) + i,
      literal);
  END ValLiteralsSet;

PROCEDURE NewValProg(codeSize: Data.Int): T =
  VAR pointer, sizePointer: Data.Pointer;
  BEGIN
    IF codeSize < 0 THEN Data.Fault("") END;
    Store.Align();
    Store.LayPointer(progFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayPointer(Data.MinPointer); (* Literals *)
    Store.LayInt(codeSize); (* CodeSize *)
    FOR i := 0 TO codeSize - 1 DO
      Store.LayPointee(Data.MinPointee); (* Code *)
    END;
    Store.SetInt(sizePointer, Store.hp - pointer);
    RETURN pointer;
  END NewValProg;

PROCEDURE ValProgGetLiterals(prog: T): T =
  BEGIN RETURN Store.GetPointer(prog); END ValProgGetLiterals;

PROCEDURE ValProgSetLiterals(prog: T; literals: T) =
  BEGIN Store.SetPointer(prog, literals); END ValProgSetLiterals;

PROCEDURE ValProgCodeSize(prog: T): Data.Int =
  BEGIN
    RETURN Store.GetInt(prog + Data.PointeesPerPointer (* Literals *))
  END ValProgCodeSize;

PROCEDURE ValProgCodeStart(prog: T): Data.Pointer =
  BEGIN
    RETURN
      prog + Data.PointeesPerPointer (* Literals *) +
        Data.PointeesPerInt (* Size *);
  END ValProgCodeStart;

PROCEDURE NewValClosure(prog: T; globals: Data.Int): T =
  VAR
    pointer, sizePointer: Data.Pointer;
    cacheLimit: Data.Int;
    format: Data.Pointer;
  BEGIN
    IF globals < 0 THEN Data.Fault("") END;
    cacheLimit := ClosureFormatCache;
    IF globals < cacheLimit THEN
      format := closureFormat[globals];
    ELSE
      format := ClosureFormat(globals);
    END;
    Store.Align();
    Store.LayPointer(format);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayPointer(prog); (* Prog *)
    FOR i := 0 TO globals - 1 DO
      Store.LayPolymorph(Data.MinPointer); (* Globals *)
    END;
    Store.SetInt(sizePointer, Store.hp - pointer);
    RETURN pointer;
  END NewValClosure;

PROCEDURE ValClosureProg(closure: T): T =
  BEGIN RETURN Store.GetPointer(closure); END ValClosureProg;

PROCEDURE ValClosureGetGlobal(closure: T; i: Data.Card): T =
  BEGIN
    RETURN
      Store.GetPolymorph(closure + Data.PointeesPerPointer (* Prog *) + i);
  END ValClosureGetGlobal;

PROCEDURE ValClosureSetGlobal(closure: T; i: Data.Card; global: T) =
  BEGIN
    Store.SetPolymorph(closure + Data.PointeesPerPointer (* Prog *) + i,
      global);
  END ValClosureSetGlobal;

PROCEDURE ValClosureTrace(closure: T; trace: BOOLEAN) =
  VAR start: Data.Pointer; instr: Data.SmallCard;
  BEGIN
    start := ValProgCodeStart(ValClosureProg(closure));
    instr := Store.GetSmallCard(start);
    IF trace THEN
      IF instr < 16_8000 THEN 
	Store.SetSmallCard(start, instr + 16_8000)
      END;
    ELSE
      IF instr >= 16_8000 THEN
        Store.SetSmallCard(start, instr - 16_8000)
      END;
    END;
  END ValClosureTrace;

PROCEDURE ValClosureBreak(closure: T; break: BOOLEAN) =
  VAR start: Data.Pointer; instr: Data.SmallCard;
  BEGIN
    start := ValProgCodeStart(ValClosureProg(closure));
    instr := Store.GetSmallCard(start);
    IF break THEN
      IF (instr MOD 16_8000) < 16_4000 THEN
        Store.SetSmallCard(start, instr + 16_4000)
      END;
    ELSE
      IF (instr MOD 16_8000) >= 16_4000 THEN
        Store.SetSmallCard(start, instr - 16_4000)
      END;
    END;
  END ValClosureBreak;

PROCEDURE OpenReader(reader: QOS.Reader; VAR (*out*) i: INTEGER): BOOLEAN =
  BEGIN
    i := 0;
    LOOP
      IF i >= ReaderTableSize THEN RETURN FALSE END;
      IF NOT readerTable[i].inUse THEN
        readerTable[i].inUse := TRUE;
        readerTable[i].rd := reader;
        EXIT;
      END;
      i := i + 1;
    END;
    RETURN TRUE;
  END OpenReader;

PROCEDURE CloseReader(i: INTEGER) =
  BEGIN 
    readerTable[i].inUse := FALSE;
    readerTable[i].rd := NIL;
  END CloseReader;

PROCEDURE FetchReader(i: INTEGER): QOS.Reader =
  BEGIN RETURN readerTable[i].rd; END FetchReader;

PROCEDURE NewValReader(readerIndex: Data.Int): T =
  VAR pointer, sizePointer: Data.Pointer;
  BEGIN
    Store.Align();
    Store.LayPointer(readerFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayInt(readerIndex);
    Store.SetInt(sizePointer, Store.hp - pointer);
    RETURN pointer;
  END NewValReader;

PROCEDURE ValReaderIndex(reader: T): Data.Int =
  BEGIN RETURN Store.GetInt(reader); END ValReaderIndex;

PROCEDURE ValReaderInput(): T = BEGIN RETURN inputReader; END ValReaderInput;

PROCEDURE NewValReaderFile(fileName: T; VAR (*out*) reader: T): BOOLEAN =
  VAR rd: QOS.Reader; i: INTEGER;
  BEGIN
    IF QOS.OpenRead(LOOPHOLE(Store.heapAdr, ADDRESS), 
	 ValStringStart(fileName),
         ValStringLength(fileName), (*out*) rd) THEN
      IF NOT (OpenReader(rd, (*out*) i)) THEN RETURN FALSE END;
      reader := NewValReader(i);
      RETURN TRUE;
    END;
    RETURN FALSE;
  END NewValReaderFile;

PROCEDURE ValReaderClose(reader: T): BOOLEAN =
  VAR readerIndex: Data.Int;
  BEGIN
    readerIndex := ValReaderIndex(reader);
    IF NOT QOS.ReaderClose(FetchReader(readerIndex)) THEN RETURN FALSE END;
    CloseReader(readerIndex);
    RETURN TRUE;
  END ValReaderClose;

PROCEDURE ValReaderMore(reader: T; VAR (*out*) more: Data.Bool): BOOLEAN =
  VAR rd: QOS.Reader;
  BEGIN
    rd := FetchReader(ValReaderIndex(reader));
    RETURN QOS.ReaderMore(rd, (*out*) more);
  END ValReaderMore;

PROCEDURE ValReaderReady(reader: T; VAR (*out*) ready: Data.Int): BOOLEAN =
  VAR rd: QOS.Reader; rdReady: Data.Int;
  BEGIN
    rd := FetchReader(ValReaderIndex(reader));
    IF NOT QOS.ReaderReady(rd, (*out*) rdReady) THEN RETURN FALSE END;
    ready := rdReady;
    RETURN TRUE;
  END ValReaderReady;

PROCEDURE ValReaderGetPointee(
    reader: T;
    VAR (*out*) pointee: Data.Pointee)
    : BOOLEAN =
  VAR rd: QOS.Reader; ch: CHAR;
  BEGIN
    rd := FetchReader(ValReaderIndex(reader));
    IF NOT QOS.GetChar(rd, (*out*) ch) THEN RETURN FALSE END;
    pointee := LOOPHOLE(ch, Data.Pointee);
    RETURN TRUE;
  END ValReaderGetPointee;

PROCEDURE ValReaderGetString(
    reader: T;
    length: Data.Int;
    VAR (*out*) string: T)
    : BOOLEAN =
  VAR rd: QOS.Reader; readLength: Data.Int;
  BEGIN
    IF NOT (NewValString(length, Data.MinPointee, (*out*) string)) THEN
      RETURN FALSE
    END;
    rd := FetchReader(ValReaderIndex(reader));
    IF NOT QOS.GetSub(rd, LOOPHOLE(Store.heapAdr, ADDRESS), ValStringStart(string), length,
             (*out*) readLength) THEN
      RETURN FALSE
    END;
    IF readLength # length THEN RETURN FALSE END;
    RETURN TRUE;
  END ValReaderGetString;

PROCEDURE ValReaderGetSubString(
    reader: T;
    string: T;
    start, length: Data.Int)
    : BOOLEAN =
  VAR
    rd: QOS.Reader;
    stringLength: Data.Int;
    stringStart, readLength: Data.Int;
  BEGIN
    stringLength := ValStringLength(string);
    IF (start < 0) OR (length < 0) OR (start + length > stringLength) THEN
      RETURN FALSE;
    END;
    rd := FetchReader(ValReaderIndex(reader));
    stringStart := ValStringStart(string);
    IF NOT QOS.GetSub(rd, LOOPHOLE(Store.heapAdr, ADDRESS), stringStart + start, length,
             (*out*) readLength) THEN
      RETURN FALSE
    END;
    IF readLength # length THEN RETURN FALSE END;
    RETURN TRUE;
  END ValReaderGetSubString;

PROCEDURE OpenWriter(writer: QOS.Writer; VAR (*out*) i: INTEGER): BOOLEAN =
  BEGIN
    i := 0;
    LOOP
      IF i >= WriterTableSize THEN RETURN FALSE END;
      IF NOT writerTable[i].inUse THEN
        writerTable[i].inUse := TRUE;
        writerTable[i].wr := writer;
        EXIT;
      END;
      i := i + 1;
    END;
    RETURN TRUE;
  END OpenWriter;

PROCEDURE CloseWriter(i: INTEGER) =
  BEGIN 
    writerTable[i].inUse := FALSE; 
    writerTable[i].wr := NIL;
  END CloseWriter;

PROCEDURE FetchWriter(i: INTEGER): QOS.Writer =
  BEGIN RETURN writerTable[i].wr; END FetchWriter;

PROCEDURE NewValWriter(writerIndex: Data.Int): T =
  VAR pointer, sizePointer: Data.Pointer;
  BEGIN
    Store.Align();
    Store.LayPointer(writerFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayInt(writerIndex);
    Store.SetInt(sizePointer, Store.hp - pointer);
    RETURN pointer;
  END NewValWriter;

PROCEDURE ValWriterIndex(writer: T): Data.Int =
  BEGIN RETURN Store.GetInt(writer); END ValWriterIndex;

PROCEDURE ValWriterOutput(): T =
  BEGIN RETURN outputWriter; END ValWriterOutput;

PROCEDURE NewValWriterFile(fileName: T; VAR (*out*) writer: T): BOOLEAN =
  VAR wr: QOS.Writer; i: INTEGER;
  BEGIN
    IF QOS.OpenWrite(LOOPHOLE(Store.heapAdr, ADDRESS), 
	 ValStringStart(fileName),
         ValStringLength(fileName), (*out*) wr) THEN
      IF NOT (OpenWriter(wr, (*out*) i)) THEN RETURN FALSE END;
      writer := NewValWriter(i);
      RETURN TRUE;
    END;
    RETURN FALSE;
  END NewValWriterFile;

PROCEDURE ValWriterClose(writer: T): BOOLEAN =
  VAR writerIndex: Data.Int;
  BEGIN
    writerIndex := ValWriterIndex(writer);
    IF NOT QOS.WriterClose(FetchWriter(writerIndex)) THEN RETURN FALSE END;
    CloseWriter(writerIndex);
    RETURN TRUE;
  END ValWriterClose;

PROCEDURE ValWriterFlush(writer: T): BOOLEAN =
  VAR wr: QOS.Writer;
  BEGIN
    wr := FetchWriter(ValWriterIndex(writer));
    RETURN QOS.WriterFlush(wr);
  END ValWriterFlush;

PROCEDURE ValWriterPutPointee(writer: T; pointee: Data.Pointee): BOOLEAN =
  VAR wr: QOS.Writer;
  BEGIN
    wr := FetchWriter(ValWriterIndex(writer));
    RETURN QOS.PutChar(wr, LOOPHOLE(pointee, CHAR));
  END ValWriterPutPointee;

PROCEDURE ValWriterPutString(writer: T; string: T): BOOLEAN =
  VAR wr: QOS.Writer;
  BEGIN
    wr := FetchWriter(ValWriterIndex(writer));
    RETURN
      QOS.PutSub(wr, LOOPHOLE(Store.heapAdr, ADDRESS), ValStringStart(string),
        ValStringLength(string));
  END ValWriterPutString;

PROCEDURE ValWriterPutSubString(
    writer: T;
    string: T;
    start, length: Data.Int)
    : BOOLEAN =
  VAR wr: QOS.Writer; stringLength: Data.Int; stringStart: Data.Int;
  BEGIN
    stringLength := ValStringLength(string);
    IF (start < 0) OR (length < 0) OR (start + length > stringLength) THEN
      RETURN FALSE
    END;
    wr := FetchWriter(ValWriterIndex(writer));
    stringStart := ValStringStart(string);
    RETURN QOS.PutSub(wr, LOOPHOLE(Store.heapAdr, ADDRESS), stringStart + start, length);
  END ValWriterPutSubString;

PROCEDURE NewValTime(seconds, microSeconds: Data.Int): T =
  VAR pointer, sizePointer: Data.Pointer;
  BEGIN
    Store.Align();
    Store.LayPointer(timeFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayInt(seconds);
    Store.LayInt(microSeconds);
    Store.SetInt(sizePointer, Store.hp - pointer);
    RETURN pointer;
  END NewValTime;

PROCEDURE ValTimeNow(): T =
  VAR seconds, microseconds: Data.Int;
  BEGIN
    QOS.TimeNow( (*out*) seconds, (*out*) microseconds);
    RETURN NewValTime(seconds, microseconds);
  END ValTimeNow;

PROCEDURE ValTimeSeconds(time: T): Data.Int =
  BEGIN RETURN Store.GetInt(time); END ValTimeSeconds;

PROCEDURE ValTimeMicroSeconds(time: T): Data.Int =
  BEGIN
    RETURN Store.GetInt(time + Data.PointeesPerInt (*seconds*));
  END ValTimeMicroSeconds;

PROCEDURE ValTimeEqual(time1, time2: T): Data.Bool =
  BEGIN
    RETURN
      (ValTimeSeconds(time1) = ValTimeSeconds(time2))
      AND (ValTimeMicroSeconds(time1) = ValTimeMicroSeconds(time2));
  END ValTimeEqual;

PROCEDURE ValTimeBefore(time1, time2: T): Data.Bool =
  BEGIN
    RETURN
      (ValTimeSeconds(time1) < ValTimeSeconds(time2))
      OR ((ValTimeSeconds(time1) = ValTimeSeconds(time2))
          AND (ValTimeMicroSeconds(time1) < ValTimeMicroSeconds(time2)));
  END ValTimeBefore;

PROCEDURE ValTimeAfter(time1, time2: T): Data.Bool =
  BEGIN
    RETURN
      (ValTimeSeconds(time1) > ValTimeSeconds(time2))
      OR ((ValTimeSeconds(time1) = ValTimeSeconds(time2))
          AND (ValTimeMicroSeconds(time1) > ValTimeMicroSeconds(time2)));
  END ValTimeAfter;

PROCEDURE NewValDynamic(type, value: T): T =
  VAR pointer, sizePointer: Data.Pointer;
  BEGIN
    Store.Align();
    Store.LayPointer(dynamicFormat);
    sizePointer := Store.hp;
    Store.LayInt(0); (* size *)
    pointer := Store.hp;
    Store.LayPointer(type);
    Store.LayPolymorph(value);
    Store.SetInt(sizePointer, Store.hp - pointer);
    Store.Align();
    RETURN pointer;
  END NewValDynamic;

PROCEDURE ValDynamicType(dynamic: T): T =
  BEGIN RETURN Store.GetPointer(dynamic); END ValDynamicType;

PROCEDURE ValDynamicValue(dynamic: T): T =
  BEGIN
    RETURN
      Store.GetPolymorph(dynamic + Data.PointeesPerPointer (* type *));
  END ValDynamicValue;

PROCEDURE ValDynamicIntern(reader: T; VAR (*out*) dynamic: T): BOOLEAN =
  VAR hp: Data.Pointer;
  BEGIN
    IF Format.Intern(FetchReader(ValReaderIndex(reader)), (*out*) hp) THEN
      dynamic := hp;
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END ValDynamicIntern;

PROCEDURE ValDynamicExtern(writer: T; dynamic: T): BOOLEAN =
  BEGIN
    RETURN Format.Extern(FetchWriter(ValWriterIndex(writer)), dynamic);
  END ValDynamicExtern;

PROCEDURE ValDynamicInternPortable(reader: T; VAR (*out*) dynamic: T): BOOLEAN =
  VAR hp: Data.Pointer;
  BEGIN
    IF Format.InternPortable(FetchReader(ValReaderIndex(reader)), 
         (*out*) hp) THEN
      dynamic := hp;
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END ValDynamicInternPortable;

PROCEDURE ValDynamicExternPortable(writer: T; dynamic: T): BOOLEAN =
  BEGIN
    RETURN Format.ExternPortable(FetchWriter(ValWriterIndex(writer)), dynamic);
  END ValDynamicExternPortable;

(* --
PROCEDURE OpenRPC(VAR (*out*) pos: INTEGER): BOOLEAN;
  VAR i: INTEGER;
  BEGIN
    FOR i := RPCServer + 1 TO RPCTableSize - 1 DO
      IF NOT rpcTable[i].inUse THEN
        rpcTable[i].inUse := TRUE;
        pos := i;
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END OpenRPC;

PROCEDURE ValRPCImport(class, instance, localOnly: T; VAR rpc: T): BOOLEAN;
  VAR classText, instanceText: Text.T; pos: INTEGER;
  BEGIN
    classText :=
      Text.FromSub(Store.heap, ValStringStart(class),
        ValStringLength(class));
    IF ValStringLength(instance) = 0 THEN
      instanceText := NIL;
    ELSE
      instanceText :=
        Text.FromSub(Store.heap, ValStringStart(instance),
          ValStringLength(instance));
    END;
    IF OpenRPC(pos)
       AND QRPC.Import(classText, instanceText,
             Data.BoolOfImmediate(localOnly), rpcTable[pos].rpc) THEN
      rpc := NewValInt(pos);
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END ValRPCImport;

PROCEDURE ValRPCClose(rpc: T);
  BEGIN rpcTable[Data.IntOfImmediate(rpc)].inUse := FALSE; END ValRPCClose;

PROCEDURE ValRPCInitCall(rpc: T): BOOLEAN;
  BEGIN
    RETURN QRPC.InitCall(rpcTable[Data.IntOfImmediate(rpc)].rpc);
  END ValRPCInitCall;

PROCEDURE ValRPCSendCall(rpc: T): BOOLEAN;
  BEGIN
    RETURN QRPC.SendCall(rpcTable[Data.IntOfImmediate(rpc)].rpc);
  END ValRPCSendCall;

PROCEDURE ValRPCEndCall(rpc: T): BOOLEAN;
  BEGIN
    RETURN QRPC.EndCall(rpcTable[Data.IntOfImmediate(rpc)].rpc);
  END ValRPCEndCall;

PROCEDURE ValRPCExport(class, instance, localOnly: T): BOOLEAN;
  VAR instanceText, classText: Text.T;
  BEGIN
    classText :=
      Text.FromSub(Store.heap, ValStringStart(class),
        ValStringLength(class));
    IF ValStringLength(instance) = 0 THEN
      instanceText := NIL;
    ELSE
      instanceText :=
        Text.FromSub(Store.heap, ValStringStart(instance),
          ValStringLength(instance));
    END;
    RETURN
      QRPC.Export(classText, instanceText, Data.BoolOfImmediate(localOnly));
  END ValRPCExport;

PROCEDURE ValRPCGetRequest(): T;
  BEGIN
    QRPC.GetRequest(rpcTable[RPCServer].rpc);
    RETURN NewValInt(RPCServer);
  END ValRPCGetRequest;

PROCEDURE ValRPCStartResponse(rpc: T);
  BEGIN QRPC.StartResponse(rpcTable[RPCServer].rpc); END ValRPCStartResponse;

PROCEDURE ValRPCEndResponse(rpc: T);
  BEGIN QRPC.EndResponse(rpcTable[RPCServer].rpc); END ValRPCEndResponse;

PROCEDURE ValRPCPut(object: SYSTEM.ADDRESS; length: INTEGER; rpc: T): BOOLEAN;
  BEGIN
    RETURN QRPC.Put(object, length, rpcTable[Data.IntOfImmediate(rpc)].rpc);
  END ValRPCPut;

PROCEDURE ValRPCPutDynamic(rpc: T; dynamic: T): BOOLEAN;
  BEGIN
    RETURN
      Format.ExternMarshal(rpcTable[Data.IntOfImmediate(rpc)].rpc,
        dynamic);
  END ValRPCPutDynamic;

PROCEDURE ValRPCPutLocalSpaceId(rpc: T): BOOLEAN;
  BEGIN
    RETURN QRPC.PutLocalSpaceId(rpcTable[Data.IntOfImmediate(rpc)].rpc);
  END ValRPCPutLocalSpaceId;

PROCEDURE ValRPCGet(object: SYSTEM.ADDRESS; length: INTEGER; rpc: T): BOOLEAN;
  BEGIN
    RETURN QRPC.Get(object, length, rpcTable[Data.IntOfImmediate(rpc)].rpc);
  END ValRPCGet;

PROCEDURE ValRPCGetDynamic(rpc: T; VAR (*out*) dynamic: T): BOOLEAN;
  BEGIN
    RETURN
      Format.InternMarshal(rpcTable[Data.IntOfImmediate(rpc)].rpc,
        dynamic);
  END ValRPCGetDynamic;
*)

(* -- PROCEDURE NewValType(): T; VAR pointer, sizePointer: Data.Pointer;
   BEGIN Store.Align(); Store.LayPointer(typeFormat); sizePointer :=
   Store.hp; Store.LayInt(0); (* size *) pointer := Store.hp;
   Store.LayPointer(Data.MinPointer); Store.SetInt(sizePointer, Store.hp
   - pointer); RETURN pointer; END NewValType; *)

(* PROCEDURE ValTypePolymorph(type: Data.Pointer): Data.Immediate; BEGIN
   RETURN Store.GetImmediate(type); END ValTypePolymorph; *)

PROCEDURE NewValStack(sizeInPolymorphs: Data.Int): T =
  VAR array: T;
  BEGIN
    IF NOT (NewValArray(sizeInPolymorphs, Data.MinPointer, (*out*) array))
    THEN
      Data.Fault("")
    END;
    RETURN array;
  END NewValStack;

PROCEDURE NewValLevelStack(sizeInPolymorphs: Data.Int): T =
  VAR array: T;
  BEGIN
    IF NOT (NewValArray(sizeInPolymorphs, Data.MinPointer,
              (*out*) array)) THEN
      Data.Fault("")
    END;
    RETURN array;
  END NewValLevelStack;

PROCEDURE Same(val1, val2: T): BOOLEAN = BEGIN RETURN val1 = val2; END Same;

PROCEDURE ValStringOfChar(ch: CHAR): T =
  VAR val: T;
  BEGIN
    IF NOT (NewValString(1, Data.MinPointee, (*out*) val)) THEN
      Data.Fault("")
    END;
    IF NOT (ValStringSetPointee(val, 0, LOOPHOLE(ch, Data.Pointee))) THEN
      Data.Fault("")
    END;
    RETURN val;
  END ValStringOfChar;

PROCEDURE ValStringOfChars(chars: TEXT): T =
  VAR length: INTEGER; val: T;
  BEGIN
    length := Text.Length(chars);
    IF NOT (NewValString(length, Data.MinPointee, (*out*) val)) THEN
      Data.Fault("")
    END;
    FOR i := 0 TO length - 1 DO
      IF NOT (ValStringSetPointee(val, i, 
	  LOOPHOLE(Text.GetChar(chars, i), Data.Pointee))) THEN
        Data.Fault("")
      END;
    END;
    RETURN val;
  END ValStringOfChars;

PROCEDURE PrintValString(wr: QOS.Writer; string: T) =
  VAR length: INTEGER; pointee: Data.Pointee;
  BEGIN
    QOS.OutChar(wr, '\"');
    (* -- escape sequences *)
    length := ValStringLength(string);
    FOR i := 0 TO length - 1 DO
      IF NOT (ValStringGetPointee(string, i, (*out*) pointee)) THEN
        Data.Fault("")
      END;
      QOS.OutChar(wr, LOOPHOLE(pointee, CHAR));
    END;
    QOS.OutChar(wr, '\"');
  END PrintValString;

PROCEDURE PrintValue(wr: QOS.Writer; value: T) =
  BEGIN QOS.OutChar(wr, '?'); END PrintValue;

PROCEDURE Setup() =
  VAR i: INTEGER;
  BEGIN

    stringFormat :=
      Format.New(
        Format.NewIter(Data.PointeesPerInt, Format.pointee));

    badgeFormat :=
      Format.New(
        Format.NewSeq(
          Format.NewList(Format.pointer,
            Format.NewList(Format.pointer,
              Format.NewList(Format.immediate,
                Format.NewList(Format.immediate,
                  Format.NewList(Format.immediate,
                    Format.NewList(Format.immediate,
                      Format.NewList(Format.immediate,
                        Format.emptyList)))))))));

    literalsFormat :=
      Format.New(
        Format.NewIter(Data.PointeesPerInt, Format.pointer));

    progFormat :=
      Format.New(
        Format.NewSeq(
          Format.NewList(Format.pointer,
            Format.NewList(
              Format.NewIter(Data.PointeesPerInt, Format.pointee),
              Format.emptyList))));

    FOR i := 0 TO ClosureFormatCache - 1 DO
      closureFormat[i] := ClosureFormat(i);
    END;

    FOR i := 0 TO TupleFormatCache - 1 DO
      tupleFormat[i] := TupleFormat(i);
    END;

    optionFormat :=
      Format.New(
        Format.NewSeq(
          Format.NewList(Format.smallInt,
            Format.NewList(Format.polymorph, Format.emptyList))));

    arrayFormat :=
      Format.New(
        Format.NewIter(Data.PointeesPerInt, Format.polymorph));

    readerFormat := Format.New(Format.int);

    writerFormat := Format.New(Format.int);

    timeFormat := Format.New(Format.NewRepeat(2, Format.int));

    dynamicFormat :=
      Format.New(
        Format.NewSeq(
          Format.NewList(Format.pointer,
            Format.NewList(Format.polymorph, Format.emptyList))));

    typeFormat := Format.New(Format.pointer);

    excPacketFormat :=
      Format.New(
        Format.NewSeq(
          Format.NewList(Format.pointer,
            Format.NewList(Format.polymorph, Format.emptyList))));

    FOR i := 0 TO ReaderTableSize - 1 DO
      readerTable[i].inUse := FALSE;
      readerTable[i].rd := QOS.stdin;
    END;
    IF OpenReader(QOS.stdin, (*out*) i) THEN END;
    inputReader := NewValReader(i);

    FOR i := 0 TO WriterTableSize - 1 DO
      writerTable[i].inUse := FALSE;
      writerTable[i].wr := QOS.stdout;
    END;
    IF OpenWriter(QOS.stdout, (*out*) i) THEN END;
    outputWriter := NewValWriter(i);

    (* RPC: FOR i := 0 TO RPCTableSize - 1 DO rpcTable[i].inUse := FALSE; END; *)

    valTimeZero := NewValTime(0, 0);

    typeNone := Data.MinPointer;

  END Setup;

BEGIN
END Value.
