(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)
(* Last modified on Mon Jun  4 02:34:58 1990 by luca           *)

interface ByteCode
import nester :Nester value :Value byteArray :ByteArray
export

(* Bytecode for the Quest Machine (QM) *)

(*
|  [ .. ]	pointee
|  [[ .. ]]	smallInt = [ .. ] [ .. ]
|  [[[ .. ]]]	int = [ .. ] [ .. ] [ .. ] [ .. ]
|  < .. > 	address
|  << .. >> 	address list = [[ .. ]] < .. > .. < .. >
|  { .. } 	target     (relative to beginning of target)
|  {{ .. }} 	target list = [[ .. ]] { .. } .. { .. }
|
|  [[OpMove]] <moveSrc> <moveDst>
|  [[OpFrame]] [[frameSize]] [[frameTempNo]]
|  [[OpApply]] <applyFun>
|  [[OpReturn]] [[returnFrameSize]]
|  [[OpArguments]] [[argumentsResNo]] <<argumentSrc>>
|  [[OpResults]] <<resultsDst>>
|  [[OpClosure]] <closureProg> <<closureGlobals>> <clousureDst>
|  [[OpDumClosure]] <dumClosurProg><dumClosureSize><dumClosureDst>
|  [[OpRecClosure]] <recClosureSrc> <<recClosurGlobals>>
|  [[OpTuple]] <<tupleSrc>> <tupleDst>
|  [[OpJumpAlways]] {target}
|  [OpJumpWhen] [condition] <addr1> <addr2> {target}
|  [[OpCaseCeck]] <caseSrc> <index>
|  [[OpCase]] <caseSrc> {{caseBranchTargets}}
|  [[OpCaseFault]]
|  [[OpArray]] <<arraySrc>> <arrayDst>
|  [[OpTrap]] <trapFrame> {target}
|  [[OpUntrap]]
|  [[OpRaise]] <raiseSel> <raiseSrc>
|  [[OpUnwind]]
|  [[OpCrash]]
|  [[OpEnd]] (* -- obsolete? *)
|  [[OpStart]] <machineStateSrc> <(suspended)machineStateDst>
|  [[OpStop]]
|  [[OpNoOp]]
|  [OpData...] [...] ...
|  (* optimizations *)
|  [[OpApplyArguments0Results1]] <applyFun>
|  [[OpApplyArguments1Results1]] <applyFun> <argumentSrc>
|  [[OpApplyArguments2Results1]] <applyFun> <argument1Src> <argument2Src>
|  [[OpApplyArgumentsResults1]] <applyFun> <<argumentsSrc>>
|  [[OpResults1]] <resultsDst>
|
|  [OpAddrImmedOk] []
|  [OpAddrImmedBool] [bool]
|  [OpAddrImmedChar] [char]
|  [OpAddrImmedInt2] [] [[smallInt]]
|  [OpAddrImmedInt4] [] [[[int]]]
|  [OpAddrImmedReal] [] [[[real]]]
|  [OpAddrLiteral2] [] [[smallInt]]
|  [OpAddrFrame2] [] [[smallInt]]
|  [OpAddrFrame2Locative] [] [[smallInt]]
|  [OpAddrGlobal2] [] [[smallInt]]
|  [OpAddrIndexed2] [] <addr> [[smallInt]]
|  [OpAddrStackDispl] [] <addr>
|  [OpAddrSpecial] [special]
*)

  Def Code = 
    byteArray.T
  (* A bytecode sequence, as described above. *)

  Def Rec Prog =
    Tuple 
      code :Code 
      literals :LitList 
    end
  (* Reflects Literal_Prog, except that Code here is a byteArray. *)
  (* This is the same structure as a Machine_Prog, which is 
     accepted by machine.exec, which converts a Prog into an 
     appropriate run-time structure. *)

  and Lit =
    Option
      litStringCase with string :String end
      litBadgeCase with badge :Badge end
      litProgCase with prog :Prog end
      litTypeCase
      litValueProgCase with prog :Prog end
      litValueCase with val :value.T end
    end
  (* Reflects Literal_T. *)

  and Badge =
    Tuple
      moduleName,progName :String
      machineId,processId :Int
      timeSecs,timeMicroSecs,timeSerialNo :Int
    end
  (* Reflects Literal_Badge. *)

  and LitList =
    Option
      nil
      cons with first :Lit rest :LitList end
    end
  (* Reflects Literal_TList. *)

  newProg(code :Code literals :LitList) :Prog
  (* Create a new Prog. *)

  emptyLitList :LitList
  newLitList(first :Lit rest :LitList) :LitList
  (* Create LitList's. *)

  newLitString(string :String) :Lit
  newLitBadge(badge :Badge) :Lit
  newLitProg(prog :Prog) :Lit
  newLitType() :Lit
  newLitValueProg(prog :Prog) :Lit
  newLitValue(val :value.T) :Lit
  (* Create various literals. *)

  printProg(
    wr :nester.T
    prog :Prog
    fromLiteral :Int
    indent :String)
    : Ok
  (* Print a program (for debugging). *)

end;
