(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)
(* Last modified on Mon Jun  4 01:31:27 1990 by luca           *)

interface Comp
import :Msg ident :Ident :Term :Exp :Top :Code :Literal
export

  (* Data stuctures for the translation of abstract syntax trees
    into Quest Machine instructions. CompCode keeps all the
    compile-time information about a piece of code. *)

  Def Rec CompAddrList =
    Option
      nil
      cons with
	ide :ident.T
	modif :Bool
	type :Term_TypExp
	var addr :Code_OpAddr
	var rest :CompAddrList
      end
    end
  (* Identifier info. Name, type, whether modifiable, address. *) 

  Def CompLocal =
    Tuple
      var items :CompAddrList
      var argumentNo, resultNo, temporaryNo :Int
    end
  (* List of local identifiers for a program (frame), either 
     Arguments, Results, or Temporaries. *)

  Def CompGlobal =
    Tuple var items :CompAddrList var globalNo :Int end
  (* List of global idenfiers for a program. *)

  Def CompLiteral = 
    Tuple var items :Literal_TList var literalNo :Int end
  (* List of literals for a program. *)

  Def ProgMap =
    Tuple var objPos,srcBegPos,srcEndPos :Array(Int) end
  (* Map from code positions into source positions. *)

  Def Rec ProgInfo =
    Option
      nil
      cons with
        badge :Literal_Badge
        local,global :CompAddrList
        literal :Literal_TList
        map :ProgMap
        rest :ProgInfo
      end
    end
  (* Program information strored in ".info.x" files. *)

  Def CompCode = 
    Tuple
      badge :Literal_Badge
      var first, last :Code_T
      local :CompLocal 
      global :CompGlobal
      literal :CompLiteral
      var collectiveProgInfo :ProgInfo
      var labelCounter :Int
      var patchList :Code_PatchList
      var currentDefName :String
    end
  (* Temporary data structure for compilation. *)

  Def CompRecInfo =
    Option
      valExpNoneCase
      valExpFunCase with funFree :CompGlobal end
      (* other constructors... (tuples etc.) *)
    end
  (* Temporary data structure for recursive definitions. *)

  Def Rec CompRecInfoList =
    Option
      nil
      cons with first :CompRecInfo rest :CompRecInfoList end
    end
  (* List of the above. *)

  emptyCompAddrList :CompAddrList
  newCompAddrList(
    ide :ident.T
    modif :Bool
    type :Term_TypExp
    addr :Code_OpAddr 
    rest :CompAddrList) 
    :CompAddrList
  (* Create CompAddrList's. *)

  compAddrListFetch(
    ide :ident.T 
    list :CompAddrList 
    out modif :Bool
    out type :Term_TypExp
    out addr :Code_OpAddr) 
    :Bool
  (* Fetch identifier attributes in a CompAddrList. *)

  newCompLocal() :CompLocal
  newCompGlobal() :CompGlobal
  newCompLiteral() :CompLiteral
  newCompCode(
    badge :Literal_Badge
    local :CompLocal 
    global :CompGlobal 
    literal :CompLiteral
    collectiveProgInfo :ProgInfo) 
    :CompCode
  (* Initial complation structures. *)

  emptyProgInfo :ProgInfo
  newProgInfo(
    badge :Literal_Badge
    local,global :CompAddrList
    literal :Literal_TList
    rest :ProgInfo)
    :ProgInfo
  (* Create ProgInfo's. *)

  getProgInfo(badge :Literal_Badge progInfo :ProgInfo) :ProgInfo
  (* Get the ProgInfo item identified by badge out of a ProgInfo list. *)

  emptyRecInfo :CompRecInfo
  newFunRecInfo(funFree :CompGlobal) :CompRecInfo
  (* Create CompRecInfo's. *)

  emptyCompRecInfoList :CompRecInfoList
  newCompRecInfoList(
    first :CompRecInfo
    rest :CompRecInfoList)
    :CompRecInfoList
  (* Create CompRecInfoList's. *)

  addLiteralSlot(lit :Literal_T literals :CompLiteral) :Code_OpAddr
  (* Add literal slot to literals and return its address. *)

  addLiteralSharedSlot(lit :Literal_T literals :CompLiteral) :Code_OpAddr
  (* Same as above. Maybe used in the future to share identical literals. *)

  addTemporarySlot(local :CompLocal) :Code_OpAddr
  (* Add a nameless temporary slot to local and return its address. *)

  addTemporaryDelayedSlot(local :CompLocal) :Code_OpAddr
  (* Add a nameless temporary delayed slot to local and return its address. *)

  addTemporaryIde(
    ide :ident.T 
    modif :Bool
    type :Term_TypExp
    local :CompLocal) 
    :Code_OpAddr
  (* Add a temporary slot to local and return its address. *)

  addTemporaryDelayedIde(
    ide :ident.T 
    modif :Bool
    type :Term_TypExp
    local :CompLocal) 
    :Code_OpAddr
  (* Add a delayed temporary slot to local and return its address. *)

  addArgumentIde(
    ide :ident.T 
    modif :Bool
    type :Term_TypExp
    local :CompLocal) 
    :Code_OpAddr
  (* Add an argument slot to local and return its address. *)

  addResultSlot(
    type :Term_TypExp
    local :CompLocal) 
    :Code_OpAddr
  (* Add a (the) result slot to local and return its address. *)

  setDefName(ide :ident.T buff :CompCode) :Ok
  (* Remember the current function name. *)

  moveToHome(addr :Code_OpAddr literals :Literal_TList) :Literal_Prog
  (* Produce a program to move addr to home and stop. *)

  moveFromHome(addr :Code_OpAddr literals :Literal_TList) :Literal_Prog
  (* Produce a program to move home into addr and stop. *)

  localFetch(
    ide :ident.T 
    local :CompLocal
    out modif :Bool
    out type :Term_TypExp) 
    :Code_OpAddr
  (* Get the attributes of a local identifier or raise error. *)

  globalFetchOrAdd(
    ide :ident.T
    type :Term_TypExp
    global :CompGlobal) 
    :Code_OpAddr
  (* Get the address of an existing global identifier, or if not found
     add a new global slot for that identifier. *)

  frameSize(buff :CompCode) :Int
  (* The run-time size of a frame for this program. *)

  emit(instr :Code_Instr where :Msg_Where buff :CompCode) :Ok
  (* Append an instruction to the current program. *)

  emitLabel(buff :CompCode) :Code_T
  (* Append a label to the current program; return the code starting with it. *)

  backpatch(patch :Code_Patch buff :CompCode) :Ok
  (* Append a label to the current program and store it into patch for
     future backpatching at that address. *)

  fixCode(buff :CompCode) :Ok
  (* Close up code generation: resolve the delayed addresses and
     generate frame displacements for temporaries, arguements and
     results. *)
     
  extractProg(
    buff :CompCode 
    pruneLocalAnon: Bool 
    out collectiveProgInfo :ProgInfo)
    :Literal_Prog
  (* Obtain a final Literal_Prog from the temporary compilation
     data structures (after running fixCode). Also produce a ProgInfo 
     for ".info.x" files. If pruneLocalAnon, remove the info for all
     the anonimous local identifiers. *)

  freshCompCode(
    moduleName,progName :String
    collectiveProgInfo :ProgInfo) 
    :CompCode
  (* Create an initial CompCode for compilation. *)

end;
