(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)
(* Last modified on Mon Jun  4 03:12:59 1990 by luca           *)

interface Nester
import writer:Writer
export

(* Translated into Quest from code written by Greg Nelson (CGN), and 
   modified by John DeTreville and John Ellis. *)

(* Formatted streams of characters and formatting codes

   A formatted stream is a stream containing a mixture of character codes and
   formatting codes that group the characters into logical objects and specify
   desirable places to break objects into lines. This module contains
   procedures for creating formatted streams and sending them characters and
   formatting codes.

   For a full explanation of the meaning of the codes, see CGN8 and CGN9.
   There are also brief comments below nest, unnest, and break
   which may be helpful. *)

  T::TYPE

  Def Parameters = 
    Tuple
      destination:writer.T
      margin:Int
      flushFrequently:Bool
    end
  (* If flushFrequently is TRUE, the formatted stream will call writer.flush on
     the destination after every output line. *)

  error:Exception(Ok)

  new(p:Parameters):T
  (* May raise writer.error *)
  (* Set f := a newly allocated formatted stream whose output characters 
     will be sent to p.destination, and whose margin property is p.margin. *)

  setParameters(f:T p:Parameters):Ok
  (* May raise nothing *)
  (* Set the parameters of f to  p. *)

  getParameters(f:T):Parameters
  (* May raise nothing *)
  (* Returns the parameters of f. *)

  putChar(f:T c:Char):Ok
  (* May raise writer.error *)
  (* Add the character c to the stream f. If c = newline, this
    is the same as a forcedBreak (see below) with offset 0 *)

  putString(f:T s:String):Ok
  (* May raise writer.error *)
  (* Add the characters of s in order to the stream f. *)

  nest(f:T offset:Int):Ok
  (* May raise writer.error *)
  (* Mark the beginning of a logical object, and the standard indentation
   increment for the object. That is, offset will be added to the offset of 
   any breakpoints of the object that get broken. *)

  unnest(f:T):Ok
  (* May raise error writer.error *)
  (* Mark the end of a logical object. Calls to nest and unnest
     should be properly nested; an unmatched unnest will raise the 
     exception error. *)

  break(f:T offset:Int forced:Bool):Ok
  (* May raise writer.error *)
  (* Mark a breakpoint (a desirable place to break the current object). If a
   linebreak does occur at this breakpoint, the indentation of the new line
   will exceed the indentation of the first character of the current logical
   object by offset spaces. (Unless this rule would produce a negative
   indentation, in which case the new line's indentation is zero.) If forced =
   TRUE, then a linebreak will be forced at this breakpoint if a linebreak
   occurs anywhere within the current object; if forced = FALSE, a 
   linebreak will occur at this breakpoint only if necessary in order to 
   fit the object within the margin. *)

  forcedBreak(f:T offset:Int):Ok
  (* May raise writer.error *)
  (* Inserts a breakpoint with the given offset that is always broken
   even if the object containing it would fit on a line. *)

  flush(f:T):Ok
  (* May raise writer.error *)
  (* The implementation of an nester.T contains a buffer of format codes that
   have been output by the client but not yet delivered to the output stream
   by the implementation. This buffer is necessary because it is impossible to
   determine whether a breakpoint should be broken until well after the
   breakpoint is output. The effect of flush(f) is (1) to output a
   forcedBreak to f, i.e., a breakpoint that is always broken,
   thus establishing a state in which there is no ambiguity about the format
   of the buffer contents, and (2) to empty the buffer. *)

end;
