(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)
(* Last modified on Fri Jun  1 10:21:58 1990 by luca           *)

interface Scan
import reader:Reader writer:Writer symbol:Symbol
export

  Def TokenClass::TYPE =
    Option 
      ideCase infixCase charCase intCase realCase
      stringCase delimCase
    end 

  finish:Exception(Ok)
    (* End of input stream exception. *)

  table:symbol.Table
    (* Table for parser keywords *)
  keySymbol,ideSymbol:Symbol_Kind
    (* Kinds of parser keywords *)

  clear():Ok
    (* Clean up the scanner state, but preserve the input file queue. *)

  reset():Ok
    (* Reinitialize the scanner state. *)

  (* === Source === *)

  pushInput(fileName:String rd:reader.T):Ok
    (* Switch the scanner input to a new reader. *)

  popInput():Ok
    (* Switch back to the previous reader. Raises finish. *)

  begWhere(out fileName:String out begPos:Int):Ok
    (* Get the fileName associated with the current input reader. 
       Skip blanks and return the charNo of the first non-blank char. *)

  endWhere(out endPos:Int):Ok
    (* Get the charNo of the first blank char following the last accepted 
       token. (Accepted means obtanied by "Get" routines, not by "Look" 
       routines.) *)

  setCharNo(charNo:Int):Ok
    (* Set the charNo counter associtated with the current input reader to the
       given number, for error reporting purposes. The charNo is initialized 
       to 0 and incremented every time a new character is scanned *)

  topLevel():Bool
    (* Whether the scanner is reading from the top level or a file. *)

  flushInput():Ok
    (* Flush the pending input (for top-level use). *)

  setPrompt(firstPrompt,nextPrompt:String):Ok
    (* Set the prompt strings. The firstPrompt will be generated once,
       then the nextPrompt all the other times, until firstPrompt is called. *)

  firstPrompt():Ok
    (* Reset the first prompt (see setPrompt). *)

  (* === Error Messages === *)

  printContext():Ok
    (* Print the current syntactic context *)

  error(A::TYPE str:String):A
    (* Print str (followed by newline if non-empty), flush the input (if
       top-level), reset the scanner, and raise msg.fail. *)

  syntax(A::TYPE cause,culprit:String):A
    (* A more elaborate version of "error". It uses printLocation and
       printContext to show the error context. *)

  (* === Generic token primitives === *)

  lookToken():TokenClass
    (* Returns the TokenClass of the next token without skipping it *)

  getToken():TokenClass
    (* Returns the TokenClass of the next token and skips it *)

  (* === Specific token primitives === *)

  (* ~~~ Char Tokens ~~~ *)

  getTokenChar():Char
    (* Returns the value of the next (char) token (or syntax error) *)

  (* ~~~ Natural Tokens ~~~ *)

  getTokenNat():Int
    (* Returns the value of the next (natural) token (or syntax error) *)

  (* ~~~ Integer Tokens ~~~ *)

  getTokenInt():Int
    (* Returns the value of the next (integer) token (or syntax error) *)

  (* ~~~ Real Tokens ~~~ *)

  getTokenReal():Real
    (* Returns the value of the next (real) token (or syntax error) *)

  (* ~~~ String Tokens ~~~ *)

  getTokenString():String
    (* Returns the value of the next (string) token (or syntax error) *)

  (* ~~~ Identifier Tokens ~~~ *)

  lookTokenIde():String
    (* Returns the value of the next identifier as a text without skipping it.
       Syntax error if the next token is not an identifier *)

  getTokenIde():String
    (* Returns the value of the next identifier as a text and skips it. Syntax
       error if the next token is not an identifier *)

  testLookTokenIde(ide:String):Bool
    (* Tests the presence of a given identifier without skipping it if found. 
      No syntax errors generated *)

  testGetTokenIde(ide:String):Bool
    (* Tests the presence of a given identifier and skips it if found. 
      No syntax errors generated *)

  haveTokenIde(ide:String):Ok
    (* Requires the presence of a given identifier. Syntax error if not 
       found *)

  (* ~~~ Symbol Tokens ~~~ *)

  lookTokenSym():symbol.T
    (* Returns the value of the next identifier as a symbol without skipping 
       it. Syntax error if the next token is not an identifier *)

  getTokenSym():symbol.T
    (* Returns the value of the next identifier as a symbol and skips it. 
       Syntax error if the next token is not an identifier *)

  testLookTokenSym(sym:symbol.T):Bool
    (* Tests the presence of a given identifier without skipping it if found. 
      No syntax errors generated *)

  testGetTokenSym(sym:symbol.T):Bool
    (* Tests the presence of a given identifier and skips it if found. 
      No syntax errors generated *)

  haveTokenSym(sym:symbol.T):Ok
    (* Requires the presence of a given identifier. Syntax error if not 
       found *)

  (* ~~~ Delimiter Tokens ~~~ *)

  lookTokenDelim():Char
    (* Returns the value of the next delimiter without skipping it. 
       Syntax error if the next token is not a delimiter *)

  getTokenDelim():Char
    (* Returns the value of the next delimiter and skips it. Syntax error 
       if the next token is not a delimiter *)

  testLookTokenDelim(delim:Char):Bool
    (* Tests the presence of a given delimiter without skipping it if found. 
       No syntax errors generated *)

  testGetTokenDelim(delim:Char):Bool
    (* Tests the presence of a given delimiter and skips it if found. No syntax
       errors generated *)

  haveTokenDelim(delim:Char):Ok
    (* Requires the presence of a given delimiter. Syntax error if not found *)

end;
