(* We implement properties with association lists.  The top-level property
   list, attached to the VBT.T, contains alists for each type.  Each alist
   associates TEXTs with values or REFs to values.

   Certain property-names are pre-defined, namely the state properties. *)

TYPE
  PropList =
    REF RECORD
      refany, text, integer, real, longreal, boolean: List.T := NIL
    END;

PROCEDURE GetIntegerProperty (fv: T; name, property: TEXT): INTEGER
  RAISES {Error, Unimplemented} =
  VAR
    vbt                := GetVBT (fv, name);
    alist: PropList    := VBT.GetProp (vbt, TYPECODE (PropList));
    ri   : REF INTEGER;
  BEGIN
    IF alist # NIL THEN
      WITH pair = List.Assoc (alist.integer, property) DO
        IF pair # NIL THEN ri := pair.tail.first; RETURN ri^ END
      END
    END;
    RAISE Error (name & " has no integer property named " & property)
  END GetIntegerProperty;
   
PROCEDURE PutIntegerProperty (fv: T; name, property: TEXT; num: INTEGER)
  RAISES {Error, Unimplemented} =
  VAR
    vbt             := GetVBT (fv, name);
    alist: PropList := VBT.GetProp (vbt, TYPECODE (PropList));
    ri              := Sx.NewInteger (num);
  BEGIN
    IF alist = NIL THEN
      VBT.PutProp (
        vbt,
        NEW (PropList, integer := List.List1 (List.List2 (property, ri))))
    ELSE
      WITH pair = List.Assoc (alist.integer, property) DO
        IF pair # NIL THEN
          List.Push (alist.integer, List.List2 (property, ri))
        ELSE
          pair.tail.first := ri
        END
      END
    END
  END PutIntegerProperty;

PROCEDURE GetRealProperty (fv: T; name, property: TEXT): REAL
  RAISES {Error, Unimplemented} =
  VAR
    vbt             := GetVBT (fv, name);
    alist: PropList := VBT.GetProp (vbt, TYPECODE (PropList));
    ri   : REF REAL;
  BEGIN
    IF alist # NIL THEN
      WITH pair = List.Assoc (alist.real, property) DO
        IF pair # NIL THEN ri := pair.tail.first; RETURN ri^ END
      END
    END;
    RAISE Error (name & " has no real property named " & property)
  END GetRealProperty;
   
PROCEDURE PutRealProperty (fv: T; name, property: TEXT; num: REAL)
  RAISES {Error, Unimplemented} =
  VAR
    vbt             := GetVBT (fv, name);
    alist: PropList := VBT.GetProp (vbt, TYPECODE (PropList));
    ri              := Sx.NewReal (num);
  BEGIN
    IF alist = NIL THEN
      VBT.PutProp (vbt,
                   NEW (PropList,
                        real := List.List1 (List.List2 (property, ri))))
    ELSE
      WITH pair = List.Assoc (alist.real, property) DO
        IF pair # NIL THEN
          List.Push (alist.real, List.List2 (property, ri))
        ELSE
          pair.tail.first := ri
        END
      END
    END
  END PutRealProperty;
  
PROCEDURE GetBooleanProperty (fv: T; name, property: TEXT): BOOLEAN
  RAISES {Error, Unimplemented} =
  VAR
    vbt                := GetVBT (fv, name);
    alist: PropList    := VBT.GetProp (vbt, TYPECODE (PropList));
    ri   : REF BOOLEAN;
  BEGIN
    IF alist # NIL THEN
      WITH pair = List.Assoc (alist.boolean, property) DO
        IF pair # NIL THEN ri := pair.tail.first; RETURN ri^ END
      END
    END;
    RAISE Error (name & " has no boolean property named " & property)
  END GetBooleanProperty;
   
PROCEDURE PutBooleanProperty (fv: T; name, property: TEXT; val: BOOLEAN)
  RAISES {Error, Unimplemented} =
  VAR
    vbt             := GetVBT (fv, name);
    alist: PropList := VBT.GetProp (vbt, TYPECODE (PropList));
    ri              := Sx.NewBoolean (val);
  BEGIN
    IF alist = NIL THEN
      VBT.PutProp (
        vbt,
        NEW (PropList, boolean := List.List1 (List.List2 (property, ri))))
    ELSE
      WITH pair = List.Assoc (alist.boolean, property) DO
        IF pair # NIL THEN
          List.Push (alist.boolean, List.List2 (property, ri))
        ELSE
          pair.tail.first := ri
        END
      END
    END
  END PutBooleanProperty;

PROCEDURE GetRefProperty (fv: T; name, property: TEXT): REFANY
  RAISES {Error, Unimplemented} =
  VAR
    vbt             := GetVBT (fv, name);
    alist: PropList := VBT.GetProp (vbt, TYPECODE (PropList));
  BEGIN
    IF alist # NIL THEN
      WITH pair = List.Assoc (alist.refany, property) DO
        IF pair # NIL THEN RETURN pair.tail.first END
      END
    END;
    RAISE Error (name & " has no refany property named " & property)
  END GetRefProperty;
   
PROCEDURE PutRefProperty (fv: T; name, property: TEXT; val: REFANY)
  RAISES {Error, Unimplemented} =
  VAR
    vbt             := GetVBT (fv, name);
    alist: PropList := VBT.GetProp (vbt, TYPECODE (PropList));
  BEGIN
    IF alist = NIL THEN
      VBT.PutProp (
        vbt,
        NEW (PropList, refany := List.List1 (List.List2 (property, val))))
    ELSE
      WITH pair = List.Assoc (alist.refany, property) DO
        IF pair # NIL THEN
          List.Push (alist.refany, List.List2 (property, val))
        ELSE
          pair.tail.first := val
        END
      END
    END
  END PutRefProperty;
