(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created 1990-01-19 by Jorge Stolfi                          *)
(*   based on Real.i3 by Eric Muller and Bill Kalsow           *)
(* Last modified on Tue Jun  2 13:38:38 PDT 1992 by stolfi     *)
(*      modified on Tue Feb 11 21:39:52 PST 1992 by muller     *)


INTERFACE ByteRGB;

(* RGB values coded as triples in [0..255].

   This package defines some standard types and operations on RGB
   values coded as triples of unsigned bytes [0..255].

   This data type should be used only for image encoding and other
   applications where a compact representation is important.
   When that is not the case, real-valued RGB.T's should be used instead.

   This interface does not assume any specific relation between
   byte values and light intensities.
   
   When (if) Modula-3 has generics, this interface will be useful.

   Index: RGB triples; images; color;
*)

TYPE 
  T = ARRAY [0..2] OF Byte;

  Byte = BITS 8 FOR [0..255];
  
CONST Default = T{0, 0, 0};

PROCEDURE New (value: T): REF T;
  (* 
    Allocates and initializes a new heap value *)

PROCEDURE NewArray (size: CARDINAL;  value: T := T{0,0,0}): REF ARRAY OF T;
  (* 
    Allocates a new array from the heap
    and initializes all its elements with the given value *)

PROCEDURE UntracedNew (value: T): UNTRACED REF T;
  (* 
    Allocates and initializes a new untraced value *)

PROCEDURE UntracedNewArray (
    size: CARDINAL;  
    value: T := T{0,0,0}
  ): UNTRACED REF ARRAY OF T;
  (* 
    Allocates a new untraced array from the heap
    and initializes all its elements with the given value *)

PROCEDURE Hash (a: T): INTEGER;
  (* 
    Returns a suitable hash value *)

TYPE Bytes = ARRAY [0..BYTESIZE(T)-1] OF Byte;

PROCEDURE ToBytes(x: T): Bytes;
(* A safe LOOPHOLE of /x/ into ARRAY OF BYTE. *)

END ByteRGB.

