INTERFACE M3Args;

(***************************************************************************)
(*                      Copyright (C) Olivetti 1989                        *)
(*                          All Rights reserved                            *)
(*                                                                         *)
(* Use and copy of this software and preparation of derivative works based *)
(* upon this software are permitted to any person, provided this same      *)
(* copyright notice and the following Olivetti warranty disclaimer are     *) 
(* included in any copy of the software or any modification thereof or     *)
(* derivative work therefrom made by any person.                           *)
(*                                                                         *)
(* This software is made available AS IS and Olivetti disclaims all        *)
(* warranties with respect to this software, whether expressed or implied  *)
(* under any law, including all implied warranties of merchantibility and  *)
(* fitness for any purpose. In no event shall Olivetti be liable for any   *)
(* damages whatsoever resulting from loss of use, data or profits or       *)
(* otherwise arising out of or in connection with the use or performance   *)
(* of this software.                                                       *)
(***************************************************************************)

(* This package provides argument decoding and help support for tools
that operate in either a Unix style command line environment, or in
a window based environment.  In particular, several tools can coexist
in a single program. Typical usage is:

In the initialisation code for the tool, place a call to 'New', an
record the tool handle.  E.g

MODULE MyTool;
..
BEGIN
  tool_g := M3Args.New("mytool", "My Wonderful Tool", "1.1");
  (* now register argument keywords, e.g. *)
  M3Args.RegisterFlag(tool_g, "PrintUnits", 
    "print name of units as they are compiled");
END MyTool.

In the action routines of the tool, call 'Find' to gather arguments
and then 'GetFlag, GetString, GetStringList' as appropriate.

In a main program which includes MyTool, always start with a call
to 'CheckHelp' to ensure that a simple call by the user for help works,
and then go about your business as usual.
*)


TYPE
  T <: REFANY;  (* tool handle *)

  Opt = {Optional, Required, Positional};

PROCEDURE New(
    toolName, toolDescription, version: TEXT;
    master := FALSE;
    ): T RAISES {};
(* Register a new tool, returning a handle for later use. 
   'toolName' is the Unix-like name for the tool, e.g. m3c.
      It is used to prefix messages generated by the Err module
      after a call to Find.
   'toolDescription' is a longer explanation output when help
      is requested, e.g. Modula-3 Compiler Front End.
   'version' is the version of the tool.
   'master' indicates whether this is a master of controlling tool.
   A master tool info will always be displayed first in the list.
*)

PROCEDURE SetMaster(t: T): T RAISES {};
(* set 't' to be the master tool, returning the previous master,
if any, else NIL. *)

PROCEDURE RegisterFlag(
    t: T;
    argName: TEXT;
    usage: TEXT) RAISES {};

PROCEDURE RegisterString(
    t:T;
    argName: TEXT;
    usage: TEXT;
    opt: Opt := Opt.Optional) RAISES {};

PROCEDURE RegisterStringList(
    t:T;
    argName: TEXT;
    usage: TEXT;
    opt: Opt := Opt.Optional) RAISES {};

(* Register either a flag (boolean) argument, single string or a stringlist 
argument name for tool 't'.  'usage' gets displayed in response to a call for 
help. If 'opt = Required', it is an error for the argument to be omitted, 
which will cause 'Find' to return 'FALSE'.  If 'opt = Positional', then
'opt = Optional' is also implied, and this and only this argument can be
given in, say, a command line without the keyword.  'argName' should
be in lower case because short forms of arguments are indicated by capitalised
letters. E.g. "PrintUnits" is interpreted as "pu" being a short form. 
Generally arguments are tool specific and some implementations may not permit
duplicates (but see MarkShared). *)

PROCEDURE MarkShared(argName: TEXT) RAISES {};
(* Indicates that this argument is to be shared amongst several tools.
The interpretation of this is implementation dependent, but it is guaranteed
to permit duplicate Register calls from multiple tools.  In a command
line implementation the same value will be given to all tools. 
*)

PROCEDURE Help(t: T; preamble := TRUE) RAISES {};
(* Gives help on tool 't', using 'usage' args passed to registration procs. 
If 'preamble = TRUE', general info on argument types precedes it. *)

PROCEDURE CheckHelp(display := TRUE): BOOLEAN RAISES {};
(* If the user has indicated, in whatever way the implementation of this
interface chooses, that help or tool identification is required, this 
procedure gives help on all registered tools, by calling 'Help(t)', 
for all 't'. It returns TRUE if help was requested.  If 'display=FALSE'
the calls on 'Help(t)' are suppressed, which enable a master tool to
subvert the behaviour.
*)

PROCEDURE HelpPreamble(t: T) RAISES {};
(*  Display the general info on argument types. *)

PROCEDURE Find(t: T): BOOLEAN RAISES {};
(* Gather the arguments for tool 't'.  Bad args results
in a message and a return of FALSE.  TRUE means go for it. *)

(* Getting/Setting argument values. Unbound arguments return NIL for TEXTs,
   and REF ARRAY OF TEXTs. The empty string list is denoted by a result
   whose NUMBER() = 0. 
   It is a checked runtime error to provide an argument name that has
   not been registered. *)

PROCEDURE GetFlag(t: T; s: TEXT): BOOLEAN RAISES {};

PROCEDURE GetString(t: T; s: TEXT): TEXT 
    RAISES {};

PROCEDURE GetStringList(t: T; s: TEXT): REF ARRAY OF TEXT 
    RAISES {};

PROCEDURE SetFlag(t: T; s: TEXT; f: BOOLEAN) RAISES {};

PROCEDURE SetString(t: T; s: TEXT; val: TEXT) 
    RAISES {};

PROCEDURE SetStringList(t: T; s: TEXT; sl: REF ARRAY OF TEXT) 
    RAISES {};

PROCEDURE SetStringAsList(t: T; s: TEXT; 
    sl: TEXT) RAISES {};
(* Takes a comma or space separated string and stores it as a string list. *)

END M3Args.
