INTERFACE M3CHash;

(***************************************************************************)
(*                      Copyright (C) Olivetti 1989                        *)
(*                          All Rights reserved                            *)
(*                                                                         *)
(* Use and copy of this software and preparation of derivative works based *)
(* upon this software are permitted to any person, provided this same      *)
(* copyright notice and the following Olivetti warranty disclaimer are     *) 
(* included in any copy of the software or any modification thereof or     *)
(* derivative work therefrom made by any person.                           *)
(*                                                                         *)
(* This software is made available AS IS and Olivetti disclaims all        *)
(* warranties with respect to this software, whether expressed or implied  *)
(* under any law, including all implied warranties of merchantibility and  *)
(* fitness for any purpose. In no event shall Olivetti be liable for any   *)
(* damages whatsoever resulting from loss of use, data or profits or       *)
(* otherwise arising out of or in connection with the use or performance   *)
(* of this software.                                                       *)
(***************************************************************************)

IMPORT Text;


(* Hash table for texts/arrays of char *)

TYPE
  Table <: ROOT;  (* Hash table *)
  Id <: ROOT;     (* Entry in table *)


TYPE
  IdCreator = OBJECT METHODS new(text: Text.T): Id RAISES {} END;

PROCEDURE New(
    size: CARDINAL;
    idCreator: IdCreator := NIL;
    init: Table := NIL)
    : Table
    RAISES {};
(* When creating a table the user specifies:
1) the table size (it is a bucket and chain style hash table; 'size' specifies
the number of buckets).
2) the id creation object. If the user supplies an 'idCreator' object then this
objects 'new' method will be called whenever a an 'id' is to be entered into
the table. The 'id' it returns (which can be any subtype of 'Id') will be the
entry for the text given as the 'text' argument. Note the 'text' argument is
given because 'IdToText' cannot be used until the id is properly initialized
and that will not be the case until after the 'new' method has returned. If the
user does not specify an id creation object simple 'Id's will be entered.
  The final 'init' argument to 'New' is only used whe creating an object which
is a subtype of 'Table' - it allows an already allocated table to be properly
initialized. *)


PROCEDURE SetCreator(t: Table; idCreator: IdCreator): IdCreator RAISES {};
(* The creator object can be changed when the table is already set up; the old
creator object is returned *)


PROCEDURE Enter(t: Table; text: Text.T): Id RAISES {};
(* Enter the given text into the table and return its id *)

PROCEDURE Lookup(t: Table; text: Text.T; VAR id: Id): BOOLEAN RAISES {};
(* Check to see if the given text is in the table; if so return TRUE and set
'id' to its id. If not return FALSE, leaving 'id' unchanged *)

<*INLINE*> PROCEDURE IdToText(id: Id): Text.T RAISES {};
(* Return the text corresponding to 'id' *)


(* The following routines expose more of the hashing mechanism. They allow a
hash value to be built up incrementally and then specified when a text or
array of characters is entered into the table. They are intended for use by
the lexer which has to build up arrays of characters (e.g. identifiers,
reserved words) incrementally and hence can build up the hash value at the
same time *)

TYPE
  Value <: REFANY;

<*INLINE*> PROCEDURE NewValue(): Value RAISES {};
(* New value *)

<*INLINE*> PROCEDURE ResetValue(v: Value) RAISES {};
(* Reset a value to its initial state, as returned by 'NewValue' *)

<*INLINE*> PROCEDURE AddCharToValue(ch: CHAR; v: Value) RAISES {};
(* Add a character to the hash value. Values should be added in sequence e.g.
the hash value for "ab" is constructed by:
  v := M3CHash.NewValue();
  M3CHash.AddCharToValue('a', v);
  M3CHash.AddCharToValue('b', v);
*)

PROCEDURE EnterCharsWithValue(
    t: Table;
    v: Value;
    READONLY chars: ARRAY OF CHAR)
    : Id
    RAISES {};
(* Enter 'chars' into the table, using precalculated hash value *)

PROCEDURE EnterTextWithValue(t: Table; v: Value; text: Text.T): Id RAISES {};
(* Enter 'text' into the table, using precalculated hash value *)

END M3CHash.
