(* Copyright 1992 Digital Equipment Corporation.           *)
(* Distributed only by permission.                         *)
(* Last modified on Wed Aug  5 12:12:02 PDT 1992 by karsenty   *)
(*      modified on Wed Jul 22 01:10:06 1992 by mhb        *)
<* PRAGMA LL *>
MODULE MFViews;

IMPORT MaxflowViewClass, Filter, GraphVBT, PaintOpCache, R2, ColorName,
       View, ZeusPanel, Graph, List,MFAlgs, Font,
       PaintOp, VBT, MaxflowIE, Rect, ZeusDataView, RGB;

CONST ScalingFactor = 1.5;

TYPE
  T = MaxflowViewClass.T BRANDED OBJECT
	gvbt : GraphVBT.T;
        source, sink: GraphVBT.Vertex;
	nVertices: INTEGER;
	nEdges: INTEGER;

      OVERRIDES
        <* LL=0 *>
        startrun := Startrun;
        oeSetup := Setup;
        oeHighlightPath := HighlightPath;
        oeRemoveHighlight := RemoveHighlight;
        oeDecFlow := DecFlow;
        oeIncFlow := IncFlow;
        oeFinalResult := FinalResult;

        <* LL=VBT.mu *>
        ueAddVBTVertex := AddVBTVertex;
        ueAddVBTEdge := AddVBTEdge;

        mouse := MouseProcess;

      END;

(* upon a click create a new vertex *)
PROCEDURE MouseProcess(self: T; READONLY cd: VBT.MouseRec) =
  VAR r : R2.T; (* the world coordinates *)
      rect: Rect.T;
  BEGIN
    IF (cd.clickType = VBT.ClickType.FirstDown) THEN
      CASE cd.whatChanged OF
      | VBT.Modifier.MouseL =>
        rect := VBT.Domain (self);
        r := R2.T {FLOAT(cd.cp.pt.h) / FLOAT(rect.east-rect.west), 
                   1.0 - (FLOAT(cd.cp.pt.v) / FLOAT(rect.south-rect.north))};
        MaxflowIE.AddVertex (self, r);
      ELSE
      END;
    END;
  END MouseProcess;

PROCEDURE AddVBTVertex (self: T; v: MFAlgs.MFVertex; pos: R2.T; name: TEXT) =
  <* LL = VBT.mu *>
  VAR vbt: GraphVBT.Vertex;
  BEGIN
    CreateVBTVertex (v, self.gvbt);
    vbt := v.data;
    vbt.move (pos, FALSE);
    vbt.setLabel (name);
    v.label := vbt.label;
    v.pos := pos;
    self.gvbt.redisplay();
  END AddVBTVertex;

PROCEDURE AddVBTEdge (self: T; e: MFAlgs.MFEdge) =
  <* LL = VBT.mu *>
  BEGIN
    CreateVBTEdge (e, self.gvbt);
    self.gvbt.redisplay();
  END AddVBTEdge;

PROCEDURE PrintEdge( <* UNUSED *> e: Graph.Edge): TEXT =
  BEGIN
    RETURN "rien a dire";
  END PrintEdge;

PROCEDURE Color(color: TEXT): PaintOp.T =
  BEGIN
    RETURN PaintOpCache.FromRGB(ColorName.ToRGB (color));
  END Color;

(* sat is the blue saturation 0 for light, 1 for full saturation *)
PROCEDURE BlueColor(sat: REAL): PaintOp.T =
  VAR r : REAL := 0.6  -  (0.6 * sat);
  BEGIN
    RETURN PaintOpCache.FromRGB (RGB.T {r, r, 1.0});
  END BlueColor;

PROCEDURE Startrun (view: T) =
  (* sleazy hack: remove the old GraphVBT and just ignore it;
     heck, what else are VM and GC good for? *)
  BEGIN
    EVAL Filter.Replace(view, NEW(GraphVBT.T).init());
    (* call the superclass startrun in ZeusClass.T *)
    MaxflowViewClass.T.startrun(view); 
  END Startrun;

PROCEDURE CreateVBTVertex (obj, arg: REFANY) =
  VAR v : Graph.Vertex;
      vbt : GraphVBT.Vertex;
      g := NARROW (arg, GraphVBT.T);
      stdsize : R2.T;
  BEGIN
    stdsize := R2.T{0.08,0.08};   
    v := NARROW (obj, Graph.Vertex);
    vbt := NEW (GraphVBT.Vertex, graph := g, 
                pos := R2.T {0.5, 0.5},
	        sizeW := stdsize,
                shape := GraphVBT.VertexShape.Ellipse).init();

    v.data := vbt;
    vbt.setFont (Font.FromName (ARRAY OF TEXT {"helvetica-bold-r-*-*-*-140-*"}));
    vbt.setFontColor (Color("Black"));
    vbt.setColor(Color ("gray"));

    EVAL  NEW (GraphVBT.VertexHighlight, vertex := vbt,
                color := Color ("gray"),
                borderMM := R2.T{1.0, 1.0}).init();	

  END CreateVBTVertex;

PROCEDURE CreateVBTEdge (obj: REFANY; <* UNUSED *> arg: REFANY) =
  VAR e : Graph.Edge;
      mfe : MFAlgs.MFEdge;
      ebt : GraphVBT.Edge;
  BEGIN
    e := NARROW (obj, Graph.Edge);
    ebt := NEW (GraphVBT.Edge,  
                vertex0 := NARROW (e.from.data, GraphVBT.Vertex), 
                vertex1 := NARROW (e.to.data, GraphVBT.Vertex)).init();
    ebt.setColor(Color("cornflowerblue"));
    ebt.setWidthMM (0.0);
    e.data := ebt;

    (* create the capacity edge : in order to simulate borders, we
       create two edges, a black one and a white one on top, with width
       slightly smaller *)

    ebt := NEW (GraphVBT.Edge,  
                vertex0 := ebt.vertex0, vertex1 := ebt.vertex1).init();
    mfe := NARROW (obj, MFAlgs.MFEdge);

    ebt.setWidthMM ((mfe.capacity * ScalingFactor) + 1.0);
    ebt.toFront();
    ebt.setColor(Color ("black"));
    ebt := NEW (GraphVBT.Edge,  
                vertex0 := ebt.vertex0, vertex1 := ebt.vertex1).init();
    ebt.setWidthMM (mfe.capacity * ScalingFactor);
    ebt.toFront();
    ebt.setColor(Color ("white"));

  END CreateVBTEdge;

(* beautify retrieve the position and label in Graph.Vertex and 
   applies it to the twin GraphVBT.Vertex *)

PROCEDURE Beautify (g: Graph.T) =
  VAR v: Graph.VertexList := g.vertices;
      vert: MFAlgs.MFVertex;
    
  BEGIN
    WHILE v # NIL DO
      vert := v.vertex;
      NARROW (vert.data, GraphVBT.Vertex).move (vert.pos, FALSE);
      NARROW (vert.data, GraphVBT.Vertex).setLabel (vert.label);
      v := v.next;
    END;  
  END Beautify;


(* create a GraphVBT.T equivalent to Graph.T *)
PROCEDURE GraphToGraphVBT (g: Graph.T) : GraphVBT.T =
  VAR
      wc := GraphVBT.WorldRectangle{
            w := 0.0, s := 0.0, e := 1.0, n := 1.0};
      gvbt : GraphVBT.T;

  BEGIN
    gvbt := NEW(GraphVBT.T, world := wc).init();

    EVAL g.vertexApply (CreateVBTVertex, gvbt);
    EVAL g.edgeApply (CreateVBTEdge, gvbt);

    Beautify (g);

    RETURN gvbt;
  END GraphToGraphVBT;

(* initialize the graph with the number of edges and vertices *)
PROCEDURE Setup (view: T; g: Graph.T; source, sink: Graph.Vertex) =

  BEGIN
    view.gvbt := GraphToGraphVBT(g);

    view.source := source.data;
    view.sink := sink.data;

    view.source.setColor(Color ("cornflowerblue"));
    view.source.setLabel ("Source");
    view.source.setSizeW (R2.T{0.1, 0.1});

    view.sink.setColor(Color ("cornflowerblue"));
    view.sink.setLabel ("Sink");
    view.sink.setSizeW (R2.T{0.1, 0.1});

    EVAL Filter.Replace(view, view.gvbt);
    view.gvbt.redisplay();
    view.gvbt.animate(0.0, 1.0);
  END Setup;

PROCEDURE New (): View.T =
  VAR a : T :=  NEW(T).init(NEW(GraphVBT.T).init());
  BEGIN
    a.gvbt := NIL;
    RETURN a;
  END New;

PROCEDURE HighlightPath (view: T; path : List.T; <* UNUSED *> maxC: REAL)=
  VAR vh, sourcevh: GraphVBT.VertexHighlight;
      curvert: Graph.Vertex;
      curedge: Graph.Edge;
      p: List.T;
      vbt_vertex: GraphVBT.Vertex;

  BEGIN
    curedge := path.first;
    p := path.tail;
    curvert := curedge.to;
    vbt_vertex := NARROW(curedge.from.data, GraphVBT.Vertex);
    vh := vbt_vertex.vertexHighlights.first;
    sourcevh := vh;
    vh.setColor(Color("black"));
    vbt_vertex := NARROW(curedge.to.data, GraphVBT.Vertex);
    vh := vbt_vertex.vertexHighlights.first;
    vh.setColor(Color("black"));
    
    WHILE p # NIL DO
      curedge := p.first;
      p := p.tail;
      IF (curedge.from = curvert) THEN
        vbt_vertex := NARROW(curedge.to.data, GraphVBT.Vertex);
        vh := vbt_vertex.vertexHighlights.first;
        vh.setColor(Color("black"));
        curvert := curedge.to;
      ELSE
        vbt_vertex := NARROW(curedge.from.data, GraphVBT.Vertex);
        vh := vbt_vertex.vertexHighlights.first;
        vh.setColor(Color("black"));
        curvert := curedge.from;
      END;
    END;

    sourcevh.setColor(Color("yellow"));
    view.gvbt.redisplay();
  END HighlightPath;

(* reminder: edge.vertex0 / 1 = from / to *)
PROCEDURE IncFlow(view: T; edge: Graph.Edge; flow: REAL;
                  number: CARDINAL; maxC, capacity: REAL) =
  VAR 
    e : GraphVBT.Edge;
    vh: GraphVBT.VertexHighlight;
    evbt : GraphVBT.Edge := edge.data;

  BEGIN
    (* animate the flowing thru the edge e *)
    e := NEW (GraphVBT.Edge,  
              vertex0 := evbt.vertex0, vertex1 := evbt.vertex0).init();
    e.setWidthMM(flow * ScalingFactor);
    e.move (e.vertex0, evbt.vertex1, NIL, NIL, TRUE);
    e.setColor (Color("cornflowerblue"));
    e.toFront ();

    view.gvbt.redisplay(); 
    view.gvbt.animate(0.0, 1.0);

    e.remove();

    (* the new flow/edge *)
    evbt.setWidthMM(flow * ScalingFactor);
    evbt.toFront();
    IF (flow = capacity) THEN
      evbt.setColor(Color("magenta"));
    ELSE
      evbt.setColor(Color("cornflowerblue"));
    END;

    vh:= evbt.vertex0.vertexHighlights.first;
    vh.setColor(Color("gray"));
    vh:= evbt.vertex1.vertexHighlights.first;
    vh.setColor(Color("yellow"));

    view.gvbt.redisplay();
END IncFlow;

PROCEDURE DecFlow(view: T; edge: Graph.Edge; oldflow, newflow: REAL;
                  number: CARDINAL; maxC, capa: REAL) =
  VAR 
    e : GraphVBT.Edge;
    vh: GraphVBT.VertexHighlight;
    evbt : GraphVBT.Edge := edge.data;

  BEGIN

    e := NEW (GraphVBT.Edge,  
              vertex0 := evbt.vertex0, vertex1 := evbt.vertex1).init();
    e.setWidthMM(oldflow * ScalingFactor);
    e.move (evbt.vertex0, evbt.vertex0, NIL, NIL, TRUE);
    e.setColor(Color ("limegreen"));
    e.toFront ();

    evbt.setWidthMM(newflow * ScalingFactor);
    evbt.setColor(Color("cornflowerblue"));

    view.gvbt.redisplay(); 
    view.gvbt.animate(0.0, 1.0);

    e.remove();

    vh:= evbt.vertex0.vertexHighlights.first;
    vh.setColor(Color("yellow"));
    vh:= evbt.vertex1.vertexHighlights.first;
    vh.setColor(Color("gray"));

    view.gvbt.redisplay();
  END DecFlow;

PROCEDURE RemoveHighlight(view: T; sinkvertex: Graph.Vertex) =
  VAR
     v: GraphVBT.Vertex := sinkvertex.data;
    vh: GraphVBT.VertexHighlight;

  BEGIN
    vh:= v.vertexHighlights.first;
    vh.setColor(Color("gray"));    
  END RemoveHighlight;


PROCEDURE FinalResult(view: T; <* UNUSED *> b: BOOLEAN) =
  BEGIN
    view.gvbt.redisplay();
  END FinalResult;

BEGIN
  ZeusPanel.RegisterView (New, "Animated Maxflow", "Maxflow");
  ZeusPanel.RegisterView (ZeusDataView.New, "Data View", "Maxflow");

END MFViews.
