(* Copyright 1992 Digital Equipment Corporation. *)
(* Distributed only by permission. *)
(* Last modified on Sun Jul 26 23:16:38 1992 by karsenty*)
(*      modified on Tue Jul 21 06:24:57 1992 by steven *)

MODULE MFAlgs;

IMPORT Algorithm, MaxflowAlgClass, MaxflowIE, FormsVBT, Random, Thread,
       VBT, ZeusPanel, Graph, R2, List;

TYPE 
  T = MaxflowAlgClass.T BRANDED OBJECT 
      graph : Graph.T;
      source, sink : MFVertex;
    OVERRIDES 
      run := Run; 
    END;

  (* edges and vertices use the data field as a pointer
     to their GraphVBT twin *)

  MFEdge = Graph.Edge BRANDED OBJECT
      capacity, flow : REAL;
  END;

  MFVertex = Graph.Vertex BRANDED OBJECT
      wherefrom : MFEdge;
      marked : BOOLEAN;
  END;

      
PROCEDURE Run (alg: T) RAISES {Thread.Alerted} =
  VAR
    Vertices  : INTEGER;        (* number of vertices *)
    Edges     : INTEGER;        (* number of edges *)
    v0, v1    : Graph.Vertex;
    stdsize: R2.T;
  BEGIN
    LOCK VBT.mu DO
      Edges := FormsVBT.GetInteger(alg.data, "Edges");
      Vertices := FormsVBT.GetInteger(alg.data, "Vertices");
    END;

    stdsize := R2.T{5.0, 5.0};
    v0 := NEW (MFVertex, data := NIL).init(alg.graph);
    alg.source := NARROW (v0, MFVertex);
(*    FOR i := 1 TO Vertices DO
      v1 := NEW (MFVertex, data := NIL).init(alg.graph);
      EVAL NEW (MFEdge, from := v0, to := v1, data := NIL).init(alg.graph);
      v0 := v1;
    END;
    alg.sink := NARROW (v0, MFVertex); *)
    v

    MaxflowIE.Setup(alg, alg.graph);
    InitFlow(alg.graph);
    FindFlow(alg);

  END Run;

PROCEDURE NullText (<* UNUSED*> g: Graph.T) : TEXT =
  BEGIN
    RETURN "--";
  END NullText;

PROCEDURE NewAlg (): Algorithm.T =
  VAR fv := FormsVBT.NewFromFile("maxflowinput.fv");
      g := NEW(Graph.T).init();
  BEGIN
    RETURN NEW(T, data := fv, graph := g).init();
  END NewAlg;



PROCEDURE InitFlow (g: Graph.T) =
  VAR e: Graph.EdgeList;
     ed: MFEdge;
    
  BEGIN
    e := g.edges;
    WHILE e # NIL DO
      ed := e.edge;
      e := e.next;
      ed.flow := 0.0;
      ed.capacity := 1.0;
    END;
  END InitFlow;

PROCEDURE FindFlow (alg: T) =
  VAR p : List.T;
      c : REAL;
     ed : MFEdge;

  BEGIN
    p := FindPath (alg, c);
    WHILE p # NIL DO
      WHILE p # NIL DO 
        ed := List.Pop (p);
        ed.flow := ed.flow + c;
        MaxflowIE.ChangeFlow (alg, ed.data, ed.flow);
      END;
      p := FindPath (alg, c);
    END;
  END FindFlow;

(* Returns a list of edges that form a path from source to sink,
  with nonzero residual capacity.
  cap is the minimum residual capacity of all edges
  in the path *)
PROCEDURE FindPath (alg: T; VAR cap: REAL): List.T =
  VAR
    queue           : List.T := NIL;
    path            : List.T := NIL;
    v               : Graph.Vertex;
    neighbours: Graph.EdgeList;
    neighbour, current_edge  : MFEdge;
    current_vertex, neighbourto, neighbourfrom : MFVertex;
    residual_capacity : REAL;
    found_the_sink  : BOOLEAN        := FALSE;
    vlist: Graph.VertexList;
    vert: MFVertex;
  BEGIN
    (* do a bfs starting at the source *)

    vlist := alg.graph.vertices;
    WHILE vlist # NIL DO
      vert := vlist.vertex;
      vlist := vlist.next;
      vert.marked := FALSE;
    END;    

    List.Push(queue, alg.source);
    WHILE queue # NIL DO

      (* pull a vertex v off the queue *)
      v := List.Last(queue);
      EVAL List.Delete(queue, v);
      neighbours := v.edges;

      (* go through the neighbors of v and see if their residual
         capacity is nonzero *)
      WHILE (neighbours # NIL) AND NOT found_the_sink DO
        neighbour := neighbours.edge;
        neighbours := neighbours.next;
        neighbourto := NARROW (neighbour.to, MFVertex);
        neighbourfrom := NARROW (neighbour.from, MFVertex);

        (* first check if edge is a forward edge *)
        IF  (neighbourfrom = v) AND (NOT neighbourto.marked)
             AND (neighbour.flow < neighbour.capacity) THEN
          neighbourto.marked := TRUE;
          neighbourto.wherefrom := neighbour;
          (* as soon as find sink, can stop the bfs *)
          IF neighbourto = alg.sink THEN found_the_sink := TRUE; END;
          List.Push(queue, neighbourto);
        END;

        (* then check if edge is back edge *)
        IF   (neighbourto = v) AND (NOT neighbourfrom.marked)
             AND (neighbour.flow > 0.0) THEN
          neighbourfrom.marked := TRUE;
          neighbourfrom.wherefrom := neighbour;
          List.Push(queue, neighbourfrom);
        END;
      END;

      (* once find sink, reconstruct the path *)
      IF found_the_sink THEN
        current_edge := alg.sink.wherefrom;
        cap := current_edge.capacity - current_edge.flow;
        current_vertex := alg.sink;
        WHILE (alg.source # current_vertex) DO
          List.Push (path, current_edge);
          IF current_edge.from = current_vertex THEN
            (* The edge is a back edge, reduce the flow *)
            residual_capacity := current_edge.flow;
            current_vertex := current_edge.to;
          ELSE
            (* The edge is a forward edge, increase the flow *)
            residual_capacity := current_edge.capacity - current_edge.flow;
            current_vertex := current_edge.from;
          END;
          IF residual_capacity < cap THEN
            cap := residual_capacity;
          END;
          current_edge := current_vertex.wherefrom;
        END;
      END;
    END;
    RETURN path;
  END FindPath;

BEGIN
  ZeusPanel.RegisterAlg(NewAlg, "Ford-Fulkerson", "Maxflow");
END MFAlgs.


