(* Copyright (C) 1992, Digital Equipment Corporation *)
(* All rights reserved. *)
(* See the file COPYRIGHT for a full description. *)

(* Last modified on Fri Sep 25 13:03:01 PDT 1992 by steveg *)
(*      modified on Tue Aug  4 07:42:08 PDT 1992 by jdd *)

MODULE GraphVBT EXPORTS GraphVBT, GraphVBTExtras;

IMPORT Animate, Axis, FloatMode, Font, List, Math, MG, MGV, PaintOp,
       PaintOpCache, Pts, R2, Random, RealFloat, RealPath, Rect, Region,
       VBT, VBTClass;

<* PRAGMA LL *>

REVEAL
  T =
    TPublic BRANDED OBJECT
      <* LL >= {SELF.mu} *>
      rect: Rect.T;             (* the display rectangle in pixels *)
      res := ARRAY Axis.T OF REAL{1.0, 1.0}; (* copied from screentype *)
      realMarginMM: R2.T;       (* enough at left and bottom to center the
                                   display rectangle *)
      edges           : List.T (* OF Edge *)            := NIL;
      vertexHighlights: List.T (* OF VertexHighlight *) := NIL;
      polygons        : List.T (* OF Polygon *)         := NIL;
      vertexGroup: MG.Group;    (* all the MG.Group objects for the
                                   vertices *)
      edgeGroup: MG.Group;      (* all the MG.Group objects for the
                                   edges *)
      vertexHighlightGroup: MG.Group;  (* all the MG.Group objects for the
                                          vertex highlights *)
      polygonGroup: MG.Group;   (* all the MG.Group objects for the
                                   polygons *)
      animations := 0;          (* number of animations in progress *)
      needRefresh := FALSE;     (* if objects moved or screen resolution
                                   while an animation was in effect *)
      initialized := FALSE;
    OVERRIDES
      init               := InitGraph;
      reshape            := Reshape;
      shape              := Shape;
      redisplay          := RedisplayGraph;
      animate            := AnimateGraph;
      clear              := ClearGraph;
      verticesAt         := GraphVerticesAt;
      edgesAt            := GraphEdgesAt;
      vertexHighlightsAt := GraphVertexHighlightsAt;
    END;

REVEAL
  Vertex = VertexPublic BRANDED OBJECT
             <* LL >= {SELF.graph.mu} *>
             mg   : MG.T;       (* the corresponding MG.T *)
             group: MG.Group;   (* containing the MG.T *)
             colorScheme: PaintOp.ColorScheme;  (* containing the color and
                                                   the fontColor *)
             animated: BOOLEAN;
             path    : AnimationPath;
             new: RECORD
                    pos: R2.T;  (* position after next animation *)
                  END;
             initialized := FALSE;
           OVERRIDES
             init           := InitVertex;
             move           := MoveVertex;
             setSizeMM      := SetVertexSizeMM;
             setSizeW       := SetVertexSizeW;
             setShape       := SetVertexShape;
             setColor       := SetVertexColor;
             setLabel       := SetVertexLabel;
             setFont        := SetVertexFont;
             setFontColor   := SetVertexFontColor;
             setBorderMM    := SetVertexBorderMM;
             setBorderColor := SetVertexBorderColor;
             toFront        := VertexToFront;
             toBack         := VertexToBack;
             remove         := RemoveVertex;
           END;

REVEAL
  Edge =
    EdgePublic BRANDED OBJECT
      <* LL >= {SELF.vertex0.graph.mu} *>
      graph   : T;              (* from the vertices *)
      straight: BOOLEAN;        (* whether the line is straight *)
      mg      : MG.T;           (* the corresponding MG.Line or MG.Shape *)
      isLine  : BOOLEAN;        (* whether the mg is an MG.Line *)
      end: ARRAY [0 .. 1] OF MG.LineEnd;  (* the ends of the MG.Line, or
                                             NIL *)
      arrowLine: ARRAY [0 .. 1], [0 .. 1] OF MG.Line;
      (* the arrowheads, or NIL.  the first index is which arrowhead
         (forward then backward); the second index is which which line (the
         one on the left in its direction, then the one on the right). *)
      arrowEnd: ARRAY [0 .. 1] OF ARRAY [0 .. 1], [0 .. 1] OF MG.LineEnd;
      (* their ends, or NIL.  the third index is the end on the edge, then
         the far end. *)
      arrowPos: ARRAY [0 .. 1] OF ARRAY [0 .. 1], [0 .. 1] OF R2.T;
      (* the positions of the ends, in world coordinates. *)
      group: MG.Group;          (* a group for all the endpoints *)
      pos: ARRAY [0 .. 1] OF R2.T;  (* current positions of the endpoints,
                                       in world coordinates *)
      cpos: ARRAY [0 .. 1] OF R2.T;  (* current positions of control
                                        points, in world coordinates, or
                                        same as pos if no control points *)
      new: RECORD
             vertex0, vertex1: Vertex;  (* vertices after next animation *)
             control0, control1: Vertex;  (* control vertices after next
                                             animation *)
           END;
      colorScheme: PaintOp.ColorScheme;
      initialized                        := FALSE;
    OVERRIDES
      init       := InitEdge;
      move       := MoveEdge;
      setWidthMM := SetEdgeWidthMM;
      setColor   := SetEdgeColor;
      setArrow   := SetEdgeArrow;
      toFront    := EdgeToFront;
      toBack     := EdgeToBack;
      remove     := RemoveEdge;
    END;

REVEAL
  VertexHighlight =
    VertexHighlightPublic BRANDED OBJECT
      <* LL >= {SELF.vertex.graph.mu} *>
      graph: T;                 (* from the vertex *)
      mg   : MG.T;              (* the corresponding MG.T *)
      group: MG.Group;          (* containing the MG.T *)
      colorScheme: PaintOp.ColorScheme;  (* the ColorScheme for the MG.T *)
      shape      : VertexShape;          (* current shape *)
      pos: R2.T;                (* current position of the vertex
                                   highlight, in world coordinates *)
      sizeMM: R2.T;             (* dimensions of the vertex highlight
                                   rectangle, in millimeters *)
      new: RECORD
             vertex: Vertex;    (* vertex after next animation *)
           END;
      initialized := FALSE;
    OVERRIDES
      init        := InitVertexHighlight;
      move        := MoveVertexHighlight;
      setBorderMM := SetVertexHighlightBorderMM;
      setBorderW  := SetVertexHighlightBorderW;
      setColor    := SetVertexHighlightColor;
      toFront     := VertexHighlightToFront;
      toBack      := VertexHighlightToBack;
      remove      := RemoveVertexHighlight;
    END;

REVEAL
  Polygon =
    PolygonPublic BRANDED OBJECT
      <* LL >= {List.First(SELF).graph.mu} *>
      graph: T;                 (* from the vertices *)
      mg   : MG.T;              (* the corresponding MG.Shape *)
      pos: List.T (* OF REF R2.T *);  (* current positions of the corners,
                                         in world coordinates *)
      group: MG.Group;          (* a group for the MG.T *)
      new: RECORD
             vertices: List.T (* OF Vertex *);  (* vertices after next
                                                   animation *)
           END;
      colorScheme: PaintOp.ColorScheme;
      initialized                        := FALSE;
    OVERRIDES
      init     := InitPolygon;
      move     := MovePolygon;
      setColor := SetPolygonColor;
      toFront  := PolygonToFront;
      toBack   := PolygonToBack;
      remove   := RemovePolygon;
    END;

<* LL.sup < graph.mu *>

PROCEDURE InitGraph (graph: T): T =
  BEGIN
    IF graph.initialized THEN RETURN graph; END;

    graph.border := ARRAY Axis.T OF REAL{0.0, 0.0}; (* no MG border *)

    EVAL MG.V.init(graph);

    graph.sizeMM := R2.T{1.0, 1.0};
    graph.realMarginMM := R2.Origin;

    LOCK graph.mu DO
      graph.vertexGroup := NEW(MG.Group).init();
      graph.edgeGroup := NEW(MG.Group).init();
      graph.vertexHighlightGroup := NEW(MG.Group).init();
      graph.polygonGroup := NEW(MG.Group).init();

      graph.displayList.addAfter(graph, graph.vertexGroup, NIL);
      graph.displayList.addAfter(graph, graph.edgeGroup, NIL);
      graph.displayList.addAfter(graph, graph.vertexHighlightGroup, NIL);
      graph.displayList.addAfter(graph, graph.polygonGroup, NIL);

      graph.initialized := TRUE;
    END;

    RETURN graph;
  END InitGraph;

<* LL.sup >= graph.mu *>

PROCEDURE Reshape (graph: T; READONLY cd: VBT.ReshapeRec) =
  BEGIN
    graph.rect := cd.new;
    IF graph.st # NIL THEN graph.res := graph.st.res; END;
    (* avoid arithmetic problems *)
    IF graph.rect.east = graph.rect.west THEN
      graph.rect.east := graph.rect.east + 1;
    END;
    IF graph.rect.south = graph.rect.north THEN
      graph.rect.south := graph.rect.south + 1;
    END;

    RefreshGraph(graph);
    MG.V.reshape(graph, cd);

    IF graph.animations > 0 THEN graph.needRefresh := TRUE; END;
  END Reshape;

<* LL.sup >= graph.mu *>

PROCEDURE Shape (graph: T; ax: Axis.T; <* UNUSED *> n: CARDINAL):
  VBT.SizeRange =
  BEGIN
    RETURN VBT.SizeRange{VBT.DefaultShape.lo,
                         MAX(ROUND(100.0 * graph.st.res[ax]),
                             VBT.DefaultShape.lo), VBT.DefaultShape.hi};
  END Shape;

<* LL.sup <= VBT.mu *>

PROCEDURE RedisplayGraph (graph: T) =
  BEGIN
    NARROW(graph, MG.V).mgRedisplay(Region.Full);
  END RedisplayGraph;

<* LL.sup < vertex.graph.mu *>

PROCEDURE InitVertex (vertex: Vertex): Vertex =
  BEGIN
    IF vertex.initialized THEN RETURN vertex; END;

    WITH graph = vertex.graph DO
      LOCK graph.mu DO
        IF vertex.sizeW[0] # 0.0 OR vertex.sizeW[1] # 0.0 THEN
          vertex.sizeMM := Abs(WorldSizeToMM(graph, vertex.sizeW));
        END;
        vertex.colorScheme :=
          PaintOpCache.MakeColorScheme(vertex.color, vertex.fontColor);
        CASE vertex.shape OF
        | VertexShape.Rectangle =>
            vertex.mg := NEW(MG.Rectangle, color := vertex.colorScheme,
                             label := vertex.label, font := vertex.font,
                             weight := 0.0).init(R2.Origin, R2.Origin);
        | VertexShape.Ellipse =>
            vertex.mg := NEW(MG.Ellipse, color := vertex.colorScheme,
                             label := vertex.label, font := vertex.font,
                             weight := 0.0).init(R2.Origin, R2.Origin);
        END;
        vertex.group := NEW(MG.Group).init();

        vertex.vertexHighlights := NIL;
        vertex.new.pos := vertex.pos;
        vertex.animated := FALSE;
        vertex.group.addBefore(graph, vertex.mg, NIL);
        graph.vertexGroup.addBefore(graph, vertex.group, NIL);
        List.Push(graph.vertices, vertex);

        VAR centerPP := WorldPosToPts(graph, vertex.pos);
        BEGIN
          MG.TranslateToLocked(
            vertex.mg, graph, Finite2(centerPP), center := TRUE);
          AdjustVertex(vertex);
        END;
        vertex.initialized := TRUE;
      END;
    END;
    RETURN vertex;
  END InitVertex;

<* LL.sup < edge.vertex0.graph.mu *>

PROCEDURE InitEdge (edge: Edge): Edge =
  BEGIN
    IF edge.initialized THEN RETURN edge; END;

    WITH graph = edge.vertex0.graph DO
      LOCK graph.mu DO
        edge.graph := graph;
        <*ASSERT edge.vertex1.graph = graph*>

        edge.straight := (edge.control0 = NIL) AND (edge.control1 = NIL);
        IF edge.control0 # NIL THEN
          <*ASSERT edge.control0.graph = graph*>
        END;
        IF edge.control1 # NIL THEN
          <*ASSERT edge.control1.graph = graph*>
        END;

        edge.colorScheme :=
          PaintOpCache.MakeColorScheme(edge.color, edge.color);

        IF edge.straight THEN
          edge.mg :=
            NEW(MG.Line, weight := Pts.FromMM(edge.widthMM),
                color := edge.colorScheme).init(R2.Origin, R2.Origin);
          edge.end[0] :=
            NEW(MG.LineEnd, line := edge.mg, controlsFrom := TRUE).init();
          edge.end[1] :=
            NEW(MG.LineEnd, line := edge.mg, controlsFrom := FALSE).init();
          edge.isLine := TRUE;
        ELSE
          VAR path := NEW(RealPath.T);
          BEGIN
            path.init();
            edge.mg := NEW(MG.Shape, weight := Pts.FromMM(edge.widthMM),
                           color := edge.colorScheme).init(
                         R2.Origin, path, fill := FALSE);
          END;
          edge.end[0] := NIL;
          edge.end[1] := NIL;
          edge.isLine := FALSE;
        END;

        FOR i := 0 TO 1 DO
          IF edge.arrow[i] THEN
            FOR j := 0 TO 1 DO
              edge.arrowLine[i][j] :=
                NEW(MG.Line, weight := Pts.FromMM(edge.widthMM),
                    color := edge.colorScheme).init(R2.Origin, R2.Origin);
              edge.arrowEnd[i][j][0] :=
                NEW(MG.LineEnd, line := edge.arrowLine[i][j],
                    controlsFrom := TRUE).init();
              edge.arrowEnd[i][j][1] :=
                NEW(MG.LineEnd, line := edge.arrowLine[i][j],
                    controlsFrom := FALSE).init();
            END;
          END;
        END;

        FOR i := 0 TO 1 DO
          FOR j := 0 TO 1 DO
            FOR k := 0 TO 1 DO edge.arrowPos[i][j][k] := R2.Origin; END;
          END;
        END;

        edge.new.vertex0 := edge.vertex0;
        edge.new.vertex1 := edge.vertex1;

        List.Push(edge.vertex0.edges, edge);
        List.Push(edge.vertex1.edges, edge);

        edge.new.control0 := edge.control0;
        edge.new.control1 := edge.control1;

        IF edge.control0 # NIL THEN
          List.Push(edge.control0.edges, edge);
        END;
        IF edge.control1 # NIL THEN
          List.Push(edge.control1.edges, edge);
        END;

        edge.group := NEW(MG.Group).init();
        graph.edgeGroup.addBefore(graph, edge.group, NIL);
        IF edge.isLine THEN
          edge.group.addBefore(graph, edge.end[0], NIL);
          edge.group.addBefore(graph, edge.end[1], NIL);
        ELSE
          edge.group.addBefore(graph, edge.mg, NIL);
        END;

        FOR i := 0 TO 1 DO
          IF edge.arrow[i] THEN
            FOR j := 0 TO 1 DO
              FOR k := 0 TO 1 DO
                edge.group.addBefore(graph, edge.arrowEnd[i][j][k], NIL);
              END;
            END;
          END;
        END;

        List.Push(graph.edges, edge);

        RefreshEdge(edge);
        edge.initialized := TRUE;
      END;
    END;
    RETURN edge;
  END InitEdge;

<* LL.sup < vertexHighlight.vertex.graph.mu *>

PROCEDURE InitVertexHighlight (vertexHighlight: VertexHighlight):
  VertexHighlight =
  BEGIN
    IF vertexHighlight.initialized THEN RETURN vertexHighlight; END;

    WITH graph = vertexHighlight.vertex.graph DO
      LOCK graph.mu DO
        vertexHighlight.graph := graph;

        vertexHighlight.colorScheme :=
          PaintOpCache.MakeColorScheme(vertexHighlight.color, PaintOp.Fg);

        vertexHighlight.sizeMM := R2.Origin;

        vertexHighlight.shape := vertexHighlight.vertex.shape;
        CASE vertexHighlight.shape OF
        | VertexShape.Rectangle =>
            vertexHighlight.mg :=
              NEW(MG.Rectangle, color := vertexHighlight.colorScheme,
                  weight := 0.0).init(R2.Origin, R2.Origin);
        | VertexShape.Ellipse =>
            vertexHighlight.mg :=
              NEW(MG.Ellipse, color := vertexHighlight.colorScheme,
                  weight := 0.0).init(R2.Origin, R2.Origin);
        END;

        vertexHighlight.group := NEW(MG.Group).init();

        vertexHighlight.pos := vertexHighlight.vertex.pos;

        vertexHighlight.new.vertex := vertexHighlight.vertex;
        List.Push(vertexHighlight.vertex.vertexHighlights, vertexHighlight);

        vertexHighlight.group.addBefore(graph, vertexHighlight.mg, NIL);
        graph.vertexHighlightGroup.addBefore(
          graph, vertexHighlight.group, NIL);

        List.Push(graph.vertexHighlights, vertexHighlight);

        IF vertexHighlight.borderW[0] # 0.0
             OR vertexHighlight.borderW[1] # 0.0 THEN
          vertexHighlight.borderMM :=
            Abs(WorldSizeToMM(graph, vertexHighlight.borderW));
        END;

        VAR centerPP := WorldPosToPts(graph, vertexHighlight.vertex.pos);
        BEGIN
          MG.TranslateToLocked(
            vertexHighlight.mg, graph, Finite2(centerPP), center := TRUE);
        END;
        EVAL AdjustVertexHighlightSizeandShape(vertexHighlight);
        vertexHighlight.initialized := TRUE;
      END;
    END;
    RETURN vertexHighlight;
  END InitVertexHighlight;

<* LL.sup < List.First(polygon.vertices).graph.mu *>

PROCEDURE InitPolygon (polygon: Polygon): Polygon =
  BEGIN
    IF polygon.initialized THEN RETURN polygon; END;

    <*ASSERT polygon.vertices # NIL*>
    WITH graph = NARROW(List.First(polygon.vertices), Vertex).graph DO
      LOCK graph.mu DO
        polygon.graph := graph;
        VAR vertices := List.Tail(polygon.vertices);
        BEGIN
          WHILE vertices # NIL DO
            VAR vertex: Vertex := List.Pop(vertices);
            BEGIN
              <*ASSERT vertex.graph = graph*>
            END;
          END;
        END;

        polygon.colorScheme :=
          PaintOpCache.MakeColorScheme(polygon.color, polygon.color);

        VAR path := NEW(RealPath.T);
        BEGIN
          path.init();
          polygon.mg := NEW(MG.Shape, weight := 0.0,
                            color := polygon.colorScheme).init(
                          R2.Origin, path, fill := FALSE);
        END;

        polygon.new.vertices := polygon.vertices;

        VAR vertices := polygon.vertices;
        BEGIN
          REPEAT
            VAR vertex: Vertex := List.Pop(vertices);
            BEGIN
              List.Push(vertex.polygons, polygon);
            END;
          UNTIL vertices = NIL;
        END;

        polygon.group := NEW(MG.Group).init();
        polygon.group.addBefore(graph, polygon.mg, NIL);
        graph.polygonGroup.addBefore(graph, polygon.group, NIL);
        List.Push(graph.polygons, polygon);

        RefreshPolygon(polygon);
        polygon.initialized := TRUE;
      END;
    END;
    RETURN polygon;
  END InitPolygon;

<* LL.sup < graph.mu *>

PROCEDURE AnimateGraph (graph: T; t0, t1: REAL) =
  BEGIN
    LOCK graph.mu DO
      graph.animations := graph.animations + 1;
      VAR
        tz := t0 / t1;
        a  := 1.0 / (1.0 - tz);
        b  := -tz / (1.0 - tz);
      BEGIN
        IF NOT RealFloat.Finite(a + b) THEN a := 1.0; b := 0.0; END;
        VAR tf := NEW(AffineTimeFunction, a := a, b := b);
        BEGIN
          AnimateGraphVertices(graph, tf);
          AnimateGraphEdges(graph, tf);
          AnimateGraphVertexHighlights(graph, tf);
          AnimateGraphPolygons(graph, tf);
        END;
      END;
    END;
    MGV.Animation(graph, t1);
    VAR redisplay := FALSE;
    BEGIN
      LOCK graph.mu DO
        graph.animations := graph.animations - 1;
        IF graph.animations > 0 THEN
          graph.needRefresh := TRUE;
        ELSIF graph.needRefresh THEN
          RefreshGraph(graph);
        END;
        PostAnimateGraphVertices(graph);
        IF PostAnimateGraphEdges(graph) THEN redisplay := TRUE; END;
        IF PostAnimateGraphVertexHighlights(graph) THEN
          redisplay := TRUE;
        END;
        IF PostAnimateGraphPolygons(graph) THEN redisplay := TRUE; END;
      END;
      IF redisplay THEN graph.redisplay(); END;
    END;
  END AnimateGraph;

<* LL.sup >= graph.mu *>

PROCEDURE AnimateGraphVertices (graph: T; tf: Animate.TimeFunction) =
  VAR vertices := graph.vertices;
  BEGIN
    WHILE vertices # NIL DO
      VAR vertex: Vertex := List.Pop(vertices);
      BEGIN
        IF vertex.animated THEN
          IF vertex.path # NIL THEN
            VAR
              path: AlongGivenPath := NEW(AlongGivenPath, graph := graph,
                                          pos := vertex.pos,
                                          path := vertex.path).init(tf);
            BEGIN
              AddAnimationLocked(graph, path, vertex.mg);
            END;
          ELSE
            VAR linear: Animate.Linear := NEW(Animate.Linear).init(tf);
            BEGIN
              linear.setVector(
                graph,
                WorldSizeToPts(graph, R2.Sub(vertex.new.pos, vertex.pos)));
              AddAnimationLocked(graph, linear, vertex.mg);
            END;
          END;
          vertex.pos := vertex.new.pos;
        END;
      END;
    END;
  END AnimateGraphVertices;

<* LL.sup >= graph.mu *>

PROCEDURE AnimateGraphEdges (graph: T; tf: Animate.TimeFunction) =
  VAR edges := graph.edges;
  BEGIN
    WHILE edges # NIL DO
      VAR
        edge       : Edge := List.Pop(edges);
        oldVertex0        := edge.vertex0;
        oldVertex1        := edge.vertex1;
        oldControl0       := edge.control0;
        oldControl1       := edge.control1;
        oldPos0           := edge.pos[0];
        oldPos1           := edge.pos[1];
      BEGIN
        IF edge.vertex0 # edge.new.vertex0
             OR edge.vertex1 # edge.new.vertex1
             OR edge.control0 # edge.new.control0
             OR edge.control1 # edge.new.control1 THEN

          edge.vertex0.edges := List.DeleteQ(edge.vertex0.edges, edge);
          edge.vertex1.edges := List.DeleteQ(edge.vertex1.edges, edge);

          IF edge.control0 # NIL THEN
            edge.control0.edges := List.DeleteQ(edge.control0.edges, edge);
          END;
          IF edge.control1 # NIL THEN
            edge.control1.edges := List.DeleteQ(edge.control1.edges, edge);
          END;

          edge.vertex0 := edge.new.vertex0;
          edge.vertex1 := edge.new.vertex1;

          List.Push(edge.vertex0.edges, edge);
          List.Push(edge.vertex1.edges, edge);

          edge.control0 := edge.new.control0;
          edge.control1 := edge.new.control1;
          edge.straight := (edge.control0 = NIL) AND (edge.control1 = NIL);

          IF edge.control0 # NIL THEN
            List.Push(edge.control0.edges, edge);
          END;
          IF edge.control1 # NIL THEN
            List.Push(edge.control1.edges, edge);
          END;
        END;

        IF NOT ((edge.vertex0 = oldVertex0 AND NOT edge.vertex0.animated)
                  AND (edge.vertex1 = oldVertex1
                         AND NOT edge.vertex1.animated)
                  AND (edge.control0 = oldControl0
                         AND (edge.control0 = NIL
                                OR NOT edge.control0.animated))
                  AND (edge.control1 = oldControl1
                         AND (edge.control1 = NIL
                                OR NOT edge.control1.animated))
                  AND edge.isLine = edge.straight) THEN
          IF NOT edge.straight AND edge.isLine THEN
            (* was straight, now curved.  change to straight bezier before
               animation. *)
            edge.group.remove(graph, edge.end[0]);
            edge.group.remove(graph, edge.end[1]);
            VAR path := NEW(RealPath.T);
            BEGIN
              path.init();
              edge.mg := NEW(MG.Shape, weight := Pts.FromMM(edge.widthMM),
                             color := edge.colorScheme).init(
                           R2.Origin, path, fill := FALSE);
            END;
            edge.end[0] := NIL;
            edge.end[1] := NIL;
            edge.isLine := FALSE;
            edge.group.addBefore(graph, edge.mg, NIL);
          END;

          IF edge.isLine THEN
            IF edge.vertex0 = oldVertex0 AND edge.vertex0.animated
                 AND edge.vertex0.path # NIL THEN
              VAR
                path: AlongGivenPath := NEW(
                                          AlongGivenPath, graph := graph,
                                          pos := edge.pos[0],
                                          path := edge.vertex0.path).init(
                                          tf);
              BEGIN
                AddAnimationLocked(graph, path, edge.end[0]);
              END;
            ELSIF edge.vertex0 # oldVertex0 OR edge.vertex0.animated THEN
              VAR linear: Animate.Linear := NEW(Animate.Linear).init(tf);
              BEGIN
                linear.setVector(
                  graph, WorldSizeToPts(graph, R2.Sub(edge.vertex0.new.pos,
                                                      edge.pos[0])));
                AddAnimationLocked(graph, linear, edge.end[0]);
              END;
            END;
            IF edge.vertex1 = oldVertex1 AND edge.vertex1.animated
                 AND edge.vertex1.path # NIL THEN
              VAR
                path: AlongGivenPath := NEW(
                                          AlongGivenPath, graph := graph,
                                          pos := edge.pos[1],
                                          path := edge.vertex1.path).init(
                                          tf);
              BEGIN
                AddAnimationLocked(graph, path, edge.end[1]);
              END;
            ELSIF edge.vertex1 # oldVertex1 OR edge.vertex1.animated THEN
              VAR linear: Animate.Linear := NEW(Animate.Linear).init(tf);
              BEGIN
                linear.setVector(
                  graph, WorldSizeToPts(graph, R2.Sub(edge.vertex1.new.pos,
                                                      edge.pos[1])));
                AddAnimationLocked(graph, linear, edge.end[1]);
              END;
            END;
          ELSE
            VAR bezierAnimation: BezierAnimation;
            BEGIN
              bezierAnimation := NEW(BezierAnimation, graph := graph);
              IF edge.vertex0 = oldVertex0 AND edge.vertex0.animated
                   AND edge.vertex0.path # NIL THEN
                bezierAnimation.pathA := edge.vertex0.path;
              ELSE
                bezierAnimation.pathA :=
                  NEW(StraightPath, p0 := edge.pos[0],
                      p1 := edge.vertex0.new.pos);
              END;
              IF edge.control0 # NIL THEN
                IF edge.control0 = oldControl0 AND edge.control0.animated
                     AND edge.control0.path # NIL THEN
                  bezierAnimation.pathB := edge.control0.path;
                ELSE
                  bezierAnimation.pathB :=
                    NEW(StraightPath, p0 := edge.cpos[0],
                        p1 := edge.control0.new.pos);
                END;
              ELSE
                bezierAnimation.pathB :=
                  NEW(StraightPath, p0 := edge.cpos[0],
                      p1 := edge.vertex0.new.pos);
              END;
              IF edge.control1 # NIL THEN
                IF edge.control1 = oldControl1 AND edge.control1.animated
                     AND edge.control1.path # NIL THEN
                  bezierAnimation.pathC := edge.control1.path;
                ELSE
                  bezierAnimation.pathC :=
                    NEW(StraightPath, p0 := edge.cpos[1],
                        p1 := edge.control1.new.pos);
                END;
              ELSE
                bezierAnimation.pathC :=
                  NEW(StraightPath, p0 := edge.cpos[1],
                      p1 := edge.vertex1.new.pos);
              END;
              IF edge.vertex1 = oldVertex1 AND edge.vertex1.animated
                   AND edge.vertex1.path # NIL THEN
                bezierAnimation.pathD := edge.vertex1.path;
              ELSE
                bezierAnimation.pathD :=
                  NEW(StraightPath, p0 := edge.pos[1],
                      p1 := edge.vertex1.new.pos);
              END;
              bezierAnimation := bezierAnimation.init(tf);
              AddAnimationLocked(graph, bezierAnimation, edge.mg);
            END;
          END;
          edge.pos[0] := edge.vertex0.new.pos;
          edge.pos[1] := edge.vertex1.new.pos;
          IF edge.control0 # NIL THEN
            edge.cpos[0] := edge.control0.new.pos;
          ELSE
            edge.cpos[0] := edge.pos[0];
          END;
          IF edge.control1 # NIL THEN
            edge.cpos[1] := edge.control1.new.pos;
          ELSE
            edge.cpos[1] := edge.pos[1];
          END;
          VAR
            new    := ArrowPos(edge);
            vertex := ARRAY [0 .. 1] OF Vertex{edge.vertex0, edge.vertex1};
            oldVertex := ARRAY [0 .. 1] OF Vertex{oldVertex0, oldVertex1};
            oldPos    := ARRAY [0 .. 1] OF R2.T{oldPos0, oldPos1};
          BEGIN
            FOR i := 0 TO 1 DO
              IF edge.arrow[i] THEN
                IF vertex[i] = oldVertex[i] AND vertex[i].animated
                     AND vertex[i].path # NIL THEN
                  VAR
                    oldOffset, newOffset: ARRAY [0 .. 1], [0 .. 1] OF R2.T;
                  BEGIN
                    FOR j := 0 TO 1 DO
                      FOR k := 0 TO 1 DO
                        oldOffset[j][k] :=
                          R2.Sub(edge.arrowPos[i][j][k], oldPos[i]);
                        newOffset[j][k] :=
                          R2.Sub(new[i][j][k], edge.pos[i]);
                      END;
                    END;
                    FOR j := 0 TO 1 DO
                      FOR k := 0 TO 1 DO
                        VAR
                          path: AlongGivenPath := NEW(AlongGivenPath,
                                                      graph := graph,
                                                      pos := edge.arrowPos[
                                                               i][j][k],
                                                      path :=
                                                        NEW(OffsetPath,
                                                            path :=
                                                              vertex[
                                                                i].path,
                                                            offset0 :=
                                                              oldOffset[j][
                                                                k],
                                                            offset1 :=
                                                              newOffset[j][
                                                                k])).init(
                                                    tf);
                        BEGIN
                          AddAnimationLocked(
                            graph, path, edge.arrowEnd[i][j][k]);
                        END;
                      END;
                    END;
                  END;
                ELSE
                  FOR j := 0 TO 1 DO
                    FOR k := 0 TO 1 DO
                      VAR
                        linear: Animate.Linear := NEW(Animate.Linear).init(
                                                    tf);
                      BEGIN
                        linear.setVector(
                          graph, WorldSizeToPts(
                                   graph, R2.Sub(new[i][j][k],
                                                 edge.arrowPos[i][j][k])));
                        AddAnimationLocked(
                          graph, linear, edge.arrowEnd[i][j][k]);
                      END;
                    END;
                  END;
                END;
              END;
            END;
            edge.arrowPos := new;
          END;
        END;
      END;
    END;
  END AnimateGraphEdges;

<* LL.sup >= graph.mu *>

PROCEDURE AnimateGraphVertexHighlights (graph: T; tf: Animate.TimeFunction) =
  VAR vertexHighlights := graph.vertexHighlights;
  BEGIN
    WHILE vertexHighlights # NIL DO
      VAR vertexHighlight: VertexHighlight := List.Pop(vertexHighlights);
      BEGIN
        IF vertexHighlight.new.vertex # vertexHighlight.vertex THEN
          VAR animation: Animate.T;
          BEGIN
            VAR
              linearResize: LinearResize;
              center0PP := WorldPosToPts(graph, vertexHighlight.pos);
              center1PP := WorldPosToPts(
                             graph, vertexHighlight.new.vertex.new.pos);
              size0MM := R2.T{
                           MAX(ABS(vertexHighlight.vertex.sizeMM[0])
                                 + 2.0 * vertexHighlight.borderMM[0], 0.0),
                           MAX(ABS(vertexHighlight.vertex.sizeMM[1])
                                 + 2.0 * vertexHighlight.borderMM[1], 0.0)};
              size1MM := R2.T{
                           MAX(ABS(vertexHighlight.new.vertex.sizeMM[0])
                                 + 2.0 * vertexHighlight.borderMM[0], 0.0),
                           MAX(ABS(vertexHighlight.new.vertex.sizeMM[1])
                                 + 2.0 * vertexHighlight.borderMM[1], 0.0)};
            BEGIN
              linearResize :=
                NEW(LinearResize, graph := graph,
                    shape := vertexHighlight.vertex.shape,
                    corner0 := ARRAY [0 .. 1] OF
                                 R2.T{SWFromCenterMM(center0PP, size0MM),
                                      SWFromCenterMM(center1PP, size1MM)},
                    corner1 := ARRAY [0 .. 1] OF
                                 R2.T{NEFromCenterMM(center0PP, size0MM),
                                      NEFromCenterMM(center1PP, size1MM)});
              EVAL linearResize.init(tf);
              animation := linearResize;
            END;
            AddAnimationLocked(graph, animation, vertexHighlight.mg);
          END;
          vertexHighlight.pos := vertexHighlight.new.vertex.new.pos;
        ELSIF vertexHighlight.vertex.animated THEN
          IF vertexHighlight.vertex.path # NIL THEN
            VAR
              path: AlongGivenPath := NEW(
                                        AlongGivenPath, graph := graph,
                                        pos := vertexHighlight.pos,
                                        path := vertexHighlight.vertex.path).init(
                                        tf);
            BEGIN
              AddAnimationLocked(graph, path, vertexHighlight.mg);
            END;
          ELSE
            VAR linear: Animate.Linear := NEW(Animate.Linear).init(tf);
            BEGIN
              linear.setVector(
                graph, WorldSizeToPts(
                         graph, R2.Sub(vertexHighlight.vertex.new.pos,
                                       vertexHighlight.pos)));
              AddAnimationLocked(graph, linear, vertexHighlight.mg);
            END;
          END;
          vertexHighlight.pos := vertexHighlight.vertex.new.pos;
        END;
        IF vertexHighlight.vertex # vertexHighlight.new.vertex THEN
          vertexHighlight.vertex.vertexHighlights :=
            List.DeleteQ(
              vertexHighlight.vertex.vertexHighlights, vertexHighlight);
          vertexHighlight.vertex := vertexHighlight.new.vertex;
          List.Push(
            vertexHighlight.vertex.vertexHighlights, vertexHighlight);
        END;
      END;
    END;
  END AnimateGraphVertexHighlights;

<* LL.sup >= graph.mu *>

PROCEDURE AnimateGraphPolygons (graph: T; tf: Animate.TimeFunction) =
  VAR polygons := graph.polygons;
  BEGIN
    WHILE polygons # NIL DO
      VAR
        polygon     : Polygon := List.Pop(polygons);
        oldVertices           := polygon.vertices;
        oldPositions          := polygon.pos;
      BEGIN
        IF NOT List.Equal(polygon.vertices, polygon.new.vertices) THEN
          VAR vertices := polygon.vertices;
          BEGIN
            WHILE vertices # NIL DO
              VAR vertex: Vertex := List.Pop(vertices);
              BEGIN
                vertex.polygons := List.DeleteQ(vertex.polygons, polygon);
              END;
            END;
          END;

          polygon.vertices := polygon.new.vertices;

          VAR vertices := polygon.vertices;
          BEGIN
            WHILE vertices # NIL DO
              VAR vertex: Vertex := List.Pop(vertices);
              BEGIN
                List.Push(vertex.polygons, polygon);
              END;
            END;
          END;
        END;

        VAR
          vertices          := polygon.vertices;
          positions: List.T := NIL;
        BEGIN
          WHILE vertices # NIL DO
            VAR vertex: Vertex := List.Pop(vertices);
            BEGIN
              List.Push(positions, NewR2(vertex.new.pos));
            END;
          END;
          polygon.pos := List.ReverseD(positions);
        END;

        VAR anyAnimated := FALSE;
        BEGIN
          VAR vertices := polygon.vertices;
          BEGIN
            WHILE vertices # NIL DO
              VAR vertex: Vertex := List.Pop(vertices);
              BEGIN
                IF vertex.animated THEN anyAnimated := TRUE; EXIT; END;
              END;
            END;
          END;
          IF NOT List.Equal(polygon.vertices, oldVertices) OR anyAnimated THEN
            VAR paths: List.T := NIL;
            BEGIN
              VAR
                list                      := polygon.vertices;
                oldList                   := oldVertices;
                positionList              := polygon.pos;
                oldPositionList           := oldPositions;
                vertex, oldVertex: Vertex;
                pos, oldPos      : R2.T;
              BEGIN
                WHILE list # NIL OR oldList # NIL DO
                  IF list # NIL THEN
                    vertex := List.Pop(list);
                    pos := NARROW(List.Pop(positionList), REF R2.T)^;
                  END;
                  IF oldList # NIL THEN
                    oldVertex := List.Pop(oldList);
                    oldPos := NARROW(List.Pop(oldPositionList), REF R2.T)^;
                  END;
                  VAR path: AnimationPath;
                  BEGIN
                    IF vertex = oldVertex AND vertex.animated
                         AND vertex.path # NIL THEN
                      path := vertex.path;
                    ELSE
                      path := NEW(StraightPath, p0 := oldPos, p1 := pos);
                    END;
                    List.Push(paths, path);
                  END;
                END;
              END;
              paths := List.ReverseD(paths);
              VAR
                polygonAnimation := NEW(PolygonAnimation, graph := graph,
                                        paths := paths);
              BEGIN
                polygonAnimation := polygonAnimation.init(tf);
                AddAnimationLocked(graph, polygonAnimation, polygon.mg);
              END;
            END;
          END;
        END;
      END;
    END;
  END AnimateGraphPolygons;

<* LL.sup >= graph.mu *>

PROCEDURE PostAnimateGraphVertices (graph: T) =
  VAR vertices := graph.vertices;
  BEGIN
    WHILE vertices # NIL DO
      VAR vertex: Vertex := List.Pop(vertices);
      BEGIN
        vertex.animated := FALSE;
      END;
    END;
  END PostAnimateGraphVertices;

<* LL.sup >= graph.mu *>

PROCEDURE PostAnimateGraphEdges (graph: T): BOOLEAN =
  VAR
    edges     := graph.edges;
    redisplay := FALSE;
  BEGIN
    WHILE edges # NIL DO
      VAR edge: Edge := List.Pop(edges);
      BEGIN
        IF edge.straight AND NOT edge.isLine THEN
          (* was curved, now straight.  replace with straight line. *)
          edge.group.remove(graph, edge.mg);
          edge.mg :=
            NEW(MG.Line, weight := Pts.FromMM(edge.widthMM),
                color := edge.colorScheme).init(R2.Origin, R2.Origin);
          edge.end[0] :=
            NEW(MG.LineEnd, line := edge.mg, controlsFrom := TRUE).init();
          edge.end[1] :=
            NEW(MG.LineEnd, line := edge.mg, controlsFrom := FALSE).init();
          edge.isLine := TRUE;
          edge.group.addBefore(graph, edge.end[0], NIL);
          edge.group.addBefore(graph, edge.end[1], NIL);
          RefreshEdge(edge);
          redisplay := TRUE;
        END;
      END;
    END;
    RETURN redisplay;
  END PostAnimateGraphEdges;

<* LL.sup >= graph.mu *>

PROCEDURE PostAnimateGraphVertexHighlights (graph: T): BOOLEAN =
  VAR
    vertexHighlights := graph.vertexHighlights;
    redisplay        := FALSE;
  BEGIN
    WHILE vertexHighlights # NIL DO
      VAR vertexHighlight: VertexHighlight := List.Pop(vertexHighlights);
      BEGIN
        IF AdjustVertexHighlightSizeandShape(vertexHighlight) THEN
          redisplay := TRUE;
        END;
      END;
    END;
    RETURN redisplay;
  END PostAnimateGraphVertexHighlights;

<* LL.sup >= graph.mu *>

PROCEDURE PostAnimateGraphPolygons (<* UNUSED *> graph: T): BOOLEAN =
  BEGIN
    RETURN FALSE;
  END PostAnimateGraphPolygons;

<* LL.sup >= graph.mu *>

PROCEDURE ClearGraph (graph: T) =
  BEGIN
    WHILE graph.vertices # NIL DO
      VAR vertex: Vertex := List.First(graph.vertices);
      BEGIN
        vertex.remove();
      END;
    END;
  END ClearGraph;

<* LL.sup >= graph.mu *>

PROCEDURE GraphVerticesAt (graph: T; pixelRect: Rect.T):
  List.T (* OF Vertex *) =
  VAR
    vertices         := graph.vertices;
    list    : List.T := NIL;
    hRectMM := R2.T{FLOAT(pixelRect.west - graph.rect.west) / graph.res[
                      Axis.T.Hor], FLOAT(pixelRect.east - graph.rect.west)
                                     / graph.res[Axis.T.Hor]};
    vRectMM := R2.T{
                 FLOAT(graph.rect.south - pixelRect.south) / graph.res[
                   Axis.T.Ver], FLOAT(graph.rect.south - pixelRect.north)
                                  / graph.res[Axis.T.Ver]};
  BEGIN
    WHILE vertices # NIL DO
      VAR
        vertex  : Vertex := List.Pop(vertices);
        centerMM         := WorldPosToMM(graph, vertex.pos);
        sizeMM           := Abs(vertex.sizeMM);
        hVertexMM := R2.T{centerMM[0] - sizeMM[0] / 2.0,
                          centerMM[0] + sizeMM[0] / 2.0};
        vVertexMM := R2.T{centerMM[1] - sizeMM[1] / 2.0,
                          centerMM[1] + sizeMM[1] / 2.0};
      BEGIN
        IF R2Intersect(hRectMM, hVertexMM)
             AND R2Intersect(vRectMM, vVertexMM) THEN
          CASE vertex.shape OF
          | VertexShape.Rectangle => List.Push(list, vertex);
          | VertexShape.Ellipse =>
              IF hRectMM[0] < hVertexMM[0] AND hVertexMM[1] < hRectMM[1]
                   OR vRectMM[0] < vVertexMM[0]
                        AND vVertexMM[1] < vRectMM[1] THEN
                List.Push(list, vertex);
              ELSE
                VAR
                  dx0 := (centerMM[0] - hRectMM[0]) / sizeMM[0];
                  dx1 := (centerMM[0] - hRectMM[1]) / sizeMM[0];
                  dy0 := (centerMM[1] - vRectMM[0]) / sizeMM[1];
                  dy1 := (centerMM[1] - vRectMM[1]) / sizeMM[1];
                BEGIN
                  IF dx0 * dx0 + dy0 * dy0 < 0.25
                       OR dx0 * dx0 + dy1 * dy1 < 0.25
                       OR dx1 * dx1 + dy0 * dy0 < 0.25
                       OR dx1 * dx1 + dy1 * dy1 < 0.25 THEN
                    List.Push(list, vertex);
                  END;
                END;
              END;
          END;
        END;
      END;
    END;
    RETURN List.ReverseD(list);
  END GraphVerticesAt;

<* LL.sup >= graph.mu *>

PROCEDURE GraphEdgesAt (graph: T; pixelRect: Rect.T):
  List.T (* OF Edge *) =
  VAR
    edges         := graph.edges;
    list : List.T := NIL;
    hRectMM := R2.T{FLOAT(pixelRect.west - graph.rect.west) / graph.res[
                      Axis.T.Hor], FLOAT(pixelRect.east - graph.rect.west)
                                     / graph.res[Axis.T.Hor]};
    vRectMM := R2.T{
                 FLOAT(graph.rect.south - pixelRect.south) / graph.res[
                   Axis.T.Ver], FLOAT(graph.rect.south - pixelRect.north)
                                  / graph.res[Axis.T.Ver]};
  BEGIN
    WHILE edges # NIL DO
      VAR edge: Edge := List.Pop(edges);
      BEGIN
        IF EdgeInBox(edge, hRectMM, vRectMM) THEN
          List.Push(list, edge);
        END;
      END;
    END;
    RETURN list;
  END GraphEdgesAt;

<* LL.sup >= graph.mu *>

PROCEDURE EdgeInBox (edge: Edge; hBoxMM, vBoxMM: R2.T): BOOLEAN =
  BEGIN
    WITH graph = edge.graph DO
      hBoxMM := R2.T{hBoxMM[0] - MAX(ABS(edge.widthMM) / 2.0,
                                     2.0 / graph.res[Axis.T.Hor]),
                     hBoxMM[1] + MAX(ABS(edge.widthMM) / 2.0,
                                     2.0 / graph.res[Axis.T.Hor])};
      vBoxMM := R2.T{vBoxMM[0] - MAX(ABS(edge.widthMM) / 2.0,
                                     2.0 / graph.res[Axis.T.Ver]),
                     vBoxMM[1] + MAX(ABS(edge.widthMM) / 2.0,
                                     2.0 / graph.res[Axis.T.Ver])};
      IF edge.straight THEN
        RETURN StraightEdgeInBox(edge, hBoxMM, vBoxMM);
      ELSE
        RETURN BezierEdgeInBox(edge, hBoxMM, vBoxMM);
      END;
    END;
  END EdgeInBox;

<* LL.sup >= graph.mu *>

PROCEDURE StraightEdgeInBox (edge: Edge; hBoxMM, vBoxMM: R2.T): BOOLEAN =
  BEGIN
    WITH graph = edge.graph DO
      VAR
        pos0MM := WorldPosToMM(graph, edge.pos[0]);
        pos1MM := WorldPosToMM(graph, edge.pos[1]);
      BEGIN
        VAR
          hEdgeMM := R2.T{MIN(pos0MM[0], pos1MM[0]),
                          MAX(pos0MM[0], pos1MM[0])};
          vEdgeMM := R2.T{MIN(pos0MM[1], pos1MM[1]),
                          MAX(pos0MM[1], pos1MM[1])};
        BEGIN
          IF NOT (R2Intersect(hBoxMM, hEdgeMM)
                    AND R2Intersect(vBoxMM, vEdgeMM)) THEN
            RETURN FALSE;
          END;
        END;
        VAR
          x0 := pos0MM[0];
          y0 := pos0MM[1];
          x1 := pos1MM[0];
          y1 := pos1MM[1];
        BEGIN
          IF hBoxMM[0] <= x0 AND x0 <= hBoxMM[1] AND vBoxMM[0] <= y0
               AND y0 <= vBoxMM[1]
               OR hBoxMM[0] <= x1 AND x1 <= hBoxMM[1] AND vBoxMM[0] <= y1
                    AND y1 <= vBoxMM[1] THEN
            RETURN TRUE;
          END;
          VAR t := (hBoxMM[0] - x0) / (x1 - x0);
          BEGIN
            IF 0.0 <= t AND t <= 1.0 THEN
              VAR y := y0 + (y1 - y0) * t;
              BEGIN
                IF vBoxMM[0] <= y AND y <= vBoxMM[1] THEN RETURN TRUE; END;
              END;
            END;
          END;
          VAR t := (hBoxMM[1] - x0) / (x1 - x0);
          BEGIN
            IF 0.0 <= t AND t <= 1.0 THEN
              VAR y := y0 + (y1 - y0) * t;
              BEGIN
                IF vBoxMM[0] <= y AND y <= vBoxMM[1] THEN RETURN TRUE; END;
              END;
            END;
          END;
          VAR t := (vBoxMM[0] - y0) / (y1 - y0);
          BEGIN
            IF 0.0 <= t AND t <= 1.0 THEN
              VAR x := x0 + (x1 - x0) * t;
              BEGIN
                IF hBoxMM[0] <= x AND x <= hBoxMM[1] THEN RETURN TRUE; END;
              END;
            END;
          END;
        END;
      END;
    END;
    RETURN FALSE;
  END StraightEdgeInBox;

<* LL.sup >= graph.mu *>

PROCEDURE BezierEdgeInBox (edge: Edge; hBoxMM, vBoxMM: R2.T): BOOLEAN =
  PROCEDURE BezierStep (c0MM, c1MM, c2MM, c3MM: R2.T): BOOLEAN =
    BEGIN
      VAR end := c0MM;
      BEGIN
        IF (hBoxMM[0] <= end[0] AND end[0] <= hBoxMM[1])
             AND (vBoxMM[0] <= end[1] AND end[1] <= vBoxMM[1]) THEN
          RETURN TRUE;
        END;
      END;
      VAR end := c3MM;
      BEGIN
        IF (hBoxMM[0] <= end[0] AND end[0] <= hBoxMM[1])
             AND (vBoxMM[0] <= end[1] AND end[1] <= vBoxMM[1]) THEN
          RETURN TRUE;
        END;
      END;
      VAR hBoundsMM := CubicBounds(c0MM[0], c1MM[0], c2MM[0], c3MM[0]);
      BEGIN
        IF NOT R2Intersect(hBoxMM, hBoundsMM) THEN RETURN FALSE; END;
        VAR vBoundsMM := CubicBounds(c0MM[1], c1MM[1], c2MM[1], c3MM[1]);
        BEGIN
          IF NOT R2Intersect(vBoxMM, vBoundsMM) THEN RETURN FALSE; END;
          IF ABS(hBoundsMM[1] - hBoundsMM[0]) < hHalfPixelMM
               AND ABS(vBoundsMM[1] - vBoundsMM[0]) < hHalfPixelMM THEN
            RETURN FALSE;
          END;
        END;
      END;
      VAR
        d0MM := c0MM;
        d1MM := c1MM;
        d2MM := c2MM;
        d3MM := c3MM;
      BEGIN
        SubCubic2(d0MM, d1MM, d2MM, d3MM, half := 0);
        IF BezierStep(d0MM, d1MM, d2MM, d3MM) THEN RETURN TRUE; END;
      END;
      VAR
        d0MM := c0MM;
        d1MM := c1MM;
        d2MM := c2MM;
        d3MM := c3MM;
      BEGIN
        SubCubic2(d0MM, d1MM, d2MM, d3MM, half := 1);
        IF BezierStep(d0MM, d1MM, d2MM, d3MM) THEN RETURN TRUE; END;
      END;
      RETURN FALSE;
    END BezierStep;
  VAR hHalfPixelMM, vHalfPixelMM: REAL;
  BEGIN
    WITH graph = edge.graph DO
      hHalfPixelMM := 0.5 / graph.res[Axis.T.Hor];
      vHalfPixelMM := 0.5 / graph.res[Axis.T.Ver];
      RETURN BezierStep(WorldPosToMM(graph, edge.pos[0]),
                        WorldPosToMM(graph, edge.cpos[0]),
                        WorldPosToMM(graph, edge.cpos[1]),
                        WorldPosToMM(graph, edge.pos[1]));
    END;
  END BezierEdgeInBox;

<* LL.sup >= graph.mu *>

PROCEDURE GraphVertexHighlightsAt (graph: T; pixelRect: Rect.T):
  List.T (* OF VertexHighlight *) =
  VAR
    vertexHighlights         := graph.vertexHighlights;
    list            : List.T := NIL;
    hRectMM := R2.T{FLOAT(pixelRect.west - graph.rect.west) / graph.res[
                      Axis.T.Hor], FLOAT(pixelRect.east - graph.rect.west)
                                     / graph.res[Axis.T.Hor]};
    vRectMM := R2.T{
                 FLOAT(graph.rect.south - pixelRect.south) / graph.res[
                   Axis.T.Ver], FLOAT(graph.rect.south - pixelRect.north)
                                  / graph.res[Axis.T.Ver]};
  BEGIN
    WHILE vertexHighlights # NIL DO
      VAR
        vertexHighlight: VertexHighlight := List.Pop(vertexHighlights);
        centerMM := WorldPosToMM(graph, vertexHighlight.pos);
        sizeMM := R2.T{ABS(vertexHighlight.vertex.sizeMM[0])
                         + 2.0 * vertexHighlight.borderMM[0],
                       ABS(vertexHighlight.vertex.sizeMM[1])
                         + 2.0 * vertexHighlight.borderMM[1]};
        hHighlightMM := R2.T{centerMM[0] - sizeMM[0] / 2.0,
                             centerMM[0] + sizeMM[0] / 2.0};
        vHighlightMM := R2.T{centerMM[1] - sizeMM[1] / 2.0,
                             centerMM[1] + sizeMM[1] / 2.0};
      BEGIN
        IF R2Intersect(hRectMM, hHighlightMM)
             AND R2Intersect(vRectMM, vHighlightMM) THEN
          CASE vertexHighlight.shape OF
          | VertexShape.Rectangle => List.Push(list, vertexHighlight);
          | VertexShape.Ellipse =>
              IF hRectMM[0] < hHighlightMM[0]
                   AND hHighlightMM[1] < hRectMM[1]
                   OR vRectMM[0] < vHighlightMM[0]
                        AND vHighlightMM[1] < vRectMM[1] THEN
                List.Push(list, vertexHighlight);
              ELSE
                VAR
                  dx0 := (centerMM[0] - hRectMM[0]) / (sizeMM[0]);
                  dx1 := (centerMM[0] - hRectMM[1]) / (sizeMM[0]);
                  dy0 := (centerMM[1] - vRectMM[0]) / (sizeMM[1]);
                  dy1 := (centerMM[1] - vRectMM[1]) / (sizeMM[1]);
                BEGIN
                  IF dx0 * dx0 + dy0 * dy0 < 0.25
                       OR dx0 * dx0 + dy1 * dy1 < 0.25
                       OR dx1 * dx1 + dy0 * dy0 < 0.25
                       OR dx1 * dx1 + dy1 * dy1 < 0.25 THEN
                    List.Push(list, vertexHighlight);
                  END;
                END;
              END;
          END;
        END;
      END;
    END;
    RETURN List.ReverseD(list);
  END GraphVertexHighlightsAt;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE MoveVertex (vertex  : Vertex;
                      pos     : R2.T;
                      animated: BOOLEAN       := FALSE;
                      path    : AnimationPath := NIL    ) =
  BEGIN
    WITH graph = vertex.graph DO
      vertex.new.pos := pos;
      IF animated THEN
        vertex.animated := TRUE;
        vertex.path := path;
      ELSE
        vertex.animated := FALSE;
        MG.TranslateToLocked(
          vertex.mg, graph, Finite2(WorldPosToPts(graph, pos)),
          center := TRUE);
        vertex.pos := pos;
        VAR edges := vertex.edges;
        BEGIN
          WHILE edges # NIL DO
            VAR edge: Edge := List.Pop(edges);
            BEGIN
              MoveEdge(edge, edge.vertex0, edge.vertex1, edge.control0,
                       edge.control1, animated := FALSE);
            END;
          END;
        END;
        VAR vertexHighlights := vertex.vertexHighlights;
        BEGIN
          WHILE vertexHighlights # NIL DO
            VAR
              vertexHighlight: VertexHighlight := List.Pop(
                                                    vertexHighlights);
            BEGIN
              MoveVertexHighlight(
                vertexHighlight, vertexHighlight.vertex, animated := FALSE);
            END;
          END;
        END;
        VAR polygons := vertex.polygons;
        BEGIN
          WHILE polygons # NIL DO
            VAR polygon: Polygon := List.Pop(polygons);
            BEGIN
              MovePolygon(polygon, polygon.vertices, animated := FALSE);
            END;
          END;
        END;
        IF graph.animations > 0 THEN graph.needRefresh := TRUE; END;
      END;
    END;
  END MoveVertex;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE SetVertexSizeMM (vertex: Vertex; sizeMM: R2.T) =
  BEGIN
    vertex.sizeMM := sizeMM;
    vertex.sizeW := R2.Origin;
    AdjustVertex(vertex);
  END SetVertexSizeMM;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE SetVertexSizeW (vertex: Vertex; sizeW: R2.T) =
  BEGIN
    vertex.sizeW := sizeW;
    WITH graph = vertex.graph DO
      IF vertex.sizeW[0] # 0.0 OR vertex.sizeW[1] # 0.0 THEN
        vertex.sizeMM := Abs(WorldSizeToMM(graph, vertex.sizeW));
        AdjustVertex(vertex);
      END;
    END;
  END SetVertexSizeW;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE SetVertexShape (vertex: Vertex; shape: VertexShape) =
  BEGIN
    IF shape = vertex.shape THEN RETURN; END;
    WITH graph = vertex.graph DO
      vertex.shape := shape;
      vertex.group.remove(graph, vertex.mg);
      CASE vertex.shape OF
      | VertexShape.Rectangle =>
          vertex.mg := NEW(MG.Rectangle, color := vertex.colorScheme,
                           label := vertex.label, font := vertex.font,
                           weight := 0.0).init(R2.Origin, R2.Origin);
      | VertexShape.Ellipse =>
          vertex.mg := NEW(MG.Ellipse, color := vertex.colorScheme,
                           label := vertex.label, font := vertex.font,
                           weight := 0.0).init(R2.Origin, R2.Origin);
      END;
      vertex.group.addBefore(graph, vertex.mg, NIL);
      AdjustVertex(vertex);
    END;
  END SetVertexShape;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE SetVertexColor (vertex: Vertex; color: PaintOp.T) =
  BEGIN
    WITH graph = vertex.graph DO
      vertex.color := color;
      vertex.colorScheme :=
        PaintOpCache.MakeColorScheme(vertex.color, vertex.fontColor);
      vertex.mg.setColor(graph, vertex.colorScheme);
    END;
  END SetVertexColor;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE SetVertexLabel (vertex: Vertex; label: TEXT) =
  BEGIN
    vertex.label := label;
    WITH graph = vertex.graph DO vertex.mg.setLabel(graph, label); END;
  END SetVertexLabel;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE SetVertexFont (vertex: Vertex; font: Font.T) =
  BEGIN
    WITH graph = vertex.graph DO
      vertex.font := font;
      vertex.mg.setFont(graph, vertex.font);
    END;
  END SetVertexFont;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE SetVertexFontColor (vertex: Vertex; fontColor: PaintOp.T) =
  BEGIN
    WITH graph = vertex.graph DO
      vertex.fontColor := fontColor;
      vertex.colorScheme :=
        PaintOpCache.MakeColorScheme(vertex.color, vertex.fontColor);
      vertex.mg.setColor(graph, vertex.colorScheme);
    END;
  END SetVertexFontColor;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE SetVertexBorderMM (vertex: Vertex; borderMM: REAL) =
  BEGIN
    vertex.borderMM := borderMM;
    AdjustVertex(vertex);
  END SetVertexBorderMM;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE SetVertexBorderColor (vertex: Vertex; borderColor: PaintOp.T) =
  BEGIN
    WITH graph = vertex.graph DO vertex.borderColor := borderColor; END;
  END SetVertexBorderColor;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE VertexToFront (vertex: Vertex) =
  BEGIN
    WITH graph = vertex.graph DO
      graph.vertexGroup.remove(graph, vertex.group);
      graph.vertexGroup.addBefore(graph, vertex.group, NIL);
      graph.vertices := List.DeleteQ(graph.vertices, vertex);
      List.Push(graph.vertices, vertex);
    END;
  END VertexToFront;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE VertexToBack (vertex: Vertex) =
  BEGIN
    WITH graph = vertex.graph DO
      graph.vertexGroup.remove(graph, vertex.group);
      graph.vertexGroup.addAfter(graph, vertex.group, NIL);
      graph.vertices := List.DeleteQ(graph.vertices, vertex);
      graph.vertices := List.Append(graph.vertices, List.List1(vertex));
    END;
  END VertexToBack;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE RemoveVertex (vertex: Vertex) =
  BEGIN
    IF NOT vertex.initialized THEN RETURN; END;
    WHILE vertex.edges # NIL DO
      VAR edge: Edge := List.First(vertex.edges);
      BEGIN
        RemoveEdge(edge);
      END;
    END;
    WHILE vertex.vertexHighlights # NIL DO
      VAR
        vertexHighlight: VertexHighlight := List.First(
                                              vertex.vertexHighlights);
      BEGIN
        RemoveVertexHighlight(vertexHighlight);
      END;
    END;
    WHILE vertex.polygons # NIL DO
      VAR polygon: Polygon := List.First(vertex.polygons);
      BEGIN
        RemovePolygon(polygon);
      END;
    END;
    WITH graph = vertex.graph DO
      graph.vertices := List.DeleteQ(graph.vertices, vertex);
      graph.vertexGroup.remove(graph, vertex.group);
    END;
    vertex.initialized := FALSE;
  END RemoveVertex;

<* LL.sup >= edge.vertex0.graph.mu *>

PROCEDURE SetEdgeWidthMM (edge: Edge; widthMM: REAL) =
  BEGIN
    WITH graph = edge.graph DO
      edge.widthMM := widthMM;
      edge.mg.setWeight(graph, Pts.FromMM(edge.widthMM));
      VAR arrows := FALSE;
      BEGIN
        FOR i := 0 TO 1 DO
          IF edge.arrow[i] THEN
            FOR j := 0 TO 1 DO
              edge.arrowLine[i][j].setWeight(
                graph, Pts.FromMM(edge.widthMM));
              arrows := TRUE;
            END;
          END;
        END;
        IF arrows THEN
          RefreshEdge(edge);
        ELSE
          (* work around MG bug *)
          IF NOT edge.isLine THEN RefreshEdge(edge); END;
        END;
      END;
    END;
  END SetEdgeWidthMM;

<* LL.sup >= edge.vertex0.graph.mu *>

PROCEDURE SetEdgeColor (edge: Edge; color: PaintOp.T) =
  BEGIN
    WITH graph = edge.graph DO
      edge.color := color;
      edge.colorScheme :=
        PaintOpCache.MakeColorScheme(edge.color, edge.color);
      edge.mg.setColor(graph, edge.colorScheme);
      edge.mg.setWeight(graph, Pts.FromMM(edge.widthMM)); (* work around MG
                                                             bug *)
      FOR i := 0 TO 1 DO
        IF edge.arrow[i] THEN
          FOR j := 0 TO 1 DO
            edge.arrowLine[i][j].setColor(graph, edge.colorScheme);
            edge.arrowLine[i][j].setWeight(graph, Pts.FromMM(edge.widthMM));
          END;
        END;
      END;
    END;
  END SetEdgeColor;

<* LL.sup >= edge.vertex0.graph.mu *>

PROCEDURE SetEdgeArrow (edge: Edge; arrow: ARRAY [0 .. 1] OF BOOLEAN) =
  BEGIN
    WITH graph = edge.graph DO
      FOR i := 0 TO 1 DO
        IF arrow[i] AND NOT edge.arrow[i] THEN
          FOR j := 0 TO 1 DO
            edge.arrowLine[i][j] :=
              NEW(MG.Line, weight := Pts.FromMM(edge.widthMM),
                  color := edge.colorScheme).init(R2.Origin, R2.Origin);
            edge.arrowEnd[i][j][0] :=
              NEW(MG.LineEnd, line := edge.arrowLine[i][j],
                  controlsFrom := TRUE).init();
            edge.arrowEnd[i][j][1] :=
              NEW(MG.LineEnd, line := edge.arrowLine[i][j],
                  controlsFrom := FALSE).init();
            FOR k := 0 TO 1 DO
              edge.group.addBefore(graph, edge.arrowEnd[i][j][k], NIL);
              edge.arrowPos[i][j][k] := R2.Origin;
            END;
          END;
        ELSIF edge.arrow[i] AND NOT arrow[i] THEN
          FOR j := 0 TO 1 DO
            edge.arrowLine[i][j] := NIL;
            FOR k := 0 TO 1 DO
              edge.group.remove(graph, edge.arrowEnd[i][j][k]);
              edge.arrowEnd[i][j][k] := NIL;
            END;
          END;
        END;
      END;
      edge.arrow := arrow;
      RefreshEdge(edge);
    END;
  END SetEdgeArrow;

<* LL.sup >= edge.vertex0.graph.mu *>

PROCEDURE MoveEdge (edge              : Edge;
                    vertex0, vertex1  : Vertex;
                    control0, control1: Vertex   := NIL;
                    animated          : BOOLEAN  := FALSE) =
  VAR straight := (control0 = NIL) AND (control1 = NIL);
  BEGIN
    WITH graph = edge.graph DO
      <*ASSERT vertex0.graph = graph*>
      <*ASSERT vertex1.graph = graph*>

      IF straight THEN
        <*ASSERT control1 = NIL*>
      ELSE
        <*ASSERT control1 # NIL*>
        <*ASSERT control0.graph = graph*>
        <*ASSERT control1.graph = graph*>
      END;

      edge.new.vertex0 := vertex0;
      edge.new.vertex1 := vertex1;
      edge.new.control0 := control0;
      edge.new.control1 := control1;

      IF NOT animated THEN

        edge.vertex0.edges := List.DeleteQ(edge.vertex0.edges, edge);
        edge.vertex1.edges := List.DeleteQ(edge.vertex1.edges, edge);

        IF edge.control0 # NIL THEN
          edge.control0.edges := List.DeleteQ(edge.control0.edges, edge);
        END;
        IF edge.control1 # NIL THEN
          edge.control1.edges := List.DeleteQ(edge.control1.edges, edge);
        END;

        edge.vertex0 := edge.new.vertex0;
        edge.vertex1 := edge.new.vertex1;

        List.Push(edge.vertex0.edges, edge);
        List.Push(edge.vertex1.edges, edge);

        edge.control0 := edge.new.control0;
        edge.control1 := edge.new.control1;
        edge.straight := (edge.control0 = NIL) AND (edge.control1 = NIL);

        IF edge.control0 # NIL THEN
          List.Push(edge.control0.edges, edge);
        END;
        IF edge.control1 # NIL THEN
          List.Push(edge.control1.edges, edge);
        END;

        IF NOT edge.straight AND edge.isLine THEN
          (* was straight, now curved. *)
          edge.group.remove(graph, edge.end[0]);
          edge.group.remove(graph, edge.end[1]);
          VAR path := NEW(RealPath.T);
          BEGIN
            path.init();
            edge.mg := NEW(MG.Shape, weight := Pts.FromMM(edge.widthMM),
                           color := edge.colorScheme).init(
                         R2.Origin, path, fill := FALSE);
          END;
          edge.end[0] := NIL;
          edge.end[1] := NIL;
          edge.isLine := FALSE;
          edge.group.addBefore(graph, edge.mg, NIL);
        ELSIF edge.straight AND NOT edge.isLine THEN
          (* was curved, now straight. *)
          edge.group.remove(graph, edge.mg);
          edge.mg :=
            NEW(MG.Line, weight := Pts.FromMM(edge.widthMM),
                color := edge.colorScheme).init(R2.Origin, R2.Origin);
          edge.end[0] :=
            NEW(MG.LineEnd, line := edge.mg, controlsFrom := TRUE).init();
          edge.end[1] :=
            NEW(MG.LineEnd, line := edge.mg, controlsFrom := FALSE).init();
          edge.isLine := TRUE;
          edge.group.addBefore(graph, edge.end[0], NIL);
          edge.group.addBefore(graph, edge.end[1], NIL);
        END;
        RefreshEdge(edge);
        IF graph.animations > 0 THEN graph.needRefresh := TRUE; END;
      END;
    END;
  END MoveEdge;

<* LL.sup >= edge.graph.mu *>

PROCEDURE EdgeToFront (edge: Edge) =
  BEGIN
    WITH graph = edge.graph DO
      graph.edgeGroup.remove(graph, edge.group);
      graph.edgeGroup.addBefore(graph, edge.group, NIL);
      graph.edges := List.DeleteQ(graph.edges, edge);
      List.Push(graph.edges, edge);
    END;
  END EdgeToFront;

<* LL.sup >= edge.graph.mu *>

PROCEDURE EdgeToBack (edge: Edge) =
  BEGIN
    WITH graph = edge.graph DO
      graph.edgeGroup.remove(graph, edge.group);
      graph.edgeGroup.addAfter(graph, edge.group, NIL);
      graph.edges := List.DeleteQ(graph.edges, edge);
      graph.edges := List.Append(graph.edges, List.List1(edge));
    END;
  END EdgeToBack;

<* LL.sup >= edge.vertex0.graph.mu *>

PROCEDURE RemoveEdge (edge: Edge) =
  BEGIN
    IF NOT edge.initialized THEN RETURN; END;
    WITH graph = edge.graph DO
      graph.edges := List.DeleteQ(graph.edges, edge);
      edge.vertex0.edges := List.DeleteQ(edge.vertex0.edges, edge);
      edge.vertex1.edges := List.DeleteQ(edge.vertex1.edges, edge);
      IF edge.control0 # NIL THEN
        edge.control0.edges := List.DeleteQ(edge.control0.edges, edge);
      END;
      IF edge.control1 # NIL THEN
        edge.control1.edges := List.DeleteQ(edge.control1.edges, edge);
      END;
      graph.edgeGroup.remove(graph, edge.group);
    END;
    edge.initialized := FALSE;
  END RemoveEdge;

<* LL.sup >= vertex.vertex.graph.mu *>

PROCEDURE MoveVertexHighlight (vertexHighlight: VertexHighlight;
                               vertex         : Vertex;
                               animated       : BOOLEAN           := FALSE) =
  BEGIN
    WITH graph = vertex.graph DO
      <*ASSERT vertex.graph = graph*>
      vertexHighlight.new.vertex := vertex;
      IF NOT animated THEN
        vertexHighlight.pos := vertexHighlight.new.vertex.pos;
        vertexHighlight.vertex.vertexHighlights :=
          List.DeleteQ(
            vertexHighlight.vertex.vertexHighlights, vertexHighlight);
        vertexHighlight.vertex := vertexHighlight.new.vertex;
        List.Push(vertexHighlight.vertex.vertexHighlights, vertexHighlight);
        VAR centerPP := WorldPosToPts(graph, vertexHighlight.vertex.pos);
        BEGIN
          MG.TranslateToLocked(
            vertexHighlight.mg, graph, Finite2(centerPP), center := TRUE);
        END;
        EVAL AdjustVertexHighlightSizeandShape(vertexHighlight);
        IF graph.animations > 0 THEN graph.needRefresh := TRUE; END;
      END;
    END;
  END MoveVertexHighlight;

<* LL.sup >= vertexHighlight.vertex.graph.mu *>

PROCEDURE SetVertexHighlightBorderMM (vertexHighlight: VertexHighlight;
                                      borderMM       : R2.T             ) =
  BEGIN
    BEGIN
      vertexHighlight.borderMM := borderMM;
      EVAL AdjustVertexHighlightSizeandShape(vertexHighlight);
    END;
  END SetVertexHighlightBorderMM;

<* LL.sup >= vertexHighlight.vertex.graph.mu *>

PROCEDURE SetVertexHighlightBorderW (vertexHighlight: VertexHighlight;
                                     borderW        : R2.T             ) =
  BEGIN
    BEGIN
      vertexHighlight.borderW := borderW;
      IF vertexHighlight.borderW[0] # 0.0
           OR vertexHighlight.borderW[1] # 0.0 THEN
        WITH graph = vertexHighlight.graph DO
          vertexHighlight.borderMM :=
            Abs(WorldSizeToMM(graph, vertexHighlight.borderW));
        END;
      END;
      EVAL AdjustVertexHighlightSizeandShape(vertexHighlight);
    END;
  END SetVertexHighlightBorderW;

<* LL.sup >= vertexHighlight.vertex.graph.mu *>

PROCEDURE SetVertexHighlightColor (vertexHighlight: VertexHighlight;
                                   color          : PaintOp.T        ) =
  BEGIN
    vertexHighlight.color := color;
    vertexHighlight.colorScheme :=
      PaintOpCache.MakeColorScheme(vertexHighlight.color, PaintOp.Fg);
    WITH graph = vertexHighlight.graph DO
      vertexHighlight.mg.setColor(graph, vertexHighlight.colorScheme);
    END;
  END SetVertexHighlightColor;

<* LL.sup >= vertexHighlight.graph.mu *>

PROCEDURE VertexHighlightToFront (vertexHighlight: VertexHighlight) =
  BEGIN
    WITH graph = vertexHighlight.graph DO
      graph.vertexHighlightGroup.remove(graph, vertexHighlight.group);
      graph.vertexHighlightGroup.addBefore(
        graph, vertexHighlight.group, NIL);
      graph.vertexHighlights :=
        List.DeleteQ(graph.vertexHighlights, vertexHighlight);
      List.Push(graph.vertexHighlights, vertexHighlight);
    END;
  END VertexHighlightToFront;

<* LL.sup >= vertexHighlight.graph.mu *>

PROCEDURE VertexHighlightToBack (vertexHighlight: VertexHighlight) =
  BEGIN
    WITH graph = vertexHighlight.graph DO
      graph.vertexHighlightGroup.remove(graph, vertexHighlight.group);
      graph.vertexHighlightGroup.addAfter(
        graph, vertexHighlight.group, NIL);
      graph.vertexHighlights :=
        List.DeleteQ(graph.vertexHighlights, vertexHighlight);
      graph.vertexHighlights :=
        List.Append(graph.vertexHighlights, List.List1(vertexHighlight));
    END;
  END VertexHighlightToBack;

<* LL.sup >= vertexHighlight.vertex.graph.mu *>

PROCEDURE RemoveVertexHighlight (vertexHighlight: VertexHighlight) =
  BEGIN
    IF NOT vertexHighlight.initialized THEN RETURN; END;
    WITH graph = vertexHighlight.graph DO
      graph.vertexHighlights :=
        List.DeleteQ(graph.vertexHighlights, vertexHighlight);
      vertexHighlight.vertex.vertexHighlights :=
        List.DeleteQ(
          vertexHighlight.vertex.vertexHighlights, vertexHighlight);
      graph.vertexHighlightGroup.remove(graph, vertexHighlight.group);
    END;
    vertexHighlight.initialized := FALSE;
  END RemoveVertexHighlight;

<* LL.sup >= List.First(polygon.vertices).graph.mu *>

PROCEDURE SetPolygonColor (polygon: Polygon; color: PaintOp.T) =
  BEGIN
    WITH graph = polygon.graph DO
      polygon.color := color;
      polygon.colorScheme :=
        PaintOpCache.MakeColorScheme(polygon.color, polygon.color);
      polygon.mg.setColor(graph, polygon.colorScheme);
      (*
      polygon.mg.setWeight(graph, 0.0); (* work around MG bug *)
      *)
    END;
  END SetPolygonColor;

<* LL.sup >= List.First(polygon.vertices).graph.mu *>

PROCEDURE MovePolygon (polygon : Polygon;
                       vertices: List.T (* OF Vertex *);
                       animated: BOOLEAN                  := FALSE) =
  BEGIN
    WITH graph = polygon.graph DO
      <*ASSERT vertices # NIL*>
      VAR list := vertices;
      BEGIN
        WHILE list # NIL DO
          VAR vertex: Vertex := List.Pop(list);
          BEGIN
            <*ASSERT vertex.graph = graph*>
          END;
        END;
      END;

      polygon.new.vertices := vertices;

      IF NOT animated THEN
        VAR vertices := polygon.vertices;
        BEGIN
          VAR vertex: Vertex := List.Pop(vertices);
          BEGIN
            vertex.polygons := List.DeleteQ(vertex.polygons, polygon);
          END;
        END;

        polygon.vertices := polygon.new.vertices;

        VAR vertices := polygon.vertices;
        BEGIN
          VAR vertex: Vertex := List.Pop(vertices);
          BEGIN
            List.Push(vertex.polygons, polygon);
          END;
        END;

        RefreshPolygon(polygon);
        IF graph.animations > 0 THEN graph.needRefresh := TRUE; END;
      END;
    END;
  END MovePolygon;

<* LL.sup >= polygon.graph.mu *>

PROCEDURE PolygonToFront (polygon: Polygon) =
  BEGIN
    WITH graph = polygon.graph DO
      graph.polygonGroup.remove(graph, polygon.group);
      graph.polygonGroup.addBefore(graph, polygon.group, NIL);
      graph.polygons := List.DeleteQ(graph.polygons, polygon);
      List.Push(graph.polygons, polygon);
    END;
  END PolygonToFront;

<* LL.sup >= polygon.graph.mu *>

PROCEDURE PolygonToBack (polygon: Polygon) =
  BEGIN
    WITH graph = polygon.graph DO
      graph.polygonGroup.remove(graph, polygon.group);
      graph.polygonGroup.addAfter(graph, polygon.group, NIL);
      graph.polygons := List.DeleteQ(graph.polygons, polygon);
      graph.polygons := List.Append(graph.polygons, List.List1(polygon));
    END;
  END PolygonToBack;

<* LL.sup >= List.First(polygon.vertices).graph.mu *>

PROCEDURE RemovePolygon (polygon: Polygon) =
  BEGIN
    IF NOT polygon.initialized THEN RETURN; END;
    WITH graph = polygon.graph DO
      graph.polygons := List.DeleteQ(graph.polygons, polygon);
      VAR vertices := polygon.vertices;
      BEGIN
        WHILE vertices # NIL DO
          VAR vertex: Vertex := List.Pop(vertices);
          BEGIN
            vertex.polygons := List.DeleteQ(vertex.polygons, polygon);
          END;
        END;
      END;
      graph.polygonGroup.remove(graph, polygon.group);
    END;
    polygon.initialized := FALSE;
  END RemovePolygon;

(* REDISPLAY *)

<* LL.sup >= graph.mu *>

PROCEDURE RefreshGraph (graph: T) =
  VAR pixels, borderPixels: ARRAY [0 .. 1] OF INTEGER;
  BEGIN
    pixels[0] := graph.rect.east - graph.rect.west;
    pixels[1] := graph.rect.south - graph.rect.north;

    borderPixels[0] := CEILING(graph.marginMM * graph.res[Axis.T.Hor]);
    borderPixels[1] := CEILING(graph.marginMM * graph.res[Axis.T.Ver]);

    pixels[0] := pixels[0] - 2 * borderPixels[0];
    IF pixels[0] < 0 THEN
      pixels[0] := 0;
      borderPixels[0] := (graph.rect.east - graph.rect.west) DIV 2;
    END;
    pixels[1] := pixels[1] - 2 * borderPixels[1];
    IF pixels[1] < 0 THEN
      pixels[1] := 0;
      borderPixels[1] := (graph.rect.south - graph.rect.north) DIV 2;
    END;

    IF pixels[0] >= graph.pixelSizeDivisor[0] THEN
      pixels[0] := pixels[0] - pixels[0] MOD graph.pixelSizeDivisor[0];
    END;
    IF pixels[1] >= graph.pixelSizeDivisor[1] THEN
      pixels[1] := pixels[1] - pixels[1] MOD graph.pixelSizeDivisor[1];
    END;

    IF graph.aspect # 0.0 THEN
      IF FLOAT(pixels[0]) * graph.res[Axis.T.Ver] * graph.aspect
           > FLOAT(pixels[1]) * graph.res[Axis.T.Hor] THEN
        pixels[0] :=
          ROUND(FLOAT(pixels[1])
                  * (graph.res[Axis.T.Hor] / graph.res[Axis.T.Ver])
                  / graph.aspect / FLOAT(graph.pixelSizeDivisor[0]))
            * graph.pixelSizeDivisor[0];
        IF pixels[0] + 2 * borderPixels[0]
             > graph.rect.east - graph.rect.west THEN
          pixels[0] := pixels[0] - graph.pixelSizeDivisor[0];
        END;
        IF pixels[0] = 0 THEN
          pixels[0] :=
            (graph.rect.east - graph.rect.west) - 2 * borderPixels[0];
        END;
      ELSE
        pixels[1] :=
          ROUND(FLOAT(pixels[0])
                  * (graph.res[Axis.T.Ver] / graph.res[Axis.T.Hor])
                  * graph.aspect / FLOAT(graph.pixelSizeDivisor[1]))
            * graph.pixelSizeDivisor[1];
        IF pixels[1] + 2 * borderPixels[1]
             > graph.rect.south - graph.rect.north THEN
          pixels[1] := pixels[1] - graph.pixelSizeDivisor[1];
        END;
        IF pixels[1] = 0 THEN
          pixels[1] :=
            (graph.rect.south - graph.rect.north) - 2 * borderPixels[1];
        END;
      END;
    END;

    borderPixels[0] :=
      ((graph.rect.east - graph.rect.west) - pixels[0]) DIV 2;
    borderPixels[1] :=
      ((graph.rect.south - graph.rect.north) - pixels[1]) DIV 2;

    graph.sizeMM[0] := FLOAT(pixels[0]) / graph.res[Axis.T.Hor];
    graph.sizeMM[1] := FLOAT(pixels[1]) / graph.res[Axis.T.Ver];

    graph.realMarginMM[0] :=
      FLOAT(borderPixels[0]) / graph.res[Axis.T.Hor];
    graph.realMarginMM[1] :=
      FLOAT(borderPixels[1]) / graph.res[Axis.T.Ver];

    VAR vertices := graph.vertices;
    BEGIN
      WHILE vertices # NIL DO
        VAR vertex: Vertex := List.Pop(vertices);
        BEGIN
          RefreshVertex(vertex);
        END;
      END;
    END;

    VAR edges := graph.edges;
    BEGIN
      WHILE edges # NIL DO
        VAR edge: Edge := List.Pop(edges);
        BEGIN
          RefreshEdge(edge);
        END;
      END;
    END;

    VAR polygons := graph.polygons;
    BEGIN
      WHILE polygons # NIL DO
        VAR polygon: Polygon := List.Pop(polygons);
        BEGIN
          RefreshPolygon(polygon);
        END;
      END;
    END;
  END RefreshGraph;

<* LL.sup >= vertex.graph.mu *>

PROCEDURE RefreshVertex (vertex: Vertex) =
  BEGIN
    WITH graph = vertex.graph DO
      VAR centerPP := WorldPosToPts(graph, vertex.pos);
      BEGIN
        IF vertex.sizeW[0] # 0.0 OR vertex.sizeW[1] # 0.0 THEN
          vertex.sizeMM := Abs(WorldSizeToMM(graph, vertex.sizeW));
          AdjustVertex(vertex);
        END;
        MG.TranslateToLocked(
          vertex.mg, graph, Finite2(centerPP), center := TRUE);
        VAR vertexHighlights := vertex.vertexHighlights;
        BEGIN
          WHILE vertexHighlights # NIL DO
            VAR
              vertexHighlight: VertexHighlight := List.Pop(
                                                    vertexHighlights);
            BEGIN
              MG.TranslateToLocked(vertexHighlight.mg, graph,
                                   Finite2(centerPP), center := TRUE);
              IF vertexHighlight.borderW[0] # 0.0
                   OR vertexHighlight.borderW[1] # 0.0 THEN
                vertexHighlight.borderMM :=
                  Abs(WorldSizeToMM(graph, vertexHighlight.borderW));
                EVAL AdjustVertexHighlightSizeandShape(vertexHighlight);
              END;
            END;
          END;
        END;
      END;
    END;
  END RefreshVertex;

<* LL.sup >= edge.vertex0.graph.mu *>

PROCEDURE RefreshEdge (edge: Edge) =
  BEGIN
    WITH graph = edge.graph DO
      edge.pos[0] := edge.vertex0.pos;
      edge.pos[1] := edge.vertex1.pos;
      IF edge.control0 # NIL THEN
        edge.cpos[0] := edge.control0.pos;
      ELSE
        edge.cpos[0] := edge.vertex0.pos;
      END;
      IF edge.control1 # NIL THEN
        edge.cpos[1] := edge.control1.pos;
      ELSE
        edge.cpos[1] := edge.vertex1.pos;
      END;
      IF edge.isLine THEN
        MG.TranslateToLocked(
          edge.end[0], graph, Finite2(WorldPosToPts(graph, edge.pos[0])));
        MG.TranslateToLocked(
          edge.end[1], graph, Finite2(WorldPosToPts(graph, edge.pos[1])));
      ELSE
        VAR
          origin := WorldPosToPts(graph, edge.pos[0]);
          path   := NEW(RealPath.T);
        BEGIN
          path.init();
          path.moveTo(R2.Origin);
          path.curveTo(
            Finite2(R2.Sub(WorldPosToPts(graph, edge.cpos[0]), origin)),
            Finite2(R2.Sub(WorldPosToPts(graph, edge.cpos[1]), origin)),
            Finite2(R2.Sub(WorldPosToPts(graph, edge.pos[1]), origin)));
          NARROW(edge.mg, MG.Shape).reshape(
            graph, origin, path, fill := FALSE);
        END;
      END;
      edge.arrowPos := ArrowPos(edge);
      FOR i := 0 TO 1 DO
        IF edge.arrow[i] THEN
          FOR j := 0 TO 1 DO
            FOR k := 0 TO 1 DO
              MG.TranslateToLocked(
                edge.arrowEnd[i][j][k], graph,
                Finite2(WorldPosToPts(graph, edge.arrowPos[i][j][k])));
            END;
          END;
        END;
      END;
    END;
  END RefreshEdge;

CONST
  (* arrowhead lines extend 22.5 degrees from the edge *)
  SinTheta = 0.38268343;
  CosTheta = 0.92387953;

<* LL.sup >= edge.vertex0.graph.mu *>

PROCEDURE ArrowPos (edge: Edge):
  ARRAY [0 .. 1], [0 .. 1], [0 .. 1] OF R2.T =
  VAR new: ARRAY [0 .. 1], [0 .. 1], [0 .. 1] OF R2.T;
  BEGIN
    WITH graph = edge.graph DO
      IF edge.arrow[0] THEN
        VAR tip, delta: R2.T;
        BEGIN
          ComputeArrowTip(edge, TRUE, tip, delta);
          new[0][0][0] := MMPosToWorld(graph, tip);
          new[0][1][0] := new[0][0][0];
          delta := R2.Scale(5.0 * edge.widthMM, delta);
          new[0][0][1] :=
            R2.Add(
              new[0][0][0],
              MMSizeToWorld(
                graph, R2.T{CosTheta * delta[0] + SinTheta * delta[1],
                            -SinTheta * delta[0] + CosTheta * delta[1]}));
          new[0][1][1] :=
            R2.Add(
              new[0][1][0],
              MMSizeToWorld(
                graph, R2.T{CosTheta * delta[0] - SinTheta * delta[1],
                            SinTheta * delta[0] + CosTheta * delta[1]}));
        END;
      ELSE
        FOR j := 0 TO 1 DO
          FOR k := 0 TO 1 DO new[0][j][k] := edge.pos[0]; END;
        END;
      END;
      IF edge.arrow[1] THEN
        VAR tip, delta: R2.T;
        BEGIN
          ComputeArrowTip(edge, FALSE, tip, delta);
          new[1][0][0] := MMPosToWorld(graph, tip);
          new[1][1][0] := new[1][0][0];
          delta := R2.Scale(5.0 * edge.widthMM, delta);
          new[1][0][1] :=
            R2.Add(
              new[1][0][0],
              MMSizeToWorld(
                graph, R2.T{CosTheta * delta[0] + SinTheta * delta[1],
                            -SinTheta * delta[0] + CosTheta * delta[1]}));
          new[1][1][1] :=
            R2.Add(
              new[1][1][0],
              MMSizeToWorld(
                graph, R2.T{CosTheta * delta[0] - SinTheta * delta[1],
                            SinTheta * delta[0] + CosTheta * delta[1]}));
        END;
      ELSE
        FOR j := 0 TO 1 DO
          FOR k := 0 TO 1 DO new[1][j][k] := edge.pos[1]; END;
        END;
      END;
    END;
    RETURN new;
  END ArrowPos;

<* LL.sup >= edge.vertex0.graph.mu *>

PROCEDURE ComputeArrowTip (            edge      : Edge;
                                       forward   : BOOLEAN;
                           VAR (*OUT*) tip, delta: R2.T     ) =
  BEGIN
    IF edge.isLine THEN
      ComputeArrowTipOfLine(edge, forward, tip, delta);
    ELSE
      ComputeArrowTipOfBezier(edge, forward, tip, delta);
    END;
  END ComputeArrowTip;

<* LL.sup >= edge.vertex0.graph.mu *>

PROCEDURE ComputeArrowTipOfLine (            edge      : Edge;
                                             forward   : BOOLEAN;
                                 VAR (*OUT*) tip, delta: R2.T     ) =
  VAR
    vertex0   : Vertex;
    pos0, pos1: R2.T;
  BEGIN
    WITH graph = edge.graph DO
      IF forward THEN
        vertex0 := edge.vertex0;
        pos0 := edge.pos[0];
        pos1 := edge.pos[1];
      ELSE
        vertex0 := edge.vertex1;
        pos0 := edge.pos[1];
        pos1 := edge.pos[0];
      END;
      pos0 := WorldPosToMM(graph, pos0);
      pos1 := WorldPosToMM(graph, pos1);
      VAR
        d      := R2.Sub(pos1, pos0);
        length := R2.Length(d);
      VAR t: REAL;
      BEGIN
        WITH sizeMM = vertex0.sizeMM DO
          CASE vertex0.shape OF
          | VertexShape.Rectangle =>
              t := MIN(ABS(sizeMM[0] / (2.0 * d[0])),
                       ABS(sizeMM[1] / (2.0 * d[1])));
          | VertexShape.Ellipse =>
              t :=
                FLOAT(Math.sqrt(
                        FLOAT(0.25 / (d[0] * d[0] / (sizeMM[0] * sizeMM[0])
                                        + d[1] * d[1]
                                            / (sizeMM[1] * sizeMM[1])),
                              LONGREAL)), REAL);
          END;
          tip :=
            R2.Add(pos0,
                   R2.Scale(MIN(MAX(t + ABS(edge.widthMM) / (2.0 * length),
                                    0.0), 1.0), d));
          delta := R2.Scale(1.0 / length, d);
        END;
      END;
    END;
    IF NOT RealFloat.Finite(tip[0] + tip[1]) THEN tip := pos0; END;
    IF NOT RealFloat.Finite(delta[0] + delta[1]) THEN
      delta := R2.T{1.0, 0.0};
    END;
  END ComputeArrowTipOfLine;

<* LL.sup >= edge.vertex0.graph.mu *>

PROCEDURE ComputeArrowTipOfBezier (            edge      : Edge;
                                               forward   : BOOLEAN;
                                   VAR (*OUT*) tip, delta: R2.T     ) =
  VAR
    vertex0                 : Vertex;
    pos0, pos1, cpos0, cpos1: R2.T;
  BEGIN
    WITH graph = edge.graph DO
      IF forward THEN
        vertex0 := edge.vertex0;
        pos0 := edge.pos[0];
        pos1 := edge.pos[1];
        cpos0 := edge.cpos[0];
        cpos1 := edge.cpos[1];
      ELSE
        vertex0 := edge.vertex1;
        pos0 := edge.pos[1];
        pos1 := edge.pos[0];
        cpos0 := edge.cpos[1];
        cpos1 := edge.cpos[0];
      END;
      pos0 := WorldPosToMM(graph, pos0);
      pos1 := WorldPosToMM(graph, pos1);
      cpos0 := WorldPosToMM(graph, cpos0);
      cpos1 := WorldPosToMM(graph, cpos1);
      VAR t: REAL;
      BEGIN
        WITH sizeMM = vertex0.sizeMM DO
          VAR
            x1       := cpos0[0] - pos0[0];
            x2       := cpos1[0] - pos0[0];
            x3       := pos1[0] - pos0[0];
            y1       := cpos0[1] - pos0[1];
            y2       := cpos1[1] - pos0[1];
            y3       := pos1[1] - pos0[1];
            widthMM  := (ABS(vertex0.sizeMM[0]) + ABS(edge.widthMM) / 2.0);
            heightMM := (ABS(vertex0.sizeMM[1]) + ABS(edge.widthMM) / 2.0);
          BEGIN
            PROCEDURE Inside (t: REAL): BOOLEAN =
              BEGIN
                VAR x := Cubic(0.0, x1, x2, x3, t);
                BEGIN
                  x := (x + x) / widthMM;
                  IF ABS(x) > 1.0 THEN RETURN FALSE; END;
                  VAR y := Cubic(0.0, y1, y2, y3, t := t);
                  BEGIN
                    y := (y + y) / heightMM;
                    IF ABS(y) > 1.0 THEN RETURN FALSE; END;
                    CASE vertex0.shape OF
                    | VertexShape.Rectangle => RETURN TRUE;
                    | VertexShape.Ellipse => RETURN x * x + y * y < 1.0;
                    END;
                  END;
                END;
              END Inside;
            BEGIN
              VAR i := 0;
              BEGIN
                LOOP
                  t := Random.Real();
                  IF NOT Inside(t) THEN EXIT; END;
                  i := i + 1;
                  IF i = 100 THEN t := 0.0; EXIT; END;
                END;
              END;
              VAR
                t0          := 0.0;
                t1          := t;
                pixelLength := PixelLength(graph, x1, x2, x3, y1, y2, y3);
              BEGIN
                REPEAT
                  t := (t0 + t1) / 2.0;
                  IF Inside(t) THEN t0 := t; ELSE t1 := t; END;
                  pixelLength := pixelLength DIV 2;
                UNTIL pixelLength = 0;
                t := t1;
              END;
            END;
            tip := R2.T{pos0[0] + Cubic(0.0, x1, x2, x3, t := t),
                        pos0[1] + Cubic(0.0, y1, y2, y3, t := t)};
            VAR
              d := R2.T{DCubic(0.0, x1, x2, x3, t := t),
                        DCubic(0.0, y1, y2, y3, t := t)};
              length := R2.Length(d);
            BEGIN
              delta := R2.Scale(1.0 / length, d);
              IF NOT RealFloat.Finite(delta[0] + delta[1]) THEN
                delta := R2.T{1.0, 0.0};
              END;
            END;
          END;
        END;
      END;
    END;
  END ComputeArrowTipOfBezier;

<* LL.sup >= graph.mu *>

PROCEDURE PixelLength (graph: T; x1, x2, x3, y1, y2, y3: REAL): INTEGER =
  BEGIN
    RETURN CEILING((ABS(x1) + ABS(x2 - x1) + ABS(x3 - x2)) * graph.res[
                     Axis.T.Hor] + (ABS(y1) + ABS(y2 - y1) + ABS(y3 - y2))
                                     * graph.res[Axis.T.Ver]);
  END PixelLength;

<* LL.sup >= List.First(polygon.vertices).graph.mu *>

PROCEDURE RefreshPolygon (polygon: Polygon) =
  BEGIN
    WITH graph = polygon.graph DO
      VAR
        vertices          := polygon.vertices;
        positions: List.T := NIL;
      BEGIN
        WHILE vertices # NIL DO
          VAR vertex: Vertex := List.Pop(vertices);
          BEGIN
            List.Push(positions, NewR2(vertex.pos));
          END;
        END;
        positions := List.ReverseD(positions);
        polygon.pos := positions;
      END;
      VAR
        origin: R2.T;
        path         := NEW(RealPath.T);
      BEGIN
        path.init();
        path.moveTo(R2.Origin);
        VAR positions := polygon.pos;
        BEGIN
          VAR pos := NARROW(List.Pop(positions), REF R2.T)^;
          BEGIN
            origin := WorldPosToPts(graph, pos);
          END;
          VAR previous := origin;
          BEGIN
            WHILE positions # NIL DO
              VAR pos := NARROW(List.Pop(positions), REF R2.T)^;
              BEGIN
                VAR posPts := WorldPosToPts(graph, pos);
                BEGIN
                  path.lineTo(Finite2(R2.Sub(posPts, origin)));
                  previous := posPts;
                END;
              END;
            END;
          END;
        END;
        path.close();
        NARROW(polygon.mg, MG.Shape).reshape(
          graph, origin, path, fill := TRUE);
      END;
    END;
  END RefreshPolygon;

(* CONSISTENCY UTLITIES *)

<* LL.sup >= vertex.graph.mu *>

PROCEDURE AdjustVertex (vertex: Vertex) =
  BEGIN
    WITH graph = vertex.graph DO
      VAR centerPP := WorldPosToPts(graph, vertex.pos);
      BEGIN
        VAR
          borderMM := MIN(MIN(vertex.borderMM, vertex.sizeMM[0] / 2.0),
                          vertex.sizeMM[1] / 2.0);
          insetSizeMM := R2.T{MAX(vertex.sizeMM[0] - 2.0 * borderMM, 0.0),
                              MAX(vertex.sizeMM[1] - 2.0 * borderMM, 0.0)};
        BEGIN
          CASE vertex.shape OF
          | VertexShape.Rectangle =>
              NARROW(vertex.mg, MG.Rectangle).reshape(
                graph, Finite2(SWFromCenterMM(centerPP, insetSizeMM)),
                Finite2(NEFromCenterMM(centerPP, insetSizeMM)));
              NARROW(vertex.mg, MG.Rectangle).setWeight(
                graph, Pts.FromMM(borderMM));
          | VertexShape.Ellipse =>
              NARROW(vertex.mg, MG.Ellipse).reshape(
                graph, Finite2(SWFromCenterMM(centerPP, insetSizeMM)),
                Finite2(NEFromCenterMM(centerPP, insetSizeMM)));
              NARROW(vertex.mg, MG.Ellipse).setWeight(
                graph, Pts.FromMM(borderMM));
          END;
        END;
        VAR edges := vertex.edges;
        BEGIN
          WHILE edges # NIL DO
            VAR edge: Edge := List.Pop(edges);
            BEGIN
              RefreshEdge(edge);
            END;
          END;
        END;
        VAR vertexHighlights := vertex.vertexHighlights;
        BEGIN
          WHILE vertexHighlights # NIL DO
            VAR
              vertexHighlight: VertexHighlight := List.Pop(
                                                    vertexHighlights);
            BEGIN
              EVAL AdjustVertexHighlightSizeandShape(vertexHighlight);
            END;
          END;
        END;
        VAR polygons := vertex.polygons;
        BEGIN
          WHILE polygons # NIL DO
            VAR polygon: Polygon := List.Pop(polygons);
            BEGIN
              RefreshPolygon(polygon);
            END;
          END;
        END;
      END;
    END;
  END AdjustVertex;

<* LL.sup >= vertexHighlight.vertex.graph.mu *>

PROCEDURE AdjustVertexHighlightSizeandShape (vertexHighlight: VertexHighlight):
  BOOLEAN =
  BEGIN
    WITH graph = vertexHighlight.graph DO
      VAR
        sizeMM := R2.T{MAX(ABS(vertexHighlight.vertex.sizeMM[0])
                             + 2.0 * vertexHighlight.borderMM[0], 0.0),
                       MAX(ABS(vertexHighlight.vertex.sizeMM[1])
                             + 2.0 * vertexHighlight.borderMM[1], 0.0)};
      BEGIN
        IF (vertexHighlight.sizeMM[0] = sizeMM[0]
              AND vertexHighlight.sizeMM[1] = sizeMM[1])
             AND vertexHighlight.shape = vertexHighlight.vertex.shape THEN
          RETURN FALSE;
        END;
        vertexHighlight.sizeMM := sizeMM;
        IF vertexHighlight.shape # vertexHighlight.vertex.shape THEN
          vertexHighlight.group.remove(graph, vertexHighlight.mg);
          vertexHighlight.shape := vertexHighlight.vertex.shape;
          CASE vertexHighlight.shape OF
          | VertexShape.Rectangle =>
              vertexHighlight.mg :=
                NEW(MG.Rectangle, color := vertexHighlight.colorScheme,
                    weight := 0.0).init(R2.Origin, R2.Origin);
          | VertexShape.Ellipse =>
              vertexHighlight.mg :=
                NEW(MG.Ellipse, color := vertexHighlight.colorScheme,
                    weight := 0.0).init(R2.Origin, R2.Origin);
          END;
          vertexHighlight.group.addBefore(graph, vertexHighlight.mg, NIL);
        END;
      END;
      VAR centerPP := WorldPosToPts(graph, vertexHighlight.vertex.pos);
      BEGIN
        CASE vertexHighlight.shape OF
        | VertexShape.Rectangle =>
            NARROW(vertexHighlight.mg, MG.Rectangle).reshape(
              graph,
              Finite2(SWFromCenterMM(centerPP, vertexHighlight.sizeMM)),
              Finite2(NEFromCenterMM(centerPP, vertexHighlight.sizeMM)));
        | VertexShape.Ellipse =>
            NARROW(vertexHighlight.mg, MG.Ellipse).reshape(
              graph,
              Finite2(SWFromCenterMM(centerPP, vertexHighlight.sizeMM)),
              Finite2(NEFromCenterMM(centerPP, vertexHighlight.sizeMM)));
        END;
      END;
    END;
    RETURN TRUE;
  END AdjustVertexHighlightSizeandShape;

(* UTILITIES ON COORDINATES *)

<* LL.sup >= graph.mu *>

PROCEDURE WorldPosToPts (graph: T; posW: R2.T): R2.T =
  BEGIN
    RETURN R2.T{Pts.FromMM((posW[0] - graph.world.w)
                             / (graph.world.e - graph.world.w)
                             * graph.sizeMM[0] + graph.realMarginMM[0]),
                Pts.FromMM((posW[1] - graph.world.s)
                             / (graph.world.n - graph.world.s)
                             * graph.sizeMM[1] + graph.realMarginMM[1])};
  END WorldPosToPts;

<* LL.sup >= graph.mu *>

PROCEDURE WorldPosToMM (graph: T; posW: R2.T): R2.T =
  BEGIN
    RETURN R2.T{(posW[0] - graph.world.w) / (graph.world.e - graph.world.w)
                  * graph.sizeMM[0] + graph.realMarginMM[0],
                (posW[1] - graph.world.s) / (graph.world.n - graph.world.s)
                  * graph.sizeMM[1] + graph.realMarginMM[1]};
  END WorldPosToMM;

<* LL.sup >= graph.mu *>

PROCEDURE MMPosToWorld (graph: T; posMM: R2.T): R2.T =
  BEGIN
    RETURN R2.T{(posMM[0] - graph.realMarginMM[0]) / graph.sizeMM[0]
                  * (graph.world.e - graph.world.w) + graph.world.w,
                (posMM[1] - graph.realMarginMM[1]) / graph.sizeMM[1]
                  * (graph.world.n - graph.world.s) + graph.world.s};
  END MMPosToWorld;

<* LL.sup >= graph.mu *>

PROCEDURE WorldSizeToPts (graph: T; sizeW: R2.T): R2.T =
  BEGIN
    RETURN R2.T{Pts.FromMM(sizeW[0] / (graph.world.e - graph.world.w)
                             * graph.sizeMM[0]),
                Pts.FromMM(sizeW[1] / (graph.world.n - graph.world.s)
                             * graph.sizeMM[1])};
  END WorldSizeToPts;

<* LL.sup >= graph.mu *>

PROCEDURE WorldSizeToMM (graph: T; sizeW: R2.T): R2.T =
  BEGIN
    RETURN
      R2.T{sizeW[0] / (graph.world.e - graph.world.w) * graph.sizeMM[0],
           sizeW[1] / (graph.world.n - graph.world.s) * graph.sizeMM[1]};
  END WorldSizeToMM;

<* LL.sup >= graph.mu *>

PROCEDURE MMSizeToWorld (graph: T; sizeMM: R2.T): R2.T =
  BEGIN
    RETURN
      R2.T{sizeMM[0] / graph.sizeMM[0] * (graph.world.e - graph.world.w),
           sizeMM[1] / graph.sizeMM[1] * (graph.world.n - graph.world.s)};
  END MMSizeToWorld;

<* LL arbitrary *>

PROCEDURE SWFromCenterMM (centerPP: R2.T; sizeMM: R2.T): R2.T =
  BEGIN
    RETURN R2.T{centerPP[0] - Pts.FromMM(sizeMM[0] / 2.0),
                centerPP[1] - Pts.FromMM(sizeMM[1] / 2.0)};
  END SWFromCenterMM;

<* LL arbitrary *>

PROCEDURE NEFromCenterMM (centerPP: R2.T; sizeMM: R2.T): R2.T =
  BEGIN
    RETURN R2.T{centerPP[0] + Pts.FromMM(sizeMM[0] / 2.0),
                centerPP[1] + Pts.FromMM(sizeMM[1] / 2.0)};
  END NEFromCenterMM;

(* MISCELLANEOUS UTILITIES *)

<* LL arbitrary *>

PROCEDURE Abs (d: R2.T): R2.T =
  BEGIN
    RETURN R2.T{ABS(d[0]), ABS(d[1])};
  END Abs;

<* LL arbitrary *>

PROCEDURE Finite2 (z: R2.T): R2.T =
  BEGIN
    IF NOT RealFloat.Finite(z[0]) THEN z[0] := 0.0; END;
    IF NOT RealFloat.Finite(z[1]) THEN z[1] := 0.0; END;
    RETURN z;
  END Finite2;

(* AddAnimationLocked is the same as MGV.AddAnimation, but locked *)

<* LL.sup >= graph.mu *>

PROCEDURE AddAnimationLocked (v: MG.V; anim: Animate.T; mg: MG.T) =
  BEGIN
    IF v.animations = NIL THEN
      v.animations := NEW(Animate.Group).init();
    END;
    v.animations.add(v, NEW(Animate.Composite, t := anim, mg := mg));
  END AddAnimationLocked;

(* TIME FUNCTION *)

TYPE
  AffineTimeFunction = Animate.TimeFunction OBJECT
                         a, b: REAL;
                       OVERRIDES
                         map := AffineMap;
                       END;

<* LL arbitrary *>

PROCEDURE AffineMap (self: AffineTimeFunction; t: REAL): REAL =
  BEGIN
    RETURN MIN(MAX(self.a * t + self.b, 0.0), 1.0);
  END AffineMap;

(* ANIMATION PATHS *)

TYPE
  StraightPath = AnimationPath OBJECT
                   p0, p1: R2.T;
                 OVERRIDES
                   pos := StraightPathPos;
                 END;

<* LL arbitrary *>

PROCEDURE StraightPathPos (self: StraightPath; t: REAL): R2.T =
  BEGIN
    RETURN R2.Add(self.p0, R2.Scale(t, R2.Sub(self.p1, self.p0)));
  END StraightPathPos;

TYPE
  OffsetPath = AnimationPath OBJECT
                 path            : AnimationPath;
                 offset0, offset1: R2.T;
               OVERRIDES
                 pos := OffsetPathPos;
               END;

<* LL arbitrary *>

PROCEDURE OffsetPathPos (self: OffsetPath; t: REAL): R2.T =
  BEGIN
    RETURN R2.Add(self.path.pos(t),
                  R2.Add(self.offset0,
                         R2.Scale(t, R2.Sub(self.offset1, self.offset0))));
  END OffsetPathPos;

(* ANIMATIONS *)

TYPE
  AlongGivenPath =
    Animate.T BRANDED OBJECT
      (* READONLY after initialization: MUST be initialized by client *)
      (* CONST *)
      graph: T;
      pos: R2.T;                (* the current center, in world
                                   coordinates *)
      path: AnimationPath;
    OVERRIDES
      length := LengthAlongGivenPath;
      doStep := DoStepAlongGivenPath;
    END;

<* LL.sup >= alongGivenPath.graph.mu *>

PROCEDURE LengthAlongGivenPath (<* UNUSED *> alongGivenPath: AlongGivenPath;
                                <* UNUSED *> v : MG.V;
                                <* UNUSED *> mg: MG.T  ): INTEGER =
  BEGIN
    RETURN 100;
  END LengthAlongGivenPath;

<* LL >= alongGivenPath.graph.mu *>

PROCEDURE DoStepAlongGivenPath (alongGivenPath: AlongGivenPath;
                                time          : REAL;
                                <* UNUSED *> timePrev: REAL;
                                             v       : MG.V;
                                             mg      : MG.T  ) =
  VAR newPos := alongGivenPath.path.pos(time);
  BEGIN
    MG.RTranslateLocked(
      mg, v, Finite2(WorldSizeToPts(alongGivenPath.graph,
                                    R2.Sub(newPos, alongGivenPath.pos))));
    alongGivenPath.pos := newPos;
  END DoStepAlongGivenPath;

TYPE
  BezierAnimation =
    Animate.T BRANDED OBJECT
      (* READONLY after initialization: MUST be initialized by client *)
      (* CONST *)
      graph                     : T;
      pathA, pathB, pathC, pathD: AnimationPath;
    OVERRIDES
      length := LengthBezierAnimation;
      doStep := DoStepBezierAnimation;
    END;

<* LL.sup >= bezierAnimation.graph.mu *>

PROCEDURE LengthBezierAnimation (<* UNUSED *> bezierAnimation: BezierAnimation;
                                 <* UNUSED *> v : MG.V;
                                 <* UNUSED *> mg: MG.T  ): INTEGER =
  BEGIN
    RETURN 100;
  END LengthBezierAnimation;

<* LL < bezierAnimation.graph.mu *>

PROCEDURE DoStepBezierAnimation (bezierAnimation: BezierAnimation;
                                 time           : REAL;
                                 <* UNUSED *> timePrev: REAL;
                                              v       : MG.V;
                                              mg      : MG.T  ) =
  VAR
    a := WorldPosToPts(
           bezierAnimation.graph, bezierAnimation.pathA.pos(time));
    b := WorldPosToPts(
           bezierAnimation.graph, bezierAnimation.pathB.pos(time));
    c := WorldPosToPts(
           bezierAnimation.graph, bezierAnimation.pathC.pos(time));
    d := WorldPosToPts(
           bezierAnimation.graph, bezierAnimation.pathD.pos(time));
    path := NEW(RealPath.T);
  BEGIN
    path.init();
    path.moveTo(R2.Origin);
    path.curveTo(
      Finite2(R2.Sub(b, a)), Finite2(R2.Sub(c, a)), Finite2(R2.Sub(d, a)));
    NARROW(mg, MG.Shape).reshape(v, a, path, fill := FALSE);
  END DoStepBezierAnimation;

TYPE
  LinearResize =
    Animate.T BRANDED OBJECT
      (* READONLY after initialization: MUST be initialized by client *)
      (* CONST *)
      graph: T;
      shape: VertexShape;
      corner0, corner1: ARRAY [0 .. 1] (* time *)
                          OF
                          R2.T;  (* coordinates, in points *)
    OVERRIDES
      length := LengthLinearResize;
      doStep := DoStepLinearResize;
    END;

<* LL.sup = linearResize.graph.mu *>

PROCEDURE LengthLinearResize (             linearResize: LinearResize;
                                           v           : MG.V;
                              <* UNUSED *> mg          : MG.T          ):
  INTEGER =
  BEGIN
    RETURN
      ROUND(
        MAX(Pts.ToPixels(
              v, MAX(ABS(linearResize.corner0[1][0]
                           - linearResize.corner0[0][0]),
                     ABS(linearResize.corner1[1][0]
                           - linearResize.corner1[0][0])), Axis.T.Hor),
            Pts.ToPixels(
              v, MAX(ABS(linearResize.corner0[1][1]
                           - linearResize.corner0[0][1]),
                     ABS(linearResize.corner1[1][1]
                           - linearResize.corner1[0][1])), Axis.T.Ver)));
  END LengthLinearResize;

<* LL < linearResize.graph.mu *>

PROCEDURE DoStepLinearResize (             linearResize: LinearResize;
                                           time        : REAL;
                              <* UNUSED *> timePrev    : REAL;
                                           v           : MG.V;
                                           mg          : MG.T          ) =
  BEGIN
    CASE linearResize.shape OF
    | VertexShape.Rectangle =>
        NARROW(mg, MG.Rectangle).reshape(
          v,
          Finite2(R2.Add(linearResize.corner0[0],
                         R2.Scale(time, R2.Sub(linearResize.corner0[1],
                                               linearResize.corner0[0])))),
          Finite2(R2.Add(linearResize.corner1[0],
                         R2.Scale(time, R2.Sub(linearResize.corner1[1],
                                               linearResize.corner1[0])))));
    | VertexShape.Ellipse =>
        NARROW(mg, MG.Ellipse).reshape(
          v,
          Finite2(R2.Add(linearResize.corner0[0],
                         R2.Scale(time, R2.Sub(linearResize.corner0[1],
                                               linearResize.corner0[0])))),
          Finite2(R2.Add(linearResize.corner1[0],
                         R2.Scale(time, R2.Sub(linearResize.corner1[1],
                                               linearResize.corner1[0])))));
    END;
  END DoStepLinearResize;

TYPE
  PolygonAnimation =
    Animate.T BRANDED OBJECT
      (* READONLY after initialization: MUST be initialized by client *)
      (* CONST *)
      graph: T;
      paths: List.T (* OF AnimationPath *);
    OVERRIDES
      length := LengthPolygonAnimation;
      doStep := DoStepPolygonAnimation;
    END;

<* LL.sup >= PolygonAnimation.graph.mu *>

PROCEDURE LengthPolygonAnimation (<* UNUSED *> polygonAnimation: PolygonAnimation;
                                  <* UNUSED *> v : MG.V;
                                  <* UNUSED *> mg: MG.T  ): INTEGER =
  BEGIN
    RETURN 100;
  END LengthPolygonAnimation;

<* LL < polygonAnimation.graph.mu *>

PROCEDURE DoStepPolygonAnimation (polygonAnimation: PolygonAnimation;
                                  time            : REAL;
                                  <* UNUSED *> timePrev: REAL;
                                               v       : MG.V;
                                               mg      : MG.T  ) =
  VAR
    p                                     := NEW(RealPath.T);
    paths : List.T (* OF AnimationPath *) := polygonAnimation.paths;
    origin: R2.T;
  BEGIN
    p.init();
    p.moveTo(R2.Origin);
    VAR path: AnimationPath := List.Pop(paths);
    BEGIN
      origin :=
        Finite2(WorldPosToPts(polygonAnimation.graph, path.pos(time)));
    END;
    VAR previous := origin;
    BEGIN
      WHILE paths # NIL DO
        VAR path: AnimationPath := List.Pop(paths);
        BEGIN
          VAR
            posPts := WorldPosToPts(polygonAnimation.graph, path.pos(time));
          BEGIN
            p.lineTo(Finite2(R2.Sub(posPts, origin)));
            previous := posPts;
          END;
        END;
      END;
    END;
    p.close();
    NARROW(mg, MG.Shape).reshape(v, origin, p, fill := TRUE);
  END DoStepPolygonAnimation;

(* BEZIER UTILITIES *)

<* LL arbitrary *>

PROCEDURE Cubic (c0, c1, c2, c3: REAL; t: REAL): REAL =
  BEGIN
    RETURN c0 + t * (3.0 * (c1 - c0)
                       + t * (3.0 * ((c2 - c1) - (c1 - c0))
                                + t * ((c3 - c0) - 3.0 * (c2 - c1))));
  END Cubic;

<* LL arbitrary *>

PROCEDURE DCubic (c0, c1, c2, c3: REAL; t: REAL): REAL =
  BEGIN
    RETURN 3.0 * ((c1 - c0) + t * (2.0 * ((c2 - c1) - (c1 - c0))
                                     + t * ((c3 - c0) - 3.0 * (c2 - c1))));
  END DCubic;

<* LL arbitrary *>

PROCEDURE SubCubic (VAR (*INOUT*) c0, c1, c2, c3: REAL; half: [0 .. 1]) =
  BEGIN
    CASE half OF
    | 0 =>
        c3 := (c0 + 3.0 * (c1 + c2) + c3) / 8.0;
        c2 := (c0 + 2.0 * c1 + c2) / 4.0;
        c1 := (c0 + c1) / 2.0;
    | 1 =>
        c0 := (c0 + 3.0 * (c1 + c2) + c3) / 8.0;
        c1 := (c1 + 2.0 * c2 + c3) / 4.0;
        c2 := (c2 + c3) / 2.0;
    END;
  END SubCubic;

<* LL arbitrary *>

PROCEDURE SubCubic2 (VAR (*INOUT*) c0, c1, c2, c3: R2.T; half: [0 .. 1]) =
  BEGIN
    SubCubic(c0[0], c1[0], c2[0], c3[0], half);
    SubCubic(c0[1], c1[1], c2[1], c3[1], half);
  END SubCubic2;

<* LL arbitrary *>

PROCEDURE CubicBounds (c0, c1, c2, c3: REAL): R2.T =
  BEGIN
    RETURN
      R2.T{MIN(MIN(c0, c1), MIN(c2, c3)), MAX(MAX(c0, c1), MAX(c2, c3))};
  END CubicBounds;

(* RANDOM UTILITIES *)

<* LL arbitrary *>

PROCEDURE R2Intersect (x, y: R2.T): BOOLEAN =
  BEGIN
    RETURN MAX(x[0], y[0]) <= MIN(x[1], y[1]);
  END R2Intersect;

PROCEDURE NewR2 (x: R2.T): REF R2.T =
  VAR a := NEW(REF R2.T);
  BEGIN
    a^ := x;
    RETURN a;
  END NewR2;

BEGIN
  <* ASSERT FloatMode.IEEE *>
END GraphVBT.
