(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* by Steve Glassman, Mark Manasse and Greg Nelson           *)
(* Last modified on Tue Oct 20 18:15:58 PDT 1992 by msm      *)
(*      modified on Tue Mar 10 19:11:14 1992 by steveg   *)
(*      modified on Mon Feb 24 13:56:12 PST 1992 by muller   *)
(*      modified on Sat Nov  2 15:59:41 PST 1991 by gnelson  *)
<*PRAGMA LL*>

UNSAFE MODULE Trestle EXPORTS Trestle, TrestleImpl;

IMPORT Thread, Env, TrestleClass, VBT, TrestleComm, ParseParams, 
  VBTClass, ScrnColorMap, Point, Rect, Region, ScrnPixmap, 
  XClient, Split, StableVBT, VBTRep,
  ProperSplit, InstallQueue, Time, InstalledVBT;

FROM TrestleClass IMPORT InstallRef, Decoration;

REVEAL 
  TrestleClass.RootVBT = ProperSplit.T BRANDED OBJECT END;
  
EXCEPTION FatalError; <*FATAL FatalError*> (*EXCEPTION Unimplemented;*)

REVEAL 
  T = TrestleClass.Public BRANDED OBJECT END;

VAR
  <* LL >= {TrestleClass.connectMu} *>
  default: T := NIL;
  initMu := NEW(MUTEX);
  inited := FALSE;

PROCEDURE Init() =
  BEGIN
    IF NOT inited THEN
      LOCK initMu DO
      	IF NOT inited THEN
          XClient.Init();
	  (* Add other classes of window system here *)
          inited := TRUE
        END
      END
    END
  END Init;
  
PROCEDURE SetDefault (t: T) =
  BEGIN
    Init();
    LOCK TrestleClass.connectMu DO default := t END
  END SetDefault;

PROCEDURE Default(): T RAISES {TrestleComm.Failure} =
  BEGIN
    Init();
    LOCK TrestleClass.connectMu DO
      LOCK TrestleClass.closeMu DO
        IF default # NIL AND NOT default.closed THEN RETURN default END
      END;
      default := TrestleClass.Connect(NIL);
      RETURN default
    END
  END Default;

PROCEDURE DeleteHook(ch: VBT.T) = 
  VAR ir: InstallRef := VBT.GetProp(ch, TYPECODE(InstallRef)); BEGIN
    ir.installed := FALSE;
    Thread.Broadcast(ir.c);
    VBT.RemProp(ch, TYPECODE(InstallRef))
  END DeleteHook;

TYPE
  InstallObject = Closure OBJECT
    trsl: T;
    v: VBT.T;
    dec: Decoration
  OVERRIDES
    apply := DoInstall
  END;

PROCEDURE Install(
    v: VBT.T;
    applName: TEXT := NIL;
    inst: TEXT := NIL;
    windowTitle: TEXT := NIL;
    iconTitle: TEXT := NIL;
    bgColorR: REAL := 1.0;
    bgColorG: REAL := 1.0;
    bgColorB: REAL := 1.0;
    iconWindow: VBT.T := NIL;
    trsl: T := NIL)
  RAISES {TrestleComm.Failure} =
  BEGIN
    PreAttach(v, trsl);
    Fork(NEW(InstallObject, trsl := trsl, v := v,
             dec := NEW(Decoration, 
               applName := applName, inst := inst,
               windowTitle := windowTitle, iconTitle := iconTitle,
               bgColorR := bgColorR, bgColorG := bgColorG, 
               bgColorB := bgColorB, iconWindow := iconWindow)))
  END Install;

PROCEDURE Decorate(
    v: VBT.T;
    instance: TEXT := NIL;
    windowTitle: TEXT := NIL;
    iconTitle: TEXT := NIL;
    bgColorR: REAL := -1.0;
    bgColorG: REAL := -1.0;
    bgColorB: REAL := -1.0;
    applName: TEXT := NIL;
    iconWindow: VBT.T := NIL)
  RAISES {TrestleComm.Failure} = BEGIN
    InnerDecorate(v, 
      NEW(Decoration, applName := applName, inst := instance,
               windowTitle := windowTitle, iconTitle := iconTitle,
               bgColorR := bgColorR, bgColorG := bgColorG, 
               bgColorB := bgColorB, iconWindow := iconWindow))
  END Decorate;

PROCEDURE InnerDecorate(v: VBT.T; new: Decoration) 
  RAISES {TrestleComm.Failure} =
  VAR trsl: T; ch: VBT.T; old: Decoration; BEGIN
    IF NOT RootChild(v, trsl, ch) THEN RETURN END;
    old := VBT.GetProp(ch, TYPECODE(Decoration));
    IF old = NIL THEN
      IF new.applName = NIL THEN 
        new.applName := ParseParams.GetParameter(0)
      END;
      IF new.windowTitle = NIL THEN 
        IF new.inst = NIL THEN
          new.windowTitle := new.applName
        ELSE 
          new.windowTitle := new.applName & " " & new.inst
        END
      END;
      IF new.iconTitle = NIL THEN 
        IF new.inst = NIL THEN
          new.iconTitle := new.applName
        ELSE 
          new.iconTitle := new.inst
        END
      END;
      IF new.inst = NIL AND NOT Env.Get("WINSTANCE", new.inst) THEN
        new.inst := "" 
      END;
      IF new.bgColorR < 0.0 THEN new.bgColorR := 1.0 END;
      IF new.bgColorG < 0.0 THEN new.bgColorG := 1.0 END;
      IF new.bgColorB < 0.0 THEN new.bgColorB := 1.0 END
    ELSE
      IF new.applName = NIL THEN new.applName := old.applName END;
      IF new.windowTitle = NIL THEN new.windowTitle := old.windowTitle END;
      IF new.iconTitle = NIL THEN new.iconTitle := old.iconTitle END;
      IF new.iconWindow = NIL THEN new.iconWindow := old.iconWindow END;
      IF new.inst = NIL THEN new.inst := old.inst END;
      IF new.bgColorR < 0.0 THEN new.bgColorR := old.bgColorR END;
      IF new.bgColorG < 0.0 THEN new.bgColorG := old.bgColorG END;
      IF new.bgColorB < 0.0 THEN new.bgColorB := old.bgColorB END
    END;
    VBT.PutProp(ch, new);
    trsl.decorate(ch, old, new) 
  END InnerDecorate;

PROCEDURE GetDecoration(v: VBT.T;
  VAR instance, windowTitle, iconTitle, applName: TEXT;
  VAR bgColorR, bgColorG, bgColorB: REAL;
  VAR iconWindow: VBT.T): BOOLEAN =
  VAR trsl: T; ch: VBT.T; old: Decoration; BEGIN
    IF NOT RootChild(v, trsl, ch) THEN RETURN FALSE END;
    old := VBT.GetProp(ch, TYPECODE(Decoration));
    IF old = NIL THEN RETURN FALSE END;
    instance := old.inst;
    windowTitle := old.windowTitle;
    iconTitle := old.iconTitle;
    applName := old.applName;
    bgColorR := old.bgColorR;
    bgColorG := old.bgColorG;
    bgColorB := old.bgColorB;
    iconWindow := old.iconWindow;
    RETURN TRUE
  END GetDecoration;

PROCEDURE RootChild(v: VBT.T; VAR trsl: T; VAR ch: VBT.T): BOOLEAN =
  VAR ir: InstallRef := VBT.GetProp(v, TYPECODE(InstallRef)); BEGIN
    IF ir = NIL THEN RETURN FALSE END;
    ch := v; 
    trsl := ir.trsl;
    WHILE ch # NIL AND ch.parent # trsl DO ch := ch.parent END;
    RETURN ch # NIL
  END RootChild;

PROCEDURE DoInstall(self: InstallObject) =
  BEGIN
    TRY
      InnerAttach(self.v, self.trsl)
    EXCEPT
      TrestleComm.Failure => Delete(self.v); RETURN
    END;
    TRY
      InnerDecorate(self.v, self.dec);
      MoveNear(self.v, NIL)
    EXCEPT
      TrestleComm.Failure => (*skip*)
    END;
  END DoInstall;

PROCEDURE PreAttach(v: VBT.T; VAR trsl: T) 
RAISES {TrestleComm.Failure} <* LL.sup <= VBT.mu *> =
  BEGIN
    IF trsl = NIL THEN trsl := Default() END;
    LOCK v DO
     VAR ir: InstallRef :=  VBTClass.GetProp(v, TYPECODE(InstallRef)); BEGIN
      IF ir = NIL THEN
        ir := NEW(InstallRef, installed := TRUE,
          c := NEW(Thread.Condition), trsl := trsl);
        VBTClass.PutProp(v, ir)
      ELSE
	ir.trsl := trsl;
	ir.installed := TRUE
      END
     END
    END
  END PreAttach;

PROCEDURE Attach(v: VBT.T; trsl: T) RAISES {TrestleComm.Failure} =
  BEGIN
    PreAttach(v, trsl);
    InnerAttach(v, trsl)
  END Attach;

PROCEDURE InnerAttach(v: VBT.T; trsl: T) 
  RAISES {TrestleComm.Failure} <* LL = {VBT.mu} *> =
  BEGIN
    trsl.attach(StableVBT.New(InstalledVBT.New(v, DeleteHook)))
  END InnerAttach;

PROCEDURE ScreenOf(  
  v: VBT.T;
  READONLY pt: Point.T): ScreenOfRec RAISES {} =
  BEGIN
    LOCK v DO
      WITH p = v.parent DO
        IF p = NIL THEN 
          RETURN ScreenOfRec{id := NoScreen, q := pt, 
                   trsl := NIL, dom := Rect.Empty}
        ELSE
          RETURN p.screenOf(v, pt)
        END
      END
    END
  END ScreenOf;

PROCEDURE Overlap(
    v: VBT.T;
    id: ScreenID;
    READONLY nw: Point.T)
  RAISES {TrestleComm.Failure} =
  VAR trsl: T; ch: VBT.T; BEGIN
    IF NOT RootChild(v, trsl, ch) THEN RETURN END;
    trsl.overlap(ch, id, nw)
  END Overlap;

PROCEDURE Iconize(v: VBT.T) RAISES {TrestleComm.Failure} =
  VAR trsl: T; ch: VBT.T; BEGIN
    IF NOT RootChild(v, trsl, ch) THEN RETURN END;
    trsl.iconize(ch)
  END Iconize;

PROCEDURE MoveNear(v, w: VBT.T) RAISES {TrestleComm.Failure} =
  VAR trsl, wtrsl: T; ch, wch: VBT.T; BEGIN
    IF NOT RootChild(v, trsl, ch) THEN RETURN END;
    IF w = NIL OR NOT RootChild(w, wtrsl, wch) OR wtrsl # trsl THEN 
      wch := NIL 
    END;
    trsl.moveNear(ch, wch)
  END MoveNear;

PROCEDURE InstallOffscreen(
    v: VBT.T;
    width, height: CARDINAL;
    st: VBT.ScreenType) RAISES {TrestleComm.Failure} =
  VAR trsl: T; ch: VBT.T; BEGIN
    IF NOT RootChild(v, trsl, ch) OR ch.st # NIL THEN RAISE FatalError END;
    WHILE st # NIL AND ISTYPE(st, VBTRep.OffscreenType) DO
      st := NARROW(st, VBTRep.OffscreenType).st
    END;
    trsl.installOffscreen(ch, width, height, st)
  END InstallOffscreen;
 
VAR 
  mu := NEW(MUTEX);
  c := NEW(Thread.Condition);
  workQ := InstallQueue.Empty;
  (* workQ is protected by mu, and contains the closure that need to
     be activated.  c is signalled when workQ becomes non-empty.  *)
  worker, pinger: Thread.T := NIL;
  (* worker is the Thread handle of the thread doing Work, or NIL.
     pinger is the Thread handle of the thread timing out Work, or NIL.  *)

TYPE Closure = InstallQueue.Closure;

PROCEDURE Fork(cl: Closure) =
  VAR mustSignal: BOOLEAN;
  BEGIN
    LOCK mu DO
      mustSignal := InstallQueue.IsEmpty(workQ);
      InstallQueue.Insert(workQ, cl);
      IF worker = NIL THEN 
        worker := Thread.Fork(NEW(Thread.SizedClosure, apply := Work, 
          stackSize := 20000)) 
      END;
      IF pinger = NIL THEN 
        pinger := Thread.Fork(NEW(Thread.SizedClosure, apply := Ping, 
          stackSize := 20000)) 
      END
    END;
    IF mustSignal THEN Thread.Signal(c) END
  END Fork;

PROCEDURE Work(<*UNUSED*>self: Thread.Closure): REFANY RAISES {} =
  <*FATAL InstallQueue.Exhausted*>
  VAR cl: Closure; live: INTEGER;
  BEGIN
    LOOP
      live := 2;
      LOCK mu DO
        WHILE InstallQueue.IsEmpty(workQ) AND live # 0 DO 
          Thread.Wait(mu, c); DEC(live)
        END;
        IF InstallQueue.IsEmpty(workQ) THEN worker := NIL; RETURN NIL END;
        cl := InstallQueue.Remove(workQ); 
      END;
      LOCK VBT.mu DO cl.apply() END
    END
  END Work;

PROCEDURE Ping(<*UNUSED*>self: Thread.Closure): REFANY RAISES {} =
  BEGIN
    LOOP
      LOCK mu DO
        IF worker = NIL THEN pinger := NIL; RETURN NIL END
      END;
      Time.LongPause(5);
      Thread.Signal(c)
    END
  END Ping;

PROCEDURE Connect(inst: TEXT := NIL): T RAISES {TrestleComm.Failure} =
  <* LL.sup <= TrestleClass.connectMu *>
  BEGIN
    Init();
    IF inst = NIL THEN 
      RETURN Default()
    ELSE
      RETURN TrestleClass.Connect(inst)
    END
  END Connect;

PROCEDURE AwaitDelete(v: VBT.T) = 
  VAR ir: InstallRef;
  BEGIN
    LOCK VBT.mu DO
      ir := VBT.GetProp(v, TYPECODE(InstallRef));
      IF ir = NIL THEN RETURN END;
      WHILE ir.installed DO Thread.Wait(VBT.mu, ir.c) END
    END
  END AwaitDelete;
  
PROCEDURE Delete(v: VBT.T) RAISES {} = 
  <*FATAL Split.NotAChild*>
  VAR trsl: T; ch: VBT.T; BEGIN
    IF RootChild(v, trsl, ch) THEN Split.Delete(trsl, ch) END
  END Delete;

<* UNUSED *>
PROCEDURE SetColorMap(v: VBT.T; cm: ScrnColorMap.T) RAISES {} = 
  VAR trsl: T; ch: VBT.T; BEGIN
    IF RootChild(v, trsl, ch) THEN trsl.setColorMap(ch, cm) END
  END SetColorMap;

PROCEDURE Capture(
    id: ScreenID;
    READONLY clip: Rect.T;
    VAR  br: Region.T;
    trsl: T := NIL)
    : ScrnPixmap.T
  RAISES {TrestleComm.Failure} = BEGIN 
    IF trsl = NIL THEN trsl := Default() END;
    RETURN trsl.captureScreen(id, clip, br)
  END Capture;

PROCEDURE GetScreens(trsl: T := NIL): ScreenArray RAISES {TrestleComm.Failure} =
  BEGIN 
    IF trsl = NIL THEN trsl := Default() END;
    RETURN trsl.getScreens() 
  END GetScreens;

PROCEDURE AllCeded(trsl: T := NIL): BOOLEAN RAISES {TrestleComm.Failure} = 
  BEGIN 
    IF trsl = NIL THEN trsl := Default() END;
    RETURN trsl.allCeded() 
  END AllCeded;

PROCEDURE TickTime(trsl: T := NIL): INTEGER = 
  BEGIN 
    TRY
      IF trsl = NIL THEN trsl := Default() END;
      RETURN trsl.tickTime() 
    EXCEPT
      TrestleComm.Failure => RETURN 1
    END
  END TickTime;

(* PROCEDURE SetScreens(
    sa: ScreenArray;
    trsl: T := NIL)
    : BOOLEAN
  RAISES {TrestleComm.Failure, Unimplemented} = BEGIN RAISE Unimplemented END SetScreens; *)

(* PROCEDURE Swap(v, w: VBT.T) RAISES {TrestleComm.Failure, Unimplemented} = BEGIN RAISE Unimplemented END Swap; *)

(* PROCEDURE GetName(v: VBT.T): TEXT RAISES {TrestleComm.Failure} = 
  BEGIN RAISE Unimplemented END GetName; *)

(* PROCEDURE NameList(
    nm: TEXT;
    trsl: T := NIL)
    : REF ARRAY OF TEXT
  RAISES {TrestleComm.Failure} = BEGIN RAISE Unimplemented END NameList; *)

(* PROCEDURE MoveNearByName(v: VBT.T; nm: TEXT) RAISES {TrestleComm.Failure} = 
  BEGIN RAISE Unimplemented END MoveNearByName; *)

(* PROCEDURE SwapByName(
    v: VBT.T;
    nm: TEXT)
  RAISES {TrestleComm.Failure, Unimplemented} = 
  BEGIN RAISE Unimplemented END SwapByName; *)

(* PROCEDURE DeleteByName(nm: TEXT; trsl: T := NIL) RAISES {TrestleComm.Failure} = BEGIN RAISE Unimplemented END DeleteByName; *)

(* PROCEDURE TakeOver(
    id: ScreenID;
    v: VBT.T;
    trsl: T := NIL)
  RAISES {TrestleComm.Failure, Unimplemented} = 
  BEGIN RAISE Unimplemented END TakeOver; *)

(* PROCEDURE Restore(
    id: ScreenID;
    v: VBT.T)
  RAISES {TrestleComm.Failure, Unimplemented} = 
  BEGIN RAISE Unimplemented END Restore; *)

(* PROCEDURE TakeOverMouse(
    id: ScreenID;
    v: VBT.T;
    trsl: T := NIL)
  RAISES {TrestleComm.Failure} = 
  BEGIN RAISE Unimplemented END TakeOverMouse; *)

(* PROCEDURE ReleaseMouse(id: ScreenID; v: VBT.T) RAISES {TrestleComm.Failure} = 
  BEGIN RAISE Unimplemented END ReleaseMouse; *)

(* PROCEDURE SetHighlight(
    id: ScreenID;
    READONLY r: Rect.T;
    border: CARDINAL;
    trsl: T := NIL)
  RAISES {TrestleComm.Failure} = BEGIN RAISE Unimplemented END SetHighlight; *)

(* PROCEDURE AddParent(
    prnt: VBT.T;
    id: ScreenID;
    trsl: T := NIL)
  RAISES {TrestleComm.Failure, Unimplemented} = 
  BEGIN RAISE Unimplemented END AddParent; *)

(* PROCEDURE RemParent(
    prnt: VBT.T;
    id: ScreenID;
    trsl: T := NIL)
  RAISES {TrestleComm.Failure, Unimplemented} = 
  BEGIN RAISE Unimplemented END RemParent; *)

(* PROCEDURE WarpCursor(
    id: ScreenID;
    READONLY pt: Point.T;
    trsl: T := NIL)
  RAISES {TrestleComm.Failure} = BEGIN RAISE Unimplemented END WarpCursor; *)

(* PROCEDURE LastCeded(
    trsl: T := NIL)
    : VBT.TimeStamp
  RAISES {TrestleComm.Failure, Unimplemented} = 
  BEGIN RAISE Unimplemented END LastCeded; *)

(* PROCEDURE GetParameters(trsl: T := NIL): Parameters 
  RAISES {TrestleComm.Failure} = BEGIN RAISE Unimplemented END GetParameters; *)

(* PROCEDURE SetParameters(
    p: Parameters;
    trsl: T := NIL)
  RAISES {TrestleComm.Failure} = BEGIN RAISE Unimplemented END SetParameters;
*)

BEGIN END Trestle.
