(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Mon Aug  3 11:32:56 PDT 1992 by meehan *)
(*      modified on Tue Jun 16 13:08:32 PDT 1992 by muller *)
(*      modified on Mon Jun 15 21:49:42 1992 by mhb *)
(* modified on Fri Mar 20 22:35:51 1992 by steveg *)

(* A "NumericVBT" is a "VBT" class for displaying and changing
   an integer within some range.  A "NumericVBT" has three parts
   (from left to right): a minus button, a type-in field, and
   a plus button.  The type-in field is restricted to be an
   integer within the range "[min ..  max]"; it can be changed
   by editing or typing carriage return, or by clicking on the
   plus or minus buttons.  The plus/minus buttons are trill
   buttons, so clicking and holding will cause the value of the
   "NumericVBT" to continuously increment/decrement.

   The "NumericVBT" has a method that is called each time the
   user changes the value. *)

INTERFACE NumericVBT;

IMPORT AnyEvent, Filter, Font, Shadow, TextPort, VBT;

TYPE
  T <: Public;
  Public = Filter.T OBJECT
           METHODS
             init (min       : INTEGER;
                   max       : INTEGER;
                   allowEmpty: BOOLEAN;
                   naked     : BOOLEAN;
                   font      : Font.T;
                   shadow    : Shadow.T ): T;
             callback (event: AnyEvent.Code);
           END;

(* The call to "v.init(...)" initializes "v" as a "NumericVBT"
   and returns "v".  The integer stored with "v", referred to as
   {\em the value in} "v", is constrained to be in the range
| [min .. MAX (min, max)]
   The initial value in "v" is equal to "min".  If "allowEmpty"
   is "TRUE", then ``empty'' (no text in the type-in area) is a
   distinct and valid state, and can be tested by "IsEmpty".
   The call "Get(v)" in the empty state will return "FIRST(INTEGER)",
   regardless of whether this is in the valid range.  For
   "NumericVBT"s of limited range, this may be useful as a simple
   out-of-band signal that doesn't require the special "IsEmpty"
   test.  If "naked" is "TRUE", then the numeric interactor
   appears as just a type-in field, without plus or minus
   buttons.  The values of "font" and "shadow" are used for
   displaying the type-in area and the buttons.

   The implementation calls "v.callback(event)" when the user
   changes the value in "v".  The "event" parameter reports the
   details of the event as either an "AnyEvent.MouseCode"
   (clicking on the plus/minus buttons) or an "AnyEvent.KeyCode"
   (typing carriage return in the type-in area).  The value
   in "v" is changed before "v.callback" is invoked. *)


PROCEDURE Put (v: T; n: INTEGER);
(* Change the value in "v" to be "MIN(v.max, MAX(v.min, n))" and
   display this value in the type-in area.  Note that "v.callback"
   is not invoked; the value in "v" has changed, but not due
   to a user gesture. *)

PROCEDURE Get (v: T): INTEGER;
(* Return the value in "v". *)

PROCEDURE PutMin (v: T; minVal: INTEGER);
(* Change "v.min" to be "minVal" and then call "Put(v, v.value)".
   The call to "Put" has the effect of projecting the value of
   "v" into the new lower bound. *)

PROCEDURE GetMin (v: T): INTEGER;
(* Return "v.min". *)

PROCEDURE PutMax (v: T; maxVal: INTEGER);
(* Change "v.max" to be "maxVal" and then call "Put(v, v.value)".
   The call to "PutMax" has the effect of projecting the value of
   "v" into the new upper bound. *)

PROCEDURE GetMax (v: T): INTEGER;
(* Return "v.max". *)

PROCEDURE SetEmpty (v: T);
(* Set "v" to the empty state.  This is a no-op unless "allowEmpty" was
   "TRUE" when "v" was initialized. *)

PROCEDURE IsEmpty (v: T): BOOLEAN;
(* Test whether "v" is in the empty state.  If "allowEmpty" was
   not "TRUE" when "v" was initialized, this procedure will always
   return "FALSE". *)

PROCEDURE TakeFocus (v         : T;
                     time      : VBT.TimeStamp;
                     alsoSelect                  := TRUE):
  BOOLEAN;
(* Cause the type-in area to grab the keyboard focus.  If the
   focus could be grabbed and if "alsoSelect" is set, the type-in
   area will make its entire text the primary selection.  Returns
   whether the keyboard focus could be acquired. *)

PROCEDURE GetTypein (v: T): TextPort.T;
(* Return "v"'s type-in area.  Clients should manipulate the
   "TextPort" with extreme care. *)

END NumericVBT.

