(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: Max.m3                                                *)
(* Last Modified On Wed Apr 15 09:51:44 PDT 1992 By kalsow     *)
(*      Modified On Fri Dec 21 01:02:24 1990 By muller         *)

MODULE Max;

IMPORT CallExpr, Expr, Type, Procedure, Error, Emit;
IMPORT Int, Reel, LReel, EReel;
IMPORT IntegerExpr, EnumExpr, ReelExpr, Temp;

VAR Z: CallExpr.MethodList;

PROCEDURE TypeOf (<*UNUSED*> proc: Expr.T; VAR args: Expr.List): Type.T =
  BEGIN
    RETURN Type.Base (Expr.TypeOf (args[0]));
  END TypeOf;

PROCEDURE Check (<*UNUSED*> proc: Expr.T; VAR args: Expr.List;  <*UNUSED*> VAR cs: Expr.CheckState): Type.T =
  BEGIN
    RETURN DoCheck ("MAX", args);
  END Check;

PROCEDURE DoCheck (name: TEXT;  args: Expr.List): Type.T =
  VAR ta, tb: Type.T;
  BEGIN
    ta := Type.Base (Expr.TypeOf (args[0]));
    tb := Type.Base (Expr.TypeOf (args[1]));

    IF (NOT Type.IsEqual (ta, tb, NIL)) THEN
      Error.ID (name, "incompatible argument types");
    ELSIF (ta = Int.T) OR (Type.Number (ta) >= 0) THEN
      (* ok *)
    ELSIF (ta = Reel.T) OR (ta = LReel.T) OR (ta = EReel.T) THEN
      (* ok *)
    ELSE
      Error.ID (name, "wrong argument types");
      ta := Int.T;
    END;
    RETURN ta;
  END DoCheck;

PROCEDURE Compile (<*UNUSED*> proc: Expr.T; args: Expr.List): Temp.T =
  VAR t1, t2, t3, t4: Temp.T;
  BEGIN
    t1 := Expr.Compile (args[0]);
    t2 := Expr.Compile (args[1]);
    t3 := Temp.AllocEmpty (Expr.TypeOf (args[0]));
    t4 := Temp.Alloc (args[1]);
    Emit.OpTT ("@ = @;\n", t3, t1);
    Emit.OpTT ("@ = @;\n", t4, t2);
    Emit.OpTT ("if (@ < @) ", t3, t4);
    Emit.OpTT ("@ = @;\n", t3, t4);
    Temp.Free (t1);
    Temp.Free (t2);
    Temp.Free (t4);
    RETURN t3;
  END Compile;

PROCEDURE Fold (<*UNUSED*> proc: Expr.T; args: Expr.List): Expr.T =
  VAR e1, e2, e3: Expr.T;  i1, i2: INTEGER;  t: Type.T;
  BEGIN
    e1 := Expr.ConstValue (args[0]);
    IF (e1 = NIL) THEN RETURN NIL END;
    e2 := Expr.ConstValue (args[1]);
    IF (e2 = NIL) THEN RETURN NIL END;

    IF IntegerExpr.Split (e1, i1) AND IntegerExpr.Split (e2, i2) THEN
      RETURN IntegerExpr.New (MAX (i1, i2));
    ELSIF EnumExpr.Split (e1, i1, t) AND EnumExpr.Split (e2, i2, t) THEN
      RETURN EnumExpr.New (t, MAX (i1, i2));
    ELSIF ReelExpr.Max (e1, e2, e3) THEN
      RETURN e3;
    ELSE
      RETURN NIL;
    END;
  END Fold;

PROCEDURE Initialize () =
  BEGIN
    Z := CallExpr.NewMethodList (2, 2, TRUE, FALSE, NIL,
                                 TypeOf, Check, Compile, Fold,
                                 CallExpr.IsNever, (* writable *)
                                 CallExpr.IsNever(* designator *));
    Procedure.Define ("MAX", Z, TRUE);
  END Initialize;

BEGIN
END Max.
