(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: KeywordExpr.m3                                        *)
(* Last modified on Mon Mar  2 10:52:47 PST 1992 by kalsow     *)
(*      modified on Sat Dec  8 00:21:11 1990 by muller         *)

MODULE KeywordExpr;

IMPORT Expr, ExprRep, Type, String, Temp, MBuf;

TYPE
  P = Expr.T OBJECT
        expr : Expr.T;
	key  : String.T;
      OVERRIDES
        typeOf       := TypeOf;
        check        := Check;
        compile      := Compile;
        evaluate     := Fold;
        fprint       := FPrinter;
        write        := ExprRep.NoWriter;
        isEqual      := EqCheck;
        getBounds    := Bounder;
        isWritable   := IsWritable;
        isDesignator := IsDesignator;
	isZeroes     := IsZeroes;
	genLiteral   := GenLiteral;
      END;

PROCEDURE New (key: String.T;  arg: Expr.T): Expr.T =
  VAR p: P;
  BEGIN
    p := NEW (P);
    ExprRep.Init (p);
    p.key  := key;
    p.expr := arg;
    RETURN p;
  END New;

PROCEDURE Is (e: Expr.T): BOOLEAN =
  BEGIN
    TYPECASE e OF
    | NULL => RETURN FALSE;
    | P    => RETURN TRUE;
    ELSE      RETURN FALSE;
    END;
  END Is;

PROCEDURE Split (e: Expr.T;  VAR key: String.T;  VAR value: Expr.T): BOOLEAN =
  BEGIN
    TYPECASE e OF
    | NULL => RETURN FALSE;
    | P(p) => key := p.key;  value := p.expr;  RETURN TRUE;
    ELSE      RETURN FALSE;
    END;
  END Split;

PROCEDURE TypeOf (p: P): Type.T =
  BEGIN
    RETURN Expr.TypeOf (p.expr);
  END TypeOf;

PROCEDURE Check (p: P;  VAR cs: Expr.CheckState) =
  BEGIN
    Expr.TypeCheck (p.expr, cs);
    p.type := Expr.TypeOf (p.expr);
  END Check;

PROCEDURE EqCheck (a: P;  e: Expr.T): BOOLEAN =
  BEGIN
    TYPECASE e OF
    | NULL => RETURN FALSE;
    | P(b) => RETURN Expr.IsEqual (a.expr, b.expr);
    ELSE      RETURN Expr.IsEqual (a.expr, e);
    END;
  END EqCheck;

PROCEDURE Compile (p: P): Temp.T =
  BEGIN
    RETURN Expr.Compile (p.expr);
  END Compile;

PROCEDURE Bounder (p: P;  VAR min, max: INTEGER) =
  BEGIN
    Expr.GetBounds (p.expr, min, max);
  END Bounder;

PROCEDURE IsDesignator (p: P): BOOLEAN =
  BEGIN
    RETURN Expr.IsDesignator (p.expr);
  END IsDesignator;

PROCEDURE IsWritable (p: P): BOOLEAN =
  BEGIN
    RETURN Expr.IsWritable (p.expr);
  END IsWritable;

PROCEDURE Fold (p: P): Expr.T =
  BEGIN
    WITH e = Expr.ConstValue (p.expr) DO
      IF e = NIL THEN
        RETURN NIL
      ELSE
        p.expr := e; 
        RETURN p; 
      END;
    END;
  END Fold;

PROCEDURE FPrinter (p: P; map: Type.FPMap; wr: MBuf.T) =
  BEGIN
    MBuf.PutText (wr, ":= ");
    String.Put (wr, p.key);
    Expr.Fingerprint (p.expr, map, wr);
  END FPrinter;

PROCEDURE IsZeroes (p: P): BOOLEAN =
  BEGIN
    RETURN Expr.IsZeroes (p.expr);
  END IsZeroes;

PROCEDURE GenLiteral (p: P) =
  BEGIN
    Expr.GenLiteral (p.expr);
  END GenLiteral;

BEGIN
END KeywordExpr.
