(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: NotExpr.m3                                            *)
(* Last modified on Mon Jul  8 18:25:23 1991 by kalsow         *)
(*      modified on Thu Nov 29 03:28:44 1990 by muller         *)

MODULE NotExpr;

IMPORT Expr, ExprRep, Type, Bool, Error, Emit, EnumExpr, Temp, MBuf;

TYPE
  P = ExprRep.Ta BRANDED "NotExpr.P" OBJECT
      OVERRIDES
        typeOf       := ExprRep.NoType;
        check        := Check;
        compile      := Compile;
        evaluate     := Fold;
        fprint       := FPrinter;
        write        := Writer;
        isEqual      := ExprRep.EqCheckA;
        getBounds    := ExprRep.NoBounds;
        isWritable   := ExprRep.IsNever;
        isDesignator := ExprRep.IsNever;
	isZeroes     := ExprRep.IsNever;
	genLiteral   := ExprRep.NoLiteral;
      END;

PROCEDURE New (a: Expr.T): Expr.T =
  VAR p: P;
  BEGIN
    p := NEW (P);
    ExprRep.Init (p);
    p.a    := a;
    p.type := Bool.T;
    RETURN p;
  END New;

PROCEDURE Check (p: P;  VAR cs: Expr.CheckState) =
  VAR ta: Type.T;
  BEGIN
    Expr.TypeCheck (p.a, cs);
    ta := Type.Base (Expr.TypeOf (p.a));
    IF (ta # Bool.T) THEN
      Error.Msg ("operand of NOT must be of type BOOLEAN");
    END;
  END Check;

PROCEDURE Compile (p: P): Temp.T =
  VAR t1, t2: Temp.T;
  BEGIN
    t1 := Expr.Compile (p.a);
    t2 := Temp.AllocMacro (p, FALSE);
    Temp.Depend (t2, t1);
    RETURN t2;
  END Compile;

PROCEDURE Writer (<*UNUSED*> p: P;  t1: Temp.T;  <*UNUSED*> t2: Temp.T) =
  BEGIN
    Emit.OpT ("(!@)", t1);
  END Writer;

PROCEDURE Fold (p: P): Expr.T =
  VAR e1, e3: Expr.T; i: INTEGER; t: Type.T;
  BEGIN
    e1 := Expr.ConstValue (p.a);
    e3 := NIL;
    IF (e1 # NIL)
      AND (EnumExpr.Split (e1, i, t))
      AND (Type.Base (t) = Bool.T) THEN
      e3 := Bool.Map[ORD (FALSE) = i]
    END;
    RETURN e3;
  END Fold;

PROCEDURE FPrinter (p: P;  map: Type.FPMap;  wr: MBuf.T) =
  BEGIN
    MBuf.PutText (wr, "NOT ");
    Expr.Fingerprint (p.a, map, wr);
  END FPrinter;

BEGIN
END NotExpr.
