(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: PlusExpr.m3                                           *)
(* Last modified on Mon Jul  8 18:26:11 1991 by kalsow         *)
(*      modified on Thu Nov 29 03:29:06 1990 by muller         *)

MODULE PlusExpr;

IMPORT Expr, ExprRep, Type, Error, Int, Reel, LReel, EReel, Temp, MBuf;

TYPE
  P = ExprRep.Ta BRANDED "PlusExpr.P" OBJECT
      OVERRIDES
        typeOf       := TypeOf;
        check        := Check;
        compile      := Compile;
        evaluate     := Fold;
        fprint       := FPrinter;
        write        := ExprRep.NoWriter;
        isEqual      := EqCheck;
        getBounds    := ExprRep.NoBounds;
        isWritable   := ExprRep.IsNever;
        isDesignator := ExprRep.IsNever;
	isZeroes     := ExprRep.IsNever;
	genLiteral   := ExprRep.NoLiteral;
      END;

PROCEDURE New (a: Expr.T): Expr.T =
  VAR p: P;
  BEGIN
    p := NEW (P);
    ExprRep.Init (p);
    p.a := a;
    RETURN p;
  END New;

PROCEDURE TypeOf (p: P): Type.T =
  BEGIN
    RETURN Expr.TypeOf (p.a);
  END TypeOf;

PROCEDURE Check (p: P;  VAR cs: Expr.CheckState) =
  VAR t: Type.T;
  BEGIN
    Expr.TypeCheck (p.a, cs);
    p.type := Expr.TypeOf (p.a);
    t := Type.Base (p.type);
    IF (t # Int.T) AND (t # Reel.T) AND (t # LReel.T) AND (t # EReel.T) THEN
      Error.Msg ("illegal operands for unary \'+\'");
    END;
  END Check;

PROCEDURE EqCheck (a: P;  e: Expr.T): BOOLEAN =
  BEGIN
    TYPECASE e OF
    | NULL => RETURN FALSE;
    | P(b) => RETURN Expr.IsEqual (a.a, b.a);
    ELSE      RETURN Expr.IsEqual (a.a, e);
    END;
  END EqCheck;

PROCEDURE Compile (p: P): Temp.T =
  BEGIN
    RETURN Expr.Compile (p.a);
  END Compile;

PROCEDURE Fold (p: P): Expr.T =
  BEGIN
    RETURN Expr.ConstValue (p.a);
  END Fold;

PROCEDURE FPrinter (p: P;  map: Type.FPMap;  wr: MBuf.T) =
  BEGIN
    MBuf.PutText (wr, "U+ ");
    Expr.Fingerprint (p.a, map, wr);
  END FPrinter;

BEGIN
END PlusExpr.
