(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: Frame.m3                                              *)
(* Last Modified On Mon Mar  2 11:06:47 PST 1992 by kalsow     *)

MODULE Frame;

IMPORT Target, Host, Type, Temp, Emit, OpenArrayType, Fault;

CONST
  Grain     = MAX (Target.INTSIZE, Target.ADDRSIZE);
  FixedSize = (Target.FixedFrameSize + Grain - 1) DIV Grain;
  PageSize  = (Target.GuardPageSize + Grain - 1) DIV Grain;

PROCEDURE Push (VAR t: T;  locals: INTEGER;  formatted_code := FALSE) =
  BEGIN
    t := cur;
    cur.size := FixedSize + locals;
    cur.max  := 0;
    cur.gone := 0;
    Emit.PushProcedure (formatted_code);
    Temp.PushSet ();
    Emit.Op ("\n");
  END Push;

PROCEDURE Pop  (READONLY t: T) =
  VAR size := cur.size + cur.gone;
  BEGIN
    (* finish the code body *)
    Emit.Op ("\002}\n");

    (* finish the decls & insert the stack check *)
    IF (size > PageSize) AND (Host.doStackChk) THEN
      EVAL Emit.SwitchToDecls ();
      Fault.StackOverflow ();
    END;

    Temp.PopSet ();
    Emit.PopProcedure ();
    cur := t;
  END Pop;

PROCEDURE PushBlock (VAR save: INTEGER;  new := 0) =
  BEGIN
    save := cur.size;
    INC (cur.size, new);
    Emit.Op ("{\001\n");
  END PushBlock;

PROCEDURE PopBlock  (save: INTEGER) =
  BEGIN
    IF (cur.size > cur.max) THEN cur.max := cur.size END;
    INC (cur.gone, cur.size - save);
    cur.size := save;
    Emit.Op ("\002}\n");
  END PopBlock;

PROCEDURE NoteDeclaration (t: Type.T) =
  VAR sz := Type.Size (t);
  BEGIN
    IF (sz > 0) THEN
      INC (cur.size, (sz + Grain - 1) DIV Grain);
    ELSE (* open array => declare space for the dope vector *)
      INC (cur.size, 1 + OpenArrayType.OpenDepth (t));
    END;
  END NoteDeclaration;

BEGIN
END Frame.
