(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: RecordType.m3                                         *)
(* Last modified on Thu Sep  3 09:42:22 PDT 1992 by rustan     *)
(*      modified on Tue Jun  9 15:31:23 PDT 1992 by kalsow     *)
(*      modified on Tue Mar 26 02:49:22 1991 by muller         *)

MODULE RecordType;

IMPORT Type, TypeRep, Scope, String, Expr, Value, Token;
IMPORT Emit, Error, Field, Ident, MBuf, PackedType, Target;
IMPORT Int, Word, M3;
IMPORT TrOffsets;
FROM Scanner IMPORT Match, Match1, GetToken, cur;

TYPE
  P = Type.T OBJECT
        scope      : Scope.T;
        fields     : Scope.ValueList;
        recSize    : INTEGER;
        align      : INTEGER;
      OVERRIDES
        check      := Check;
        base       := TypeRep.SelfBase;
        isEqual    := EqualChk;
        isSubtype  := TypeRep.NoSubtypes;
        count      := TypeRep.NotOrdinal;
        bounds     := TypeRep.NotBounded;
        size       := Sizer;
        minSize    := Sizer;
        alignment  := Aligner;
	isEmpty    := IsEmpty;
        dependsOn  := DependsOn;
        compile    := Compiler;
        initCost   := InitCoster;
        initValue  := GenInit;
        tracedOffs := TracedOffs;
        paramEncoding := ParamEnc;
        mapper     := GenMap;
        fprint     := FPrinter;
        class      := MyClass;
      END;

PROCEDURE Parse (READONLY fail: Token.Set): Type.T =
  VAR t: Type.T;
  BEGIN
    t := New (Scope.PushNew (FALSE, NIL));
    Match (Token.T.tRECORD, fail, Token.Set {Token.T.tEND});
    ParseFieldList (fail + Token.Set {Token.T.tEND});
    Match1 (Token.T.tEND, fail);
    Scope.PopNew ();
    RETURN t;
  END Parse;

PROCEDURE ParseFieldList (READONLY fail: Token.Set) =
  TYPE TK = Token.T;
  VAR
    fail2   : Token.Set;
    type    : Type.T;
    dfault  : Expr.T;
    j, n    : INTEGER;
    nFields : INTEGER;
  BEGIN
    fail2 := fail + Token.Set {TK.tCOLON, TK.tASSIGN, TK.tSEMI};
    nFields := 0;
    WHILE (cur.token = TK.tIDENT) DO
      n := Ident.ParseList (fail2);
      type := NIL;
      IF (cur.token = TK.tCOLON) THEN
        GetToken (); (* : *)
        type := Type.Parse (fail + Token.Set {TK.tASSIGN, TK.tSEMI}
                                                            + Token.ExprStart);
      END;
      dfault := NIL;
      IF (cur.token = TK.tEQUAL) THEN
        Error.Msg ("default value must begin with ':='");
        cur.token := TK.tASSIGN;
      END;
      IF (cur.token = TK.tASSIGN) THEN
        GetToken (); (* := *)
        dfault := Expr.Parse (fail + Token.Set {TK.tSEMI});
      END;
      IF (type = NIL) AND (dfault = NIL) THEN
        Error.Msg ("fields must include a type or default value");
      END;
      j := Ident.top - n;
      FOR i := 0 TO n - 1 DO
        Scope.Insert (Field.New (Ident.stack[j + i], nFields, type, dfault));
        INC (nFields);
      END;
      DEC (Ident.top, n);
      IF (cur.token # TK.tSEMI) THEN EXIT END;
      GetToken (); (* ; *)
    END;
  END ParseFieldList;

PROCEDURE New (fields: Scope.T): Type.T =
  VAR p: P;
  BEGIN
    p := NEW (P);
    TypeRep.Init (p);
    p.scope  := fields;
    p.fields := NIL;
    RETURN p;
  END New;

PROCEDURE Split (t: Type.T;  VAR fields: Scope.T): BOOLEAN =
  BEGIN
    TYPECASE Type.Strip (t) OF
    | NULL =>  RETURN FALSE;
    | P(p) =>  fields := p.scope;  RETURN TRUE;
    ELSE       RETURN FALSE;
    END;
  END Split;

PROCEDURE LookUp (t: Type.T;  field: String.T;  VAR obj: Value.T): BOOLEAN =
  BEGIN
    TYPECASE Type.Strip (t) OF
    | NULL =>  RETURN FALSE;
    | P(p) =>  obj := Scope.LookUp (p.scope, field, TRUE);  RETURN (obj # NIL);
    ELSE       RETURN FALSE;
    END;
  END LookUp;

(***********************************************************************)

PROCEDURE MyClass (<*UNUSED*> p: P): TypeRep.Class =
  BEGIN
    RETURN TypeRep.Class.Record;
  END MyClass;

PROCEDURE Check (p: P) =
  VAR
    objs  : Scope.ValueList;
    n     : INTEGER;
    types : ARRAY [0..39] OF Type.T;
    cs    := M3.OuterCheckState;
  BEGIN
    Scope.TypeCheck (p.scope, cs);
    Scope.ToList (p.scope, objs, n);
    IF (n <= NUMBER (types))
      THEN CheckFields (p, objs, n, types);
      ELSE CheckFields (p, objs, n, NEW (REF ARRAY OF Type.T, n)^);
    END;
  END Check;

PROCEDURE CheckFields (p       : P;
                       objs    : Scope.ValueList;
                       nFields : INTEGER;
                   VAR types   : ARRAY OF Type.T) =
  VAR
    j       : INTEGER;
    o       : Value.T;
    type    : Type.T;
    hash    : INTEGER;
    fname   : String.T;
  BEGIN
    (* sort the fields in their order of occurance *)
    p.fields := NEW (Scope.ValueList, nFields);
    FOR i := 0 TO nFields - 1 DO
      o := objs[i];
      Field.SplitX (o, j, type);
      p.fields[j] := o;
      types[j] := Type.Strip (type);
      p.isTraced := p.isTraced OR Type.IsTraced (type);
      p.hasUntraced := p.hasUntraced OR Type.HasUntraced (type);
      p.isLocalOnly := p.isLocalOnly OR Type.IsLocalOnly (type)
    END;

    (* assign the final offsets to each field *)
    SizeAndAlignment (p.scope, p.recSize, p.align);

    (* compute the hash value for this record *)
    hash := Word.Plus (Word.Times (943, p.recSize), p.align);
    FOR i := 0 TO nFields-1 DO
      fname := Value.CName (p.fields[i]);
      hash := Word.Plus (Word.Times (hash, 41), String.Hash (fname));
      hash := Word.Plus (Word.Times (hash, 37), Type.Size (types[i]));
    END;
    p.hash := hash;
  END CheckFields;

PROCEDURE SizeAndAlignment (fields: Scope.T;  VAR recSize, recAlign: INTEGER) =
  VAR objs    : Scope.ValueList;
      nFields : INTEGER;
  BEGIN
    Scope.ToList (fields, objs, nFields);
    SizesOffsetsAlignments (objs, nFields, recSize, recAlign)
  END SizeAndAlignment;

(* Procedure SizesOffsetsAlignments computes the size of a record, given
   its fields.  The size and alignment of the record are returned in the
   OUT-parameters recSize and recAlign.  If proc is non-NIL, it is called
   for every non-packed field in the record, with the type of the field
   and its offset in number of addressable units from the beginning of the
   record. *)
PROCEDURE SizesOffsetsAlignments (objs: Scope.ValueList; nFields: INTEGER;
                                  VAR recSize, recAlign: INTEGER;
                                  proc: PROCEDURE( fieldType: Type.T;
                                                   fieldOffset: CARDINAL )
                                    := NIL) =
  VAR
    b              : Type.T;
    s              : INTEGER;
    fieldAlign     : INTEGER;
    fieldSize      : INTEGER;
    leftInThisUnit : INTEGER;
    intAlign       : INTEGER;
    intSize        : INTEGER;
    starting       : INTEGER;
    packed         := FALSE;
    previousPacked := FALSE;
    o              : Value.T;
    offset         : INTEGER;
    type           : Type.T;
  BEGIN
    (* compute the size of the record *)
    recSize  := 0; (* total size of the record *)
    recAlign := Target.STRUCTURESIZEBOUNDARY; (* minimum allowed alignment *)
    intAlign := Type.Alignment (Int.T);
    intSize  := Type.Size (Int.T);

    (* compute the field offsets *)
    FOR i := 0 TO nFields - 1 DO
      o := objs[i];
      Field.SplitX (o, offset, type);
      <*ASSERT offset = i *>

      packed := PackedType.Split (type, s, b);
      fieldSize := Type.Size (type);
      fieldAlign := Type.Alignment (type);

      recAlign := MAX (recAlign, fieldAlign);

      IF packed THEN
        IF Target.PCCBITFIELDTYPEMATTERS THEN
          recAlign := MAX (recAlign, intAlign);
        END;
        (* can we fit the field there ? *)
        IF previousPacked THEN
          IF leftInThisUnit < fieldSize THEN
            IF leftInThisUnit # 0 THEN
              Error.Msg ("SRC Modula-3 does not support this type");
            END;
            INC (recSize, intSize);
	    IF Target.BITFIELDCANOVERLAP THEN
	      starting := RoundUp (intSize - leftInThisUnit, intAlign);
              leftInThisUnit := 2 * intSize - fieldSize - starting;
            ELSE
              leftInThisUnit := intSize - fieldSize;
            END;
          ELSE
            DEC (leftInThisUnit, fieldSize);
          END;
        ELSE
          recSize := RoundUp (recSize, intSize);
          INC (recSize, intSize);
          leftInThisUnit := intSize - fieldSize;
        END;
      ELSE
        recSize := RoundUp (recSize, fieldAlign);
        IF proc # NIL THEN
          proc( type, recSize DIV Target.ADDRUNIT )
        END;
        INC (recSize, fieldSize);
      END;
      previousPacked := packed;
    END;

    IF recSize # 0 THEN
      recAlign := MAX (recAlign, Target.STRUCTURESIZEBOUNDARY);
      recSize  := RoundUp (recSize, recAlign);
    END
  END SizesOffsetsAlignments;

PROCEDURE RoundUp (size, alignment: INTEGER): INTEGER =
  BEGIN
    IF (alignment = 0)
      THEN RETURN size;
      ELSE RETURN ((size + alignment - 1) DIV alignment) * alignment;
    END;
  END RoundUp;

PROCEDURE Compiler (p: P) =
  BEGIN
    Scope.InitValues (p.scope);
    IF TypeRep.IsCompiled (p) THEN RETURN END;
    GenDecl (p);

    IF TypeRep.StartLinkInfo (p) THEN RETURN END;

    GenDependencies (p);

    Emit.Op ("C\n");
    GenDecl (p);
    Emit.Op ("*\n");
  END Compiler;

PROCEDURE GenDecl (p: P) =
  BEGIN
    Emit.OpF ("struct _rec@ {\n\001", p);
    FOR i := 0 TO LAST (p.fields^) DO
      Field.EmitDeclaration (p.fields[i]);
    END;
    IF LAST (p.fields^) < 0 THEN
      Emit.Op ("int _filler;\n"); 
    END;
    Emit.Op ("\002};\n");
    Emit.OpFF ("typedef struct _rec@ @;\n", p, p);
  END GenDecl;

PROCEDURE GenDependencies (p: P) =
  VAR offset: INTEGER;  o: Value.T;  type: Type.T;
  BEGIN
    FOR i := 0 TO LAST (p.fields^) DO
      o := p.fields[i];
      Field.SplitX (o, offset, type);
      Emit.OpF ("d@\n", type);
    END;
  END GenDependencies;

PROCEDURE EqualChk (a: P;  t: Type.T;  x: Type.Assumption): BOOLEAN =
  VAR b: P;
  BEGIN
    TYPECASE Type.Strip (t) OF
    | NULL => RETURN FALSE;
    | P(p) => b := p;
    ELSE      RETURN FALSE;
    END;

    (* first, make some simple sanity checks *)
    IF (a.recSize # b.recSize) THEN RETURN FALSE END;
    IF (a.align # b.align) THEN RETURN FALSE END;
    IF (a.isTraced # b.isTraced) THEN RETURN FALSE END;
    IF (NUMBER (a.fields^) # NUMBER (b.fields^)) THEN RETURN FALSE END;

    (* compare the fields *)
    FOR i := 0 TO LAST (a.fields^) DO
      IF NOT Field.IsEqual (a.fields[i], b.fields[i], x) THEN RETURN FALSE END;
    END;

    RETURN TRUE;
  END EqualChk;

PROCEDURE Sizer (p: P): INTEGER =
  BEGIN
    IF (NOT p.checked) THEN Check (p) END;
    RETURN p.recSize;
  END Sizer;

PROCEDURE Aligner (p: P): INTEGER =
  BEGIN
    IF (NOT p.checked) THEN Check (p) END;
    RETURN p.align;
  END Aligner;

PROCEDURE IsEmpty (p: P): BOOLEAN =
  VAR n, offs: INTEGER;  t: Type.T;  fields: Scope.ValueList;
  BEGIN
    Scope.ToList (p.scope, fields, n);
    FOR i := 0 TO n-1 DO
      Field.SplitX (fields[i], offs, t);
      IF Type.IsEmpty (t) THEN RETURN TRUE END;
    END;
    RETURN FALSE;
  END IsEmpty;

PROCEDURE DependsOn (p: P;  t: Type.T): BOOLEAN =
  VAR n, offs: INTEGER;  ft: Type.T;  fields: Scope.ValueList;
  BEGIN
    Scope.ToList (p.scope, fields, n);
    FOR i := 0 TO n-1 DO
      Field.SplitX (fields[i], offs, ft);
      IF Type.DependsOn (ft, t) THEN RETURN TRUE END;
    END;
    RETURN FALSE;
  END DependsOn;

PROCEDURE InitCoster (p: P;  zeroed: BOOLEAN): INTEGER =
  VAR j, cost: INTEGER;  t: Type.T;  field: Value.T;
  BEGIN
    cost := 0;
    FOR i := 0 TO LAST (p.fields^) DO
      field := p.fields[i];
      Field.SplitX (field, j, t);
      IF Field.GetDefault (field) # NIL
        THEN INC (cost, MAX (1, Type.InitCost (t, FALSE)));
        ELSE INC (cost, Type.InitCost (t, zeroed ));
      END;
      IF (cost < 0) THEN RETURN LAST (INTEGER) END;
    END;
    RETURN cost;
  END InitCoster;

PROCEDURE GenInit (<*UNUSED*> p: P) =
  BEGIN
    <* ASSERT FALSE *>
  END GenInit;

PROCEDURE TracedOffs (p: P;  offset: CARDINAL): TrOffsets.T =
  BEGIN
    RETURN TracedOffsets( p.fields, NUMBER( p.fields^ ), offset )
  END TracedOffs;

PROCEDURE TracedOffsets (fields: Scope.ValueList;
                         nFields: INTEGER;
                         offset: CARDINAL := 0): TrOffsets.T =
  PROCEDURE ProcessField (fieldType: Type.T; fieldOffset: CARDINAL) =
    VAR troff    : TrOffsets.T;
    BEGIN
      troff := Type.TracedOffsets( fieldType, offset + fieldOffset );
      IF head = NIL THEN head := troff END;
      tail := TrOffsets.Append( tail, troff )
    END ProcessField;
  VAR head, tail : TrOffsets.T := NIL;
      recSize, recAlign: INTEGER;
  BEGIN
    SizesOffsetsAlignments (fields, nFields, recSize, recAlign, ProcessField);
    RETURN head
  END TracedOffsets;

PROCEDURE ParamEnc (p: P): TEXT =
  VAR enc: TEXT := "";
      offset: INTEGER;
      type: Type.T;
  BEGIN
    FOR i := FIRST( p.fields^ ) TO LAST( p.fields^ ) DO
      Field.SplitX( p.fields[i], offset, type );
      enc := enc & Type.ParamEncoding( type )
    END;
    RETURN enc
  END ParamEnc;

VAR dot: String.T := NIL;

PROCEDURE GenMap (p: P;  VAR prefix: String.Stack) =
  BEGIN
    IF (dot = NIL) THEN dot := String.Add (".") END;
    prefix.stk[prefix.top] := dot;
    INC (prefix.top, 2);
    FOR i := 0 TO LAST (p.fields^) DO
      prefix.stk[prefix.top-1] := Value.CName (p.fields[i]);
      Type.GenMap (Value.TypeOf (p.fields[i]), prefix);
    END;
    DEC (prefix.top, 2);
  END GenMap;

PROCEDURE FPrinter (p: P;  map: Type.FPMap;  wr: MBuf.T) =
  BEGIN
    MBuf.PutText (wr, "RECORD");
    FOR i := 0 TO LAST (p.fields^) DO
      MBuf.PutText (wr, " ");
      Value.Fingerprint (p.fields[i], map, wr);
    END;
  END FPrinter;

BEGIN
END RecordType.
