(* new interface KRML *)
UNSAFE INTERFACE RTStackKRML;
(* This interface is marked as UNSAFE because of the untraced references
   used in the implementation.  Unintended calls to, for example,
   Dispose or GetBottom, may cause trouble. *)

IMPORT Word, Thread;

TYPE
  T <: ADDRESS;

PROCEDURE New( t: Thread.T; size: CARDINAL;
               modelFrame: UNTRACED REF ARRAY OF Word.T;
               VAR stackLow, stackHigh: ADDRESS ): T;
(* REQUIRES Thread.inSystemCritical AND size # 0 *)
(* Return a new stack.  't' is the thread that is getting a stack.
   'size' indicates the number of Word.T that the stack will consist
   of in addition to the frame, and must be non-0.  'modelFrame' is
   an array consisting of an initial frame with which to initialize
   the stack.  The OUT parameters stackLow and stackHigh return
   as values that can be passed to SetCurrentStackLimits. *)

PROCEDURE Dispose( VAR stack: T );
(* REQUIRES Thread.inSystemCritical *)
(* Dispose of a stack.  't' returns as NIL, and its previous value
   should not be deferenced after a call to this procedure.
   Note, a thread can never dispose of its own stack, since it needs
   its stack to call this function. *)

PROCEDURE GetBounds( VAR resumeKey: ADDRESS;
                     VAR low, high: UNTRACED REF Word.T ): BOOLEAN;
(* Returns the inclusive bounds 'low' and 'high' of the "next" (explained
   below) stack.  This procedure is intended to be called from the
   garbage collector thread.

   This procedure iterates over the stacks in the system, skipping
   the stack of the garbage collecting thread (which is intended to
   be the calling thread).  A iteration of all such stacks is
   started with a call with resumeKey := NIL.  The value returned
   in 'resumeKey' is then used as the IN-parameter of the next call.
   The procedure returns TRUE as long as there was another stack
   to be queried, and returns FALSE when all stacks have been
   queried.  When the procedure returns FALSE, 'resumeKey', 'low', and
   'high' are undefined and should not be used.

   Once a described iteration begins, it is assumed that it will
   be completed before the next iteration begins.  Moreover, it
   is assumed that no context switch or other call to the interface
   is made during a described iteration. *)

PROCEDURE InitGcStack( stack: ADDRESS; totalSize: CARDINAL;
                       modelFrame: UNTRACED REF ARRAY OF Word.T;
                       VAR stackLow, stackHigh: ADDRESS );
(* Initializes the stack of the garbage collector. *)

PROCEDURE GetMainStackLimits( VAR low, high: ADDRESS );
(* Returns the stack limits for the main thread. *)

PROCEDURE SetCurrentStackLimits( low, high: ADDRESS );
(* Sets the stack limits to be used for checking stack overflows. *)

END RTStackKRML.
