(* Copyright (C) 1990, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last modified on Thu Mar  5 13:33:52 PST 1992 by kalsow     *)
(*      modified on Mon Nov 18 15:10:00 PST 1991 by muller     *)

UNSAFE MODULE RTStack;

IMPORT Unix, RTOS, RTMisc;

CONST Protected   = RTOS.Read_mode;
CONST Unprotected = RTOS.Read_mode + RTOS.Write_mode;
(* the guard page is marked read-only so that when Thread.InitContext
   writes the "seal" on the non-protected pages it doesn't generate
   a nil fault. *)

VAR page_bytes : CARDINAL := 0;
VAR stack_slop : CARDINAL;

PROCEDURE Init () =
  BEGIN
    page_bytes := Unix.getpagesize ();
    stack_slop := 2 * (page_bytes DIV BYTESIZE (INTEGER));
  END Init;

PROCEDURE New (size: INTEGER): T =
  VAR t: T;  i: INTEGER;  start: ADDRESS;
  BEGIN
    IF (page_bytes = 0) THEN Init () END;

    (* allocate enough so that we're guaranteed to get a full, aligned page *)
    t := NEW (T, size + stack_slop);

    (* find the aligned page and unmap it *)
    start := RTMisc.Align (ADR (t[0]), page_bytes);
    i := RTOS.mprotect (start, page_bytes, Protected);
    <* ASSERT i = 0 *>

    (* finally, return the new stack *)
    RETURN t;
  END New;

PROCEDURE Dispose (t: T) =
  VAR i: INTEGER;  start := RTMisc.Align (ADR (t[0]), page_bytes);
  BEGIN
    (* find the aligned page and re-map it *)
    i := RTOS.mprotect (start, page_bytes, Unprotected);
    <* ASSERT i = 0 *>

    (* and finally, free the storage *)
    DISPOSE (t);
  END Dispose;

PROCEDURE GetBounds (t: T;  VAR(*OUT*) first, last: ADDRESS) =
  VAR start := RTMisc.Align (ADR (t[0]), page_bytes);
  BEGIN
    first := start + page_bytes + ADRSIZE (t[0]);
    last  := ADR (t [0]) + NUMBER (t^) * ADRSIZE (t[0]);
  END GetBounds;

BEGIN
END RTStack.
