/* Copyright (C) 1994, Klaus Preschern.                        */
/* All rights reserved.                                        */
/* See the file COPYRIGHT.KP for a full description.           */

#include <sys/syscalls.h>
#include <signal.h>
#include <sigaction.h>
#include <errno.h>
#include <sys/message.h>

#define MAX_SIG			_NSIG

static struct sigvec {
  SignalHandler handler [MAX_SIG];
} sig_vector;

extern int __pid;

int _kill (int pid, int sig)
  {
    MESSAGE mess;
    mess.type = m_kill;
    mess.m.kill_m.pid    = pid;
    mess.m.kill_m.signal = sig;
    return call_system (&mess);
  }

int _sigsetmask (int mask)
  {
    MESSAGE mess;
    mess.type = m_sigmask;
    mess.m.sigmask_m.mask = mask;
    return call_system (&mess);
  }

int _sigaction (int sig, struct sigaction * act, struct sigaction * oact)
  {
     SignalHandler handler; int save;
     
     if ((sig <= 0) || (sig >= NSIG)) {
       errno = EINVAL;
       return -1;
     }
     if (act == NULL) {
       if (oact == NULL) {
         return 0;
       }
       /* race conditon - but there is no other way to do it */
       handler = signal (sig, SIG_IGN);
       if (handler == SIG_ERR) {
         return -1;
       }
       save = errno;
       (void) signal (sig, handler);
       errno = save;
     } else {
       if ((act->sa_flags != 0) || (act->sa_mask != 0)) {
         errno = ENOSYS;
	 return -1;
	 handler = signal (sig, act->sa_handler);
	 if (handler == SIG_ERR) {
	   return -1;
	 }
       }
     }
     
     if (oact != NULL) {
       oact->sa_handler = handler;
       oact->sa_mask = 0;
       oact->sa_flags = 0;
     }
     return 0;
  }

SignalHandler _signal (int sig, SignalHandler action)
  {
    SignalHandler old_val; int mask, old_mask;
    if ((sig > 0) && (sig < MAX_SIG)) {
      mask = 1 << (sig - 1);
      old_val = sig_vector.handler [sig];
      sig_vector.handler [sig] = action;
      old_mask = _sigsetmask (0);	        /* get old mask */
      mask = old_mask & ~mask;			/* remove signal from mask */
      _sigsetmask (mask);		     	/* now catch this signal */
    } else {
      errno = EINVAL;
      return SIG_ERR;
    }
    return old_val;
  }
  
/* __signal () is the procedure where signals arrive. 'mess' is pushed 
 * by the kernel and contains a signal message (i.e. all registers with
 * values before the occurrence of this signal). Pending system calls 
 * (i.e. wait ()) are preemted and return EINVAL. 
 * The kernel automatically disables further arrivals of the same signal.
 * To enable the signal again it is necessary to use sigsetmask (). If
 * a disabled signal arrives the process is killed.
 * SIGKILL is handled by the kernel (it never arrives at __signal ()).
 */  
 
static void __signal (MESSAGE mess)
  {
    SignalHandler handler; 
    int signal;
    
    /* 'mess.result' contains the signal number */
    signal = mess.result;
    
    /* Check wether this is a valid signal */
    if ((signal > 0) && (signal < MAX_SIG)) {
      handler = sig_vector.handler [signal]; /* get the signal handler */
    } else {
      handler = SignalBad;
    }
    
    if (handler == SignalDefault) {
      _exit (-1);
    } else if (handler == SignalBad) {
      _exit (-1);
    } else if (handler == SignalIgnore) {
       				     	   /* ignore this signal */
    } else {
      handler ();    			   /* call the handler */
    }
    /* do a process jump */
    _procjmp (__pid, mess.m.signal_m.regs);
  }  
  
void init_signals ()
  {
    /* Send the address of __signal () to the system. */
    MESSAGE mess; int i;
    __pid = getpid ();			/* get my pid */
    for (i = 1; i < MAX_SIG; i++) { 	/* init the signal handlers */
      sig_vector.handler [i] = SignalDefault;
    }
    mess.type = m_sighand;
    mess.m.sighand_m.handler = (word32) &__signal;
    call_system (&mess);
  }  
