MODULE CompositePart;

IMPORT CRandom, VarParams, AtomicPart, Document, GenParams, Connection,
       Globals, OO7, OO7Rep, Fmt, BaseAssemblyList, CompositePartList,
       PartIdSet;
FROM IO IMPORT Put, PutInt;
FROM Support IMPORT PrintOp;

REVEAL T = OO7.CompositePart BRANDED "CompositePart.T" OBJECT
OVERRIDES
  init := Init;
  traverse := Traverse;
  traverse7 := Traverse7;
END;

PROCEDURE Init (self: T; cpId: INTEGER): OO7.CompositePart =
  VAR
    to: INTEGER;
    atomicid: INTEGER;
    atomicParts := NEW(REF ARRAY OF OO7.AtomicPart,
                       VarParams.NumAtomicPerComp);
    cn: Connection.T;
  BEGIN
    IF Globals.debugMode THEN
      Put("CompositePart::CompositePart(cpId = "); PutInt(cpId); Put(")\n");
    END;

    (* initialize the simple stuff *)
    self.id := cpId;
    VAR typeNo := CRandom.random() MOD GenParams.NumTypes;
    BEGIN self.type := Globals.types[typeNo]; END;

    self.parts := atomicParts;

    (* for the build date, decide if this part is young or old, and then *)
    (* randomly choose a date in the required range *)
    
    IF cpId MOD GenParams.YoungCompFrac = 0 THEN
      (* young one *)
      IF Globals.debugMode THEN
        Put("(young composite part, id = "); PutInt(self.id); Put(".)\n");
      END;
      self.buildDate :=
          GenParams.MinYoungCompDate +
          (CRandom.random() MOD (GenParams.MaxYoungCompDate-GenParams.MinYoungCompDate+1));
    ELSE
      (* old one *)
      IF Globals.debugMode THEN
        Put("(old composite part, id = "); PutInt(self.id); Put(".)\n");
      END;
      self.buildDate :=
          GenParams.MinOldCompDate +
          (CRandom.random() MOD (GenParams.MaxOldCompDate-GenParams.MinOldCompDate+1));
    END;

    (* initialize the documentation (indexed by its title and id) ... *)
    self.documentation := NEW(Document.T).init(cpId, self);
    
    (* insert title into document index *)
    EVAL Globals.DocumentIdx.put(Fmt.F("Composite Part %08s", Fmt.Int(cpId)),
                                 self.documentation);

    (* insert id into document id index *)
    EVAL Globals.DocumentIdIdx.put(cpId, self.documentation);

    (* now create the atomic parts (indexed by their ids) ... *)
    FOR i := 0 TO VarParams.NumAtomicPerComp - 1   DO
      atomicid := Globals.nextAtomicId + i;

      (* create a new atomic part *)
      (* the AtomicPart constructor takes care of setting up *)
      (* the back pointer to the containing CompositePart() *)
      atomicParts[i] := NEW (AtomicPart.T).init(atomicid, self);
      
      (* stick the id of the part into the index *)
      EVAL Globals.AtomicPartIdx.put(atomicid, atomicParts[i]);
    
      (* first atomic part is the root part *)
      IF i = 0 THEN self.rootPart := atomicParts[i]; END;
    END;
    
    (* ... and then wire them semi-randomly together (as a ring plus random
       additional connections to ensure full part reachability for traversals)
    *)
    
    FOR from := 0 TO VarParams.NumAtomicPerComp - 1  DO
      FOR i := 0 TO VarParams.NumConnPerAtomic - 1 DO
        IF i = 0 THEN
          to := (from + 1) MOD VarParams.NumAtomicPerComp;
        ELSE
          to := CRandom.random() MOD  VarParams.NumAtomicPerComp;
        END;
        cn := NEW (Connection.T).init(atomicParts[from], atomicParts[to]);
      END
    END;
    INC(Globals.nextAtomicId, VarParams.NumAtomicPerComp);
    
    (* finally insert this composite part as a child of the base
       assemblies that use it *)
    
    (* first the assemblies using the comp part as a shared component *)
    
    (* get the first base assembly *)
    VAR
      ba := Globals.shared_cp[cpId];
    BEGIN
      WHILE ba # NIL DO
        (* add this assembly to the list of assemblies in which
           this composite part is used as a shared member *)
        self.usedInShar := BaseAssemblyList.Cons(ba.head, self.usedInShar);

        (* then add the composite part cp to the list of shared parts used
           in this assembly *)
        ba.head.componentsShar :=
            CompositePartList.Cons(self, ba.head.componentsShar);

        (* continue with next base assembly *)
        ba := ba.tail;
      END
    END;
    
    (* next the assemblies using the comp part as a private component *)
    
    (* get the first base assembly *)
    VAR
      ba := Globals.private_cp[cpId];
    BEGIN
      WHILE ba # NIL DO
        (* add this assembly to the list of assemblies in which
           this composite part is used as a shared member *)
        self.usedInPriv := BaseAssemblyList.Cons(ba.head, self.usedInPriv);

        (* then add the composite part cp to the list of shared parts used
           in this assembly *)
        ba.head.componentsPriv :=
            CompositePartList.Cons(self, ba.head.componentsPriv);

        (* continue with next base assembly *)
        ba := ba.tail;
      END
    END;
    RETURN self;
  END Init; 
  
PROCEDURE Traverse (self: T;  op: OO7.BenchmarkOp): INTEGER =
  BEGIN
    IF Globals.debugMode THEN
      Put("\t\t\tCompositePart::traverse(id = "); PutInt(self.id);
      Put(", op = "); PrintOp(op); Put(")\n");
    END;

    (* do DFS of the composite part's atomic part graph *)
    IF op >= OO7.BenchmarkOp.Trav1 AND op <= OO7.BenchmarkOp.Trav3c THEN
      (*  do parameterized DFS of atomic part graph *)
      VAR
        visitedIds := NEW(PartIdSet.T);
        result := self.rootPart.traverse(op, visitedIds);
      BEGIN
        visitedIds := visitedIds.dispose();
        RETURN result;
      END;
    ELSIF op = OO7.BenchmarkOp.Trav4 THEN
      (* search document text for a certain character *)
      RETURN self.documentation.searchText('I');
    ELSIF op = OO7.BenchmarkOp.Trav5do THEN
      (* conditionally change initial part of document text *)
      RETURN self.documentation.replaceText("I am", "This is");
    ELSIF op = OO7.BenchmarkOp.Trav5undo THEN
      (* conditionally change back initial part of document text *)
      RETURN self.documentation.replaceText("This is", "I am");
    ELSIF op = OO7.BenchmarkOp.Trav6 THEN
      (* visit the root part only (it knows how to handle this) *)
      VAR
        visitedIds := NEW(PartIdSet.T);
        result := self.rootPart.traverse(op, visitedIds);
      BEGIN
        visitedIds := visitedIds.dispose();
        RETURN result;
      END;
    ELSE 
      (* composite parts don't respond to other traversals *)
      Put("*** CompositePart::PANIC -- illegal traversal!!! ***\n");
      RETURN 0;
    END
  END Traverse;

PROCEDURE Traverse7 (self: T): INTEGER =
  VAR
    visitedBaseIds := NEW(PartIdSet.T);
    visitedComplexIds := NEW(PartIdSet.T);
    count := 0;
  BEGIN
    IF Globals.debugMode THEN
      Put("\tCompositePart::traverse7(id = "); PutInt(self.id); Put(")\n");
    END;

    (* search up the design hierarchy (along the private path) *)
    
    (* establish iterator of private base assemblies *)
    VAR
      ba := self.usedInPriv;
    BEGIN
      WHILE ba # NIL DO
        IF NOT visitedBaseIds.contains(ba.head.id) THEN
          visitedBaseIds.insert(ba.head.id);
          INC(count, ba.head.traverse7(visitedComplexIds));
        END;
        ba := ba.tail;
      END
    END;
    visitedBaseIds := visitedBaseIds.dispose();
    visitedComplexIds := visitedComplexIds.dispose();
    RETURN count;
  END Traverse7;
  
BEGIN
END CompositePart.
