UNSAFE MODULE Transaction;

IMPORT RTIO, RTDB, VirtualResource, Atom, AtomList, Txn;
FROM RTHeapDep IMPORT Page;
FROM Resource IMPORT resource, m;

REVEAL Private = <*TRANSIENT*> ROOT BRANDED "Transaction.Private" OBJECT END;
REVEAL T = Public BRANDED "Transaction.T" OBJECT
  open := FALSE;
OVERRIDES
  begin := Begin;
  commit := Commit;
  chain := Chain;
  abort := Abort;
  checkpoint := Checkpoint;
  isOpen := IsOpen;
  lock := Lock;
END;

EXCEPTION FatalError; <*FATAL FatalError*>
PROCEDURE fatalAtoms (t: AtomList.T) =
  BEGIN
    WHILE t # NIL DO
      RTIO.PutText(Atom.ToText(t.head)); RTIO.PutChar('\n');
      t := t.tail;
    END;
    RTIO.Flush();
    RAISE FatalError;
  END fatalAtoms;

PROCEDURE Begin(self: T)
  RAISES { InProgress, Disabled } =
  VAR outer: Txn.Level;
  BEGIN
    LOCK m DO
      IF self.open THEN RAISE InProgress; END;
      outer := resource.getTransactionLevel();
      IF outer # Txn.EnvelopeLevel THEN
        IF NOT RTDB.Flush() THEN RAISE Disabled END;
      END;
      TRY
        resource.beginTransaction();
      EXCEPT
      | VirtualResource.FatalError(t) => fatalAtoms(t);
      END;
      IF outer = Txn.EnvelopeLevel THEN
        RTDB.Invalidate();
      END;
      self.open := TRUE;
    END;
  END Begin;

PROCEDURE Commit(self: T)
  RAISES { NotInProgress, Disabled } =
  VAR outer: Txn.Level;
  BEGIN
    LOCK m DO
      IF NOT self.open THEN RAISE NotInProgress; END;
      IF NOT RTDB.Flush() THEN RAISE Disabled END;
      TRY
        resource.commitTransaction();
        outer := resource.getTransactionLevel();
      EXCEPT
      | VirtualResource.NotInTransaction => RAISE NotInProgress;
      | VirtualResource.FatalError(t) => fatalAtoms(t);
      END;
      IF outer = Txn.EnvelopeLevel THEN
        RTDB.Release();
      END;
      self.open := FALSE;
    END;
  END Commit;

PROCEDURE Chain(self: T)
  RAISES { NotInProgress, Disabled } =
  BEGIN
    LOCK m DO
      IF NOT self.open THEN RAISE NotInProgress; END;
      IF NOT RTDB.Flush() THEN RAISE Disabled END;
      TRY
        resource.chainTransaction();
      EXCEPT
      | VirtualResource.NotInTransaction => RAISE NotInProgress;
      | VirtualResource.FatalError(t) => fatalAtoms(t);
      END;
    END
  END Chain; 

PROCEDURE Abort(self: T) RAISES { NotInProgress } =
  VAR outer: Txn.Level;
  BEGIN
    LOCK m DO
      IF NOT self.open THEN RAISE NotInProgress; END;
      TRY
        resource.abortTransaction();
        outer := resource.getTransactionLevel();
      EXCEPT
      | VirtualResource.NotInTransaction => RAISE NotInProgress;
      | VirtualResource.FatalError(t) => fatalAtoms(t);
      END;
      IF outer = Txn.EnvelopeLevel THEN
        RTDB.Release();
      ELSE
        RTDB.Invalidate();
      END;
      self.open := FALSE;
    END;
  END Abort;

PROCEDURE Checkpoint(self: T)
  RAISES { NotInProgress, Disabled } =
  BEGIN
    LOCK m DO
      IF NOT self.open THEN RAISE NotInProgress; END;
      IF NOT RTDB.Flush() THEN RAISE Disabled END;
    END;
  END Checkpoint;

PROCEDURE IsOpen(self: T): BOOLEAN =
  BEGIN
    LOCK m DO
      RETURN self.open;
    END;
  END IsOpen;

PROCEDURE Lock(self: T; object: REFANY; mode: LockMode)
  RAISES { NotInProgress } =
  VAR
    db: RTDB.T;
    p: Page;
  BEGIN
    LOCK m DO
      IF NOT self.open THEN RAISE NotInProgress; END;
      IF RTDB.RefPageMap(object, db, p) THEN
        db.lock(p, mode);
      END;
    END;
  END Lock;

BEGIN
  m := NEW(MUTEX);
END Transaction.
