INTERFACE LogEvents;

(***************************************************************************)
(** Created by:  Roland Baumann						   *)

(** $Author: hosking $
    $Revision: 1.2 $
    $Date: 1999/04/28 15:12:38 $
    $Log: LogEvents.i3,v $
    Revision 1.2  1999/04/28 15:12:38  hosking
    Changes for PM3.

    Revision 1.1.1.1  1998/12/03 18:53:56  hosking
    Import of GRAS3-1.1

    Revision 1.1  1997/12/02 17:56:41  roland
    New event types and event contexts for user recovery operations
    introduced.

*)
(***************************************************************************)

IMPORT Event, Txn;
FROM EventType IMPORT Mismatch, Unknown;


TYPE
  T = Event.T;
    (** All LogEvents have the following attributes:
          poolName,
          pool      : the pool in which the event occurred its name.
          isPreEvent: Events can be signaled before (pre) or after (post)
                        the operation has been performed.
          level     : the transaction-level of the transaction
                        that signaled the event.
          graphNo,
          graph     : the graph in which the event occurred and its number.

        Additionally to this basic set of event attributes, different event
        types have different attributes.

        Checkpoint events have no additional attributes

        Undo/Redo events have no additional attributes

        RedoIth events
          SonNo     : the number of the son checkpoint to go to

        Backstep/Forstep events have no additional attributes

    *)


  Operation = {Checkpoint, Undo, Redo, RedoNext, RedoPrev, RedoIth,
               Backstep, Forstep};

CONST
  EventTypeName = ARRAY Operation OF
                    TEXT{"Checkpoint", "Undo", "Redo", "RedoNext",
                         "RedoPrev", "RedoIth", "Backstep", "Forstep"};
  (* LogEvents are declared in EventTypes.i3 with these names. *)

CONST
  (* Attribute names of the above event types *)
  LevelAttribute       = "Level";
  PoolNameAttribute    = "PoolName";
  PoolAttribute        = "Pool";
  IsPreEventAttribute  = "IsPreEvent";
  GraphNumberAttribute = "GraphNumber";
  GraphAttribute       = "Graph";

  SonNoAttribute = "SonNo";

PROCEDURE SignalCheckpoint (transUnit : CARDINAL;
                            poolName  : TEXT;
                            pool      : REFANY;
                            graphNo   : CARDINAL;
                            graph     : REFANY;
                            isPreEvent: BOOLEAN;
                            level     : Txn.Level);

PROCEDURE SignalUndo (transUnit : CARDINAL;
                      poolName  : TEXT;
                      pool      : REFANY;
                      graphNo   : CARDINAL;
                      graph     : REFANY;
                      isPreEvent: BOOLEAN;
                      level     : Txn.Level);


PROCEDURE SignalRedo (transUnit : CARDINAL;
                      poolName  : TEXT;
                      pool      : REFANY;
                      graphNo   : CARDINAL;
                      graph     : REFANY;
                      isPreEvent: BOOLEAN;
                      level     : Txn.Level);


PROCEDURE SignalRedoNext (transUnit : CARDINAL;
                          poolName  : TEXT;
                          pool      : REFANY;
                          graphNo   : CARDINAL;
                          graph     : REFANY;
                          isPreEvent: BOOLEAN;
                          level     : Txn.Level);


PROCEDURE SignalRedoPrev (transUnit : CARDINAL;
                          poolName  : TEXT;
                          pool      : REFANY;
                          graphNo   : CARDINAL;
                          graph     : REFANY;
                          isPreEvent: BOOLEAN;
                          level     : Txn.Level);


PROCEDURE SignalRedoIth (transUnit : CARDINAL;
                         poolName  : TEXT;
                         pool      : REFANY;
                         graphNo   : CARDINAL;
                         graph     : REFANY;
                         isPreEvent: BOOLEAN;
                         level     : Txn.Level;
                         son       : CARDINAL           );


PROCEDURE SignalBackstep (transUnit : CARDINAL;
                          poolName  : TEXT;
                          pool      : REFANY;
                          graphNo   : CARDINAL;
                          graph     : REFANY;
                          isPreEvent: BOOLEAN;
                          level     : Txn.Level);


PROCEDURE SignalForstep (transUnit : CARDINAL;
                         poolName  : TEXT;
                         pool      : REFANY;
                         graphNo   : CARDINAL;
                         graph     : REFANY;
                         isPreEvent: BOOLEAN;
                         level     : Txn.Level);


(* Queries for event attributes.*)
PROCEDURE GetOperation (ev: T): Operation RAISES {Unknown};
PROCEDURE GetPoolName (ev: T): TEXT RAISES {Mismatch, Unknown};
PROCEDURE GetPool (ev: T): REFANY RAISES {Mismatch, Unknown};
PROCEDURE GetGraphNo (ev: T): CARDINAL RAISES {Mismatch, Unknown};
PROCEDURE GetGraph (ev: T): REFANY RAISES {Mismatch, Unknown};
PROCEDURE GetIsPreEvent (ev: T): BOOLEAN RAISES {Mismatch, Unknown};
PROCEDURE GetLevel (ev: T): Txn.Level RAISES {Mismatch, Unknown};

(* RedoIth *)
PROCEDURE GetSonNo (ev: T): CARDINAL RAISES {Mismatch, Unknown};

END LogEvents.
