UNSAFE MODULE Shore;

IMPORT IP;
IMPORT TCP;
IMPORT ConnFD;
IMPORT Thread;
IMPORT Wr;
IMPORT Rd;
IMPORT IO;
IMPORT Transaction;
IMPORT TextF;
IMPORT Scan;
IMPORT RTParams;
IMPORT Process;
FROM RTHeapDep IMPORT PageData;
IMPORT Uin;
IMPORT FloatMode;
IMPORT Lex;
IMPORT RTDB;


VAR
  socket: ConnFD.T;

PROCEDURE getAll(VAR message: ARRAY OF CHAR)
  RAISES { Rd.Failure, ConnFD.TimedOut, Thread.Alerted } =
  VAR
    nread := 0;
    nleft := NUMBER(message);
    index := FIRST(message);
  BEGIN
    WHILE nleft > 0 DO
      nread := socket.get( SUBARRAY(message, index, nleft) );
      DEC(nleft, nread);
      INC(index, nread);
    END;
  END getAll; 

PROCEDURE getResponseXct(): RTDB.Id=
  VAR
    xct_id_tmp: INTEGER;
    xct_id: RTDB.Id;
  BEGIN
    TRY    
      (* receive response type *)
      getAll(LOOPHOLE(xct_id_tmp, ARRAY OF CHAR));      
      xct_id := Uin.ntohl(xct_id_tmp);
    EXCEPT
    | Rd.Failure, ConnFD.TimedOut, Thread.Alerted =>
      IO.Put("Database create - transmission error\n");
      Process.Exit();
    END;
    RETURN xct_id;
  END getResponseXct; 

PROCEDURE createReq(xct_id: RTDB.Id; database_name: TEXT) =
  VAR 
    ch: ARRAY [0..0] OF CHAR;
    xct_id_tmp: INTEGER;
  BEGIN
    TRY
      (* send message indicating type of operation *)
      ch[0] := VAL(ORD(RequestType.CREATE_R), CHAR);
      socket.put(ch);
      (* send transaction id *)
      xct_id_tmp := Uin.htonl(xct_id);
      socket.put(LOOPHOLE(xct_id_tmp, ARRAY OF CHAR));
      (* send length of database name and subsequently actual name *)
      ch[0] := VAL(NUMBER(database_name^), CHAR);
      socket.put(ch);
      socket.put(database_name^);
    EXCEPT
    | Wr.Failure, Thread.Alerted =>
      IO.Put("Database create - transmission error\n");
      Process.Exit();
    END;
  END createReq; 

PROCEDURE createResp(VAR db_id:RTDB.Id): ResponseType  =
  VAR 
    ch: ARRAY [0..0] OF CHAR;
    resp: ResponseType;
    db_id_tmp: RTDB.Id;
  BEGIN
    TRY
      (* receive response code *)
      getAll(ch);
      resp := VAL(ORD(ch[0]), ResponseType);
      IF resp = ResponseType.OK THEN
        (* receive transaction id *)
        getAll(LOOPHOLE(db_id_tmp, ARRAY OF CHAR));
        db_id := Uin.ntohl(db_id_tmp);
      END;
    EXCEPT
    | Rd.Failure, ConnFD.TimedOut, Thread.Alerted =>
      IO.Put("Database create - transmission error\n");
      Process.Exit();
    END;
    RETURN resp;
  END createResp; 

PROCEDURE openReq(xct_id: RTDB.Id; database_name: TEXT) =
  VAR 
    ch: ARRAY [0..0] OF CHAR;
    xct_id_tmp: INTEGER;
  BEGIN
    TRY
      (* send message indicating type of operation *)
      ch[0] := VAL(ORD(RequestType.OPEN_R), CHAR);
      socket.put(ch);
      (* send transaction id *)
      xct_id_tmp := Uin.htonl(xct_id);
      socket.put(LOOPHOLE(xct_id_tmp, ARRAY OF CHAR));
     (* send length of database name
         and subsequently actual name  *)
      ch[0] := VAL(NUMBER(database_name^), CHAR);
      socket.put(ch);
      socket.put(database_name^);
    EXCEPT
    | Wr.Failure, Thread.Alerted =>
      IO.Put("Database open - transmission error\n");
      Process.Exit();
    END;
  END openReq; 

PROCEDURE openResp(VAR db_id:RTDB.Id): ResponseType  =
  VAR
    ch: ARRAY [0..0] OF CHAR;
    resp: ResponseType;
    db_id_tmp: INTEGER;
  BEGIN
    TRY
      (* receive response code *)
      getAll(ch);      
      resp := VAL(ORD(ch[0]), ResponseType);
      IF resp = ResponseType.OK THEN
        (* receive transaction id *)
        getAll(LOOPHOLE(db_id_tmp, ARRAY OF CHAR));
        db_id := Uin.ntohl(db_id_tmp);
      END;
    EXCEPT
    | Rd.Failure, ConnFD.TimedOut, Thread.Alerted =>
      IO.Put("Database create - transmission error\n");
      Process.Exit();
    END;
    RETURN resp;
  END openResp;


PROCEDURE getPageReq(xct_id: RTDB.Id; 
                     db_id: RTDB.Id; 
                     p_id: RTDB.Id;
                     len: INTEGER;
                     type: RequestType) = 
  VAR
    ch: ARRAY [0..0] OF CHAR;
    xct_id_tmp: INTEGER;
    db_id_tmp: INTEGER;
    p_id_tmp: INTEGER;
    bytes := Uin.htonl(len);
  BEGIN
    TRY
      
      ch[0] := VAL(ORD(type), CHAR);
      (* send message indicating type of operation *)
      socket.put(ch);
      (* send transaction id *)
      xct_id_tmp := Uin.htonl(xct_id);
      socket.put(LOOPHOLE(xct_id_tmp, ARRAY OF CHAR));
      (* send database id *)
      db_id_tmp := Uin.htonl(db_id);
      socket.put(LOOPHOLE(db_id_tmp, ARRAY OF CHAR));
      (* send page id *)
      p_id_tmp := Uin.htonl(p_id);
      socket.put(LOOPHOLE(p_id_tmp, ARRAY OF CHAR));
      (* send page length *)
      socket.put(LOOPHOLE(bytes, ARRAY OF CHAR));
    EXCEPT
    | Wr.Failure, Thread.Alerted =>
      IO.Put("Get page request - transmission error\n");
      Process.Exit();
    END;
  END getPageReq;  


PROCEDURE getPageResp(VAR buffer: PageData): ResponseType =
  VAR 
    ch: ARRAY [0..0] OF CHAR;
    resp: ResponseType;
  BEGIN
    TRY
      (* receive response code *)
      getAll(ch);
      resp := VAL(ORD(ch[0]), ResponseType);      
      IF resp = ResponseType.OK  THEN
        (* receive page content *)       
        getAll(LOOPHOLE(buffer, ARRAY OF CHAR));
      END;
    EXCEPT
    | Rd.Failure, ConnFD.TimedOut, Thread.Alerted =>
      IO.Put("Get page response - transmission error\n");
      Process.Exit();
    END;
    RETURN resp;
  END getPageResp;  


PROCEDURE putPageReq(xct_id: RTDB.Id; db_id: RTDB.Id; p_id: RTDB.Id;
                     VAR data: PageData)=
  VAR
    ch: ARRAY [0..0] OF CHAR;
    xct_id_tmp: INTEGER;
    db_id_tmp: INTEGER;
    p_id_tmp: INTEGER;
    bytes := Uin.htonl(NUMBER(data) );
  BEGIN
    TRY
      (* send message indicating type of operation *)
      ch[0] := VAL(ORD(RequestType.WRITE_PAGE_R), CHAR);
      socket.put(ch);
      (* send transaction id *)
      xct_id_tmp := Uin.htonl(xct_id);
      socket.put(LOOPHOLE(xct_id_tmp, ARRAY OF CHAR));
      (* send database id *)
      db_id_tmp := Uin.htonl(db_id);
      socket.put(LOOPHOLE(db_id_tmp, ARRAY OF CHAR));
      (* send page id *)
      p_id_tmp := Uin.htonl(p_id);
      socket.put(LOOPHOLE(p_id_tmp, ARRAY OF CHAR));
      (* send page length *)
      socket.put(LOOPHOLE(bytes, ARRAY OF CHAR));
      (* send page content *)
      socket.put(LOOPHOLE(data, ARRAY OF CHAR));

    EXCEPT
    | Wr.Failure, Thread.Alerted =>
      IO.Put("Put page req - transmission error\n");
      Process.Exit();
    END;
  END putPageReq;

PROCEDURE putPageResp(): ResponseType =
  VAR 
    ch: ARRAY [0..0] OF CHAR;
    resp: ResponseType;
  BEGIN
    TRY
      (* receive response code *)
      getAll(ch);
      resp := VAL(ORD(ch[0]), ResponseType);
    EXCEPT
    | Rd.Failure, ConnFD.TimedOut, Thread.Alerted =>
      IO.Put("Put page - transmission error\n");
      Process.Exit();
    END;
    RETURN resp;
  END putPageResp;


PROCEDURE lockPageReq(xct_id: RTDB.Id; db_id: RTDB.Id; p_id: RTDB.Id;
                   mode: Transaction.LockMode) =
  VAR
    ch: ARRAY [0..0] OF CHAR;
    xct_id_tmp: INTEGER;
    db_id_tmp: INTEGER;
    p_id_tmp: INTEGER;
  BEGIN
    TRY
      (* send message indicating type of operation *)
      ch[0] := VAL(ORD(RequestType.LOCK_PAGE_R), CHAR);
      socket.put(ch);
      (* send transaction id *)
      xct_id_tmp := Uin.htonl(xct_id);
      socket.put(LOOPHOLE(xct_id_tmp, ARRAY OF CHAR));
      (* send database id *)
      db_id_tmp := Uin.htonl(db_id);
      socket.put(LOOPHOLE(db_id_tmp, ARRAY OF CHAR));
      (* send page id *)
      p_id_tmp := Uin.htonl(p_id);
      socket.put(LOOPHOLE(p_id_tmp, ARRAY OF CHAR));
      (* send lock type *)
      ch[0] := VAL(ORD(mode), CHAR);
      socket.put(ch);
    EXCEPT
    | Wr.Failure, Thread.Alerted =>
      IO.Put("Lock page request - transmission error\n");
      Process.Exit();
    END;
  END lockPageReq;


PROCEDURE lockPageResp(): ResponseType =
  VAR 
    ch: ARRAY [0..0] OF CHAR;
    resp: ResponseType;
  BEGIN
    TRY
      (* receive response code *)
      getAll(ch);
      resp := VAL(ORD(ch[0]), ResponseType);
    EXCEPT
    | Rd.Failure, ConnFD.TimedOut, Thread.Alerted =>
      IO.Put("Lock page response - transmission error\n");
      Process.Exit();
    END;
    RETURN resp;
  END lockPageResp;

PROCEDURE beginReq() =
  VAR
    ch: ARRAY [0..0] OF CHAR;
  BEGIN
    TRY
      (* send message indicating type of operation *)
      ch[0] := VAL(ORD(RequestType.BEGIN_R), CHAR);
      socket.put(ch);
    EXCEPT
    | Wr.Failure, Thread.Alerted =>
      IO.Put("Transaction begin request - transmission error\n");
      Process.Exit();
    END;
  END beginReq;

PROCEDURE beginResp(): ResponseType =
  VAR 
    ch: ARRAY [0..0] OF CHAR;
    resp: ResponseType;
  BEGIN
    TRY
      (* receive response code *)
      getAll(ch);
      resp := VAL(ORD(ch[0]), ResponseType);
    EXCEPT
    | Rd.Failure, ConnFD.TimedOut, Thread.Alerted =>
      IO.Put("Transaction begin response - transmission error\n");
      Process.Exit();
    END;
    RETURN resp;
  END beginResp;

PROCEDURE commitReq(xct_id: RTDB.Id) =
  VAR 
    ch: ARRAY [0..0] OF CHAR;
    xct_id_tmp: INTEGER;
  BEGIN
    TRY
      (* send message indicating type of operation *)
      ch[0] := VAL(ORD(RequestType.COMMIT_R), CHAR);
      socket.put(ch);
      (* send transaction id *)
      xct_id_tmp := Uin.htonl(xct_id);
      socket.put(LOOPHOLE(xct_id_tmp, ARRAY OF CHAR));
    EXCEPT
    | Wr.Failure, Thread.Alerted =>
      IO.Put("Transaction commit request - transmission error\n");
      Process.Exit();
    END;
  END commitReq;



PROCEDURE commitResp(): ResponseType =
  VAR 
    ch: ARRAY [0..0] OF CHAR;
    resp: ResponseType;
  BEGIN
    TRY
      (* receive response code *)
      getAll(ch);
      resp := VAL(ORD(ch[0]), ResponseType);
    EXCEPT
    | Rd.Failure, ConnFD.TimedOut, Thread.Alerted =>
      IO.Put("Transaction commit response - transmission error\n");
      Process.Exit();
    END;
    RETURN resp;
  END commitResp;



PROCEDURE chainReq(xct_id: RTDB.Id) =
  VAR 
    ch: ARRAY [0..0] OF CHAR;
    xct_id_tmp: INTEGER;
  BEGIN
    TRY
      (* send message indicating type of operation *)
      ch[0] := VAL(ORD(RequestType.CHAIN_R), CHAR);
      socket.put(ch);
      (* send transaction id *)
      xct_id_tmp := Uin.htonl(xct_id);
      socket.put(LOOPHOLE(xct_id_tmp, ARRAY OF CHAR));
    EXCEPT
    | Wr.Failure, Thread.Alerted =>
      IO.Put("Transaction chain request - transmission error\n");
      Process.Exit();
    END;
  END chainReq;


PROCEDURE chainResp(): ResponseType =
  VAR 
    ch: ARRAY [0..0] OF CHAR;
    resp: ResponseType;
  BEGIN
    TRY
      (* receive response code *)
      getAll(ch);
      resp := VAL(ORD(ch[0]), ResponseType);
    EXCEPT
    | Rd.Failure, ConnFD.TimedOut, Thread.Alerted =>
      IO.Put("Transaction chain response - transmission error\n");
      Process.Exit();
    END;
    RETURN resp;
  END chainResp;


PROCEDURE abortReq(xct_id: RTDB.Id) =
  VAR 
    ch: ARRAY [0..0] OF CHAR;
    xct_id_tmp: INTEGER;
  BEGIN
    TRY
      (* send message indicating type of operation *)
      ch[0] := VAL(ORD(RequestType.ABORT_R), CHAR);
      socket.put(ch);
      (* send transaction id *)
      xct_id_tmp := Uin.htonl(xct_id);
      socket.put(LOOPHOLE(xct_id_tmp, ARRAY OF CHAR));
    EXCEPT
    | Wr.Failure, Thread.Alerted =>
      IO.Put("Transaction abort request - transmission error\n");
      Process.Exit();
    END;
  END abortReq;


PROCEDURE abortResp(): ResponseType =
  VAR 
    ch: ARRAY [0..0] OF CHAR;
    resp: ResponseType;
  BEGIN
    TRY
      (* receive response code *)
      getAll(ch);
      resp := VAL(ORD(ch[0]), ResponseType);
    EXCEPT
    | Rd.Failure, ConnFD.TimedOut, Thread.Alerted =>
      IO.Put("Transaction abort response - transmission error\n");
      Process.Exit();
    END;
    RETURN resp;
  END abortResp;



VAR
  host: TEXT;
  port: INTEGER;
  ip_endpoint: IP.Endpoint;
BEGIN
  TRY
    host := IP.GetCanonicalByAddr(IP.GetHostAddr());
  EXCEPT
  | IP.Error =>
    IO.Put("Initialization failed\n");
    IO.Put("Unable to obtain local machine address\n");
    Process.Exit();
  END;

  port := 9000;
  IF RTParams.IsPresent("host") THEN
    host := RTParams.Value("host");
  END;
  IF RTParams.IsPresent("port") THEN
    TRY
      port := Scan.Int(RTParams.Value("port"));
    EXCEPT
    | FloatMode.Trap =>
      IO.Put("Initialization failed\n");
      IO.Put("Command line parameters incorrect\n");
      Process.Exit();
    | Lex.Error =>
      IO.Put("Initialization failed\n");
      IO.Put("Command line parameters incorrect\n");
      Process.Exit();
    END;
  END;

  TRY
    IF NOT IP.GetHostByName(host, ip_endpoint.addr) THEN
      IO.Put("Initialization failed\n");
      Process.Exit();
    END;
    ip_endpoint.port := port;
    socket := TCP.Connect(ip_endpoint);
  EXCEPT
  | IP.Error, Thread.Alerted =>
    IO.Put("Initialization failed\n");
    Process.Exit();
  END;
END Shore.
