(* 
 * Copyright 1997 University of Washington
 * All rights reserved.
 * See COPYRIGHT file for a full description
 *
 *
 * SecurityManager
 *    The primary public interface to dynamic security management;
 *    defines the core types of security management and lets
 *    the currently active thread discover and enforce security.
 *
 * HISTORY
 *
 * 03-Dec-97  Robert Grimm (rgrimm) at the University of Washington
 *      Added CheckSecurity.
 *
 * 30-Jul-97  Robert Grimm (rgrimm) at the University of Washington
 *      Created
 *
 * html
 *)

INTERFACE SecurityManager;

IMPORT SecurityError, AccessMode;

CONST
  Brand = "SecurityManager";

TYPE
  UID = [-16_7FFFFFFF-1..16_7FFFFFFF];
  GID = [-16_7FFFFFFF-1..16_7FFFFFFF];
  (*
     User and group identifiers.
       Used for auditing, accounting and discretionary access control.
       32 bits for Unix compatibility.
   *)

  SID = INTEGER;
  (*
     Security identifiers.
       Used for mandatory access control.
       Associated with both subjects and objects.
       The real currency of dynamic security management.
   *)

CONST
  TazUid      =  0; (* User id for Taz.                        *)
  AnonUid     = -1; (* User id for Anon.                       *)
  AllUsersGid =  0; (* Group id for the group with all users.  *)

  TazSid      =  0; (* Security id for Taz.                    *)
  TazDataSid  =  1; (* Security id for data generated by Taz.  *)
  AnonSid     = -1; (* Security id for Anon.                   *)
  AnonDataSid = -2; (* Security id for data generated by Anon. *)
  (*
     The system starts up under 'Taz',
     and 'Anon' is not trusted with anything.
   *)


(*
         Context and user/group id management
    (context management is on a per-session basis,
     as contexts are shared by threads spawned by
                  the initial thread)
   ------------------------------------------------
 *)

PROCEDURE GetCurrentProperty( name: TEXT; remove: BOOLEAN := FALSE) : REFANY;
  (*
     Get the property value associated with name,
     return NIL if no such property exists,
     destroy property if remove is TRUE.
   *)

PROCEDURE SetCurrentProperty( name: TEXT; property: REFANY ) : REFANY;
  (*
     Associate the specified property value with name.
     Return old value if already exists.
   *)

PROCEDURE GetCurrentUid() : UID;
  (* Get current user id *)

PROCEDURE GetGroups( uid : UID ) : REF ARRAY OF GID RAISES { SecurityError.T };
  (*
     Get groups a user is a member in;
     raise "SecurityError.EC.InvalidId" if invalid id.
   *)

PROCEDURE GetUserName ( uid : UID ) : TEXT RAISES { SecurityError.T };
  (*
     Turn "uid" into human-readable form;
     raise "SecurityError.EC.InvalidId" if invalid id.
   *)

PROCEDURE GetGroupName( gid : GID ) : TEXT RAISES { SecurityError.T };
  (*
     Turn "gid" into human-readable form;
     raise "SecurityError.EC.InvalidId" if invalid id.
   *)

(*
    Security ID Management
   ------------------------
 *)

PROCEDURE GetCurrentSubjectSid() : SID;
  (* Get current subject security id. *)

PROCEDURE GetCurrentObjectSid() : SID;
  (* Get current default object security id. *)

PROCEDURE SetCurrentObjectSid( sid : SID ) RAISES { SecurityError.T };
  (*
     Set default object security security id to "sid".
     Raise "SecurityError.EC.Unauthorized"
     if subject is unauthorized for this "sid".
     Note that the subject security id is implicitly managed
     by the security manager and can not be changed.
   *)

PROCEDURE GetSid( o : REFANY ) : SID RAISES { SecurityError.T };
  (*
     Get the security identifier associated with object "o".
     Raise "SecurityError.EC.InvalidType" if the objects
     type is insecure.
   *)
     
PROCEDURE GetSidName( sid : SID ) : TEXT RAISES { SecurityError.T };
  (*
     Turn "sid" into human-readable form;
     raise "SecurityError.EC.InvalidId" if invalid id.
   *)

(*
    Mediation
   -----------
 *)

PROCEDURE CheckPermissions( k2 : SID; m : AccessMode.T ) : BOOLEAN;
  (*
     Given the object SID "k2" and the access mode "m" required for an
     operation on that object, determine if the current subject is
     allowed to perform that operation. This interface is provided
     to perform an explicit check efficiently.
   *)

PROCEDURE CheckSimplePermissions( k2 : SID; s : AccessMode.SimpleT )
  : BOOLEAN;
  (*
     Given the object SID "k2" and the simple permissions "s" required
     for an operation on that object, determine if the current subject
     is allowed to perform that operation. This interface is provided
     to perform an explicit check efficiently.
   *)

PROCEDURE CheckObjectPermissions( obj : REFANY; m : AccessMode.T ) : BOOLEAN;
  (*
     Given the object "obj" and the access mode "m" required for an
     operation on that object, determine if the current subject is
     allowed to perform that operation. This interface is provided
     to perform an explicit check efficiently.
   *)

PROCEDURE CheckObjectSimplePermissions( obj : REFANY; s : AccessMode.SimpleT )
  : BOOLEAN;
  (*
     Given the object "obj" and the simple permissions "s" required
     for an operation on that object, determine if the current subject
     is allowed to perform that operation. This interface is provided
     to perform an explicit check efficiently.
   *)


(*
    State Information 
   -------------------
 *)

PROCEDURE CheckSecurity() : BOOLEAN;
  (*
     Returns "TRUE" iff security is activated.
   *)


END SecurityManager.
