/*  ZIP.C:  The ZIP Machine Emulator */

#include "zip.h"


extern char unifytable[];

int T0, T1, INTFLAG;
int *LORG, *LTOP, *GORG, *GTOP, *HORG, *HTOP, *TRORG, *TRTOP, *H;
int syswords[SYSWSIZE+1], modstk[LENMODSTK], *modptr;

int IR;
int BP, M1;
int *TR, *XC, *L, *BL, *G;
int *CL;


/* The Scratchpad (used by FUNCTOR and arithmetic instructions) */

#define LENPAD		5000
#define MAXPAD		&pad[LENPAD-1]

int pad[LENPAD];


/*
  In line unification of something in a register (a) with an atom (b).
  NOTE: corrupts register IQ because of in_trail.  Callers use IP.
*/

#define in_unatomic(a,b)\
  deref(a);\
  if (tag(a) == UNDEF)\
  {\
    memoff0(a) = b;\
    in_trail(a);\
  }\
  else if (a != b) goto Fail;


/*
  Set up the argument of a metacall.
  Note that lookups in the heap (findfunct, findproc) are used here.
  M0 is set to either a PROC, or a term having an unknown procedure.
  NOTE:  corrupts R0, R1, R2, M0.
*/

#define metaproc \
{\
  M0 = R0; \
  R1 = 0; \
  if (tag(R0) == ATOM) \
  { \
    R2 = findfunct(R0,0,FALSE); \
    if (tag(R2) != TERMIN) M0 = findproc(R2,*modptr,FALSE); \
  } \
  else if (tag(R0) == TERM) \
  { \
    R2 = termfunctor(R0); \
    M0 = findproc(R2,*modptr,FALSE); \
    R1 = termarity(R0); \
  } \
  if (tag(M0) == TERMIN) M0 = R2;\
  if (tag(M0) == TERMIN) M0 = R0; \
}

/*
   Push a Trail entry, but only if necessary.
   NOTE: corrupts register IQ.  Callers use IP.
*/

#define in_trail(w) \
  IQ = (int *) val(w); \
  if ( ((IQ > LORG) && (IQ < BL)) || (IQ < (int *) *(BL + GOFF)) ) \
    { *TR++ = w; if (TR > TRTOP) fatality(68);} 

/*
  tidy trail.  Used by cut and callx('!').
  NOTE:  corrupts  IP, IQ R0.
*/

#define tidytrail \
  if (TR1 < TR) \
  { \
    register int *v, *bg; \
    IP = IQ = TR1; \
    bg = (int *) *(BL + GOFF);\
    while (IP < TR) \
    { \
      R0 = *IP; \
      v = (int *) val(R0); \
      if ( (tag(R0) == CLAUSE) || ((v > LORG) && (v < BL)) || (v < bg) ) \
      { \
        if (IP != IQ) *IQ = R0; \
        IQ++; \
      } \
      IP++; \
    } \
    TR = IQ; \
  } 


/*
  Pop trail down to dest, resetting entries as we go.
  Take action on trailed clause references.
  Notice that untrailed clauses referred to by M0, M1 are retrailed.
  NOTE:  Corrupts R0, R1, R2, IP, IQ.
*/

#define untrail \
  if (TR > TR1) \
  { \
    IP = TR; \
    R1 = 0;\
    while (IP > TR1) \
    { \
      R0 = *--IP; \
      if (tag(R0) == UNDEF) mem(val(R0)) = R0; \
      else \
      { \
        proctrap((tag(R0) != CLAUSE), 50); \
        IQ = (int *) val(R0) + CLAFLAOFF; \
        R2 = *IQ; \
        *IQ = R2 & ~CFCLAIMED; \
/*\
prclause(" unclaimed",R0);\
*/\
        if ((R2 & CFDOOMED) && !(R2 & CFREFMASK))\
        {\
          if (R0 == M0 || R0 == M1)\
          {\
            R1 = R0;\
            *IQ |= CFCLAIMED;\
          }\
          else dealloc(R0);\
        }\
      } \
    } \
    TR = IP; \
    if (R1) *TR++ = R1;\
  } 


/*
  Claim the current clause and its lookahead by pushing on the Trail.
  NOTE:  Corrupts IP.
*/

#define claimclause	\
{\
  IP = XC + CLAFLAOFF;\
  if (!(*IP & CFCLAIMED))\
  {\
/*\
prclause(" claim: ",(int) XC);\
*/\
    *IP |= CFCLAIMED;\
    *TR++ = makeword(CLAUSE, (int) XC);\
    if (TR > TRTOP) fatality(68);\
  }\
  if (tag(M0) != TERMIN)\
  {\
    IP = (int *) (val(M0)) + CLAFLAOFF;\
    if (!(*IP & CFCLAIMED))\
    {\
  /*\
  prclause(" claimahead: ",M0);\
  */\
      *IP |= CFCLAIMED; \
      *TR++ = M0;\
      if (TR > TRTOP) fatality(68);\
    }\
  }\
}\

/*
  Possible failure because of (a) unknown procedure, or (b) procedure with
  no clauses.  If merely an empty database key, then Fail.  Otherwise,
  call the unknown_handler after trying to find a culprit name.
*/

#define unknown \
{ \
  if (tag(M0) != ATOM) \
  { \
    if (tag(M0) == PROC) R0 = memoff(M0,PROFUNOFF); else R0 = M0; \
    if (R0 == syswords[SWDATABASE] || R0 == syswords[SWSOURCE]) goto Fail; \
    R0 = memoff(R0,FUNATOFF); \
  } \
  *(CL+A1OFF) = R0;\
  M0 = syswords[SWUNKNOWN];\
  goto Arrive;\
} \


/*
  Use indexing information to update Match register M0 to point to the
  next clause.  Does the least possible amount of work based on the
  situation at hand, even if this means duplicating a few lines of code.
  The register declaration is optional.
  Possibly the only goto's in the whole system.
*/

#define nextmatch(found,none,next)	\
{\
  if (tag(M0) == TERMIN) goto none; /* no matching clause */\
  R1 = memoff(M0,CLAKEYOFF);\
  if (tag(R1) == UNDEF) goto found; /* clause is not indexed */\
  R0 = *(CL+A1OFF); /* Grab the argument */\
  deref(R0);\
  if (tag(R0) == UNDEF) goto found; /* arg will match anything */\
  if (tag(R0) == TERM) R0 = termfunctor(R0); \
next:\
  switch(tag(R1))\
  {\
    case FUNCTOR:\
    case ATOM:\
    case INT:\
    case CLAUSE:\
      if (R0 == R1) goto found;\
      break;\
    case CONS:\
    case BOX:\
      if (tag(R0) == tag(R1)) goto found; /* match any box */\
      break;\
    default:\
      processortrap(2);\
  }\
    M0 = memoff(M0,CLAFOROFF);\
    if (tag(M0) == TERMIN) goto none;\
    R1 = memoff(M0,CLAKEYOFF);\
    if (tag(R1) == UNDEF) goto found;\
    goto next;\
}


/* a macro for blkcpy */

#ifdef os370
#define blkcpy(a,b,n) memcpy(a,b,sizeof(int)*n)
#else
#ifdef BERK41
#define MOVC3 92
#define blkcpy(to,from,count) asm(MOVC3,from,to,sizeof(int)*count)
#else
#define blkcpy(a,b,n) R0=(n); IP=(b); IQ=(a); while (R0--) *IQ++ = *IP++
#endif
#endif

/* The ZIP Emulator */

interpret()
{
  register char *PC;
  register int R0, R1, R2;      /* microcode temporaries */
  register int *IP, *IQ;    /* microcode temporaries */
  register int PM, M0;
  register int *D;
  register int *TR1, *SP;
  G = GORG;
  L = LORG;
  TR = TRORG;
  modstk[0] = syswords[SWUSER];
  modptr = modstk;
  M0 = syswords[SWSTART];
  CL = L;
  *(CL + CLOFF) = NULL;
  BL = NULL;
Arrive:
  ++T1;
  if (T0) trace(T0 & TMARMESS,"arrive",M0,T0 & TMARRIVE, T0 & TMARWAIT,PC);
  if (tag(M0) != PROC) unknown;
  R0 = memoff(M0,PROCLAOFF);
  if (tag(R0) == INT)
  {
    if (primhandle(val(R0)))
    {
      PC = (char *) ((int *) val(*(CL + CPOFF)));
      XC = (int *) val(*(CL+XCOFF));
      CL = (int *) val(*(CL + CLOFF));
      D = L + ACTSIZE;
      PM = COPY;
      goto Continue;
    }
    else goto Fail;
  }
  R0 = memoff(R0,FIXFIROFF);
  if (tag(R0) == TERMIN) unknown;
  M0 = R0;
  nextmatch(L2,Fail,L4)
L2:
  M1 = M0;
  XC = (int *) val(M0);
  checklocal(10+(*(XC+CLAFLAOFF) & CFSIZE));
  M0 = *(XC+CLAFOROFF);
  nextmatch(L6,L7,L8)
L6:
    *(CL + BPOFF) = BP;
    *(CL + BLOFF) = (int) BL;
    *(CL + TROFF) = (int) TR;
    *(CL + GOFF) = (int) G;
    BP = M0;
    BL = CL;
L7:
Setup:
  PC = (char *) ((int *) val(*(XC + CLATEXOFF)) + 1);
  PM = MATCH;
  D = L + ACTSIZE;
  SP = pad;
Continue:
  if (T0) trace(T0 & TMSIMESS,NULL,NULL,T0 & TMSINGLE,T0 & TMSIWAIT,PC);
  switch(*PC++ | PM)
  {
    default:
      processortrap(18);
    case 0|MATCH:
    case 0|COPY:
      goto Continue;    /* ORION only.  Simulates word fetch. */
    case IFAIL|MATCH:
    case IFAIL|COPY:
      goto Fail;
    case IPOP|MATCH:
    case IPOP|COPY:
      PM = *SP--;
      D = (int *) *SP--;
      goto Continue;
    case POPARG|COPY:
      SP--;
      D = (int *) *SP--;
      goto Continue;
    case INITVAR|COPY:
      IP = CL + *PC++;
      *IP = makeword(UNDEF,(int) IP);
      goto Continue;
    case IVAR|MATCH:
      R0 = *D++;
      R1 = *(CL + *PC++);
      goto Unify;
    case IVAR|COPY:
      R0 = *(CL + *PC++);
      deref(R0);
      if (tag(R0) == UNDEF)
      {
        if ((int *) val(R0) < D) *D = makeword(LINK,val(R0));
        else
        {
          *D = makeword(UNDEF,(int) D);
          memoff0(R0) = makeword(LINK,(int) D);
          in_trail(R0);
        }
      }
      else *D = R0;
      D++;
      goto Continue;
    case VARARG|COPY:
VarArg:
      R0 = *(CL + *PC++);
      *D++ = (tag(R0) == UNDEF) ? makeword(LINK,val(R0)) : R0;
      goto Continue;
    case FIRSTVAR|MATCH:
      R0 = *D++;
      deref(R0);
      *(CL + *PC++) = (tag(R0) == UNDEF) ? makeword(LINK,val(R0)) : R0;
      goto Continue;
    case FIRSTVAR|COPY:
      *(CL + *PC++) = makeword(LINK, (int) D);
      *D = makeword(UNDEF,(int) D);
      ++D;
      goto Continue;
    case FIRVARARG|COPY:
FirVarArg:
      IP = CL + *PC++;
      *IP = makeword(UNDEF,(int) IP);
      *D++ = makeword(LINK,(int) IP);
      goto Continue;
    case GLOFIRVAR|COPY:
      if (BL >= CL) goto FirVarArg;
      *(CL + *PC++) = *D++ = makeword(LINK,(int) G);
      *G = makeword(UNDEF,(int) G);
      ++G;
      goto Continue;
    case GLOVAR|COPY:
      if (BL >= CL) goto VarArg;
      R0 = *(CL + *PC++);
      deref(R0);
      if (tag(R0) == UNDEF)
      {
        if ((int *) val(R0) >= CL)
        {
          memoff0(R0) = *D++ = makeword(LINK,(int) G);
          *G = makeword(UNDEF,(int) G);
          ++G;
        }
        else *D++ = makeword(LINK,val(R0));
      }
      else *D++ = R0;
      goto Continue;
    case VOIDN|COPY:
      R0 = *PC++;
      while (R0--) {*D = makeword(UNDEF,(int) D); ++D;}
      goto Continue;
    case VOID|COPY:
      *D = makeword(UNDEF,(int) D);
      ++D;
      goto Continue;
    case VOID|MATCH:
      D++;
      goto Continue;
    case VOIDN|MATCH:
      D += *PC++;
      goto Continue;
    case IFUNCTOR|MATCH:
      R0 = *D++;
      deref(R0);
      if (tag(R0) == UNDEF)
      {
        in_trail(R0);
        memoff0(R0) = makeword(TERM,(int)G);
        *G++ = *(XC + CLAUSELEN + *PC++);
        *++SP = (int) D;
        *++SP = PM;
        D = G;
        G += (*PC++);
        checkglobal(10);
        PM = COPY;
      }
      else
      {
        if (tag(R0) != TERM) goto Fail;
        if (termfunctor(R0) != *(XC + CLAUSELEN + *PC++)) goto Fail;
        PC++;
        *++SP = (int) D;
        *++SP = PM;
        D = (int *) val(R0) + 1;
      }
      goto Continue;
    case CONSLIST|MATCH:
      R0 = *D++;
      deref(R0);
      if (tag(R0) == UNDEF)
      {
        checkglobal(12);
        in_trail(R0);
        memoff0(R0) = makeword(CONS,(int)G);
        *++SP = (int) D;
        *++SP = PM;
        D = G;
        G += 2;
        PM = COPY;
      }
      else
      {
        if (tag(R0) != CONS) goto Fail;
        *++SP = (int) D;
        *++SP = PM;
        D = (int *) val(R0);
      }
      goto Continue;
    case IFUNCTOR|COPY:
      *D++ = makeword(TERM,(int)G);
      *G++ = *(XC + CLAUSELEN + *PC++);
      *++SP = (int) D;
      *++SP = PM;
      D = G;
      G += *PC++;
      checkglobal(10);
      goto Continue;
    case CONSLIST|COPY:
      checkglobal(12);
      *D++ = makeword(CONS,(int)G);
      *++SP = (int) D;
      *++SP = PM;
      D = G;
      G += 2;
      goto Continue;
    case LASTFUNCTOR|MATCH:
      R0 = *D;
      deref(R0);
      if (tag(R0) == UNDEF)
      {
        in_trail(R0);
        memoff0(R0) = makeword(TERM,(int) G);
        *G++ = *(XC + CLAUSELEN + *PC++);
        D = G;
        G += *PC++;
        checkglobal(10);
        PM = COPY;
      }
      else
      {
        if (tag(R0) != TERM) goto Fail;
        if (termfunctor(R0) != *(XC + CLAUSELEN + *PC++)) goto Fail;
        PC++;
        D = (int *) val(R0) + 1;
      }
      goto Continue;
    case LASTCONSLIST|MATCH:
      R0 = *D;
      deref(R0);
      if (tag(R0) == UNDEF)
      {
        checkglobal(12);
        in_trail(R0);
        memoff0(R0) = makeword(CONS,(int) G);
        D = G;
        G += 2;
        PM = COPY;
      }
      else
      {
        if (tag(R0) != CONS) goto Fail;
        D = (int *) val(R0);
      }
      goto Continue;
    case LASTFUNCTOR|COPY:
      *D = makeword(TERM,(int)G);
      *G++ = *(XC + CLAUSELEN + *PC++);
      D = G;
      G += *PC++;
      checkglobal(10);
      goto Continue;
    case LASTCONSLIST|COPY:
      checkglobal(12);
      *D = makeword(CONS,(int)G);
      D = G;
      G += 2;
      goto Continue;
    case IMMED|MATCH:
      R1 = makeword(INT,(*PC++ & 0XFF));
      goto ConstMatch;
    case CONSTNIL|MATCH:
      R1 = syswords[SWNIL];
      goto ConstMatch;
    case CONSTANT|MATCH:
      R1 = *(XC + CLAUSELEN + *PC++);
ConstMatch:
      R0 = *D++;
      deref(R0);
      if (tag(R0) == UNDEF)
      {
        in_trail(R0);
        memoff0(R0) = R1;
      }
      else if (R0 != R1) goto Fail;
      goto Continue;
    case IMMED|COPY:
      *D++ = makeword(INT,(*PC++ & 0XFF));
      goto Continue;
    case CONSTANT|COPY:
      *D++ = *(XC + CLAUSELEN + *PC++);
      goto Continue;
    case CONSTNIL|COPY:
      *D++ = syswords[SWNIL];
      goto Continue;
    case ENTER|MATCH:
      PM = COPY;
    case ENTER|COPY:
      claimclause;
      L += *PC++;
      D = L + ACTSIZE;
      if (INTFLAG)
      {
        INTFLAG = 0;
        M0 = syswords[SWBREAK];
        goto Call;
      }
      goto Continue;
    case CUT|MATCH:
      PM = COPY;
    case CUT|COPY:
      if (BL >= CL)
      {
        L = CL + *PC++;
        while (CL <= (IP = (int *) *(BL+BLOFF))) BL = IP;
LocalCut:
        TR1 = (int *) *(BL + TROFF);
        BP = *(BL + BPOFF);
        BL = (int *) *(BL + BLOFF);
        tidytrail;
        D = L + ACTSIZE;
      }
      else PC++;
      goto Continue;
    case LOCALCUT|MATCH:
      PM = COPY;
    case LOCALCUT|COPY:
      IQ = (int *) *(CL + *PC++);
      if (BL >= IQ)
      {
        while (IQ <= (IP = (int *) *(BL+BLOFF))) BL = IP;
        L = BL;
      }
      goto LocalCut;
    case ICONTINUE|COPY:
      PC = PC + (*PC & 0XFF) + 1;
      goto Continue;
    case SAVEL|COPY:
      *(CL + *PC++) = (int) L;
      goto Continue;
    case DISJUNCT|COPY:
      *(L + CLOFF) = (int) CL;
      *(L + BLOFF) = (int) BL;
      *(L + BPOFF) = BP;
      *(L + GOFF) = (int) G;
      *(L + TROFF) = (int) TR;
      *(L + XCOFF) = (int) XC;
      BL = L;
      /* fall through case for ORELSE */
      BP = makeword(INT,(int) (PC + (*PC & 0XFF) + 1));
      ++PC;
      L += ACTSIZE;
      D = L + ACTSIZE;
      goto Continue;
    case ENDOR|MATCH:
      PM = COPY;
    case ENDOR|COPY:
      BP = *(L + BPOFF);
      BL = (int *) *(L + BLOFF);
      D = L + ACTSIZE;
      goto Continue;
    case CALLX|COPY:
      R0 = *(CL + *PC++);
      deref(R0);
      metaproc;
      checkglobal(R1+10);
      if (tag(M0) == PROC && tag(R0) == TERM)
      {
        IP = (int *) val(R0);
        while (R1--)
        {
          R0 = *++IP;
          *D++ = (tag(R0) == UNDEF) ? makeword(LINK, (int) IP) : R0;
        }
      }
      goto Call;
    case DEPART|COPY:
      M0 = *(XC + CLAUSELEN + *PC++);
      if (BL >= CL)
      {
        PC++;
        *(L + XCOFF) = *(CL + XCOFF);
        *(L + CPOFF) = *(CL + CPOFF);
        *(L + CLOFF) = *(CL + CLOFF);
        CL = L;
        goto Arrive;
      }
      blkcpy(CL+ACTSIZE, L+ACTSIZE, *PC++);   /* may corrupt R0,IP,IQ */
      L = CL;
      goto Arrive;
    case CALL|COPY:
      M0 = *(XC + CLAUSELEN + *PC++);
Call:
      *(L + CPOFF) = (int) PC;
      *(L + XCOFF) = (int) XC;
      *(L + CLOFF) = (int) CL;
      CL = L;
      goto Arrive;
    case IEXIT|COPY:
      if (!(BL >= CL))
      {
        L = CL;
        D = L + ACTSIZE;
      }
      goto Return;
    case IRETURN|MATCH:
      PM = COPY;
    case IRETURN|COPY:
      claimclause;
      if (BL >= CL)  L += *PC;
      ++PC;
      D = L + ACTSIZE;
Return:
      XC = (int *) val(*(CL + XCOFF));
      PC = (char *) ((int *) val(*(CL + CPOFF)));
      CL = (int *) val(*(CL + CLOFF));
      goto Continue;
    case PROVAR|MATCH:
    case PROVAR|COPY:
      ++T1;
      R0 = *(CL + *PC++);
      deref(R0);
      if (tag(R0) != UNDEF) goto Fail;
      goto Continue;
    case PRONONVAR|MATCH:
    case PRONONVAR|COPY:
      ++T1;
      R0 = *(CL + *PC++);
      deref(R0);
      if (tag(R0) == UNDEF) goto Fail;
      goto Continue;
    case PROATOM|MATCH:
    case PROATOM|COPY:
      ++T1;
      R0 = *(CL + *PC++);
      deref(R0);
      if (tag(R0) != ATOM) goto Fail;
      goto Continue;
    case PROINT|MATCH:
    case PROINT|COPY:
      ++T1;
      R0 = *(CL + *PC++);
      deref(R0);
      if (tag(R0) != INT) goto Fail;
      goto Continue;
    case PROAIC|MATCH:
    case PROAIC|COPY:
      ++T1;
      R0 = *(CL + *PC++);
      deref(R0);
      if ((tag(R0) == UNDEF) || (tag(R0) == TERM) || (tag(R0) == CONS)) goto Fail;
      goto Continue;
    case PROARG|MATCH:
      PM = COPY;
    case PROARG|COPY:
      ++T1;
      D = L + ACTSIZE;
      R0 = *D;
      deref(R0);
      if (tag(R0) != INT) goto Fail;
      R0 = signextend(R0);
      if (R0 <= 0) goto Fail;
      R1 = *(D+1);
      deref(R1);
      if (tag(R1) == CONS)
      {
        if (R0 > 2) goto Fail;
        --R0;
      }
      else if ((tag(R1) != TERM) || (R0 > termarity(R1))) goto Fail;
      R1 = memoff(R1,R0);
      R0 = *(D+2);
      goto Unify;
    case PROFUNCTOR|MATCH:
      PM = COPY;
    case PROFUNCTOR|COPY:
      ++T1;
      D = L + ACTSIZE;
      R0 = *D;
      deref(R0);
      switch (tag(R0))
      {
        default:
          goto Fail;
        case ATOM:
        case INT:
          R1 = *(D+1);
          in_unatomic(R1,R0);
          R1 = *(D+2);
          in_unatomic(R1,makeword(INT,0));
          break;
        case CONS:
          R1 = *(D+1);
          in_unatomic(R1,syswords[SWDOT]);
          R1 = *(D+2);
          in_unatomic(R1,makeword(INT,2));
          break;
        case TERM:
          R0 = val(termfunctor(R0));
          R1 = *(D+1);
          in_unatomic(R1,mem(R0+FUNATOFF));
          R1 = *(D+2);
          in_unatomic(R1,mem(R0+FUNAROFF));
          break;
        case UNDEF:
          R0 = *(D+1);
          deref(R0);
          if (tag(R0) == UNDEF) goto Fail;
          R1 = *(D+2);
          deref(R1);
          if (tag(R1) != INT) goto Fail;
          R1 = val(R1);
          if (R1 == 0)
          {
            R1 = *D;
            in_unatomic(R1,R0);
          }
          else
          {
            checkglobal(R1+10);
            if (R0 == syswords[SWDOT] && R1 == 2)
            {
              R0 = makeword(CONS, (int) G);
              *G = makeword(UNDEF, (int) G);
              ++G;
              *G = makeword(UNDEF, (int) G);
              ++G;
              R1 = *D;
              goto Unify;
            }
            *G = findfunct(R0,R1,TRUE);
            R0 = makeword(TERM, (int) G);
            G++;
            while (R1--) {*G = makeword(UNDEF,(int) G); ++G;}
            R1 = *D;
            goto Unify;
          }
    }
    goto Continue;
  case PROSUCC|MATCH:
    PM = COPY;
  case PROSUCC|COPY:
    ++T1;
      D = L + ACTSIZE;
      R0 = *D;
      deref(R0);
      if (tag(R0) == INT)
      {
        R1 = *(D+1);
        in_unatomic(R1,makeword(INT,val(signextend(R0)+1)));
      }
      else
      {
        R0 = *(D+1);
        deref(R0);
        if (tag(R0) != INT) goto Fail;
        R1 = *D;
        in_unatomic(R1,makeword(INT,val(signextend(R0)-1)));
      }
    goto Continue;
  case PROEQUAL|MATCH:
    PM = COPY;
  case PROEQUAL|COPY:
    ++T1;
    D = L + ACTSIZE;
    R0 = *D;
    R1 = *(D+1);
    goto Unify;
  case EVAL|COPY:
  case EVAL|MATCH:
    R0 = *(CL + *PC++);
    deref(R0);
    IP = CL + *PC++;
    if (tag(R0) == INT) *IP = R0;
    else
    {
      M0 = syswords[SWISPROC];
      *IP = R1 = makeword(UNDEF, (int) IP);
      *D++ = (tag(R1) == UNDEF) ? makeword(LINK, (int) val(R1)) : R1;
      *D++ = (tag(R0) == UNDEF) ? makeword(LINK, (int) val(R0)) : R0;
      goto Call;
    }
    goto Continue;
  case PUSHB|COPY:
  case PUSHB|MATCH:
    *++SP = *PC++ & 0XFF;
    goto Continue;
  case PUSHI|COPY:
  case PUSHI|MATCH:
    R0 = *(XC + CLAUSELEN + *PC++);
    *++SP = signextend(R0);
    goto Continue;
  case PUSHV|COPY:
  case PUSHV|MATCH:
    R0 = *(CL + *PC++);
    deref(R0);
    *++SP = signextend(R0);
    goto Continue;
  case RESULT|COPY:
  case RESULT|MATCH:
    ++T1;
    R1 = *(CL + *PC++);
    deref(R1);
    if (tag(R1) == UNDEF)
    {
      in_trail(R1);
      memoff0(R1) = makeword(INT,val(*SP--));
    }
    else if (makeword(INT,val(*SP--)) != R1) goto Fail;
    goto Continue;
  case FIRRESULT|MATCH:
  case FIRRESULT|COPY:
    ++T1;
    *(CL + *PC++) = makeword(INT,val(*SP--));
    goto Continue;
  case ISADD|COPY:
  case ISADD|MATCH:
    --SP;
    *SP += *(SP+1);
    goto Continue;
  case ISSUB|COPY:
  case ISSUB|MATCH:
    --SP;
    *SP -= *(SP+1);
    goto Continue;
  case ISMUL|COPY:
  case ISMUL|MATCH:
    --SP;
    *SP *= *(SP+1);
    goto Continue;
  case ISDIV|COPY:
  case ISDIV|MATCH:
    if (!(*SP)) goto Fail;
    --SP;
    *SP /= *(SP+1);
    goto Continue;
  case ISMOD|COPY:
  case ISMOD|MATCH:
    if (!(*SP)) goto Fail;
    --SP;
    *SP %= *(SP+1);
    goto Continue;
  case ISSHR|COPY:
  case ISSHR|MATCH:
    --SP;
    *SP >>= *(SP+1);
    goto Continue;
  case ISSHL|COPY:
  case ISSHL|MATCH:
    --SP;
    *SP <<= *(SP+1);
    goto Continue;
  case ISAND|COPY:
  case ISAND|MATCH:
    --SP;
    *SP &= *(SP+1);
    goto Continue;
  case ISOR|COPY:
  case ISOR|MATCH:
    --SP;
    *SP |= *(SP+1);
    goto Continue;
  case ISNOT|COPY:
  case ISNOT|MATCH:
    *SP = ~(*SP);
    goto Continue;
  case ISNEG|COPY:
  case ISNEG|MATCH:
    *SP = -(*SP);
    goto Continue;
  case ISEQ|COPY:
  case ISEQ|MATCH:
    ++T1;
    if (*SP != *(SP-1)) goto Fail;
    SP -= 2;
    goto Continue;
  case ISNE|COPY:
  case ISNE|MATCH:
    ++T1;
    if (*SP == *(SP-1)) goto Fail;
    SP -= 2;
    goto Continue;
  case ISLT|COPY:
  case ISLT|MATCH:
    ++T1;
    if (*(SP-1) >= *SP) goto Fail;
    SP -= 2;
    goto Continue;
  case ISLE|COPY:
  case ISLE|MATCH:
    ++T1;
    if (*(SP-1) > *SP) goto Fail;
    SP -= 2;
    goto Continue;
  case ISGT|COPY:
  case ISGT|MATCH:
    ++T1;
    if (*(SP-1) <= *SP) goto Fail;
    SP -= 2;
    goto Continue;
  case ISGE|COPY:
  case ISGE|MATCH:
    ++T1;
    if (*(SP-1) < *SP) goto Fail;
    SP -= 2;
    goto Continue;
  }
Fail:
  if (T0) trace(T0 & TMFAMESS,"fail",BP,T0 & TMFAILS, T0 & TMFAWAIT,PC);
  G = (int *) *(BL + GOFF);
  TR1 = (int *) *(BL + TROFF);
  if (BL < L)
  {
    L = BL;
    if (tag(BP) == CLAUSE) CL = L;
    else
    {
      CL = (int *) *(L + CLOFF);
      XC = (int *) *(L + XCOFF);
      PC = (char *) ((int *) val(BP));
      untrail;
      goto Continue;
    }
  }
  M1 = BP;
  XC = (int *) val(BP);
  checklocal(10+(*(XC+CLAFLAOFF) & CFSIZE));
  M0 = *(XC+CLAFOROFF);
  untrail;
  nextmatch(LA,LB,LC)
LA:
  BP = M0;
  goto Setup;
LB:
  BP = *(CL + BPOFF);
  BL = (int *) *(CL + BLOFF);
  goto Setup;
Unify:
  deref(R0);
  deref(R1);
  if (R0 == R1) goto Continue;
  if (tag(R0) == UNDEF)
  {
    if (tag(R1) == UNDEF)
    {
      if (val(R1) < val(R0)) { R1 = makeword(LINK,val(R1)); goto R0var;}
      else { R0 = makeword(LINK,val(R0)); goto R1var;}
    }
    else
    {
R0var:
      memoff0(R0) = R1;
      in_trail(R0);
    }
  }
  else
  {
    if (tag(R1) == UNDEF)
    {
R1var:
      memoff0(R1) = R0;
      in_trail(R1);
    }
    else
    {
      register int a, b, c, *bot;
      c = 1;
      IP = bot = G;
      while (TRUE)
      {
        deref(R0);
        deref(R1);
        if (R0 != R1) switch (unifytable[tagcode(R0,R1)])
        {
          case 0:
            processortrap(14);
          case 1:
            if (R0 != R1)
            {
              if (val(R1) < val(R0))
              {
                memoff0(R0) = makeword(LINK,val(R1));
                in_trail(R0);
              }
              else
              {
                memoff0(R1) = makeword(LINK,val(R0));
                in_trail(R1);
              }
            }
            break;
          case 2:
            memoff0(R0) = R1;
            in_trail(R0);
            break;
          case 3:
            memoff0(R1) = R0;
            in_trail(R1);
            break;
          case 4:
            if (termfunctor(R0) != termfunctor(R1)) goto Fail;
            if (c > 1)
            {
              *++IP = a;
              *++IP = b;
              *++IP = c;
            }
            a = adroff(val(R0),1);
            b = adroff(val(R1),1);
            c = termarity(R0);
            R0 = mem(a);
            R1 = mem(b);
            continue;
          case 5:
            if (boxcmp(val(R0),val(R1)) == FALSE) goto Fail;
            break;
          case 6:
            /* Taken care of by explicit test before the switch */
          case 7:
          case 8:
            goto Fail;
          case 9:
            if (c > 1)
            {
              *++IP = a;
              *++IP = b;
              *++IP = c;
            }
            a = val(R0);
            b = val(R1);
            c = 2;
            R0 = mem(a);
            R1 = mem(b);
            continue;
          case 10:
            if (tablesize(R0) != tablesize(R1))  goto Fail;
            if (c > 1)
            {
              *++IP = a;
              *++IP = b;
              *++IP = c;
            }
            a = adroff(val(R0),1);
            b = adroff(val(R1),1);
            c = tablesize(R0);
            R0 = mem(a);
            R1 = mem(b);
            continue;
        }
        if (--c == 0)
        {
          if (IP == bot) goto Continue;
          c = (*IP--) - 1;
          b = *IP--;
          a = *IP--;
        }
        a = adroff(a,1);
        b = adroff(b,1);
        R0 = mem(a);
        R1 = mem(b);
      }
    }
  }
  goto Continue;
}
