/*
**	This file demonstrates the use of the connectionless client/server
**	model.
**	In one window, start icp, consult this file and enter the command
**	    server.
**	You will get an output of the form
**	    | ?- server.
**	    Server started. You can start a client with the following command:
**		client(5569,2460554499) .
**	In one or more other windows, start icp, consult this file and enter
**	the command suggested by the server
**	An example session will be like this
**	    client(5569,2460554499) . 
**	    enter term, "end" ends this channel, "stop" stops server>
**	Prolog terms entered at the client prompt will be transmitted to the
**	server. To close the connection enter the atom 'end'. To close the
**	server enter 'stop'.
*/

/************************* connectionless server ******************************/
/************************************************************************/
/*	server								*/
/* This starts off the server.						*/
/************************************************************************/
server :-
	tcp_connectionless(0, Socket),
	tcp_getsockaddr(Socket, Port, Address),
	writeseqnl(['server Port:', Port, '  Address:', Address]),
	writeseqnl(['Server started.',
		'You can start a client with the following command: ','\n\t',
		client(Port, Address),'.']),
	server_loop(ok, Socket).

/************************************************************************/
/*	server_loop(Data, Socket)					*/
/* If the Data is stop, the server socket is closed and no new		*/
/* connections are accepted. Connected clients continue to be served	*/
/* by the forked processes.						*/
/* Otherwise, reads data from the socket until, the atom 'stop' is	*/
/* received.								*/
/************************************************************************/
server_loop(stop, Socket) :- !, tcp_close(Socket).
server_loop(_, Socket) :-
	tcp_recvfrom(Socket, Data, Port, Address),
	writeseqnl(['got>',Data,'<','from Port:',Port,'  Address:',Address]),
	server_loop(Data, Socket).

/************************* connectionless client ******************************/
/************************************************************************/
/*	client(Port, Address)						*/
/* This starts off the client						*/
/************************************************************************/
client(Port, Address) :-
	tcp_connectionless(0, Socket),
	client_loop(ok, Socket, Port, Address).

/************************************************************************/
/*	client_loop(In, Socket, Port, Address)				*/
/* Prompts the user for data to be sent to the server and sends it. The	*/
/* connection is closed when the atom 'stop' or 'end' is sent.		*/
/************************************************************************/
client_loop(stop, Socket, _, _) :- !, tcp_close(Socket).
client_loop(end, Socket, _, _) :- !, tcp_close(Socket).
client_loop(_, Socket, Port, Address) :-
	write('enter term, "end" ends this channel, "stop" stops server> '),
	flush,
	read(Data),
	tcp_sendto(Socket, Data, Port, Address),
	client_loop(Data, Socket, Port, Address).

?- ensure_loaded(tcp).
?- writeseqnl(['use the command\n\t \033[1mserver.\033[0m\nto start the server']).
