/*
**      This file demonstrates the use of the connected client/server model.
**      In one window, start icp, consult this file and enter the command
**          server.
**      You will get an output of the form
**          | ?- server.
**          use client(5569,2460554499).
**      In one or more other windows, start icp, consult this file and enter
**      the command suggested by the server
**      An example session will be like this
**          client(5569,2460554499) .
**          enter term, "end" ends this channel, "stop" stops server>
**      Prolog terms entered at the client prompt will be transmitted to the
**      server, and echoed on standard output. To close the connection enter
**	the atom 'end', 'stop' to close the server.
*/

/**************************** connected server **************************/

/************************************************************************/
/*      server                                                          */
/* This starts off the server						*/
/************************************************************************/
server :-
        tcp_server(0, Socket),
        tcp_getsockaddr(Socket, Port, Address),
        write('use '), write(client(Port, Address)), write('.\n'),
        server_loop(ok, Socket).

/************************************************************************/
/*      server_loop(Flag, Socket)                                       */
/* If the Flag is 'stop', the server socket is closed and no new	*/
/* connections are accepted. Connected clients continue to be served    */
/* by the forked processes.                                             */
/* Otherwise, accepts a connection on socket Socket and uses the	*/
/* predicate process_request/3 which deals with communication		*/
/* with the client. When this is finished the server looks for more	*/
/* connections.								*/
/************************************************************************/
server_loop(stop, Socket) :- !, tcp_close(Socket).
server_loop(_, Socket) :-
        tcp_accept(Socket, NewSocket),
        process_request(ok, NewSocket, T),
        server_loop(T, Socket).

/************************************************************************/
/*      process_request(In, Socket, Out)				*/
/* Echoes input till a 'stop' or 'end' is received.			*/
/************************************************************************/
process_request(stop, Socket, stop) :- !, tcp_close(Socket).
process_request(end, Socket, end) :- !, tcp_close(Socket).
process_request(_, Socket, Final) :-
        tcp_recv(Socket, T),
        writenl(tcp_recv(Socket, T)),
        process_request(T, Socket, Final).
process_request(_, Socket, end) :- tcp_close(Socket).

/************************** connected client ****************************/
/************************************************************************/
/*      client(Port, Address)                                           */
/* This starts off the client, connecting to the specified server port	*/
/* and address								*/
/************************************************************************/
client(Port, Address) :-
        tcp_client(Port, Address, Socket),
        client_loop(ok, Socket).

/************************************************************************/
/*      client_loop(Data, Socket)                                       */
/* Prompts the user for data to be sent to the server and sends it. The */
/* connection is closed when the atom 'stop' or 'end' is sent.		*/
/************************************************************************/
client_loop(stop, Socket) :- !, tcp_close(Socket).
client_loop(end, Socket) :- !, tcp_close(Socket).
client_loop(_, Socket) :-
        write('enter term, "end" ends this channel, "stop" stops server> '),
        flush,
        read(T),
        tcp_send(Socket, T), !,
        client_loop(T, Socket).
client_loop(_, Socket) :- !, tcp_close(Socket).

?- ensure_loaded(tcp).
