;; Emacs interface for IC PROLOG ][
;; Written for the Logic & Objects extension to IC PROLOG ][
;; Based on the Emacs interface for GDB
;; written by Z. Bobolakis, Imperial College (zb@doc.ic.ac.uk)
;; Last modified: 7 Dec 1992

  
(require 'shell)

(defvar icp-prompt-pattern "^| ?- *"
  "A regexp to recognize the prompt for ICP") 

(defvar icp-mode-map nil
  "Keymap for icp-mode.")

(if icp-mode-map
   nil
  (setq icp-mode-map (copy-keymap shell-mode-map))
  (define-key icp-mode-map "\C-l" 'icp-refresh))

(defun icp-mode ()
  "Major mode for interacting with an inferior ICP process."
  (interactive)
  (send-string (get-buffer-process (current-buffer)) "ensure_loaded(gnu).\n")
  (send-string (get-buffer-process (current-buffer)) "[lo].\n")
  (kill-all-local-variables)
  (setq major-mode 'icp-mode)
  (setq mode-name "Inferior ICP")
  (setq mode-line-process '(": %s"))
  (use-local-map icp-mode-map)
  (make-local-variable 'last-input-start)
  (setq last-input-start (make-marker))
  (make-local-variable 'last-input-end)
  (setq last-input-end (make-marker))
  (make-local-variable 'icp-last-frame)
  (setq icp-last-frame nil)
  (make-local-variable 'icp-last-frame-displayed-p)
  (setq icp-last-frame-displayed-p t)
  (make-local-variable 'icp-delete-prompt-marker)
  (setq icp-delete-prompt-marker nil)
  (make-local-variable 'icp-filter-accumulator)
  (setq icp-filter-accumulator nil)
  (make-local-variable 'shell-prompt-pattern)
  (setq shell-prompt-pattern icp-prompt-pattern)
  (run-hooks 'shell-mode-hook 'icp-mode-hook))

(defvar current-icp-buffer nil)

(defvar icp-command-name "icp"
  "Pathname for executing ICP.")

(defun icp ()
  "Run icp in buffer *icp-main*."
  (interactive)
  (progn
    (switch-to-buffer "*icp-main*")
    (or (bolp) (newline))
    (make-shell "icp-main" icp-command-name)
    (icp-mode)
    (set-process-filter (get-buffer-process (current-buffer)) 'icp-filter)
    (set-process-sentinel (get-buffer-process (current-buffer)) 'icp-sentinel)
    (icp-set-buffer)))

(defun icp-set-buffer ()
  (cond ((eq major-mode 'icp-mode)
	(setq current-icp-buffer (current-buffer)))))

(defun icp-filter (proc string)
  (let ((inhibit-quit t))
    (if icp-filter-accumulator
	(icp-filter-accumulate-marker proc
				      (concat icp-filter-accumulator string))
	(icp-filter-scan-input proc string))))

(defun icp-filter-accumulate-marker (proc string)
  (setq icp-filter-accumulator nil)
  (if (> (length string) 1)
      (if (= (aref string 1) ?\032)
	  (let ((end (string-match "\n" string)))
	    (if end
		(progn
		  (let* ((first-colon (string-match ":" string 2))
			 (second-colon
			  (string-match ":" string (1+ first-colon))))
		    (setq icp-last-frame
			  (cons (substring string 2 first-colon)
				(string-to-int
				 (substring string (1+ first-colon)
					    second-colon)))))
		  (setq icp-last-frame-displayed-p nil)
		  (icp-filter-scan-input proc
					 (substring string (1+ end))))
	      (setq icp-filter-accumulator string)))
	(icp-filter-insert proc "\032")
	(icp-filter-scan-input proc (substring string 1)))
    (setq icp-filter-accumulator string)))

(defun icp-filter-scan-input (proc string)
  (if (equal string "")
      (setq icp-filter-accumulator nil)
      (let ((start (string-match "\032" string)))
	(if start
	    (progn (icp-filter-insert proc (substring string 0 start))
		   (icp-filter-accumulate-marker proc
						 (substring string start)))
	    (icp-filter-insert proc string)))))

(defun icp-filter-insert (proc string)
  (let ((moving (= (point) (process-mark proc)))
	(output-after-point (< (point) (process-mark proc)))
	(old-buffer (current-buffer))
	start)
    (set-buffer (process-buffer proc))
    (unwind-protect
	(save-excursion
	  ;; Insert the text, moving the process-marker.
	  (goto-char (process-mark proc))
	  (setq start (point))
	  (insert string)
	  (set-marker (process-mark proc) (point))
	  (icp-maybe-delete-prompt)
	  ;; Check for a filename-and-char number.
	  (icp-display-frame
	   ;; Don't display the specified file
	   ;; unless (1) point is at or after the position where output appears
	   ;; and (2) this buffer is on the screen.
	   (or output-after-point
	       (not (get-buffer-window (current-buffer))))
	   ;; Display a file only when a new filename-and-char-number appears.
	   t))
      (set-buffer old-buffer))
    (if moving (goto-char (process-mark proc)))))

(defun icp-sentinel (proc msg)
  (cond ((null (buffer-name (process-buffer proc)))
	 ;; buffer killed
	 ;; Stop displaying an arrow in a source file.
	 (setq overlay-arrow-position nil)
	 (set-process-buffer proc nil))
	((memq (process-status proc) '(signal exit))
	 ;; Stop displaying an arrow in a source file.
	 (setq overlay-arrow-position nil)
	 ;; Fix the mode line.
	 (setq mode-line-process
	       (concat ": "
		       (symbol-name (process-status proc))))
	 (let* ((obuf (current-buffer)))
	   ;; save-excursion isn't the right thing if
	   ;;  process-buffer is current-buffer
	   (unwind-protect
	       (progn
		 ;; Write something in *compilation* and hack its mode line,
		 (set-buffer (process-buffer proc))
		 ;; Force mode line redisplay soon
		 (set-buffer-modified-p (buffer-modified-p))
		 (if (eobp)
		     (insert ?\n mode-name " " msg)
		   (save-excursion
		     (goto-char (point-max))
		     (insert ?\n mode-name " " msg)))
		 ;; If buffer and mode line will show that the process
		 ;; is dead, we can delete it now.  Otherwise it
		 ;; will stay around until M-x list-processes.
		 (delete-process proc))
	     ;; Restore old buffer, but don't restore old point
	     ;; if obuf is the icp buffer.
	     (set-buffer obuf))))))


(defun icp-refresh ()
  "Fix up a possibly garbled display, and redraw the arrow."
  (interactive)
  (redraw-display)
  (icp-display-frame))

(defun icp-display-frame (&optional nodisplay noauto)
  "Find, obey and delete the last filename-and-line marker from GDB.
The marker looks like \\032\\032FILENAME:CHARPOS:\\n.
Obeying it means displaying in another window the specified file and line."
  (interactive)
  (icp-set-buffer)
  (and icp-last-frame (not nodisplay)
       (or (not icp-last-frame-displayed-p) (not noauto))
       (progn (icp-display-line (car icp-last-frame) (cdr icp-last-frame))
	      (setq icp-last-frame-displayed-p t))))

;; Make sure the file named TRUE-FILE is in a buffer that appears on the screen
;; and that its char CHAR is visible.
;; Put the overlay-arrow on the line containing char CHAR in that buffer.

(defun icp-display-line (true-file char)
  (let* ((buffer (find-file-noselect true-file))
	 (window (display-buffer buffer t))
	 (pos))
    (save-excursion
      (set-buffer buffer)
      (save-restriction
	(widen)
	(goto-char char)
	(beginning-of-line)
	(setq pos (point))
	(setq overlay-arrow-string "=>")
	(or overlay-arrow-position
	    (setq overlay-arrow-position (make-marker)))
	(set-marker overlay-arrow-position (point) (current-buffer)))
      (cond ((or (< pos (point-min)) (> pos (point-max)))
	     (widen)
	     (goto-char pos))))
    (set-window-point window overlay-arrow-position)))


(defun icp-maybe-delete-prompt ()
  (if (and icp-delete-prompt-marker
	   (> (point-max) (marker-position icp-delete-prompt-marker)))
      (let (start)
	(goto-char icp-delete-prompt-marker)
	(setq start (point))
	(beginning-of-line)
	(delete-region (point) start)
	(setq icp-delete-prompt-marker nil))))


