/*****************************************************************************
LOGIC & OBJECTS ------------- IC PROLOG ][ version

written by F.G.McCabe (fgm@doc.ic.ac.uk)
modified for IC PROLOG ][ by Z.Bobolakis (zb@doc.ic.ac.uk)
last modified: 25 Jan 1993

Logic Programming Section
Dept. of Computing
Imperial College
London
******************************************************************************/

% L&O File 

/*
  Tracing in generated via a series of hooks which are compiled in to the 
 Prolog code.
  These hooks are defined below.

*/

% rl_trace marks the initial entry to a method in a class body
rl_trace(def(_,_,_,_,tr(off,_,_,_,_,_,_,_,_,_)),_,_,_,true):-!.
rl_trace(def(_,_,_,_,tr(on,_,_,_,_,_,_,_,_,_)),A,L,S,'#r#'(A,L,S)).

% cl_trace marks entry and exit in a clause
cl_trace(defnd(_,_,_,_,_,_,_,tr(off,_,_,_,_,_,_,_,_,_)),_,_,true,true):-!.
cl_trace(defnd(_,_,_,_,S,L,_,tr(on,_,_,_,_,_,_,_,_,_)),
          A,C,'#e#'(A,L,S,C),'#s#'(A,L,S,C)).

% vc_trace marks the use of a variable labelled call
vc_trace(def(_,_,_,_,tr(off,_,_,_,_,_,_,_,_,_)),_,_,_,true):-!.
vc_trace(def(_,_,_,_,tr(on,_,_,_,_,_,_,_,_,_)),A,L,S,'#l#'(A,L,S)).


% fn_trace marks the initial entry to a function in a class body
fn_trace(def(_,_,_,_,tr(_,off,_,_,_,_,_,_,_,_)),_,_,_,_,true):-!.
fn_trace(def(_,_,_,_,tr(_,on,_,_,_,_,_,_,_,_)),F,A,L,S,'#f#'(E,L,S)):-
  A=..[_|V],
  append(Args,[_],V),
  E=..[F|Args].

% eq_trace marks entry and exit of an equation
eq_trace(defnd(_,_,_,_,_,_,_,tr(_,off,_,_,_,_,_,_,_,_)),_,_,_,true,true):-!.
eq_trace(defnd(_,_,_,_,S,L,_,tr(_,on,_,_,_,_,_,_,_,_)),
         EX,C,V,'#q#'(EX,L,S,C),'#v#'(EX,V,L,S,C)).

% ve_trace marks a call to a variable labelled expression
ve_trace(def(_,_,_,_,tr(_,off,_,_,_,_,_,_,_,_)),_,_,_,_,true):-!.
ve_trace(def(_,_,_,_,tr(_,on,_,_,_,_,_,_,_,_)),F,A,L,S,'#k#'(E,L,S)):-
    A=..[_|V],
    append(Args,[_],V),
    E=..[F|Args].

% ih_trace marks the use of inheritance
ih_trace(defnd(_,_,_,_,_,_,_,tr(_,_,off,_,_,_,_,_,_,_)),_,_,_,true):-!.
ih_trace(defnd(_,_,_,_,S,L,_,tr(_,_,on,_,_,_,_,_,_,_)),M,A,C,'#i#'(L,M,A,C,S)).

% dc_trace marks entry and exit of a variable declaration
dc_trace(defnd(_,_,_,_,_,_,_,tr(_,_,_,off,_,_,_,_,_,_)),_,_,_,true):-!.
dc_trace(defnd(_,_,_,_,S,L,_,tr(_,_,_,on,_,_,_,_,_,_)),C,V,W,'#d#'(V,W,L,S,C)).

% a_trace marks the assignment of a variable
a_trace(defnd(_,_,_,_,_,_,_,tr(_,_,_,off,_,_,_,_,_,_)),_,_,_,_,true):-!.
a_trace(defnd(_,_,_,_,_,_,_,tr(_,_,_,on,_,_,_,_,_,_)),C,L,V,W,'#a#'(L,V,W,C)).

% db_trace marks a user specified print statement
db_trace(defnd(_,_,_,_,_,_,_,tr(_,_,_,_,off,_,_,_,_,_)),_,_,_,true):-!.
db_trace(defnd(_,_,_,_,S,L,_,tr(_,_,_,_,on,_,_,_,_,_)),W,M,G,(G,'#u#'(M,L,W,S))).

% pci_trace marks the use of a %'f primitive
pci_trace(defnd(_,_,_,_,_,_,_,tr(_,_,_,_,_,_,_,_,_,off)),_,true,true):-!.
pci_trace(defnd(_,_,_,_,S,L,_,tr(_,_,_,_,_,_,_,_,_,on)),A,'#%#'(A,L,S),'%#%'(A,L,S)).




% highlight the declaration of a variable
'#d#'(V,R,L,S,C):- is_traced(dynamic_code),is_traced(entry),!,
                   trace_dialog(report(declare),C,L,S,V:=R).
'#d#'(_,_,_,_,_).

% highlight the assignment of a variable
'#a#'(L,V,R,C):- is_traced(dynamic_code),is_traced(entry),!,
                 trace_dialog(report(assign),C,L,L,V:=R).
'#a#'(_,_,_,_).

% highlight entry to a relation from outside a class body
'#r#'(A,L,S):- is_traced(clauses),is_traced(entry),!,
               (trace_dialog(enter(relation),true,S,L,A);
               trace_dialog(fail(relation),true,S,L,A),fail).
'#r#'(_,_,_).

% highlight entry to a clause
'#e#'(A,L,S,C):- is_traced(clauses),is_traced(entry),!,
                 (trace_dialog(enter(clause),C,S,L,A);
                  trace_dialog(redo(clause),C,S,L,A),fail).
'#e#'(_,_,_,_).

% highlight exit from a clause
'#s#'(A,L,S,C):- is_traced(clauses),is_traced(exit),!,
                 trace_dialog(exit(clause),C,S,L,A).
'#s#'(A,L,S,C):- skip_trace(S,L,A),!,   % did we skip?
                 '#s#'(A,L,S,C).
'#s#'(_,_,_,_).



% highlight entry to a function from outside a class body
'#f#'(E,L,S):- is_traced(equations),is_traced(entry),!,
               trace_dialog(enter(function),true,S,L,E).
'#f#'(_,_,_).

% highlight entry to an equation
'#q#'(E,L,S,C):-  is_traced(equations),is_traced(entry),!,
                  trace_dialog(enter(equation),C,S,L,E).
'#q#'(_,_,_,_).

% highlight exit from an equation
'#v#'(EX,R,L,S,C):-  is_traced(equations),is_traced(exit),!,
                  trace_dialog(exit(equation),C,S,L,EX=R).
'#v#'(EX,R,L,S,C):-skip_trace(S,L,EX),!,
                  '#v#'(EX,R,L,S,C).
'#v#'(_,_,_,_,_).

% highlight the use of a class rule
'#i#'(L,M,_,C,S):-  is_traced(inheritance),is_traced(entry),!,
                 trace_dialog(report(inherit),C,S,L,(L<<M)).
'#i#'(_,_,_,_,_).

% highlight a user specified print selection
'#u#'(M,L,C,S):- is_traced(user),is_traced(entry),!,
               trace_dialog(report(user),C,S,L,M).
'#u#'(_,_,_,_).



% make a labelled variable call
'#l#'(A,L,S):- is_traced(clauses),is_traced(entry),!,
               (trace_dialog(entry(relation),true,S,L,A);
                trace_dialog(fail(relation),true,S,L,A),fail).
'#l#'(_,_,_).

% evaluate a variable labelled expression
'#k#'(E,L,S):-is_traced(equations),is_traced(entry),!,
               trace_dialog(entry(function),true,S,L,E).
'#k#'(_,_,_).

% enter a %'f primitive
'#%#'(A,L,S):-
  tracing_percents,!,
  trace_dialog(entry('%''f'),true,S,L,A).
'#%#'(_,_,_).

% exit a %'f primitive
'%#%'(A,L,S):-
  tracing_percents,!,
  trace_dialog(exit('%''f'),true,S,L,A).
'%#%'(_,_,_).




is_traced(X) :- 
	atom(X),!,
	get_prop('##traced##',X,on).


lo_trace :- 
	del_props('##traced##'),
	set_prop('##traced##',clauses,on),
	set_prop('##traced##',equations,on),
	set_prop('##traced##',inheritance,on),
	set_prop('##traced##',dynamic_code,on),
	set_prop('##traced##',entry,on),
	set_prop('##traced##',user,on),
	set_prop('##traced##',exit,on),
	set_prop('##traced##',fail,on),
	set_prop('##traced##',redo,on),
	set_prop('##traced##',percentfs,off).


lo_notrace :- 
	del_props('##traced##'),
	set_prop('##traced##',clauses,off),
	set_prop('##traced##',equations,off),
	set_prop('##traced##',inheritance,off),
	set_prop('##traced##',dynamic_code,off),
	set_prop('##traced##',entry,off),
	set_prop('##traced##',user,off),
	set_prop('##traced##',exit,off),
	set_prop('##traced##',fail,off),
	set_prop('##traced##',redo,off),
	set_prop('##traced##',percentfs,off).


/*
Implement a dialog to support tracing in IC-PROLOG ][ version of L&O
*/

trace_dialog(Info,Source,Sf,LbL,Call):-
  runtracehooks(Source),
  (LbL==Sf->Lb=Sf;Lb=LbL/Sf),
  writeqseq(user,[Info,Lb:Call,?]),
  flush_output(user_output),
  get0(user,B),
  trace_respond(B,_,Info,Source,Sf,LbL,Call).

runtracehooks(S) :-
  defined(gnutracehook/1),
  !,
  gnutracehook(S).
runtracehooks(_).


trace_respond(10,_,_,_,_,_,_) :- !.
trace_respond(0'q,_,_,_,_,_,_):- !,
  skip(user,10),
  abort.
trace_respond(0'f,_,_,_,_,_,_):- !,
  skip(user,10),
  fail.
trace_respond(0's,_,enter(_),_,Sf,LbL,Call):- !,
  skip(user,10),
  is_tracing(T),
  set_prop('#tracing#','#mode#',trace(Sf,LbL,Call,T)),
  tracing(off).
trace_respond(0'c,_,_,_,_,_,_):- !,
  skip(user,10).
trace_respond(0'h,I,Type(Info),Source,Sf,LbL,Call):- !,
  trace_respond(0'?,I,Type(Info),Source,Sf,LbL,Call).
trace_respond(_,_,Info,Source,Sf,LbL,Call):-
  skip(10),
  write(user_error, '\n\
Options :\n\
  <CR> - continue\n\
   c   - continue\n\
   s   - skip goal (only at entry)\n\
   f   - fail goal\n\
   q   - abort query\n\
   h   - display this list\n\n'),
  trace_dialog(Info,Source,Sf,LbL,Call).


% skip_trace is called by the hooks to see if we can finish skipping...
skip_trace(S,L,A):-
  get_prop('#tracing#','#mode#',trace(S,L,A,T)),!,
  tracing(T),
  del_prop('#tracing#','#mode#').


is_tracing(X):-
  get_props('##traced##',L),
  is_tr(L,X).

is_tr([],[]):-!.
is_tr([X|L],[X|T]):-
  get_prop('##traced##',X,on),!,
  is_tr(L,T).
is_tr([_|L],T):-
  is_tr(L,T).



%
% Give access to the tracing state - used by the preprocessor 
%
tr_mode(tr(Clauses,Equations,Inherit,Decls,User,Entry,Exit,Fail,Redo,Cent)):-
  get_prop('##traced##',clauses,Clauses),
  get_prop('##traced##',equations,Equations),
  get_prop('##traced##',inheritance,Inherit),
  get_prop('##traced##',dynamic_code,Decls),
  get_prop('##traced##',user,User),
  get_prop('##traced##',entry,Entry),
  get_prop('##traced##',exit,Exit),
  get_prop('##traced##',fail,Fail),
  get_prop('##traced##',redo,Redo),
  get_prop('##traced##',percentfs,Cent).



%
% tracing_percents is true if we are tracing the %f's
%
tracing_percents:-
  get_prop('##traced##',percentfs,on).

% 
% Support primitive to define tracing level 
%
tracing(all):-!,
  tracing([entry,exit,fail,redo],
  [clauses,equations,inheritance,dynamic_code,percentfs,user]).
tracing(off):-!,
  clear_tracing.
tracing(X):-!,
  tracing([entry,exit,fail,redo],X).
tracing(X,Y):-
  clear_tracing,
  enable_tracing(X),
  enable_tracing(Y).

clear_tracing :- lo_notrace.

enable_tracing([]).
enable_tracing([X|Y]) :- set_prop('##traced##',X,on),
	enable_tracing(Y).

