/*****************************************************************************
LOGIC & OBJECTS ------------- IC PROLOG ][ version

written by F.G.McCabe (fgm@doc.ic.ac.uk)
modified for IC PROLOG ][ by Z.Bobolakis (zb@doc.ic.ac.uk)
last modified: 25 Jan 1993

Logic Programming Section
Dept. of Computing
Imperial College
London
******************************************************************************/

% L&O File

/* translate all the templates which occur in a file/window */
tr_all_templates([],[],[],_,_).
tr_all_templates([tpl(Lb,Id,Pb,Body,Rules)|Ts], Rels, [Lb|LbLs],Tr,SuperObj):-
  tr_template(Lb, Body, Rules, Rels, Rs, Tr,Id,Pb,SuperObj),
  tr_all_templates(Ts,Rs,LbLs,Tr,SuperObj).

/* translate a complete class template */
tr_template(Lb/LbA, Body, Rules,[pr(LbS/3,[Access,Where|Cls])|Rs], Rs,Tr,Id,Pb,_):-
  var(Body),!,
  label_symbol(Lb,LbA,LbS),
  label_rule(LbS,Lb,LbA,Access,Where,'@w'(Id,Pb,1)),
  tr_all_rules(Rules, Cls, LbS,Tr).
tr_template(Lb/LbA, Body, Rules, Rls, Rs,Tr,Id,Pb,SuperObj):-
  label_symbol(Lb,LbA,LbS),
  tr_rules(Rules, Ri, Rs, Inherit, LbS, Body,Tr,SuperObj),
  tr_class_body(Body, Rls, Ri, Inherit,LbS,LbA,Tr,Id,Pb).

/*   Generate the clauses for a class body. */
tr_class_body((L:{Clauses}),[pr(LbS/3,EntryPs)|R],Rs, In, LbS, _, Tr,Id,Pb):-
  sort_sentences(Clauses,Relations,Preds,Functions,Funs,Assigns,IVars,Id,0,LbS),
  termin(Preds),
  termin(Funs),
  termin(IVars),
  Defnd=def(LbS, Funs, Preds, IVars, Tr),
  tr_methods(Relations, R, Ri, Defnd, L, In),
  tr_functions(Functions, Ri, Rj, Defnd, L),!,
  mk_entry_rels(Preds,EntryPs,Ei,Defnd),
  mk_entry_funs(Funs, Ei, Ej,Defnd),
  tr_decls(Assigns, Ej, Ek, Defnd, L),
  mk_entry_vars(IVars, Ek, El, Rj, Rs, Defnd),
  mk_super_rule(LbS, L, El, [], '@w'(Id,Pb,1)),!.

sort_sentences([], _, _, _, _, _, _, _, _, _).
sort_sentences([Cl-P|Clauses],Rels,Preds,Functs,Funs,Assigns,IVars,Id,Ix,L):-
  sort_sentence(Cl,Rels,Preds,Functs,Funs,Assigns,IVars,Id,P,Ix,L), Ix1 is Ix+1,
  sort_sentences(Clauses,Rels,Preds,Functs,Funs,Assigns,IVars,Id,Ix1,L).

sort_sentence(Cl,_,_,Functions,Funs,_,_,Id,P,Ix,L):-
  defining_fun(Cl,F,A),!,      % is this an equation?
  occ(sym(F,A,LP),Funs),
  (var(LP)->
    local_fun(L,F,A,LP);true), % construct local function symbol for this func
  occ(eqs(F,A,Eqs),Functions),
  add_occ(Cl,Eqs,1,Id,P,Ix).
sort_sentence(Cl,_,_,_,_,Assigns,IVars,Id,P,Ix,_):-
  defining_ivar(Cl,Var),!,
  occ(Var,IVars),
  add_occ(Cl,Assigns,1,Id,P,Ix). % exactly one initialisation statement allowed
sort_sentence(Cl,Relations,Preds,_,_,_,_,Id,P,Ix,L):-
  defining_pred(Cl,Pred,Arity),!,
  occ(sym(Pred,Arity,LP),Preds),
  (var(LP)->local_pred(L,Pred,Arity,LP);true),
  occ(cls(Pred,Arity,Cls),Relations),
  add_occ(Cl,Cls,1,Id,P,Ix).
sort_sentence((:-G), _,_,_,_,_,_,_,_,_,_):-
  (call(G)->true;true).

/* pick out the function symbol and arity of an equation */
defining_fun((LHS=_:-_),Fun,Arity):-!,
  functor(LHS,Fun,Arity).
defining_fun((LHS=_),Fun,Arity):-
  functor(LHS,Fun,Arity).

/* pick out the predicate and arity of the clause */
defining_pred((Head:-_),Pred,Arity):-
  !,
  functor(Head,Pred,Arity).
defining_pred(Head,Pred,Arity):-
  functor(Head,Pred,Arity).

% declare an instanced variable
defining_ivar(IV:=_,IV):-
  atom(IV).

%
% map the relations within the class body into Prolog-style clauses
%
tr_methods([], Rls, Rls, _, _, _):-!.
tr_methods([cls(Pred,Arity,Clses)|Rels],[pr(P/A,Cls)|R], Rs,Defnd, LbL,Super):-
  termin(Clses),
  local_pred_name(Defnd,Pred,Arity,P),
  tr_rel(Clses, Cls, Cli, Defnd, LbL),!,
  inherit_clause(Super,Cli,[],Pred,Arity,Defnd),  % construct super reference
  Cls = [(H:-_)|_], functor(H,_,A),   
  tr_methods(Rels, R, Rs, Defnd, LbL, Super).
%
% map a relation into the appropriate suite of clauses
%
tr_rel([], C, C, _, _).
tr_rel([[Cnt|Cl]|Cls], [(H:-B)|PC], PCls, Defnd, LbL):-
  tr_clause(Cl, H, B, Defnd, LbL,Cnt),
  tr_rel(Cls, PC, PCls, Defnd, LbL).

%
% map the functions within the class body into Prolog style clauses
%
tr_functions([], Rls, Rls, _, _):-!.
tr_functions([eqs(Fun,Arity,Eqs)|Fs], [pr(P/A,Cls)|R], Rs, Defnd, LbL):-
  termin(Eqs),
  local_fun_name(Defnd,Fun,Arity,P),
  tr_fun(Eqs, Cls, [], Defnd, LbL),
  Cls = [(H:-_)|_], functor(H,_,A),   
  tr_functions(Fs, R, Rs, Defnd, LbL).

tr_fun([], C, C, _, _).
tr_fun([[Cnt|(LHS=RHS:-Cond)]|Eqs], [(H:-B)|PC], PEqs, Defnd, LbL):-!,
  tr_eqn(LHS, RHS, Cond, H, B, Defnd, LbL,Cnt),
  tr_fun(Eqs, PC, PEqs, Defnd, LbL).
tr_fun([[Cnt|(LHS=RHS)]|Eqs], [(H:-B)|PC], PEqs, Defnd, LbL):-
  tr_eqn(LHS, RHS, true, H, B, Defnd, LbL,Cnt),
  tr_fun(Eqs, PC, PEqs, Defnd, LbL).

%
% map the variable initialization statements
%
tr_decls([], Cls, Cls, _, _):-!.  % no assignments to take care of
tr_decls(Assigns, [(Head:-InitVs)|Cls], Cls, Dfnd, LbL):-
  termin(Assigns),
  label_arg_form(LbL, head, Assigns, Dfnd, Defnd),
  label_n_self(Defnd, LbS, LbA, SV),
  tr_decs(Assigns, Defnd, Is),
  triple(LbS,'#init#',LbA,SV,Head),
  simplify((Is,fail),InitVs).

tr_decs([], _, true):-!.
tr_decs([[Ctxt|Ass]|Assigns], Defnd, (I,Inits)):-
  tr_dec(Ass, I, Defnd,Ctxt),
  tr_decs(Assigns, Defnd, Inits).

/* handle one instance variable declaration */
tr_dec(VAR:=RHS, InitG, Defnd,Ctxt):-!,
  self_var(Defnd, SelfVar),
  tr_term(RHS, R, GR, Defnd),
  dc_trace(Defnd,Ctxt,VAR,R,TGX),
  simplify((GR,initialise(SelfVar,VAR,R),TGX), InitG).

%
% the clauses that represent the predicate entry points into the class body
%
mk_entry_rels([],OCls,OCls,_).
mk_entry_rels([sym(Pr,Ar,LocP)|Preds],
       [(Head:-!,Goal)|OCls],OCls1,Defnd):-
  defnd_label(Defnd,LbS),
  functor(Atom,Pr,Ar),   % create a skeleton head
  Atom =.. [_|V],
  append(V,[Lab,Self],LocA),
  B=..[LocP|LocA],
  rl_trace(Defnd,Atom,Lab,Self,TGE),
  triple(LbS,Atom,Lab,Self,Head),
  simplify((TGE,B),Goal),
  mk_entry_rels(Preds,OCls, OCls1,Defnd).
%
% the clauses that represent the function entry points
%
mk_entry_funs([],OCls,OCls,_).
mk_entry_funs([sym(F,Ar,LocP)|Funs],
      [(Head:-!,Goal)|OCls],OCls1,Defnd):-
  defnd_label(Defnd,LbS),
  Ar1 is Ar+1,
  conc([F,'*'],Fn),
  functor(Atom,Fn,Ar1),
  Atom=..[_|V],
  append(V,[Lab,Self],LocA),
  B=..[LocP|LocA],
  fn_trace(Defnd,F,Atom,Lab,Self,TGE),
  triple(LbS,Atom,Lab,Self,Head),
  simplify((TGE,B),Goal),
  mk_entry_funs(Funs,OCls,OCls1,Defnd).

%
% set up entry points for instance variables, to look like functions (sort of)
%
mk_entry_vars([], Cls, Cls, Rs, Rs,_):-!.
mk_entry_vars([V|IVars],
       [(Head:-!,VCall)|OCls],OCls1,
       [pr(Vsymb/2,
          [(Vhead:-retrieve(Sf,V,Val))])|Ri],Rs,Defnd):-
  defnd_label(Defnd,LbS),
  conc([V, *], LocP),
  conc(['<',V,'>'],Vsymb),
  unary(LocP,VX,Label),
  binary(Vsymb,Self,VX,VCall),
  triple(LbS,Label,_,Self,Head),
  binary(Vsymb,Sf,Val,Vhead),
  mk_entry_vars(IVars, OCls, OCls1,Ri,Rs,Defnd).

%
% final entry point which picks up on the class rules, and defines the label symbol
%
mk_super_rule(LbS,LbL,[Label,Where,(MHead:-Body)|Cls],Cls,Pb):-
  functor(LbL,Lb,LbA),
  label_rule(LbS,Lb,LbA,Label,Where,Pb),
  triple(LbS,A,L,S,MHead),
  triple(SP,A,L,S,Body),
  conc([LbS,super],SP).

%
% generate the label declaration rule
%
label_rule(LbS,Lb,LbA, (Head:-!),(Where:-!),Pb):-
  functor(LbL,Lb,LbA),
  triple(LbS,'?label?'(LbS),LbL,_,Head),
  triple(LbS,'?where?'(Pb),LbL,_,Where).
%
% generate a clause for the ?::? predicate to give access to the label
%
access_rule(LbS, Lb, Ar, (?::?(Label,Atom,Self):-Body)):-
  functor(Label,Lb,Ar),
  triple(LbS,Atom,Label,Self,Body).

/* convert a class body axiom into the set of clauses needed to implement it */

tr_clause((H:-B),Head,Body,Dfnd,L,Cnt):-!,
  label_arg_form(L, H, B, Dfnd, Defnd),
  H=..[HPr|Hargs],
  functor(H,HPr,HAr),
  tr_list(Hargs, Nargs, [], SGH, Defnd),
  form_local_pred(HPr, HAr, Nargs, Head, Defnd),
  tr_body(B, Bdy, Defnd,Cnt),
  cl_trace(Defnd,H,Cnt,GE,GX),
  simplify((GE,Bdy,SGH,GX), Body).  
tr_clause(H,Head,Body,Dfnd,L,Cnt):-
  label_arg_form(L, H, true, Dfnd, Defnd),
  H=..[HPr|Hargs],
  functor(H,_,HAr),
  tr_list(Hargs, Nargs, [], SGH, Defnd),
  form_local_pred(HPr, HAr, Nargs, Head, Defnd),
  cl_trace(Defnd,H,Cnt,GE,GX),
  simplify((GE,SGH,GX), Body).

/* compile an equation - similar to compiling a normal clause */
tr_eqn(LHS, RHS, B,Head,Body,Dfnd,L,Cnt):-!,
  label_arg_form(L, (LHS=RHS), B, Dfnd, Defnd),
  functor(LHS,F,FAr),
  LHS=..[_|Args],                        % split LHS into function and arguments
  label_n_self(Defnd, _, LbArg, SelfVar),
  tr_list(Args, Nargs, [R,LbArg,SelfVar], SGH, Defnd),
  local_fun_name(Defnd, F, FAr, LocP),
  Head=..[LocP|Nargs],
  tr_body(B, Bdy, Defnd,Cnt),
  tr_term(RHS, R, GR, Defnd),
  eq_trace(Defnd,LHS,Cnt,R,TGE,TGX),
  simplify((SGH,TGE,Bdy,!,GR,TGX), Body).

/*
The next section handles conditions in clauses and equations which occur in class
bodies.  Each condition is converted into a special form.
Also any disjunctions negations, and arrows are extracted from a clause, leaving 
just a conjunction of calls to deal with.
Any functional expressions are converted  into calls to predicates.
*/

%
% translate the body conditions that can occur in the body of a clause
%
% split into labelled and unlabelled conditions ...
%
%
% first of all check for a variable call
%
tr_body(G, Goal, Defnd, _):-
  var(G),!,
  label_n_self(Defnd, LbS, LbArg, SelfVar),
  triple(LbS, G, LbArg, SelfVar, Goal).

% or a variable predicate symbol ...
tr_body(G, Goal, Defnd,Where):-
  functor(G,GP,_),
  var(GP),!,
  tr_unlabelled_body(G, Goal, Defnd, Where).

% a super condition ...
tr_body(:Call, Goal, Defnd,Where):-!,
  tr_labelled_body(Call, super, Goal, Defnd,Where).

% a labelled condition ...
tr_body(LbL:Call, Goal, Defnd,Where):-!,
  tr_term(LbL,Lb,SGL,Defnd),
  tr_labelled_body(Call, Lb, Gl, Defnd,Where),
  simplify((SGL,Gl),Goal).

% a debug statement ...
tr_body(trace(Message),Goal,Defnd,Where):-!,
  tr_term(Message,L,SG,Defnd),
  db_trace(Defnd,Where,L,SG,TG),
  simplify(TG,Goal).

tr_body(G, Goal, Defnd,Where):-
  tr_unlabelled_body(G,Goal,Defnd,Where).

% handle a labelled condition
tr_labelled_body(Call, Lb, Goal, Defnd, _):-
  var(Call),!,
  (var(Lb)->(self_var(Defnd,Sf),
            Goal= ?:?(Call,Lb,Sf));
            label_symbol(Lb,LbSymb),
            triple(LbSymb,Call,Lb,Lb,Goal)).

% a variable predicate symbol
tr_labelled_body(Call, Lb, Goal, Defnd, _):-
  functor(Call,V,_), var(V),!,
  tr_term(Call,Cll,SGC,Defnd),
  (var(Lb)->(self_var(Defnd,Sf),
            simplify((SGC,?:?(Cll,Lb,Sf)), Goal));
            label_symbol(Lb,LbSymb),
            triple(LbSymb,Cll,Lb,Lb,GL),
            simplify((SGC,GL),Goal)).

/* label movement rules */
tr_labelled_body((E,O), Label, Goal, Defnd,Where):-!,
  tr_labelled_body(E, Label, G1, Defnd,Where),
  tr_labelled_body(O, Label, G2, Defnd,Where),
  simplify((G1,G2), Goal).
tr_labelled_body((E;O),Label, (GE;GO), Defnd,Where):-
  tr_labelled_body(E,Label, G1, Defnd,Where),
  tr_labelled_body(O,Label, G2, Defnd,Where),
  simplify(G1,GE),
  simplify(G2,GO).
tr_labelled_body(\+E, Label, \+Goal, Defnd,Where):-
  tr_labelled_body(E, Label, Goal, Defnd,Where).
tr_labelled_body((not(E)), Label, not(Goal), Defnd,Where):-
  tr_labelled_body(E, Label, Goal, Defnd,Where).
tr_labelled_body((T->Th;El), L, (Test->Then;Else), Defnd,Where):-
  tr_labelled_body(T, L, Test, Defnd,Where),
  tr_labelled_body(Th, L, Then, Defnd,Where),
  tr_labelled_body(El, L, Else, Defnd,Where).

tr_labelled_body((Lab:C), _, Goal, Defnd,Where):-!,
  tr_term(Lab,Lb,SGL,Defnd),
  tr_labelled_body(C, Lb, Gl, Defnd,Where),
  simplify((SGL,Gl),Goal).

tr_labelled_body(true, _, true, _, _).

% the higher-order predicates
tr_labelled_body(setof(T,C,L), LbL, setof(T1,C1,L), Defnd, Where):-!,
  tr_term(T, T1, TG, Defnd),
  tr_labelled_body(C, LbL, SC, Defnd, Where),
  simplify((SC,TG),C1).
tr_labelled_body(bagof(T,C,L), LbL, bagof(T1,C1,L), Defnd, Where):-!,
  tr_term(T, T1, TG, Defnd),
  tr_labelled_body(C, LbL, SC, Defnd, Where),
  simplify((SC,TG),C1).
tr_labelled_body(findall(T,C,L), LbL, findall(T1,C1,L), Defnd, Where):-!,
  tr_term(T, T1, TG, Defnd),
  tr_labelled_body(C, LbL, SC, Defnd, Where),
  simplify((SC,TG),C1).
tr_labelled_body(Q^C, Q^NC, LbL, Defnd, Where):-!,
 tr_labelled_body(C, LbL, NC, Defnd, Where).

% a special case where a condition is an equality in the body
tr_labelled_body((LHS=RHS), L, G, Defnd, _):-!,
  tr_term(L:LHS, NLHS, LG, Defnd),
  tr_term(L:RHS, NRHS, RG, Defnd),
  tr_body_eq(LHS, NLHS, RHS, NRHS, LG, RG, G, Defnd).

% an assignment 'statement'
tr_labelled_body(V:=RHS, L, G, Defnd,Where):-
  atom(V),!,
  tr_term(RHS, RV, G1, Defnd),
  a_trace(Defnd,Where,L,V,RV,TG),
  simplify((G1,TG,assign(L,V,RV)), G).

% variable labelled call
tr_labelled_body(Call, Lb, Goal, Defnd, _):-
  var(Lb),!,
  tr_term(Call,Cll,SGC,Defnd),
   self_var(Defnd,Sf),
  simplify((SGC,?:?(Cll,Lb,Sf)), Goal).

% super labelled call
tr_labelled_body(Call, super, Goal, Defnd, _):-!,
  tr_term(Call,Cll,SGC,Defnd),
  label_n_self(Defnd, LbS, LbArg, SelfVar),
  super_pred(LbS, LocP),
  triple(LocP,Cll, LbArg, SelfVar,GL),
  simplify((SGC,GL), Goal). 

% self labelled call
tr_labelled_body(Call, self, Goal, Defnd, _):-!,
  tr_term(Call,Cll,SG,Defnd),
  self_var(Defnd, SelfVar),
  simplify((SG,?:?(Cll,SelfVar,SelfVar)), Goal).

% labelled condition - generate the special labelled call.
tr_labelled_body(Call, Lb, Goal, Defnd, _):-
  tr_term(Call,Cll,SGC,Defnd),
  (var(Lb)->(self_var(Defnd,Sf),
            simplify((SGC,?:?(Cll,Lb,Sf)), Goal));
            label_symbol(Lb,LbSymb),
            triple(LbSymb,Cll,Lb,Lb,GL),
            simplify((SGC,GL),Goal)).

%
% unlabelled conditions use the label which is part of the class body

tr_unlabelled_body(true, true, _, _).
tr_unlabelled_body((At,B), (Na,Nb),  Defnd,Where) :-
  !,
  tr_body(At, Na, Defnd,Where),
  tr_body(B, Nb, Defnd,Where).
%
% Look for ; -> not etc.
% 
tr_unlabelled_body((T->Th), (Test->Then), Defnd,Where):-
  tr_body(T, Test, Defnd,Where),
  tr_body(Th, Then, Defnd,Where).

% disjunction
tr_unlabelled_body((Ei;Or), (Either; Orr), Defnd,Where):-
  tr_body(Ei, Either, Defnd,Where),
  tr_body(Or, Orr, Defnd,Where).

% negation
tr_unlabelled_body(\+NT, \+Not, Defnd,Where) :-
  tr_body(NT, Not, Defnd,Where).
tr_unlabelled_body(not(NT), not(Not), Defnd,Where) :-
  tr_body(NT, Not, Defnd,Where).
tr_unlabelled_body(forall(IF, THEN), forall(If, Then), Defnd,Where) :-
  tr_body(IF, If, Defnd,Where),
  tr_body(THEN, Then, Defnd,Where).

% the higher-order predicates
tr_unlabelled_body(setof(T,C,L), setof(T1,C1,L), Defnd, Where):-!,
  tr_term(T, T1, TG, Defnd),
  tr_body(C, SC, Defnd, Where),
  simplify((SC,TG),C1).
tr_unlabelled_body(bagof(T,C,L), bagof(T1,C1,L), Defnd, Where):-!,
  tr_term(T, T1, TG, Defnd),
  tr_body(C, SC, Defnd, Where),
  simplify((SC,TG),C1).
tr_unlabelled_body(findall(T,C,L), findall(T1,C1,L), Defnd, Where):-!,
  tr_term(T, T1, TG, Defnd),
  tr_body(C, SC, Defnd, Where),
  simplify((SC,TG),C1).
tr_unlabelled_body(Q^C, Q^NC, Defnd, Where):-!,
 tr_unlabelled_body(C, NC, Defnd, Where).

% a special case where a condition is an equality in the body
tr_unlabelled_body((LHS=RHS), G, Defnd, _):-!,
  tr_term(LHS, NLHS, LG, Defnd),
  tr_term(RHS, NRHS, RG, Defnd),
  tr_body_eq(LHS, NLHS, RHS, NRHS, LG, RG, G, Defnd).

% an assignment 'statement'
tr_unlabelled_body(V:=RHS, G, Defnd,Where):-!,
  atom(V),
  tr_term(RHS, RV, G1, Defnd),
  self_var(Defnd,Self),
  a_trace(Defnd,Where,Self,V,RV,TG),
  simplify((G1,TG,assign(Self,V,RV)), G).

% cuts are left alone
tr_unlabelled_body(!, !, _, _):-!.

tr_unlabelled_body(`Call, Goal, Defnd, _):-
  tr_quoted(Call, G, C, Defnd),
  simplify((C,G), Goal).

tr_unlabelled_body(`#Call, Goal, Defnd, _):-
  Call=..[F|A],
  tr_list(A, AA, [], C, Defnd),
  G=..[F|AA],
  simplify((C,G), Goal).

% basic condition - extract any expressions from inside it
tr_unlabelled_body(Call, Goal, Defnd, _):-
  Call=..[Pr|Args],
  functor(Call, _, PAr),
  tr_list(Args, NArgs, [], SGC, Defnd),
  tr_body_call(Pr, PAr, NArgs, SGC, Goal, Defnd).

tr_body_call(Pr, _, Args, SGC, Goal, Defnd):-
  var(Pr),!,
  label_n_self(Defnd, _, LbArg, SelfVar),
  Atom=..[Pr|Args],
  simplify((SGC,'?:?'(Atom,LbArg,SelfVar)), Goal).
tr_body_call(Pr, PAr, Args, SGC, Goal, Defnd):-
  form_local_pred(Pr,PAr,Args,LocA,Defnd),
  simplify((SGC,_,LocA),Goal).

form_local_pred(Pr,Ar,Args,Atom,defnd(_,_,Preds,_,SelfVar,LbArg,_,_)):-
  occ(sym(Pr,Ar,LocP),Preds),               % locate the local predicate symbol
  append(Args,[LbArg,SelfVar],AArgs),
  Atom=..[LocP|AArgs].
form_local_pred(Pr, Ar, Args, Goal, _):-
  primitive_pred(Pr,Ar),!,
  Goal=..[Pr|Args].
form_local_pred(Pr, _, Args, (TGE,Goal,TGX), Defnd):-
  perCentF(Pr),!,            % deal with %f's separateley
  Goal=..[Pr|Args],
  pci_trace(Defnd,Goal,TGE,TGX).
form_local_pred(Pr, _, Args, Goal, Defnd):-
  A=..[Pr|Args],            % construct a super call
  label_n_self(Defnd, Lb, LbArg, SelfVar),
  super_pred(Lb, LbS),
  triple(LbS,A,LbArg,SelfVar,Goal).

tr_body_eq(LHS, NLHS, RHS, NRHS, LG, RG, Goal, _):-
  (reduced_term(LHS, NLHS) ->   % has the lhs been reduced?
        (NLHS = NRHS,
	simplify((LG, RG), Goal));
   (reduced_term(RHS, NRHS) ->   % has the rhs been reduced?
	    (NRHS = NLHS,
	    simplify((LG, RG), Goal)))).
tr_body_eq(_, NLHS, _, NRHS, LG, RG, Goal, _):-
  simplify((LG,RG,NLHS=NRHS), Goal).

reduced_term(A, B):-
  var(B), A\==B.  % if it is the same variable then keep the equality

%  
% determine the form of the label argument in the translated clauses
%
label_arg_form(L, H, B, def(LbS,Fs,Prs,IVs,Tr), defnd(LbS,Fs,Prs,IVs,_,L,shared,Tr)):-
  varsin(L, LabVars),
  occurrences(H, LabVars, HO),
  body_average(B, LabVars, BO),
  shareable(HO,BO),!.        % the number of occs. of label vars is OK
label_arg_form(L, _, _, def(LbS,Fs,Prs,IVs,Tr), defnd(LbS,Fs,Prs,IVs,_,_,LabVars,Tr)):-
  L=..[_|LabVars],
  good_args(LabVars),!.
label_arg_form(L, _, _, def(LbS,Fs,Prs,IVs,Tr), defnd(LbS,Fs,Prs,IVs,_,L,shared,Tr)).

body_length(X,C,C1):- var(X),!,C1 is C+1.
body_length((_,B),I,C):-
  I1 is I+1,
  body_length(B,I1,C).
body_length(_,C,C1):- C1 is C+1.

body_average(B,L,C):-
  body_length(B,0,BL),
  occurrences(B,L,BO),
  C is BO/BL.

occurrences(T, L, C):-
  varsin(T, VT),
  count_occs(VT,L,0,C).

count_occs([],_,C,C).
count_occs([V|L],VL,C,CH):-
  occ_ident(V,VL),!,
  C1 is C+1,
  count_occs(L,VL,C1,CH).
count_occs([_|L],VL,C,CH):-
  count_occs(L,VL,C,CH).

% the conditions under which it is OK to copy the label term around
shareable(H,0):-H>0,!.
shareable(_,B):-B>0.75.

% good_args checks that the top-level arguments to a label are all vars or atoms
good_args([]).
good_args([X|L]):-
  var(X),!,good_args(L).
good_args([X|L]):-
  atomic(X),good_args(L). 

label_argument(Var,defnd(_,_,_,_,_,LX,L,_),Ix,LX):-
  var(LX), L \== shared,
  index_occ(Var, L, 1, Ix).

self_var(defnd(_,_,_,_,Self,_,_,_), Self).
label_n_self(defnd(LbS,_,_,_,Self,LbArg,_,_),LbS, LbArg,Self).
defnd_label(defnd(LbS,_,_,_,_,_,_,_),LbS):-!.
defnd_label(def(LbS,_,_,_,_),LbS).


/*
The next section deals with class rules.
Two sets of clauses are generated for each label - those for
normal class rules and those for overriding class rules
*/

tr_rules([], Rels, Rlo, absent, Lb, (Label:_), Tr, SuperObj):-!,
  tr_rules([[true|(Label<<SuperObj:-true)]], Rels, Rlo, _, Lb, [], Tr, _).

tr_rules(Rules, [pr(LbSuper/3,Cls)|Rels], Rlo, Inherit, Lb, _, Tr, _):-
  conc([Lb,super],LbSuper),
  conc([Lb,inherit],LbInherit),
  tr_all_rules(Rules, Cls, LbSuper, Tr),
  tr_over_rules(Rules, Over, [], LbInherit, Tr),
  (Over \== []->(Inherit=present, Rels = [pr(LbInherit/3,Over)|Rlo]);
                Inherit=absent, Rels=Rlo).

% handles both <= and <<
tr_all_rules([],[], _,_).
tr_all_rules([[C|(Lbl<=Mbl:-B)]|Rules], [R|Cls], Lb, Tr):-!,
  tr_rule(Lbl,Mbl,B, R, Lb,Tr,C),
  tr_all_rules(Rules, Cls, Lb, Tr).
tr_all_rules([[C|(Lbl<<Mbl:-B)]|Rules], [R|Cls], Lb, Tr):-!,
  tr_rule(Lbl,Mbl,B, R, Lb, Tr,C),
  tr_all_rules(Rules, Cls, Lb, Tr).

% translate the non-overriding class rules only
tr_over_rules([], Cls, Cls, _, _).
tr_over_rules([[C|(Lbl<=Mbl:-B)]|Rules], [R|RCls], Cls, Lb, Tr):-!,
  tr_rule(Lbl,Mbl,B, R, Lb, Tr, C),
  tr_over_rules(Rules, RCls, Cls, Lb, Tr).
tr_over_rules([_|Rules], RCls, Cls, Lb, Tr):-  % ignore << this time
  tr_over_rules(Rules, RCls, Cls, Lb, Tr).

% translate an individual class rule
tr_rule(LbL, MbL, G, (Head:-Body), LbS, Tr, Cnt):-
  Defnd = defnd(LbS,[],[],[],Self,L,shared,Tr),
  tr_term(LbL, L, G1,  Defnd),
  tr_mabel(MbL, M, G2, GX, Atom, Defnd),
  tr_body(G, G3, Defnd,Cnt),
  ih_trace(Defnd,M,Atom,Cnt,TG),
  triple(LbS,Atom,L,Self,Head),
  (var(M)->(simplify((G1,G3,G2,TG,?#?(Atom,M,Self),GX), Body));
        functor(M,MbS,MbA),
        (var(MbS)->
              simplify((G1,G3,G2,TG,?#?(Atom,M,Self),GX), Body);
              label_symbol(MbS,MbA,MMbS),
              triple(MMbS,Atom,M,Self,GL),
              simplify((G1,G3,G2,TG,GL,GX), Body))),!.

tr_mabel(Var, Var, true, true, _, _):-
   var(Var),!.
tr_mabel(Minus(MbL,Diff), M, G, no_occ(Atom,DL), Atom, Defnd):-
   nonvar(Minus),Minus= -,!,   % a fix to allow for variable functors
   tr_term(MbL, M, G, Defnd),
   tr_diff(Diff, DL).
tr_mabel(MbL, M, G, true, _, Defnd):-
   tr_term(MbL, M, G, Defnd).

% Handle the differential inheritance list
tr_diff([],[]).
tr_diff([P/A|L], [PQ|LL]):-!,
  functor(PQ, P, A),
  tr_diff(L, LL).
tr_diff([A|L],[A|LL]):-
  tr_diff(L,LL).

inherit_clause(absent, Cls, Cls, _, _, _):-!.
inherit_clause(present,[(Hd:-Body)|Cls], Cls, Pred, Arity, Defnd):-
  defnd_label(Defnd,LbS),
  conc([LbS, inherit], LP),
  triple(LP,Atom,Lab,Self,Body),
  local_pred_name(Defnd,  Pred, Arity, LocP),
  length(Args, Arity),      % form the basic arguments
  Atom=..[Pred|Args],
  append(Args,[Lab,Self], HA),
  Hd=..[LocP|HA],!.
