/* Copyright (C) 1992 Imperial College */
/* public list :
        ['$ comp_relation'/7, '$ assert_compiler'/6,
         '$ indxless'/2, '$ v_less'/2, '$ compare_perms'/2]
*/

/*
   In this section we generate the code for a complete predicate/arity 
   combination.  We can assume that all the clauses of a given predicate
   and arity have been collected together into a single list of clauses.
*/

'$ comp_relation'(Clses,Pred,Ar,Labin,LabOut,[Pred/Ar|C],Cout):-
    comp_clauses(Clses,CodeFragments,Labin, Li),
    new_label(Li,Lab1,Deflt),
    default_sequence_try(CodeFragments, DefC, Cout, Lab1, Lab2, 1,
               V,V,V,V,V,V,V, [],Nil,Int,List,Tpl,Con,Float, Ar),
    construct_switch(V,Nil,Int,List,Tpl,Con,Float,C,[Deflt|DefC],Lab2,LabOut,Deflt,Ar).

comp_clauses([], [], L, L).
comp_clauses([Clause|Clauses],[cx(ClCode,Rest,First)|Frags],Li,Lo) :-
    '$comp_clause$'(Clause,ClCode,Rest,First, Li, Lj), !,
    comp_clauses(Clauses, Frags, Lj, Lo).

/* In default_sequence we construct the default clause sequence, and a table
   of all the labels and first arguments for each clause.  This latter is 
   used by the index table construction programs.
*/
default_sequence_try([], Co, Co, Lab, Lab, _, V,N,I,L,T,C,F, V,N,I,L,T,C,F, _).
default_sequence_try([Clause], [Lab| Cli], Clo, Li, Lo, Clno,
        V,N,I,L,T,C,F, V1,N1,I1,L1,T1,C1,F1, _) :- !,
    Clause = cx(Cli, Clo, First),
    new_label(Li, Lo, Lab),
    switch_entry(First, Lab(Clno),
        V,N,I,L,T,C,F, V1,N1,I1,L1,T1,C1,F1).
default_sequence_try([cx(Cli,[Lb1|Rest],First)|Cls], [tryme(A,Lb1),Lb0|Cli],
                      Clo, Labi, Labo, Clno,
        V,N,I,L,T,C,F, V2,N2,I2,L2,T2,C2,F2, A) :-
    switch_entry(First, Lb0(Clno), V,N,I,L,T,C,F, V1,N1,I1,L1,T1,C1,F1),
    new_label(Labi, Lab1, Lb0),
    aux_label(Lab1, Lb1),
    'increment%f'(Clno, Clno1),
    default_sequence_retry(Cls, Rest, Clo, Lab1, Labo, Clno1,
        V1,N1,I1,L1,T1,C1,F1, V2,N2,I2,L2,T2,C2,F2, A).

% last clause of a sequence
default_sequence_retry([Clause], [trustme(), Lab| Cli], Clo, Li, Lo, Clno,
        V,N,I,L,T,C,F, V1,N1,I1,L1,T1,C1,F1, _) :- !,
    Clause = cx(Cli, Clo, First),
    new_label(Li, Lo, Lab),
    switch_entry(First, Lab(Clno), V,N,I,L,T,C,F, V1,N1,I1,L1,T1,C1,F1).
% 'middle' clauses of a sequence
default_sequence_retry([cx(Cli,[Lb1|Rest],First)|Cls], [retryme(Lb1), Lb0|Cli],
                        Clo, Labi, Labo, Clno,
        V,N,I,L,T,C,F, V2,N2,I2,L2,T2,C2,F2, Arity) :-
    switch_entry(First, Lb0(Clno), V,N,I,L,T,C,F, V1,N1,I1,L1,T1,C1,F1),
    new_label(Labi, Lab1, Lb0),
    aux_label(Lab1, Lb1),
    'increment%f'(Clno, Clno1),
    default_sequence_retry(Cls, Rest, Clo, Lab1, Labo, Clno1,
        V1,N1,I1,L1,T1,C1,F1, V2,N2,I2,L2,T2,C2,F2, Arity).

/* switch_entry is used during the construction of the default sequence
   to build up the initial table.  This table is then sorted and used as
   the basis of the index code itself.  Note that the list for variables
   is being 'instantiated' rather than 'constructed'.  This is because
   all the other lists have the variable list as their tail.
*/
switch_entry(var, Cl, [Cl|Vars], Nil, Int, Lst, Tpl, Con, Flt,
                        Vars,    Nil, Int, Lst, Tpl, Con, Flt).
switch_entry(nil, Cl, Vars,   Nil,    Int, Lst, Tpl, Con, Flt, 
                      Vars, [Cl|Nil], Int, Lst, Tpl, Con, Flt).
switch_entry(int(I), Lb(Cl), Vars, Nil,      Int,       Lst, Tpl, Con, Flt, 
                             Vars, Nil, [Lb(Cl,I)|Int], Lst, Tpl, Con, Flt).
switch_entry(list, Cl, Vars, Nil, Int,   Lst,    Tpl, Con, Flt,
                       Vars, Nil, Int, [Cl|Lst], Tpl, Con, Flt).
switch_entry(tuple(T), Lb(Cl), Vars, Nil, Int, Lst,      Tpl,       Con, Flt, 
                               Vars, Nil, Int, Lst, [Lb(Cl,T)|Tpl], Con, Flt).
switch_entry(con(C), Lb(Cl), Vars, Nil, Int, Lst, Tpl,      Con,       Flt,
                             Vars, Nil, Int, Lst, Tpl, [Lb(Cl,C)|Con], Flt).
switch_entry(float, Cl, Vars, Nil, Int, Lst, Tpl, Con,    Flt  , 
                        Vars, Nil, Int, Lst, Tpl, Con, [Cl|Flt]).
switch_entry(none, _, V,N,I,L,T,C,F, V,N,I,L,T,C,F).

/* construct_switch actually generates the initial switch and following
   index code and individual switch branches

   Had to split second clause into two parts to overcome limit of 25 local variables.
*/
construct_switch(V, V, V, V, V, V, V, C, C, Lb, Lb, _, _) :- !.
construct_switch(Vars, Nil, Int, Lst, Tpl, Con, Float,
                 [arg_switch(1, Dfl, Lint, Lfloat, Lcon, Lnil, Llst, Ltpl)|Swi], Co,
                 Lb, Lbo, Dfl, Arity) :-
    aux_construct_switch(Vars, Nil, Lst, Fl, Lb, Lb2, Swi, C2, Arity, Lnil, Llst),
    gen_index(Con, Vars, Lcon, Fl, Lb2, Lb3, C2, C3, Arity, indexc),
    gen_index(Int, Vars, Lint, Fl, Lb3, Lb4, C3, C4, Arity, indexi),
    gen_try_all(Float, Vars, Lfloat, Fl, Lb4, Lb5, C4, C5, Arity),
    gen_tpl_index(Tpl, Vars, Ltpl, Fl, Lb5, Lbo, C5, Co, Arity).

aux_construct_switch(Vars, Nil, Lst, Fl, Lb, Lb2, Swi, C2, Arity, Lnil, Llst) :-
    gen_trys(Vars, [], Fl, '?fail?', Lb, Lb0, Swi, C0, Arity),
    gen_trys(Nil, Vars, Lnil, Fl, Lb0, Lb1, C0, C1, Arity),
    gen_trys(Lst, Vars, Llst, Fl, Lb1, Lb2, C1, C2, Arity).



/* gen_index implements the creation of the indexing code for integers.
   The input (Indx, Vars) is a difference list of the clauses with integer
   first argument.  The tail (Vars) is a list of clauses with variable
   first argument.  (gen_index is also used for indexing constants.)
*/
gen_index(Indx, Vars, Lab, Deflt, Li, Lo, 
          [Lab, Ixins(1, Deflt, Xlist)|Ci], Co, Arity, Ixins) :-
    worth_indexing(Indx, Vars, Sorted),
    !,
    new_label(Li, L1, Lab),
    gen_index_trys(Sorted, Vars, Xlist, Deflt, L1, Lo, Ci, Co, Arity).
gen_index(Indx, Vars, Lab, Deflt, Li, Lo, Ci, Co, Arity, _):-
    gen_try_all(Indx, Vars, Lab, Deflt, Li, Lo, Ci, Co, Arity).

/* gen_tpl_index creates indexing code for tuples.  Apart from the usual
   constant functors, the other non-variable functors are grouped together
   and called 'unusuals'.  A separate chain 'all' is used when we have a
   variable functor at run-time.
*/
gen_tpl_index(Tpl, Vlist, Lab, Fl, Li, Lo,
              [Lab, indext(1, Deflt, All, Unusual, Xlist)|Ci], Co, Arity) :-
    separate_functors(Tpl, Vlist, Cons, Vars, Others),
    worth_indexing(Cons, Vars, ConList),
    !,
    new_label(Li, L1, Lab),
    gen_trys(Vars, [], Deflt, Fl, L1, L2, Ci, C1, Arity),
    gen_try_all(Tpl, Vlist, All, Fl, L2, L3, C1, C2, Arity),
    gen_trys(Others, Vars, Unusual, Deflt, L3, L4, C2, C3, Arity),
    gen_index_trys(ConList, Vars, Xlist, Deflt, L4, Lo, C3, Co, Arity).  
gen_tpl_index(Tpl, Vlist, Lab, Fl, Li, Lo, Ci, Co, Arity) :-
    gen_try_all(Tpl, Vlist, Lab, Fl, Li, Lo, Ci, Co, Arity).

/* This splits the list of functors into three categories - con, var and
   unusual.  The lists for con and unusual include the var functors.  All
   three lists also have as their tail the clauses with var first args.
*/
separate_functors([NextLb(NextCl,NextType)|Rest], _, Con, Var, Others) :- !,
    split_functors(NextType, Rest, NextLb, NextCl, Con, Var, Others, Var).
separate_functors(Var, Var, Var, Var, Var).

split_functors(con(C), [NextLb(NextCl,NextType)|Rest], Lb, Cl,
                [Lb(Cl,C)|Cons], Vars, Others, V) :- !,
    split_functors(NextType, Rest, NextLb, NextCl, Cons, Vars, Others, V).
split_functors(con(C), Vars, Lb, Cl, [Lb(Cl,C)|V], Vars, V, V).
split_functors(var, [NextLb(NextCl,NextType)|Rest], Lb, Cl,
                Cons, [Lb(Cl)|Vars], Others, V) :- !,
    split_functors(NextType, Rest, NextLb, NextCl, Cons, Vars, Others, V).
split_functors(var, Vars, Lb, Cl, V, [Lb(Cl)|Vars], V, V).
split_functors(unusual, [NextLb(NextCl,NextType)|Rest], Lb, Cl,
                Cons, Vars, [Lb(Cl)|Others], V) :- !,
    split_functors(NextType, Rest, NextLb, NextCl, Cons, Vars, Others, V).
split_functors(unusual, Vars, Lb, Cl, V, Vars, [Lb(Cl)|V], V).

/* index only if there are at least 2 ground cases and less than 3 vars */
worth_indexing(Cons, Vars, Indx) :-
    makelist(Cons, Vars, ConL),
    qsort(ConL, Sorted, '$ indxless'),
    bunch_up(Sorted, Indx, Vars),
    length(Indx, Il),
    2 =< Il,
    length(Vars, Vl),
    Vl < 3.

/* constructs the list of try sequences for each case in the index */
gen_index_trys([], _, [], _, Lb, Lb, Co, Co, _).
gen_index_trys([[Val|Ilist]|Si], V, [Val(Ilab)|Xlist], Deflt,Li,Lo,Ci,Co,Ar):-
    gen_trys(Ilist, V, Ilab, Deflt, Li, L1, Ci, C1, Ar),
    gen_index_trys(Si, V, Xlist, Deflt, L1, Lo, C1, Co, Ar).

/* recast the labels and generate a sequence of trys */
gen_try_all(Vlist, Vlist, Fl, Fl, L, L, C, C, _) :- !.
gen_try_all(Tpl, Vlist, Lab, Fl, Li, Lo, Ci, Co, Arity) :-
    strip_x_vals(Tpl, Vlist, Tup),
    gen_trys(Tup, Vlist, Lab, Fl, Li, Lo, Ci, Co, Arity).

/* strip_x_vals recasts the list of labels.  The tail of the list
   is already in the correct format. */
strip_x_vals(L, L, L) :- !.
strip_x_vals([Lb(Cl,_)|Indx], L, [Lb(Cl)|Inx]):- !,
    strip_x_vals(Indx, L, Inx).
strip_x_vals([H|Indx], L, [H|Inx]):-
    strip_x_vals(Indx, L, Inx).

/* bunch_up produces a list of lists of equal index values.  The list of 
   variable clauses is appended to the end of the list for each value */
bunch_up([], [], _).
bunch_up([Lb(Cl, Val)|L], [[Val, Lb(Cl)|Vl]|Ll], Vars):-
    bunch(Val, L, Vars, Vl, Lo),
    bunch_up(Lo, Ll, Vars).

bunch(Val, [Lb(Cl, Val)|L], Vars, [Lb(Cl)|Vl], Ll):-  !,
    bunch(Val, L, Vars, Vl, Ll).
bunch(_, L, Vars, Vars, L).



/* gen_trys constructs a sequence of TRY etc. instructions */
gen_trys(L, L, Fail, Fail, Lb, Lb, C, C, _) :- !.
gen_trys([Vclb(_)], _, Vclb, _, Lb, Lb, Co, Co, _) :- !.
gen_trys(List, _, Lab, _, L0, L1, [Lab,try(Arity,Lb)|Ci], Co, Arity) :-
    qsort(List, [Lb(_),Next(_)|Rest], '$ v_less'),
    new_label(L0, L1, Lab),
    gen_try(Rest, Next, Ci, Co).

gen_try([], Vclb, [trust(Vclb)|C], C).
gen_try([Next(_)|Rest], Vclb, [retry(Vclb)|Ci], Co) :-
    gen_try(Rest, Next, Ci, Co).



/* orderings used in sorting */
'$ v_less'(_(Clno1), _(Clno2)) :-
    Clno1 < Clno2.

'$ indxless'(_(_, V1), _(_, V2)):-
    V1@<V2,
    !.
'$ indxless'(_(Cl1, V), _(Cl2, V)):-
    Cl1<Cl2.

/*------------------------------------------------------------------*/

/* Compile a single clause 
   We must analyse the variables in the clause and then
   generate code based on that analysis
*/
'$comp_clause$'(Locals1-(Head:-B),Code,Cout,FirstArg,Li,Lo) :- !,
    ex_cl(Locals1, Head, B, Bdy, Locals),      % extract complex conditions and meta-calls
    split_up_body(Bdy, First, Body),
    analyse_vars(Locals, VarTable, Head, First, Body),
    functor(Head, _, Arity),
    gen_allocate(Body, Al0, Al1),
    get_args(1, Arity, Head, VarTable, Code, C0, Gc0-Gc1+Al0-Al1, Al, 0, S0),
    gen_first_cut(Bdy, C0, C1),       % only if first goal is '!'
    put_allocate(C1, C2, Al, _),      % just in case it hasn't been done already
    compile_body(First, Body, VarTable, C2, Cout, Li, Lo, S0, Space),
    gen_gc_test(Space, Arity, Gc0, Gc1),
    first_arg_type(Head, Locals, FirstArg).
'$comp_clause$'(Locals-Head, Code, Cout, FirstArg, Li, Li) :-
    functor(Head, _, Arity),
    make_v_table(Locals,VarTable),
    ass_get_args(1,Arity,Head,VarTable,C0,C1,0,S0),
    gen_gc_test(S0,Arity,Code,C0),
    fix_vars(VarTable,Arity,LastReg),
    assert_deallocate(LastReg, C1, [succ()|Cout]),
    first_arg_type(Head, Locals, FirstArg).

/* decides whether allocate/deallocate is needed */ 
compile_body(true, true, _, [succ()|C], C, Li, Li, Sp, Sp) :- !.
compile_body(A, true, VarTable, Code, Cout, Li, Lo, S0, S) :- !,
    comp_condition(A, 1, VarTable, Code, Cout, Li, Lo, S0, S, last).
compile_body(A, B, VarTable, Code, Cout, Li, Lo, S0, S) :-
    comp_body((A,B), 1, VarTable, Code, Cout, Li, Lo, S0, S).

/* comp_body compiles a sequence of calls in the body of a clause */
comp_body((A,B), Pos, VarTable, Code, Cout, Li, Lo, S0, S):-!,
    comp_condition(A, Pos, VarTable, Code, C0, Li, Lj, S0, S1, notlast),
    'increment%f'(Pos, NP),
    comp_body(B, NP, VarTable, C0, Cout, Lj, Lo, S1, S).
comp_body(A, Pos, VarTable, Code, Cout, Li, Lo, S0, S):-
    comp_condition(A, Pos, VarTable, Code, Cout, Li, Lo, S0, S, deallocate).

/* comp_condition compiles a single call in the body of a clause */
comp_condition(true, _, _, [deallocate(),succ()|C], C, Li, Li, S, S, deallocate) :- !.
comp_condition(true, _, _, C, C, Li, Li, S, S, _) :- !.
comp_condition(otherwise, _, _, [deallocate(),succ()|C], C, Li, Li, S, S, deallocate) :- !.
comp_condition(otherwise, _, _, C, C, Li, Li, S, S, _) :- !.
comp_condition(!, _, _, [cut()|Code], Code, Li, Li, S, S, notlast) :- !.
comp_condition(!, _, _, [deallocate(),cut(),succ()|Code], Code, Li, Li, S, S, deallocate) :- !.
comp_condition(fail, _, _, [fail()|Code], Code, Li, Li, S, S, _) :- !.
comp_condition(false, _, _, [fail()|Code], Code, Li, Li, S, S, _) :- !.
comp_condition('$arith$'(Arith,Is,_,OV), Pos, VarTable, Code, Cout, Li, Lo, S0, S, Where):-
        !,
        all_subseq_var(OV, VarTable, Is),
        length(Arith, Len),
        'mul%f'(Len, 4, Space),
        S1 is S0 + Space,
        new_label(Li, L1, Label1),
        Code = [set_err(Label1)|Code1],
        comp_arith_body(Arith, VarTable, Code1, Code2, Pos, S1, S),
        new_label(L1, L2, Label2),
        Code2 = [clr_err(), go_to(Label2), Label1|Code3],
        comp_condition('$ arith_err'(Is), Pos, VarTable, Code3, [Label2|Code4], L2, Lo, 0, _, notlast),
        fix_arith_end(Where, Code4, Cout).
comp_condition(Cond, Pos, VarTable, [put_slash(1)|Code], Cout, Li, Lo, S0, S, Where):-
    functor(Cond, '?sb?'(Cls), Arity), !,
    put_args(2, Arity, Cond, VarTable, Pos, Code, C0, S0, S),
    gen_deallocate(Where, C1, C2, Last),
    comp_jsr(Last, Cls, Arity, Pos, VarTable, C0, C1, C2, Cout, Li, Lo).
comp_condition(Cond, Pos, VarTable, Code, Cout, Li, Lo, S0, S, Where):-
    functor(Cond, Pred, Arity),
    put_args(1, Arity, Cond, VarTable, Pos, Code, C0, S0, S),
    gen_deallocate(Where, C1, C2, Last),
    comp_jsr(Last, Pred, Arity, Pos, VarTable, C0, C1, C2, Cout, Li, Lo).

comp_arith_body([], _, Code, Code, _, S, S).
comp_arith_body([Cond|T], VarTable, Code, Cout, Pos, S0, S) :-
        functor(Cond, Pred, Arity),
        '$ escape_code'(Pred, Arity, Esc),
        put_args(1, Arity, Cond, VarTable, Pos, Code, [escape(Esc)|Co], S0, S1),
        comp_arith_body(T, VarTable, Co, Cout, Pos, S1, S).

all_subseq_var([], _, _).
all_subseq_var([Var|Rest], VarTable, Is) :-
        is_a_var(Var,VarTable,_,_,_,First),
        nonvar(First), !,
        all_subseq_var(Rest, VarTable, Is).
all_subseq_var([Var|_], _, Is) :-
        writeseqnl(user_error, ['\nERROR:', Var, 'is unbound in arithmetic expression', Is]),
        fail.

fix_arith_end(notlast, C, C).
fix_arith_end(last, [succ()|C], C).
fix_arith_end(deallocate, [deallocate(),succ()|C], C).

% generate the appropriate call instruction
comp_jsr(notlast, Var, Arity, Pos, VarTable, Code, C0, C1, C, Li, Lo) :-
    body_jsr(Var, Var, Arity, Pos, VarTable, Code, C0, C1, C, Li, Lo).
comp_jsr(last, Var, Arity, Pos, VarTable, Code, C0, C1, C, Li, Lo) :-
    last_jsr(Var, Var, Arity, Pos, VarTable, Code, C0, C1, C, Li, Lo).

% cases where it is not the last goal ...
% note : arguments 2, 7 and 8 are place holders only

% thread in a local jsr
% NOTE : a data instruction is inserted to follow the convention for
% accessing the local environment size etc. (ie. a fixed offset before
% the continuation label.
body_jsr([CL|Cses], _, Arity, Pos, VarT, [callme(Lb,Ln,Gc)|Ci],
         _, _, Cout, Li, Lo) :- !,
    comp_local_clauses(tryme(Arity),[CL|Cses],Ci,[data(Ln,Gc),Lb|Cout],Li,Lj),
    env_length(VarT, Pos, 0, Ln, 0, Gc),
    new_label(Lj, Lo, Lb).
% variable predicate symbol...
body_jsr(Var, _, Arity, Pos, VarTable, Code, _, _, C, Li, Li) :-
    atom(Var),
    is_a_var(Var, VarTable, Type, LastOcc, Index, First), !,
    put_var_arg(Type, Index, 0, First, LastOcc, Pos, Code, [callv(Arity,Ln,Gc)|C], 0,_),
    env_length(VarTable, Pos, 0, Ln, 0, Gc).
% foreign function...
body_jsr(Foreign, _, Arity, _, _, Code, _, _, Cd, Li, Li) :-
    '$ escape_code'(Foreign, Arity, Esc), !,
    Code = [escape(Esc)|Cd].
% normal code...
body_jsr(P, _, A, Pos, VarTable, [call(P/A,Ln,Gc)|C], _, _, C, Li, Li):-
    env_length(VarTable, Pos, 0, Ln, 0, Gc).

% cases where it is the last goal ...
% note : argument 2 is a place holder only

% thread in a local jsr
% in the following clause, we need to force a 'jmp' instruction (to set SB)
last_jsr([CL|Cses], _, Arity, _, _, Code, Code, [jmp(Lb),Lb|C1], Cout, Li, Lo):- !,
    new_label(Li,Lj,Lb),
    comp_local_clauses(tryme(Arity), [CL|Cses], C1, Cout, Lj, Lo).
% variable predicate symbol...
last_jsr(Var, _, Arity, Pos, VarTable, Code, C0, C1, C, Li, Li) :-
    atom(Var),
    is_a_var(Var, VarTable, Type, LastOcc, Index, First), !,
    put_var_arg(Type, Index, 0, First, LastOcc, Pos, Code, C0, 0, _),
    C1 = [executev(Arity)|C].
% foreign function...
last_jsr(Foreign, _, Arity, _, _, Ci, Ci, Code, Co, Li, Li) :-
    '$ escape_code'(Foreign, Arity, Esc), !,
    Code = [escape(Esc),succ()|Co].
% normal code...
last_jsr(P, _, A, _, _, Ci, Ci, [execute(P/A)|Co], Co, Li, Li).



comp_local_clauses(_, [], Cx, Cx, Li, Li) :- !.
comp_local_clauses(tryme(_), [Cl], Cx, Cy, Li, Lo):- !,
    '$comp_clause$'(Cl, Cx, Cy, _, Li, Lo).
comp_local_clauses(tryme(Ar), [Cl|Clauses], [tryme(Ar,Lb)|Cx], Cz, Li, Lo):- !,
    '$comp_clause$'(Cl, Cx, [Lb|Cy], _, Li, Lj),
    new_label(Lj, Lk, Lb),
    comp_local_clauses(retryme, Clauses, Cy, Cz, Lk, Lo).
comp_local_clauses(retryme, [Cl], [trustme()|Cx], Cy, Li, Lo):- !,
    '$comp_clause$'(Cl, Cx, Cy, _, Li, Lo).
comp_local_clauses(retryme, [Cl|Clauses], [retryme(Lb)|Cx], Cz, Li, Lo):-
    '$comp_clause$'(Cl, Cx, [Lb|Cy], _, Li, Lj),
    new_label(Lj, Lk, Lb),
    comp_local_clauses(retryme, Clauses, Cy, Cz, Lk, Lo).

/* env_length computes the remaining length of the local environment
   and generates a usage bitmap
*/
env_length([],_,Length,Length, Map, Map).
env_length([_(Last,Type,Ix,First)|V],Pos,L,Length, Bit, Map):-
    on(Type, [safe, unsafe]),
    Pos < Last, !,
    'increment%f'(Ix, Index),
    max(L,Index,Len),
    set_map_bit(First, Ix, Bit, Mp),
    env_length(V,Pos,Len,Length, Mp, Map).
env_length([_|V],Pos,L,Length, Bit, Map):-
    env_length(V,Pos,L,Length, Bit, Map).

/* changed 32 to 26 in next clause because of 27-bit integers */ 
set_map_bit(F, Ix, M, Map):-
    nonvar(F), Ix < 26, !,
    Map is M\/(1<<Ix).
set_map_bit(_, _, Map, Map).



/* get_args.
   Pos, Arity are the current register and max register
   Head is the predication to compile
   VarTable is the variable lookup table
*/
get_args(Pos,Arity,_,_,Code,Code,A,A,S,S):-
    Pos>Arity,!.
get_args(Pos,Arity,Head,VarTable,Code,Cdout,Al,A,S,Space):-
    arg(Pos,Head,Argument),
    'tag%f'(Argument,N),
    get_arg(N,Argument,Pos,VarTable,Code,C0,Al,A0,S,S0),
    'increment%f'(Pos, NP),
    get_args(NP,Arity,Head,VarTable,C0,Cdout,A0,A,S0,Space).

/* get_arg handles one argument at a time. */
get_arg(1, Int, Reg, _, [get_int(Reg, Int)|Code], Code,A,A,S,S).
get_arg(2, Float, Reg, _, [get_float(Reg, Float)|Code], Code,A,A,S,S).
get_arg(3, Var, Reg, Vars, Code, Co,Al,A,S,S):-
    is_a_var(Var, Vars, Type, _, Index, First),
    !,
    get_var_arg(Type, Index, Reg, First, Code, Co,Al,A).
get_arg(3, Con, Reg, _, [get_const(Reg, Con)|Code], Code,A,A,S,S).
get_arg(4, [], Reg, _, [get_nil(Reg)|Code], Code,A,A,S,S).
get_arg(5, [H|T], Reg, Vars, Code, Cout,Al,A,S,Space):-
    put_gc_test(Code, [get_list(Reg)|C0], Al, Al0),
    S0 is S++2,
    gen_push_pop(outer, H, C0, C1, C2, C3),    % generate a push inst'n if nec.
    'tag%f'(H, HN),
    get_term(HN, H, Vars, C1, C2, Al0, Al1, S0, S1, inner),
    'tag%f'(T, TN),
    get_term(TN, T, Vars, C3, Cout, Al1, A, S1, Space, outer).
get_arg(6, Tuple, Reg, Vars, Code, Cdout,Al,A,S,Space):-
    'arity%f'(Tuple, Count),
    put_gc_test(Code, [get_tpl(Reg, Count)|C0], Al, Al0),
    S0 is S++Count++1,
    get_tuple(Tuple, 1, Count, Vars, C0, Cdout,Al0,A,S0,Space,outer).

/* get_var_arg analyse what kind of variable is involved, and what code to
    generate */
get_var_arg(void, _,     _,   _,     Code, Code,A,A).
get_var_arg(safe, Indx,  Reg, First, Code, Co,Al,A):-
    var(First), !,
    First=local,
    put_allocate(Code,[get_y_var(Reg,Indx)|Co],Al,A).
get_var_arg(safe, Index, Reg, _,     [get_y_val(Reg,Index)|Co], Co,A,A).
get_var_arg(reg,  Reg,   Reg, _,     Code, Code,A,A):-!.
% must come after gc instr'n because it might use a temporary register
get_var_arg(reg,  Index, Reg, First, Code, Co,Al,A):-
    var(First), !,
    First=local,
    put_gc_test(Code, [get_x_var(Reg,Index)|Co], Al, A).
get_var_arg(reg,  Index, Reg, _,     [get_x_val(Reg,Index)|Co], Co,A,A).
  
/* get_term compiles an inner term argument using unify_ instructions */
get_term(1, Int,_,[unify_int(Int)|Code], Code,A,A,S,S,_).
get_term(2, Float,_,[unify_float(Float)|Code], Code,A,A,S,S,_).
get_term(3, Var,VarTable,Code,Cout,Al,A,S,S,_):-
    is_a_var(Var, VarTable, Type,_, Index, First),
    !,
    get_var_term(Type,First,Index,Code,Cout,Al,A).
get_term(3, Con,_,[unify_const(Con)|Code],Code,A,A,S,S,_).
get_term(4, [],_,[unify_nil()|Code],Code,A,A,S,S,_).
get_term(5, [Head|Tail],VarTable,[unify_list()|Code],Cout,Al,A,S,Space,Where):-
    S0 is S++2,
    gen_push_pop(Where, Head, Code, C1, C2, C3),    % generate a push instr'n if nec.
    'tag%f'(Head, HN),
    get_term(HN, Head, VarTable, C1, C2,Al,A0,S0,S1,inner),
    'tag%f'(Tail, TN),
    get_term(TN, Tail, VarTable, C3, Cout,A0,A,S1,Space,Where).
get_term(6, Tuple,VarTable,[unify_tpl(Count)|Code],Cout,Al,A,S,Space,Where):-
    'arity%f'(Tuple, Count),
    S0 is S++Count++1,
    get_tuple(Tuple, 1, Count, VarTable, Code, Cout,Al,A,S0,Space,Where).

get_tuple(Tuple, Pos, Arity, VarTable, Code, Cout,Al,A,S,Space,Where):-
    get_tuple_args(Tuple, Pos, Arity, VarTable, C, [],Al,A,S,Space,Where),
    merge_unify_voids(C, Code, Cout).

get_tuple_args(Tuple, Arity, Arity, VarTable, Code, Cout,Al,A,S,Space,Where):- !,
    '$ nth'(Tuple,Arity,Term),
    'tag%f'(Term, N),
    get_term(N,Term,VarTable,Code,Cout,Al,A,S,Space,Where).
get_tuple_args(Tuple, Pos, Arity, VarTable, Code, Cout,Al,A,S,Space,Where):-
    '$ nth'(Tuple,Pos,Term),
    gen_push_pop(Where, Term, Code, C1, C2, C3),
    'tag%f'(Term, N),
    get_term(N, Term, VarTable, C1, C2,Al,A0,S,S0,inner),
    'increment%f'(Pos, NP),
    get_tuple_args(Tuple, NP, Arity, VarTable, C3, Cout,A0,A,S0,Space,Where).

merge_unify_voids([], Cout, Cout).
merge_unify_voids([unify_void(1)|Rest], Code, Cout) :- !,
    count_unify_voids(Rest, Code, Cout, 1).
merge_unify_voids([Other|Rest], [Other|Code], Cout) :-
    merge_unify_voids(Rest, Code, Cout).

count_unify_voids([], [unify_void(N)|Cout], Cout, N).
count_unify_voids([unify_void(1)|Rest], Code, Cout, N) :- !,
    'increment%f'(N, M),
    count_unify_voids(Rest, Code, Cout, M).
count_unify_voids([Other|Rest], [unify_void(N), Other|Code], Cout, N) :-
    merge_unify_voids(Rest, Code, Cout).

get_var_term(void,   _,      _,    [unify_void(1)|Code],Code,A,A).
get_var_term(reg,    First,  Indx, [unify_x_var(Indx)|Code],Code,A,A) :-
    var(First), !,
    First=global.
get_var_term(reg,    local,  Indx, [unify_loc_x_val(Indx)|Code],Code,A,A) :- !.
get_var_term(reg,    global, Indx, [unify_x_val(Indx)|Code],Code,A,A).
% the next two cases should never happen
% get_var_term(unsafe, local,  Indx, [unify_loc_y_val(Indx)|Code],Code,A,A) :- !.
% get_var_term(unsafe, global, Indx, [unify_y_val(Indx)|Code],Code,A,A).
get_var_term(safe,   First,  Indx, Code,C,Al,A) :-
    var(First),!,
    First=global,
    put_allocate(Code,[unify_y_var(Indx)|C],Al,A).
get_var_term(safe,   local,  Indx, [unify_loc_y_val(Indx)|Code],Code,A,A) :- !.
get_var_term(safe,   global, Indx, [unify_y_val(Indx)|Code],Code,A,A).



/* put_args constructs the arguments for body goals */
put_args(Arg, Arity, _, _, _, Code, Code, S, S):-
    Arg>Arity,!.
put_args(Arg, Arity, Cond, VarTable, Pos, Code, Cdout, S, Space):-
    arg(Arg, Cond, Argument),
    'tag%f'(Argument, N),
    put_arg(N, Argument, Arg, VarTable, Pos, Code, C0, S, S0),
    'increment%f'(Arg, NArg),
    put_args(NArg, Arity, Cond, VarTable, Pos, C0, Cdout, S0, Space).

/* put_arg handles one argument at a time */
put_arg(1, Int, Reg, _, _, [put_int(Int,Reg)|Code], Code, S, S).
put_arg(2, Float, Reg, _, _, [put_float(Float,Reg)|Code], Code, S, S).
put_arg(3, Var, Reg, Vars, Pos, Code, Co, S, Space):-
    is_a_var(Var, Vars, Type, LastOcc, Index, First),
    !,
    put_var_arg(Type, Index, Reg, First, LastOcc, Pos, Code, Co, S, Space).
put_arg(3, Con, Reg, _, _, [put_const(Con,Reg)|Code], Code, S, S).
put_arg(4, [], Reg, _, _, [put_nil(Reg)|Code], Code, S, S).
put_arg(5, [H|T], Reg, Vars, _, [put_list(Reg)|Code], Cout, S, Space):-
    S0 is S++2,
    gen_push_pop(outer, H, Code, C1, C2, C3),    % generate a push inst'n if nec.
    'tag%f'(H, HN),
    put_term(HN, H, Vars, C1, C2,S0,S1,inner),
    'tag%f'(T, TN),
    put_term(TN, T, Vars, C3, Cout,S1,Space,outer).
put_arg(6, Tuple, Reg, Vars, _, [put_tpl(Count,Reg)|Code], Co, S, Space):-
    'arity%f'(Tuple, Count),
    S0 is S++Count++1,
    put_tuple(Tuple, 1, Count, Vars, Code, Co, S0, Space,outer).

/* put_var_arg analyse what kind of variable is involved, and what code to
    generate */
put_var_arg(void,  _,    Reg,_,    _, _, [put_void(Reg)|Code],Code,S,Space) :-
    'increment%f'(S, Space).
% the next case should never happen
%put_var_arg(safe,  Index,Reg,First,_, _, [put_y_var(Index,Reg)|Co],Co,S,S) :-
%    var(First), !,
%    First=local.
put_var_arg(safe,  Index,Reg,_,    _, _, [put_y_val(Index,Reg)|Co],Co,S,S).
put_var_arg(reg,   Reg,  Reg,Used, _, _, Code,Code,S,S):-
    nonvar(Used), !.
put_var_arg(reg,   Reg,  Reg,First,_, _, [put_void(Reg)|Co],Co,S,Space):-
    var(First), !,
    First=global,
    'increment%f'(S, Space).
put_var_arg(reg,   Index,Reg,First,_, _, [put_x_var(Index,Reg)|Co],Co,S,Space):-
    var(First), !,
    First=global,
    'increment%f'(S, Space).
put_var_arg(reg,   Index,Reg,_,    _, _, [put_x_val(Index,Reg)|Co],Co,S,S).
put_var_arg(unsafe,Index,Reg,First,_, _, [put_y_var(Index,Reg)|Co],Co,S,S):-
    var(First), !,
    First=local.
put_var_arg(unsafe,Index,Reg,_,    Lo,Lo,[put_unsafe_y(Index,Reg)|Co],Co,S,Space):- !,
    'increment%f'(S, Space).
put_var_arg(unsafe,Index,Reg,_,    _, _, [put_y_val(Index, Reg)|Co],Co,S,S).


  
/* put_term compiles an inner term argument using set_ instructions */
put_term(1, Int,_,[set_int(Int)|Code], Code,S,S,_).
put_term(2, Float,_,[set_float(Float)|Code], Code,S,S,_).
put_term(3, Var,VarTable,[Instr|C],C,S,S,_):-
    is_a_var(Var, VarTable, Type,_, Index, First),
    !,
    put_var_term(Type,First,Index,Instr).
put_term(3, Con,_,[set_const(Con)|Code],Code,S,S,_).
put_term(4, [],_,[set_nil()|Code],Code,S,S,_).
put_term(5, [Head|Tail],VarTable,[set_list()|Code],Cout,S,Space,Where):-
    S0 is S++2,
    gen_push_pop(Where, Head, Code, C1, C2, C3),    % generate a push instr'n if nec.
    'tag%f'(Head, HN),
    put_term(HN, Head, VarTable, C1, C2,S0,S1,inner),
    'tag%f'(Tail, TN),
    put_term(TN, Tail, VarTable, C3, Cout,S1,Space,Where).
put_term(6, Tuple,VarTable,[set_tpl(Count)|Code],Cout,S,Space,Where):-
    'arity%f'(Tuple, Count),
    S0 is S++Count++1,
    put_tuple(Tuple, 1, Count, VarTable, Code, Cout,S0,Space,Where).

put_tuple(Tuple, Pos, Arity, VarTable, Code, Cout,S,Space,Where):-
    put_tuple_args(Tuple, Pos, Arity, VarTable, C, [],S,Space,Where),
    merge_set_voids(C, Code, Cout).

put_tuple_args(Tuple, Arity, Arity, VarTable, Code, Cout,S,Space,Where):- !,
    '$ nth'(Tuple,Arity,Term),
    'tag%f'(Term, N),
    put_term(N, Term,VarTable,Code,Cout,S,Space,Where).
put_tuple_args(Tuple, Pos, Arity, VarTable, Code, Cout,S,Space,Where):-
    '$ nth'(Tuple,Pos,Term),
    gen_push_pop(Where, Term, Code, C1, C2, C3),
    'tag%f'(Term, N),
    put_term(N, Term, VarTable, C1, C2,S,S0,inner),
    'increment%f'(Pos, NP),
    put_tuple_args(Tuple, NP, Arity, VarTable, C3, Cout,S0,Space,Where).

merge_set_voids([], Cout, Cout).
merge_set_voids([set_void(1)|Rest], Code, Cout) :- !,
    count_set_voids(Rest, Code, Cout, 1).
merge_set_voids([Other|Rest], [Other|Code], Cout) :-
    merge_set_voids(Rest, Code, Cout).

count_set_voids([], [set_void(N)|Cout], Cout, N).
count_set_voids([set_void(1)|Rest], Code, Cout, N) :- !,
    'increment%f'(N, M),
    count_set_voids(Rest, Code, Cout, M).
count_set_voids([Other|Rest], [set_void(N), Other|Code], Cout, N) :-
    merge_set_voids(Rest, Code, Cout).

put_var_term(void,   _,      _,    set_void(1)).
put_var_term(reg,    First,  Indx, set_x_var(Indx)):-
    var(First), !,
    First=global.
put_var_term(reg,    local,  Indx, set_loc_x_val(Indx)) :- !.
put_var_term(reg,    global, Indx, set_x_val(Indx)).
put_var_term(unsafe, local,  Indx, set_loc_y_val(Indx)) :- !.
% the next case should never happen
% put_var_term(unsafe, global, Indx, set_y_val(Indx)).
put_var_term(safe,   First,  Indx, set_y_var(Indx)):-
    var(First), !,
    First=global.
put_var_term(safe,   local,  Indx, set_loc_y_val(Indx)) :- !.
put_var_term(safe,   global, Indx, set_y_val(Indx)).



/* determine the first goal of the body, excluding ! and true */
split_up_body((!,Body),First,Rest):-!,
    split_up_body(Body,First,Rest).
split_up_body((true,Body),First,Rest):-!,
    split_up_body(Body,First,Rest).
split_up_body(!,true,true):-!.
split_up_body((Arith,Body),true,(Arith,Body)):-
        Arith = '$arith$'(_,_,_,_), !.
split_up_body((First,Body),First,Body):-!.
split_up_body(Arith,true,Arith) :-
        Arith = '$arith$'(_,_,_,_), !.
split_up_body(First,First,true).

/* determine the type of the first arg (for indexing) */
first_arg_type(Head, Varslist, FirstArg) :-
    arg(1,Head,F), !,
    'tag%f'(F, N),
    analyse_first_arg(N, F,Varslist,FirstArg).
first_arg_type(_, _, none).

/* analyse first arg returns the type of the first argument for indexing etc. */
analyse_first_arg(1, Int, _, int(Int)).
analyse_first_arg(2, _, _, float).
analyse_first_arg(3, Var, Varslist, var):-
    on(Var, Varslist), !.
analyse_first_arg(3, Con, _, con(Con)).
analyse_first_arg(4, [], _, nil).
analyse_first_arg(5, _, _, list).
analyse_first_arg(6, Tuple, Varslist, tuple(Type)):-
    functor(Tuple, Functor, Arity),
    analyse_functor(Functor, Arity, Varslist, Type).

analyse_functor(Var, _, Varslist, var) :-
    on(Var, Varslist), !.
analyse_functor(Con, Arity, _, con(Con/Arity)) :-
    'atom%f'(Con), !.
analyse_functor(_, _, _, unusual).

/* only need a gc instruction if space is required */
gen_gc_test(0, _, [gc0()|Code], Code):-!.
gen_gc_test(Space, Arity, [gc(Space,Arity)|Code], Code).

/* insert the gc test here, but leave the 'allocate' instr'n till later */
put_gc_test(CodeIn, CodeOut, CodeIn-CodeOut+A-A1, G-G+A-A1).

/* inserts a cut instruction if it is the first goal */
gen_first_cut(!, [cut()|C], C) :- !.
gen_first_cut((!,_), [cut()|C], C) :- !.
gen_first_cut((true,Body), C1, C2) :- !,
    gen_first_cut(Body, C1, C2).
gen_first_cut(_, C, C).

/* allocate if >1 body goal */
gen_allocate(true, C, C) :- !.
gen_allocate(_, [allocate()|C], C).

/* this is where the actual insertion is performed.  Empty structures
   are returned so that subsequent calls to put_allocate do nothing.
   Note that the gc test (if any) must come before the allocate instr'n. */
put_allocate(CodeIn, CodeOut, CodeIn-C+C-CodeOut, G-G+A-A).

/* only generate a deallocate instruction if this is
   the last goal and not the only goal in the body */
gen_deallocate(deallocate, [deallocate()|C], C, last) :- !.
gen_deallocate(Last, C, C, Last).

/* gen_push_pop determines if a push and pop sequence
   is needed to preserve the unification structure  */
gen_push_pop(_, Term, C, C, O, O):-
    atomic(Term), !.
gen_push_pop(outer, _, [f_push_t()|C], C, [pop_t()|O], O) :- !. 
gen_push_pop(inner, _, [push_t()|C], C, [pop_t()|O], O).

/*------------------------------------------------------------------*/

/*
   transform a clause containing complex conditions such as disjunction etc.
   The complex conditions are replaced by special sets of clauses.
*/
ex_cl(V, H, B, Body, Vout):-
        tran(B, V, (H,B), outer, Body, [], NewVars),
        append(NewVars, V, Vout).

tran(((A,B),C), V, Term, Level, Call, NVI, NVO) :- !,
    tran((A,B,C), V, Term, Level, Call, NVI, NVO).
tran((A,B), V, Term, Level, (A1,B1), NVI, NVO) :- !,
    tran(A, V, Term, Level, A1, NVI, NV1),
    tran(B, V, Term, Level, B1, NV1, NVO).
tran(!, _, _, inner, 'deepcut%f'('?bk?'), NV, NV) :- !.
tran(!, _, _, outer, !, NV, NV) :- !.
tran((A;B), V, Term, Level, Call, NVI, NVO) :- !,
    var_analyse((A;B), V, Term, Vars, Shared),
    '%=..%'(Head, [[], '?bk?'|Shared]),
    tran_disjunction((A;B), Vars, Term, Disjuncts, [], NVI, NVO),
    gen_clauses(Disjuncts, Head, ['?bk?'|Vars], Clauses),
    construct_call(Level, Clauses, Shared, Call).
tran((A->B), V, Term, Level, Call, NVI, NVO) :- !,
    var_analyse((A->B), V, Term, Vars, Shared),
    '%=..%'(Head, [[], '?bk?'|Shared]),
    construct_call(Level, [['?bk?'|Vars]-(Head:-If,!,Then)], Shared, Call),
    tran(A, Vars, Term, inner, If, NVI, NV1),
    tran(B, Vars, Term, inner, Then, NV1, NVO).
tran(is(Val, Exp), _, _, _, Code, NV, NV) :-
	integer(Exp), !,
	Code = (Val = Exp).
tran(is(Val, Exp), V, _, _, '$arith$'(GL,is(Val, Exp),NV,OV), NVI, NVO) :-
        not((atom(Exp), on(Exp,V))), !,
        '$ compile_expr'(Exp, Val, V, L, [], [], OV),
        '?toground?'((L,NVI,V), GT, _, _, NV),
        GT = (GL,_,_),
        append(NV, NVI, NVO).
tran(\+(A), V, Term, Level, Call, NVI, NVO) :- !,
    var_analyse(\+(A), V, Term, Vars, Shared),
    '%=..%'(Head, [[], '?bk?'|Shared]),
    construct_call(Level, [['?bk?'|Vars]-(Head:-Not,!,fail),
                           ['?bk?'|Vars]-Head], Shared, Call),
    tran(A, Vars, Term, inner, Not, NVI, NVO).
tran(not(A), V, Term, Level, Call, NVI, NVO) :- !,
    var_analyse(not(A), V, Term, Vars, Shared),
    '%=..%'(Head, [[], '?bk?'|Shared]),
    construct_call(Level, [['?bk?'|Vars]-(Head:-Not,!,fail),
                           ['?bk?'|Vars]-Head], Shared, Call),
    tran(A, Vars, Term, inner, Not, NVI, NVO).
tran(G, V, _, Level, Call, NV, NV) :-
    atom(G),
    on(G, V), !,
    construct_call(Level, '?call?', [G], Call).
tran(G, _, _, _, G, NV, NV).

tran_disjunction((A;B), Vars, Term, G0, G, NVI, NVO) :- !,
    tran_disjunction(A, Vars, Term, G0, G1, NVI, NV1),
    tran_disjunction(B, Vars, Term, G1, G, NV1, NVO).
tran_disjunction((A->B), Vars, Term, [(If,!,Then)|Rest], Rest, NVI, NVO) :- !,
    tran(A, Vars, Term, inner, If, NVI, NV1),
    tran(B, Vars, Term, inner, Then, NV1, NVO).
tran_disjunction(A, Vars, Term, [Call|Rest], Rest, NVI, NVO) :-
    tran(A, Vars, Term, inner, Call, NVI, NVO).

gen_clauses([], _, _, []).
gen_clauses([H|T], Head, Vars, [Vars-(Head:-H)|Clauses]) :-
    gen_clauses(T, Head, Vars, Clauses).

/*
'Subvars' is the set of variables in 'Sub',
'Shared' is the set of shared variables in 'Sub' and 'Term'.
*/
var_analyse(Term, Vars, Term, Subvars, Subvars) :- !,
    extract_vars(Term, Vars, Subvars).
var_analyse(Sub, Vars, Term, Subvars, Shared) :-
    replace(Term, Sub, NewTerm),
    extract_vars(Sub, Vars, Subvars),
    extract_vars(NewTerm, Vars, Othervars),
    intersection(Subvars, Othervars, Shared).

replace(T, T, []) :- !.
replace((A,B), T, (A1,B)) :-
    replace(A, T, A1), !.
replace((A,B), T, (A,B1)) :-
    replace(B, T, B1).
replace((A;B), T, (A1,B)) :-
     replace(A, T, A1), !.
replace((A;B), T, (A,B1)) :-
     replace(B, T, B1).
replace((A->B), T, (A1,B)) :-
     replace(A, T, A1), !.
replace((A->B), T, (A,B1)) :-
     replace(B, T, B1).
replace(\+ A, T, A1) :-
     replace(A, T, A1).
replace(not A, T, A1) :-
     replace(A, T, A1).


extract_vars(Term, Names, Vars) :-
    tohollow(Term, HTerm, Names, V),
    varsin(HTerm, Set),
    filter_vars(Names, V, Set, Vars).

filter_vars([], _, _, []).
filter_vars([Name|Names], [V|Vars], Set, [Name|Rest]) :-
    on(Element, Set),
    V == Element, !,
    filter_vars(Names, Vars, Set, Rest).
filter_vars([_|Names], [_|Vars], Set, Rest) :-
    filter_vars(Names, Vars, Set, Rest).

intersection([], _, []).
intersection([H|T], L, [H|Rest]) :-
    on(H, L), !,
    intersection(T, L, Rest).
intersection([_|T], L, Rest) :-
    intersection(T, L, Rest).

construct_call(inner, Functor, Args, Call) :-
    '%=..%'(Call, [Functor, '?bk?'|Args]).
construct_call(outer, Functor, Args, Call) :-
    '%=..%'(Call, ['?sb?'(Functor), '?bk?'|Args]).

/*------------------------------------------------------------------*/

/* in this window we analyse the occurrences of variables in a clause */

/* the output is a table giving, for each variable, a list of all 
   its occurrences in the head, first call and body of a clause
    together with an assignment of variable name to register/permanent/label
    location */

analyse_vars([],[],_,_,_) :- !.
analyse_vars(Locals,VarTable,Head,First,Body):-
    make_v_table(Locals,Table),
    find_arg_vars(Head,Table,head_arg),
    find_arg_vars(First,Table,first_arg),
    find_body_vars(Body,Table,2),
    functor(Head,_,Hc),
    functor(First,_,Fc),
    max(Hc,Fc,Max),
    'increment%f'(Max, Free),
    allocate_vars(Table,VarTable,Free,Hc,Perms,[],0,NoPerms,_),
    qsort(Perms,Sorted,'$ compare_perms'),
    allocate_perms(Sorted,NoPerms).

make_v_table([],[]).
make_v_table([V|R],[V(_)|T]):-
    make_v_table(R,T).

find_arg_vars(Head,Table,Where):-
    functor(Head,Pred,Arity),
    find_args_vars(1,Arity,Table,Head,Where),
    'tag%f'(Pred, N),
    find_var(N, Pred,Table,Where,0).

find_args_vars(Pos,Arity,_,_,_):-
    Pos>Arity,!.
find_args_vars(Pos,Arity,Table,Compound,Where):-
    arg(Pos,Compound,Term),
    'tag%f'(Term, N),
    find_var(N, Term,Table,Where,Pos),
    'increment%f'(Pos, NP),
    find_args_vars(NP,Arity,Table,Compound,Where).

find_body_vars(true,_,_) :- !.
find_body_vars((Atom,Body),Table,Ix):-!,
    find_atom_vars(Atom,Table,Ix,body_arg),
    'increment%f'(Ix, Ix1),
    find_body_vars(Body,Table,Ix1).
find_body_vars(Atom,Table,Ix):-
    find_atom_vars(Atom,Table,Ix,last_arg).

find_atom_vars('$arith$'(_,is(Val,_),NewV,OldV),Table,Ix,Where):- !,
        '%=..%'(DummyGoal, [[],NewV,OldV,Val|NewV]),
        functor(DummyGoal,_,Arity),
        find_atom_args_vars(1,Arity,Table,DummyGoal,Where,Ix).
find_atom_vars(Head,Table,Ix,Where):-
    functor(Head,Pred,Arity),
    find_atom_args_vars(1,Arity,Table,Head,Where,Ix),
    'tag%f'(Pred, N),
    find_var(N, Pred,Table,Where,Ix/0).

find_atom_args_vars(Pos,Arity,_,_,_,_):-
    Pos>Arity,!.
find_atom_args_vars(Pos,Arity,Table,Compound,Where,Ix):-
    arg(Pos,Compound,Term),
    'tag%f'(Term, N),
    find_var(N, Term,Table,Where,Ix/Pos),
    'increment%f'(Pos, NP),
    find_atom_args_vars(NP,Arity,Table,Compound,Where,Ix).

find_var(1, _,_,_,_).
find_var(2, _,_,_,_).
find_var(3, Con,Table,Where,Ix) :-
    record_occ(Con,Table,Where,Ix).
find_var(4, [],_,_,_).
find_var(5, [Head|Tail],Table,Where,Ix):-
    inner_of(Where,Inner),
    'tag%f'(Head, HN),
    find_var(HN, Head,Table,Inner,Ix),
    'tag%f'(Tail, TN),
    find_var(TN, Tail,Table,Inner,Ix).
find_var(6, Compound,Table,Where,Ix):-
    'arity%f'(Compound,Arity),
    inner_of(Where,Inner),
    find_tuple_vars(1,Arity,Table,Compound,Inner,Ix).

find_tuple_vars(Pos,Arity,_,_,_,_):-
    Pos>Arity,!.
find_tuple_vars(Pos,Arity,Table,Compound,Where,Ix):-
    '$ nth'(Compound,Pos,Term),
    'tag%f'(Term, N),
    find_var(N, Term,Table,Where,Ix),
    'increment%f'(Pos, NP),
    find_tuple_vars(NP,Arity,Table,Compound,Where,Ix).

record_occ(Con,Table,Where,Ix) :-
    occ(Con(Occs),Table), !,
    add_occ(Where(Ix),Occs).
record_occ(_,_,_,_).

add_occ(El, Var) :-
    var(Var), !,
    Var = [El|_].
add_occ(El, [_|Tail]) :-
    add_occ(El, Tail).

inner_of(head_arg,   head_term).
inner_of(head_term,  head_term).
inner_of(first_arg,  first_term).
inner_of(first_term, first_term).
inner_of(body_arg,   body_term).
inner_of(body_term,  body_term).
inner_of(last_arg,   last_term).
inner_of(last_term,  last_term).



/* allocate vars 
    given an initial determination where variables happen we now look
    to see how we can allocate a variable into register permanent and label */

allocate_vars([],[],_,_,P,P,NP,NP,_).
allocate_vars([V(Occs)|Table],[Entry|Vt],F,Hc,P,Perms,NP,NPr,SafeR):-
    nonvar(Occs),
    '$termin$'(Occs),
    classify_v(Occs, Type),
    alloc_var(Type,V,Occs,Entry,F,R1,Hc,P,Pe,NP,NP1,SafeR), !,
    allocate_vars(Table,Vt,R1,Hc,Pe,Perms,NP1,NPr,SafeR).
% unused variable names.  Do not delete this clause !!
% Unused variables may come from incomplete var analysis in ex_body/6
allocate_vars([_|Table],Vt,F,Hc,P,Perms,NP,NPr,SafeR):-
    allocate_vars(Table,Vt,F,Hc,P,Perms,NP,NPr,SafeR).

classify_v([_(_)], void) :- !.    % only one occurrence of the variable
classify_v([F|Rest], reg) :-      % try for a register allocation
    goal_number(F, N),
    same_goal(Rest, N).
classify_v([First|_], Type) :-    % otherwise it is a permanent variable
    classify_var(First, Type).

classify_var(head_arg(_),   safe).
classify_var(head_term(_),  safe).
classify_var(first_arg(_),  unsafe).
classify_var(first_term(_), safe).
classify_var(body_arg(_),   unsafe).
classify_var(body_term(_),  safe).
classify_var(last_arg(_),   unsafe).
classify_var(last_term(_),  safe).

same_goal([], _).
same_goal([F|Rest], N) :-
    goal_number(F, N),
    same_goal(Rest, N).

goal_number(head_arg(_),    1).
goal_number(head_term(_),   1).
goal_number(first_arg(_),   1).
goal_number(first_term(_),  1).
goal_number(body_arg(N/_),  N).
goal_number(body_term(N/_), N).
goal_number(last_arg(_),    0).  % Special case : if first occurrence 
goal_number(last_term(_),   0).  % is in last goal, so must the rest !!


% alloc_var computes the actual position of the variable depending on type
alloc_var(safe,V,[O|Occs],V(Last,safe,Lc,_),R,R,_,[Last(Lc)|P],P,NoP,NP,_):-
    'increment%f'(NoP, NP),
    lastocc(Occs,O,Last).
alloc_var(unsafe,V,[O|Occs],V(Last,unsafe,Lc,_),R,R,_,[Last(Lc)|P],P,NoP,NP,_):-
    'increment%f'(NoP, NP),
    lastocc(Occs,O,Last).
alloc_var(reg,V,Occs,V(_,reg,Reg,Used),R,NR,Hc,P,P,NP,NP,SafeR):-
    alloc_register(Occs,Reg,R,NR,Hc,Used,SafeR), !.
alloc_var(reg,V,[O|Occs],V(Last,safe,Lc,_),R,R,_,[Last(Lc)|P],P,NoP,NP,_):-
    'increment%f'(NoP, NP),
    lastocc(Occs,O,Last).
alloc_var(void,V,_,V(_,void,_,_),R,R,_,P,P,NP,NP,_).

lastocc([],Last,Ix) :- 
    lastoccur(Last,Ix).
lastocc([Occ|L],_,Ix) :-
    lastocc(L,Occ,Ix).

lastoccur(_(Ix/_),Ix) :- !.
lastoccur(_(Ix),Ix) :- !.

% if a variable ONLY occurs as a head argument then we can use that location
alloc_register(Occs,Reg,Free,Free,_,local,SafeR) :-
    del_occ(head_arg(R1),Occs,Occ1),
    no_occ(first_arg(_),Occ1),
    no_occ(first_term(_),Occ1),
    which_reg(head_arg(R),Occ1,R,R1,Reg), !,
    occ(Reg, SafeR).
% the 'best' is when it occurs as an argument in the first call
alloc_register(Occs,Reg,Free,Free,Hc,_,_):-
    occ(first_arg(Reg),Occs),
    Reg>Hc, !,
    Reg < 32.
% the 'best' is when it occurs as an argument in the first call
alloc_register([Head(Pos)|Occs],Reg,Free,Free,_,Used,SafeR):-
    occ(first_arg(Reg),Occs),
    on(Head,[head_arg,head_term]),
    Pos>=Reg,
    safe_reg(SafeR, Reg), !,
    Reg < 32,
    preset_reg(Reg,[Head(Pos)|Occs],Used).
% if a variable occurs as two first arguments then we can use 2nd location
alloc_register(Occs,Reg,Free,Free,_,_,_) :-
    del_occ(first_arg(R1),Occs,Occ1),
    no_occ(head_arg(_),Occ1),
    no_occ(head_term(_),Occ1), !,
    which_reg(first_arg(R),Occ1,R,R1,Reg).
% if a variable occurs as two body arguments then we can use 2nd location
alloc_register(Occs,Reg,Free,Free,_,_,_) :-
    del_occ(body_arg(_/R1),Occs,Occ1), !,
    which_reg(body_arg(_/R),Occ1,R,R1,Reg).
% if a variable occurs as two last arguments then we can use 2nd location
alloc_register(Occs,Reg,Free,Free,_,_,_) :-
    del_occ(last_arg(_/R1),Occs,Occ1), !,
    which_reg(last_arg(_/R),Occ1,R,R1,Reg).
% otherwise allocate a free register at the end of the file
alloc_register([H(_)|_],Reg,Reg,Free,_,_,_):-
    on(H, [head_arg, head_term, first_arg, first_term]),
    'increment%f'(Reg, Free),
    Reg < 32.

% determine which register to use - prefer 2nd location if appears twice
which_reg(Term, Occs, In, _, In) :-
    occ(Term, Occs),
    In < 32, !.
which_reg(_, _, _, Default, Default) :-
    Default < 32.

% check if the reg may be overwritten
safe_reg(List, _) :-
    var(List), !.
safe_reg([H|T], R) :-
    R \= H,
    safe_reg(T, R).

preset_reg(Reg,Occs,local):-
    occ(head_arg(Reg),Occs),!.
preset_reg(_,_,_).

del_occ(Occ, [Occ|List], List) :- !.
del_occ(Occ, [El|List], [El|Rest]) :-
    del_occ(Occ, List, Rest).



/* order local variables by last occurrence */
'$ compare_perms'(O1(_),O2(_)):-
    O1<O2.


/* allocate_perms determine the offset of a permanent variable within the 
   environment.
   This is chosen so that we can get rid of variables as soon as possible.
   This requires that we sort the list of permanent variables */
allocate_perms([],0).
allocate_perms([_(Offset)|Perms], Off):-
    Offset is Off--1,
    allocate_perms(Perms,Offset).


  
/* is_a_var retrieves the relevant data about a variable from the table */
/* each entry in the table looks like :-

   name(occurrence, type, allocation, local/global flag)

   'occurrence' is the goal# in which it last appears.
   'type' is one of void, reg, safe and unsafe.
   'allocation' is reg# for reg type, and var# for safe and unsafe variables.

*/
is_a_var(Var,VarTable,Type,Occs,Index,First):-
    occ(Var(Occs,Type,Index,First),VarTable).

/* various miscellaneous functions needed to support the compiler */

new_label(L, Lo, Label):-
    concat('L', L, Label),
    'increment%f'(L, Lo).

aux_label(L, Label):-
    concat('L', L, Lbl),
    concat(Lbl, a, Label).

% convert a difference list into an ordinary list, removing duplicates
makelist(End, End, L) :- !,
    '$termin$'(L).
makelist([H|T], End, L) :-
    occ(H, L),
    makelist(T, End, L).

/*------------------------------------------------------------------*/

/* quick compiler */

/* the clauses to be compiled are of the form '$c$pred'(..Args.. , Body). */
'$ assert_compiler'(Head, Locals, Code, Cout, Li, Li) :-    % assertions
    functor(Head, _, Arity),
    make_v_table(Locals,VarTable),
    ass_get_args(1,Arity,Head,VarTable,C0,C1,0,S0),
    gen_gc_test(S0,Arity,Code,C0),
    fix_vars(VarTable,Arity,LastReg),
    assert_deallocate(LastReg, C1, [succ()|Cout]).

/*------------------------------------------------------------------*/

/* optimised compiler for assertions */

ass_get_args(Pos,Arity,_,_,Code,Code,S,S):-
    Pos>Arity,!.
ass_get_args(Pos,Arity,Head,VarTable,Code,Cdout,S,Space):-
    arg(Pos,Head,Argument),
    'tag%f'(Argument,N),
    ass_get_arg(N,Argument,Pos,VarTable,Code,C0,S,S0),
    'increment%f'(Pos, NP),
    ass_get_args(NP,Arity,Head,VarTable,C0,Cdout,S0,Space).

/* get_arg handles one argument at a time. */
ass_get_arg(1, Int, Reg, _, [get_int(Reg, Int)|Code], Code,S,S).
ass_get_arg(2, Float, Reg, _, [get_float(Reg, Float)|Code], Code,S,S).
ass_get_arg(3, Var, Reg, Vars, Code, Co,S,S):-
    occ(Var(Occs), Vars), !,
    add_occ(arg(Reg, Code, Co), Occs).
ass_get_arg(3, Con, Reg, _, [get_const(Reg, Con)|Code], Code,S,S).
ass_get_arg(4, [], Reg, _, [get_nil(Reg)|Code], Code,S,S).
ass_get_arg(5, [H|T], Reg, Vars, [get_list(Reg)|C0], Cout,S,Space):-
    S0 is S++2,
    gen_push_pop(outer, H, C0, C1, C2, C3),    % generate a push inst'n if nec.
    'tag%f'(H, HN),
    ass_get_term(HN, H, Vars, C1, C2, S0, S1, inner),
    'tag%f'(T, TN),
    ass_get_term(TN, T, Vars, C3, Cout, S1, Space, outer).
ass_get_arg(6, Tuple, Reg, Vars, [get_tpl(Reg,Count)|C0], Cout,S,Space):-
    'arity%f'(Tuple, Count),
    S0 is S++Count++1,
    ass_get_tuple_args(Tuple, 1, Count, Vars, C0, Cout,S0,Space,outer).

/* ass_get_term compiles an inner term argument using unify_ instructions */
ass_get_term(1, Int,_,[unify_int(Int)|Code],Code,S,S,_).
ass_get_term(2, Float,_,[unify_float(Float)|Code],Code,S,S,_).
ass_get_term(3, Var,VarTable,Code,Cout,S,S,_):-
    occ(Var(Occs), VarTable), !,
    add_occ(term(Code, Cout), Occs).
ass_get_term(3, Con,_,[unify_const(Con)|Code],Code,S,S,_).
ass_get_term(4, [],_,[unify_nil()|Code],Code,S,S,_).
ass_get_term(5, [Head|Tail],VarTable,[unify_list()|Code],Cout,S,Space,Where):-
    S0 is S++2,
    gen_push_pop(Where, Head, Code, C1, C2, C3),    % generate a push instr'n if nec.
    'tag%f'(Head, HN),
    ass_get_term(HN, Head, VarTable, C1, C2,S0,S1,inner),
    'tag%f'(Tail, TN),
    ass_get_term(TN, Tail, VarTable, C3, Cout,S1,Space,Where).
ass_get_term(6, Tuple,VarTable,[unify_tpl(Count)|Code],Cout,S,Space,Where):-
    'arity%f'(Tuple, Count),
    S0 is S++Count++1,
    ass_get_tuple_args(Tuple, 1, Count, VarTable, Code, Cout,S0,Space,Where).

ass_get_tuple_args(Tuple, Arity, Arity, VarTable, Code, Cout,S,Space,Where):- !,
    '$ nth'(Tuple,Arity,Term),
    'tag%f'(Term, N),
    ass_get_term(N,Term,VarTable,Code,Cout,S,Space,Where).
ass_get_tuple_args(Tuple, Pos, Arity, VarTable, Code, Cout,S,Space,Where):-
    '$ nth'(Tuple,Pos,Term),
    gen_push_pop(Where, Term, Code, C1, C2, C3),
    'tag%f'(Term, N),
    ass_get_term(N, Term, VarTable, C1, C2,S,S0,inner),
    'increment%f'(Pos, NP),
    ass_get_tuple_args(Tuple, NP, Arity, VarTable, C3, Cout,S0,Space,Where).

fix_vars([], Reg, Reg).
fix_vars([_([Occ|Occs])|Rest], Reg, RegOut) :-
    '$termin$'(Occs),
    fix_var(Occs, Occ, Reg, Reg1),
    fix_vars(Rest, Reg1, RegOut).

fix_var([], arg(_,C,C), Reg, Reg) :- !.
fix_var([], term([unify_void(1)|C],C), Reg, Reg) :- !.
fix_var(Occs, arg(Pos,C,C), Reg, Reg) :- !,
    fix_occs(Occs, Pos).
fix_var(Occs, Occ, Reg, Reg) :-
    del_occ(arg(Pos,C,C), Occs, Rest), !,
    fix_occs([Occ|Rest], Pos).
/* run out of registers, use permanent variables */
fix_var(Occs, term([allocate(),unify_y_var(0)|C],C), 31, y(0)) :- !,
    fix_term_occs(Occs, y(0)).
fix_var(Occs, term([unify_x_var(Reg1)|C],C), Reg, Reg1) :-
    integer(Reg), !,
    'increment%f'(Reg, Reg1),
    fix_term_occs(Occs, Reg1).
fix_var(Occs, term([unify_y_var(Var1)|C],C), y(Var), y(Var1)) :-
    'increment%f'(Var, Var1),
    fix_term_occs(Occs, y(Var1)).

fix_occs([], _).
fix_occs([arg(Pos, [get_x_val(Pos,Reg)|C],C)|Rest], Reg) :- !,
    fix_occs(Rest, Reg).
fix_occs([term([unify_loc_x_val(Reg)|C],C)|Rest], Reg) :-
    fix_occs(Rest, Reg).

fix_term_occs([], _).
fix_term_occs([term([Code|C],C)|Rest], Reg) :-
    aux_fix_term_occs(Reg, Code),
    fix_term_occs(Rest, Reg).

aux_fix_term_occs(y(Var), unify_y_val(Var)) :- !.
aux_fix_term_occs(Reg, unify_x_val(Reg)).

assert_deallocate(y(_), [deallocate()|C], C) :- !.
assert_deallocate(_, C, C).


'$ nth'(T,N,E):- M is N--1, arg(M,T,E).

/* terminate an unsafe list */
'$termin$'([]) :- !.
'$termin$'([_|L]) :- '$termin$'(L).

'$ warn_locals'(on, Idx, Locals, Head, Bdy) :-
	split_up_body(Bdy, First, Body),
	analyse_vars(Locals, VarTable, Head, First, Body),
	'$warn_locals$'(on, Idx, VarTable, Head).
'$ warn_locals'(_, _, _, _, _).

'$warn_locals$'(on, Idx, VarTable, Head) :-
	collect_voids(VarTable, [V|Voids]),
	functor(Head, Pred, Arity), !,
	'$ style_warning'(single_var, vars(Idx, Pred/Arity, [V|Voids])).
'$warn_locals$'(_, _, _, _).

collect_voids([], []).
collect_voids([Var(_,void,_,_)|Rest], [Var|Voids]) :-
	\+ concat('_', _, Var), !,
	collect_voids(Rest, Voids).
collect_voids([_|Rest], Voids) :- collect_voids(Rest, Voids).
